#!/usr/bin/perl
# @(#) test_selector.pl	05-05-2004	Ulrich Jansen
#
# Bereitstellen eines Parameter-Strings fr die Tests.
#
#    ========== licence begin  GPL
#    Copyright (C) 2001 SAP AG
#
#    This program is free software; you can redistribute it and/or
#    modify it under the terms of the GNU General Public License
#    as published by the Free Software Foundation; either version 2
#    of the License, or (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#    ========== licence end
#

BEGIN {
	if ($^O !~ /win32/i) {
		unshift @INC, ("/devtool/TOOL/tool/lib/perl5", "/devtool/TOOL/tool/lib/Perl", "/devtool/TOOL/tool/bin", "/SAP_DB/TESTDB");
	}
	unshift @INC, ($^O =~ /win32/i ? "\\SAP_DB\\TESTDB\\lib" : "/SAP_DB/TESTDB/lib");
}

use strict;

use Net::HTTP;
use HTTP::Status;
use Sys::Hostname;
use File::Basename;
use XML::Simple;
use Getopt::Long;
use Time::Local;
use QAConnect;

$| = 1;
print "\ntest_selector 1.02 (c)2004, SAP AG\n\n";

# Local Variables:
my $DEBUG		= 1;
my $hostname 		= lc(hostname());
my $outstr          = "";
my $act_relver      = "";
my $work_profile	= "";
my $single_mode 	= 0;
my $lockfile_dir	= ($^O =~ /win32/i) ? "\\SAP_DB\\TESTDB\\Locks" : "/SAP_DB/TESTDB/Locks";
my %test_pkgs;
my $test_count 		= 0;
my $max_days		= 0;
my $clear_locks		= 0;
my $testgroups		= 0;
my $delay			= 0;
my $use_packid		= 0;
my $no_test			= 0;
my $relver_filter	= "";
my $is_chr			= 0;
my $max_tests		= 0;
my %test_order;
my $daytime			= (((localtime(time))[6] > 0) && ((localtime(time))[6] < 6)) ? ((((localtime(time))[2] > 8) && ((localtime(time))[2] < 18)) ? 1 : 0) : 0;
my %zombies;

# Read command line parameters:
GetOptions('profile=s' => \$work_profile, 'single_mode' => \$single_mode, 'max_days=i' => \$max_days, 'clear_locks' => \$clear_locks, 'testgroups=i' => \$testgroups, 'delay=i' => \$delay, 'use_packid' => \$use_packid, 'no_test' => \$no_test, 'rel_filter=s' => \$relver_filter, 'is_chr' => \$is_chr, 'max_tests=i' => $max_tests);
$work_profile = "workday" unless ($work_profile);
$max_tests = 6 unless ($max_tests);

QAConnect::setdbg($DEBUG);

# Prepare Lock dir:
mkdir ($lockfile_dir, 0777) unless (-d $lockfile_dir);

if ($clear_locks) {
	QAConnect::dbgout("Clearing lock files.");
	unlink <$lockfile_dir/*.lock>;
}

if ($no_test) {
	exit(0);
}


################################################################################


# Get Servers data from QADB.
my ($rc, $href)     = QAConnect::httpsql_request("QADB_HTTPSQL", "select IDPLATFORM, RELEASE_FILTER, TESTGROUPS, DELAY_WORKDAY, DELAY_WEEKEND, SYNC_LOW, MAX_DAYS from TESTER.SERVERS where HOST like '$hostname\%'", 1); 
(ref($href) and $href->{'Rows'}->{'Row'}[0]->{'IDPLATFORM'}) or QAConnect::throw_error("UNKNOWN HOSTNAME", "This host ($hostname) has no entry in the SERVERS table!", 1);

# Copy href data to local variables:
my $platform_id     = $href->{'Rows'}->{'Row'}[0]->{'IDPLATFORM'};
if (($href->{'Rows'}->{'Row'}[0]->{'RELEASE_FILTER'} ne "") && ($href->{'Rows'}->{'Row'}[0]->{'RELEASE_FILTER'} ne "?") && $relver_filter && ($relver_filter !~ /$href->{'Rows'}->{'Row'}[0]->{'RELEASE_FILTER'}/) ) {
	QAConnect::throw_error("RELEASE FILTER MISMATCH", "Manual filter '$relver_filter' and Machine filter '$href->{'Rows'}->{'Row'}[0]->{'RELEASE_FILTER'}' don't go together!", 1);
}
   $relver_filter   = ($relver_filter ne "" ? $relver_filter : $href->{'Rows'}->{'Row'}[0]->{'RELEASE_FILTER'});
   $testgroups      = ($single_mode ? 1 : ($testgroups>0 ? $testgroups : ($href->{'Rows'}->{'Row'}[0]->{'TESTGROUPS'} > 0 ? $href->{'Rows'}->{'Row'}[0]->{'TESTGROUPS'} : 2)));
   $delay           = ($single_mode ? 0 : ($delay>0 ? $delay : ($work_profile =~ /weekend/i ? ($href->{'Rows'}->{'Row'}[0]->{'DELAY_WEEKEND'} > 0 ? $href->{'Rows'}->{'Row'}[0]->{'DELAY_WEEKEND'} : 36000) : ($href->{'Rows'}->{'Row'}[0]->{'DELAY_WORKDAY'} > 0 ? $href->{'Rows'}->{'Row'}[0]->{'DELAY_WORKDAY'} : 7200))));
my $sync_low        = ($href->{'Rows'}->{'Row'}[0]->{'SYNC_LOW'} > 0 ? $href->{'Rows'}->{'Row'}[0]->{'SYNC_LOW'} : 0); 
   $max_days        = ($max_days > 0 ? $max_days : ($href->{'Rows'}->{'Row'}[0]->{'MAX_DAYS'} > 0 ? $href->{'Rows'}->{'Row'}[0]->{'MAX_DAYS'} : "3"));
my $time_ago        = QAConnect::getisotime (time - ($max_days*24*60*60));
   $max_tests		= 1 if ($single_mode);

################################################################################

# First, let's check all makes not tested, yet:
while (1)
{
	last unless scan_makes("SELECT * FROM TESTER.PRIOMAKES WHERE TS>$time_ago AND IDPLATFORM=$platform_id AND RELEVANCE='1' AND IDOBJSTATUS<=1000 ORDER BY PRIORITY DESC, ID DESC", 0);
}


# Finally, check all already tested makes:
while (1)
{
	last unless scan_makes("SELECT * FROM TESTER.PRIOMAKES WHERE TS>$time_ago AND IDPLATFORM=$platform_id AND RELEVANCE='1' AND IDOBJSTATUS>1000 ORDER BY PRIORITY DESC, ID DESC", 1);
}


# Finalize:
QAConnect::throw_error("OOPS!", "There seem to be nothing to test for me! ", 111) unless ($test_count > 0);

QAConnect::dbgout("\n");

QAConnect::dbgout("\nSELECTED THE FOLLOWING PACKAGE(S):\n");
foreach my $ordernum (sort keys %test_order) {
		my $pack = $test_order{$ordernum};
		if ($use_packid) {
			$outstr .= " -packid $test_pkgs{$pack}";
		} else {
			$outstr .= " -package $pack";
		}
		QAConnect::dbgout("\t$pack ($test_pkgs{$pack})");
}

QAConnect::dbgout("\nGenerated statement: $outstr");

QAConnect::dbgout("\n");
# Dump output 
open(ENV_OUT, ">>$ENV{'DTM_TASKEXPORTFILE'}") or QAConnect::throw_error("ENV OPEN ERROR", "Can't open temporary env output file!", 1);
print ENV_OUT "\nTEST_PKGS=$outstr\n";
print ENV_OUT "TESTGROUPS=-testgroups $testgroups -delay $delay\n" unless ($single_mode or ($testgroups == 1));
print ENV_OUT "SYNC_LOW=-sync_low\n" if ($sync_low);
close (ENV_OUT) or QAConnect::throw_error("OOPS!", "Can't close temoprary env file?!");
QAConnect::dbgout("Successfully printed '$outstr' to env file: $ENV{'DTM_TASKEXPORTFILE'}");
exit(0);

##############################################################################
# check_timeout() - Check/Wait for timeout.
##############################################################################
sub scan_makes
{
	my ($stmt, $tested) = @_;

	return 0 if ($test_count >= $max_tests);

	QAConnect::dbgout("SQL-Statement: $stmt");

	my ($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", $stmt, 1);
	return 0 unless ($res->{'Rows'}->{'Row'});
	my $max_rows = scalar(@{$res->{'Rows'}->{'Row'}});
	QAConnect::dbgout("Found $max_rows entries.");
	my $row_count = -1;
	
	while ($ok)
	{
		# First, guarantee abort condition:
		$row_count ++;
		last if ($row_count >= $max_rows);
		
		# Now, copy data:
		my $make_id 		= $res->{'Rows'}->{"Row"}[$row_count]->{'ID'};
		my $act_relver 		= $res->{'Rows'}->{"Row"}[$row_count]->{'RELVER'} . $res->{'Rows'}->{"Row"}[$row_count]->{'QASTATUS'};
		my $status		= $res->{'Rows'}->{"Row"}[$row_count]->{'IDOBJSTATUS'};
		my $time_started	= $res->{'Rows'}->{"Row"}[$row_count]->{'TS'};
		my $testruns		= $res->{'Rows'}->{"Row"}[$row_count]->{'TESTRUNS'};
		my $timeout		= $res->{'Rows'}->{"Row"}[$row_count]->{'LOCKTIMEOUT'};
		my $idqastatus 		= $res->{'Rows'}->{"Row"}[$row_count]->{'IDQASTATUS'};
		
		QAConnect::dbgout("Checking ID $make_id ($act_relver)...");
		
		# Check, if relver filter matches:
		if(($relver_filter ne "?") && ($act_relver !~ /$relver_filter/))
		{
			QAConnect::dbgout("Release filter '$relver_filter' didn't match --> Skipping.");
			next;
		}

		# Check, if relver is chr compatible:
		if(($is_chr) && ($act_relver =~ /7402/))
		{
			QAConnect::dbgout("Release 7402 is not a changeroot release --> Skipping.");
			next;
		}

		# Check, if make_id is already checked:
		if (-e "$lockfile_dir/$make_id.lock")
		{
			QAConnect::dbgout("Make ID $make_id is already locked.");
			next;
		}

		my $already_tested = get_testruns($make_id);
		if (!$tested && $already_tested) {
			QAConnect::dbgout("Make ID $make_id has already been tested. We're looking for untested makes for now.");
			next;
		}
		
		# NOW, we're sure that we found potential candidate (depends on lock timeout), so LOCK IT!:
		open (LOCK_FILE, ">$lockfile_dir/$make_id.lock") or QAConnect::throw_error("FATAL ERROR", "Can't open lock file '$lockfile_dir/$make_id.lock' for writing!\n$!", 1);
		print LOCK_FILE "Checked: " . scalar(localtime) . "\n";
		close (LOCK_FILE);
		
		QAConnect::dbgout("Locked ID $make_id.");
		
		if ($tested && !$already_tested) {
			if ($status >= 1000)
			{
				QAConnect::dbgout("ID $make_id was late...Taking it anyway!");
				$zombies{$make_id} = 0;
			} else {
				QAConnect::dbgout("ID $make_id is still not ready...discarding it finally.");
				$zombies{$make_id} = 1;
				next;
			}
		}
		
		# Check, if we've got an older version of an already checked release:
		if (check_later_version($make_id, $res->{'Rows'}->{"Row"}[$row_count]->{'RELVER'}, $res->{'Rows'}->{"Row"}[$row_count]->{'BUILDPFX'}, $res->{'Rows'}->{"Row"}[$row_count]->{'QASTATUS'}, $timeout, $idqastatus)) {
			next;
		}
		
		# Check, if make is still running (lbound) / already tested often enough (ubound).
		if ($status >= 1000)
		{
			QAConnect::dbgout("ID $make_id has finished making.");
			# Check ubound test count:
			QAConnect::dbgout("ID $make_id has already been tested $already_tested times.");
			if ($already_tested < $testruns)
			{
				QAConnect::dbgout("Taking package $act_relver (ID $make_id)!");
				select_relver($make_id, $act_relver);
			} else {
				QAConnect::dbgout("ID $make_id has been tested often enough...");
			}
		}
		elsif ($status == 999)
		{
			QAConnect::dbgout("Make with ID $make_id FAILED (Status $status)! Skipping...");
			next;
		}
		elsif ($status == 990)
		{
			QAConnect::dbgout("Make with ID $make_id was cancelled! Skipping...");
			next;
		}
		else
		{
			if ($daytime) {
				QAConnect::dbgout("ID $make_id is still making...");
				QAConnect::dbgout("It's daytime! No time to wait for latecomers...");
				$zombies{$make_id} = 1;
				if (unlink ("$lockfile_dir/$make_id.lock")) {
					QAConnect::dbgout("Unlocked ID $make_id again.");
				
				} else {
					QAConnect::dbgout("Unlocking ID $make_id FAILED!");
				}
				next;
			}
			
			QAConnect::dbgout("ID $make_id is still making...");
			# Check lbound timeout:
			my $interval_steps = int(($timeout - (time() - QAConnect::isotime2long($time_started))) / 60) + 1;
			if ($interval_steps <= 1)
			{
				QAConnect::dbgout("Seems to be a zombie make...");
				$zombies{$make_id} = 1;
				if (unlink ("$lockfile_dir/$make_id.lock")) {
					QAConnect::dbgout("Unlocked ID $make_id again.");
				
				} else {
					QAConnect::dbgout("Unlocking ID $make_id FAILED!");
				}
				next;
			}
			QAConnect::dbgout("Waiting for up to $interval_steps minutes...");
			while ($interval_steps > 0)
			{
				sleep (60);
				my ($loc_ok, $loc_res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select idobjstatus from makes where id=$make_id", 1);
				$status = $loc_res->{'Rows'}->{"Row"}[0]->{'IDOBJSTATUS'};
				if ($status >= 1000)
				{
					QAConnect::dbgout("Make $make_id is done! Taking package $act_relver.");
					select_relver($make_id, $act_relver);
					return 1;
				}
				elsif ($status == 999)
				{
					QAConnect::dbgout("Make $make_id FAILED! Skipping...");
					return 1;
				}
				$interval_steps --;
				if ($DEBUG)
				{
					if ($interval_steps%10) {print "."; }
					else { print "$interval_steps"; }
				}
			}
			if ($interval_steps <= 0)
			{
				QAConnect::dbgout("Sorry, time's up! I'm taking the next one (this one may be a zombie)...");
				$zombies{$make_id} = 1;
				QAConnect::dbgout("Unlocking ID $make_id again...");
				if (unlink ("$lockfile_dir/$make_id.lock")) {
					QAConnect::dbgout("Unlocked ID $make_id again.");
				
				} else {
					QAConnect::dbgout("Unlocking ID $make_id FAILED!");
				}
				return 1;
			}
			return 1;
		}
		
		# Abort condition:
		if ($test_count >= $max_tests) {
			QAConnect::dbgout("Maximum tests reached!");
			last;
		}
	}
	
	return 0;	
}


##############################################################################
# get_testruns() - Retrieves the number of started Tests.
##############################################################################
sub get_testruns
{
	my $id = shift;
	my $testcount = 0;
	my $sharp = 0;
	QAConnect::dbgout("Checking Testruns for ID $id:");
	my ($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select IDOBJSTATUS from makes where id=$id", 1);
	return 0 unless $ok;
	my $stat = $res->{'Rows'}->{"Row"}[0]->{'IDOBJSTATUS'};
	if ($stat == 1000)
	{
		QAConnect::dbgout("ID $id has marked as 'ready to be tested', so let's test it!");
		return 0;
	}
	($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select OBJSTATUS from monitor_sessions where idmake=$id and SESSION_TYPE='Nighttest' order by id desc", 1);
	return $testcount unless ($ok && defined ($res->{'Rows'}->{'Row'}));
	my $maxrow = scalar(@{$res->{'Rows'}->{'Row'}});
	return 0 unless ($maxrow);
	QAConnect::dbgout("\tFound $maxrow sessions.");
	$testcount ++;
	my $actrow = 1;
	while ($actrow < $maxrow) { $testcount ++ if ($res->{'Rows'}->{"Row"}[$actrow]->{'OBJSTATUS'} =~ /^end\sof\stesting/i); $actrow++; }
	QAConnect::dbgout("\tFound $testcount running/completed tests.");
	return $testcount;
}

##############################################################################
# select_relver() - finally adds a release to selected array.
##############################################################################
sub select_relver {
    my ($make_id, $act_relver) = @_;
	if ($test_pkgs{$act_relver}) {
		QAConnect::dbgout("WARNING: Release $act_relver is already selected!");
		if ($test_pkgs{$act_relver} > $make_id) {
				QAConnect::dbgout("         Discarded ID $make_id!");
		} else {
				QAConnect::dbgout("         Discarded ID $test_pkgs{$act_relver}!");
				$test_pkgs{$act_relver} = $make_id;
		}
	} else {
		$test_pkgs{$act_relver} = $make_id;
		$test_count ++;
		$test_order{$test_count} = $act_relver;
		QAConnect::dbgout("Took Test $test_count of $max_tests.");
	}
    QAConnect::dbgout("Selected package $act_relver (ID $test_pkgs{$act_relver})");
}

##############################################################################
# check_later_versions() - Check for any later version.
##############################################################################
sub check_later_version
{
	my ($id, $ver, $build, $state, $timeout, $idqastatus) = @_;
	my ($ok, $res) = (0, "");
	my $row_count = 0;
	if ($timeout == 0) {
		($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select * from priomakes where relver='$ver' and buildpfx='$build' and IDPLATFORM=$platform_id and idobjstatus>=1000 and relevance='1' order by id desc", 1);
	} else {
		($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select * from priomakes where relver='$ver' and buildpfx='$build' and IDPLATFORM=$platform_id and relevance='1' order by id desc", 1);
	}
	while ($res->{'Rows'}->{'Row'}[$row_count]->{'ID'} != $id) {
		my $test_id = $res->{'Rows'}->{'Row'}[$row_count]->{'ID'};
		if ($zombies{$test_id} == 1) {
			QAConnect::dbgout("ID $test_id is newer than $id, but has been marked as zombie! --> Skipping!");
		} elsif ($res->{'Rows'}->{'Row'}[$row_count]->{'IDOBJSTATUS'} =~ /^\d*9$/) {
			QAConnect::dbgout("ID $test_id is newer than $id, but seem to have errors! --> Skipping!");
		} elsif ($res->{'Rows'}->{'Row'}[$row_count]->{'IDOBJSTATUS'} =~ /^990$/) {
			QAConnect::dbgout("ID $test_id is newer than $id, but was cancelled! --> Skipping!");
		} elsif ($res->{'Rows'}->{'Row'}[$row_count]->{'IDQASTATUS'} < $idqastatus) {
			QAConnect::dbgout("WARNING: ID $test_id is newer than $id, BUT HAS A LOWER QASTATE!!! ($res->{'Rows'}->{'Row'}[$row_count]->{'QASTATUS'} < $state) --> Skipping!");
		} else {
			QAConnect::dbgout("Found a newer version: ID $test_id is newer than ID $id.");
			return 1;
		}
		$row_count ++;
	}
	
	$row_count = 0;
	if ($timeout == 0) {
		($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select * from priomakes where relver='$ver' and qastatus='$state' and IDPLATFORM=$platform_id and idobjstatus>=1000 and relevance='1' order by id desc", 1);
	} else {
		($ok, $res) = QAConnect::httpsql_request("QADB_HTTPSQL", "select * from priomakes where relver='$ver' and qastatus='$state' and IDPLATFORM=$platform_id and relevance='1' order by id desc", 1);
	}
	while ($res->{'Rows'}->{'Row'}[$row_count]->{'ID'} != $id) {
		my $test_id = $res->{'Rows'}->{'Row'}[$row_count]->{'ID'};
		if ($zombies{$test_id} == 1) {
			QAConnect::dbgout("ID $test_id is newer than $id, but has been marked as zombie! --> Skipping!");
		} elsif ($res->{'Rows'}->{'Row'}[$row_count]->{'IDOBJSTATUS'} =~ /^\d*9$/) {
			QAConnect::dbgout("ID $test_id is newer than $id, but seem to have errors! --> Skipping!");
		} elsif ($res->{'Rows'}->{'Row'}[$row_count]->{'IDOBJSTATUS'} =~ /^990$/) {
			QAConnect::dbgout("ID $test_id is newer than $id, but was cancelled! --> Skipping!");
        } elsif ($res->{'Rows'}->{'Row'}[$row_count]->{'BUILDPFX'} < $build) {
			QAConnect::dbgout("ID $test_id is newer than $id, but HAS A LOWER BUILD PREFIX!!! --> Skipping!");
		} else {
			QAConnect::dbgout("Found a newer version: ID $test_id is newer than ID $id.");
			return 1;
		}
		$row_count ++;
	}
	QAConnect::dbgout("ID $id (Rel $ver.$build($state)) is the latest (ready one)!");
	return 0;
}

