#!/usr/bin/env python
# createSimplePackage.py

import sys
import string
import re
import os
from ftplib import FTP

class Packer:
    def __init__ (self, rootdir, excludes = []):
        self.rootdir = rootdir
        self.excludes = map (re.compile, excludes)
        self.infiles = []

    def addTarget (self, vmakeTarget):
        #
        # create distribution list
        #
        output = perlOutput ('imf', '-F', vmakeTarget)
        rex = re.compile ("Creating distribution file '(.*)'")
        for line in output:
            match = rex.match (line)
            if match:
                listfile = match.group (1)
        #
        # add files
        #
        normroot = normfile (self.rootdir)
        rootlen = len (normroot)
        instroot = os.environ ['INSTROOT']
        entries = string.split (open (listfile, 'r').read ())
        for entry in entries:
            if self.isExcluded (entry):
                continue
            fullpath = os.path.join (instroot, entry)
            normpath = normfile (fullpath)
            if normpath [:rootlen] == normroot:
                self.infiles.append (fullpath [rootlen + 1:])

    def isExcluded (self, fname):
        for exclude in self.excludes:
            if exclude.search (fname):
                return 1
        return None


    def createArchive (self, name, outdir):
        archiveName = self.archiveName (name)
        extension = detectExtension ()
        outfile = os.path.join (outdir, archiveName) + extension
        os.chdir (self.rootdir)
        packFiles (outfile, self.infiles)
        return outfile

    def archiveName (self, name):
        prefix = 'maxdb'
        platform = detectPlatform ()
        release = detectRelease ()
        return '%(prefix)s-%(name)s-%(platform)s-%(release)s' % locals ()

def perlOutput (cmd, *args):
    perlfile = os.path.join (os.environ ['TOOL'], 'bin', cmd + '.pl')
    cmdline = 'perl ' + perlfile + ' ' + string.join (args)
    lines = os.popen (cmdline, 'r').readlines ()
    return map (string.strip, lines)

toolvars = {}
def readToolvars ():
    lines = perlOutput ('toolvarsDump')
    state = 'wkey'
    for line in lines:
        if state == 'wkey':
            if not line:
                continue
            key = line
            state = 'wvalue'
        elif state == 'wvalue':
            toolvars [key] = line
            state = 'wkey'
            del key
        else:
            raise 'invalid state', state
    return
    items = toolvars.items ()
    items.sort ()
    for key, value in items:
        print '%s=%s' % (key, value)
readToolvars ()


osspecMap = {
    'WIN32': 'win',
    'LINUX': 'linux',
    'SVR4': 'solaris',
    'AIX': 'aix',
    'HPUX': 'hpux',
    'OSF1': 'tru64',
}

machMap = {
    'I386': 'i386',
    'SUN': 'sparc',
    '_IBMR2': 'ppc',
    'PA20W': 'hppa',
    'PA11': 'hppa',
    'ALPHA': 'alpha',
    'IA64': 'ia64',
    'HP_IA64': 'ia64',
}

def detectPlatform ():
    osspec = osspecMap [toolvars ['OSSPEC']]
    mach = machMap [toolvars ['MACH']]
    return osspec + '-' + mach

def detectRelease ():
    relver = toolvars ['RELVER']
    major = int (relver [1])
    minor = int (relver [2])
    corr = int (toolvars ['CORRECTION_LEVEL'])
    build = int (toolvars ['BUILDPRAEFIX'])
    if toolvars ['RELSTAT'] == 'TEST':
        alpha = 'a'
    else:
        alpha = ''
    return '%d.%d.%02d.%02d%s' % (major, minor, corr, build, alpha)

def detectExtension ():
    if sys.platform == 'win32':
        return '.zip'
    else:
        return '.tgz'

def normfile (fname):
    return os.path.normcase (os.path.normpath (fname))

def packFiles (outfile, files):
    if sys.platform == 'win32':
        pipe = os.popen ('zip -@ ' + outfile, 'w')
        for file in files:
            pipe.write (file + '\n')
    else:
        os.system ('tar cf - ' + string.join (files) + ' | gzip -c > ' + outfile)

def split2 (text, delimiter, optionalIsSecond = 1):
    pos = string.find (text, delimiter)
    if pos == -1:
        if optionalIsSecond:
            return (text, '')
        else:
            return ('', text)
    else:
        return (text [:pos], text [pos + 1:])

def parseTarget (target):
    userinfo, location = split2 (target, '@', None)
    if userinfo:
        username, pwd = split2 (userinfo, ':', 1)
    else:
        username, pwd = ('anonymous', 'anonymous@invalidserver.net')
    host, dirname = split2 (location, ':', 1)
    return host, username, pwd, dirname

def ftpTransfer (ftpinfo, outfile):
    host, username, pwd, targetdir = parseTarget (ftpinfo)
    ftp = FTP(host, username, pwd)
    if targetdir:
        ftp.cwd (targetdir)
    cmd = 'STOR ' + os.path.basename (outfile)
    stream = open (outfile, 'rb')
    ftp.storbinary (cmd, stream, 8192)
    stream.close ()
    ftp.close ()

def checkOptions (options, args):
    if options.test:
        print detectPlatform ()
        sys.exit (0)

def main (options, args):
    """create an archive containing all distrib files
    """
    checkOptions (options, args)
    packer = Packer (options.rootdir, options.exclude)
    for arg in args:
        packer.addTarget (arg)
    outfile = packer.createArchive (options.name, options.outdir)
    if options.ftp:
        ftpTransfer (options.ftp, outfile)
        os.remove (outfile)
    else:
        print outfile

def _options ():
    return [
        # (optstring, varname, typechar, default, help)
        ('rootdir', None, ':', os.environ ['INSTROOT'], 'files are archived relative to this dir'),
        ('outdir', None, ':', os.environ ['TMP'], 'output directory'),
        ('exclude', None, '::', None, 'exclude files matching these patterns'),
        ('name', None, ':', 'unknown', 'name of package (default: first target)'),
        ('test', None, '', None, 'test environment'),
        ('ftp', None, ':', None, '[<user>:<pwd>@]host[:<path>] for FTP transfer'),
        ]

if __name__ == '__main__':
    import optlib
    optlib.optMain2 (main, _options ())

