#ifndef __PLAYER_H
#define __PLAYER_H
/*-------------------------------------------------------------------------
 * Copyright (c) 2000 Kenneth W. Sodemann (stufflehead@bigfoot.com)
 *-------------------------------------------------------------------------
 * player
 *
 * Synopsis:
 *    The player class.
 *
 * $Id: player.h,v 1.1 2000/09/05 15:16:12 stuffle Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to 
 * Free Software Foundation, Inc. 
 * 59 Temple Place, Suite 330 
 * Boston, MA  02111-1307  USA
 *
 *-------------------------------------------------------------------------
 */
/*! \file player.h
 
  \brief Define player objects

  The player objects store data about both human and computer players.

  The definition for the player structure is not visible outside of
  this module.  All player operations should be handled via the defined
  interface methods.  This may seem silly at this point, but the idea is
  to provide a consistent interface that can be easily expanded if the
  functionallity of the player increases as I develope this game.

  \author Kenneth W. Sodemann <stufflehead@bigfoot.com>
  $Revision: 1.1 $
  $Date: 2000/09/05 15:16:12 $
*/

#include <glib.h>

/*! \enum player_type
 * Define the type of player this player object is.  Currently there are
 * two types of players, HUMAN and COMPUTER.
 */
/*! \var HUMAN
 * The player object represents a human player.
 */
/*! \var COMPUTER
 * The player object represents a computer player.
 */
typedef enum
{
   HUMAN = 1,
   COMPUTER
} player_type;

#ifndef __PLAYER_BODY
/*! \typedef  player_object
 *
 * Use the player_object type to create new player objects.  Player objects 
 * can be human or computer players.  The individual fields of this structure
 * are not for public consumption, and are thus hidden.  Use the defined
 * interface methods to manipulate player objects.
 */
typedef struct player_object player_object;

/*! \fn create_player (const gchar   *name, 
                       player_type  type)
  
   \brief Create a player object
  
   \param name
   The name that the player should have
  
   \param type
   The type of player to create (see player_type)
  
   \return
   A pointer to the newly created player object.
*/
player_object *create_player (const gchar   *name,
                              player_type    type);

/*! \fn destroy_player (player_object *player)
  
   \brief Destroy a player object, freeing all memory used by it
  
   \param player
   The player to be destroyed.
  
   \return
   Nothing.
*/
void destroy_player (player_object *player);

/*! \fn reset_player (player_object *player)
  
   \brief Reset the player for another game
  
   Clear the score, and perform any other tasks that are needed to 
   make the player ready for the start of a game.
  
   \param player
   The player to be reset.
  
   \return
   Nothing.
*/
void reset_player (player_object *player);

/*! add_to_player_score (player_object  *player
                         gint            score)
  
   \brief Add to a player's score
  
   \param player
   The player whose score needs to be adjusted.
  
   \param score
   The amount to adjust the player's score by (can be zero or negative).
  
   \return
   The new score for the player.
*/
gint add_to_player_score (player_object *player,
                          gint           score);

/*! get_type_of_player (player_object *player)
  
   \brief Return the type of player for the given player object.
  
   \param player
   The player in question.
  
   \return
   The type of player for the player object.
*/
player_type get_type_of_player (player_object *player);

/*! get_player_score (player_object *player)
  
   \brief Return the score for the given player object.
  
   \param player
   The player in question.
  
   \return
   The score for the player object.
*/
gint get_player_score (player_object *player);

/*! get_player_name (player_object *player)
  
   \brief Return the type of player for the given player object.
  
   \param player
   The player in question.
  
   \return
   The name for the player object.
*/
const gchar *get_player_name (player_object *player);

#endif /* not in player body */
#endif
