/* This file is part of the KDE project

   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/


#ifndef SOCKETSAFE_H
#define SOCKETSAFE_H

#include <qca.h>
#include <kbufferedsocket.h>

class QDomElement;

namespace Mailody {

/**
 * @class SocketSafe
 * Responsible for communicating with the server with safe IMAPs
 * @author Tom Albers <tomalbers@kde.nl>
 */
class SocketSafe : public QObject
{
    Q_OBJECT

    public:
        enum Secure
        {
            NONE = 0,
            SSL,
            TLS,
            CLEAR
        };

        /**
         * Contructor, it will not auto connect. Call reconnect() to connect to
         * the parameters given.
         * @param parent the parent
         * @param name the name, will be printed  on debug.
         * @param server the server to connect to
         * @param port the port to connect to
         * @param safe the safety level
         */
        explicit SocketSafe(QObject* parent, const char* name,
                            const QString& server, int port, Secure safe);

        /**
        * Destructor
        */
        ~SocketSafe();

        /**
         * Call this when you will terminate the connection, it will
         * prevent a popup to the user 'do you want to reconnect'
         */
        void aboutToClose() { m_aboutToClose = true; };

        /**
         * Existing connection will be closed and a new connection will be
         * made
         */
        virtual void reconnect();

        /**
         * Write @p text to the socket
         */
        virtual void write(const QString& text);

        /**
         * @return true when the connection is live and kicking
         */
        virtual bool available();

    private:

        QCA::TLS*           m_ssl;
        QCA::Cert           cert;
        QPtrList<QCA::Cert> rootCerts;


        KNetwork::KBufferedSocket*    m_socket;
        QString             m_server;
        int                 m_port;
        Secure              m_safe;

        bool                m_aboutToClose;
        bool                m_crypted;

        void login();
        void showCertInfo(const QCA::Cert &cert);
        QString resultToString(int result);
        QPtrList<QCA::Cert> getRootCerts(const QString &store);

    signals:
        /**
         * emits the incoming data
         */
        void data(const QString&);

        /**
         * emitted when there is a connection (ready to send something)
         */
        void connected();

        /**
         * emitted when the dns request is ok (pretty useless, but still).
         */
        void hostFound();

        /**
         * emitted when there is something wrong, please issue a self
         * dustruct on me!
         */
        void error(const QString&);

        /**
         * emitted when disconnected, but only when the aboutToClose() is
         * not called before. Please delete this class when this signal
         * is emitted.
         */
        void disconnected();

        /**
         * this is the signal when tls succeeded
         */
        void tlscomplete();

    private slots:
        void slotStateChanged( int );
        void slotError(int i);
        void unCryptedReadyRead();
        void slotDisconnected();

        void ssl_error(int x);
        void ssl_closed();
        void ssl_readyReadOutgoing(int);
        void ssl_handshaken();
        void cryptedSocketRead();
};

}

#endif

