/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Bar.cc
    \brief Implementation of the Template class Bar.
    
    Magics Team - ECMWF 2004
    
    Started: Wed 5-May-2004
    
    Changes:
    
*/



#include "Bar.h"
#include "PointsHandler.h"
#include "LegendVisitor.h"
#include "Flag.h"

using namespace magics;

Bar::Bar() 
{}


Bar::~Bar() 
{}

/*!
 Class information are given to the output-stream.
*/		
void Bar::print(ostream& out)  const
{
	out << "Bar[";
	out << "]";
}

void  Bar::visit(Transformation& transformation, Data<UserPoint>&)
{
	if ( !annotation_.empty() )
		transformation.needTopAxis(true);
}

void Bar::operator()(Data<UserPoint>& data, BasicGraphicsObjectContainer& out)
{
	vector<double> xpos;
	
	vector<double> ypos;
    
    const Transformation& transformation= out.transformation();

    double y = transformation.getMinY() + (transformation.getMaxY()-transformation.getMinY())*0.2;

	CustomisedPointsList points; 
	std::set<string> request;
	data.customisedPoints(request, points);
	if (points.empty()) return;
	double previous =  (**points.begin())["x"];
	double next =  (**points.begin())["x"];
	double x, top, bottom, left, right;
	
	if (width_ == INT_MAX) {
		if ( points.size() > 1 ) 
			width_ = abs((*points[0])["x"] - (*points[1])["x"])*0.8;
		else 
			width_ = abs((*points[0])["x"] - (*points[1])["x"]);
			 
	}
	
	vector<string>::iterator annotation = annotation_.begin();
	
	for (CustomisedPointsList::const_iterator point = points.begin(); point != points.end(); ++point)
	{
		
		x = (**point)["x"];
		next = ((point+1) != points.end() ) ? (**(point+1))["x"] : x;
		bottom = (min_value_ == INT_MAX) ? (**point)["y_lower"]: min_value_;
		top = (**point)["y_upper"];
		left = x - width_/2;

		right = x + width_/2;
		previous = x;
		Polyline* box  = new Polyline();
		box->setColour(*line_colour_);
		
		box->setThickness(thickness_);
		box->setLineStyle(style_);
		if ( clipping_) {
			box->push_back(transformation.clip_point(UserPoint(left, bottom)));
			box->push_back(transformation.clip_point(UserPoint(left, top)));
			box->push_back(transformation.clip_point(UserPoint(right, top)));
			box->push_back(transformation.clip_point(UserPoint(right, bottom)));
			box->push_back(transformation.clip_point(UserPoint(left, bottom)));
		}
		else {
			box->push_back(transformation(UserPoint(left, bottom)));
			box->push_back(transformation(UserPoint(left, top)));
			box->push_back(transformation(UserPoint(right, top)));
			box->push_back(transformation(UserPoint(right, bottom)));
			box->push_back(transformation(UserPoint(left, bottom)));
		}
		

		const_cast<GraphShadeStyle*>(&shade_->getStyle())->setColour(colour_->clone());
		
		(*shade_)(*box);
		PaperPoint point(transformation.x(x), y);
		if ( annotation != annotation_.end() && transformation.in(point) ) {
			Text* annot = new Text();
			annot->addText(*annotation, Colour(annotation_colour_), annotation_size_);
			annot->setVerticalAlign(MBOTTOM);
			annot->push_back(point);
			texts_.push_back(annot);
			annotation++;
		}
		
		out.push_back(box);			
	}



}







void Bar::visit(LegendVisitor& legend)
{
	if ( !legend_) return;
	Polyline* curve  = new Polyline();
	(*curve).setColour(*colour_);
	(*curve).setLineStyle(style_);
	(*curve).setThickness(thickness_);
  
     curve->setFilled(true);
	 
	 shade_->legend(*curve);
		   		
	
    
	LegendEntry * entry = new BoxEntry(legend_text_, curve);

	legend.add(entry);
}

GraphFlag::GraphFlag() 
{}


GraphFlag::~GraphFlag() 
{}

/*!
 Class information are given to the output-stream.
*/		

void GraphFlag::print(ostream& out)  const
{
	out << "GraphFlag[";
	out << "]";
}



void GraphFlag::operator()(Data<UserPoint>& data, BasicGraphicsObjectContainer& out)
{
	CustomisedPointsList points; 
	std::set<string> request;
	request.insert("intensity");
	request.insert("direction");
	data.customisedPoints(request, points);
    
    if (points.empty() ) return;
	
	Flag* flags = new Flag();
	flags->setColour(*colour_); 
	flags->setLength(length_);
   
	flags->setOriginMarker("magics_15");
	flags->setOriginHeight(0.05);
	flags->setConvention(KNOTS);
	int i = 0;
	double x, missing;
	double intensity, direction;
    
	for (CustomisedPointsList::const_iterator point = points.begin(); point != points.end(); ++point) {
		i++;
       
		x = (**point)["step"];
		missing  = ((*point)->find("intensity") != (*point)->end()) ? (**point)["missing"] : -99999999;
        double lat = 10;
        if ( (*point)->find("latitude") != (*point)->end() )
            lat = (**point)["latitude"];
        flags->setHemisphere(lat >0 ? NORTH: SOUTH);
		if ( (*point)->find("intensity") != (*point)->end() && (*point)->find("direction") != (*point)->end() )   {
			intensity = (**point)["intensity"];
			direction = (**point)["direction"];
			if ( intensity == missing || direction == missing) 
				continue;	
			PaperPoint pos(x, 0);
			flags->push_back(ArrowPoint(intensity, direction, pos));
			flags->back().set(intensity, direction);
		}
	}
	if ( !flags->empty() ) out.push_back(flags);	
	
	

}







void GraphFlag::visit(LegendVisitor& legend)
{
	
}
GraphArrow::GraphArrow() 
{}


GraphArrow::~GraphArrow() 
{}

/*!
 Class information are given to the output-stream.
*/		

void GraphArrow::print(ostream& out)  const
{
	out << "GraphArrow[";
	out << "]";
}



void GraphArrow::operator()(Data<UserPoint>& data, BasicGraphicsObjectContainer& out)
{
	CustomisedPointsList points; 
	std::set<string> request;
	request.insert("intensity");
	request.insert("direction");
	data.customisedPoints(request, points);
	
	Arrow* arrow = new Arrow();
	arrow->setColour(*colour_); 
	
   
	arrow->setOriginMarker("magics_15");
	arrow->setOriginHeight(0.05);
	arrow->setScale(unit_velocity_);
	int i = 0;
	double x, missing;
	double intensity, direction;
	for (CustomisedPointsList::const_iterator point = points.begin(); point != points.end(); ++point) {
		i++;
       
		x = (**point)["step"];
		missing  = ((*point)->find("intensity") != (*point)->end()) ? (**point)["missing"] : -99999999;
        
		if ( (*point)->find("intensity") != (*point)->end() && (*point)->find("direction") != (*point)->end() )   {
			intensity = (**point)["intensity"];
			direction = (**point)["direction"];
			if ( intensity == missing || direction == missing) 
				continue;	
			PaperPoint pos(x, 0);
			arrow->push_back(ArrowPoint(intensity, direction, pos));
			arrow->back().set(intensity, direction);
		}
	}
	if ( !arrow->empty() ) out.push_back(arrow);	
	
	

}




void Bar::visit(TopAxisVisitor& top)
{

	for ( vector<Text*>::iterator text = texts_.begin(); text != texts_.end(); ++text)
		top.push_back(*text);


}


void GraphArrow::visit(LegendVisitor& legend)
{
	
}
