/*  gtktiemu - a TI89/92/92+ emulator
 *  (c) Copyright 2000-2001, Romain Lievin and Thomas Corvazier
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "intl.h"

/*
 * Hardware handling of the TI92 (linkport)
 *
 */
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <signal.h>
#include <time.h>
#include "str.h"
#include "sysdeps.h"
#include "config.h"
#include "options.h"
#include "hardware.h"
#include "memory.h"
#include "keyboard.h"
#include "newcpu.h"
#include "packets.h"
#include "globinfo.h"
#include "callbacks.h"
#include "lib68k.h"
#include "tilibs.h"
#include "errcodes.h"


LinkCable lc;
int byteAvail = 0;
UBYTE lastByte;
int internalLink = 0;
int timeOut = 0;
int iput = 0;
int iget = 0;
int byte_t2f, byte_f2t;
LinkCable *ilc = NULL;
TicalcFncts itc;
InfoUpdate iu = { 0 };


/*
  Linkport management (open/close)
*/

void print_lc_error(int errnum)
{
  char msg[MAXCHARS] = "No error -> bug !\n";

  ticable_get_error(errnum, msg);
  DISPLAY(_("Link cable error: code = %i, msg = %s\n"), errnum, msg);
  //iupdate_msgbox(_("Error"), msg);
}

void ti68k_open_linkport()
{
  int err;

  ticable_set_param(&(glob_inf.linkParam));
  ticable_set_cable(glob_inf.linkParam.link_type, &lc);
  if( (err=lc.init_port()) )
    {
      print_lc_error(err);
      return;
    }
  if( (err=lc.open_port()) )
    {
      print_lc_error(err);
      return;
    }
}

void ti68k_close_linkport()
{
  int err;

  if( (err=lc.close_port()) )
    {
      print_lc_error(err);
      return;
    }
  if( (err=lc.term_port()) )
    { 
      print_lc_error(err);
      return;
    }
}


/*
  Linkport management (send/receive)
*/


void linkport_putbyte(UBYTE arg)
{
  int err;
  
  timeOut = 0;
  if(internalLink)
    {
      byte_t2f = arg;
      iget = 1;
    }  
  else
    {
      if( (err=lc.put(arg)) )
	{
	  print_lc_error(err);
	  return;
	}
    }
}

UBYTE linkport_getbyte(void)
{
  timeOut = 0;
  if(internalLink) 
    {
      iput = 0;
      return byte_f2t;
    }

  byteAvail = 0;
  return lastByte;
}

int linkport_byteavail(void)
{
  if(internalLink) 
    return iput;

  return byteAvail;  
}

int linkport_checkread(void)
{
  int err = 0;
  int status = 0;

  if(internalLink) 
    return iput;

  if(byteAvail)
    return 0;

  if( (err=lc.check_port(&status)) )
    {
      print_lc_error(err);
      byteAvail = 0;
    }
  if(status & STATUS_RX)
    {
      if( (err=lc.get(&lastByte)) )
        {
          print_lc_error(err);
        }
      byteAvail = 1;
    }
  else
    byteAvail = 0;
  
  return byteAvail;
}

/*
  Internal link port emulation for sending/receiving files
*/

int ilp_init_port()     { return 0; }
int ilp_open_port()     { return 0; }
int ilp_put(byte data)
{ 
  byte_f2t = data; 
  iput = 1;
  while(iput) 
    { 
      ti68k_doInstructions(1); 
    };
  
  return 0; 
}
int ilp_get(byte *data)
{ 
  while(!iget) 
    { 
      ti68k_doInstructions(1); 
    };
  *data = byte_t2f;
  iget = 0; 
  
  return 0; 
}
int ilp_probe_port()    { return 0; }
int ilp_close_port()    { return 0; }
int ilp_term_port()     { return 0; }
int ilp_check_port(int *st) { return 0; }

void ilp_start()   { }
void ilp_stop()    { }
void ilp_refresh() { }
void ilp_msgbox(const char *t, char *s)  { }
void ilp_pbar()    { }
void ilp_label()   { }
int  ilp_choose(char *c, char *n) { return 0; }

int init_linkfile()
{
  if(ilc != NULL)
    free(ilc);
  ilc = (LinkCable *)malloc(sizeof(LinkCable));
  if(ilc == NULL)
    return ERR_MALLOC;

  ilc->init_port  = ilp_init_port;
  ilc->open_port  = ilp_open_port;
  ilc->put        = ilp_put;
  ilc->get        = ilp_get;
  ilc->close_port = ilp_close_port;
  ilc->term_port  = ilp_term_port;
  ilc->probe_port = ilp_probe_port;
  ilc->check_port = ilp_check_port;

  switch(ti68k_getCalcType())
    {
    case TI92: ticalc_set_calc(CALC_TI92, &itc, ilc);
      break;
    case TI89: ticalc_set_calc(CALC_TI89, &itc, ilc);
      break;
    case TI92 | MODULEPLUS: ticalc_set_calc(CALC_TI92P, &itc, ilc);
      break;
    default: return ERR_INTERNAL;
      break;
    }
  /*
    ticalc_set_update(&iu, ilp_start, ilp_stop, ilp_refresh, ilp_msgbox,
    ilp_pbar, ilp_label, ilp_choose);
  */

  return ERR_NONE;
}

int test_sendfile()
{
  FILE *f;

  speedyLink = 1;
  internalLink = 1;  
  printf("result: %i\n", ticalc_open_ti_file("/root/str.89s", "rt", &f));
  itc.send_var(f, 0);
  ticalc_close_ti_file(f);
  internalLink = 0;
  speedyLink = 0;

  return ERR_NONE;
}

int send_ti_file(char *filename)
{
  FILE *f;
  char *ext;

  /* Get extension */
  ext = strrchr(filename, '.');
  if(ext == NULL)
    return ERR_TI_FILE;
  else
    ext++;
  
  /* TI file ? */
  if(!(strstr(ext, "89") && (ti68k_getCalcType() == TI89)) ||
     (strstr(ext, "92") && (ti68k_getCalcType() == TI92))  ||
     (strstr(ext, "9x") && (ti68k_getCalcType() == TI92p)) ||
     (strstr(ext, "9X") && (ti68k_getCalcType() == TI92p)) )
    return ERR_TI_FILE;
  
  /* FLASH APP file ? */
  else if( (ext[2] == 'k') || (ext[2] =='K'))
    {
      speedyLink = 1;
      internalLink = 1;  
      ticalc_open_ti_file(filename, "rt", &f);
      itc.send_flash(f, MODE_APPS);
      ticalc_close_ti_file(f);
      internalLink = 0;
      speedyLink = 0;
    }

  /* FLASH OS file ? */
  else if( (ext[2] == 'u') || (ext[2] == 'U'))
    {
      speedyLink = 1;
      internalLink = 1;  
      ticalc_open_ti_file(filename, "rt", &f);
      itc.send_flash(f, MODE_AMS);
      ticalc_close_ti_file(f);
      internalLink = 0;
      speedyLink = 0;
    }
  
  /* Backup file ? */
  else if( (ext[2] == 'b') || (ext[2] == 'B'))
    {
      speedyLink = 1;
      internalLink = 1;  
      ticalc_open_ti_file(filename, "rt", &f);
      itc.send_backup(f, MODE_NORMAL);
      ticalc_close_ti_file(f);
      internalLink = 0;
      speedyLink = 0;
    }

  /* Group file ? */
  else if( (ext[2] == 'g') || (ext[2] == 'G'))
    {
      speedyLink = 1;
      internalLink = 1;  
      ticalc_open_ti_file(filename, "rt", &f);
      itc.send_var(f, MODE_NORMAL);
      ticalc_close_ti_file(f);
      internalLink = 0;
      speedyLink = 0;
    }

  /* Single file */
  else
    {
      speedyLink = 1;
      internalLink = 1;  
      ticalc_open_ti_file(filename, "rt", &f);
      itc.send_var(f, MODE_NORMAL);
      ticalc_close_ti_file(f);
      internalLink = 0;
      speedyLink = 0;
    }

  return ERR_NONE;
}




