/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/


#include "TeProxMatrixConstructionStrategy.h"
#include "TeDatabase.h"
#include "TeProgress.h"
#include "TeSTElementSet.h"

//////////////////////////////////////////////////////////////////////
// Auxiliary Functions for construction strategies
//////////////////////////////////////////////////////////////////////

bool
TeFindObjectCentroid (TeSTInstance& obj, TeGeomRep rep, TeCoord2D& p)
{		
		if(rep == TePOLYGONS)
		{
			TePolygonSet pols;
			if(obj.getGeometry (pols))
				p = TeFindCentroid (pols);
			else
				return false;
		}
		else if (rep == TeLINES)
		{
			TeLineSet lines;
			if(obj.getGeometry (lines))
				p = TeFindCentroid (lines);
			else
				return false;
		}
		else if (rep == TePOINTS)
		{
			TePointSet points;
			if(obj.getGeometry (points))
				p = TeFindCentroid (points);
			else
				return false;
		}
		else if (rep == TeCELLS)
		{
			TeCellSet cells;
			if(obj.getGeometry (cells))
				p = TeFindCentroid (cells);
			else
				return false;
		}
		return true;
}

bool
TeFindObjectsCentroid (TeSTElementSet* objects, const string& object_id, TeGeomRep rep, TeCoord2D& p)
{
	if(rep==TePOLYGONS)
	{
		TePolygonSet polygons;
		if(objects->getGeometry (object_id, polygons))
				p = TeFindCentroid (polygons);

	}
	else if (rep==TeLINES)
	{
		TeLineSet lines;
		if(objects->getGeometry (object_id, lines))
			p = TeFindCentroid (lines);		
	}
	else if (rep==TePOINTS)
	{
		TePointSet points;
		if(objects->getGeometry (object_id, points))
			p = TeFindCentroid (points);					

	}
	else if (rep==TeCELLS)
	{
		TeCellSet cells;
		if(objects->getGeometry (object_id, cells))
			p = TeFindCentroid (cells);
	}
	return true;
}


bool  TeSelectEntryPoints (TeNodeSet  entry_set, map<string, string> entry_geom_map, TeGraphNetwork* net,
						double max_dist, TeSTElementSet* objects, TeGeomRep rep, TeProxMatrixImplementation* imp)
{

	// Compute minimum path for all the new nodes/geometries near the network
	// and connect the ones close enough
	TeNodeSet::iterator it_node1 = entry_set.begin();
	while ( it_node1 != entry_set.end())
	{
		vector<double> minimum_path;
		if (net->minimumPath ((*it_node1), entry_set, minimum_path))
		{
		int j  = 0;
		// Check maximum connection distance and connect selected objects
		TeNodeSet::iterator it_node2 = entry_set.begin();
		while ( it_node2 != entry_set.end())
		{
			// Get ids to simplify the code
			string node1_id   = (*it_node1).objectId(); // Created Node id, nearest point to Geom1
			string node2_id   = (*it_node2).objectId();	// Created Node id, nearest point to Geom1		
			string object1_id = entry_geom_map[node1_id]; // Input Geom1 id
			string object2_id = entry_geom_map[node2_id]; // Input Geom2 id

			// Check identity and maximum allowed distance through the network
	  		if ((object1_id != object2_id) && (minimum_path[j] <= max_dist))
				{	
					// Compute attributes and connect objects
					TeProxMatrixAttributes attr;
					
					TeCoord2D p1, p2;
					if (!TeFindObjectsCentroid (objects, object1_id, rep, p1))
						return false;

					if (!TeFindObjectsCentroid (objects, object2_id, rep, p2))
						return false;
				
					// Local istance between input geometries
					attr.CentroidDistance (TeDistance (p1, p2));

					// Total distance from input geometries centroids to network
					attr.NetworkObjectsDistance (TeDistance (p1, (*it_node1).location()) + 
												 TeDistance (p2, (*it_node2).location()));	
				
					// Minimum path from the nodes relatives to the two geometries
					attr.NetworkMinimumPath (minimum_path[j]);  
			
					imp->connectObjects (object1_id, object2_id, attr); // for networks, only one direction is connected
				}
			it_node2++;
			j++;
		}
		}
		it_node1++;
	}
	return true;
}


bool  TeSelectEntryPoints2 (TeNodeSet    entry_set1,
						  TeNodeSet    entry_set2,
						map<string, string> entry_geom_map,
						TeGraphNetwork* net,
						double max_dist, 
						TeSTElementSet* objects1,
						TeGeomRep rep1,
						TeSTElementSet* objects2,
						TeGeomRep rep2,
						TeProxMatrixImplementation* imp)
{

	// Compute minimum path for all the new nodes/geometries near the network
	// and connect the ones close enough
	TeNodeSet::iterator it_node1 = entry_set1.begin();
	while ( it_node1 != entry_set1.end())
	{
		vector<double> minimum_path;
		net->minimumPath ((*it_node1), entry_set2, minimum_path); 
		if (minimum_path.size() > 0)
		{
			int j  = 0;
		// Check maximum connection distance and connect selected objects
			TeNodeSet::iterator it_node2 = entry_set2.begin();
			while ( it_node2 != entry_set2.end())
			{
			    // Get ids to simplify the code
				string node1_id   = (*it_node1).objectId(); // Created Node id, nearest point to Geom1
				string node2_id   = (*it_node2).objectId();	// Created Node id, nearest point to Geom1		
				string object1_id = entry_geom_map[node1_id]; // Input Geom1 id
				string object2_id = entry_geom_map[node2_id]; // Input Geom2 id
	
		
				// Check identity and maximum allowed distance through the network
	  			if ((object1_id != object2_id) && (minimum_path[j] <= max_dist))
				{
					
					// Compute attributes and connect objects
					TeProxMatrixAttributes attr;

					TeCoord2D p1, p2;
					if (!TeFindObjectsCentroid (objects1, object1_id, rep1, p1))
						return false;

					if (!TeFindObjectsCentroid (objects2, object2_id, rep2,  p2))
						return false;
		
					// Local distance
					attr.CentroidDistance (TeDistance (p1, p2));

					// Total distance from input geometries centroids to network
					attr.NetworkObjectsDistance (TeDistance (p1, (*it_node1).location()) + 
												 TeDistance (p2, (*it_node2).location()));	
				
					// Minimum path from the nodes relatives to the two geometries
					attr.NetworkMinimumPath (minimum_path[j]);  
			
					imp->connectObjects (object1_id, object2_id, attr); // for networks, only one direction is connected
				}
				it_node2++;
				j++;
			}
		}
		it_node1++;
	}
	return true;
}


//////////////////////////////////////////////////////////////////////
// TeProxMatrixConstructionStrategy 
//////////////////////////////////////////////////////////////////////

TeProxMatrixConstructionStrategy::TeProxMatrixConstructionStrategy() : 
	objects_(0), 
	geomRep_(TeGEOMETRYNONE),
	type_("")
	{ } 

TeProxMatrixConstructionStrategy::TeProxMatrixConstructionStrategy(TeSTElementSet*  objects, TeGeomRep geomRep, const string& type): 
	objects_(objects), 
	geomRep_(geomRep),
	type_(type)
	{ } 

TeProxMatrixConstructionStrategy::TeProxMatrixConstructionStrategy (const TeProxMatrixConstructionStrategy& st)
{
	objects_ = st.objects_; 
	geomRep_ = st.geomRep_;
	type_ = st.type_;
}

void 
TeProxMatrixConstructionStrategy::setSTObjects (TeSTElementSet*  objects, TeGeomRep  geomRep)
{	
	objects_ = objects; 
	geomRep_ = geomRep; 
}

TeSTElementSet* 
TeProxMatrixConstructionStrategy::objects() 
{ 
	return objects_; 
}

bool 
TeProxMatrixConstructionStrategy::IsEqual (const TeProxMatrixConstructionStrategy& other) const
{
	return ((type_==other.type_) && 
		((&objects_)==(&other.objects_)) && 
		(geomRep_ == other.geomRep_)); 
}


//////////////////////////////////////////////////////////////////////
// TeProxMatrixLocalAdjacencyStrategy 
//////////////////////////////////////////////////////////////////////

TeProxMatrixLocalAdjacencyStrategy::TeProxMatrixLocalAdjacencyStrategy (bool calcDistance) : 
		TeProxMatrixConstructionStrategy (0, TeGEOMETRYNONE, "LocalAdjacency"),
		calcDistance_(calcDistance)	
		{ }


TeProxMatrixLocalAdjacencyStrategy::TeProxMatrixLocalAdjacencyStrategy (TeSTElementSet*  objects, TeGeomRep	geomRep, bool calcDistance):
		TeProxMatrixConstructionStrategy (objects, geomRep, "LocalAdjacency"),
		calcDistance_(calcDistance)	
		{ }


TeProxMatrixLocalAdjacencyStrategy::TeProxMatrixLocalAdjacencyStrategy (TeProxMatrixLocalAdjacencyStrategy& st):
		TeProxMatrixConstructionStrategy (st)
		{
			calcDistance_ = st.calculateDistance();
		}

bool 
TeProxMatrixLocalAdjacencyStrategy::Construct (TeProxMatrixImplementation* imp)
{
	if (imp == 0) 
		return false;
	int n = 0;

	// Iterate over all selected objects, selecting their neighbours
	TeSTElementSet::iterator itobj1 = objects_->begin();
	
	TeDatabase* db = objects_->theme()->layer()->database();
	if(!db)
		return false;

	string		geomTable = objects_->theme()->layer()->tableName(geomRep_);
	if(geomTable.empty())
		return false;

	//verify if the theme has collection
	string collTable = ""; 
	if(db->tableExist(objects_->theme()->collectionTable()))
		collTable = objects_->theme()->collectionTable();

	// ----- progress bar
	int step = 0;
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(objects_->numElements());
	// -----

	TePrecision::instance().setPrecision(TeGetPrecision(objects_->theme()->layer()->projection()));
		
	while ( itobj1 != objects_->end())
	{
		// Get object1 id and representation
		string object_id1 = (*itobj1).objectId();
		n++;
			
		Keys IdsIn, IdsOut;
		IdsIn.push_back(object_id1);

		//spatial relation Touch
		if(!db->spatialRelation(geomTable, geomRep_, IdsIn, IdsOut, TeINTERSECTS, collTable))
		{
			++itobj1;
			continue;
		}

		for(unsigned int i=0; i<IdsOut.size(); i++)
		{
			string object_id2 = IdsOut[i];
		
			if ((object_id1 != object_id2) && (!imp->isConnected (object_id1,object_id2)) && 
				(objects_->hasElement(object_id2)))
			{
				TeProxMatrixAttributes attr;

				if(calcDistance_)
				{

					TeCoord2D p1, p2;
					if (!TeFindObjectsCentroid (objects_, object_id1, geomRep_, p1))
						return false;

					if (!TeFindObjectsCentroid (objects_, object_id2, geomRep_,  p2))
						return false;

					attr.CentroidDistance (TeDistance (p1, p2)); 
				}

				imp->connectObjects (object_id2, object_id1, attr);
				imp->connectObjects (object_id1, object_id2, attr);
			}

		}
		++itobj1;


		if(TeProgress::instance())
		{
			if (TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			else
				TeProgress::instance()->setProgress(step);
		}	
		++step;
	}

	if (TeProgress::instance())
		TeProgress::instance()->reset();
	return true; 
}


TeProxMatrixLocalAdjacencyStrategy& 
TeProxMatrixLocalAdjacencyStrategy::operator= (const TeProxMatrixLocalAdjacencyStrategy& rhs)
{
	if ( this != &rhs )
	{
		type_ = rhs.type_;
		objects_ = rhs.objects_;
		geomRep_ = rhs.geomRep_;
	}
	return *this;
}

bool 
TeProxMatrixLocalAdjacencyStrategy::operator== (const TeProxMatrixLocalAdjacencyStrategy& rhs) const   
{ 
	return (TeProxMatrixConstructionStrategy::IsEqual(rhs)); 
}
	

//////////////////////////////////////////////////////////////////////
// TeProxMatrixLocalDistanceStrategy 
//////////////////////////////////////////////////////////////////////

TeProxMatrixLocalDistanceStrategy::TeProxMatrixLocalDistanceStrategy (double max_distance):
		TeProxMatrixConstructionStrategy (0, TeGEOMETRYNONE, "LocalDistance"), 
		max_distance_ (max_distance)
		{ }


TeProxMatrixLocalDistanceStrategy::TeProxMatrixLocalDistanceStrategy (TeSTElementSet*  objects, TeGeomRep geomRep, double max_distance): 
		TeProxMatrixConstructionStrategy(objects, geomRep, "LocalDistance"),
		max_distance_ (max_distance) 
		{ }

	
TeProxMatrixLocalDistanceStrategy::TeProxMatrixLocalDistanceStrategy (const TeProxMatrixLocalDistanceStrategy& st): 
		TeProxMatrixConstructionStrategy(st)
{
	max_distance_ = st.max_distance_;
} 

bool 
TeProxMatrixLocalDistanceStrategy::Construct (TeProxMatrixImplementation* imp)
{
	if (imp == 0) 
		return false;

	TeDatabase* db = objects_->theme()->layer()->database();
	if(!db)
		return false;

	string geomTable =  objects_->theme()->layer()->tableName(geomRep_);
	if(geomTable.empty())
		return true;
	
	//verify if the theme has restriction
	string collTable = "";
	if(objects_->theme()->hasRestriction())
		collTable = objects_->theme()->collectionTable();
	
	// ----- progress bar
	int step = 0;
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(objects_->numElements());
	// -----
	
	TeSTElementSet::iterator itobj1 = objects_->begin();
	while ( itobj1 != objects_->end())
	{
		// Get object1 id and representation
		string object_id1 = (*itobj1).objectId();
		
		TeCoord2D p1; 
		if (!TeFindObjectCentroid ((*itobj1), geomRep_, p1))
			return false;
	
		KeysToDist result;
		
		//within distance
		if(!db->withinDistance(geomTable, geomRep_, p1, result, max_distance_, collTable))
		{
			++itobj1;
			continue;
		}

		KeysToDist::iterator it = result.begin();
		while(it != result.end())
		{
			string object_id2 = it->first;

			if ((object_id1 != object_id2) && (!imp->isConnected (object_id1,object_id2)) && 
				(objects_->hasElement(object_id2)))
			{
				TeProxMatrixAttributes attr;
				attr.CentroidDistance (it->second);
				imp->connectObjects (object_id1, object_id2, attr);
				imp->connectObjects (object_id2, object_id1, attr);
			}

			++it;
		}

		if(TeProgress::instance())
		{

			if (TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			else
				TeProgress::instance()->setProgress(step);
		}	
		++step;
		++itobj1;
	}

	if (TeProgress::instance())
		TeProgress::instance()->reset();
	return true;
}

	
bool 
TeProxMatrixLocalDistanceStrategy::IsEqual (const TeProxMatrixConstructionStrategy& other) const
{
	return ((*this) == (TeProxMatrixLocalDistanceStrategy&) other); 
}

bool 
TeProxMatrixLocalDistanceStrategy::operator== (const TeProxMatrixLocalDistanceStrategy& s) const 
{
	return ( TeProxMatrixConstructionStrategy::IsEqual(s) && 
			(max_distance_ == s.max_distance_)); 
}
	
TeProxMatrixLocalDistanceStrategy& 
TeProxMatrixLocalDistanceStrategy::operator= (const TeProxMatrixLocalDistanceStrategy& rhs)
{
	if ( this != &rhs )
	{
		type_ = rhs.type_;
		objects_ = rhs.objects_;
		geomRep_ = rhs.geomRep_;
		max_distance_ = rhs.max_distance_;
	}
	return *this;
}

//////////////////////////////////////////////////////////////////////
// TeProxMatrixClosedNetworkStrategy 
//////////////////////////////////////////////////////////////////////

TeProxMatrixClosedNetworkStrategy::TeProxMatrixClosedNetworkStrategy (TeSTElementSet*  objects,
										TeGeomRep rep, double max_local_distance,
										double max_net_distance, double  max_connection_distance, 
										TeGraphNetwork* input_net) : 	
	TeProxMatrixConstructionStrategy(objects, rep, "ClosedNetwork"),
	max_local_distance_ (max_local_distance), 
	max_net_distance_ (max_net_distance), 
	max_connection_distance_ (max_connection_distance), 
	net_ (input_net) 
	{ }



TeProxMatrixClosedNetworkStrategy::TeProxMatrixClosedNetworkStrategy (const TeProxMatrixClosedNetworkStrategy& rhs) :
	TeProxMatrixConstructionStrategy(rhs)
{
		max_local_distance_ = rhs.max_local_distance_;
		max_net_distance_ = rhs.max_net_distance_;
		max_connection_distance_ = rhs.max_connection_distance_;
		
		TeGraphNetwork* n = new TeGraphNetwork();
		*n = *rhs.net_; 
		net_ = n;
}

	  
bool 
TeProxMatrixClosedNetworkStrategy::Construct (TeProxMatrixImplementation* imp)
{
	if (imp == 0) return false;

	// Connect local neighbours, based on the Local Distance Strategy

	TeProxMatrixLocalDistanceStrategy local(objects_, geomRep_, max_local_distance_);
	local.Construct (imp);

	// Connect neighbours through the network. The process is the following:
	// 1. The nearest node (entry points) in the network for all the input geometries
	//    are computed;
	// 2. If the distance from the geometry centroid to the entry point is smaller than the maximum allowed, 
	//    create a new node in the network corresponding to this entry point.
	// 3. Compute the minimum path among all these new nodes (entry points).
	// 4. For each pair of entry points, if the minimum path distance is smaller than the
	//    maximum allowed, the corresponding geometries will be connected.

	if (net_ == 0) return false;

	map<string, string> entry_geom_map; // maps input geometries to network entry points
	TeNodeSet entry_set; // entry points

	// ----- progress bar
	int step = 0;
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(objects_->numElements());
	// -----

	// Iterate over all objects and select the ones that are close enough to the network
	TeSTElementSet::iterator itobj1 = objects_->begin();
	while ( itobj1 != objects_->end())
	{
		// Get object1 id and representation
		string object_id1 = (*itobj1).objectId();
		TeCoord2D p1, p2; 
			

		int i = 0;
		double min_distance = 0.;
	
		net_->nearestNodePoint (p1, i, p2, min_distance);
				
		TeNode node;
		if ((min_distance <= max_net_distance_) && net_->getNode(i, node))
		{
			entry_geom_map[node.objectId()] = object_id1; // Associates geometry with closest net nodes
			entry_set.add (node);  // This set will be used as initial points in the shortest path algorithm
		}

		if(TeProgress::instance())
		{
			if (TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			else
				TeProgress::instance()->setProgress(step);
		}
		++step;
		++itobj1;
	}

	TeSelectEntryPoints (entry_set, entry_geom_map, net_, max_connection_distance_, objects_, geomRep_, imp);
	if (TeProgress::instance())
		TeProgress::instance()->reset();
		
	return true;
}


bool 
TeProxMatrixClosedNetworkStrategy::IsEqual (const TeProxMatrixConstructionStrategy& other) const
{
	return (*this == (TeProxMatrixClosedNetworkStrategy&) other); 
}
	
TeProxMatrixClosedNetworkStrategy& 
TeProxMatrixClosedNetworkStrategy::operator= (const TeProxMatrixClosedNetworkStrategy& rhs)
{
	if (!(this==&rhs))
	{
		type_ = rhs.type_;
		objects_ = rhs.objects_;
		geomRep_ = rhs.geomRep_;
		max_local_distance_ = rhs.max_local_distance_;
		max_net_distance_ = rhs.max_net_distance_;
		max_connection_distance_ = rhs.max_connection_distance_;
	
		TeGraphNetwork* n = new TeGraphNetwork(); 
		*n = *rhs.net_; 
		net_ = n;       
	}
	return (*this);
}

	
bool 
TeProxMatrixClosedNetworkStrategy::operator== (const TeProxMatrixClosedNetworkStrategy& other) const 
{
	return ( TeProxMatrixConstructionStrategy::IsEqual(other) &&
			(max_local_distance_ == other.max_local_distance_) &&
			(max_net_distance_ == other.max_net_distance_) &&
			(max_connection_distance_ == other.max_connection_distance_));
}
	   
//////////////////////////////////////////////////////////////////////
// TeProxMatrixOpenNetworkStrategy 
//////////////////////////////////////////////////////////////////////

TeProxMatrixOpenNetworkStrategy::TeProxMatrixOpenNetworkStrategy (TeSTElementSet*  objects, TeGeomRep rep, 
										double max_local_distance, 
										double	max_net_distance, double  max_connetion_distance, 
										TeGraphNetwork* input_net) : 	
	TeProxMatrixConstructionStrategy(objects, rep, "OpenNetwork"),
	max_local_distance_ (max_local_distance), 
	max_net_distance_ (max_net_distance), 
	max_connection_distance_ (max_connetion_distance), 
	net_ (input_net)
	{ }


TeProxMatrixOpenNetworkStrategy::TeProxMatrixOpenNetworkStrategy (const TeProxMatrixOpenNetworkStrategy& rhs): 
	TeProxMatrixConstructionStrategy(rhs)
{
	max_local_distance_ = rhs.max_local_distance_;
	max_net_distance_ = rhs.max_net_distance_;
	max_connection_distance_ = rhs.max_connection_distance_;
	
	TeGraphNetwork* n = new TeGraphNetwork();
	*n = *rhs.net_; 
	net_ = n;
}

bool 
TeProxMatrixOpenNetworkStrategy::Construct (TeProxMatrixImplementation* imp)
{
	if (imp == 0) 
		return false;

	// Connect local neighbours, based on the Local Distance Strategy
	if (max_local_distance_ > 0)
	{
		TeProxMatrixLocalDistanceStrategy local(objects_, geomRep_, max_local_distance_);
		local.Construct (imp);
	}

	// Connect neighbours through the network. The process is the following:
	// 1. The nearest point (entry points) in the network for all the input geometries
	//    are computed;
	// 2. If the distance from the geometry centroid to the entry point is smaller than the maximum allowed, 
	//    create a new node in the network corresponding to this entry point.
	// 3. Compute the minimum path among all these new nodes (entry points).
	// 4. For each pair of entry points, if the minimum path distance is smaller than the
	//    maximum allowed, the corresponding geometries will be connected.
	if (net_ == 0) 
		return false;

	map<string, string> entry_geom_map; // maps input geometries to network entry points
	TeNodeSet entry_set; // entry points

//	// ----- progress bar
//	int step = 0;
//	if(TeProgress::instance())

//		TeProgress::instance()->setTotalSteps(objects_->numObjects());
	// -----

	// Iterate over all objects and select the ones that are close enough to the network
	TeSTElementSet::iterator itobj1 = objects_->begin();
	int id = 0;  // Nodes to be created will have sequential object_ids. 
				 // The existing network nodes have object_id equal to their position
	while ( itobj1 != objects_->end())

	{
		// Get object1 id and representation
		string object_id1 = (*itobj1).objectId();
		cout << object_id1 << "id " << id << endl;

		TeCoord2D p1; 
		if (!TeFindObjectCentroid ((*itobj1), geomRep_, p1))
			return false;


		int line_index;
		TeCoord2D pinter;
		double min_dist;

		if (net_->nearestNetworkPoint (p1, line_index, pinter, min_dist))
		{
			if (min_dist < max_net_distance_)
			{
				TeNode new_node;
				new_node.add (pinter);
				new_node.objectId (Te2String (id));
				id++;

				net_->breakLineSet (new_node, line_index);
				entry_geom_map[new_node.objectId()] = object_id1; // Associates geometry with closest net nodes
				entry_set.add (new_node);  // This set will be used as initial points in the shortest path algorithm
			}
		}
		++itobj1;
/*		if(TeProgress::instance())
		{
			if (TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			else
				TeProgress::instance()->setProgress(step);
		}	
		++step;
		*/
	
	}

    TeLineSet lineSet = net_->getLineSet();
    map<string, double> lineCosts =  net_->getLineCosts();
	TeGraphNetwork net(lineSet, lineCosts);
	TeSelectEntryPoints (entry_set, entry_geom_map, &net, max_connection_distance_, objects_, geomRep_, imp);
	if (TeProgress::instance())
		TeProgress::instance()->reset();
	return true;
}


bool 
TeProxMatrixOpenNetworkStrategy::IsEqual (const TeProxMatrixConstructionStrategy& other) const
{
	return ((*this)==(TeProxMatrixOpenNetworkStrategy&)other); 
}

bool 
TeProxMatrixOpenNetworkStrategy::operator== (const TeProxMatrixOpenNetworkStrategy& other) const 
{
	return (TeProxMatrixConstructionStrategy::IsEqual(other) &&
		(max_local_distance_ == other.max_local_distance_) &&
		(max_net_distance_ == other.max_net_distance_) &&
		(max_connection_distance_ == other.max_connection_distance_));
}


TeProxMatrixOpenNetworkStrategy& 
TeProxMatrixOpenNetworkStrategy::operator= (const TeProxMatrixOpenNetworkStrategy& rhs)
{
	if (!(this==&rhs))
	{
		type_ = rhs.type_;
		objects_ = rhs.objects_;
		geomRep_ = rhs.geomRep_;
		max_local_distance_ = rhs.max_local_distance_;
		max_net_distance_ = rhs.max_net_distance_;
		max_connection_distance_ = rhs.max_connection_distance_;
	
		TeGraphNetwork* n = new TeGraphNetwork(); 
		*n = *rhs.net_;                           
		net_ = n;								
	}
	return (*this);
}


/////////////////////////////////////////////////////////////////////
// TeProxMatrixOpenNetworkStrategy2 
//////////////////////////////////////////////////////////////////////

TeProxMatrixOpenNetworkStrategy2::TeProxMatrixOpenNetworkStrategy2 (TeSTElementSet*  objects1, 
										TeGeomRep rep1,
										TeSTElementSet*  objects2, TeGeomRep rep2,
										double max_local_distance, 
										double	max_net_distance, double  max_connetion_distance, 
										TeGraphNetwork* input_net) : 	
	TeProxMatrixConstructionStrategy(objects1, rep1, "OpenNetwork2"),
	objects2_ (objects2), 
	geomRep2_(rep2),
	max_local_distance_ (max_local_distance), 
	max_net_distance_ (max_net_distance), 
	max_connection_distance_ (max_connetion_distance), 
	net_ (input_net)
	{ }  

TeProxMatrixOpenNetworkStrategy2::TeProxMatrixOpenNetworkStrategy2 (const TeProxMatrixOpenNetworkStrategy2& rhs)
                    : TeProxMatrixConstructionStrategy(rhs)
{
	objects2_ = rhs.objects2_;
	geomRep2_ = rhs.geomRep2_;
	max_local_distance_ = rhs.max_local_distance_;
	max_net_distance_ = rhs.max_net_distance_;
	max_connection_distance_ = rhs.max_connection_distance_;
	
	TeGraphNetwork* n = new TeGraphNetwork();
	*n = *rhs.net_; 
	net_ = n;
}


bool 
TeProxMatrixOpenNetworkStrategy2:: Construct (TeProxMatrixImplementation* imp)
	{
		if (imp == 0) 
			return false;


		// Connect neighbours through the network. The process is the following:
		// 1. The nearest point (entry points) in the network for all the input geometries
		//    are computed;
		// 2. If the distance from the geometry centroid to the entry point is smaller than the maximum allowed, 
		//    create a new node in the network corresponding to this entry point.
		// 3. Compute the minimum path among all these new nodes (entry points).
		// 4. For each pair of entry points, if the minimum path distance is smaller than the
		//    maximum allowed, the corresponding geometries will be connected.
		if (net_ == 0) 
			return false;

		map<string, string> entry_geom_map; // maps input geometries to network entry points

	
		int id = 0;  // Nodes to be created will have sequential object_ids. 
					 // The existing network nodes have object_id equal to their position
		// Iterate over all objects and select the ones that are close enough to the network

		TeNodeSet entry_set1; // entry points
		TeSTElementSet::iterator itobj1 = objects_->begin();	
		while ( itobj1 != objects_->end())
		{
			// Get object1 id and representation
			string object_id1 = (*itobj1).objectId();
					cout << "object1 " << object_id1 << "id" << id << endl;

			TeCoord2D p1; 
			if (!TeFindObjectCentroid ((*itobj1), geomRep_, p1))
				return false;
		

			int line_index;
			TeCoord2D pinter;
			double min_dist;

			if (net_->nearestNetworkPoint (p1, line_index, pinter, min_dist))
			{
				if (min_dist < max_net_distance_)
				{
					TeNode new_node;
					new_node.add (pinter);
					new_node.objectId (Te2String (id));
					id++;

					net_->breakLineSet (new_node, line_index);
					entry_geom_map[new_node.objectId()] = object_id1; // Associates geometry with closest net nodes
					entry_set1.add (new_node);  // This set will be used as initial points in the shortest path algorithm
				}
			}
			++itobj1;
		}


	// Iterate over all objects of interst (objects2) 
		TeNodeSet entry_set2; // entry points
		TeSTElementSet::iterator itobj2 = objects2_->begin();		
		while ( itobj2 != objects2_->end())
		{
			// Get object1 id and representation
			string object_id2 = (*itobj2).objectId();
					cout << "object2 " << object_id2 << "id" << id << endl;

			TeCoord2D p2; 
			if (!TeFindObjectCentroid ((*itobj2), geomRep2_, p2))
				return false;
		
			int line_index;
			TeCoord2D pinter;
			double min_dist;

			if (net_->nearestNetworkPoint (p2, line_index, pinter, min_dist))
			{ 
				TeNode new_node;
				new_node.add (pinter);
				new_node.objectId (Te2String (id));
				id++;

				net_->breakLineSet (new_node, line_index);
				entry_geom_map[new_node.objectId()] = object_id2; // Associates geometry with closest net nodes
				entry_set2.add (new_node);  // This set will be used as initial points in the shortest path algorithm
			}
			++itobj2;
		}

        TeLineSet lineSet = net_->getLineSet();
        map<string, double> lineCosts = net_->getLineCosts();
		TeGraphNetwork net(lineSet, lineCosts);
		TeSelectEntryPoints2 (entry_set1, entry_set2, entry_geom_map, &net, max_connection_distance_, 
			objects_, geomRep_, objects2_, geomRep2_, imp);
			return true;
	}

bool 
TeProxMatrixOpenNetworkStrategy2::IsEqual (const TeProxMatrixConstructionStrategy& other) const
{
	return ((*this)==(TeProxMatrixOpenNetworkStrategy2&)other); 
}

bool 
TeProxMatrixOpenNetworkStrategy2:: operator== (const TeProxMatrixOpenNetworkStrategy2& other) const 
{

	return ((type_==other.type_) &&
		(max_local_distance_ == other.max_local_distance_) &&
		(max_net_distance_ == other.max_net_distance_) &&
		(max_connection_distance_ == other.max_connection_distance_));
}

TeProxMatrixOpenNetworkStrategy2& 
TeProxMatrixOpenNetworkStrategy2:: operator= (const TeProxMatrixOpenNetworkStrategy2& rhs)
{
		if (!(this==&rhs))
		{
			type_ = rhs.type_;
			objects_ = rhs.objects_;
			objects2_ = rhs.objects2_;
	
			max_local_distance_ = rhs.max_local_distance_;
			max_net_distance_ = rhs.max_net_distance_;
			max_connection_distance_ = rhs.max_connection_distance_;
		
			TeGraphNetwork* n = new TeGraphNetwork(); 
			*n = *rhs.net_;  
			net_ = n;       
		}
		return (*this);
}


