/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeOverlay.h"
#include "TeIntersector.h"
#include "TeGeometryAlgorithms.h"
#include "TeFragmentation.h"

#include <map>
#include <vector>
#include <algorithm>

using namespace std;


//---------------- Auxiliary structure ----------------//
struct xyOrder
{
	bool operator()(const TeCoord2D& c1, const TeCoord2D& c2) const
	{
		if(TeGeometryAlgorithmsPrecision::IsGreater(c2.x_, c1.x_))
			return true;

		if(TeGeometryAlgorithmsPrecision::IsGreater(c1.x_, c2.x_))
			return false;

		if(TeGeometryAlgorithmsPrecision::IsGreater(c2.y_, c1.y_))
			return true;		

		return false;
	}
};

typedef multimap<TeCoord2D, pair<unsigned int, TeLine2D>, xyOrder> TeLineIndex;



inline void TeRemoveDuplicatedSegments(TeLine2D& l)
{
	bool hasDuplicated = false;

	for(unsigned int i = 1; i < l.size() - 1; ++i)
		if(TeEquals(l[i - 1], l[i + 1]))
		{
			l.erase(i + 1);
			hasDuplicated = true;
		}

	if(hasDuplicated)
		TeRemoveDuplicatedCoordinates(l);

	return;
}

void TeClonePolygonSet(const TePolygonSet& pSetIn, TePolygonSet& pSetOut)
{
	for(unsigned int i = 0; i < pSetIn.size(); ++i)
	{
		TePolygon p;
		for(unsigned int j = 0; j < pSetIn[i].size(); ++j)
		{
			TeLine2D l;

			for(unsigned int k = 0; k < pSetIn[i][j].size(); ++k)
				l.add(pSetIn[i][j][k]);

			TeRemoveDuplicatedCoordinates(l);

			TeRemoveDuplicatedSegments(l);

			TeLinearRing r(l);

			p.add(r);		
		}

		pSetOut.add(p);
	}
}

void TeCloneLineSet(const TeLineSet& lSetIn, TeLineSet& lSetOut)
{
	for(unsigned int i = 0; i < lSetIn.size(); ++i)
	{
		TeLine2D l;

		for(unsigned int j = 0; j < lSetIn[i].size(); ++j)
			l.add(lSetIn[i][j]);

		TeRemoveDuplicatedCoordinates(l);

		lSetOut.add(l);
	}
}


// Verifica qual operao a realizar e j coloca as fronteiras dos polgonos na ordem correta e caso no seja necessrio operao, ele j retorna o resultado
TePolygonSet TeChooseOperation(const short& operation, TePolygonSet& redPols, TePolygonSet& bluePols, short& locationRedFragments, short& locationBlueFragments)
{
	TePolygonSet result;

	unsigned int i = 0;

	switch(operation)
	{							
		// Intersection gets all fragments from red polygon wich are
		// inside the blue polygon and the blue fragments that are inside
		// the red polygon.
		case TeINTERSECTION:  if(TeDisjoint(redPols.box(), bluePols.box()))
								  return result;

							  // Need to reverse orientation
							  for(i = 0; i < bluePols.size(); ++i)
							  {
								  if(TeOrientation(bluePols[i][0]) == TeCOUNTERCLOCKWISE)
									  reverse(bluePols[i][0].begin(), bluePols[i][0].end());

								  for(unsigned int j = 1; j < bluePols[i].size(); ++j)
								  {
									  if(TeOrientation(bluePols[i][j]) == TeCLOCKWISE)
										  reverse(bluePols[i][j].begin(), bluePols[i][j].end());
								  }
							  }

							  // Need to reverse orientation
							  for(i = 0; i < redPols.size(); ++i)
							  {
								  if(TeOrientation(redPols[i][0]) == TeCOUNTERCLOCKWISE)
									  reverse(redPols[i][0].begin(), redPols[i][0].end());

								  for(unsigned int j = 1; j < redPols[i].size(); ++j)
								  {
									  if(TeOrientation(redPols[i][j]) == TeCLOCKWISE)
										  reverse(redPols[i][j].begin(), redPols[i][j].end());
								  }
							  }
			
			                  locationRedFragments  = TeINSIDE;
			                  locationBlueFragments = TeINSIDE;

							  break;

		// Union gets all fragments from red polygon wich are
		// outside the blue polygon and the blue fragments that are outside
		// the red polygon.
		case TeUNION:         if(TeDisjoint(redPols.box(), bluePols.box()))
							  {
								  result = redPols;

								  for(i = 0; i < bluePols.size(); ++i)
									  result.add(bluePols[i]);

								  return result;
							  }

							  // Need to reverse orientation
							  for(i = 0; i < bluePols.size(); ++i)
							  {
								  if(TeOrientation(bluePols[i][0]) == TeCOUNTERCLOCKWISE)
									  reverse(bluePols[i][0].begin(), bluePols[i][0].end());

								  for(unsigned int j = 1; j < bluePols[i].size(); ++j)
								  {
									  if(TeOrientation(bluePols[i][j]) == TeCLOCKWISE)
										  reverse(bluePols[i][j].begin(), bluePols[i][j].end());
								  }
							  }

							  // Need to reverse orientation
							  for(i = 0; i < redPols.size(); ++i)
							  {
								  if(TeOrientation(redPols[i][0]) == TeCOUNTERCLOCKWISE)
									  reverse(redPols[i][0].begin(), redPols[i][0].end());

								  for(unsigned int j = 1; j < redPols[i].size(); ++j)
								  {
									  if(TeOrientation(redPols[i][j]) == TeCLOCKWISE)
										  reverse(redPols[i][j].begin(), redPols[i][j].end());
								  }
							  }

							  locationRedFragments  = TeOUTSIDE;
			                  locationBlueFragments = TeOUTSIDE;

							  break;

		// Difference gets all fragments from red polygon wich are
		// outside the blue polygon and the blue fragments that are inside
		// the red polygon.
		case TeDIFFERENCE:	  if(TeDisjoint(redPols.box(), bluePols.box()))
								return redPols;
			
						      locationRedFragments  = TeOUTSIDE;
			                  locationBlueFragments = TeINSIDE;

							  // Need to reverse orientation
							  for(i = 0; i < bluePols.size(); ++i)
							  {
								  if(TeOrientation(bluePols[i][0]) == TeCLOCKWISE)
									  reverse(bluePols[i][0].begin(), bluePols[i][0].end());

								  for(unsigned int j = 1; j < bluePols[i].size(); ++j)
								  {
									  if(TeOrientation(bluePols[i][j]) == TeCOUNTERCLOCKWISE)
										  reverse(bluePols[i][j].begin(), bluePols[i][j].end());
								  }
							  }

							  // Need to reverse orientation
							  for(i = 0; i < redPols.size(); ++i)
							  {
								  if(TeOrientation(redPols[i][0]) == TeCOUNTERCLOCKWISE)
									  reverse(redPols[i][0].begin(), redPols[i][0].end());

								  for(unsigned int j = 1; j < redPols[i].size(); ++j)
								  {
									  if(TeOrientation(redPols[i][j]) == TeCLOCKWISE)
										  reverse(redPols[i][j].begin(), redPols[i][j].end());
								  }
							  }							  

		                      break;
	}

	return result;
}

TeMultiGeometry TeChooseOperation(const short& operation, TeLineSet& redLines, TePolygonSet& bluePols, short& locationRedFragments)
{
	TeMultiGeometry result;

	switch(operation)
	{							
		// Intersection gets all fragments from red polygon wich are
		// inside the blue polygon and the blue fragments that are inside
		// the red polygon.
		case TeINTERSECTION:  if(TeDisjoint(redLines.box(), bluePols.box()))
								  return result;
			
			                  locationRedFragments  = TeINSIDE;
							  break;

		// Union gets all fragments from red polygon wich are
		// outside the blue polygon and the blue fragments that are outside
		// the red polygon.
		case TeUNION:         if(TeDisjoint(redLines.box(), bluePols.box()))
							  {
								  result.setGeometry(redLines);
								  result.setGeometry(bluePols);

								  return result;
							  }

							  locationRedFragments  = TeOUTSIDE;

							  break;

		// Difference gets all fragments from red polygon wich are
		// outside the blue polygon and the blue fragments that are inside
		// the red polygon.
		case TeDIFFERENCE:	  if(TeDisjoint(redLines.box(), bluePols.box()))
							  {
								  result.setGeometry(redLines);
								  return result;
							  }
			
						      locationRedFragments  = TeOUTSIDE;

		                      break;
	}

	return result;
} 

void TeRemoveOpositeBoundaryFragments(TeLineIndex& redFragmentsIndex, TeLineIndex& blueFragmentsIndex)
{
	TeLineIndex::iterator indexIterator = blueFragmentsIndex.begin();
	
	while(indexIterator != blueFragmentsIndex.end())
	{
		TeLineIndex::iterator idx = redFragmentsIndex.find(indexIterator->second.second[(indexIterator->second.second.size() - 1u)]);

		if(idx != redFragmentsIndex.end())
		{
			if(TeEquals(idx->second.second[idx->second.second.size() - 1u], indexIterator->second.second[0u]))
			{
				redFragmentsIndex.erase(idx);

				TeLineIndex::iterator idxAux;
				idxAux = indexIterator;
				++indexIterator;
				blueFragmentsIndex.erase(idxAux);

				continue;
			}

		}
					

		++indexIterator;
	}
}

void TeRemoveSameBoundaryFragments(TeLineIndex& redFragmentsIndex, TeLineIndex& blueFragmentsIndex)
{
	TeLineIndex::iterator indexIterator = blueFragmentsIndex.begin();
	
	while(indexIterator != blueFragmentsIndex.end())
	{
		TeLineIndex::iterator idx = redFragmentsIndex.find(indexIterator->second.second[0u]);

		if(idx != redFragmentsIndex.end())
		{
			if(TeEquals(idx->second.second[idx->second.second.size() - 1u], indexIterator->second.second[(indexIterator->second.second.size() - 1u)]))
			{
				redFragmentsIndex.erase(idx);
				continue;
			}

		}					

		++indexIterator;
	}
}

void TeMergeFragments(TeLineIndex& fragmentsIndex, TeLineIndex& boundaryFragmentsIndex)
{
	TeLineIndex::iterator indexIterator = boundaryFragmentsIndex.begin();

	while(indexIterator != boundaryFragmentsIndex.end())
	{
		fragmentsIndex.insert(*indexIterator);
		++indexIterator;
	}

	boundaryFragmentsIndex.clear();	
}

void TeFindAndMoveClosedRings(TeLineIndex& fragmentsIndex, vector<TeLinearRing>& rings)
{
	TeLineIndex::iterator indexIterator = fragmentsIndex.begin();

	while(indexIterator != fragmentsIndex.end())
	{
		if(indexIterator->second.second.isRing())
		{
			rings.push_back(indexIterator->second.second);

			TeLineIndex::iterator idxAux;
			idxAux = indexIterator;
			++indexIterator;
			fragmentsIndex.erase(idxAux);
		}
		else
			++indexIterator;
	}
}

// Recebe um polygonset com anis externos e um vetor de polgonos que seriam buracos e
// define em qual polgono colocar os buracos
bool TeMountTopology(TePolygonSet& polysOut, vector<TeLinearRing>& holes)
{
	bool returnValue = true;

	if((polysOut.size() == 0) && (holes.size() > 0))
	{
		return false;	// Formou buracos e no formou os anis externos
	}

	if(polysOut.size() == 1)
	{
		for(unsigned int i = 0; i < holes.size(); ++i)
			polysOut[0].add(holes[i]);		
	}
	else
	{
		for(unsigned int i = 0; i < holes.size(); ++i)
		{
			TeLinearRing ring = holes[i];

			vector<TePolygon> candidates;
			vector<unsigned int> candidatesPos;

			unsigned int j = 0;

			for(j = 0; j < polysOut.size(); ++j)
			{
				if(TeWithinOrCoveredByOrEquals(ring.box(), polysOut[j].box()))
				{
					candidates.push_back(polysOut[j]);
					candidatesPos.push_back(j);
				}
			}

			if(candidates.size() == 1)
			{
				candidates[0].add(ring);
				continue;
			}

			vector<TePolygon> newCandidates;

			for(j = 0; j < candidates.size(); ++j)
			{
				short rel = TeBOUNDARY;

				unsigned int nthVert = ring.size();
				unsigned int iVert = 0u;

				while(iVert < nthVert)
				{
					rel = TeRelation(ring[iVert], candidates[j][0]);

					if(rel & TeINSIDE)
					{				
						newCandidates.push_back(candidates[j]);
						break;
					}
					else if(rel & TeOUTSIDE)
					{
						break;
					}
				
					++iVert;
				}

				if(iVert == nthVert)	
				{
					// Erro topolgico, todos os pontos esto na fronteira do anel externo...
					returnValue = false;

					TePolygon topTest;
					topTest.add(ring);

					short whatRel = TeRelation(topTest, candidates[j]);

					// Se um buraco for igual ao exterior, existe um erro topolgico
					// No momento, eliminamos o anel externo
					// e o interno... 
					// Se o buraco for coberto pelo polgono externo, ento ele ir ficar dentro deste
					// Caso contrrio  erro sem possibilidades...
					if(whatRel & TeEQUALS)
					{						
						polysOut.erase(candidatesPos[j]);
						continue;
					}
				}				
			}

			if(newCandidates.size() <= 0)
			{
				// No encontrou o anel externo ao qual o buraco pertence
				returnValue = false;
				continue;
			}

			int idxMinArea = 0;
			
			double minArea = TeMAXFLOAT;

			for(j = 0; j < newCandidates.size(); ++j)
			{
				if(TeGeometryArea(newCandidates[j].box()) < minArea)
				{
					idxMinArea = j;
					minArea = TeGeometryArea(newCandidates[j].box());
				}
			}

			newCandidates[idxMinArea].add(ring);
		}
	}

	return returnValue;
}

// Set operation for rings
bool TeOVERLAY::TeOverlay(const TePolygonSet& redPolsIn, const TePolygonSet& bluePolsIn, TePolygonSet& polsOut, const short& operation)
{
	polsOut.clear();

	short locationRedFragments = 0;		// Wich red fragments to consider.	
	short locationBlueFragments = 0;	// Wich blue fragments to consider.

	short redMask  = TeUNKNOWNPOSITION;	
	short blueMask = TeUNKNOWNPOSITION;	
	short location = TeUNKNOWNPOSITION;

	TePolygonSet redPols;
	TePolygonSet bluePols;

	// Copia o contedo dos polgonos de entrada
	TeClonePolygonSet(redPolsIn, redPols);
	TeClonePolygonSet(bluePolsIn, bluePols);

	// Ajeita a orientao das fronteiras dos polgonos e determina a localizao dos fragmentos que sero utilizados
	polsOut = TeChooseOperation(operation, redPols, bluePols, locationRedFragments, locationBlueFragments);

	if(polsOut.size() > 0)
		return true;


	// If we are here so the rings are not disjoint by box

	// Gets intersection list
	TeINTERSECTOR2::TeVectorBoundaryIP report;

	TeINTERSECTOR2::TeSafeIntersections(redPols, bluePols, report);

	// So we need to create the fragments
	TeLineIndex redFragmentsIndex;
	TeLineIndex blueFragmentsIndex;	

	TeLineIndex redBoundaryFragmentsIndex;
	TeLineIndex blueBoundaryFragmentsIndex;	


	TeLineSet redRings;
	for(unsigned int i = 0; i < redPols.size(); ++i)
		for(unsigned int j = 0; j < redPols[i].size(); ++j)
			redRings.add(redPols[i][j]);

	TeLineSet blueRings;
	for(unsigned int k = 0; k < bluePols.size(); ++k)
		for(unsigned int j = 0; j < bluePols[k].size(); ++j)
			blueRings.add(bluePols[k][j]);

	// Choose red fragments
	TeLineSet redFragments;
	TeLineSet redFragmentsBoundary;
	TeFragmentBoundary(redRings, report, redFragmentsBoundary, redFragments);

	unsigned int setSize = redFragments.size();

	TeCoord2D middle;

	if(redFragmentsBoundary.size())
		redMask = TeBOUNDARY;

	unsigned int w = 0;

	for(w = 0; w < redFragmentsBoundary.size(); ++w)
		redBoundaryFragmentsIndex.insert(TeLineIndex::value_type(redFragmentsBoundary[w][0], pair<short, TeLine2D>(TeBOUNDARY, redFragmentsBoundary[w])));

	vector<TeLinearRing> rings;

	// Do a position test for each fragment and stop if all relations have been found.
	for(w = 0; w < setSize; ++w)
	{	
		if(redFragments[w].size() ==  2)	// If the fragment has two points I need to check the middle point of this fragment.
			TeGetMiddlePoint(redFragments[w][0], redFragments[w][1], middle);
		else	// If the fragment has more than two points so I check one point between the end points.
			middle = redFragments[w][(unsigned int)((double(redFragments[w].size()) / 2.0 + 0.6)) - 1];
			
		location = TeRelation(middle, bluePols);

		// If fragment is on location type or is boundary, get it.
		if((location & locationRedFragments))
		{
			if(redFragments[w].isRing())
				rings.push_back(redFragments[w]);
			else
				redFragmentsIndex.insert(TeLineIndex::value_type(redFragments[w][0], pair<short, TeLine2D>(w, redFragments[w])));
		}

		redMask |= location;
	}

	// Choose blue fragments

	TeINTERSECTOR2::TeVectorBoundaryIP::iterator it     = report.begin();
	TeINTERSECTOR2::TeVectorBoundaryIP::iterator it_end = report.end();
	
	while(it != it_end)
	{
		swap(it->bluePartNum_, it->redPartNum_);
		swap(it->blueSegNum_, it->redSegNum_);
		++it;
	}

	// Choose blue fragments
	TeLineSet blueFragments;
	TeLineSet blueFragmentsBoundary;
	
	TeFragmentBoundary(blueRings, report, blueFragmentsBoundary, blueFragments);

		
	setSize = blueFragments.size();

	for(w = 0; w < blueFragmentsBoundary.size(); ++w)
		blueBoundaryFragmentsIndex.insert(TeLineIndex::value_type(blueFragmentsBoundary[w][0], pair<short, TeLine2D>(TeBOUNDARY, blueFragmentsBoundary[w])));

	if(blueFragmentsBoundary.size())
		blueMask = TeBOUNDARY;

	// Do a position test for each fragment and stop if all relations have been found.
	for(unsigned int j = 0; j < setSize; ++j)
	{	
		if(blueFragments[j].size() ==  2)	// If the fragment has two points I need to check the middle point of this fragment.
			TeGetMiddlePoint(blueFragments[j][0], blueFragments[j][1], middle);
		else	// If the fragment has more than two points so I check one point between the end points.
			middle = blueFragments[j][(unsigned int)((double(blueFragments[j].size()) / 2.0 + 0.6)) - 1];
			
		location = TeRelation(middle, redPols);

		// If fragment is on location type or is boundary, get it.
		if((location & locationBlueFragments))
		{
			if(blueFragments[j].isRing())
				rings.push_back(blueFragments[j]);
			else
				blueFragmentsIndex.insert(TeLineIndex::value_type(blueFragments[j][0], pair<short, TeLine2D>(w, blueFragments[j])));
		}
		
		blueMask |= location;		
	}

	// Remove closed rings => Only needs to test open segments
	TeFindAndMoveClosedRings(redBoundaryFragmentsIndex, rings);
	TeFindAndMoveClosedRings(blueBoundaryFragmentsIndex, rings);


	// Try to eliminate oposite boundaries
	TeRemoveOpositeBoundaryFragments(redBoundaryFragmentsIndex, blueBoundaryFragmentsIndex);

	// Try to eliminate same boundary fragments
	TeRemoveSameBoundaryFragments(redBoundaryFragmentsIndex, blueBoundaryFragmentsIndex);

	// Merge all red fragments in the first fragment index
	TeMergeFragments(redFragmentsIndex, redBoundaryFragmentsIndex);

	// Merge all blue fragments in the first fragment index
	TeMergeFragments(blueFragmentsIndex, blueBoundaryFragmentsIndex);


	// Make polygons from fragments.
	bool returnValue = true;

	TeLine2D  lAux;	
	TeCoord2D endLineCoordinate;

	while(!(redFragmentsIndex.empty() && blueFragmentsIndex.empty()))
	{
		if(lAux.size() == 0)
		{
			TeLineIndex::iterator indexIterator  = redFragmentsIndex.begin();
			if(indexIterator != redFragmentsIndex.end())
			{
				if(indexIterator->second.second.isRing())
				{
					rings.push_back(indexIterator->second.second);
					redFragmentsIndex.erase(indexIterator);
					continue;
				}
				
				for(unsigned int i = 0; i < indexIterator->second.second.size(); ++i)
					lAux.add(indexIterator->second.second[i]);

				lAux.setBox(::TeUnion(lAux.box(), indexIterator->second.second.box()));

				redFragmentsIndex.erase(indexIterator);
			}
			else
			{
				TeLineIndex::iterator indexIterator  = blueFragmentsIndex.begin();
				if(indexIterator != blueFragmentsIndex.end())
				{
					if(indexIterator->second.second.isRing())
					{
						rings.push_back(indexIterator->second.second);
						blueFragmentsIndex.erase(indexIterator);
						continue;
					}

					for(unsigned int i = 0; i < indexIterator->second.second.size(); ++i)
						lAux.add(indexIterator->second.second[i]);

					//lAux.copyElements(indexIterator->second);
					lAux.setBox(::TeUnion(lAux.box(), indexIterator->second.second.box()));
					blueFragmentsIndex.erase(indexIterator);
				}
				else
				{
					returnValue = false;	//No poderia vir aqui, deveria ter sado no teste do lao!!
				}
			}
		}	
		else
		{
			endLineCoordinate = lAux[lAux.size() - 1];

			// Try to find the beginning of the next fragment that is part of the polygon in the same list
			TeLineIndex::iterator indexIterator = redFragmentsIndex.find(endLineCoordinate);

			if(indexIterator != redFragmentsIndex.end())
			{
				for(unsigned int i = 1; i < indexIterator->second.second.size(); ++i)
					lAux.add(indexIterator->second.second[i]);

				lAux.setBox(::TeUnion(lAux.box(), indexIterator->second.second.box()));
				redFragmentsIndex.erase(indexIterator);
			}			
			else
			{
				indexIterator = blueFragmentsIndex.find(endLineCoordinate);

				if(indexIterator != blueFragmentsIndex.end())
				{
					for(unsigned int i = 1; i < indexIterator->second.second.size(); ++i)
						lAux.add(indexIterator->second.second[i]);

					lAux.setBox(::TeUnion(lAux.box(), indexIterator->second.second.box()));
					blueFragmentsIndex.erase(indexIterator);
				}
				else
				{
					if(lAux.isRing())
					{
						// Add ring
						rings.push_back(TeLinearRing(lAux));						
					}

					returnValue = false;	// Erro na topologia dos polgonos

					// Clear auxiliary.
					TeLine2D emptyLine;
					lAux = emptyLine;
				}
			}
		}

		if(lAux.isRing())
		{  
			// Add polygon
			rings.push_back(TeLinearRing(lAux));
			
			// Clear auxiliary.
			TeLine2D emptyLine;					
			lAux = emptyLine;
		}
	}

	if(lAux.size() > 0)
		returnValue = false;	// Erro, alguma linha no fechou!!!

	// Separate holes from islands
	vector<TeLinearRing> holes;

	//string aux = "";
	for(unsigned int z = 0; z < rings.size(); ++z)
	{
		short ori = TeOrientation(rings[z]);

		if(ori == TeCOUNTERCLOCKWISE)	// It is a hole
		{
			holes.push_back(rings[z]);			// add to holes list
		}
		else if(ori == TeCLOCKWISE)		// else if island
		{										// create a polygon
			TePolygon p;
			p.add(rings[z]);
			polsOut.add(p);
		}
		else	
		{
			returnValue = false;	// Objeto sem rea? Isso  um erro!
		}
	}
	
	if((polsOut.size() == 0) && (holes.size() == 0))
	{
		if(operation & TeUNION)
			return false;	// Na unio deve haver a formao de polgonos
		else
			return returnValue;	// Diferena e interseo podem retornar vazio
	}

	bool mountResult = TeMountTopology(polsOut, holes);

	return (returnValue && mountResult);	
}

TeMultiGeometry TeOVERLAY::TeOverlay(const TeLineSet& redLinesIn, const TePolygonSet& bluePolsIn, const short& operation)
{
	short location = TeUNKNOWNPOSITION;
	short locationRedFragments = TeUNKNOWNPOSITION;

	TeMultiGeometry mGeom;

	TeLineSet redLines;
	TePolygonSet bluePols;
	TeLineSet blueLines;

	// Copia o contedo das linhas de entrada
	TeCloneLineSet(redLinesIn, redLines);

	// Copia o contedo dos polgonos de entrada
	TeClonePolygonSet(bluePolsIn, bluePols);


	// Ajeita a orientao das fronteiras dos polgonos e determina a localizao dos fragmentos que sero utilizados
	mGeom = TeChooseOperation(operation, redLines, bluePols, locationRedFragments);

	if(!mGeom.empty())
		return mGeom;

	unsigned int i = 0u;

	// Gets intersection list
	TeINTERSECTOR2::TeVectorBoundaryIP report;

	unsigned int redLinesSize = redLines.size();

	// Loops through red lines
	for(i = 0u; i < redLinesSize; ++i)
	{
		unsigned int bluePart = 0u;

		// Loops through blue polygonset
		unsigned int bluePolsSize = bluePols.size();

		for(unsigned int j = 0u; j < bluePolsSize; ++j)
		{
			//Loops trough blue polygon rings
			unsigned int bluePolNumRings = bluePols[j].size();

			for(unsigned k = 0u; k < bluePolNumRings; ++k)
			{
				TeINTERSECTOR2::TeSafeIntersections(redLines[i], bluePols[j][k], report, i, bluePart);

				++bluePart;
			}
		}

		//++redPart;
	}

	// Fragment red Lines
	TeLineSet redFragments;
	TeLineSet redFragmentsBoundary;
	TeFragmentBoundary(redLines, report, redFragmentsBoundary, redFragments);

	// Choose red fragments
	unsigned int setSize = redFragments.size();

	TeCoord2D middle;

	for(i = 0u; i < setSize; ++i)
	{	
		if(redFragments[i].size() ==  2)	// If the fragment has two points I need to check the middle point of this fragment.
			TeGetMiddlePoint(redFragments[i][0u], redFragments[i][1u], middle);
		else	// If the fragment has more than two points so I check one point between the end points.
			middle = redFragments[i][(unsigned int)((double(redFragments[i].size()) / 2.0 + 0.6)) - 1];
			
		location = TeRelation(middle, bluePols);

		// If fragment is on location type, get it.
		if((location & locationRedFragments))
			mGeom.addGeometry(redFragments[i]);
	}

	switch(operation)
	{							
		case TeINTERSECTION:  break;

		case TeUNION:         mGeom.setGeometry(bluePols);
							  break;

		case TeDIFFERENCE:	  break;
	}

	return mGeom;

}


