/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifdef WIN32
#pragma warning ( disable: 4786 )
#endif

#include <map>

using namespace std;

#include "TeDatum.h"
#include "TeException.h"
#include "TeUtils.h"

const int NUMDATUM = 8; // number of datum
const int NUMSPH   = 8; // number of ellipsoids

struct SphParam 
{
	double rd;
	double flt;
};

typedef map<string, SphParam> SphMap;

// Spheroid       Radius(Semi Major Axis)        Flattening
char* ellipsoidList [] = 
{ 
"Airy",	       "6.3775634e+06", "3.340906e-03",	  
"Clarke-1866", "6.378206e+06",	"3.390060e-03",	 
"EarthRadius", "6.371000e+06",	"0.000000e+00",
"Everest",     "6.3772763e+06", "3.324468e-03",
"GRS80",       "6.378137e+06",  "3.352811e-03",
"Hayford", 	   "6.378388e+06",  "3.367003e-03", 	 
"UGGI67", 	   "6.378160e+06",	"3.352892e-03",	 
"WGS84",	   "6.378137e+06",	"3.352811e-03"
};



// DatumName    Spheroid        Shift_X           Shift_Y           Shift_Z

char* datumList [] = 
{ 
"Astro-Chua",   "Hayford",      "-7.700000e+01",  "2.390000e+02",  "5.000000e+00",
"CorregoAlegre","Hayford",	    "-1.387000e+02",  "1.644000e+02",  "3.440000e+01",
"Indian",	    "Everest",	    "0.000000e+00",   "0.000000e+00",  "0.000000e+00",
"NAD27",	    "Clarke-1866",	"0.000000e+00",   "0.000000e+00",  "0.000000e+00",
"NAD83",	    "GRS80",	    "0.000000e+00",   "0.000000e+00",  "0.000000e+00",
"SAD69", 	    "UGGI67",       "0.000000e+00",   "0.000000e+00",  "0.000000e+00",
"Spherical",	"EarthRadius",  "0.000000e+00",   "0.000000e+00",  "0.000000e+00",
"WGS84",	    "WGS84",	    "6.687000e+01",   "-4.370000e+00", "3.852000e+01"
};

typedef map<string, TeDatum> DatumMap;


TeDatum::TeDatum(const TeDatum& other)
{
	name_ = other.name_;
	rd_ = other.rd_ ;
	flt_= other.flt_;
	dx_ = other.dx_;
	dy_= other.dy_;
	dz_ = other.dz_;
}

TeDatum&
TeDatum::operator=(const TeDatum& rhs)
{
	if ( this != &rhs )
	{
		name_ = rhs.name_;
		rd_ = rhs.rd_;
		flt_= rhs.flt_;
		dx_ = rhs.dx_;
		dy_= rhs.dy_;
		dz_ = rhs.dz_;
	}
	return *this;
}


TeDatum
TeDatumFactory::make( const string& name )
{

	DatumMap datumMap_;
	// Step 1 - Read the Spheroids - Put in a Map

	SphMap sphMap;

	SphParam sphPar;

	int i, k = 0;

	for ( i = 0; i < NUMSPH; i++ )
	{
		string name = ellipsoidList [k++];
		sphPar.rd  = atof ( ellipsoidList [k++] );
		sphPar.flt = atof ( ellipsoidList [k++] );

		sphMap [ name ] = sphPar;
	}
	
	// Step 2 - Read the Datum List, create a Map


	k = 0;

	for ( i = 0; i < NUMDATUM; i++ )
	{
			string name = datumList [k++];
			
			string sphname = datumList [ k++];

			double 	dx  = atof ( datumList [k++] );
			double  dy  = atof ( datumList [k++] );
			double  dz  = atof ( datumList [k++] );

			SphMap::iterator it = sphMap.find ( sphname );
			
			if ( it == sphMap.end() )
			{
				throw TeException ( DATUM_ERROR );
			}
			else 
				sphPar = (*it).second;

			TeDatum dat ( name, sphPar.rd, sphPar.flt, dx, dy, dz );

			datumMap_ [ name ] = dat; // create a datum map
	}

	DatumMap::iterator it = datumMap_.find ( name );

	if ( it != datumMap_.end() ) // found it
		return (*it).second;

	return TeDatum ("Spherical");
}


bool findDatum(double semiMajor, double flatenning, TeDatum& datum)
{
	DatumMap datumMap_;

	// Step 1 - Read the Spheroids - Put in a Map

	SphMap sphMap;
	SphParam sphPar;

	int i, k = 0;

	for ( i = 0; i < NUMSPH; i++ )
	{
		string name = ellipsoidList [k++];
		sphPar.rd  = atof ( ellipsoidList [k++] );
		sphPar.flt = atof ( ellipsoidList [k++] );

		sphMap [ name ] = sphPar;
	}
	
	// Step 2 - Read the Datum List, create a Map
	k = 0;

	for ( i = 0; i < NUMDATUM; i++ )
	{
		string name = datumList [k++];
		string sphname = datumList [ k++];
		double 	dx  = atof ( datumList [k++] );
		double  dy  = atof ( datumList [k++] );
		double  dz  = atof ( datumList [k++] );

		SphMap::iterator it = sphMap.find ( sphname );
			
		if ( it == sphMap.end() )
		{
			throw TeException ( DATUM_ERROR );
		}
		else 
			sphPar = (*it).second;
		TeDatum dat ( name, sphPar.rd, sphPar.flt, dx, dy, dz );
		datumMap_ [ name ] = dat; // create a datum map
	}

	DatumMap::iterator it = datumMap_.begin();
	while (it != datumMap_.end())
	{
		TeDatum dt = (*it).second;
		if ( TeCompareDouble(dt.radius(),semiMajor,6) &&
			 TeCompareDouble(dt.flattening(),flatenning,6))
		{

			datum = dt;
			return true;
		}
		++it;
	}
	return false;
}
