/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeException.h"
#include "TeImportExport.h"
#include "TeSPRFile.h"

#include "TeDataTypes.h"
#include "TeUtils.h"
#include "TeLayer.h"
#include "TeTable.h"
#include "TeGeometry.h"
#include "TeGeometryAlgorithms.h"
#include "TeDatabase.h"
#include "TeProgress.h"
#include <time.h>

// --- Spring

bool TeExportSPRMNTData			( TeLayer* layer, const string& sprFileName, const string& tabName, const string& attrQuoteName );
bool TeExportSPRThematicData	( TeLayer* layer, const string& sprFileName, const string& tabName, const string& attThemeName );
bool TeExportSPRNetworkData		( TeLayer* layer, const string& sprFileName, const string& objName);
bool TeExportSPRCadastralData	( TeLayer* layer, const string& sprFileName, const string& objName);
bool TeExportSPRTable			(TeLayer* layer, const string& sprFileName, const string& tabName, const string& objName,TeSpringModels cat);

bool
TeExportSPR (TeLayer* layer, const string& sprFileName, const string& tableName, TeSpringModels cat, const string& name)
{
	if (tableName.empty() || sprFileName.empty())
		return false;
	switch (cat) 
	{
	case TeSPRMNT:
		return TeExportSPRMNTData(layer,sprFileName,tableName, name);
	case TeSPRThematic:
		return TeExportSPRThematicData(layer,sprFileName,tableName,name);
	case TeSPRNetwork:
		if (TeExportSPRNetworkData(layer,sprFileName,name))
			return TeExportSPRTable (layer,sprFileName,tableName,name,TeSPRNetwork);
	default:
		if (TeExportSPRCadastralData(layer,sprFileName,name))
			return TeExportSPRTable (layer,sprFileName,tableName,name,TeSPRCadastral);
	}
	return false;
}

bool TeExportSPRMNTData( TeLayer* layer, const string& sprFileName, const string& tabName, const string& attrQuoteName )
{
	clock_t	ti, tf;
	int count;
	TeDatabasePortal* portal = 0;
	try{
		
		TeSPRFile sprFile ( sprFileName+".spr", "w+" );

		if (attrQuoteName.empty() || tabName.empty())
			return false;

		if (!(layer->hasGeometry(TePOINTS) || 
			layer->hasGeometry(TeLINES)))
			return false;

		// check if layer has the attribute table
		TeTable tableE;
		if (!layer->getAttrTablesByName(tabName,tableE))
			return false;
		string linkName = tableE.linkName();
		if (linkName.empty())
			return false;

		TeDatabasePortal* portal = layer->database()->getPortal();
		if (!portal)
			return false;

		// DTM/Network data can have POINTS 3D or contour line 3D, so the
		// header SAMPLE should be written only once 
		sprFile.writeSampleHeader();
		if ( layer->projection() )
			sprFile.writeProjection ( layer->projection() );
		sprFile.writeBox ( layer->box() );
		sprFile.writeInfoEnd();

		string sql = "SELECT attT." + attrQuoteName + ", geoT.* FROM " + tabName + " attT, ";
		bool flag = true;
		if (layer->hasGeometry(TeLINES))
		{
			sql += layer->tableName(TeLINES) + " geoT ";
			sql += " WHERE attT." + linkName + "= geoT.object_id";
			if (portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				count = 0;
				do 
				{
					TeLine2D line;
					double quote = portal->getDouble(0);
					flag = portal->fetchGeometry(line);
					if (!line.empty())
					{	
						sprFile.writeLine3DHeader();
						sprFile.writeSampleQuote( quote );
						for (unsigned int k=0; k<line.size(); k++ )
							sprFile.writePoint ( line[k] );
						sprFile.writeEnd();
					}
					count++;
					if (TeProgress::instance())
					{
						tf = clock();
						if (((tf-ti)/CLOCKS_PER_SEC) > 3)
							TeProgress::instance()->setProgress(count);
						if (TeProgress::instance()->wasCancelled())
							break;
						ti = tf;
					}
				} while(flag);
				if (count)
					sprFile.writeEnd();
			}
			portal->freeResult();
		}
		if (layer->hasGeometry(TePOINTS))
		{
			sql += layer->tableName(TePOINTS) + " geoT ";
			sql += " WHERE attT." + linkName + "= geoT.object_id";
			if (portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				count = 0;
				do 
				{
					TePoint xy;
					if (count == 0)
						sprFile.writePoint3DHeader();

					double quote = portal->getDouble(0);
					flag = portal->fetchGeometry(xy);

					sprFile.writePoint3D( xy.location().x(), xy.location().y(), quote );
					count++;
					if (TeProgress::instance())
					{
						tf = clock();
						if (((tf-ti)/CLOCKS_PER_SEC) > 3)
							TeProgress::instance()->setProgress(count);
						if (TeProgress::instance()->wasCancelled())
							break;
						ti = tf;
					}
				} while (flag);
				if (count)
					sprFile.writeEnd();
			}
		}
		delete portal;
		return true;
	}
	catch(...)
	{
		if (portal)
			delete portal;
		return false;
	}
	if(TeProgress::instance())
		TeProgress::instance()->reset();
}

bool TeExportSPRThematicData( TeLayer* layer, const string& sprFileName, const string& tabName, const string& attThemeName )
{
	if (attThemeName.empty() || tabName.empty())
		return false;

	if (!layer || !(layer->hasGeometry(TePOINTS) || layer->hasGeometry(TeLINES) || 
		            layer->hasGeometry(TePOLYGONS) || layer->hasGeometry(TeCELLS)))
		return false;

	// check if layer has the attribute table
	TeTable tableE;
	if (!layer->getAttrTablesByName(tabName,tableE))
		return false;
	string linkName = tableE.linkName();
	if (linkName.empty())
		return false;

	TeDatabasePortal* portal=0;
	clock_t	ti, tf;
	int count;

	try{
		portal = layer->database()->getPortal();
		if (!portal)
			return false;
		TeSPRFile sprFile ( sprFileName+".spr", "w+" );
		
		unsigned int k;
		bool flag = true;
		
		string sql = "SELECT attT." + attThemeName + ", geoT.* FROM " + tabName + " attT, ";
		if (layer->hasGeometry(TePOINTS))
		{
			sql += layer->tableName(TePOINTS) + " geoT ";
			sql += " WHERE attT." + linkName + "= geoT.object_id";
			if(portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				count = 0;
				do 
				{
					if (count==0)
					{
						sprFile.writePointsHeader ( );
						sprFile.writeProjection ( layer->projection() );
						sprFile.writeBox ( layer->box() );
						sprFile.writeInfoEnd();
					}
					string theme = portal->getData(attThemeName);
					TePoint xy;
					flag = portal->fetchGeometry(xy);
					sprFile.writeThematicPoint(xy.location().x(), xy.location().y(), theme );
					count++;
					if (TeProgress::instance())
					{
						tf = clock();
						if (((tf-ti)/CLOCKS_PER_SEC) > 3)
							TeProgress::instance()->setProgress(count);
						if (TeProgress::instance()->wasCancelled())
							break;
						ti = tf;
					}
				} while (flag);
			}
			portal->freeResult();
			if (count)
				sprFile.writeEnd();
		}
		if (!layer->hasGeometry(TeLINES) && !layer->hasGeometry(TePOLYGONS) && !layer->hasGeometry(TeCELLS))
		{
			delete portal;
			sprFile.writeEnd();
			return true;
		}
		count = 0;
		TeSPRFile labelFile(sprFileName+"_LAB.spr", "w+");
		if (layer->hasGeometry(TeLINES))
		{
			sql += layer->tableName(TeLINES) + " geoT ";
			sql += " WHERE attT." + linkName + "= geoT.object_id";
			if (portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				count = 0;
				do{
					string theme = portal->getData(0);
					TeLine2D line;
					flag = portal->fetchGeometry(line);
					if (!line.empty())
					{
						if (count == 0)
						{
							sprFile.writeLinesHeader ( );
							if ( layer->projection() )
								sprFile.writeProjection ( layer->projection() );
							sprFile.writeBox ( layer->box() );
							sprFile.writeInfoEnd();
							labelFile.writeLabelHeader(TeSPRThematic);
						}

						for (k=0; k<line.size(); k++ )
							sprFile.writePoint ( line[k] );
						sprFile.writeEnd();
						TeCoord2D pt = TeFindCentroid( line );
						labelFile.writeThematicLabel(pt.x(), pt.y(),theme);
						count++;
						if (TeProgress::instance())
						{
							tf = clock();
							if (((tf-ti)/CLOCKS_PER_SEC) > 3)
								TeProgress::instance()->setProgress(count);
							if (TeProgress::instance()->wasCancelled())
								break;
							ti = tf;
						}
					}
				} while (flag);
			}
			portal->freeResult();
		}

		if (layer->hasGeometry(TePOLYGONS))
		{
			sql += layer->tableName(TePOLYGONS) + " geoT ";
			sql += " WHERE attT." + linkName + "= geoT.object_id";
			if (portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				count = 0;
				do 
				{
					string theme = portal->getData(0);
					TePolygon poly;
					flag = portal->fetchGeometry(poly);
					if (!poly.empty())
					{
						if (count == 0)
						{
							sprFile.writeLinesHeader ( );
							if ( layer->projection() )
								sprFile.writeProjection ( layer->projection() );
							sprFile.writeBox ( layer->box() );
							sprFile.writeInfoEnd();
							labelFile.writeLabelHeader(TeSPRThematic);
						}
						for (k = 0; k < poly.size(); k++ )
						{
							TeLinearRing ring = poly[k]; 
							
							for (unsigned int l = 0 ; l < ring.size(); l++)
								sprFile.writePoint ( ring[l] );
							sprFile.writeEnd();
						}

						TeCoord2D pt = TeFindCentroid( poly );
						labelFile.writeThematicLabel(pt.x(), pt.y(),theme);
						count++;
						if (TeProgress::instance())
						{
							tf = clock();
							if (((tf-ti)/CLOCKS_PER_SEC) > 3)
								TeProgress::instance()->setProgress(count);
							if (TeProgress::instance()->wasCancelled())
								break;
							ti = tf;
						}
					}
				} while (flag);
			}
			portal->freeResult();
		}

		if (layer->hasGeometry(TeCELLS))
		{
			sql += layer->tableName(TeCELLS) + " geoT ";
			sql += " WHERE attT." + linkName + "= geoT.object_id";
			if (portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				count = 0;
				do 
				{
					string theme = portal->getData(0);
					TeCell cell;
					flag = portal->fetchGeometry(cell);
					if (count == 0)
					{
						sprFile.writeLinesHeader ( );
						if ( layer->projection() )
							sprFile.writeProjection ( layer->projection() );
						sprFile.writeBox ( layer->box() );
						sprFile.writeInfoEnd();
						labelFile.writeLabelHeader(TeSPRThematic);
					}
					TeCoord2D ll = cell.box().lowerLeft();
					TeCoord2D ur = cell.box().upperRight();
					sprFile.writePoint (ll);
                    TeCoord2D p1(ur.x(), ll.y());
					sprFile.writePoint (p1);
					sprFile.writePoint (ur);
                    TeCoord2D p2(ll.x(),ur.y());                   
					sprFile.writePoint (p2);
					sprFile.writePoint (ll);
					sprFile.writeEnd();
					TeCoord2D pt = TeFindCentroid(cell);
					labelFile.writeThematicLabel(pt.x(), pt.y(),theme);
					count++;
					if (TeProgress::instance())
					{
						tf = clock();
						if (((tf-ti)/CLOCKS_PER_SEC) > 3)
							TeProgress::instance()->setProgress(count);
						if (TeProgress::instance()->wasCancelled())
							break;
						ti = tf;
					}
				} while (flag);
			}
			portal->freeResult();
		}
		sprFile.writeEnd();
		labelFile.writeEnd();
		delete portal;
		return true;
	}
	catch(...)
	{
		if (portal)
			delete portal;
		return false;
	}
	if(TeProgress::instance())
		TeProgress::instance()->reset();
}


bool TeExportSPRNetworkData(TeLayer* layer, const string& sprFileName, const string& objName)
{
	if (!layer || !(layer->hasGeometry(TePOINTS) || layer->hasGeometry(TeLINES)) )
		return false;

	TeDatabasePortal* portal = layer->database()->getPortal();
	if (!portal)
		return false;
	clock_t	ti,tf;
	int count;

	try{
		TeSPRFile sprFile ( sprFileName+".spr", "w+" );
		sprFile.writeNetworkHeader();
		if ( layer->projection() )
			sprFile.writeProjection ( layer->projection() );
		sprFile.writeBox ( layer->box() );
		sprFile.writeInfoEnd();
		bool flag;	;
		if (layer->hasGeometry(TeLINES))
		{
			string sql = "SELECT * FROM " + layer->tableName(TeLINES);
			if (portal->query(sql) && portal->fetchRow())
			{		
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				TeSPRFile labelFile(sprFileName+"_LAB.spr", "w+");
				do 
				{
					TeLine2D line;
					string objid = portal->getData("object_id");
					flag = portal->fetchGeometry(line);
					if (!line.empty())
					{
						sprFile.writeNetworkLineHeader(count++);
						unsigned int k;
						for (k=0; k<line.size(); k++ )
							sprFile.writePoint ( line[k] );
						sprFile.writeEnd();
						if (count==0)
							labelFile.writeLabelHeader(TeSPRNetwork);
						TeCoord2D pt = TeFindCentroid( line );
						labelFile.writeNetworkLabel(objid,pt.x(), pt.y(),objName);
						count++;
					}
					count++;
					if (TeProgress::instance())
					{
						tf = clock();
						if (((tf-ti)/CLOCKS_PER_SEC) > 3)
							TeProgress::instance()->setProgress(count);
						if (TeProgress::instance()->wasCancelled())
							break;
						ti = tf;
					}
				}while (flag);
				if (count>0)
				{
					sprFile.writeEnd();	 
					labelFile.writeEnd();
				}			
			}
			portal->freeResult();
		}
	}
	catch (...)
	{
		delete portal;
		return false;
	}
	delete portal;
	if(TeProgress::instance())
		TeProgress::instance()->reset();
	return true;
}

bool TeExportSPRCadastralData( TeLayer* layer, const string& sprFileName, const string& objName)
{
	if (!layer || !(layer->hasGeometry(TePOINTS) || layer->hasGeometry(TeLINES) || 
					layer->hasGeometry(TePOLYGONS) || layer->hasGeometry(TeCELLS)))
		return false;

	TeDatabasePortal* portal= layer->database()->getPortal();
	if (!portal)
		return false;
	bool flag;
	clock_t	ti, tf;
	int count = 0;
	unsigned int k;
	try {
		
		TeSPRFile sprFile ( sprFileName+".spr", "w+" );
		if (layer->hasGeometry(TePOINTS))
		{
			string sql = "SELECT * FROM " + layer->tableName(TePOINTS);
			if (portal->query(sql) && portal->fetchRow())
			{
				if(TeProgress::instance())
					TeProgress::instance()->setTotalSteps(portal->numRows());
				ti = clock();
				do
				{
					if (count==0)
					{
						sprFile.writePointsHeader ( );
						sprFile.writeProjection ( layer->projection() );
						sprFile.writeBox ( layer->box() );
						sprFile.writeInfoEnd();
					}
					string objid = portal->getData("object_id");
					TePoint xy;
					flag = portal->fetchGeometry(xy);
					sprFile.writeCadastralLabel(objid,xy.location().x(), xy.location().y(),objName);
					count++;
					if (TeProgress::instance())
					{
						tf = clock();
						if (((tf-ti)/CLOCKS_PER_SEC) > 3)
							TeProgress::instance()->setProgress(count);
						if (TeProgress::instance()->wasCancelled())
							break;
						ti = tf;
					}
				} while (flag);
				if (count)
					sprFile.writeEnd();
			}
			portal->freeResult();
		}

		count = 0;
		// if layer has lines, polygons or cells there should be one more file with labels
		if (layer->hasGeometry(TeLINES) || layer->hasGeometry(TePOLYGONS)|| layer->hasGeometry(TeCELLS))
		{
			TeSPRFile labelFile(sprFileName+"_LAB.spr", "w+");
			if (layer->hasGeometry(TeLINES))
			{
				string sql = "SELECT * FROM " + layer->tableName(TeLINES);
				if (portal->query(sql) && portal->fetchRow())
				{
					if(TeProgress::instance())
						TeProgress::instance()->setTotalSteps(portal->numRows());
					ti = clock();
					do 
					{
						string objid = portal->getData("object_id");
						TeLine2D line;
						flag = portal->fetchGeometry(line);
						if (!line.empty())
						{
							if (count == 0)
							{
								sprFile.writeLinesHeader ( );
								if ( layer->projection() )
									sprFile.writeProjection ( layer->projection() );
								sprFile.writeBox ( layer->box() );
								sprFile.writeInfoEnd();
								labelFile.writeLabelHeader(TeSPRCadastral);
							}
							for (k=0; k<line.size(); k++ )
								sprFile.writePoint ( line[k] );
							sprFile.writeEnd();
							TeCoord2D xy = TeFindCentroid( line );
							labelFile.writeCadastralLabel(objid,xy.x(), xy.y(),objName);
						} // end - polygon is not empty
						count++;
						if (TeProgress::instance())
						{
							tf = clock();
							if (((tf-ti)/CLOCKS_PER_SEC) > 3)
								TeProgress::instance()->setProgress(count);
							if (TeProgress::instance()->wasCancelled())
								break;
							ti = tf;
						}
					} while (flag); // end - for every record
				} // end - if there is any record
			} // end - if has lines
			portal->freeResult();
			if (layer->hasGeometry(TePOLYGONS))
			{
				string sql = "SELECT * FROM " + layer->tableName(TePOLYGONS);
				if (portal->query(sql) && portal->fetchRow())
				{
					if(TeProgress::instance())
						TeProgress::instance()->setTotalSteps(portal->numRows());
					ti = clock();
					do 
					{
						string objid = portal->getData("object_id");
						TePolygon poly;
						flag = portal->fetchGeometry(poly);
						if (!poly.empty())
						{
							if (count == 0)
							{
								sprFile.writeLinesHeader ( );
								if ( layer->projection() )
									sprFile.writeProjection ( layer->projection() );
								sprFile.writeBox ( layer->box() );
								sprFile.writeInfoEnd();
								labelFile.writeLabelHeader(TeSPRCadastral);
							}
							for (k = 0; k < poly.size(); k++ )
							{
								TeLinearRing ring = poly[k]; 			
								for (unsigned int l = 0 ; l < ring.size(); l++)
									sprFile.writePoint ( ring[l] );
								sprFile.writeEnd();
							}
							TeCoord2D xy = TeFindCentroid( poly );
							labelFile.writeCadastralLabel(objid,xy.x(), xy.y(),objName);
						} // end - poly not empty
						count++;
						if (TeProgress::instance())
						{
							tf = clock();
							if (((tf-ti)/CLOCKS_PER_SEC) > 3)
								TeProgress::instance()->setProgress(count);
							if (TeProgress::instance()->wasCancelled())
								break;
							ti = tf;
						}
					} while (flag); // end - for every record
				} // end - has any record
			} // end - if has polygons
			portal->freeResult();
			if (layer->hasGeometry(TeCELLS))
			{
				string sql = "SELECT * FROM " + layer->tableName(TeCELLS);
				if (portal->query(sql) && portal->fetchRow())
				{
					if(TeProgress::instance())
						TeProgress::instance()->setTotalSteps(portal->numRows());
					ti = clock();
					do 
					{
						string objid = portal->getData("object_id");
						TeCell cell;
						flag = portal->fetchGeometry(cell);
						if (count == 0)
						{
							sprFile.writeLinesHeader ( );
							if ( layer->projection() )
								sprFile.writeProjection ( layer->projection() );
							sprFile.writeBox ( layer->box() );
							sprFile.writeInfoEnd();
							labelFile.writeLabelHeader(TeSPRCadastral);
						}
						TeCoord2D ll = cell.box().lowerLeft();
						TeCoord2D ur = cell.box().upperRight();
						sprFile.writePoint (ll);
                        TeCoord2D p1(ur.x(),ll.y());
						sprFile.writePoint (p1);
						sprFile.writePoint (ur);
                        TeCoord2D p2(ll.x(),ur.y());
						sprFile.writePoint (p2);
						sprFile.writePoint (ll);
						sprFile.writeEnd();
						TeCoord2D xy = TeFindCentroid(cell);
						labelFile.writeCadastralLabel(objid,xy.x(), xy.y(),objName);
						count++;
						if (TeProgress::instance())
						{
							tf = clock();
							if (((tf-ti)/CLOCKS_PER_SEC) > 3)
								TeProgress::instance()->setProgress(count);
							if (TeProgress::instance()->wasCancelled())
								break;
							ti = tf;
						}
					} while (flag);
				}
			}
			portal->freeResult();
			if (count>0)
			{
				sprFile.writeEnd();
				labelFile.writeEnd();
			}
		} // end - if has lines or polygons
		delete portal;
		return true;
	}
	catch (...)
	{
		delete portal;
		return true;
	}
	if(TeProgress::instance())
		TeProgress::instance()->reset();
}

bool TeExportSPRTable(TeLayer* layer, const string& sprFileName, const string& tabName, const string& objName, TeSpringModels cat)
{
	if (!layer || tabName.empty() || sprFileName.empty())
		return false;

	string obj;
	if (objName.empty())
		obj = "Objeto";
	else
		obj = objName;

	// check if layer has the attribute table
	TeTable tableE;
	if (!layer->getAttrTablesByName(tabName,tableE))
		return false;
	int index = tableE.attrLinkPosition();
	if (index < 0)
		return false;

	// Try to get the table
	TeDatabasePortal* portal = layer->database()->getPortal();
	if (!portal)
		return false;

	string sql = "SELECT * FROM " + tabName;
	if (!portal->query(sql))
	{
		delete portal;
		return false;
	}

	// retrieve the attribute information 
	TeAttributeList attInfo = portal->AttributeList();

	TeSPRFile tabFile(sprFileName+"_TAB.spr", "w+");

	// write the attribute information ( skip the index if exists)
	tabFile.writeTableHeader( cat );
	tabFile.writeAttributeInfo (attInfo, obj, tableE.linkName());
	tabFile.writeInfoEnd();
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(portal->numRows());
	int count = 0;
	clock_t	ti, tf;
	ti = clock();
	while (portal->fetchRow())
	{	
		TeTableRow row;
		for (int j = 0; j < portal->numFields(); j++)
			row.push_back ( portal->getData (j) ); 

		string label = row[index];
		tabFile.writeGeoId (label);
		for (unsigned int col = 0; col < row.size(); col++)
		{
			if ( col == (unsigned int) index )						// skip the index - already written
				continue;		
			tabFile.writeValue(row[col], attInfo[col].rep_.type_); // write the value
		}
		tabFile.writeNewLine();
		count++;
		if (TeProgress::instance())
		{
			tf = clock();
			if (((tf-ti)/CLOCKS_PER_SEC) > 3)
				TeProgress::instance()->setProgress(count);
			if (TeProgress::instance()->wasCancelled())
				break;
			ti = tf;
		}
	}
	tabFile.writeEnd();
	delete portal;
	if(TeProgress::instance())
		TeProgress::instance()->reset();
	return true;
}

