/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/
/*! \file TeCellAlgorithms.h
    This file contains algorithms and operations to deal with on cell geometry
*/

#ifndef  __TERRALIB_INTERNAL_CELLALGORITHMS_H
#define  __TERRALIB_INTERNAL_CELLALGORITHMS_H

#include "TeSTElementSet.h"
#include "TeComputeAttributeStrategies.h"

class TeDatabase; 
class TeLayer;
class TeGraphNetwork;


using namespace std;


/** @defgroup CellAlg Functions to generate and operate on cell geometry 
 *  @{
 */

/*!
	\brief Creates a new layer with the cells that covers the polygons of a theme  
	\param layerName name of the new layer with cell representations
	\param theme pointer to the theme that has the polygons
	\param resX X resolution of each cell
	\param resY Y resolution of each cell
	\param box bounding box of interest
	\return a pointer to the new layer created if success or null if fail
*/
TeLayer* TeCreateCells(const string& layerName,TeTheme* theme, double resX, double resY, TeBox& box);

/*!
	\brief Creates a new layer with the cells that covers the entire bounding box, or the polygons  of a layer 
	\param layerName name of the new layer with cell representations
	\param layerBase pointer to the base layer
	\param resX X resolution of each cell
	\param resY Y resolution of each cell
	\param mask flag that indicates if the cells should be created only over the polygons of the layer
	\param box bounding box of interest
	\return a pointer to the new layer created if success or null if fail
*/
TeLayer* TeCreateCells( const string& layerName, TeLayer* layerBase, 
			   double resX, double resY, TeBox& box, bool mask = true);


/*!
	\brief Calculates a set of statistics of cells of a theme 
	\param themeOut			cell theme 
	\param newAttrTable		new attribute table where the generated statistics will be kept 
	\param stat				the structure that contains the statistics which will be generated 
	\param themeIn			point or cell theme
	\param db				a pointer to the database 
*/
bool 
TeCellStatistics(TeTheme* themeOut, TeTable& newAttrTable, TeGroupingAttr& stat, 
				 TeTheme* themeIn, TeDatabase* db);


/*!
	\brief Saves the statistics accessible through a TeDatabasePortal in a database table 
	\param tableAttrCell	attribute table name where the generated statistics will be kept 
	\param portal			a pointer to the database portal that contains the generated statistics  
*/
bool TeKeepStatistics(TeTable& tableAttrCell, TeDatabasePortal* portal);


/** @} */ // End of group CellAlg format


//////////////////////////////////////////////////////////////////////
//
//					Fill Cell Function (headers)
//
/////////////////////////////////////////////////////////////////////
/** @defgroup FillCellAlg Functions to fill one attribute (or group of related attributes) for all cells in the output layer based on a single input layer attribute for a given time interval. Specific operation to be applied is a parameter (Strategy pattern).
 *  @{
 */

/*!
	\brief Computes the cell attribute based on a spatial operation (TeComputeAttrSpatialStrategy) over a vector input layer.
	\param db						a pointer to the database in which input and output layers are stored
	\param input_layername			name of input layer 
	\param rep						input layer representation to be used
	\param input_tablename			input layer attribute table to be used  
	\param input_attrname			input layer attribute to be used to compute output attribute
	\param t						time interval of output cells attributes
	\param cell_layername			output cell layer  
	\param cell_tablename			output cell attribute dynamic table; if does not exist, the table will be created by the function  
	\param output_columnName		output attribute name   
	\param operation				operation to be applied (see TeComputeAttrSpatialStrategy hierarchy)  
*/
bool TeFillCellSpatialOperation (	TeDatabase* db,
											const string& input_layername, 
											TeGeomRep rep,
											const string& input_tablename, 
											const string& input_attrname,
											TeTimeInterval t,
											const string& cell_layername, 
											const string& cell_tablename, 
											const string& output_columnName,
											TeComputeAttrSpatialStrategy* operation);


											
/*!
	\brief Computes the cell attribute based on a non-spatial operation (TeComputeAttrStrategy) over a vector or cell input layer.
	\param db						a pointer to the database in which input and output layers are stored
	\param input_layername			name of input layer 
	\param rep						input layer representation to be used
	\param input_tablename			input layer attribute table to be used  
	\param input_attrname			input layer attribute to be used to compute output attribute
	\param t						time interval of output cells attributes
	\param cell_layername			output cell layer  
	\param cell_tablename			output cell attribute dynamic table; if does not exist, the table will be created by the function  
	\param output_columnName		output attribute name   
	\param operation				operation to be applied (see TeComputeAttrStrategy hierarchy)  
*/
bool TeFillCellNonSpatialOperation (TeDatabase* db,
										const string& input_layername, 
										TeGeomRep rep,
										const string& input_tablename, 
										const string& input_attrname,
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										TeComputeAttrStrategy<TeSTElementSet::propertyIterator>* operation);

/*!
	\brief Computes the cell attribute based on a non-spatial operation (TeComputeSpatialStrategy) over a raster input layer.
	\param db						a pointer to the database in which input and output layers are stored
	\param input_raster_layername	name of input layer that contains a raster representation 
	\param t						time interval of output cells attributes
	\param cell_layername			output cell layer  
	\param cell_tablename			output cell attribute dynamic table; if does not exist, the table will be created by the function  
	\param output_columnName		output attribute name   
	\param operation				operation to be applied (see TeComputeAttrStrategy hierarchy)  
*/

bool TeFillCellNonSpatialRasterOperation (TeDatabase* db,
										const string& input_raster_layername, 
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										TeComputeAttrStrategy<TeRaster::iteratorPoly>* operation);





/*!
	\brief Computes the cell attribute based on a non-spatial operation (TeComputeSpatialStrategy) over a certain attribute of a set of objects connected to each cell by a network.
	\param db						a pointer to the database in which input and output layers are stored
	\param objects					set of objects (e.g., ports) 
	\param rep						objects representation type
	\param net						netork to be used to verify and quantify cell connectivity to each cell
	\param input_attrname			connected objects attribute to be used to compute output attribute
	\param t						time interval of output cells attributes
	\param cell_layername			output cell layer  
	\param cell_tablename			output cell attribute dynamic table; if does not exist, the table will be created by the function  
	\param output_columnName		output attribute name   
	\param operation				operation to be applied (see TeComputeAttrStrategy hierarchy)  
	\param local_distance_factor	multiplier used to compute connection strenght weights: local distance factor (see GPM documentation) 
	\param net_distance_factor		multiplier used to compute connection strenght weights: distance to network factor (see GPM documentation) 
	\param net_conn_factor			multiplier used to compute connection strenght weights: connection through network factor (see GPM documentation) 
	\param mult_factor				multiplier used to compute connection strenght weights: weight multiplier factor (see GPM documentation) 
*/


bool TeFillCellConnectionOperation (	TeDatabase* db,
										TeSTElementSet objects,
										TeGeomRep rep,
										TeGraphNetwork* net,
										const string& input_attrName,

										TeTimeInterval t,

										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										
							
										TeComputeAttrStrategy<TeSTElementSet::propertyIterator>* operation,
										double local_distance_factor = 0.0,
										double net_distance_factor = 0.25,
										double net_conn_factor = 1.0,
										double mult_factor = 1000000.0);

/** @} */ // End of group FillCellAlg 

		  
/** @defgroup AgregateCellAlg Functions aggregate a set of attributes to coarse resolution cells for a given time interval. Specific (non-spatial) operation to be applied is a parameter (Strategy pattern).
*  @{
*/


/*!
	\brief Computes a set of attributes for the output cell layer based on a non-spatial operation (TeComputeAttrSpatialStrategy) over a input cell layer.
	\param db						a pointer to the database in which input and output layers are stored
	\param input_layername			name of input cell layer 
	\param input_tablename			input layer attribute table to be used  
	\param t						time interval of output cells attributes
	\param cell_layername			output cell layer  
	\param cell_tablename			output cell attribute dynamic table; if does not exist, the table will be created by the function  
	\param attrNames				names of input/output attribute names to be aggregated   
	\param operation				operation to be applied (see TeComputeAttrStrategy hierarchy)  
*/


bool TeFillCellAggregateOperation (  TeDatabase* db,
										const string& input_layername, 
										const string& input_tablename, 
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										vector<string>& attrNames,
										TeComputeAttrStrategy<TeSTElementSet::propertyIterator>* operation);

/** @} */ // End of group AgregateCellAlg 


//////////////////////////////////////////////////////////////////////
//
//					Fill Cell Auxiliary Functions (headers)
//
/////////////////////////////////////////////////////////////////////


// Load cells geometries and verify/create dynamic attribute table if does not exists
bool TeFillCellInitLoad (TeLayer* cell_layer, const string& cell_tablename, TeCellSet& cells);



// Init output cell sto
void TeFillCellInitSTO (const TeCell& cell, TeTimeInterval& t, TePropertyVector& result, TeSTElementSet& cellObjSet);

// Creates dymanic table
bool TeCreateBasicDymanicCellTable (TeLayer* cell_layer, const string cell_tablename);



#endif

