/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/


#include "connect.h"
#include "sdo_oci.h"


OCIConnection::OCIConnection() 
{
	connected = false;

	envhp = 0;
	errhp = 0;
	srvhp = 0;
	svchp = 0;
	usrhp = 0;
	txnhp = 0;
	stmthp = 0;
	tdo_geometry = 0;
	tdo_ordinates = 0;
	ordinates = 0;
}


bool OCIConnection::Connect(const char* dblink, const char* username, const char* password) 
{
	sword status;

	int mode = OCI_THREADED|OCI_OBJECT;
	

	//create and initialize the environment
	status = SDO_OCIEnvCreate(&envhp, (ub4)mode, (dvoid *)0, 
				      (dvoid*(*)(dvoid *ctxp, size_t size))0, 
				      (dvoid*(*)(dvoid *ctxp, dvoid *memptr, size_t newsize))0, 
				      (void (*)(dvoid *ctxp, dvoid *memptr))0, 
					  (size_t)0, (dvoid**)0);
	if(!CheckError(errhp, status))
		return false;

	// Allocate handles - errhp(OCIError) and srvhp (OCIServer)
	status = SDO_OCIHandleAlloc((dvoid*)envhp, (dvoid**)&errhp, (ub4)OCI_HTYPE_ERROR,
					   (size_t)0, (dvoid**)0);
	if(!CheckError(errhp, status))
		return false;

	status = SDO_OCIHandleAlloc((dvoid*)envhp, (dvoid**)&srvhp, (ub4)OCI_HTYPE_SERVER,
					   (size_t)0, (dvoid**)0);
	if(!CheckError(errhp, status))
		return false;

	// Multiple Sessions or Connections
	status = SDO_OCIServerAttach(srvhp, errhp, (text*)dblink, 
						 (sb4)strlen(dblink), (ub4)OCI_DEFAULT);
	if(!CheckError(errhp, status))
		return false;

	// Allocate handles: svchp(OCISvcCtx)
	status = SDO_OCIHandleAlloc((dvoid*)envhp, (dvoid**)&svchp, (ub4)OCI_HTYPE_SVCCTX, 
					   (size_t)0, (dvoid**)0);
	if(!CheckError(errhp, status))
		return false;

	// Set the server attribute in the service context handle
	status = SDO_OCIAttrSet((dvoid*)svchp, (ub4)OCI_HTYPE_SVCCTX, (dvoid*)srvhp, (ub4)0, 
				   (ub4)OCI_ATTR_SERVER, errhp);
	if(!CheckError(errhp, status))
		return false;

	/// Allocate a user session handle
	status = SDO_OCIHandleAlloc((dvoid*)envhp, (dvoid**)&usrhp, (ub4)OCI_HTYPE_SESSION, 
					   (size_t)0, (dvoid**)0);
	if(!CheckError(errhp, status))
		return false;

	// set user name attribute in user session handle 
	status = SDO_OCIAttrSet((dvoid*)usrhp, (ub4)OCI_HTYPE_SESSION, (dvoid*)username, 
				   (ub4)strlen(username), (ub4)OCI_ATTR_USERNAME, errhp);
	if(!CheckError(errhp, status))
		return false;
	
	// set password attribute in user session handle
	status = SDO_OCIAttrSet((dvoid*)usrhp, (ub4)OCI_HTYPE_SESSION, (dvoid*)password, 
				   (ub4)strlen(password), (ub4)OCI_ATTR_PASSWORD, errhp);
	if(!CheckError(errhp, status))
		return false;

	// Session begins - establishes a session for a user against a particular
	// server. This call is required for the user to execute operations on the server.
	status = SDO_OCISessionBegin(svchp, errhp, usrhp, OCI_CRED_RDBMS, 
								 OCI_DEFAULT);
	if(!CheckError(errhp, status))
		return false;

	// set the user session attribute in the service context handle
	status = SDO_OCIAttrSet((dvoid*)svchp, (ub4)OCI_HTYPE_SVCCTX, (dvoid*)usrhp, (ub4)0, 
				   (ub4)OCI_ATTR_SESSION, errhp);
	if(!CheckError(errhp, status))
		return false;

	// allocate transaction handle 1 and set it in the service handle (OCITrans)
	status = OCIHandleAlloc((dvoid *)envhp, (dvoid **)&txnhp, OCI_HTYPE_TRANS, 0, 0);
	if(!CheckError(errhp, status))
		return false;

	status = OCIAttrSet((dvoid *)svchp, OCI_HTYPE_SVCCTX, (dvoid *)txnhp, 0,OCI_ATTR_TRANS, errhp);
	if(!CheckError(errhp, status))
		return false;

	// locate stmthp
	status = SDO_OCIHandleAlloc((dvoid*)envhp, (dvoid**)&stmthp, 
								  (ub4)OCI_HTYPE_STMT, (size_t)0, (dvoid**)0);
	if(!CheckError(errhp, status))
		return false;

	//load the tdo type to sdo_geometry (OCIType)
	status = OCITypeByName(envhp, errhp, svchp, (const text *)
			"MDSYS", (ub4) strlen((char *)"MDSYS"),
			(const text *) "SDO_GEOMETRY",	(ub4) strlen((char *)"SDO_GEOMETRY"),
			(text *)0, 0, OCI_DURATION_SESSION, OCI_TYPEGET_HEADER, &tdo_geometry);
	if(!CheckError(errhp, status))
		return false;

	//load the tdo type to sdo_ordinates (OCIType)
	status = OCITypeByName(envhp, errhp, svchp, (const text *)
			"MDSYS", (ub4) strlen((char *)"MDSYS"),
			(const text *) "SDO_ORDINATE_ARRAY", (ub4) strlen((char *)"SDO_ORDINATE_ARRAY"),
			(text *)0, 0, OCI_DURATION_SESSION, OCI_TYPEGET_HEADER, &tdo_ordinates);
	if(!CheckError(errhp, status))
		return false;

	connected = true;
	return true;
}


void OCIConnection::Disconnect()  
{
	// Session ends
	SDO_OCISessionEnd(svchp, errhp, usrhp, (ub4)OCI_DEFAULT);
	SDO_OCIServerDetach(srvhp, errhp, (ub4)OCI_DEFAULT);

	// Finalize svchp, srvhp, errhp and stmthp
	SDO_OCIHandleFree((dvoid*)txnhp, (ub4)OCI_HTYPE_TRANS);
	SDO_OCIHandleFree((dvoid*)srvhp, (ub4)OCI_HTYPE_SERVER);
	SDO_OCIHandleFree((dvoid*)svchp, (ub4)OCI_HTYPE_SVCCTX);
	SDO_OCIHandleFree((dvoid*)errhp, (ub4)OCI_HTYPE_ERROR);
	SDO_OCIHandleFree((dvoid*)usrhp, (ub4)OCI_HTYPE_SESSION);
	SDO_OCIHandleFree((dvoid*)stmthp, (ub4)OCI_HTYPE_STMT);

	connected = false;		
	envhp = 0;
	errhp = 0;
	srvhp = 0;
	svchp = 0;
	usrhp = 0;
	txnhp = 0;
	tdo_geometry = 0;
	tdo_ordinates = 0;
	ordinates = 0;
	stmthp = 0;
}


bool OCIConnection::Prepare(const string& stmt)
{	
	if (!connected)
		return false;
	
	ub4 size = stmt.size();
	sword status = SDO_OCIStmtPrepare(svchp, (OCIStmt *)stmthp, errhp, (text*)stmt.c_str(), (ub4)size,  
		(text*)0, (ub4)0, (ub4)OCI_NTV_SYNTAX, (ub4)OCI_DEFAULT);
	if(!CheckError(errhp, status))
		return false;

	return true;
}


bool OCIConnection::BindByPos(int pos, void* value, int size, void* indicator, int type)
{
	if (!connected)
		return false;

	OCIBind *bindp = NULL;

	sword status = SDO_OCIBindByPos(stmthp, &bindp, errhp, pos, (dvoid *)value, 
								(sb4)size, type, (dvoid *)indicator, (ub2 *)0, 
								(ub2 *)0, (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT);
	if(!CheckError(errhp, status))
		return false;
	return true;
}


bool OCIConnection::BindByName(char* name, void* value, int size, void* indicator, int type)
{
	if (!connected)
		return false;

	OCIBind *bindp = NULL;
	sword status = SDO_OCIBindByName(stmthp, &bindp, errhp, (text*)name, 
								 (sb4)-1, (dvoid *)value, (sb4)size, type, 
								 (dvoid *)indicator, (ub2 *)0, (ub2 *)0, 
								 (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT);
	if(!CheckError(errhp, status))
		return false;
	return true;
}

bool OCIConnection::BindOrdinates()
{
	OCIBind		*bnd1p = NULL;
	
	/* bind coordinate varray object */
	sword status = OCIBindByName(stmthp, &bnd1p, errhp, 
	    (text *)":ordinates", (sb4)-1, (dvoid *)0, (sb4)0, SQLT_NTY, (dvoid *)0, 
		(ub2 *)0, (ub2 *)0, (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT);
	if(!CheckError(errhp, status))
		return false;

	status = OCIBindObject(bnd1p, errhp, tdo_ordinates, (dvoid **)&ordinates, (ub4 *)0, 
	    (dvoid **)0, (ub4 *)0);
	if(!CheckError(errhp, status))
		return false;

	return true;
}


int OCIConnection::QueryType()  
{	
	ub2 querytype_;
		
	sword status = OCIAttrGet((dvoid *)stmthp, (ub4)OCI_HTYPE_STMT, (ub2 *)&querytype_,
	(ub4*)NULL, (ub4)OCI_ATTR_STMT_TYPE, (OCIError *)errhp);
	if(!CheckError(errhp, status))
		return -1;
	return (querytype_);
}


bool OCIConnection::AllocateObjectOrdinates()
{
	sword status = SDO_OCIObjectNew(envhp, errhp, svchp, OCI_TYPECODE_VARRAY, tdo_ordinates, (dvoid*)NULL, 
					OCI_DURATION_SESSION, TRUE, (dvoid**)&ordinates);

	if(!CheckError(errhp, status))
		return false;

	return true;
}


bool OCIConnection::AppendOrdinates(const double& val)
{
	
	OCINumber	oci_number;
	sword       status;
	
	status = OCINumberFromReal(errhp, (dvoid *)&(val), 
		(uword)sizeof(double),&oci_number);

	status = OCICollAppend(envhp, errhp, 
		(dvoid *) &oci_number,
		(dvoid *)0, (OCIColl *)ordinates);

	if(!CheckError(errhp, status))
		return false;

	return true;
}

bool OCIConnection::TransStart()
{
	sword status = OCITransStart (svchp, errhp, 60, OCI_TRANS_NEW); 
	if(!CheckError(errhp, status))
		return false;

	return true;
}

bool OCIConnection::TransRollback()
{
	sword status = OCITransRollback(svchp, errhp, (ub4) 0);
	if(!CheckError(errhp, status))
		return false;

	return true;
}

bool OCIConnection::TransCommit()
{
	sword status = OCITransCommit(svchp, errhp, (ub4) 0);
	if(!CheckError(errhp, status))
		return false;

	return true;
}

bool OCIConnection::Execute(const string& stm)
{
	if(!Prepare(stm))
		return false;
	
	if ((QueryType()<0) || (QueryType()==1))
		return false;  //return records (query)
	
	TransStart();
	sword status = OCIStmtExecute(svchp, stmthp, errhp, (ub4)1, (ub4)0, (OCISnapshot *)NULL, 
								  (OCISnapshot *)NULL, (ub4)OCI_DEFAULT);
	
	if(!CheckError(errhp, status))
	{
		TransRollback();
		return false;
	}

	TransCommit();
	return true; 
}


bool OCIConnection::ExecuteSDOSTM(const string& stm)  
{
	if(!Prepare(stm))
		return false;

	if(!BindOrdinates())
		return false;

	if((QueryType()<0) || (QueryType()==1))
		return false;  //return records (query)
		
	TransStart();
	sword status = OCIStmtExecute(svchp, stmthp, errhp, (ub4)1, (ub4)0, (OCISnapshot *)NULL, 
								  (OCISnapshot *)NULL, (ub4)OCI_DEFAULT);
	
	if(!CheckError(errhp, status))
	{		
		OCIObjectFree(envhp, errhp, (dvoid *)ordinates, (ub2)OCI_OBJECTFREE_FORCE);
		TransRollback();
		return false;
	}

	TransCommit();
	OCIObjectFree(envhp, errhp, (dvoid *)ordinates, (ub2)OCI_OBJECTFREE_FORCE);
	return true; 
}


bool OCIConnection::CheckError(OCIError* errhp, int status) 
{
	char errbuf[512];
	sb4 errcode = 0;
	bool returnedVal = false;
		
	if (status == OCI_ERROR)
	{
		SDO_OCIErrorGet((dvoid*)errhp, (ub4)1, (text*)NULL, &errcode, 
						(text*)errbuf, (ub4)sizeof(errbuf), OCI_HTYPE_ERROR);
		errorMess = errbuf;
		return false;
	}

	switch (status)
	{ 
		case OCI_SUCCESS:
			errorMess = "Success!";
			returnedVal = true;
			break;

		case OCI_SUCCESS_WITH_INFO:
			errorMess = "Success with information!";
			returnedVal = true;
			break;

		case OCI_NEED_DATA:
			errorMess = "Need data!";
			break;
		
		case OCI_NO_DATA:
			errorMess = "No data!";
			break;

		//An invalid handle was passed as a parameter or a user callback is passed an
		//invalid handle or invalid context. No further diagnostics are available.
		case OCI_INVALID_HANDLE:
			errorMess = "Invalid handle!";
			break;

		case OCI_STILL_EXECUTING:
			errorMess = "Still executing!";
			break;

		case OCI_CONTINUE:
			errorMess = "Continue!";
			break;
		default:
			break;
	}

	return returnedVal;
}
