/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include <libsysactivity/libsysactivity.h>

void print_network_info(struct sa_network_interface* dev) {
#ifdef SA_NETWORK_INTERFACE_NAME
	printf("%s\n received: ", dev->name);
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_BYTES
	printf("%"PRIu64" bytes, ", dev->received_bytes);
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_PACKETS
	printf("%"PRIu64" packets, ", dev->received_packets);
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_ERRORS
	printf("%"PRIu64" errors, ", dev->received_errors);
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_DROP
	printf("%"PRIu64" dropped, ", dev->received_drop);
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_FIFO
	printf("%"PRIu64" fifo, ", dev->received_fifo);
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_MULTICAST
	printf("%"PRIu64" multicast, ", dev->received_multicast);
#endif
	printf("\n sent: ");
#ifdef SA_NETWORK_INTERFACE_SENT_BYTES
	printf("%"PRIu64" bytes, ", dev->sent_bytes);
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_PACKETS
	printf("%"PRIu64" packets, ", dev->sent_packets);
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_ERRORS
	printf("%"PRIu64" errors, ", dev->sent_errors);
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_DROP
	printf("%"PRIu64" dropped, ", dev->sent_drop);
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_FIFO
	printf("%"PRIu64" fifo, ", dev->sent_fifo);
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_MULTICAST
	printf("%"PRIu64" multicast, ", dev->sent_multicast);
#endif
	printf("\n");
}

void test_network_info(struct sa_network_interface* dev) {
#ifdef SA_NETWORK_INTERFACE_NAME
	size_t len = strnlen(dev->name, SA_SWAP_NAME);
	if (len == 0) {
		printf("\nERROR: The length of the device name is zero\n");
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_BYTES
#ifdef SA_NETWORK_INTERFACE_RECEIVED_PACKETS
	if (dev->received_packets > dev->received_bytes) {
		printf("\nERROR: The number of received packets is bigger than the received bytes\n");
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_MULTICAST
	if (dev->received_packets > dev->received_multicast) {
		printf("\nERROR: The number of received packets is bigger than the received multicasts\n");
		exit(EXIT_FAILURE);
	}
#endif
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_BYTES
#ifdef SA_NETWORK_INTERFACE_SENT_PACKETS
	if (dev->sent_packets > dev->sent_bytes) {
		printf("\nERROR: The number of sent packets is bigger than the sent bytes\n");
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_MULTICAST
	if (dev->sent_packets > dev->sent_multicast) {
		printf("\nERROR: The number of sent packets is bigger than the sent multicasts\n");
		exit(EXIT_FAILURE);
	}
#endif
#endif
}

void* get_network_info(void* arg) {
	int ret;
	struct sa_network_interface* interfaces = NULL;
#ifdef SA_OPEN_NETWORK
	ret = sa_open_network();
	if (ret != 0) {
		printf("sa_open_network(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif

	struct sa_network_interface dev;
#ifdef __FreeBSD_kernel__ // gcc -E -dM -x c /dev/null
	ret = sa_get_network_interface("lo0", &dev);
#else
	ret = sa_get_network_interface("lo", &dev);
#endif
	if (ret != 0) {
		printf("sa_get_network_interface(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
	print_network_info(&dev);
	printf("\n");
	test_network_info(&dev);

	int i;
	uint16_t number, written;
	for (i = 0; i < 6; i++) {
		ret = sa_count_network_interfaces(&number);
		if (ret != 0 || number < 2) {
			printf("sa_count_network_interfaces(): %s\n", strerror(ret));
			exit(EXIT_FAILURE);
		}
		printf("there are %d network interfaces\n", number);

		if (number > 0) {
			uint16_t j;
			interfaces = (struct sa_network_interface*) realloc(interfaces, number
					* sizeof(struct sa_network_interface));
			written = 0;
			ret = sa_get_network_interfaces(interfaces, number, &written);
			if (ret != 0 || written < 2) {
				printf("sa_get_network_interfaces(): %s\n", strerror(ret));
				exit(EXIT_FAILURE);
			}
			for (j = 0; j < written; j++) {
				print_network_info(&interfaces[j]);
				test_network_info(&interfaces[j]);
			}
		}
		printf("\n");
		sleep(1);
	}
	if (interfaces != NULL)
		free(interfaces);
#ifdef SA_CLOSE_NETWORK
	ret = sa_close_network();
	if (ret != 0) {
		printf("sa_close_network(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_OPEN_NETWORK
	ret = sa_open_network();
	if (ret != 0) {
		printf("sa_open_network(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
#ifdef SA_CLOSE_NETWORK
	ret = sa_close_network();
	if (ret != 0) {
		printf("sa_close_network(): %s\n", strerror(ret));
		exit(EXIT_FAILURE);
	}
#endif
	return NULL;
}

int main() {
	pthread_t thread1;
	struct timespec delay;
	delay.tv_sec = 0;
	delay.tv_nsec = 500000000;

	pthread_create(&thread1, NULL, get_network_info, NULL);
	nanosleep(&delay, NULL);
	get_network_info(NULL);

	return EXIT_SUCCESS;
}
