/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef SA_PROCESS
#warning You should not include this file directly from your program.
#endif

/**
 * \defgroup process Process interface
 * @{
 */
#ifndef SA_COMMON_PROCESS_H_
#define SA_COMMON_PROCESS_H_

/** \enum sa_process_state
 * The different types of states of a process.
 */
SA_EXPORT enum sa_process_state {
	RUNNING = 0, //!< The process is running.
	SLEEPING = 1, //!< The process is sleeping.
	SLEEPING_UNINTERRUPTIBLE = 2, //!< The process is an uninterruptible sleep
	ZOMBIE = 3, //!< It has completed execution but still has an entry in the process table.
	STOPPED = 4,
//!< The process is traced or stopped.
};

/** \struct sa_process process.h
 * This structure gathers statistics of a process.
 */
SA_EXPORT struct sa_process {
#ifdef SA_PROCESS_PID
	pid_t pid; //!< The id of the process
#endif
#ifdef SA_PROCESS_UID
	uid_t uid; //!< The id of the user running this process
#endif
#ifdef SA_PROCESS_GID
	gid_t gid; //!< Id of the user's group running this process
#endif
#ifdef SA_PROCESS_FILENAME
	char filename[SA_PROCESS_FILENAME]; //!< The name of the running file.
#endif
#ifdef SA_PROCESS_CMDLINE
	char cmdline[SA_PROCESS_CMDLINE]; //!< The command executed to launch this process.
#endif
#ifdef SA_PROCESS_STATE
	enum sa_process_state state; //!< The current state of the process.
#endif
#ifdef SA_PROCESS_PARENT_PID
	pid_t parent_pid; //!< The id of its parent
#endif
#ifdef SA_PROCESS_PGRP
	pid_t pgrp; //!< The group id the process
#endif
#ifdef SA_PROCESS_SID
	pid_t sid; //!< Session id of this process
#endif
#ifdef SA_PROCESS_TTY
	pid_t tty; //!< The id of the tty this process is running on.
#endif
#ifdef SA_PROCESS_USER_TIME // TODO Explain these better
	uint64_t user_time; //!< User mode jiffies
#endif
#ifdef SA_PROCESS_SYS_TIME
	uint64_t sys_time; //!< Kernel mode jiffies
#endif
#ifdef SA_PROCESS_NICE
	int8_t nice; //!< Priority of this process.
#endif
#ifdef SA_PROCESS_THREADS
	uint32_t threads; //!< Number of threads
#endif
#ifdef SA_PROCESS_START_TIME
	uint64_t start_time; //!< Process start time in hundredths of second since system boot
#endif
#ifdef SA_PROCESS_VM_SIZE
	uint32_t vm_size; //!< Virtual memory size
#endif
#ifdef SA_PROCESS_RSS
	uint32_t rss; //!< Resident Set memory Size
#endif
};

#ifdef SA_OPEN_PROCESS
/**
 * Prepares the resources needed for retrieving process statistics. This function exists (and is needed) only when SA_OPEN_PROCESS is defined.
 * @return If successful 0 is returned, otherwise an error code is returned. If the operative system is not supported the return value will be ENOTSUP.
 * @see sa_close_process()
 */
SA_EXPORT int sa_open_process();
#endif

#ifdef SA_CLOSE_PROCESS
/**
 * This function closes the resources used for retrieving process statistics. You should call it even when there was a previous error in another function of this API. This function exists (and is needed) only when SA_CLOSE_PROCESS is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.
 * @see sa_open_process()
 */
SA_EXPORT int sa_close_process();
#endif

/**
 * Gives the total number of processes on the systems.
 * @param number The number will be stored here
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_count_processes(pid_t* number);

/**
 * Gets a list of the current processes ids in the system. The array will be fully populated even if it's not big enough (but ENOMEM is returned).
 * @param dst A pointer to the array to populate.
 * @param dst_size The number of pid_t that fits in the array.
 * @param written The amount of pids written is stored here.
 * @return If successful 0 is returned, otherwise an error code is returned.
 * @since 0.4.1
 */
SA_EXPORT int sa_get_processes_ids(pid_t* dst, pid_t dst_size, pid_t* written);

/**
 * Retrieves statistics from a process identified by its pid.
 * @param pid The pid of the process.
 * @param dst Where the statistics will be stored.
 * @return If successful 0 is returned, otherwise an error code is returned. ESRCH is returned when the requested process was not found.
 */
SA_EXPORT int sa_get_process(pid_t pid, struct sa_process* dst);

/**
 * Retrieves statistics about all the processes on the system.
 * @param dst A buffer where the statistics will be stored.
 * @param dst_size The number of process that fits in the dst buffer. If it's not big enough dst will be filled but ENOMEM will be returned.
 * @param written The amount of processes written.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_get_processes(struct sa_process* dst, pid_t dst_size, pid_t* written);

/*@}*/
#endif /* SA_COMMON_PROCESS_H_ */
