/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.adaptor.interceptor;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import javax.management.MBeanServer;

import mx4j.log.Logger;
import mx4j.tools.adaptor.rmi.UnmarshallingMBeanServer;
import mx4j.util.MethodTernaryTree;
import mx4j.util.Utils;

/**
 * @version $Revision: 1.3 $
 */
public final class InvokerAdaptorInterceptor extends AdaptorInterceptor
{
   private MethodTernaryTree m_methods = new MethodTernaryTree();
   private MBeanServer m_unmarshaller;

   public void setMBeanServer(MBeanServer server)
   {
      super.setMBeanServer(server);
      // A decorator for the MBeanServer instance that will take care of unmarshalling RMI arguments.
      // This operation is more difficult to do in the interceptors, as the invocation is totally untyped.
      m_unmarshaller = new UnmarshallingMBeanServer(server);
   }

   public String getType()
   {
      return "invoker";
   }

   /**
    * Cannot change the status of this attribute, this interceptor is always enabled
    */
   public void setEnabled(boolean value)
   {
   }

   /**
    * This interceptor is always enabled
    */
   public boolean isEnabled()
   {
      return true;
   }

   public InvocationResult invoke(Invocation invocation) throws Exception
   {
      // Save some logging
      return doInvoke(invocation);
   }

   protected InvocationResult doInvoke(Invocation invocation) throws Exception
   {
      // Here I have to call the MBeanServer

      Logger logger = getLogger();
      boolean debug = logger.isEnabledFor(Logger.DEBUG);
      if (debug) logger.debug("Last interceptor, invoking MBeanServer");

      // Extract method name from invocation
      String name = invocation.getMethodName();
      if (debug) logger.debug("Method name is: " + name);

      // Extract signature from invocation
      String[] signature = invocation.getSignature();
      Class[] params = Utils.loadClasses(Thread.currentThread().getContextClassLoader(), signature);
      if (debug) logger.debug("Signature is: " + Arrays.asList(signature));

      // Extract arguments
      Object[] args = invocation.getArguments();
      if (debug) logger.debug("Arguments are: " + Arrays.asList(args));

      try
      {
         if (m_unmarshaller == null) throw new IllegalStateException("Target MBeanServer is not set. Either call setMBeanServer or register this MBean");
         Method m = getMBeanServerMethod(name, signature, params);
         Object result = invokeMBeanServerMethod(m, args);

         InvocationResult invocationResult = new InvocationResult();
         invocationResult.setResult(result);

         return invocationResult;
      }
      catch (InvocationTargetException x)
      {
         // Unwrap invocation target exceptions
         Throwable t = x.getTargetException();
         if (t instanceof Exception)
            throw (Exception)t;
         else
            throw (Error)t;
      }
   }

   private Method getMBeanServerMethod(String name, String[] signature, Class[] params) throws NoSuchMethodException
   {
      // Double checked locking does not work in Java :(

      synchronized (m_methods)
      {
         Method m = (Method)m_methods.get(name, signature);
         if (m == null)
         {
            m = MBeanServer.class.getMethod(name, params);
            m_methods.put(name, signature, m);
         }
         return m;
      }
   }

   private Object invokeMBeanServerMethod(Method method, Object[] params) throws Exception
   {
      return method.invoke(m_unmarshaller, params);
   }
}
