/**
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.openmbean;

import java.io.Serializable;
import java.util.Collections;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

import javax.management.MBeanAttributeInfo;

/**
 *
 * @author <a href="mailto:warrenm@users.sourceforge.net">Warren Mira</a>
 * @author <a href="mailto:shadow12@users.sourceforge.net">Bronwen Cassidy</a>
 * @version $Revision: 1.11 $
 */
public class OpenMBeanAttributeInfoSupport extends MBeanAttributeInfo implements OpenMBeanAttributeInfo, Serializable
{
	private static final long serialVersionUID = -4867215622149721849L;

	private OpenType openType;
	private Object defaultValue = null;
	private Set legalValues = null;
	private Comparable minValue = null;
	private Comparable maxValue = null;


    private transient int hashCode = 0;
    private transient String toStringName = null;

    /**
     * Constructs an OpenMBeanAttributeInfoSupport which is an <code>OpenMBean</code>
     * with the specified name,<code>OpenType</code>, description and
     * and the specified read/write access properties.
     *
     * @param name The name of the attribute. Cant be a null.
     * @param description The description of the attribute. Cant be null.
     * @param openType The <code>OpenType</code> representation
     * @param isReadable <code>true</code> if the attribute has a getter exposed.
     * @param isWritable <code>true</code> if the attribute has a setter exposed
     * @param isIs true if the attribute's getter is of the form isXXX.
     * @throws IllegalArgumentException If name or description are null or empty string, or openType is null 
     *
     */
	public OpenMBeanAttributeInfoSupport(String name, String description, OpenType openType, boolean isReadable, boolean isWritable, boolean isIs)
	{
		super(name, openType == null ? "" : openType.getClassName(), description, isReadable, isWritable, isIs);
        if ( openType == null ) 
            throw new IllegalArgumentException("OpenType can't be null");
        if ( name == null || name.length() == 0 || name.trim().equals("") ) 
            throw new IllegalArgumentException("name can't be null or empty");
        if ( description == null || description.length() == 0 || description.trim().equals("") ) 
            throw new IllegalArgumentException("description can't be null or empty");
        
        this.openType = openType;

	}


    
    /**
     * Constructs an OpenMBeanAttributeInfoSupport which is an <code>OpenMBean</code>
     * with the specified name,<code>OpenType</code>, description and
     * and the specified read/write access properties and the default value.
     *
     * @param name The name of the attribute. Cant be a null.
     * @param description The description of the attribute. Cant be null.
     * @param openType The <code>OpenType</code> representation
     * @param isReadable <code>true</code> if the attribute has a getter exposed.
     * @param isWritable <code>true</code> if the attribute has a setter exposed
     * @param isIs true if the attribute's getter is of the form isXXX.
     * @param defaultValue The default value of the attribute. Must be a    
     *              valid value for the <code>OpenType</code> specified. 
     *              <code>null</code> if no default value is set. 
     *              <code>ArrayType</code> and <code>TabularType</code> are 
     *              not supported for a default value.
     * @throws IllegalArgumentException If name or description are null 
     *              or empty string, or openType is null 
     * @throws OpenDataException If defaultValue is not of valid type; and
     *              default value not supported for ArrayType and TabularType 
     *              (should be null)
     *
     */
	public OpenMBeanAttributeInfoSupport(String name, String description, OpenType openType, boolean isReadable, boolean isWritable, boolean isIs, Object defaultValue) throws OpenDataException
	{
		this(name, description, openType, isReadable, isWritable, isIs);
        
			//then should be a proper value
		if ( openType instanceof ArrayType || 
			 openType instanceof TabularType ) 
		{ 
				if ( defaultValue != null ) 
					throw new OpenDataException("defaultValue is not " +
					 "supported for ArrayType and TabularType. Should be null");
		} 
        
		if ( defaultValue != null && !openType.isValue(defaultValue) ) 
				throw new OpenDataException("defaultValue is not " +
					"a valid value for the given OpenType");
                    
		this.defaultValue = defaultValue;
	}

    /**
     * Constructs an OpenMBeanAttributeInfoSupport which is an <code>OpenMBean</code>
     * with the specified name,<code>OpenType</code>, description and
     * and the specified read/write access properties and the default value.
     *
     * @param name The name of the attribute. Cant be a null.
     * @param description The description of the attribute. Cant be null.
     * @param openType The <code>OpenType</code> representation
     * @param isReadable <code>true</code> if the attribute has a getter exposed.
     * @param isWritable <code>true</code> if the attribute has a setter exposed
     * @param isIs true if the attribute's getter is of the form isXXX.
     * @param defaultValue The default value of the attribute. Must be a valid 
     *              value for the <code>OpenType</code> specified. 
     *              <code>null</code> if no default value is set. 
     *              <code>ArrayType</code> and <code>TabularType</code> are
     *              not supported for a default value.
     * @param legalValues each value must be a valid 
     *              value for the <code>OpenType</code> specified. 
     *              <code>null</code> if no default value is set. 
     *              <code>ArrayTypr</code> and <code>TabularType</code> are
     *              not supported for a legal value (should be null).
     * @throws IllegalArgumentException If name or description are null or 
     *              empty string, or openType is null 
     * @throws OpenDataException If defaultValue is not of valid type; and
     *              default value not supported for ArrayType and TabularType 
     *              (should be null)
     *
     */
	public OpenMBeanAttributeInfoSupport(String name, String description, OpenType openType, boolean isReadable, boolean isWritable, boolean isIs, Object defaultValue, Object[] legalValues) throws OpenDataException
	{
		this(
			name,
			description,
			openType,
			isReadable,
			isWritable,
			isIs,
			defaultValue);

		if (openType instanceof ArrayType || openType instanceof TabularType)
		{
			if (legalValues != null && legalValues.length > 0)
				throw new OpenDataException(
					"legalValues isn't allowed "
						+ "for ArrayType and TabularType. Should be null or empty "
						+ "array");

		}
		else if (legalValues != null && legalValues.length > 0)
		{
			Set tmpSet = new HashSet(legalValues.length, 1.0f);

			for (int i = 0; i < legalValues.length; i++)
			{
				Object lv = legalValues[i];
				if (openType.isValue(lv))
				{
					tmpSet.add(lv);
				}
				else
				{
					throw new OpenDataException(
						"An Entry in the set "
							+ "of legalValues is not a valid value for the given "
							+ "opentype");
				}
			}

			if (defaultValue != null && !tmpSet.contains(defaultValue))
			{
				throw new OpenDataException("The legal value set must include the default value");
			}

			this.legalValues = Collections.unmodifiableSet(tmpSet);
		}
	}

    /**
     * Constructs an OpenMBeanAttributeInfoSupport which is an <code>OpenMBean</code>
     * with the specified name,<code>OpenType</code>, description and
     * and the specified read/write access properties,default value and legal
     * values as an array
     *
     * @param name The name of the attribute. Cant be a null.
     * @param description The description of the attribute. Cant be null.
     * @param openType The <code>OpenType</code> representation
     * @param isReadable <code>true</code> if the attribute has a getter exposed.
     * @param isWritable <code>true</code> if the attribute has a setter exposed
     * @param isIs true if the attribute's getter is of the form isXXX.
     * @param defaultValue The default value of the attribute. 
     *              Must be a valid value for the <code>OpenType</code> 
     *              specified. <code>null</code> if no default
     * @param minValue must be a valid <code>OpenType</code> specified for 
     *              this attribute. Can be null, which means no minimal value 
     *              for the attribute.
     * @param maxValue must be a valid <code>OpenType</code> specified for 
     *              this attribute. Can be null, which means no miximal value 
     *              for the attribute.
     * @throws IllegalArgumentException If name or description 
                    are null or empty string, or openType is null 
     * @throws OpenDataException If defaultValue is not of valid type; 
     *              and default value not supported for ArrayType 
     *              and Tabular (should be null). If minValue and maxValue is
     *              not a valid value for specified <code>OpenType</code>.
     *              If minValue and maxValue are non-null and 
     *              <code>OpenType</code> and minValue.compareTo(maxValue) > 0
     *              is true, or both defaultValue and minValue are non-null
     *              and minValue.compareTo(defaultValue) >0 is true,
     *              or both defaultValue and maxValue are non-null and 
     *              defaultValue.compareTo(maxValue) >0 is true
     *
     */
	public OpenMBeanAttributeInfoSupport(String name, String description, OpenType openType, boolean isReadable, boolean isWritable, boolean isIs, Object defaultValue, Comparable minValue, Comparable maxValue) throws OpenDataException
	{
		this(name, description, openType, isReadable, isWritable, isIs, defaultValue);

		if ( minValue != null ) 
			if ( !openType.isValue(minValue) ) 
				throw new OpenDataException("minValue is not a valid " +
				"value for the specified openType");
		if ( maxValue != null ) 
			if ( !openType.isValue(maxValue) ) 
				throw new OpenDataException("maxValue is not a valid " +
				"value for the specified openType");
            
		if ( minValue != null && maxValue != null ) 
			if ( minValue.compareTo(maxValue) > 0 ) 
				throw new OpenDataException("minValue and/or maxValue is " +
				"invalid: minValue is greater than maxValue");
		if ( defaultValue != null && minValue != null ) 
			if ( minValue.compareTo((Comparable)defaultValue) > 0 ) 
				throw new OpenDataException("defaultvalue and/or minValue is invalid: minValue is greater than defaultValue");
		if ( defaultValue != null && maxValue != null ) 
			if ( ((Comparable)defaultValue).compareTo(maxValue) > 0 ) 
				throw new OpenDataException("defaultvalue and/or maxValue is invalid: defaultValue is greater than maxValue");
                


		this.minValue = minValue;
		this.maxValue = maxValue;

	}
    
    /**
     * Returns the <code>OpenType</code> of this attribute
     *
     */
	public OpenType getOpenType()
	{
		return openType;
	}

    /**
     * Returns the default value, if specified, null otherwise
     *
     */
	public Object getDefaultValue()
	{
		return defaultValue;
	}

    /**
     * Returns the legalValues as a <code>Set</code>
     *
     */
	public Set getLegalValues()
	{
		return legalValues;
	}

    
    /**
     * Returns the minValue
     *
     *
     */
	public Comparable getMinValue()
	{
		return minValue;
	}

    /**
     * Returns the maxValue
     *
     */
	public Comparable getMaxValue()
	{
		return maxValue;
	}

    /** 
     * <code>true</code> if has defaultValue, false otherwise
     *
     */
	public boolean hasDefaultValue()
	{

		return defaultValue != null;
	}

    /**
     * <code>true</code> if has a legalValues allowed, false otherwise.
     *
     */
	public boolean hasLegalValues()
	{
		return legalValues != null;
	}

    /**
     * <code>true</code> if has a minValue, false otherwise.
     *
     */
	public boolean hasMinValue()
	{
		return minValue != null;
	}

    /**
     * <code>true</code> if has a maxValue, false otherwise.
     *
     */
	public boolean hasMaxValue()
	{
		return maxValue != null;
	}

    
    /**
     * Test wether obj is a valid value for this attribute  
     *
     * @param obj The object being tested 
     * @return true If a valid value
     */ 
	public boolean isValue(Object obj)
	{
        if ( defaultValue != null ) 
        {
            if ( openType.isValue(obj) ) return true;     
        } 
        else //defaultValue is null 
        {
            //if obj is null- true
            if ( obj == null ) return true;
        }
        
        return false;
            
	}

    /**
     * Compares the give <code>Object</code> for equality with this instance.
     *
     * <p>
     * The operation returns true if and only if the following statements
     * are all true:
     * <ul>
     *  <li>obj is not null</li>
     *  <li>obj also implements OpenMBeanAttributeInfo</li>
     *  <li>their names are equals</li>
     *  <li>their open types are equal</li>
     *  <li>access properties (isReadable, isWritable, isIs) are equal</li>
     *  <li>default,min,max and legal values are equal</li>
     * </ul> 
     * 
     * @return boolean true if the above conditions are met
     *
     */
	public boolean equals(Object obj)
	{
        if ( obj == this ) return true;
        //obj should not be null
        if ( obj == null ) return false;
        //obj should implement OpenMBeanAttributeInfo
        if ( !(obj instanceof OpenMBeanAttributeInfo) ) return false;
       
        //this wouldn't fail
        OpenMBeanAttributeInfo other = (OpenMBeanAttributeInfo)obj;
        
        //name should be equal     
        if ( !getName().equals(other.getName()) ) return false;
        
        //opentype should be equal
        if ( !getOpenType().equals(other.getOpenType()) ) return false;
        
        //access properties should be equal
        if ( !( isReadable() == other.isReadable()) ) return false;
        if ( !( isWritable() == other.isWritable()) ) return false;
        if ( !( isIs() == other.isIs()) ) return false;

        //default, min, max and legalvalues are equal
        if ( hasDefaultValue() ) 
        {
            if ( !getDefaultValue().equals(other.getDefaultValue()) ) return false;
        } 
        else //no defaultvalue 
        {
            //other should also not have default value
            if ( other.hasDefaultValue() ) return false;
        } 

        //min
        if ( hasMinValue() ) 
        {
            if ( !minValue.equals(other.getMinValue()) ) return false;
        } 
        else 
        {
            if ( other.hasMinValue() ) return false;
        }

        //max
        if ( hasMaxValue() ) 
        {
            if ( !maxValue.equals(other.getMaxValue()) ) return false;
        } 
        else 
        {
            if ( other.hasMaxValue() ) return false;
        }

        //legalValues
        if ( hasLegalValues() ) 
        {
           if ( !legalValues.equals(other.getLegalValues() ) ) return false; 
        } 
        else
        {
            if ( other.hasLegalValues() ) return false;
        }
   
        return true;
        
        
 
	}


    
    /**
     * Computes the hashCode of this <code>OpenMBeanAttributeInfo</code>
     *
     * @return int The hashCode value
     */
	public int hashCode()
	{
        if ( hashCode == 0 ) 
        {
            hashCode = hashCode(this);
        }
        return hashCode;

	}

    /**
     * Returns a string representation of this <code>OpenMBeanAttributeInfo</code> instance. 
     *
     * @return String The representation as string
     */
	public String toString()
	{
        
        if ( toStringName == null ) {
            StringBuffer sb = new StringBuffer(getClass().getName());
            sb.append("(name=");
            sb.append(getName());
            sb.append(", opentype=");
            sb.append(openType.toString());
            sb.append(", defaultValue=");
            sb.append(hasDefaultValue() ? getDefaultValue().toString() : "null");
            sb.append(", minValue=");
            sb.append(hasMinValue() ? getMinValue().toString() : "null");
            sb.append(", maxValue=");
            sb.append(hasMaxValue() ? getMaxValue().toString() : "null");
            sb.append(", legalValues=");
            sb.append(hasLegalValues() ? getLegalValues().toString() : "null");
            sb.append(")");
            
            toStringName = sb.toString();
        }
        
        return toStringName;
        
	}

	private int hashCode(OpenMBeanAttributeInfoSupport info)
	{
		int result = info.getName().hashCode();
		result += info.getOpenType().hashCode();
		result += (info.hasDefaultValue() == false) ? 0 : info.getDefaultValue().hashCode();
		result += (info.hasLegalValues() == false) ? 0 : hashCode(info.getLegalValues());
		result += (info.hasMinValue() == false) ? 0 : info.getMinValue().hashCode();
		result += (info.hasMaxValue() == false) ? 0 : info.getMaxValue().hashCode();
		return result; 
	}
	
	private int hashCode(Set legalvalues)
	{
		int result = 0;
		Iterator i = legalvalues.iterator();
		while (i.hasNext())
		{
			Object v = i.next();
			result += v.hashCode();
		}
		return result;
	}
}
