#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util::GMP qw/next_prime prev_prime surround_primes
                              next_twin_prime powint addint/;
my $extra = defined $ENV{EXTENDED_TESTING} && $ENV{EXTENDED_TESTING};

plan tests => 2 + 3*2 + 6 + 1 + 2 + 1 + 2 + 7 + 3*$extra + 2;

my @small_primes = qw/
2 3 5 7 11 13 17 19 23 29 31 37 41 43 47 53 59 61 67 71 73 79 83 89 97
101 103 107 109 113 127 131 137 139 149 151 157 163 167 173 179 181 191 193 197
199 211 223 227 229 233 239 241 251 257 263 269 271 277 281 283 293 307 311 313
317 331 337 347 349 353 359 367 373 379 383 389 397 401 409 419 421 431 433 439
443 449 457 461 463 467 479 487 491 499 503 509 521 523 541 547 557 563 569 571
577 587 593 599 601 607 613 617 619 631 641 643 647 653 659 661 673 677 683 691
701 709 719 727 733 739 743 751 757 761 769 773 787 797 809 811 821 823 827 829
839 853 857 859 863 877 881 883 887 907 911 919 929 937 941 947 953 967 971 977
983 991 997
1009 1013 1019 1021 1031 1033 1039 1049 1051 1061 1063 1069 1087 1091 1093 1097
1103 1109 1117 1123 1129 1151 1153 1163 1171 1181 1187 1193 1201 1213 1217 1223
1229 1231 1237 1249 1259 1277 1279 1283 1289 1291 1297 1301 1303 1307 1319 1321
1327 1361 1367 1373 1381 1399 1409 1423 1427 1429 1433 1439 1447 1451 1453 1459
1471 1481 1483 1487 1489 1493 1499 1511 1523 1531 1543 1549 1553 1559 1567 1571
1579 1583 1597 1601 1607 1609 1613 1619 1621 1627 1637 1657 1663 1667 1669 1693
1697 1699 1709 1721 1723 1733 1741 1747 1753 1759 1777 1783 1787 1789 1801 1811
1823 1831 1847 1861 1867 1871 1873 1877 1879 1889 1901 1907 1913 1931 1933 1949
1951 1973 1979 1987 1993 1997 1999 2003 2011 2017 2027 2029 2039 2053 2063 2069
2081 2083 2087 2089 2099 2111 2113 2129 2131 2137 2141 2143 2153 2161 2179 2203
2207 2213 2221 2237 2239 2243 2251 2267 2269 2273 2281 2287 2293 2297 2309 2311
2333 2339 2341 2347 2351 2357 2371 2377 2381 2383 2389 2393 2399 2411 2417 2423
2437 2441 2447 2459 2467 2473 2477 2503 2521 2531 2539 2543 2549 2551 2557 2579
2591 2593 2609 2617 2621 2633 2647 2657 2659 2663 2671 2677 2683 2687 2689 2693
2699 2707 2711 2713 2719 2729 2731 2741 2749 2753 2767 2777 2789 2791 2797 2801
2803 2819 2833 2837 2843 2851 2857 2861 2879 2887 2897 2903 2909 2917 2927 2939
2953 2957 2963 2969 2971 2999 3001 3011 3019 3023 3037 3041 3049 3061 3067 3079
3083 3089 3109 3119 3121 3137 3163 3167 3169 3181 3187 3191 3203 3209 3217 3221
3229 3251 3253 3257 3259 3271 3299 3301 3307 3313 3319 3323 3329 3331 3343 3347
3359 3361 3371 3373 3389 3391 3407 3413 3433 3449 3457 3461 3463 3467 3469 3491
3499 3511 3517 3527 3529 3533 3539 3541 3547 3557 3559 3571
/;


{
  my @sp = (undef, @small_primes, 3581);
  my ($pi, $ni) = (0, 1);
  my @prevexp = map { $pi++ if $_ >  $sp[$pi+1]; $sp[$pi]; } (0 .. 3572);
  my @nextexp = map { $ni++ if $_ >= $sp[$ni  ]; $sp[$ni]; } (0 .. 3572);
  my @prev = map { prev_prime($_) } (0..3572);
  my @next = map { next_prime($_) } (0..3572);
  is_deeply( \@prev, \@prevexp, "prev_prime 0..3572" );
  is_deeply( \@next, \@nextexp, "next_prime 0..3572" );
}

my %primegaps = (
 19609 => 52,
 360653 => 96,
 2010733 => 148,
);

while (my($base, $range) = each (%primegaps)) {
  is( next_prime($base), $base+$range, "next prime of $base is $base+$range" );
  is( prev_prime($base+$range), $base, "prev prime of $base+$range is $base" );
}

is( next_prime(19608), 19609, "next prime of 19608 is 19609" );
is( next_prime(19610), 19661, "next prime of 19610 is 19661" );
is( next_prime(19660), 19661, "next prime of 19660 is 19661" );
is( prev_prime(19662), 19661, "prev prime of 19662 is 19661" );
is( prev_prime(19660), 19609, "prev prime of 19660 is 19609" );
is( prev_prime(19610), 19609, "prev prime of 19610 is 19609" );

is( prev_prime(2), undef, "Previous prime of 2 returns undef" );

# Turns out the testing of prev/next from 0-3572 still misses some cases.
{
  my @next = map { next_prime($_) } 2010733 .. 2010880;
  my @prev = map { prev_prime($_) } 2010734 .. 2010881;
  is_deeply(\@next,[(2010881)x@next],"next_prime(2010733..2010880) = 2010881");
  is_deeply(\@prev,[(2010733)x@prev],"prev_prime(2010734..2010881) = 2010733");
}
# Similar test case to 2010870, where m=0 and next_prime is at m=1
is(next_prime(1234567890), 1234567891, "next_prime(1234567890) == 1234567891)");

is( next_prime('87567547898934657346596012842304861297462937562349058673456'),
               '87567547898934657346596012842304861297462937562349058673779',
    "next_prime(8756....73456) = 8756....73779");
is( prev_prime('1353175074828899034888345292712068768032725991919855436631156'),
               '1353175074828899034888345292712068768032725991919855436630917',
    "prev_prime(1353....31156) = 1353....30917");


##########################################
is_deeply([surround_primes(0)], [undef,2], "surround_primes(2)");
is_deeply([surround_primes(2)], [undef,1], "surround_primes(2)");
is_deeply([surround_primes("29384928409238")], [79,45], "surround_primes(29384928409238)");
is_deeply([surround_primes("18446744073709551616")], [59,13], "surround_primes(2^64)");
is_deeply([surround_primes("36893488147419103273")], [90,90], "surround_primes(2^65+41)");
# Test second arg
is_deeply([surround_primes("36893488147419103273",89)], [90,90], "surround_primes(2^65+41,89)");
is_deeply([surround_primes("36893488147419103273",90)], [90,0], "surround_primes(2^65+41,90)");
if ($extra) {
  my $c163 = "3254185929142547441117000456865810587301677179676494144227638178204588790016877642497284992010196102315442455595356723557223592608572003000275424902281588892769763";
  is_deeply([surround_primes($c163,200)], [8776,4916], "surround_primes(d163,200)");
  my $c85 = "2642497284992010196102315442455595356723557223592608572003000275424902281588892769763";
  is_deeply([surround_primes($c85)], [32,456], "surround_primes(c85)");
  is_deeply([surround_primes($c85,200)], [32,0], "surround_primes(c85,200)");
}
##########################################
{
  # Only a subset of the results
  my @a124001 = (qw/2 1 1 19 7 151 37 139 37 7 277 817 61 1267 97 2371 1549 19 619 97 391 409 649 5527 2731 559 949 427 601 2797 1681 7189 2449 6751 7597 8419 16879 871 5569 10327 16111 2131 6121 23329 5179 4249 2641 2257 3997 8281 18307 7537 41347 25831 3397 7687 6637 7381 16597 9091 23599 27319 4711 22249 6517 7579 4051 1777 12949 38119 17887 3319 38671 829 19219 24721 4489 45469 2581 30511 37759/);
  my(@got, @exp);
  for my $i (0 .. $#a124001) {
    my $n = powint(10,$i);
    push @exp, addint($n, $a124001[$i]);
    push @got, next_twin_prime($n);
  }
  is_deeply(\@got, \@exp, "twin primes 10^x");
}
{
  my @a113275 = (qw/3 5 17 41 71 311 347 659 2381 5879 13397 18539 24419 62297 187907 687521 688451 850349 2868959 4869911 9923987 14656517 17382479 30752231 32822369 96894041 136283429 234966929 248641037 255949949 390817727 698542487 2466641069 4289385521 19181736269 24215097497 24857578817 40253418059 42441715487 43725662621 65095731749 134037421667 198311685749 223093059731 353503437239 484797803249 638432376191 784468515221 794623899269 1246446371789 1344856591289 1496875686461 2156652267611 2435613754109 4491437003327 13104143169251 14437327538267 18306891187511 18853633225211 23275487664899 23634280586867 38533601831027 43697538391391 56484333976919 74668675816277 116741875898981 136391104728629 221346439666109 353971046703347 450811253543219 742914612256169 1121784847637957 1149418981410179 2543288406389231 2797282815481499/);
  my @a036062 = (qw/5 11 29 59 101 347 419 809 2549 6089 13679 18911 24917 62927 188831 688451 689459 851801 2870471 4871441 9925709 14658419 17384669 30754487 32825201 96896909 136286441 234970031 248644217 255953429 390821531 698547257 2466646361 4289391551 19181742551 24215103971 24857585369 40253424707 42441722537 43725670601 65095739789 134037430661 198311695061 223093069049 353503447439 484797813587 638432386859 784468525931 794623910657 1246446383771 1344856603427 1496875698749 2156652280241 2435613767159 4491437017589 13104143183687 14437327553219 18306891202907 18853633240931 23275487681261 23634280603289 38533601847617 43697538408287 56484333994001 74668675834661 116741875918727 136391104748621 221346439686641 353971046725277 450811253565767 742914612279527 1121784847661339 1149418981435409 2543288406415499 2797282815510341/);
  if (!$extra) { $#a113275 = 30; $#a036062 = 30; }
  is_deeply([map { next_twin_prime($_) } @a113275], \@a036062, "next_twin_prime on record gaps");
}
