// Copyright 2005 Google Inc.  All Rights Reserved.
//
// Author: <bpm@google.com> (Brian McBarron)

#ifndef TALK_BASE_SOCKET_STREAM_H__
#define TALK_BASE_SOCKET_STREAM_H__

#include "talk/base/asyncsocket.h"
#include "talk/base/common.h"
#include "talk/base/stream.h"

using namespace cricket;

///////////////////////////////////////////////////////////////////////////////

class SocketStream : public StreamInterface {
 public:
  SocketStream(AsyncSocket* socket) : socket_(NULL) {
    Attach(socket);
  }
  virtual ~SocketStream() { delete socket_; }

  void Attach(AsyncSocket* socket) {
    if (socket_)
      delete socket_;
    socket_ = socket;
    if (socket_) {
      socket_->SignalConnectEvent.connect(this, &SocketStream::OnConnectEvent);
      socket_->SignalReadEvent.connect(this,    &SocketStream::OnReadEvent);
      socket_->SignalWriteEvent.connect(this,   &SocketStream::OnWriteEvent);
      socket_->SignalCloseEvent.connect(this,   &SocketStream::OnCloseEvent);
    }
  }

  AsyncSocket* Detach() {
    AsyncSocket* socket = socket_;
    if (socket_) {
      socket_->SignalConnectEvent.disconnect(this);
      socket_->SignalReadEvent.disconnect(this);
      socket_->SignalWriteEvent.disconnect(this);
      socket_->SignalCloseEvent.disconnect(this);
      socket_ = NULL;
    }
    return socket;
  }

  AsyncSocket* GetSocket() { return socket_; }

  virtual StreamState GetState() const {
    ASSERT(socket_ != NULL);
    switch (socket_->GetState()) {
      case Socket::CS_CONNECTED:
        return SS_OPEN;
      case Socket::CS_CONNECTING:
        return SS_OPENING;
      case Socket::CS_CLOSED:
      default:
        return SS_CLOSED;
    }
  }

  virtual StreamResult Read(void* buffer, size_t buffer_len,
                            size_t* read, int* error) {
    ASSERT(socket_ != NULL);
    int result = socket_->Recv(buffer, buffer_len);
    if (result < 0) {
      if (socket_->IsBlocking())
        return SR_BLOCK;
      if (error)
        *error = socket_->GetError();
      return SR_ERROR;
    }
    if ((result > 0) || (buffer_len == 0)) {
      if (read)
        *read = result;
      return SR_SUCCESS;
    }
    return SR_EOS;
  }

  virtual StreamResult Write(const void* data, size_t data_len,
                             size_t* written, int* error) {
    ASSERT(socket_ != NULL);
    int result = socket_->Send(data, data_len);
    if (result < 0) {
      if (socket_->IsBlocking())
        return SR_BLOCK;
      if (error)
        *error = socket_->GetError();
      return SR_ERROR;
    }
    if (written)
      *written = result;
    return SR_SUCCESS;
  }

  virtual void Close() { ASSERT(socket_ != NULL); socket_->Close(); }

  virtual bool GetSize(size_t* size) const { return false; }
  virtual bool Rewind() { return false; }

 private:
  void OnConnectEvent(AsyncSocket* socket) {
    ASSERT(socket == socket_);
    SignalEvent(this, SE_OPEN | SE_READ | SE_WRITE, 0);
  }
  void OnReadEvent(AsyncSocket* socket) {
    ASSERT(socket == socket_);
    SignalEvent(this, SE_READ, 0);
  }
  void OnWriteEvent(AsyncSocket* socket) {
    ASSERT(socket == socket_);
    SignalEvent(this, SE_WRITE, 0);
  }
  void OnCloseEvent(AsyncSocket* socket, int err) {
    ASSERT(socket == socket_);
    SignalEvent(this, SE_CLOSE, err);
  }
  
  AsyncSocket* socket_;

  DISALLOW_EVIL_CONSTRUCTORS(SocketStream);
};

///////////////////////////////////////////////////////////////////////////////


#endif  // TALK_BASE_SOCKET_STREAM_H__
