/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: ActionCommand.java,v 1.30 2007/01/03 22:41:22 pietschy Exp $
 */

package org.pietschy.command;

import javax.swing.*;
import javax.swing.event.EventListenerList;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;

/**
 * ActionCommands provide the base for all executable commands.  Subclasses must
 * implement {@link #handleExecute} to provide command specific behaviour.
 */
public abstract class
ActionCommand
extends Command implements ActionCommandExecutor
{

   /**
    * The key for the action event hint. This hint will be present when ever the
    * command is invoked from an attached {@link AbstractButton}.
    */
   public static final String HINT_ACTION_EVENT = "org.pietschy.command.action-event";

   /**
    * The key for the modifiers hint.  This hint will be present when ever the
    * command is invoked from an attached {@link AbstractButton}.
    *
    * @see #getModifiers
    */
   public static final String HINT_MODIFIERS = "org.pietschy.command.modifiers";

   /**
    * The key for the invoker hint. This hint will be present when ever the
    * command is invoked from an attached {@link AbstractButton}.
    *
    * @see #getInvoker
    */
   public static final String HINT_INVOKER = "org.pietschy.command.invoker";


   /**
    * The key for the invoker window hint. This hint will be present whenever
    * the a Window parent can be found from the invoker.
    *
    * @see #getInvokerWindow()
    */
   public static final String HINT_INVOKER_WINDOW = "org.pietschy.command.invoker-window";


   private String actionCommand;
   private ActionAdapter actionAdapter;

   private EventListenerList commandListeners = new EventListenerList();
   private ArrayList commandInterceptors = new ArrayList();
   private Map hints = new HashMap();

   private ActionListener actionPerformedHandler = new ActionListener()
   {
      public void actionPerformed(ActionEvent e)
      {
         putHint(HINT_ACTION_EVENT, e);
         putHint(HINT_MODIFIERS, new Integer(e.getModifiers()));
         putHint(HINT_INVOKER, e.getSource());
         execute();
      }
   };

   /**
    * Creates a new anonymous ActionCommand.  Anonymous commands must be fully programatically
    * generated and can only be added to groups manually by calling
    * <code>groupCommand.installFace(myAnonymousCommand)</code>.
    */
   public ActionCommand()
   {
      super(CommandManager.defaultInstance());
   }

   /**
    * Creates a new command with the speicifed Id that is bound to the
    * {@link CommandManager#defaultInstance()}.
    */
   public ActionCommand(String id)
   {
      super(CommandManager.defaultInstance(), id);
   }

   /**
    * Creates a new anonymous command bound to the specified {@link CommandManager#defaultInstance()}.
    */
   public ActionCommand(CommandManager commandManager)
   {
      super(commandManager);
   }

   /**
    * Creates a new ActionCommand with the specified id that is bound to the
    * specified {@link CommandManager}.
    */
   public ActionCommand(CommandManager commandManager, String commandId)
   {
      super(commandManager, commandId);
   }


   /**
    * Main entry point for command subclasses that must be implemented to provide
    * command specific behaviour.<p>
    * This method should never be called directly to invoke a comand.  All
    * command invocation must be performed using the {@link #execute()} and
    * {@link #execute(Map)} methods.
    */
   protected abstract void handleExecute();


   /**
    * Executes this command with the specified hints.  The hints are added to any existing
    * hints that have been configured using {@link #putHint}.  The hints are only available for
    * this execution of the command.
    *
    * @param hints the hints this command is to use.
    * @see #putHint
    * @see #getHint
    */
   public final void execute(Map hints)
   {
      if (internalLog.isDebugEnabled())
      {
         internalLog.enter("execute");
         internalLog.param("hints", String.valueOf(hints));
      }

      putHints(hints);
      execute();

      internalLog.exit("execute");
   }

   /**
    * Causes the command to perform it's operation.  Subclasses must implement
    * {@link #handleExecute} to customize this commands behaviour.
    */
   public final void execute()
   {
      internalLog.enter("execute");

      // extract the invoker window, we do this even if the the user doesn't
      // request it.  This ensures that it's always availabe even if the command
      // alters the component heirarchy during execution.
      putHint(HINT_INVOKER_WINDOW, extractInvokerWindow(getInvoker()));

      if (preExecute())
      {
         try
         {
            handleExecute();
         }
         finally
         {
            postExecute();
         }
      }

      hints.clear();
      internalLog.exit("execute");
   }

   /**
    * This method will find the first button from this command in the {@link javax.swing.RootPaneContainer} and set
    * it to be the default button by calling {@link javax.swing.JRootPane#setDefaultButton}.
    *
    * @param container the root pane container to check.
    */
   public void requestDefautIn(RootPaneContainer container)
   {
      JRootPane rootPane = container.getRootPane();
      JButton button = (JButton) getButtonIn(rootPane);
      if (button != null)
         rootPane.setDefaultButton(button);
   }


   /**
    * Adds the specified hint to be available the next time {@link #execute} is called.
    *
    * @param key   the name of the hint.
    * @param value the hint value.
    */
   public void putHint(Object key, Object value)
   {
      hints.put(key, value);
   }

   /**
    * Adds the specified hints to be available the next time {@link #execute} is called.
    *
    * @param hints a list of hints to insert.
    */
   public void putHints(Map hints)
   {
      this.hints.putAll(hints);
   }

   /**
    * Gets any hints that may have been specified by a call to
    * {@link #execute(java.util.Map)} or that have been explicitly set by {@link #putHint}.
    *
    * @param key the name of the hint.
    * @return the hint or <tt>null</tt> if the hint doesn't exist.
    */
   public Object getHint(Object key)
   {
      return hints.get(key);
   }

   /**
    * Gets any hints that may have been specified by a call to
    * {@link #execute(java.util.Map)} or that have been explicitly set by {@link #putHint}.
    *
    * @param key          the name of the hint.
    * @param defaultValue a default to return if the hint wasn't provided.
    * @return the hint or <tt>defaultValue</tt> if the hint wasn't specified exist.
    * @see #putHint
    */
   public Object getHint(Object key, Object defaultValue)
   {
      Object value = hints.get(key);

      return value != null ? value : defaultValue;
   }

   /**
    * Gets any hints that may have been specified by a call to
    * {@link #execute(java.util.Map)} or that have been explicitly set by {@link #putHint}.
    *
    * @return the hint or <tt>defaultValue</tt> if the hint wasn't specified exist.
    * @see #putHint
    */
   public Map getHints()
   {
      return Collections.unmodifiableMap(hints);
   }

   /**
    * Convenience method to get any modifiers that were specified.  Modifiers are always
    * copied from any action event that triggers this command.
    */
   public int getModifiers()
   {
      return ((Integer) getHint(HINT_MODIFIERS, new Integer(0))).intValue();
   }

   /**
    * Convenience method to get the object that invoked the command.  If the command was
    * invoked from from a button (or menu) then the invoker will be that button.
    *
    * @return the {@link AbstractButton} that invoked to command.  If the command was manually
    *         invoked this will return <tt>null</tt> unless the hint {@link #HINT_INVOKER} has been
    *         explicitly set.
    */
   public Object getInvoker()
   {
      return getHint(HINT_INVOKER, null);
   }

   /**
    * Convenience method to get the Window ancestor of the object that invoked the command.
    * If the invoker is null, or doesn't decend from {@link Component} then null is returned.
    * Otherwise, an attempt to find the Window ancestor is made using
    * {@link SwingUtilities#getWindowAncestor} or by traversing the ancestors and invokers of
    * popup menus.
    *
    * @return <tt>null</tt> if the invoker is <tt>null</tt>, doesn't extend {@link Component} or
    *         doesn't have a {@link Window} ancestor.  Otherwise the invokers {@link Window} ancestor is
    *         returned.
    * @see #getInvoker
    */
   public Window getInvokerWindow()
   {
      return (Window) getHint(HINT_INVOKER_WINDOW, null);
   }


   private Window
   extractInvokerWindow(Object invokerHint)
   {

      if (invokerHint instanceof Window)
         return (Window) invokerHint;

      if (invokerHint instanceof Component)
      {
         Window window = SwingUtilities.getWindowAncestor((Component) invokerHint);

         if (window == null)
         {
            // window was null so we're probably in a popup
            if (invokerHint instanceof JMenuItem)
            {
               Container parent = ((JMenuItem) invokerHint).getParent();

               if (parent instanceof JPopupMenu)
               {
                  JPopupMenu topPopup = getTopLevelPopup((JPopupMenu) parent);
                  window = SwingUtilities.getWindowAncestor(topPopup.getInvoker());
               }
               else
               {
                  window = SwingUtilities.getWindowAncestor(parent);
               }
            }
         }

         return window;
      }

      return null;
   }

   private JPopupMenu
   getTopLevelPopup(JPopupMenu popup)
   {
      Component invoker = popup.getInvoker();

      if (invoker instanceof JMenuItem)
      {
         Component parent = invoker.getParent();
         if (parent instanceof JPopupMenu)
            return getTopLevelPopup((JPopupMenu) parent);
      }

      return popup;
   }

   /**
    * Convenience method for <code>getHint(ActionCommand.HINT_ACTION_EVENT)</code>.
    */
   public ActionEvent
   getActionEvent()
   {
      return (ActionEvent) getHint(ActionCommand.HINT_ACTION_EVENT);
   }

   /**
    * Overrides the default implementation to also installFace an {@link ActionListener} to the button.
    *
    * @param button   the button to attach to.
    * @param faceName the face the button will be using.
    */
   public void
   attach(AbstractButton button, String faceName)
   {
      super.attach(button, faceName);
      button.addActionListener(actionPerformedHandler);
   }

   /**
    * Overrides the default implementation to remove the {@link ActionListener} installed by
    * {@link #attach(javax.swing.AbstractButton, String)}.
    *
    * @param button   the button to attach to.
    */
   public void
   detach(AbstractButton button)
   {
      super.detach(button);
      button.removeActionListener(actionPerformedHandler);
   }

   /**
    * This method is called to configure newly created buttons.   Subclasses may override this
    * method to perform special configuration if required.
    *
    * @param button the button to configure.
    */
   protected void
   configureButtonStates(AbstractButton button)
   {
      super.configureButtonStates(button);
      button.setActionCommand(actionCommand);
   }


   /**
    * Gets an action that mirrors the default face of this command.  This is useful for use with
    * the native cut and paste clipboard methods that require an action.
    */
   public Action
   getActionAdapter()
   {
      if (actionAdapter == null)
         actionAdapter = new ActionAdapter(this, Face.DEFAULT, actionPerformedHandler);

      return actionAdapter;
   }

   /**
    * Gets an action that mirrors this the specified face of this command.  This is useful for
    * use with the native cut and paste clipboard methods that require an action.
    */
   public Action getActionAdapter(String faceName)
   {
      return new ActionAdapter(this, faceName, actionPerformedHandler);
   }


   /**
    * Gets the value of this commands actionCommand string.  The actionCommand is provided for
    * compatability with Swing actions.  If it isn't explicity configured using
    * {@link #setActionCommand} it will be equal to the {@link #getId id} of the command.
    *
    * @return the 'actionCommand' value of this command.
    */
   public String getActionCommand()
   {
      return actionCommand;
   }

   /**
    * Sets the value of this commands actionCommand string.  The actionCommand is provided for
    * compatability with Swing actions.  If it isn't explicity configured it will be equal to the
    * {@link #getId id} of the command.
    *
    * @param actionCommand the new value of the 'actionCommand'.
    */
   public void setActionCommand(String actionCommand)
   {
      if (!areEqual(this.actionCommand, actionCommand))
      {
         String old = this.actionCommand;
         this.actionCommand = actionCommand;
         Iterator iter = buttonIterator();
         while (iter.hasNext())
         {
            AbstractButton button = (AbstractButton) iter.next();
            button.setActionCommand(this.actionCommand);
         }

         pcs.firePropertyChange("actionCommand", old, actionCommand);
      }
   }


   /**
    * Checks if the the two values are equal.  This method is null safe, that is either value
    * may be null.
    *
    * @return <tt>true</tt> if <tt>oldValue</tt> equals <tt>newValue</tt> or if both values
    *         are <tt>null</tt>, <tt>false</tt> otherwise.
    */
   protected boolean
   areEqual(Object oldValue, Object newValue)
   {
      if (oldValue == null)
         return newValue == null;
      else
         return oldValue.equals(newValue);
   }

   /**
    * This method is called prior to {@link #handleExecute} being called.  It simply
    * calls firePreExecute to notify all the register {@link CommandListener} that
    * the command is about to execute.
    */
   protected boolean
   preExecute()      
   {
      // invoke the interceptors in the reverse order they were added, ie the last interceptor
      // added gets invoked first.  Should probably really do some syncronisation here.
      for (int i = commandInterceptors.size() - 1; i >= 0; i--)
      {
         ActionCommandInterceptor interceptor = (ActionCommandInterceptor) commandInterceptors.get(i);
         if (!interceptor.beforeExecute(this))
         {
            // if the execution was cancelled, we still invoke the after execute methods
            // on all the interceptors that were previously invoked.  This ensures that afterExecute
            // is invoked on all interceptors for which beforeExcute returned true.
            for (int j = i+1; j < commandInterceptors.size(); j++)
            {
               interceptor = (ActionCommandInterceptor) commandInterceptors.get(i);
               interceptor.afterExecute(this);
            }
            return false;
         }
      }

      CommandEvent e = null;
      // Guaranteed to return a non-null array
      Object[] listeners = commandListeners.getListenerList();
      // Process the listeners last to first, notifying
      // those that are interested in this event
      for (int i = listeners.length - 2; i >= 0; i -= 2)
      {
         if (listeners[i] == CommandListener.class)
         {
            // Lazily create the event:
            if (e == null)
               e = new CommandEvent(this);
            ((CommandListener) listeners[i + 1]).beforeExecute(e);
         }
      }

      return true;
   }

   /**
    * This method is called after {@link #handleExecute} has been called.  It simply
    * calls firePostExecute to notify all the register {@link CommandListener} that
    * the command has just completed.
    */
   protected void
   postExecute()
   {
      // on the way out we invoke interceptors in the order they were added.
      // Should probably really do some syncronisation here.
      for (Iterator iter = commandInterceptors.iterator(); iter.hasNext();)
         ((ActionCommandInterceptor) iter.next()).afterExecute(this);

      CommandEvent e = null;
      // Guaranteed to return a non-null array
      Object[] listeners = commandListeners.getListenerList();
      // Process the listeners last to first, notifying
      // those that are interested in this event
      for (int i = listeners.length - 2; i >= 0; i -= 2)
      {
         if (listeners[i] == CommandListener.class)
         {
            // Lazily create the event:
            if (e == null)
               e = new CommandEvent(this);
            ((CommandListener) listeners[i + 1]).afterExecute(e);
         }
      }
   }


   /**
    * Adds a {@link CommandListener} the the command.  The listener will be notified when
    * the command is executed.
    *
    * @param l the {@link CommandListener} to register
    * @see #removeCommandListener
    * @deprecated Use {@link #addInterceptor(ActionCommandInterceptor)} instead.
    */
   public void
   addCommandListener(CommandListener l)
   {
      internalLog.enter("addCommandListener");
      internalLog.param("l", String.valueOf(l));
      commandListeners.add(CommandListener.class, l);
      internalLog.exit("addCommandListener");
   }

   /**
    * Removes a {@link CommandListener} the the command.
    *
    * @param l the {@link CommandListener} to removed
    * @see #addCommandListener
    * @deprecated use {@link #removeInterceptor(ActionCommandInterceptor)} instead.
    */
   public void
   removeCommandListener(CommandListener l)
   {
      internalLog.enter("removeCommandListener");
      internalLog.param("l", String.valueOf(l));
      commandListeners.remove(CommandListener.class, l);
      internalLog.exit("removeCommandListener");
   }


   /**
    * Adds an {@link ActionCommandInterceptor} the the command.  The interceptor will be invoked
    * before and after the command is executed.
    *
    * @param interceptor the {@link ActionCommandInterceptor} to register
    * @see #removeInterceptor(ActionCommandInterceptor)
    */
   public void
   addInterceptor(ActionCommandInterceptor interceptor)
   {
      internalLog.enter("addInterceptor");
      internalLog.param("interceptor", String.valueOf(interceptor));
      commandInterceptors.add(interceptor);
      internalLog.exit("addInterceptor");
   }

   /**
    * Removes an {@link ActionCommandInterceptor} from the command.
    *
    * @param interceptor the {@link ActionCommandInterceptor} to remove
    * @see #addInterceptor(ActionCommandInterceptor)
    */
   public void
   removeInterceptor(ActionCommandInterceptor interceptor)
   {
      internalLog.enter("removeInterceptor");
      internalLog.param("interceptor", String.valueOf(interceptor));
      commandInterceptors.remove(interceptor);
      internalLog.exit("removeInterceptor");
   }

   /**
    * Installs a shortcut into the components input and action maps using the
    * accelerator specified by the default face.
    *
    * @param component The component to install the short cut in.
    * @param condition The condition as per {@link javax.swing.JComponent#getInputMap(int)}.
    */
   public void
   installShortCut(JComponent component, int condition)
   {
      KeyStroke accelerator = getDefaultFace().getAccelerator();

      if (accelerator == null)
      {
         throw new IllegalStateException("no accelerator defined for the default face");
      }

      InputMap inputMap = component.getInputMap(condition);
      inputMap.put(accelerator, getId());
      component.getActionMap().put(getId(), getActionAdapter());
   }

   /**
    * Installs a shortcut into the components input and action maps using the
    * accelerator of the specified face.
    *
    * @param component The component to install the short cut in.
    * @param faceName The face that defines the required accelerator.
    * @param condition The condition as per {@link javax.swing.JComponent#getInputMap(int)}.
    */
   public void
   installShortCut(JComponent component, String faceName, int condition)
   {
      KeyStroke accelerator = getFace(faceName).getAccelerator();

      if (accelerator == null)
      {
         throw new IllegalStateException("no accelerator defined for the face: " + faceName);
      }

      InputMap inputMap = component.getInputMap(condition);
      inputMap.put(accelerator, this);
      component.getActionMap().put(this, getActionAdapter());
   }


   /**
    * Removes the short cut installed by a previous call to {@link #installShortCut(javax.swing.JComponent, int)}
    * @param component The component the short cut was installed on.
    * @param condition The condition as per {@link javax.swing.JComponent#getInputMap(int)}.
    */
   public void
   uninstallShortCut(JComponent component, int condition)
   {
      KeyStroke accelerator = getDefaultFace().getAccelerator();

      if (accelerator == null)
      {
         throw new IllegalStateException("no accelerator defined for the default face");
      }

      InputMap inputMap = component.getInputMap(condition);
      // only remove the entry if we're the current target.
      if (this.equals(inputMap.get(accelerator)))
      {
         inputMap.remove(accelerator);
      }
      component.getActionMap().remove(this);
   }

   /**
    * Removes the short cut installed by a previous call to {@link #installShortCut(javax.swing.JComponent, String, int)}
    * @param component The component the short cut was installed on.
    * @param condition The condition as per {@link javax.swing.JComponent#getInputMap(int)}.
    */
   public void
   uninstallShortCut(JComponent component, String faceName, int condition)
   {
      KeyStroke accelerator = getFace(faceName).getAccelerator();

      if (accelerator == null)
      {
         throw new IllegalStateException("no accelerator defined for the face:" + faceName);
      }

      InputMap inputMap = component.getInputMap(condition);
      // only remove the entry if we're the current target.
      if (this.equals(inputMap.get(accelerator)))
      {
         inputMap.remove(accelerator);
      }
      component.getActionMap().remove(this);
   }

}
