/*
 * Apfloat arbitrary precision arithmetic library
 * Copyright (C) 2002-2019  Mikko Tommila
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.apfloat;

import java.math.RoundingMode;

import junit.framework.TestSuite;
import static java.math.RoundingMode.*;

/**
 * @version 1.9.1
 * @author Mikko Tommila
 */

public class ApfloatMathTest
    extends ApfloatTestCase
{
    public ApfloatMathTest(String methodName)
    {
        super(methodName);
    }

    public static void main(String[] args)
    {
        junit.textui.TestRunner.run(suite());
    }

    public static TestSuite suite()
    {
        TestSuite suite = new TestSuite();

        suite.addTest(new ApfloatMathTest("testIntegerPow"));
        suite.addTest(new ApfloatMathTest("testScale"));
        suite.addTest(new ApfloatMathTest("testInverseRoot"));
        suite.addTest(new ApfloatMathTest("testRoot"));
        suite.addTest(new ApfloatMathTest("testAbs"));
        suite.addTest(new ApfloatMathTest("testCopySign"));
        suite.addTest(new ApfloatMathTest("testFloor"));
        suite.addTest(new ApfloatMathTest("testCeil"));
        suite.addTest(new ApfloatMathTest("testTruncate"));
        suite.addTest(new ApfloatMathTest("testFrac"));
        suite.addTest(new ApfloatMathTest("testRound"));
        suite.addTest(new ApfloatMathTest("testNegate"));
        suite.addTest(new ApfloatMathTest("testModf"));
        suite.addTest(new ApfloatMathTest("testFmod"));
        suite.addTest(new ApfloatMathTest("testMultiplyAdd"));
        suite.addTest(new ApfloatMathTest("testMultiplySubtract"));
        suite.addTest(new ApfloatMathTest("testAgm"));
        suite.addTest(new ApfloatMathTest("testFactorial"));
        suite.addTest(new ApfloatMathTest("testPi"));
        suite.addTest(new ApfloatMathTest("testLog"));
        suite.addTest(new ApfloatMathTest("testLogBase"));
        suite.addTest(new ApfloatMathTest("testExp"));
        suite.addTest(new ApfloatMathTest("testPow"));
        suite.addTest(new ApfloatMathTest("testAcosh"));
        suite.addTest(new ApfloatMathTest("testAsinh"));
        suite.addTest(new ApfloatMathTest("testAtanh"));
        suite.addTest(new ApfloatMathTest("testCosh"));
        suite.addTest(new ApfloatMathTest("testSinh"));
        suite.addTest(new ApfloatMathTest("testTanh"));
        suite.addTest(new ApfloatMathTest("testAcos"));
        suite.addTest(new ApfloatMathTest("testAsin"));
        suite.addTest(new ApfloatMathTest("testAtan"));
        suite.addTest(new ApfloatMathTest("testAtan2"));
        suite.addTest(new ApfloatMathTest("testCos"));
        suite.addTest(new ApfloatMathTest("testSin"));
        suite.addTest(new ApfloatMathTest("testTan"));
        suite.addTest(new ApfloatMathTest("testW"));
        suite.addTest(new ApfloatMathTest("testToRadians"));
        suite.addTest(new ApfloatMathTest("testToDegrees"));
        suite.addTest(new ApfloatMathTest("testProduct"));
        suite.addTest(new ApfloatMathTest("testSum"));
        suite.addTest(new ApfloatMathTest("testGamma"));
        suite.addTest(new ApfloatMathTest("testRandom"));
        suite.addTest(new ApfloatMathTest("testRandomGaussian"));
        suite.addTest(new ApfloatMathTest("testMax"));
        suite.addTest(new ApfloatMathTest("testMin"));

        return suite;
    }

    public static void testIntegerPow()
    {
        Apfloat x = new Apfloat(2, 1000);
        assertEquals("2^30", new Apfloat(1 << 30), ApfloatMath.pow(x, 30));
        assertEquals("2^60", new Apfloat(1L << 60), ApfloatMath.pow(x, 60));
        assertEquals("2^-1", new Apfloat("0.5"), ApfloatMath.pow(x, -1));
        assertEquals("2^-3", new Apfloat("0.125"), ApfloatMath.pow(x, -3));
        assertEquals("2^0", new Apfloat(1), ApfloatMath.pow(x, 0));

        assertEquals("2^1000000000", ApfloatMath.pow(x, 1000000000).precision(10), ApfloatMath.pow(x.precision(11), 1000000000).precision(10));

        x = new Apfloat("1.00000000000000000000000000000000000000001", 50);
        assertEquals("1.00000000000000000000000000000000000000001^0x7FFFFFFFFFFFFFFF", new Apfloat("1.0000000000000000000000922337203685477580700042535"), ApfloatMath.pow(x, 0x7FFFFFFFFFFFFFFFL), new Apfloat(1e-48));
        assertEquals("1.00000000000000000000000000000000000000001^0x8000000000000000", new Apfloat("0.99999999999999999999990776627963145224192000425352"), ApfloatMath.pow(x, 0x8000000000000000L), new Apfloat(1e-48));

        assertEquals("2^1000 * 5^1000 = 10^1000", ApfloatMath.pow(new Apfloat(10), 1000), ApfloatMath.pow(new Apfloat(2), 1000).multiply(ApfloatMath.pow(new Apfloat(5), 1000)));
        assertEquals("2^10000 * 5^10000 = 10^10000", ApfloatMath.pow(new Apfloat(10), 10000), ApfloatMath.pow(new Apfloat(2), 10000).multiply(ApfloatMath.pow(new Apfloat(5), 10000)));
        assertEquals("2^100000 * 5^100000 = 10^100000", ApfloatMath.pow(new Apfloat(10), 100000), ApfloatMath.pow(new Apfloat(2), 100000).multiply(ApfloatMath.pow(new Apfloat(5), 100000)));
        assertEquals("2^1000000 * 5^1000000 = 10^1000000", ApfloatMath.pow(new Apfloat(10), 1000000), ApfloatMath.pow(new Apfloat(2), 1000000).multiply(ApfloatMath.pow(new Apfloat(5), 1000000)));

        try
        {
            ApfloatMath.pow(new Apfloat(0), 0);
            fail("0^0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be undefined
        }
    }

    public static void testScale()
    {
        for (int radix = Character.MIN_RADIX; radix <= Character.MAX_RADIX; radix++)
        {
            Apfloat x = ApfloatMath.scale(new Apfloat(1, 5, radix), 5);
            assertEquals("1 precision", 5, x.precision());
            assertEquals("1 string", "100000", x.toString(true));

            x = ApfloatMath.scale(new Apfloat(-1, 5, radix), -5);
            assertEquals("-1 precision", 5, x.precision());
            assertEquals("-1 string", "-0.00001", x.toString(true));

            x = ApfloatMath.scale(new Apfloat(1, 5, radix), Long.MIN_VALUE / 2);
            x = ApfloatMath.scale(x, Long.MAX_VALUE);
            assertEquals("-2^62 precision", 5, x.precision());
            assertEquals("-2^62 string", "1e4611686018427387903", x.toString());

            x = ApfloatMath.scale(new Apfloat(1, 5, radix), Long.MIN_VALUE / -2);
            x = ApfloatMath.scale(x, Long.MIN_VALUE);
            assertEquals("2^62 precision", 5, x.precision());
            assertEquals("2^62 string", "1e-4611686018427387904", x.toString());

            x = ApfloatMath.scale(new Apfloat(1, 5, radix), Long.MIN_VALUE / 2);
            x = ApfloatMath.scale(x, Long.MIN_VALUE / 2);
            assertEquals("underflow", "0", x.toString());

            x = ApfloatMath.scale(new Apfloat(1, 5, radix), Long.MIN_VALUE / 2);
            x = ApfloatMath.scale(x, Long.MIN_VALUE);
            assertEquals("underflow 2", "0", x.toString());

            x = ApfloatMath.scale(new Apfloat(radix - 1, 5, radix), 5);
            assertEquals("9 precision", 5, x.precision());
            assertEquals("9 string", Character.forDigit(radix - 1, radix) + "00000", x.toString(true));
        }
    }

    public static void testInverseRoot()
    {
        Apfloat x = new Apfloat("1.2345678901234567890123456789"),
                invX,
                invSqrtX;
        assertEquals("1.2345678901234567890123456789^-1", new Apfloat("0.810000007290000066339000603685715"), ApfloatMath.inverseRoot(x, 1), new Apfloat(1e-30));
        assertEquals("-2^-1/3", new Apfloat(-0.79370052598409973737585281963615), ApfloatMath.inverseRoot(new Apfloat(-2.0), 3), new Apfloat(1e-15));
        assertEquals("-2^-1/5", new Apfloat(-0.87055056329612413913627001747975), ApfloatMath.inverseRoot(new Apfloat(-2.0), 5), new Apfloat(1e-15));
        assertEquals("-2^-1/7", new Apfloat(-0.9057236642639066715941728732151), ApfloatMath.inverseRoot(new Apfloat(-2.0), 7), new Apfloat(1e-15));

        x = new Apfloat("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913151557485724245415069595082953311686172785588907509838175463746493931925506040092770167113900984882401285836160356370766010471018194295559619894676783744944825537977472684710404753464620804668425906949129331367702898915210475216205696602405803815019351125338243003558764024749647326391419927260426992279678235478163600934172164121992458631503028618297455570674983850549458858692699569092721079750930295532116534498720275596023648066549911988183479775356636980742654252786255181841757467289097777279380008164706001614524919217321721477235014144197356854816136115735255213347574184946843852332390739414333454776241686251898356948556209921922218427255025425688767179049460165346680498862723279178608578438382796797668145410095388378636095068006422512520511739298489608412848862694560424196528502221066118630674427862203919494504712371378696095636437191728746776465757396241389086583264599581339047802759009946576407895126946839835259570982582262052248940772671947826848260147699090264013639443745530506820349625245174939965143142980919065925093722169646151570985838741059788595977297549893016175392846813826868386894277415599185592524595395943104997252468084598727364469584865383673622262609912460805124388439045124413654976278079771569143599770012961608944169486855584840635342207222582848864815845602850601684273945226746767889525213852254995466672782398645659611635488623057745649803559363456817432411251507606947945109659609402522887971089314566913686722874894056010150330861792868092087476091782493858900971490967598526136554978189312978482168299894872265880485756401427047755513237964145152374623436454285844479526586782105114135473573952311342716610213596953623144295248493718711014576540359027993440374200731057853906219838744780847848968332144571386875194350643021845319104848100537061468067491927819119793995206141966342875444064374512371819217999839101591956181467514269123974894090718649423196156794520809514655022523160388193014209376213785595663893778708303906979207734672218256259966150142150306803844773454920260541466592520149744285073251866600213243408819071048633173464965145390579626856100550810665879699816357473638405257145910289706414011097120628043903975951567715770042033786993600723055876317635942187312514712053292819182618612586732157919841484882916447060957527069572209175671167229109816909152801735067127485832228718352093539657251210835791513698820914442100675103346711031412671113699086585163983150197016515116851714376576183515565088490998985998238734552833163550764791853589322618548963213293308985706420467525907091548141654985946163718027098199430992448895757128289059232332609729971208443357326548938239119325974636673058360414281388303203824903758985243744170291327656180937734440307074692112019130203303801976211011004492932151608424448596376698389522868478312355265821314495768572624334418930396864262434107732269780280731891544110104468232527162010526522721116603966655730925471105578537634668206531098965269186205647693125705863566201855810072936065987648611791045334885034611365768675324944166803962657978771855608455296541266540853061434443185867697514566140680070023787765913440171274947042056223053899456131407112700040785473326993908145466464588079727082668306343285878569830523580893306575740679545716377525420211495576158140025012622859413021647155097925923099079654737612551765675135751782966645477917450112996148903046399471329621073404375189573596145890193897131117904297828564750320319869151402870808599048010941214722131794764777262241425485454033215718530614228813758504306332175182979866223717215916077166925474873898665494945011465406284336639379003976926567214638530673609657120918076383271664162748888007869256029022847210403172118608204190004229661711963779213375751149595015660496318629472654736425230817703675159067350235072835405670403867435136222247715891504953098444893330963408780769325993978054193414473774418426312986080998886874132604721569516239658645730216315981931951673538129741677294786724229246543668009806769282382806899640048243540370141631496589794092432378969070697794223625082216889573837986230015937764716512289357860158816175578297352334460428151262720373431465319777741603199066554187639792933441952154134189948544473456738316249934191318148092777710386387734317720754565453220777092120190516609628049092636019759882816133231666365286193266863360627356763035447762803504507772355471058595487027908143562401451718062464362679456127531813407833033625423278394497538243720583531147711992606381334677687969597030983391307710987040859133746414428227726346594704745878477872019277152807317679077071572134447306057007334924369311383504931631284042512192565179806941135280131470130478164378851852909285452011658393419656213491434159562586586557055269049652098580338507224264829397285847831630577775606888764462482468579260395352773480304802900587607582510474709164396136267604492562742042083208566119062545433721315359584506877246029016187667952406163425225771954291629919306455377991403734043287526288896399587947572917464263574552540790914513571113694109119393251910760208252026187985318877058429725916778131496990090192116971737278476847268608490033770242429165130050051683233643503895170298939223345172201381280696501178440874519601212285993716231301711444846409038906449544400619869075485160263275052983491874078668088183385102283345085048608250393021332197155184306354550076682829493041377655279397517546139539846833936383047461199665385815384205685338621867252334028308711232827892125077126294632295639898989358211674562701021835646220134967151881909730381198004973407239610368540664319395097901906996395524530054505806855019567302292191393391856803449039820595510022635353619204199474553859381023439554495977837790237421617271117236434354394782218185286240851400666044332588856986705431547069657474585503323233421073015459405165537906866273337995851156257843229882737231989875714159578111963583300594087306812160287649628674460477464915995054973742562690104903778198683593814657412680492564879855614537234786733039046883834363465537949864192705638729317487233208376011230299113679386270894387993620162951541337142489283072201269014754668476535761647737946752004907571555278196536213239264061601363581559074220202031872776052772190055614842555187925303435139844253223415762336106425063904975008656271095359194658975141310348227693062474353632569160781547818115284366795706110861533150445212747392454494542368288606134084148637767009612071512491404302725386076482363414334623518975766452164137679690314950191085759844239198629164219399490723623464684411739403265918404437805133389452574239950829659122850855582157250310712570126683024029295252201187267675622041542051618416348475651699981161410100299607838690929160302884002691041407928862150784245167090870006992821206604183718065355672525325675328612910424877618258297651579598470356222629348600341587229805349896502262917487882027342092222453398562647669149055628425039127577102840279980663658254889264880254566101729670266407655904290994568150652653053718294127033693137851786090407086671149655834343476933857817113864558736781230145876871266034891390956200993936103102916161528813843790990423174733639480457593149314052976347574811935670911013775172100803155902485309066920376719220332290943346768514221447737939375170344366199104033751117354719185504644902636551281622882446257591633303910722538374218214088350865739177150968288747826569959957449066175834413752239709683408005355984917541738188399944697486762655165827658483588453142775687900290951702835297163445621296404352311760066510124120065975585127617858382920419748442360800719304576189323492292796501987518721272675079812554709589045563579212210333466974992356302549478024901141952123828153091140790738602515227429958180724716259166854513331239480494707911915326734302824418604142636395480004480026704962482017928964766975831832713142517029692348896276684403232609275249603579964692565049368183609003238092934595889706953653494060340216654437558900456328822505452556405644824651518754711962184439658253375438856909411303150952617937800297412076651479394259029896959469955657612186561967337862362561252163208628692221032748892186543648022967807057656151446320469279068212073883778142335628236089632080682224680122482611771858963814091839036736722208883215137556003727983940041529700287830766709444745601345564172543709069793961225714298946715435784687886144458123145935719849225284716050492212424701412147805734551050080190869960330276347870810817545011930714122339086639383395294257869050764310063835198343893415961318543475464955697810382930971646514384070070736041123735998434522516105070270562352660127648483084076118301305279320542746286540360367453286510570658748822569815793678976697422057505968344086973502014102067235850200724522563265134105592401902742162484391403599895353945909440704691209140938700126456001623742880210927645793106579229552498872758461012648369998922569596881592056001016552563756785667227966198857827948488558343975187445455129656344348039664205579829368043522027709842942325330225763418070394769941597915945300697521482933665556615678736400536665641654732170439035213295435291694145990416087532018683793702348886894791510716378529023452924407736594956305100742108714261349745956151384987137570471017879573104229690666702144986374645952808243694457897723300487647652413390759204340196340391147320233807150952220106825634274716460243354400515212669324934196739770415956837535551667302739007497297363549645332888698440611964961627734495182736955882207573551766515898551909866653935494810688732068599075407923424023009259007017319603622547564789406475483466477604114632339056513433068449539790709030234604614709616968868850140834704054607429586991382966824681857103188790652870366508324319744047718556789348230894310682870272280973624809399627060747264553992539944280811373694338872940630792615959954626246297070625948455690347119729964090894180595343932512362355081349490043642785271383159125689892951964272875739469142725343669415323610045373048819855170659412173524625895487301676002988659257866285612496655235338294287854253404830833070165372285635591525347844598183134112900199920598135220511733658564078264849427644113763938669248031183644536985891754426473998822846218449008777697763127957226726555625962825427653183001340709223343657791601280931794017185985999338492354956400570995585611349802524990669842330173503580440811685526531170995708994273287092584878944364600504108922669178352587078595129834417295351953788553457374260859029081765155780390594640873506123226112009373108048548526357228257682034160504846627750450031262008007998049254853469414697751649327095049346393824322271885159740547021482897111777923761225788734771881968254629812686858170507402725502633290449762778944236216741191862694396506715157795867564823993917604260176338704549901761436412046921823707648878341968968611815581587360629386038101712158552726683008238340465647588040513808016336388742163714064354955618689641122821407533026551004241048967835285882902436709048871181909094945331442182876618103100735477054981596807720094746961343609286148494178501718077930681085469000944589952794243981392135055864221964834915126390128038320010977386806628779239718014613432445726400973742570073592100315415089367930081699805365202760072774967458400283624053460372634165542590276018348403068113818551059797056640075094260878857357960373245141467867036880988060971642584975951380693094494015154222219432913021739125383559150310033303251117491569691745027149433151558854039221640972291011290355218157628232831823425483261119128009282525619020526301639114772473314857391077758744253876117465786711694147764214411112635835538713610110232679877564102468240322648346417663698066378576813492045302240819727856471983963087815432211669122464159117767322532643356861461865452226812688726844596844241610785401676814208088502800541436131462308210259417375623899420757136275167457318918945628352570441335437585753426986994725470316566139919996826282472706413362221789239031760854289437339356188916512504244040089527198378738648058472689546243882343751788520143956005710481194988423906061369573423155907967034614914344788636041031823507365027785908975782727313050488939890099239135033732508559826558670892426124294736701939077271307068691709264625484232407485503660801360466895118400936686095463250021458529309500009071510582362672932645373821049387249966993394246855164832611341461106802674466373343753407642940266829738652209357016263846485285149036293201991996882851718395366913452224447080459239660281715655156566611135982311225062890585491450971575539002439315351909021071194573002438801766150352708626025378817975194780610137150044899172100222013350131060163915415895780371177927752259787428919179155224171895853616805947412341933984202187456492564434623925319531351033114763949119950728584306583619353693296992898379149419394060857248639688369032655643642166442576079147108699843157337496488352927693282207629472823815374099615455987982598910937171262182830258481123890119682214294576675807186538065064870261338928229949725745303328389638184394477077940228435988341003583854238973542439564755568409522484455413923941000162076936368467764130178196593799715574685419463348937484391297423914336593604100352343777065888677811394986164787471407932638587386247328896456435987746676384794665040741118256583788784548581489629612739984134427260860618724554523606431537101127468097787044640947582803487697589483282412392929605829486191966709189580898332012103184303401284951162035342801441276172858302435598300320420245120728725355811958401491809692533950757784000674655260314461670508276827722235341911026341631571474061238504258459884199076112872580591139356896014316682831763235673254170734208173322304629879928049085140947903688786878949305469557030726190095020764334933591060245450864536289354568629585313153371838682656178622736371697577418302398600659148161640494496501173213138957470620884748023653710311508984279927544268532779743113951435741722197597993596852522857452637962896126915723579866205734083757668738842664059909935050008133754324546359675048442352848747014435454195762584735642161981340734685411176688311865448937769795665172796623267148103386439137518659467300244345005449953997423723287124948347060440634716063258306498297955101095418362350303094530973358344628394763047756450150085075789495489313939448992161255255977014368589435858775263796255970816776438001254365023714127834679261019955852247172201777237004178084194239487254068015560359983905489857235467456423905858502167190313952629445543913166313453089390620467843877850542393905247313620129476918749751910114723152893267725339181466073000890277689631148109022097245207591672970078505807171863810549679731001678708506942070922329080703832634534520380278609905569001341371823683709919495164896007550493412678764367463849020639640197666855923356546391383631857456981471962108410809618846054560390384553437291414465134749407848844237721751543342603066988317683310011331086904219390310801437843341513709243530136776310849135161564226984750743032971674696406665315270353254671126675224605511995818319637637076179919192035795820075956053023462677579439363074630569010801149427141009391369138107258137813578940055995001835425118417213605572752210352680373572652792241737360575112788721819084490061780138897107708229310027976659358387589093956881485602632243937265624727760378908144588378550197028437793624078250527048758164703245812908783952324532378960298416692254896497156069811921865849267704039564812781021799132174163058105545988013004845629976511212415363745150056350701278159267142413421033015661653560247338078430286552572227530499988370153487930080626018096238151613669033411113865385109193673938352293458883225508870645075394739520439680790670868064450969865488016828743437861264538158342807530618454859037982179945996811544197425363443996029025100158882721647450068207041937615845471231834600726293395505482395571372568402322682130124767945226448209102356477527230820810635188991526928891084555711266039650343978962782500161101532351605196559042118449499077899920073294769058685778787209829013529566139788848605097860859570177312981553149516814671769597609942100361835591387778176984587581044662839988060061622984861693533738657877359833616133841338536842119789389001852956919678045544828584837011709672125353387586215823101331038776682721157269495181795897546939926421979155233857662316762754757035469941489290413018638611943919628388705436777432242768091323654494853667680000010652624854730558615989991401707698385483188750142938908995068545307651168033373222651756622075269517914422528081651716677667279303548515420402381746089232839170327542575086765511785939500279338959205766827896776445318404041855401043513483895312013263783692835808271937831265496174599705674507183320650345566440344904536275600112501843356073612227659492783937064784264567633881880756561216896050416113903906396016202215368494109260538768871483798955999911209916464644119185682770045742434340216722764455893301277815868695250694993646101756850601671453543158148010545886056455013320375864548584032402987170934809105562116715468484778039447569798042631809917564228098739987669732376957370158080682290459921236616890259627304306793165311494017647376938735140933618332161428021497633991898354848756252987524238730775595559554651963944018218409984124898262367377146722606163364329640633572810707887581640438148501884114318859882769449011932129682715888413386943468285900666408063140777577257056307294004929403024204984165654797367054855804458657202276378404668233798528271057843197535417950113472736257740802134768260450228515797957976474670228409995616015691089038458245026792659420555039587922981852648007068376504183656209455543461351341525700659748819163413595567196496540321872716026485930490397874895890661272507948282769389535217536218507962977851461884327192232238101587444505286652380225328438913752738458923844225354726530981715784478342158223270206902872323300538621634798850946954720047952311201504329322662827276321779088400878614802214753765781058197022263097174950721272484794781695729614236585957820908307332335603484653187302930266596450137183754288975579714499246540386817992138934692447419850973346267933210726868707680626399193619650440995421676278409146698569257150743157407938053239252394775574415918458215625181921552337096074833292349210345146264374498055961033079941453477845746999921285999993996122816152193148887693880222810830019860165494165426169685867883726095877456761825072759929508931805218729246108676399589161458550583972742098090978172932393010676638682404011130402470073508578287246271349463685318154696904669686939254725194139929146524238577625500474852954768147954670070503479995888676950161249722820403039954632788306959762493615101024365553522306906129493885990157346610237122354789112925476961760050479749280607212680392269110277722610254414922157650450812067717357120271802429681062037765788371669091094180744878140490755178203856539099104775941413215432844062503018027571696508209642734841469572639788425600845312140659358090412711359200419759851362547961606322887361813673732445060792441176399759746193835845749159880976674470930065463424234606342374746660804317012600520559284936959414340814685298150539471789004518357551541252235905906872648786357525419112888773717663748602766063496035367947026923229718683277173932361920077745221262475186983349515101986426988784717193966497690708252174233656627259284406204302141137199227852699846988477023238238400556555178890876613601304770984386116870523105531491625172837327286760072481729876375698163354150746088386636406934704372066886512756882661497307886570156850169186474885416791545965072342877306998537139043002665307839877638503238182155355973235306860430106757608389086270498418885951380910304235957824951439885901131858358406674723702971497850841458530857813391562707603563907639473114554958322669457024941398316343323789759556808568362972538679132750555425244919435891284050452269538121791319145135009938463117740179715122837854601160359554028644059024964669307077690554810288502080858008781157738171917417760173307385547580060560143377432990127286772530431825197579167929699650414607066457125888346979796429316229655201687973000356463045793088403274807718115553309098870255052076804630346086581653948769519600440848206596737947316808641564565053004988161649057883115434548505266006982309315777650037807046612647060214575057932709620478256152471459189652236083966456241051955105223572397395128818164059785914279148165426328920042816091369377737222999833270820829699557377273756676155271139225880552018988762011416800546873655806334716037342917039079863965229613128017826797172898229360702880690877686605932527463784053976918480820410219447197138692560841624511239806201131845412447820501107987607171556831540788654390412108730324020106853419472304766667217498698685470767812051247367924791931508564447753798537997322344561227858432968466475133365736923872014647236794278700425032555899268843495928761240075587569464137056251400117971331662071537154360068764773186755871487839890810742953094106059694431584775397009439883949144323536685392099468796450665339857388878661476294434140104988899316005120767810358861166020296119363968213496075011164983278563531614516845769568710900299976984126326650234771672865737857908574664607722834154031144152941880478254387617707904300015669867767957609099669360755949651527363498118964130433116627747123388174060373174397054067031096767657486953587896700319258662594105105335843846560233917967492678447637084749783336555790073841914731988627135259546251816043422537299628632674968240580602964211463864368642247248872834341704415734824818333016405669596688667695634914163284264149745333499994800026699875888159350735781519588990053951208535103572613736403436753471410483601754648830040784641674521673719048310967671134434948192626811107399482506073949507350316901973185211955263563258433909982249862406703107683184466072912487475403161796994113973877658998685541703188477886759290260700432126661791922352093822787888098863359911608192353555704646349113208591897961327913197564909760001399623444553501434642686046449586247690943470482932941404111465409239883444351591332010773944111840741076849810663472410482393582740194493566516108846312567852977697346843030614624180358529331597345830384554103370109167677637427621021370135485445092630719011473184857492331816720721372793556795284439254815609137281284063330393735624200160456645574145881660521666087387480472433912129558777639069690370788285277538940524607584962315743691711317613478388271941686066257210368513215664780014767523103935786068961112599602818393095487090590738613519145918195102973278755710497290114871718971800469616977700179139196137914171627070189584692143436967629274591099400600849835684252019155937037010110497473394938778859894174330317853487076032219829705797511914405109942358830345463534923498268836240433272674155403016195056806541809394099820206099941402168909007082133072308966211977553066591881411915778362729274615618571037217247100952142369648308641025928874579993223749551912219519034244523075351338068568073544649951272031744871954039761073080602699062580760202927314552520780799141842906388443734996814582733720726639176702011830046481900024130835088465841521489912761065137415394356572113903285749187690944137020905170314877734616528798482353382972601361109845148418238081205409961252745808810994869722161285248974255555160763716750548961730168096138038119143611439921063800508321409876045993093248510251682944672606661381517457125597549535802399831469822036133808284993567055755247129027453977621404931820146580080215665360677655087838043041343105918046068008345911366408348874080057412725867047922583191274157390809143831384564241509408491339180968402511639919368532255573389669537490266209232613188558915808324555719484538756287861288590041060060737465014026278240273469625282171749415823317492396835301361786536737606421667781377399510065895288774276626368418306801908046098498094697636673356622829151323527888061577682781595886691802389403330764419124034120223163685778603572769415417788264352381319050280870185750470463129333537572853866058889045831114507739429352019943219711716422350056440429798920815943071670198574692738486538334361457946341759225738985880016980147574205429958012429581054565108310462972829375841611625325625165724980784920998979906200359365099347215829651741357984910471116607915874369865412223483418877229294463351786538567319625598520260729476740726167671455736498121056777168934849176607717052771876011999081441130586455779105256843048114402619384023224709392498029335507318458903553971330884461741079591625117148648744686112476054286734367090466784686702740918810142497111496578177242793470702166882956108777944050484375284433751088282647719785400065097040330218625561473321177711744133502816088403517814525419643203095760186946490886815452856213469883554445602495566684366029221951248309106053772019802183101032704178386654471812603971906884623708575180800353270471856594994761242481109992886791589690495639476246084240659309486215076903149870206735338483495508363660178487710608098042692471324100094640143736032656451845667924566695510015022983307984960799498824970617236744936122622296179081431141466094123415935930958540791390872083227335495720807571651718765994498569379562387555161757543809178052802946420044721539628074636021132942559160025707356281263873310600589106524570802447493754318414940148211999627645310680066311838237616396631809314446712986155275982014514102756006892975024630401735148919457636078935285550531733141645705049964438909363084387448478396168405184527328840323452024705685164657164771393237755172947951261323982296023945485797545865174587877133181387529598094121742273003522965080891777050682592488223221549380483714547816472139768209633205083056479204820859204754998573203888763916019952409189389455767687497308569559580106595265030362661597506622250840674288982659075106375635699682115109496697445805472886936310203678232501823237084597901115484720876182124778132663304120762165873129708112307581598212486398072124078688781145016558251361789030708608701989758898074566439551574153631931919810705753366337380382721527988493503974800158905194208797113080512339332219034662499171691509485414018710603546037946433790058909577211808044657439628061867178610171567409676620802957665770512912099079443046328929473061595104309022214393718495606340561893425130572682914657832933405246350289291754708725648426003496296116541382300773133272983050016025672401418515204189070115428857992081219844931569990591820118197335001261877280368124819958770702075324063612593134385955425477819611429351635612234966615226147353996740515849986035529533292457523888101362023476246690558164389678630976273655047243486430712184943734853006063876445662721866617012381277156213797461498613287441177145524447089971445228856629424402301847912054784985745216346964489738920624019435183100882834802492490854030778638751659113028739587870981007727182718745290139728366148421428717055317965430765045343246005363614726181809699769334862640774351999286863238350887566835950972655748154319401955768504372480010204137498318722596773871549583997184449072791419658459300839426370208756353982169620553248032122674989114026785285996734052420310917978999057188219493913207534317079800237365909853755202389116434671855829068537118979526262344924833924963424497146568465912489185566295893299090352392333336474352037077010108438800329075983421701855422838616172104176030116459187805393674474720599850235828918336929223373239994804371084196594731626548257480994825099918330069765693671596893644933488647442135008407006608835972350395323401795825570360169369909886711321097988970705172807558551912699306730992507040702455685077867906947661262980822516331363995211709845280926303759224267425755998928927837047444521893632034894155210445972618838003006776179313813991620580627016510244588692476492468919246121253102757313908404700071435613623169923716948481325542009145304103713545329662063921054798243921251725401323149027405858920632175894943454890684639931375709103463327141531622328055229729795380188016285907357295541627886764982741861642187898857410716490691918511628152854867941736389066538857642291583425006736124538491606741373401735727799563410433268835695078149313780073623541800706191802673285511919426760912210359874692411728374931261633950012395992405084543756985079570462226646190001035004901830341535458428337643781119885563187777925372011667185395418359844383052037628194407615941068207169703022851522505731260930468984234331527321313612165828080752126315477306044237747535059522871744026663891488171730864361113890694202790881431194487994171540421034121908470940802540239329429454938786402305129271190975135360009219711054120966831115163287054230284700731206580326264171161659576132723515666625366727189985341998952368848309993027574199164638414270779887088742292770538912271724863220288984251252872178260305009945108247835729056919885554678860794628053712270424665431921452817607414824038278358297193010178883456741678113989547504483393146896307633966572267270433932167454218245570625247972199786685427989779923395790575818906225254735822052364248507834071101449804787266919901864388229323053823185597328697809222535295910173414073348847610055640182423921926950620831838145469839236646136398910121021770959767049083050818547041946643713122996923588953849301363565761861060622287055994233716310212784574464639897381885667462608794820186474876727272220626764653380998019668836809941590757768526398651462533363124505364026105696055131838131742611844201890888531963569869627950367384243130113317533053298020166888174813429886815855778103432317530647849832106297184251843855344276201282345707169885305183261796411785796088881503296022907056144762209150947390359466469162353968092013945781758910889319921122600739281491694816152738427362642980982340632002440244958944561291670495082358124873917996486411334803247577752197089327722623494860150466526814398770516153170266969297049283162855042128981467061953319702695072143782304768752802873541261663917082459251700107141808548006369232594620190022780874098597719218051585321473926532515590354102092846659252999143537918253145452905984158176370589279069098969111643811878094353715213322614436253144901274547726957393934815469163116249288735747188240715039950094467319543161938554852076657388251396391635767231510055560372633948672082078086537349424401157996675073607111593513319591971209489647175530245313647709420946356969822266737752099451684506436238242118535348879893956731878066061078854400055082765703055874485418057788917192078814233511386629296671796434687600770479995378833878703487180218424373421122739402557176908196030920182401884270570460926225641783752652633583242406612533115294234579655695025068100183109004112453790153329661569705223792103257069370510908307894799990049993953221536227484766036136776979785673865846709366795885837887956259464648913766521995882869338018360119323685785585581955560421562508836502033220245137621582046181067051953306530606065010548871672453779428313388716313955969058320834168984760656071183471362181232462272588419902861420872849568796393254642853430753011052857138296437099903569488852851904029560473461311382638788975517885604249987483163828040468486189381895905420398898726506976202019955484126500053944282039301274816381585303964399254702016727593285743666616441109625663373054092195196751483287348089574777752783442210910731113518280460363471981856555729571447476825528578633493428584231187494400032296906977583159038580393535213588600796003420975473922967333106493956018122378128545843176055617338611267347807458506760630482294096530411183066710818930311088717281675195796753471885372293096161432040063813224658411111577583585811350185690478153689381377184728147519983505047812977185990847076219746058874232569958288925350419379582606162118423687685114183160683158679946016520577405294230536017803133572632670547903384012573059123396018801378254219270947673371919872873852480574212489211834708766296672072723256505651293331260595057777275424712416483128329820723617505746738701282095755443059683955556868611883971355220844528526400812520276655576774959696266126045652456840861392382657685833846984997787267065551918544686984694784957346226062942196245570853712727765230989554501930377321666491825781546772920052126671434632096378918523232150189761260343736840671941930377468809992968775824410478781232662531818459604538535438391144967753128642609252115376732588667226040425234910870269580996475958057946639734190640100363619040420331135793365424263035614570090112448008900208014780566037101541223288914657223931450760716706435568274377439657890679726874384730763464516775621030986040927170909512808630902973850445271828927496892121066700816485833955377359191369501531620189088874842107987068991148046692706509407620465027725286507289053285485614331608126930056937854178610969692025388650345771831766868859236814884752764984688219497397297077371871884004143231276365048145311228509900207424092558592529261030210673681543470152523487863516439762358604191941296976904052648323470099111542426012734380220893310966863678986949779940012601642276092608234930411806438291383473546797253992623387915829984864592717340592256207491053085315371829116816372193951887009577881815868504645076993439409874335144316263303172477474868979182092394808331439708406730840795893581089665647758599055637695252326536144247802308268118310377358870892406130313364773710116282146146616794040905186152603600925219472188909181073358719641421444786548995285823439470500798303885388608310357193060027711945580219119428999227223534587075662469261776631788551443502182870266856106650035310502163182060176092179846849368631612937279518730789726373537171502563787335797718081848784588665043358243770041477104149349274384575871071597315594394264125702709651251081155482479394035976811881172824721582501094960966253933953809221955919181885526780621499231727631632183398969380756168559117529984501320671293924041445938623988093812404521914848316462101473891825101090967738690664041589736104764365000680771056567184862814963711188321924456639458144914861655004956769826903089111856879869294705135248160917432430153836847072928989828460222373014526556798986277679680914697983782687643115988321090437156112997665215396354644208691975673700057387649784376862876817924974694384274652563163230055513041742273416464551278127845777724575203865437542828256714128858345444351325620544642410110379554641905811686230596447695870540721419852121067343324107567675758184569906930460475227701670056845439692340417110898889934163505851578873534308155208117720718803791040469830695786854739376564336319797868036718730796939242363214484503547763156702553900654231179201534649779290662415083288583952905426376876689688050333172278001858850697362324038947004718976193473443084374437599250341788079722358591342458131440498477017323616947197657153531977549971627856631190469126091825912498903676541769799036237552865263757337635269693443544004730671988689019681474287677908669796885225016369498567302175231325292653758964151714795595387842784998664563028788319620998304945198743963690706827626574858104391122326187940599415540632701319898957037611053236062986748037791537675115830432084987209202809297526498125691634250005229088726469252846661046653921714820801305022980526378364269597337070539227891535105688839381132497570713310295044303467159894487868471164383280506925077662745001220035262037094660234146489983902525888301486781621967751945831677187627572005054397944124599007711520515461993050983869825428464072555409274031325716326407929341833421470904125425335232480219322770753555467958716383587501815933871742360615511710131235256334858203651461418700492057043720182617331947157008675785393360786227395581857975872587441025420771054753612940474601000940954449596628814869159038990718659805636171376922272907641977551777201042764969496110562205925024202177042696221549587264539892276976603105249808557594716310758701332088614632664125911486338812202844406941694882615295776253250198703598706743804698219420563812558334364219492322759372212890564209430823525440841108645453694049692714940033197828613181861888111184082578659287574263844500599442295685864604810330153889114994869354360302218109434667640000223625505736312946262960961987605642599639461386923308371962659547392346241345977957485246478379807956931986508159776753505539189911513352522987361127791827485420086895396583594219633315028695611920122988898870060799927954111882690230789131076036176347794894320321027733594169086500719328040171638406449878717537567811853213284082165711075495282949749362146082155832056872321855740651610962748743750980922302116099826330339154694946444910045152809250897450748967603240907689836529406579201983152654106581368237919840906457124689484702093577611931399802468134052003947819498662026240089021501661638135383815150377350229660746279529103840686855690701575166241929872444827194293310048548244545807188976330032325258215812803274679620028147624318286221710543528983482082734516801861317195933247110746622285087106661177034653528395776259977446721857158161264111432717943478859908928084866949141390977167369002777585026866465405659503948678411107901161040085727445629384254941675946054871172359464291058509099502149587931121961359083158826206823321561530868337308381732793281969838750870834838804638847844188400318471269745437093732983624028751979208023218787448828728437273780178270080587824107493575148899789117397461293203510814327032514090304874622629423443275712600866425083331876886507564292716055252895449215376517514921963671810494353178583834538652556566406572513635750643532365089367904317025978781771903148679638408288102094614900797151377170990619549696400708676671023300486726314755105372317571143223174114116806228642063889062101923552235467116621374996932693217370431059872250394565749246169782609702533594750209138366737728944386964000281103440260847128990007468077648440887113413525033678773167977093727786821661178653442317322646378476978751443320953400016506921305464768909850502030150448808342618452087305309731894929164253229336124315143065782640702838984098416029503092418971209716016492656134134334222988279099217860426798124572853458013382609958771781131021673402565627440072968340661984806766158050216918337236803990279316064204368120799003162644491461902194582296909921227885539487835383056468648816555622943156731282743908264506116289428035016613366978240517701552196265227254558507386405852998303791803504328767038092521679075712040612375963276856748450791511473134400018325703449209097124358094479004624943134550289006806487042935340374360326258205357901183956490893543451013429696175452495739606214902887289327925206965353863964432253883275224996059869747598823299162635459733244451637553343774929289905811757863555556269374269109471170021654117182197505198317871371060510637955585889055688528879890847509157646390746936198815078146852621332524738376511929901561091897779220087057933964638274906806987691681974923656242260871541761004306089043779766785196618914041449252704808819714988015420577870065215940092897776013307568479669929554336561398477380603943688958876460549838714789684828053847017308711177611596635050399793438693391197898871091565417091330826076474063057114110988393880954814378284745288383680794188843426662220704387228874139478010177213922819119923654055163958934742639538248296090369002883593277458550608013179884071624465639979482757836501955142215513392819782269842786383916797150912624105487257009240700454884856929504481107380879965474815689139353809434745569721289198271770207666136024895814681191336141212587838955773571949863172108443989014239484966592517313881716026632619310653665350414730708044149391693632623737677770958503132559900957627319573086480424677012123270205337426670531424482081681303063973787366424836725398374876909806021827857862165127385635132901489035098832706172589325753639939790557291751600976154590447716922658063151110280384360173747421524760851520990161585823125715907334217365762671423904782795872815050956330928026684589376496497702329736413190609827406335310897924642421345837409011693919642504591288134034988106354008875968200544083643865166178805576089568967275315380819420773325979172784376256611843198910250074918290864751497940031607038455494653859460274524474668123146879434416109933389089926384118474252570445725174593257389895651857165759614812660203107976282541655905060424791140169579003383565748692528007430256234194982864679144763227740055294609039401775363356554719310001754300475047191448998410400158679461792416100164547165513370740739502604427695385538343975505488710997852054011751697475813449260794336895437832211724506873442319898788441285420647428097356258070669831069799352606933921356858813912148073547284632277849080870024677763036055512323866562951788537196730346347012229395816067925091532174890308408865160611190114984434123501246469280288059961342835118847154497712784733617662850621697787177438243625657117794500644777183702219991066950216567576440449979407650379999548450027106659878136038023141268369057831904607927652972776940436130230517870805465115424693952651271010529270703066730244471259739399505146284047674313637399782591845411764133279064606365841529270190302760173394748669603486949765417524293060407270050590395031485229213925755948450788679779252539317651564161971684435243697944473559642606333910551268260615957262170366985064732812667245219890605498802807828814297963366967441248059821921463395657457221022986775997467381260693670691340815594120161159601902377535255563006062479832612498812881929373434768626892192397778339107331065882568137771723283153290825250927330478507249771394483338925520811756084529665905539409655685417060011798572938139982583192936791003918440992865756059935989100029698644609747147184701015312837626311467742091455740418159088000649432378558393085308283054760767995243573916312218860575496738322431956506554608528812019023636447127037486344217272578795034284863129449163184753475314350413920961087960577309872013524840750576371992536504709085825139368634638633680428917671076021111598288755399401200760139470336617937153963061398636554922137415979051190835882900976566473007338793146789131814651093167615758213514248604422924453041131606527009743300884990346754055186406773426035834096086055337473627609356588531097609942383473822220872924644976845605795625167655740884103217313456277358560523582363895320385340248422733716391239732159954408284216666360232965456947035771848734420342277066538373875061692127680157661810954200977083636043611105924091178895403380214265239489296864398089261146354145715351943428507213534530183158756282757338982688985235577992957276452293915674775666760510878876484534936360682780505646228135988858792599409464460417052044700463151379754317371877560398159626475014109066588661621800382669899619655805872086397211769952194667898570117983324406018115756580742841829106151939176300591943144346051540477105700543390001824531177337189558576036071828605063564799790041397618089553636696031621931132502238517916720551806592635180362512145759262383693482226658955769946604919381124866090997981285718234940066155521961122072030922776462009993152442735894887105766238946938894464950939603304543408421024624010487233287500817491798755438793873814398942380117627008371960530943839400637561164585609431295175977139353960743227924892212670458081833137641658182695621058728924477400359470092686626596514220506300785920024882918608397437323538490839643261470005324235406470420894992102504047267810590836440074663800208701266642094571817029467522785400745085523777208905816839184465928294170182882330149715542352359117748186285929676050482038643431087795628929254056389466219482687110428281638939757117577869154301650586029652174595819888786804081103284327398671986213062055598552660364050462821523061545944744899088390819997387474529698107762014871340001225355222466954093152131153379157980269795557105085074738747507580687653764457825244326380461430428892359348529610582693821034980004052484070844035611678171705128133788057056434506161193304244407982603779511985486945591520519600930412710072778493015550388953603382619293437970818743209499141595933963681106275572952780042548630600545238391510689989135788200194117865356821491185282078521301255185184937115034221595422445119002073935396274002081104655302079328672547405436527175958935007163360763216147258154076420530200453401835723382926619153083540951202263291650544261236191970516138393573266937601569144299449437448568097756963031295887191611292946818849363386473927476012269641588489009657170861605981472044674286642087653347998582220906198021732116142304194777549907387385679411898246609130916917722742072333676350326783405863019301932429963972044451792881228544782119535308989101253429755247276357302262813820918074397486714535907786335301608215599113141442050914472935350222308171936635093468658586563148555758624478186201087118897606529698992693281787055764351433820601410773292610634315253371822433852635202177354407152818981376987551575745469397271504884697936195004777209705617939138289898453274262272886471088832701737232588182446584362495805925603381052156062061557132991560848920643403033952622634514542836786982880742514225674518061841495646861116354049718976821542277224794740335715274368194098920501136534001238467142965518673441537416150425632567134302476551252192180357801692403266995417460875924092070046693403965101781348578356944407604702325407555577647284507518268904182939661133101601311190773986324627782190236506603740416067249624901374332172464540974129955705291424382080760983648234659738866913499197840131080155813439791948528304367390124820824448141280954437738983200598649091595053228579145768849625786658859991798675205545580990045564611787552493701245532171701942828846174027366499784755082942280202329012216301023097721515694464279098021908266898688342630716092079140851976952355534886577434252775311972474308730436195113961190800302558783876442060850447306312992778889427291897271698905759252446796601897074829609491906487646937027507738664323919190422542902353189233772931667360869962280325571853089192844038050710300647768478632431910002239297852553723755662136447400967605394398382357646069924652600890906241059042154539279044115295803453345002562441010063595300395988644661695956263518780606885137234627079973272331346939714562855426154676506324656766202792452085813477176085216913409465203076733918411475041401689241213198268815686645614853802875393311602322925556189410429953356400957864953409351152664540244187759493169305604486864208627572011723195264050230997745676478384889734643172159806267876718380052476968840849891850861490034324034767426862459523958903585821350064509981782446360873177543788596776729195261112138591947254514003011805034378752776644027626189410175768726804281766238606804778852428874302591452470739505465251353394595987896197789110418902929438185672050709646062635417329446495766126519534957018600154126239622864138977967333290705673769621564981845068422636903678495559700260798679962610190393312637685569687670292953711625280055431007864087289392257145124811357786276649024251619902774710903359333093049483805978566288447874414698414990671237647895822632949046798120899848571635710878311918486302545016209298058292083348136384054217200561219893536693713367333924644161252231969434712064173754912163570085736943973059797097197266666422674311177621764030686813103518991122713397240368870009968629225464650063852886203938005047782769128356033725482557939129852515068299691077542576474883253414121328006267170940090982235296579579978030182824284902214707481111240186076134151503875698309186527806588966823625239378452726345304204188025084423631903833183845505223679923577529291069250432614469501098610888999146585518818735825281643025209392852580779697376208456374821144339881627100317031513344023095263519295886806908213558536801610002137408511544849126858412686958991741491338205784928006982551957402018181056412972508360703568510553317878408290000415525118657794539633175385320921497205266078312602819611648580986845875251299974040927976831766399146553861089375879522149717317281315179329044311218158710235187407572221001237687219447472093493123241070650806185623725267325407333248757544829675734500193219021991199607979893733836732425761039389853492787774739805080800155447640610535222023254094435677187945654304067358964910176107759483645408234861302547184764851895758366743997915085128580206078205544629917232020282229148869593997299742974711553718589242384938558585954074381048826246487880533042714630119415898963287926783273224561038521970111304665871005000832851773117764897352309266612345888731028835156264460236719966445547276083101187883891511493409393447500730258558147561908813987523578123313422798665035227253671712307568610450045489703600795698276263923441071465848957802414081584052295369374997106655948944592462866199635563506526234053394391421112718106910522900246574236041");
        invX = new Apfloat("0.31830988618379067153776752674502872406891929148091289749533468811779359526845307018022760553250617191214568545351591607378582369222915730575593482146339967845847993387481815514615549279385061537743478579243479532338672478048344725802366476022844539951143188092378017380534791224097882187387568817105744619989288680049734469547891922179664619356614981233397292560939889730437576314957313392848207799174827869721996773619839992488575117034235771686223503753432109309507397601947892072951866753611860498899327061065431355100644064955563279433204589349623919633168121203360607199626782397499766557330887055951014003248135512877769914262176024439875229536275552947578126613609291595696352262485462813992155004900059551971417811380559357026305042003263549204184962321248112291240629296817849691838287042315081511240174305321360443431828151494916544519549257079975031065878162796354481871650959414665743808139995181531541569869407871796561743468512807337902332509141188665526253730005224543594230642251990087733589007525112167263423390519516256449883246668629021224707375712622727338433428413949392025850115667210623921718901967911343741990949302086324763103516167888595994199901050877513225889176661369210157058303028208097859770127763215523939861468207799915738378119618747554412375086445437860273251052247756077507776221362813530868165655705386685359911214158077212070547799249025199149855259404718819116860232965928237115542481150889891404357953958481898065458954043329920713063630708800768137974943538317752638193301392880955394137536731355620955959090070679151660376367737587553224962990611993116043816719750207025425808646316099743937375551893132692442068408881710995700758547738858707323875565857471875686940646047429167584711423727268385892036636458392833001756615866270699558199491729858053490121978737818917661006740610761094624643161886395352064566262837961949964487667034871397969500207900136776007957344719921604800547802174990970957584713652227989780653799485416699222984165780755356948607100913691216734295861691344665407097078511240417367864819912442350663678804194158714154993099761737213272193732393407494908420566243850369244966998232229913311207593935227986256599215521655598020156607200467654597581708047752311489086185202382010867599677809309842496590321414570601045442047203504662634635951862210065631021874782727929061158521436016723590975344929196094795458489621840187425157383666579177256798087173733327951346890281900727465438348622776132766145184605519469012109642555607413067556606434197546993313698160065397701348358292936701656323370662867232146199029970623963946751688841968331190830450128678862572888076776712301759543290034129413503754912118321743371571587845246951266342226599731188319378143970137748846011503839541138007643678512440677484707251361670831303459421762344359187366512977037499949741710623319661220278428908920322976905540502282214049704347949020773347280774572019934978634712362414234809577987377731138461569700461141142881272772640473704021482011497184562231443936007956495190600333917044246069283754730087093529291313194380224616953419861941718256729333838355387758286663113301339626432830420179142345724768521413569035202226901601398953684471397924976103155196190679416170950119507253129806878890841163577287136608400063019118948687850508970134982858172950288460664266491390232698913550787884807218108766710565848684406989687323029326457204955333320990628131468715106028181655976182957338686984547520526999565991403632392864246628095152579496816578234819495652757983633998789749954303979637453325892036610791665504184727013707778456936494519945056601533713875167319445839648859494812327636622791661348869703385719476647478532480486900399561988080437922696685472273528998276543083321268496587274837012245221722642399726692569419883679885485911831256667980960764235002322423334410382515860710585848226880623226822499063677185391698088769361569819479656177105940903061079828019773976817673096733044493372481745080280801823574842440904615066947945076298807768381173684623189264548867813225032478621256639064776678426306911276302350407021749191164022475087195363106803748147886807267956077634420090633836590389759183534790835575848705523481449211878320546456575893638729825789942990686510544718331859614495997161225380146643768905358867009116039049851326124097532322687306798405750904171106882881981025965454909318236264446568102333231301419675114867459097262408224343276183121232233146400016661222921236256065251739190320492643322771532296094398654155426328824031899202212656610415191669700610158191169965964373480579677102766479140416152795100858451969529414203285394876900861833172905672064986720870931447511583227326482409156199394313268044317610915886265655601104922641778037484242184897820820104309936194227441508011421926813286271490879831953887157784158685123556604447402297284985047849745742626404353347215132911965452191483132046875748725826421949717943014261478208145287082835941144453095960763025519158248005068936096440523468459800204096612492606759294935638100362366075030760575904686683910042713795418258112286357575841724114899985157951380369842064841294277038227762951280419521109791587570282808290610303167928564653081204120662378317314054417250053983278750305577472121622911389684206448155454548433027284687339854326806188106149613053701860331784273842828625874268212314307428124079569523685300595615721253393992161836844695267009549343803490090289233276569241648173471902047426058088875588170700689088526682612526149190276606204586293868933665703532656483030420358845288791436243434043512984392889879306444437531697648902381880268100886427971725724216635878792168877960382957477954466205726202629739104252410365561269891423732555651319766201070310463331968788180443612628909042425957946894297748251653644772047205421996238304113764694479671602673213675726577358572394647345072133704337126402389795614679168383813319167735705882483743643455975012207892129639171491240121784245683278874769309917986405696247286693223941464861909171814672375725812637003302125351537024327151773926149039984400215533536618939629951245364869587342621682420439041798963824888097502342118368827981577891155036924323808306186908760561787725614647515723053559422866464839397156112752603617687732812143507366427252593424485218857080221926436525530240234045105858579311189216409876818138152138400790929238328004313467203581926374578399726868217380640115932742062572247139776352807285050816741962162967038062426372494456362101690247380381293264471877744909261474545617579660082037897182734748517883624811345838247600661268999643926316077276210849157869001691065606270237686820147461211913088747249775425852579614847078148123573811568886210548525283535660116442972924578723859338598406609338665687631610712972814636888119909937870754636734532262602172471760425760865786891242809595789084674721170126540879963865271833374683887313567802563824405632459000666024472183717511840391816244830735899819089776219951783978162316232707778326788058474440996967504484724182016681951335834201683359140446635794214205857769794026199067990848830670121251687842904867729280686488131463495510681080896201235522513816949292286595308780001676530422904203247328622395974638055023828824272339240913872052672204711924280934058024313592837471493671746344662009491926679546495408983777026883072752018307512853487690033244278743861349741257631909296395503844154204734865781253814182358388255524244902344709354231280866715441233323129348949254611245826664061906681102768343638921064300285265600175910362659350994569340376731485068500339884621383126862132520868244104276631759422301509081265813140496977297394805744074519738562298721254807684665749884664357157550147114371408757475938680467380582289392289729953943886905493488592061788743850810533177814213241251510198607599533167358722256274902209943012307567487159111735944258717528926187514406442640492113132225994262055173822569208650368686538310585691714699194300851307986042243698944642647610330996058198393536302360449663850254744399584268662135284374968852431970243352793558705965970391709078421391576117370624655509086645832397810637810745078316444584651945562668556365121297646675214393620289629063966724758215342510035057652703672984207145174820713147789389750889388036351846121874165629300371485064993842552924849566442153094750113639766912820709160545225045710063086500791836978379652609975182308126420708527780025674852310375754636751724710350757013913199140056023168251774030006034051833492604168482489825678476105154502947428246204603451279366111071665256807672763153736569286609136119880374002143904185940355981166432835159736233173094295497929433095897662428086367215968862811146235869935552747979786045553121411706081674955052818948701407904290494033677681418174122943888744513005646291745991739476633794253703878084594221442701513956074525091628219007654077797472903256964036093657641165827067802767129828860080314130197758534308349014133737775668761111670023934917246816119065449455687658518791543168073771881979878510483376757587145593590330815334785406251283752192438048609398697197839812002595549384290659408433556345230357013060171051199958804199389108630905491713408807373360857296809025762364619586993925443672985871032405974493916659896386595667160691804645410522508422887416871916988867672767766355036823217634225025584667491544386796416380108126585875550613813214830609995365294248815537634497346843371763019826684415186189434771031868161791424882126944178638982576156715262316588010994178360467331921541418551980210540563017377956570988680745285458524206974027985737707100124123028686157330792171990011742987828330413060414017053124814940732147321066761052753351440376789945540944443072677022709669272492397108519877438667349640844194899913345405010892846487915013739182660387493838285961998684846110034533042418482911925192110885518190399364862616855938742264140420996464763914943519710390037292618805951887263889001838301232098457528646384259406892187168518534552773628726619713706557204866428210466152131768299171743639414248599191845356234559288777075935262785486284559062290001298442530002665372460431016746003513775558810587844378058848287783721805046040250144163829027118504896351748383060570842065377511439449950927826426833300260179969716009976369079424512557274848252076805253856102723282665982389645813423013533004343968123962130504267757832524299484749694022604373981191164460281089039538437008176219733899445844176810864773182888155227187167954086324060396967375161567152859994769646120528195067782203075700577700417750904464632016941176937122876574110853156561177667992650926998254548402940776612698163428269088178582370057748198639904551810929084338471712072910258172872513660074428979164926926021822543871560844616406735147644404347926074768860714637494101419725893628372265822608705781448937379465685834561638466173235019283777004703332308249672335018842773064055917471768152436930049504428432012686249272166473692328817558690302093512221677348535803607139749886940727002041171722162758166656000901959523060808250058972636234349635676705601116797999808890671245024375564214712282394710970016156067445197859322415069173378388774053357402576471767323264413087334808055532709235040068569287459234129212021812930398061656304640357906735526998851296925934209906533447707809409874310919954446488719593409654497942355073152580991746564527959837216380371023870321914768491492386762037729080796682886873067588930525845740096672775261052948684868526985305878987875303117248011979824432169008475181061422849300205310598561989029972166767143129921706248328470488872837678942947888993166114114068253627356617233057021238427785828096359098664805921281629010206120947645458933494502182941877705430379409021807920139622198981123276358972143516850048225068615477612149255885420905169720764622836127232289341185572846045343756318840206774624473263283423131064768512032400111379981169700259229304807643436270808879879534910404294664402567794589367777982010382583362909088005604335764414160010759288562292152873623524505317394238168984464851935459064349889266138282932671393092420283275738591958761709969926576251664483226249816553028039644459805293792079467914141624238057372652274019277936872978155912626853840736399135192857152969773930747626241094761881516121582228793671596957779148404020788645289451865813100403876914178628793552398838460578135327614243559439393803973118169783032762648747744473149431801168177231960714397103628420163625418192143936653542760395022942288931649838566862878113092984559860304535885130301079088401379405046647843645084077765689431887369248209103654436668611118896632095484527747772095183265928053973287091221765853531171941913206757793163983495562337709826817065800612307708793779871433797577150768100298204751083471648749075258857509653420745939086548795584784560923565499709947736700826982916313809964661748770439328449919695431629630125413906750141595952044745138803382922749720191423844938457266256365374320319061644703496274909819618304581291791254043853053643146280720903695033240258752981040166210945934288742929576003776985534595464470610690191000198985136455678236144990268123862510662302739798510815069761716448888107855978658497754044274358105005502516206505755404304534146182486653358420220836763003479376082546767350868083505920971520080034411049098854964438468668491627487371005709736689189579652744948526769018919390001311437885950901835426890393760282517677967351181042478349925727085981315654858103667802769813941505646284232745290809172891269965915564753703207165412639118060919769757786671044799522797523351303338587638988379404408208110536345116234470918047306927808198102051657497965820947207746020281525855053729705476851319771840504457579225949529875969590256830892684992304031249168231447574860164703630302510967189155225669236669244916964850797920744118436414720934516906864696606048885845327103611854271815852916750130473220317410508840908855044799311837661197471600526969238717487166512078911540619330975216777678768746521187705532080782547578823691400936430167896755837169688505731714575082616317493925388928283965795716589358020670075530298580943981564456174969076926671988279751963871271190852189840043460564438582649221510068319447659569693471054633438998647549092203244309741334559308713778165501500770388729892922472843024474836478542152196236662330591905317714549973966698836348178926258328140941096479239993596757736749079359501445372051564866733072752097347542263239578324478642733667037225108496035545934077006579220121499030235137067666584202405822527496969526912574823656264346686111688823121971332624313500652608574524680430497556927988348406426373301460739614344996997061362796197414638818071691479629068079327746749961407884855951227197108412361161769850739732605380894781676698912669353437946667849797247663785610436182570374867335058034908108505200186640822312344734736176569215294993803575919855412640384568168843957417890596083918093833238352743673174932386930667195004979459568045392907372219230861602881754939751233455586575038503359277397375800210609799508305523856780768941182501005035443038363925271209822630742660780623088542624685019767478527455630014599560912994003104138572240855341627465256174812592159625414508163753164928017905059375980668695801818406072087363662110870351649955350568607623442782980443138070903006421483491994066515177672045493005860639484997321203246451139634181178273748391522688481954855348279731780960011619233046934501456020179622018131611156500697794739609273436503379130516897854740268267964088227737229137455774449807984589405211708687524974954684558223379184839521786386450229780353893270132602333767591463191334661001536994381134941810825914886021872998750816711341579140675556740642148964750748657398333586360620544820986552175163993535062705490729071601442734134731398822514425404454084139134452680416592023959130719231768856005765415796999076560107919839834988138607487080905046950705719670428540942793353912014275648735282095210040348163792845575914107625159274753083262095277960381203717487735717624179306757038292361014045237560527208737038588904832337305884398233905751853880617412120772578667710001110245491461053355080971024947415842546298464372553354822600194099067681598059752439597051596736754190069272892896778818921900312789654468365710683882049678249553673722892537991034552034421126185449306549189187768903419298810470337282544057550510070814430981396727898784902997371445879211448792216928817108317118710069926927362247815703078704535420181806237759128211469796867542771570469702844298167705272070838205327335448411952821103595261214510855601294427434693604298732036703160019094437780488792129768910391452639494428636563952775351815566857209387443611764598149642691802785946491614315901991627195954849809987341591055016617466890893146069446803792699236424507045037414314338260190926463831364141085500227755343509861064902977587838126287081214473592126274169797319720532992690324280594200496044035656080428700457603008120741848297459934702154820782911178473031784227251925897587157141815900015502504692530479624208440864169959474399429376090902543971332536055108054278483879055541728814972394467216834573033502436676156576175520579582383800572681791477028265128859372367709661650909065410971280080047296778255601370965371991041435432337399305705922062745032961651152463902281448799635120689746493719518551090851266889512162487376535208198030776116728298280326291727879604894042284448754382774261347248626852263796280544296945763486628522693031130882341527397737714236404882545871108545820410534157008933915364926767862868000163248165573700158191949409909287382676730429222297967689237937519976323336597911363324077467220814125368532439433492247073521240362232229102132920109144317953379314514740019444098144495034940074961155295610028323751482568331018900345013505857809583420139230545635788961112525325858550702178141211994333315309039352546356572950678349445837015276924560562109273664768379960968648321684959956517107019688762609041149731226729580091335848090982382126911947458231875708134677836626311135018008894886810753655518071936708091445441681966581588148852960847704566681606667701055695728128703869270188825691724827568228712869040735988324302709363394499774867906258308950596145000319461759657898096772224252892096305595164524112999306417836075026803405646224121138478238850762311813205871499233090771184805805973016795899613092417686967600882308109630258899721932621578081960568048146459866283486405766957648614844402503853030856859308634361969835021506984206432459119093492378468897845415800273911244793127468899448580189220312706873073868644901230036505774762691660250318536484270038342485760196291646634199876322478125488330192926953279833337690516996565565866401662426453428439252087017124899000143466352688042529049774046111871981836656306853287820264298715173892420369687139135717631332883870569069405625490081867493712786124798319403870519367700960219488996225784059442092494245717703533570888936829380899854857014545852338921624526208699199361089345638987546165257594330889200640747024890747711283239113399268851064857589317706851377508603468281932821888696341834474023670890899877401820908392921740413356448121093711607473619942993330173442227931136636740107234906087549105352606864151305860156672474951776282119530775752671139315651620400971371335139680229910389348668960896849208652609227341255127174614663458344724253977207821799330827365637973911898914452905303602556884371062320742493000598853803142875404198331624315306878485075037179109470505622664390821192711820789525117148637417853518892312531710363466486163754820109689431777303486688458899008697474972175231398524058314145015480491904560535699944264833102222444965903350016297215804993352380855648085338858424174198842094106763153038592931299084136998515700707110524249961626515061439225863141644648108379199155927132007889691731653138641062477430196739447939334037489223096033621905810199329568981019083619972745644704778702797098141094798427537715061261102096271849675698683902324998032559515512888283389782432914049493661878944945116636597022672340910442096876629934521545661092192327437462984001447361530753690703100894797736597623667501029200489645281495939385901781145598917104603496894709454194733349669412721760709116111950319517363462902619448352157144433408974921429906536096508782601602684609701048990057429584511757414159711364940381141448630684810172962921726306439724336593751157523642347766302668153072160319435603441278135075189607285832095126635120848510314867266534033450859642200145535226973824506327026891481159380064450948672488747180153971286369239271077588121254694073503311947875404919703959241300422658877366807262510761653004455085908397725609390010054710396261810047437581079128137671343409622343427943636461588044445899364084194063406135493438365490786084497463904448436869346669362856426712011515134649985465340670403124294607924369354134120239099807405713170232338448258306634825862730044939290771665950692850359278976183206927484763355086532186302077471920192861600942607961332123622618643468456947496610744793493810156243382385896573415500665351531247755524213845918363985427828032731017488618330286192432208769962464568335795033935520348674986810874024076882866925823936097104810508420618945988135504288410397123586286850078555270897125032020572741451270339426878646330840235686932348502771085735184683174463247038279459869469200456183718642781770575974039524581214249724939080463256691255742211157904663610313536016221442413551942651724255235475324445298580489111766581289995984844899297170408122667377918510752517464454241809757276581792836410089741166161250553138025354941097460772084520567960469852676464196152853328583891363155877971442212158780620429556269020976953004560930569993026160059760744910400924529832380538357942163178751780296388445060933027731380866959507661481077417489169496948340461356591049481473393402717163701567824920814877607702492629624313851031996970470658485330805289774762494774347424641972411489099258547706986047112007981883255672098124184750027394710355519736301559720055273300284028138253623599594107310742410959498329418544159084519769119308654396432719856367865506161101855649473544251092037737439433292540495937439732232712515252579671843225112762866924206383158376359790562782249064185305348582428212108384602015719424711244167628061374874312712483260856781741164419620588301472241711725504741262798163295798534674576849997294761918350349083392809532917497012513072077486679859946453933396395975048854858617283277340980121027681227691581771048821840788821725412576272518173151852738716645117887946198752097950628697376519933085396606486278369143353214223278058353275072560747457237719581153024500534008466190526883333918631695221368854415638610641995104209854824299538380519913274642892831897868414049675991613297217026746538980606218890071283912729877557231527421897161920824447274179535240238218764022024753036189270640031809838809249964724203628833283543570684163870250133659463259839317042393055009515964294847265026854780974446217578797560572599199131174227204855862472112372788655808543284884438658356420438755876989401150877349908633532696411983075158035329874879323703926599131114391821513148784740567877905275940078378237628984864691677225933417660909701771808203008931788751278461062740830842551063499710590695771564364197604434677122342305773943351999549651174697372634378204488231723205776997207310631518567321769330484384482813764786103119755364150545519335677976258707996864499920824955657973197338519592966618597738947838714152236298724177543816791723238564996943851796306373589619071284454259627219248006455919057146784736036603659245601260429289624624296913120391968164774095485486705808229823449782343994672333013642798845842896149104999847687265210450225554820878145337724157442925102968526464940952466150483419621714228214838522871315118382086541458625856689047171464231992481663247532787940766966633244960221895134797927134692071587510697754724577034761860916042077400197274844795027418809889278885900182509771228308468979550308196445131374300904324697208392842134223091455289232186020675182528103040129777735361788400612295819236038965380173494579745395627256932108370733395370819939095708657488375883446991208745679688751175594761590439434117511905519980738438344281378106631777238350558725691743047358324759960753435409590510024903149383003428474089845265595448547738667441178758983341148511036362367051025356731498472209609155753223145248203821800220422847797519737978034845738339744753849467525801635861766814481135843478488769235323898736120904830900336573230912046634654710505335376760136644112829573425609789950229593996745734781690755837295765826485783126415823996741333355486312489399436282472291157983043385567763270567619536822307462314029451183791542127232296273934621043627617671071748696737446778443206069334794454744837895047239278560563788891485953345117296766346216863760969629709384980176565038111828708221759494968884496244072721078715243226558080230557948110196134566822600007268267872121459836472329567038197260360511776309771595253546152025459574456522701426386822682051215861051903154694849276722447469905762510965026154781940762903911851897088595889419032842361409954852158678791045613591508398270405417981863335445547434036805083832810971215743867564381412299056370794858879420983586162872196061721315942739087985641101639712087432304757148226541286675204600375153570676119869449876776717816784165940539600324140940273595380018308971529239962565835016342458679074759883679024970681810375571720472120213559611475789292879755313022366819143752188654941601369173202593617471963758337176580814009816914509006254564326905840338871316016455206095060755704307024157013036551415974687472532300010369562825745397251655511888262376274490501862941889384547803227951906363881011170132925201346443744564841869745165645048928234290034678649966435659797532941351907746221850182298416529530969563354247284137065352514292181990354736442444152667029783885214723953229977018594994494095803543384234677347117261348833200531541954548851681539871692937737247440782749259781596164149112469078304477889304513496544790629776612886498975962732810692338137311883629687372195843994599366247142327860635285870339687732259128249410088005723986511007826568260975489318689586314034665911071267464622156923415091657947557234231998543696129355858187059912968676365129027484169289802323281857080746153199236999668703501044499453810232976975014568863735031862783484612960276217447713672113077904295194619360242390376491453491104019667727565019967571560901159960481686883728519286960177776422430544489036951650083609850498775096268388731121178771456098623225277310355858634258551897316740790029675186662640587805276084267477718717976381279698745397607635434792155837408670984102261190221783025533238152134296101723084164299826152682019292115290299350426099651647187543854741812920604067962070944688585734719790152872363884010295830069906554593493180046113284056888376926425755862960634007157539321989653093681165531129861539377185942268252243507368369032352685174642024156247069245829755386500728585312956535768252591178106219862870443605944054254613518524695699420428788396579791952957833674648287958534074173567655832462700138757794784174413011872382199289109539951724631090544103627693293162033329417093123985662208354958090079160807840225423781013086394116495801157861352858770098093441092603002941663985679481803276121911172843012574052646183248322311569855125084152057191234545046738088198277646130221819506280574988211147321039798087488014888153711080821317852948707617336729000511263363196565483222985721444551913843418351873361958479301515368999175173810337961688145210107096417810895881331287946459863840486587250806842891246321429569705766604156703255041826255585819237661136340592952863443908745976885999986771671447800352554274676407931455092013305654796888817186822974133504490717815958318299885673893014221857331537926494460259207531440283178715822163918057784682672106351135055362372690072151620927142022035517495518443695670497827754876160542683391924054020932708392261228222079345175900320204040696274997807580342565931027043483766713581519692568618258169187986007360591659743586735665124568973454430400525979335568140772535107550419224950025558361331298117191036865724800943877606488918301622374717826003263716667062761124254987642768033784374263052043978502833108767899412744950204847693565123480015029630533952935783730618441592373218512955656783466849931279696970037781932222823062545447920932077464962596218014908499744891429928618243008537847167388415558517205874440391036524625186707676585928238518547884209417731088511883773123792776628326502555960586163250774594227130541750059625785583593427612430159082841168310163307242544562091157670021975977539020682780883178944471129960584286939152395500458528851310449299393416191679439281136327648605541247034244238129982453695471577412563802119653844918431745841322790196354784188921335121581262535222048919273079672482866849078188708928671062907097734414782975693783562737296108098786055382559854978739059714961590786919171468367329001140278367465503065723712453811866462808219063415173043449852604795834551328728239071026308363411032887739341891299074333786750773980375354636044014344160765720404360130871805294846375397016535882446337075868905778188157252158772902282946425791478106871031620099193140098240027774088272639421148061034663831797273571092721660073201510828188203665252659341635946191471671267751462033687012351844171651251135581836838392656724845503337735139342019513430367461763664389294942840819395289031082863758451529078042360169436124845489902844874099343868020091499860044765000456801790235763457464086418406080839686490753390171833121308500059957994146328251038905466287924783743677817714304077295911128538109609724783363434659495843468145687487760168952936717323233731171419083938530586220011337573925355660166866400237007620733832066366445953418607771159913181656754764947035515288564101817056745414574954368631065573817905075036337912769115215529846523207952695005813244667408866737228085958698576734558171290391813752985292539919197622457392473289688429536095001226213049924266434338467351088645466031528905606918158485276161257943713687532837706782580923458870564417818129106792857642043372895725185213214134620131166409089283818336835356298871140440650319376412664792417359552583984565057408770239841828484550840762945631507455985958631686811938836555988506484317208328321926325675091097262242016274971778153637433478402469392124599043245180568653801263484705311183769962663600062950050158513986898720929113973881489953244808089408009172802809067385104658874388397790302664621497727852084830237053906852460754401773603680402251299258504008068347006224378615533282547996031596819022547816479356675454863854816453960655892183825603767475224108095423562474319543142845907059912363014923463863455637355833646486158252814083545282813554376865219859784283021144815543339899329863166689511672126267569253992367822876860073101713656838699496284173700190812656064690113781872353951160925143924526745620036134403584831440215944849035557871444084244135071514011528210985798500950160062368631336416254719596570277349017000685078400611152083874932568769260774213057203917444455402803358896928695717608233450011057209862787844592330007266089080339735830623827804169544901846471621771293257098222801892403774092967166399653187133153574359077075048174884963571374458384808751921567288657702413404301424453192046181500729277930281326257137471705680501287614830333988287776072535488919005970275554323311349335142228327525904581622593742743712003010854903465636061343201098079741227248284339404702865498229928593544771459308908770800046724669366137312973037941876383780918789766649873612221710635634144473680019518236790463987664342524042633829364040042305440963500165204305586409799281496070671361592883536704290297717248898374376985870027081825985328649581653813891618945156646314192433557530401446725783101376556370506428077440808440619128099021394461444114987668688694691225983534020292228020482820048969555830706867896947034216048108788525072457367824332250069959320117395374742432990073262056943482460776755816533414394037874539394829713844305173596369025939197729359628528368665317349529015943382309129353982327622843439173941645517032767517615717152349067454553172535475829967021956720039988580288218078546537161248291064200962328600180599742559689566899198421401178500200959907375122425143475624302793540026675918922283506334148544854484410183119408483716849150556704561545115395478182568879839307040909768365285064076079058282463428814368193115560918699890513224637223925163878375233259831155018686501937163698587195360544771952862150670300568979444411501733223016043653076556320507750329968568300146215676593858161924324284802644545054005626791709453871886103573731712556063904075692881071925627662908943607321460444107171105990882656963822581144456005523248858873883121299770424520014675686148625518886284984420340313370796158173327445595952070361949515739680508235689764279152650807411168462806124234432456322373566194687606847886596815031058062974445899739681971530357294627273599475583964494956805929391999892004504433277627057297767597772195671198406640211314375416878470827320701016775357644892988440458458181476468523715553181962897473335518254327508909315013021330887912590613922247352979010361653045993558635604649917524610943206344899847145971421669368655328787700283960626880418556810999908437080296019068420101750599290325274346448416944150581700759327191266106306545992690482365222057872357442852043876752700633356719779341918676066243948651520060601040666469926767734215634971346239692048503938034072797960421289379540001210157054859796979146302934923794787617228448903520338725426956168219517903447954109671957968820337534499989189249308039618824021811645625326555789507769854524590025688148900593464497964241524253656436730518623061041076091983319134698518781216829499284631085006841264080694465014621634513595082679105367715788673427104567719751803349948337876117817998658139279761812675052073140234399789907125198778996614735606485420849811446826709138681256963614599748522524162168897822708768931804193949329168003050952675579785716880727337596389052521591671891166526751583005074320946737584951397710480007020969819710713319357864912049218225184247720540147243264306605211654257774806839887837900128506056266209773232193830182975154523385106305402875472791631781860793766045562576336948139497369028408220611821297978614203482983081201562516720957526564975204924811410668657371910308599061137507484763638899436965828597576762624201612304786443412444603128830764763694315268715368003519031514873079527588436882260913615484987164224421255816348262246924468200126563316393817982407404737664717904193939373048754045546446728402874557923714470320395998616287083326059352068567650770335200322653570406266820059057566379057495953497788001389979754455404217087182972411093300957791021795453281891836212282314597440651726753321760535037090345643977607110366988777349305998413019451535620366917469469210423790068670112084488824153853279790093007019503695670805136791694986652302169702238986595427458966142240404340778410066106150389955570468414630944912811877036077894248633305005027935802582538925155741468263695742052312805443489273645977745547388783451663325432563319780099732288601549505147892283284859501165239719629764214392264388734388510852582710833128562361486666188756940603151046780016380262895703316044161969874371593805212184979561972975372698150241098358809200591448033216024335604476361450481665485991646796767958008090867956371816930630460727450399075375830658205315419713731871412560315301464257625168529585108528505619404206250283894236872533063349690112588086438297512583188655004282940424374872277035699743434070307626677182321814337406398331062875691526753599673903103744529009862533880569663816461276844090556627501531442123838723650219700531507449616682594302660319544882286211788427434450611508269880538119637794293615813002390534788877519629380221867281032239856473510066778700951867053142966169023493813016406383420130638791549081784532969719378687569489757873600982288837696137097092811815744149025381450606936190513487801884145887201017260021702245990907103192734680399049441395613944330261388988626799574595001254532751539105830064412991608890682113345173709956991585785812092707620053158402093779097582896421174484685971428013173207202174079952126270063514106418350101110795416787219485475026785144135473109876596262875791730755905397799543105273194383085393155643534438941759122855063072357866668055692140725864078313085095483783411577563424157931646278319442982135610687805374647639833068028549269088835882410868656099125516842653615716879690404700582753514015224107965327178973214027038163064791568449313799628738321695324970161311872938068743807507167974031315887676155211982915800501474573029961899572563700430496223064706164450917657941605498787383739587075345944130555435695479256263457450135132759917296904847807878339329888785245757818355302255716043162826727546991113457728402183281571148435127979942872587248556584991556728809543563623942683165223340649571974086313863270421414725336389714832308528782044366823886472956391043623932833325661367979353383056799111554710488357338463627618282803411447525525096579456798972563655325644608765250439437301841318294794470376326120105881526936557361402321875837264567065456391917953216555884095848113508883913829404149178107960987978376687644678236945562625586485485726910128765095733353235987701705273750911788066268353096549254788125108551280860725187742376036727326268142340387194681592106126691318524101134998675725958331569133038814645972010214445536731187532487283246835857744159849240433918791368303413066675478105964765836397336629100069764717796775156952104815098239764651456873841109062590884148780649983062381952589909152483679713897715662081407360377190198797159555351582706545030052959870121955450423423629342542200129904195992388688842866220701883566235976448537466149815318315934973161546696587677598225499897024730259759174794094137960953629841044823069983254451587213471598192925931023375020040022358197553948671528928533896424263838990592394308858218835036020616727735361352694392490807160540695357540668186555877447621660979224300021923417349549350245227194949912914143199796813142117207709734426303728074040484689758163791762456510498330116616723996249471694593059669502771340977747534201354901293803306627852824509578088084437286866444103374632586701560068720209943798525803531809665049882567815580847234932337655509872887143069568970446416837479837922649162844842519935463382941497034374041316118397487499094211718850281178208870959469560014935488939287580597333114612691386258250398585317734269604461781414741139702903491764752874634290539307522090146040683866779184389662230883777343120023844825290478844333065316002140369811306212126305833277410302429467139864661713172630951327818738879750137572398498384409074773506612693542689775535861781351546777750785952527693003783711224346346326777820220956487501803730355393765169416982981949426761565802305273168549296679686886229715872409439279665503710382475339490008768070215212560781009376819052588366652902421719962781351382908740378045199063917226078339472035701898256009189960947403729021413443877618552146050231483952968711658578895619304789548023935304757054929903221446636531455467012470012069861362584187142778008814242642564739064540260767434331055657960698788175790732414358028133661045328917308581713993120680262741660001786575786085044945897843942616088880411698192654961435382183902690320761200382262136926232433698614399977039186755401385213770127835035047600632982000469716747365431571254760547665876240796161733142576598106454127826630775296320536739364094876213390344608145828571282227151956053682954843331469032804087340817922191573572894376827728525112441430427743704380312434816035529637614220036525872888106103526549143125077101789595880536498956812518603533703486989540453986833744531160790397400609272883702034970461864563159666358944266235556651421598667130881898609580758521293454186040345306795433461662275248568063610286351151383783007050120722159084500960920216870561331156046509317358035237892494495146263302419084826200264016738094728330639707581029897931750021456349674173510730218589323221715832285478280328522218811476431142556011108073920855643963832299469736906612388354964296909199683600754130272561098808235158068101950126740524540394807508992212023861710969158995839150570927217898397956618114512357199719306186429100215666244685552311648981929110964251895666955721755258616226027610738501524136812264466588105522058188437574282563151232732693905587432314329388524817932046209317888241983329898125017713039548143277160123525010495737551090752859297234363241764235914450709711509345472580182800502934260345370859874372182167215830965357614010322564964367669147557401925505008507129888088609762401249290107148292500924632292274687861249179390333633800018025487613137754632761949214967664896942952449185256063096611786810018311002293339800477234169168581134474107375453625205573091066535726240115780237028713923082617029641435344759682521817441075588005596622122150069773008158814784733034233303425250632190557487812257967314023937708833575951908653125856089431352486548588954374439105448143018533555819344978102009971974850393632995251716320814591455467981983991322276184180737606685619643431546888660110980764579747049546616630208817638947571476872933689544174646509465674400999740621592276066347783422391413946883285074636265972840418494666260265005030036570979990529367451712739098989564081413514760019221996147015376230938114247219406620783105354941865843182435791605106711524929932865151848765564088353739306697319269987829972522279695817133949078531839862534322681403551942801675593514242787256554071252784542229408190113072644409062005374036152888924962121537689720983990270334530321798454358526118501273441679153746106634166614665154090310438365407785037405670328250770933229008045250912133106162204056477125241915979957629147416876372396301694646953363780512528857109076146774409731041062273613396763419696819294372146989429450722974270278541856699288984298676117886097883823625982337931205028542699983991783572298407218223119701788251316476054560558437254583931908885561593895740261043412134272459232283003058721464723754452776171093159000537645642706997139613621969087866746227783562203794827714489149355828316323284861025773920365520750312214047317065394665214126170193318589173301043190407751000172273549144239198975832132519750774796567611035359177064958662936842987228509705197695757360706975702471334916935070510598196443952891379452478066818825238626700122002778548914434877685293655430835969403359049401269614117590362056164683680705896146349602491314418785924633238229468662169343150826871983619675948874868528885976279573241234818541013586203721751959026605771865511091168889928792958965634679240064771356894403412789597821669662053703974893302727757831965293932895531359972261820765706394458037604322330395339474121409865135903065812152545878173851605575768834254310577915581550189247547902378690709351423213401235061922161170996853499363980334594749831318760170083949302391992978663822523647468823463462048195633401833946461972778702031579145110177554821588481434689631785360967714548386535791417004204574241456728602763938020672455785773107944215916444959754092647590936488007664298187904840062899832511698052373563212059207306260707098941678037196747826481825763196143595206768880472767088045818770560218518689884030580498737150647608615273502047659871584262839006905215219556581894705091545972386434105549970703277424149930201410320519864199575377011975041402146249269516029465567798364768604983378001244877162572588685701423764804477128960507386495610564729217876480805199221274247698519862393148766338121307991381213647977311383419157494118539306299139003441827277017477594312499864905629264084247086561869795787326244314362160508026419352379100411209851318480481315311111077704582454290043760856786784228141349635812592745443310973440592227914052805857003991123906815789322103605856360711467417809976600202942192450926194840681706418598735580947222977485295659651393057317703349572192083973158121715673546791072732119847975500765860754572522214824389750698101118217110473866940748956397918210104709906314885328891262919731709983330381032695975031313479335842963650980242641219742839823954938223366674243983045359216255396251829965330314204054774241398975605862830663370466101733764042946485151112499756185698029705346556270643461636806210120964393960039347159947796336784661508636866948789401267392772160154933812498210143049672410070344245488200905442706408138027019968715278396713557540749753339119672818295078491052743146470842775395573178863639744564948042864091832847011459189174312192904476564497019657956649158348815441130069272429465522918977165561303571657431403076575556942575939877477574184929774181643399435599647007980902474634202760122730641977112830528711779937074510075659535920892498040097587939270272761728161723541462451338485395227069307929041179082859477958203084280909009852295386463702493262157506898063921274144127780963265489413960466507774420482071846203884768666400704590136833310018399461004647020899766640186364304403849720698805935105746403564232560242478575252572978011253727885940906334391328097068177263883411141070077209248224151798419954972983575265874008804329281282712896349397237433216261898343614009983060882029549251304311904896310044917907727909109505464711352129375250277159162317327373187643573029034860746923315065796268195208103349485868673831586070628927784890419975699527521152593637960957582920913616793761052193252958524480369885062661903173210622729045389837859418460190692083252930396824175564623024300201583618628284590515504849979315319580198944134506940827234519495292427997417773582115969708471455322646763752658732805490583359269301341783224678587705019559040424635030547626322397341436831746387550453903759933135123850413910884188616800949684327651886345687044342550560302108456088859218712853403218254739566227454234173647249185514888935807870840923142645017692864862771008434955668843396087017164591854872991847498901084069134133855157861849576035013978054673297911228196579808099803784662264065158893433287747463749548224446803657858715454463296437091793446250792267146382323502222363992195166774791876020849490056125804740844100957039438166262637751983604863359434643271364853184997465142218465755889890499310520810582146377153472804623312671086594118842443830268851581152493775226948094890981758316695073142095862387427455190795011580187050897729740193247223861056083758540749419249158569772442096175874840021062897035795397748521705415467749523176249530478816233046775061740161226562768067120268611883051095810600993648208405416657719140190015050539248624430832772276887076669263572005180201907124436355746513550078917177482534997940611580449222719459627520510935755279947391772546921465857749850723556905275063328491016890915969115943865230081256589900139199790049770762061393886572673894243952181496823371014063481493638426243026265574981111522423638519247440066731625008698862463956592772895089291161119922723975880317954779250120198773843164858365524168544427435440733485911138941038327684063404542915032486826537746739959266881394414986602598384592175823705159539312933507052236660282984831600646926418392649654128417394423761176328109017497000923160582098240246593615075123489884860351455208049845779254851674877567030117853027243561536377148009763500567201740139698223653714911464191529201352292837821004744897914242729810319250374074983411242539329510234834706794175762131614557681756673337349900276499159739198658616206530452825757627412047806502803962491361801370902470054842042134004116086184093890224830029664438276324588993236151941482563307641843567417838822795513922205718715102607855277990857513122198525841147586735276482507506685207885274923198032072276288766695589818440396866040522341002024877477400237093975781671566056179590077448167764972834631807250565217660270820209654351868876955740728321643973398418565858209570012122116274294318846174640174197538260293216814540214582441037204048415051087434294979467805506840635269704685132709424998456111462421840347560018491024727829807290268243888703896957528997005786078338071459736424411930411384230277590594075088694083840199858605562085656145647449557085780384122147879392313518610145299006302983595461068531639554213589991131607906122305230715827897051875231871001197536802557067318953173516757064095948293452408248636295199306725302392794521370092056553743989266936422375321962430475888711074397379409025781464037038767385217475416116984629288742986309598112189994222813477100819946165649553280589762371656270794480982585544266906431185053013613788450992957645214669333243460060772948374042346663211682095825092537860230748425403150049485818391192627763391591796700351782263227060143608836118325409683278273425012319441196839766214917099012634441641767220261294057276609175636381365607791575176238971503311276477758256702422035942642674671175483757892479944336293724260169898713706639455790039575837908554827361314027100864242161494938662944651824079659332676090727422930804093283090514304912663684739592231941892645469497486254411506610973032902520045797797751281059449400539587925502950605887262593920946535590505658606219773875231195518425288803479774797511242545865510770390157145834706713904918231315377807507521924790055012219453851019629651293296931450823800239705535132291523665737856416191114952974040993595811982235009874633988627572730872163609162307398232145129156682315638286495124424490885465205608921416403524559010427895316393146544991818863361685940005254897799794038958370049915606514507811993926625888090755621970863944423275718527198907880897748136281918508479884851409668751866385465811901936997322984143099690945575327699336995047940596646807");
        invSqrtX = new Apfloat("0.56418958354775628694807945156077258584405062932899885684408572171064246844149341448674366020210736344302834790636170735168993149482616286636548952001776899329283763705959843976035246435021797257121158024577282022055450852717321662220846330811139951276345448602306823769091874515874661585130106639835777199937733416035779887616674407623303938021635060164362169623505045831253194687945665628832581552162443454886570318987743088177818294838947510850091845853422163178258933525183301202462256222372494727003389743126429965942635042830449036331580393769598560815933782922055738292378823157923310576196477746825309017156180805952222961763246535904008534429486250289473921641811927430977422556987879577904987481913693723969207286063342476796994860958876304363372645587875154308093441973848328849137917536189566350924134892530883039496540584105725414827482347517474200939698834167771488558916783373949859158165013768598868620439641290728692896380737732673111842867880211552486621897190889680753149512014745527175912981527212206323434675741526025618498227789498695253016694969685663862309220442486143313106110173970431655443536461630026681088930477205413027483214336586471029185494908383876924769637405875410315847331707621695314489049731739811213852526289496658188107247230501730560007305550738026907023920932512936518565440802730304560855889331865258630525883472098109161522555134016344809607734062906765209247774445791565993628691381065069739922116063501838120163482794525558804589719527990309229885611554562819744432231840948954822569892318934682036700320622921003243956292571852541904557883683133559630572888014867327142640415611826594705292148828210962155611587788457586937076931656897573303633060580804267765542888724683958653812819175444600411692598232070101487977598739665524120123102085597258077416323243651968705793718699568664131637070746016467456232664785276329361255603673045814750967583692588177510251617179158854351131944977951130206446574423546416984789804083750107470135275188047822622928469596822727604371352963566243917919851346226220302225583139481689744215628304289589197458305603292803761633107007468446807754335238804309325018209344327697175660458535814618929348300084752188356315613485030052563131751150240281525357348028103775587309687205804983911678544836782163830873961661473233861336858064369130362327022920120958420518454001622129760222492050209138562077926845204513809206519104822871344651158875977646633495516427975277152900630459262666213210247412162619267265257583174894539164002150602758312143432376221368865926588667722327005130728367784026758852702612626204623304827833773093459725104897520711061902252643126658529065705282455896450805541303573152502245105699347424513194027198482005017901051218365393350015086788334128533820394325989186458860182128103237819376135272641860975261957333921775530690182470880743194704770603952658525202275727673224405352835567656782748666180528568184389534014481806430664693634465400747291871772392106949945262293257821840527730356884864031512332566992523470311027574933252903059744296764363162772990464358844708667222896394685960201086760903071360480516376504524067957805736895016949964236950047089706313626078166185619663218284700774739942287908873053227130811032498300708800237075842354520605900314384741253294008552812549695965181921615707903796758028301539201532048375107908291922688326216758396417167461672753995583775081502517299603949425222840205899473612215419053715828613438487398090522305169387544617795294705631402716509359410261616481055491159842227135691346046233610415171336576112981414233406803319679212957118651203363736195534725979410919473813218791473267598462718552551887579755934932345210480461207507601887038050614639074252484284638619434643195046376565402559660603942734216659798731847716849162696760061385456542842668296423244647412313445991707051854082795130309956986576268273713346144038281470730274602574072210808368599748628671260265492020706920851787934904475915119844073759810215367044415466899659889272993409562370756870869893890621612882247232460148836787805411328346190808716586922092480838698726338506580003055790165732524446934710088181103809873111778766773278115075262361972407994788907458748710947494173554596600935718451739462183954924577707449183853086537527561392821260619897373724956375315765474397665361414676939685574844311545379091806015409735383982631685150412224687474671032170649428537066027921001175799220944018003376216676337577031769948843876962683788692992242719239154432214615739553487871275598873589396924354559840837624070484088807241062153876063344730857065896876916585669281509326281416842018881377099616128788119056941410266822150121199132709060730452207326511286976324647213722199843633641893155401207696743515391579486258659755045027734963665557776920242282226296664209761438270452632519817671957350102784482375908272297508510350647276203060623166558525608966580786840710316141399464391830066379930928702784472551007654723726806244302893128132180457443551641501321801547887667029097728511081095549334578653493623889661317856388711379758241048704133136039581329284885697528781569100784745220935753700794899256814191993816099536329043935814009657977297776647888080611452783168101341613935036512859286861680551295885587664116213501206060215913393041424193581433120163847397278261400943217309678227731657372659580538769950456194846016356587217452912510997741871165922842956461745260227733048992174609822499941336455055700400091079139094026619822123924323424723922428816383588858192984517844322657435952794136251683862184300384584714186309603848620313884223599144544790690239680790322856835049907159632422296292072556747456395004396306275410564563213492694072240005675845072296889636611092647085466761119828383002170505052961600922439930591366771883798456302407679951433863541650905825984501011010415586978798012490727195857669714028612616168309509346497428020399568547006756578680481857811049427709468077266460548959234783947674284597729947283724066842704096514497651208435546344586199241560190818160196901488584783200289028251761707365505269338610774112846690609234968183839179097327524324038431555384028723430635476619583023307727725427579741121155397140475122779065600658606347957997361668469650035281403385522084645026356575719131899873578716912512315486999597089433813921910212313899342074502805830929449871406692805292351731421992395545477344609761584906031639614048942972666301071328474048606817649872421181709716131240864262953287620272909860845406952454795283488518090436012445194196983881664892002107549708243826682648926296774119323280341328917027476296264117572639839456951947272977348793950106760922643774907118752078897549561400641000759898506470614273580237479767044447054471927497150909129511858371599719765968104244868344376282448128597664215097026213842802467700289360451691873431840519407443676963504016351717108772840719615761870302651891909033045578918380921822890511139111583863791934383436519314201168717314955454902257282053032985559851193595577154796380631798561406946303956263451132003060823529879699361648933264550451032214731154092208702505292267199740819150784721171925635477999964074784639855761962725580342635207932783171952495635799486707200343303157624935118217506754237437914812136080460611355415434656674931375476901144056955207809833467603903667011515719710830751834573112864828378201647492746338704304511161152004594782198832498817584966808755129613399252637064415108473363874672382723788891628471922444458232732427141042428583844509121453036018577066317263450710491170511967994260525806591645178275297657373106318072207940287682611533025171585623297772427241171219907894494265102830347867759540929734258948935688591148245660056279565684973679487620178198067092822374125164044546658422771090338989251192383919047911397751108629317181946147533284126721958545813500358355215286650621626130581019555017507732202038319117838330072924168815088585215881340380753988842024903102020848280555739997781673701451416182819651587634174209657682489608499576221032916793206764245211444429628179035688719130399272377682786098770108354856605057984957821407380783030713352876450241251052784051093673990966871281774215199117706363912149542121376398460995051909804994591654614963066777817770243521595140619307007899960358235490551929110041523071463730013448020763355664560328256117464395210748442548856278416153297890806102766040904743809965266423673534789599081948006610961800601632417298086692226979086904646734764226386570194265245360198206740364521330422644369039915643166603668270735411348526634638673965388476273159285507021432124419970148735056297184109713344535168116039850386577221262777258840267116599834084532857852784270894683958342422562254146067483616849288308024521564961703047678147893490899562763535675106967594266290035099060992625882196592453308411485613143891835429474753071213431922507493266488639823235551469584404098425929283762672361692485916515890464761931881378780249497682692410893269086230317484028762713487845488555083728422686493709605742856747355328821157580551559427894474536698984131636987495359677596118905155146659311839581585060618685667804675805538141233717003287568425461861343147202485100888443557069074093880630420745513950380294309663415827363183806062191374664329701042625983659711922253593103439571636159823786014038534161070663673931749878096349722760636886649120440890423589324937767423207434372192810169194197554227998757251546063095329588910012120996656426631237286583023288290921682002981970941837509195324919622931637593287161764533163958803732389823047734690441058247778739521371717374609775245714449555490149260025099685498803310968920655683666035871747040117499074747660397877557734996684229941633736797764185843325690787007798015633190952863350605471113019923352504652706697709888058858120872950766825543166125617763407792271374983894873709276480711674840537298502648022979016117433124471050702942068467773027534941037534231738164795692164197614856666316219792467656522201380743571758834455207793464593379420281774367250844069957532118611304295713123301776002317369219717324638898061914447646983454958982087564973979885533423036797189000448063041532743994679650060058360314537808419521394105004894071240811520325105795866412887873356195607385152055908213789074362804514146976082865241775290189873794807968987979373588825020615297497615878368171807934314104482657007812162906405417523594503538333251064848840092605094720755303138100964844409242356777638786792082952852903742207317667071493369604929047782333174264773751716413823536407670941523624861860035862948836123642849569247954484080790466836274995321737872968920425849576493619401731476056314342134128772629514188849954377616074076693448527304811195312666152190150679769478492171110802496477211681693033197420180499170627862122965321945563060290182212698319146513668824097101792025538350446204857481502513979093672932091328745137030764338029554104711609849553385682729392165812882652605117732287341111444209728648982149993974452775420452465587508308613362219649748260125191273639813605255744853952948461911731008093596738005974083489816770651222363514495952535312806725639674756955564241230715587856167414841295068560759643609723228843200387064519983259830016337432276626690230660069118991810575487237351869365729194168949741851077880514610813878701865837498986898602070328686821824369826731674516737658682423615648432731828242178668565426325614668227153283108869749496812258749518911306477301618810275610685578174035354040944236016043900298393454392244212043921676043370728826281786912809430354579552272975898544635883991638754284444621374362600265233439807556649186080927036987120303468311974951650757725815955409798807178246993021519805393830380525847960327266575099161403433799942014630649126505903999464185096193792892664390710874378682259188481710386626966948083034752291854717648753938558442348002899872242984767394792422156128974757501626566915174532623869751187418085507433380375363361292573780201691087729554271989396850241038389631320836754523983174671233498748124462815905862014322180217833935555264252786617321986382251704094837371407115697204044747931033060401767969603443126968144563972837638803892436087636692246313696584802936986998696361484086577887836604543627350717511789649939260036142903439729652964222595310903879081464560680365547502947208787024328379522049677185386434267445839671903517063438639476077545301326203322943744870207068967657671535129763287321430223871292467662716440261457967115089058993565384142608932125089536680325687922620580983380160057740171470664416150569856076837461854591493126597961785777431991966054145137117647621837463767319313956436276701134141893367220340254291589008957950026920146279537306911964325047278984266716200089602387627569229444077828040255487684753698226023508991825667664583680699741169752447423982061190381433152475494706602461670259118067212821341395578400486328521183527038443964541526369395695659064642412601556400375630215068802787793507503188268608174483048333008516072573440670139179836842348936400064331702420923429341560496926331278184396850093389779302445050710591790878670419538028616560517453082417587263252138573854878957645108906511711491196708530742990751175890078958279972753355606693215423682107002101495960404243275899158183827369585152580455995100737429426810781171027219708293234192616021628481539123093280495725903059882004175857489590266545831212063797270458365851331671607430403156316802305524428220674833586440481168288260999708136357660669176779393784377081337231785546526404303268433899539624989311632133188393754096133256249250361183234705268975824034041934772204256822657510107240068239403505522467048615570938324602114698715682028850876549009635440721319742489934130094302291525213680332010704718020079487010339566446289011048935796847993605916147837600664871575134195152199300786746877250762877137387576433007029374383529129726296212038326033039051643190889093934720271631076871240568387380118186767946685249008070005357025897507678834094395658461786251659180843838908128897190275116645566752868413180738822032340570642138104725685320501743217472547700289730083078618162168537205558198425919964495819276677450448023972633392192615984360944964131141454451121673195810542185058616527373615108805784066704092099130406678819123147531922858489626906993705606837993858621140633275171253969270124185791833372740881887678470543445639093119237369966766103720372935748348515501071359977132500810604360151519619045327774794484531140989795014963704293399917724245215858814413997579100261598683885190844221783028801746272388469054528748560228599822116670883270064317053149110055417365072997219489471394555839949355495721353019177721883556052081052663016871606006097529022687887650051378639043620337613518534696849789091676633160509029855981062336019250153839756684014378004043963025627547704340531854483668585853663573963884128995024476747833434544681295569906888105050635980140934679613427184633963364257821144090300975620979774939943565600913148232988493967596431551285246333883037925290546835948042946889566343576455727621844310228658339695273231439322173441706406027893784823054299173669618680064567912657880851687992857641913018762103083155310497668456724783602305544020201745341517851141812425501346879333953130941657534876571977101231668217209789495682749471507383760150982394307681007977167742344805070924963491141368341421059751843075275630555355973905658121413656252791394474746373886494032750401857481938376498671157664695343548669478278944223516567100332335718810268943218580234725547475460337109433909851542404514281976583871553526712083552133817324471065153668702058315349296827243179616072798564094729501476756758774694773946474773163729946110397646304415465288955699460409686815385039332149058548548533946589230193164537785843719667207220715799024089188790498951228526612678321792971146648955529360353075992896743183900464168025007307255525513334885759433456866942390941912984086508789075447890577149906689032831759822619963465575731337545762535798966416494022060810473614372537955827669230144962093483454316335451733074498509868630488787484865283874291914972130359201865590655000934083470353173766871685936431318728381996764888097708246030375619387241546365114179266082058593433531781815554697258454884210079218456167518591734209322097918415297015184123495780621056975931257663985605996992211602687344005196616783519016617564714084215178858611206652384090381645212315533415156210652763022487220268608150842562028980687055338239823518490161300633713777207277569639104230537845851419310867923123053682709373081356906591074047135430010853003440046606841906377406541903783177695131196794169753941492690897071593152474884615211358649705445910231625736423389999691173983000215621159878966095864842786974113389649867557903712028359236014304664587111233568332162153391596936724231482963812914058093742899912499751117119583395660793897107411834769799400489795923333513429198809889704910715424032899481801045978504546310563546858658788737311497729591022199583534762687520319143912514750446605827442021440806634293972826504461895478582430532863694151757279340558268489692568651820266099783573866995036060022647115502591545627141591773345948843127198110692967701909947201780927655115278079777845489761962555497989413403785235558131808751703391877864852538132481530419381237349192443588113118055720295823517916544504160744442254260585626111486242497236480524121126826801927659218641944773316160259890348092290459929256137071417944992513375451387278537657554529939217005598860089951757037912979960256501787029388712333369782396124355995651686368552792319309529384531004533759956488227454742612834748358560504949068977163454178181680237850208855568885078211178976568703721270885402965723481209879468538307007194285567666322940473227549795387459718352155346329930922781877406458718723512720783422577255690995216924971800695393568600497575581020740036008255157285709410654031527729306574350209495683762079509554575640414846822714053031575490745232886806180795802406265018967031866713865061463213414388662642854490092266679046238487257856800398820307239118469100135405146750825910759924344207147136964725042069590715109055775419542015459828625052625925106327684682287368362689577485611964457102117294970541980205367373760624130585186818123542107276433159268321437252310594519065030330277861943404583047658399731258240494161136315773859225598362056152280846430308277895525060137610331500768314060912636549086956626048727578439242616081621743301817316873004784407547620267652272235469814979008299677598257270261380642113256265809701523033979527486123230157851959075520482079187731679706021813899644608564202186500347804127108825845433999850607688488515680858473500375769612444211429650643716997619295529654966379416847591629504055169301647752706504672461836310059406961587540446499020937289377385335827720561840789413813531175545568519226050778045428932542058110909766202670907782405907689333625245121559149225027284627100803901302821670743586521988639632721895877119099794615730193532478094414640648946093154583483050973995291381089804133598999053401651324228186129999980665419025080235730995667967087425700625902055690593983268729213051075865997852518161113035191421704544774501622736901966674978305250175253719735947958514160296892633552886769863003371876663171326384379121920596720872792055031044673077906141463322557446270358409248727078342584489209988362673452399022501428572587016389349881612681878736610180141825279998284369305923218071148672446451461392048452952497907250327570721212098780448983545222512265972130313909180982987583256572096385838466364938394399136492065975602452446905990785604672532160335009060248829378928073103537410037940716915371372284799167144437570576864922988994498195526548134362473353142919941929902510707416998176980628481100871834249328381290757706383186306294353243879880271793853976240173240626266871122832749561177774679072636737877553146578089168244042814314903254275744032135024536230603243238622728530240965153874221687432187090472010599892635878029166214586357163028713200975362098924415828067255248973222565259747869964457126545033045062478306226071041992295571072201465413241752357796786796545046390154853122226506503213591548224729444303527048949121640051584801619979249550785493377923491911687014328396089014049020238803523411525711242330670213670710788069043953294512006902219248987245668155252471207225304879419117646136382265883918093243232715241022669795311391555173266264472813079522067825078432093897966786158897461720574776149194596986412047612931817397241039096151551580771707122829855714363107263263769207567656177427085211727391839965994695753975704893459684503568905189372386759093260248683088813952815638480889165381606844959179569760918419373780739642468615773153214566311625805576003529472378616258579510318340299247011724241070260173699716685481217990703423940432516548300602084634292033235730178599658447715117961918128755468455366028432826387159156202364015327436828964586721206255546954444889857213547463026892788555286688826247059820801626263307852434965408318060867525541429697467555598765686713383837245917097163413291202647218128121717906428623969522638790995176958706759844947346118114974685585042045475592032104896322633097939651776315866617643387882385869205486987222160165997009337492747922118643670059158984423539328793808276446970533226041739011403759720564096962437264233110695787355661479022682413710975468309723645790608219273437284680926599975143930764819663715615692560218307438471680345469933254076213729139488783857297633247637341237762984105503534049614682481564402163496422093603258110191110250674990967331594285501898991113222666865928364052768417211499114367581350769026784606942325440922487327659961082101721739778536447039295674609139260343144403301625490841567132733084888118046132130533471620063061924733983468760625230705263823701863436345006363419652218686481100456134661207470179606522358562730161734669663969799026572465838829741169910068813205049309819594139732418597922716355248819728328201989371082019372817660140758706049864824452112070702537874934134791449754025492740917573593471060392201353226961021567889249516894328587987644010418031046197996158979818962567968127218865462171663429024883296501500787051080330380927705746709258809151416449303408654568706713047633088300423630056642432964040375733170426341186574625023086538414236894534076412875694924729257452018578737237130283424485124923589249975967185949759455621348041364752798452194528803625858136368543546123360725878558579061099423435552095064006588196709240600849425038888121485985714251878960070326454427805433847642409050865464228648725070249705095872037043571719116100061877339269887135933673667859082287082442430954197699113096553135414003096449179113497399796850237666349826852015587803910207973507367369020353642578279876846626239251249993719267564412123666697141217968795560104143588048089081763990194998422785119971962923744252241099984929207692226790962915043918461001195434448045959023373551761152274959465869566114711721517139078245005951876971841756387730786218576851887255287387917099483191281300918624948118826306978851934638151644342687862802004288181801537557079329196656786486441748160862616034218688058713588471951913343894875926031672928848472543240150468672372117779614553121490048147219639094177073842457794189854061010200557476293488154978227856047675455628330948797216848484436670145639895737986566351543388644533833751647071563116225966020606631816742978849543067059869033967044941285519607546475414818941810334715274155964597831988906818340531287181211494057010015718416112249849724823531501076728369551688005636324195725032282819248690440508095436449288930523733708206388993132065454248190524213692229430860168805694937287323879464690305303711764561214360521165977010252146185279489206132965253386546945277891056285465401863556053060205014851139636984355648458653189318591013491891658950640205512644425899711169929301216722345334723011813834514206356350878279630088233076726062564619827826417590940099062045382002122389718900596796226238129814576671556495940814012804805548084773524480201407021459104801215699649229020306738643738332171936059379105328436864118816707750618717219945200192118844832934569273654091256566205574709842879173294284843601509334585582454531138858890223907013388617916184607613797535609828543846592357106188899734533395052765438452939195205298899268007875228989968070118707812626445771883569052226731992163164564544758381995163843389042232643568348822164918614379276586450183512210194370356157070713912837909724552115228106276742172217339796757381575830324675574147146167538280406991108758071229776539748274935582115563740211943742851360679246975027441095991239177230686649746883833355869919662091814123375578077046277935881816131505053913915261659068437286938804755406865617931677353693932004600822036676128722375097772101721660753560001861738121701024981991015815375546966794123296382119959959498650757275270952378914609833197712576803454868404649215447993306636011806302633185983908073777127065360684565178813461701916835105020570254470862219410242390356857120987465344209833272265511162826120243237180045500064657024489830288838424967628094970277048524696937777091866834978933904784157584118910481511377924727512134361917046371633138723656965467157301668771496685112793759023736116313650167181730218905192670216203407067492812489589294773375730604412221566363676297959168433396586933439676155008698957263457863870764337837814160931632327785139804064028252251194968347212639663310793352381503142154998878844015545951278942521763477136471352135756338176927124903500279523274738563889096115182110529072283343061034931876354944031786302381387781503102509068071554408077846800762798136663741600593007861610196710731949315813711250584864325032147355652348572550647280687600578696840101162428253742705304948378075475939789202880784076546884260096696780110346189758151969909735102521050451354995439636539955757066720816744534564945255937551653408852041743076230975991118003033405193316677893289425577144943228501930231649701860364950937399096381401905832934220881501578866870862104272384147204612801069223565729437398050833346636015232203702820767076329854890536765857882557113324453223906451079347934323227248010741617006052446486707613922170456232822423799348039716991911262749688740757188466944167802002970003291615850941585652168446400092051089110305171191331229081479969188237529808578243590929605050091148141164057703520348498526780227966041996838883144307526222307641364893767796485828212956949401737577223753076592091120194826525464498392064242535715873823006643626455079882413405336751630065487847944863766481902388865962024611571009959555677966390245823223367065440145922830291340375422506387339663464736659130343294464324482695854365860004867427047116620709352120780676509175430295805788421857322839708030045402042818904949374180784459729369832436304231492340134070165795410466801417395677736384366744492196620004143263030545361364953313318736302953875283376285497145380934283510605065403710362143612033170643448152127584386093394064273116557499244991868191843603859848086141174917761773467642614964713805438306644926551453847698465411559675433153543598308488251494647190040643318419763945530433136969744856330388497980065029662109147556968914026504272506299615042262185647037366442157098193196371675655169206529567978360854804610716919982566625767200363908558581103517769873851417867464282886954359485879013889544848631312924097024402160255155878121690390854243086100939673659308822252321367044940347087443305601751479177833134040125964006266287601044190078000958479663405032769159350019266290816735002443722477915494964468862740420286579584750506763938749694314789212345135729166854284450199227575007536060900420533072031274023309801518122302075336241186633754181608043706206836300030304353835473298873506616469554752703566476608268815275201711406565275877192488811829402653141223299025971245057070033866697190781791452487268475672323989924823897589138411165302055457519285324494552324215126116989555773252521272559957334304041259705596737204179342983503046654148990514430754204198053248680656394121888953544250158445606585360364862462814538281433200811443175819943535970093089742660860349361248456619270610612509233427406064737218459412761353215340931659772044081843401973837778406472401160553880643499729960692644287881209134299576418958454077245720430776492830389067981668904325958912838848283324755755093782122014435179519300781761159331065950981022275013416952005410462557894479522526889213879954950238447021357136627494438210386319653399932538792787623154016983634036020869802225862941655111958490769696820751189202210834250486921449395477127413279324455256313537452604540557959008456718999590443681558665049097728698922586805543821142284655648889958921848884091637154000752636806484347016173152136633972981973165504421768338727937988573352426468796880902084721016365461039241542116423898293553541492599545972399945668300130334209472440281416680018973351557520350730518245777176390197528307676061749336296430557634026579982251544769064862841861573407094719672153146456333343186691338195203484507835632564030510700999726345514480736823480698522279158580821217816671963689736331562189200618227404620929638299878546194845037130672879377659224991547350982501462640949473672497141913074747543519960144987857672307823791305279342623677722445067994162002585024283622548956809516797993695704610870620278407765055264490665443414245193944208228431460132605681057600342783644227269588741507965559132996467183143983286109974545255583548936245657444081454475810341724072157334915289158801864865232437234971310559679230769459240535594066868037534147397799978094935867816576314249141567570111266177941865607347594157096733173888821775829191609501837566301155402348489348586535117513470050574365177243885410387998425291899042838056077399362190375041890152117166707261222297125687723003282882409866758916377167064877898127548532828929449232801289129243042812045786249377383456876467690920805515789619915358826590621391030363769336915168973714928622568669081378390409990318211364608821942615261635542940836610167454888432111357590820833184944695774879557717285049231493286657924636909512037412238175083367879830724151600517843878700806808955881285792501851406614041792421636656804898827999291447513489623865532873078202491952235835407228719768512421934539781991336000458237326909455509057916657307558718961453717203571801033117689181066676042439900992422882190039700411853202791441464924562449567008287569659268844591289155753455180207513877986289776470994409934417784884010506977881148202460466702157057602867874589112325876988604761594962559775438048493130308029865952198079005876788540240077877103258070345233352943632634803380421525473739767142817901351285193180042797067389931666342135125523710581488293273345197531199043120406763931575313873624196110212044896645275100932050966928981592531149568457448565371803578485163917663482922279850550860690635355071885509495215371945507710410126931087569045411545451766132590325058839512304916137519097876065051894000382265571067628819898182254161799148586525297905635081339544041770897814755960402003890631647862429475570942575002021635407819684347687281772414813443359297644741230321895690234461596107170831526861263966666484836032779379224118913914122337571766765857524477968290341943875446104682371173874405645598643698138444482351732808832361741792206343111729080024085646360438014558636788095561876833879479541197057163517798024173352521141339098955635323390783508801823395369999451161031400442708953398431408938416860330544282939096576335931494237928560222492643287923183427062910310739531894007981749150363850607414275763764770976085965917501522817651994862717435111426895263197827991057818849478726836589199005115961181195993755121756911343039807346418142613184422739224859183876469700064714199087087135997879413172449220883565546233940421330506885631031055724922534429852856642208367820914000099264280494375723015117803496267107569253696418287757425194411247036736181075897352292112928316290543827438282791343625832020864921357428060965390399267010952331828355790661228793821536655147952135135554259838382541166467166120391514469371480913648643885605974022742568988237661718665990894676010257119500219839480524039094684178872646700931265352675991331754916754186016538675040431712026013004699878512006856916493041500562476179171753161324580190459161855963430941942854859412998565334299660959467956727977158854838716624177508590600666246210525203834239226568418579577423182050809706889609286013581032181800368200991375448713255161831163730561976193824297317312450354660186339843563866457309549721406045816419779333459374467081225774646262881685105916713563080462183970974314136264597143577855898367992789366715915095405824377728156136841728201202310517927312848694594460693278060508724662347333146500289729025473315314838787536423928512689975588166515992183736786829630889803656361009833164442902391281516602621581050707961177239097610790220406496090488875763270854745412544893325791165325448753930414604760971292922066668599764981726653664893563964505984262667229509152426024151071957183117632214134028028005214063746691752941540449052794286772751757383426500866527912616412349006847436212611835502711409169616793604374202293940270813048383629371718599030185529584917228778479550646778172300258013724498923508504116852166885218766709681422404864053316752139104335264594359987849581690097204549675629667753530709748294088541409101191564929941531822699737845598596791205139742871551421839343333451039015497895793642892726564140597350888148985270345031157238060035071163409808929095801945837024617191991916362600659412611218361301167316409612479824316935407256705750127782439797383539988375258102175306565714586927104906232268336277345221050136288791807017690450485590868657504119639099103122538089576305428421338847267161579377087185802268279839377048786256525759012396414616670897035498921253119385887729610585541065185884972490579775069450417370721505917845044297396610824993113405630726773093137861490740631007390280642646491734361115473167795838597679113726203270398258537656196891096956763138813494729269481972500207532852226162211794002037062775732585322821313236981056518929909641359707336186588155467733773192639157091144154125152275549819953232402428111413060062890082113943831901948377192018065274324068759101537593098906531180742229164003932589251149325392926026798784804369859765629216036874850509599643571759554929900123434861769053685716466939954687367821943481316199440136616601312916534361075793310940577967977611647831657577848350427377304628809301634398451780763075770418267762295823150309961401428734422152022504890759010378570440015855867749071557128246329182365211753708288466926215990360395139784847156787816020104974985752915248436429501295255625842955476903722249516714603343907402960195865664535908209552903460218951301749991444348841461283164695303624533733891661098139937784513263640741715661857209261220807360444784191058290587800666530799586869332094136148593831573040696364165675463748458159189966101775833012040868280679889763435694651529022035980500528306606396975109576885950832277740871164026444023683781744254107155861468098151299058055316859504364576721273439517750004641036764490363431652139504528575512285790679814483105718548921711021124476707367125158802692884394791260487921011196804575374216464101323545355210766966077240554737375275331181352073503892742550311621683414175225021118131266351018516293492689303559949171794211795320793999333795112555749329892557817314398571101596423018835545533975141027591064264000731629506803891441493499436780972099489506428211838813331172807643842966540366214143309342227809321040498306909937135203286341137383868981850575178369523507113432977570916327865562301348128399954148753386666280679774243544027448345995920155330745004025108891030319790677896896803475550386329215937987583413765345818848789458683062578127615402072055138301398760798133167921766536248632887863334409472052492438681732640358984530978250250337904040938525179023804823373713713772438199184295305874001449927842994151984956081568712325510232395191424424022549987066576098452437764820208200339401938225133842002254516419082874741188978322238216768587307497991593090301378484950591910451159723784222792429068305017162468046983966659787875730198968051714375422404197570920507302728609212497355339425267539665939538974928788990103234713347058318963072234201747659584858386904387189362909349033034056716667354886129020706762894855370309020121465650810719599319306374434799183614982688936025032810662026801783371829619330589471574636772192885714786857368030315198935926201635681689965783795569936347862083213878115592357185049621680965479174893925005842066383558887609535094364444137527940355993959374640755406659481560578349740621508046618209882138858601035547064316663393608475152924451336999140594053464556850438446478205862404894012378553858327664927622003510286392084108445509689818333725077859456425378863838699794105272587090645456094593905442616178871280079909651300751681762060820373173715722035073050311027766814505402288422146244459735951178067165830246585532910283667764975453653118379859124514681750171866404862984802238436805990150002587900733449092338716814664461570693578701358045010275408479972280661764092810471657337951376502622510415185255948652534379309646341907424926058463197685023299070892390210469355832008474302007656343732079067097996324161829671005834849478401911612443089355492507168226863009486733636558850561242462253709551922639268784382933538811699008311220169651849753688429943702695290854722277444268549413063162728115133641573070911911782993397295332910475705512402110447323438417187197737136722756745852396564876739482160481394327152677816386227659438521962238027898025740778498786940929861472375857184599019639472477325389062637607547821083968838627596491663128890620487519342719408198705949842723372319833251206242933273209964960675997491591948555490067067348606925620418619910377419920186981515191887059615892203354008993072887145913354141938271480145005180879372417970315546321541483793041229464188052283478111188747252670334574291968229559159776620722159679835580365902318411295086621215249841248714929435099758157784912410615552742259061635727755249956991096223181379026247534051447135737024416386393101335410056963346429098806714567233308813090504497097523116831801108742297531646162912567185485572165103273920263321444253651204336413057149402074174856024702367887184750909483175977873345761016374300249413902698072789058385338275345341067979233196511962398558651647180272883332117218010768802021043715341695965989129416123646380530914592664896376303765275539759716792072312287934531018863301310849719525759367267292434533641233906679363871919137669789467575934113856375737743160283596212959333394866238444725207557386543776635699073170255744935961292318399236975960356636307647876874425244501464175115553674460289792531291514796043448186831454481615865151164852958996755136070832110517491976352717944841667882260702411089184709437670086185930401122725133456955442896821995662358875638010207157962784813292205759941759656222237744661138459979006589544461690097691728242179566774838447900182230036857823947969270136908858257319898881258308082446414440738076296281919588470217283309846329466269766642857080867811611134378112390120153286644233282386903904276339427228301883666242774277752585502599629246833432424085728697064674894075523471504525207065829728983495647320208053471237455398678481422194063153342300414020788633969896977764474053639308060285657592124613859582053688062329678616374997567526252105153246187519757227186596078619513381165358983401320685181522435018200350486585149888380517468801566788809668258674374896086841865036886053076504713178713122666271872254999945140985916108397803350282525896941471967809799744488484976894985858740768559053750535637806602672838986188239865088233489317783631407301317078435468375735790540120692655800759160654151191575767738444970164618400862272370868690980512248273853922209296119892431490866832486792064327788306199487130160425062334508621633124083052147570343719828018762312197036276394157702803172635369638567666643194377947523746545254397178413711108598766834846320442809183109846707669331120546190913631804199071013108833525777155821929592649168607980572401999447691656873018239865999555589222154246054145397555816510996619367050773567183390560430314974162064969903974941942020884015383463609055890678404835069939757620519379009321383347412151710957907180335502659722535790618753430334686648042297463699885695891220097658210153652387457928821750219041338625776680437574082651740432496117797099029909567384800599825286640015660497457346077546457631221870125396189693782762183790932889090948449725045110707446342336564813878467717394044579168176642460533897973647496171363272228303770853359009702646163704855449493040896205332826761927558244377578112013983844960882996129070225541635660117587525951422135219282546369139990908928998427110839673990996709475389376370696416615133638934370457999148637850365820795081180932895330085811229445655662047945642322739487297564101669704669374578951168400710798922557463626082805112351376493034943637805512535283356247962285738419207818337458738886185339004954952774221684110059245259957129861532481425057119704149226450811212617737396318270318035486629210282417909090797453114682964205716395326903434704394644024072895798842288814616310595490559654241608928726321397558815045978249539247710220064361951967172685783202489405334625160046303289095372516262008325569567636769198289275540483106456381544912544900297034790527893516823527714895360064341969938349556567282266612878865820067947907905016552257156259378562393585660312404292549705646450249108979391314423723069240981231802732986301980166925076627033905481314205273084372564458242450400365980895738512000562712191774656225746491884214195277780179391606282817704703971505199283871784598761660928913585995220728437725553749627778265420177911145761851276825349090572238304310755260001596501633998050348681626910407450940865914919009775113069181431620300851370899091050662688302974401732231303625624050998460996835282293542499353795944717484210973667293249154510688714625703837833251085327061010734690187412533256567505051088681631032940214025400220401929590177649674891364576239032980991045119387594268253910489625557596450642712125131132659773487224235125502434179768706921724439574217325015106243871419949970477943793818740965986671578313715576463722628417041801438008463881917132175737477326208269896854827986155604451390841993043419729738139893368108536099623927401121472512317603299417266402791763991806033114846760298240274072197195713968046379504845197155185810631460486069556463509978221365742467948442296240778745428999651223650860178705277342092174930956399352047297904347082874622565237641847850947747018973970457538976013107422531354691627859435376293671420904117466731177378090584665768829707835522820016128291853207976902191922957638821128624576171797016267082732188232941507631287476697279784397291466751803945493483242196432284836152842964875731807917757474836675603874715420942939414665423007241073074348446719392981330390572291265165148167845682963478577396009518141982137315661750937806657478345527876530326637333212429014094447245128994010185817834905236656474841899769555373920228584893148856253942949260168774147861847595804348967712581573368778264298084356634974228596217570771122603438463712648413841575142006448402650048151948245113271420708270456730523105470373022892620718808012779411498828049677847661225624810513198978787582585437092316571865295688393270480581833433809662318587898259937262025214006572671066732153913809037089528006363788069885644598114612220220803546295202649297095627578750368909400276419318110813141573139281863258930437868586445758566800131953618360662872608199890115599081688065345837828435904255876445673604373882498293017892981675255539120252488641300966948166368872654133777557792187918710661168553464562746001766516006936837270595091757260691898243643032683027235524131627285811431883610073897637453399095715152016961322950033387576967543483474106566686585566267727296770901583810254543395105906569155626980328087757646583808379337990825536781426453355491948168844011581557664646904374154294286125344207431948618117047194233108429752062227100609306476963230362268948834549289551406678925942973651707809976276504326005887444530598484120131598581284622127775485515313524520949475206312360502362909718243489764519618711815901203996781195792208576369339970117158510060757514689835975738313931333525482213232885800978724021630286009378059454896971853760874199278360117890787223268735539244333250276696519676211597404909568550826085820790100553886828398557177685630668749485938954364037199829198959696556529031582099253407789261311750002305161209983784683802954002978171976840224277456358368977325267418268576880318063977616674248005394203768966266872972034708325083957501041416711456461731623241601840202407894200891752598165690532924663771836721189605654677212819969601210412338178021457968717201795329361356305233462324671310786788568824508792924972205435157632633620355606985779061246962592099137147898056101377178865088729983494671908568325998572423847507503466152358836940039081187169730459771000462425185839335400390512328985909580928544568832935476899972567743763734744478399119380158686604593738502476402136285252411316102736825864762758036385234640338274330206016809946997693818362801382388686946326937882564171843388938441468404128912322218589100923475211453953442992112726049028637605561316141508181083869727995031111789636940167987222700509054326459636456675505524471146636555429305034294436089650152979963889854381474099240069584052288639630057210875028655975684791449996620095598221840186045076728959446763280273606564669130296075011213002123518450405976992852124748335821704962445792179374123195470436820452133215201048776157603793388034014662581116696159735977336175540742727569993765893393954575145261458408126074723808876511281819631865523960494950459737073227291091406224569716322417125935502086296729446859183287675716063986159859101828219922609374841899990769466870444497366975070651597344813193965232216286975541341643182071935542869939242907453799854464071726555123374927134301912514148451367396450327345126417544786214763188646845879494345357696921446035381891171866543070176578755787449936155559457815468459581378274320618824091168366042379443266819205441009671281815069540316572248737483443664003137821294360814586881941452340139066155048675308311572658324402187503127983720227379890612386283827977840292805497256753188033696806443748702109142030596800276389037113484381634045031667010303464845362163245624390394532523033618951628257634017661258998708804157107355006491175725363557862355573362567026154252137966674357366746622895290682112292813296372176755333174599781232457309995661489865827016611609454563211089613354737503751376107815996529518304062277964699785906005958153392636875844676839015217145843971286214423962402949822546957749201615908515877944653806889628787945155073028186718255355034897842586077713270449598540163082843037757237796809206807967191941422255652257118877223167582954689676970543239953396348842872751905967640585666458503990989703281894088215578947037393970419325062332911776585342080156011776656942318397630087586384889853963199560947407805151856059929225948653760776868699112393245819580966350481976954864950343035294539781499003571548821633934817311504976767017867603543729422321588717731180440615608030468278822427246974195752782987085286934728817838459342041002134096673097425883604325692166582243888009022729968246423088462144178637219521299793495721966202301108457153623432772235095734390926124170582461037790488748130546479613612363764001568569016797281694500341996127091512245928820442247465758549390669004955694160527232934553262659352239591224811679250179393965131762285788608917840412302352013795578968793189830812565071294343442681040414323018114614877539159720317309391222206880324169207614957600883792832788002290969454938895659675662082014367948886769498981163998497407729809536056204924844979205004726125769529862213382057292567730634293263638762367788508812912141430829730545275226352453993499469392234324095544754353435569505157625042532991398705204739017753567989044577913740084198725670169209992614526687777227645763441568442854496824852860074354143319241556995686406984325090673172447760168694945453744944016710495212240946431956501133258649241938325736439016929196750797666245393316248682509499691892813466839117780245224760511382709024460283858255121841450380619808471325771144673093258681874270812660569364045085303728684797952271943774194848842024127297476147100215788503493026696817404678873300858113781159351407449492400904409486930108489662991244454686547463067788100450840819487534074076791483766923275275067061199754365166961471463906135865208552604365987312265644867183948023024810814962900878173053788479206362481141595770069317744250682340782987311807574896630124838003309621111261630822781282153001630634216845321602857164338793646457957328126688357843272528011597204929990914356697955645726975294370202965112194215648119634505508737525417997657488374870088477076227111237826796274427808084811023606907309680380975420595867284306504284075624964292546152706939357078704581131578517493128065078116968654499759110081956264855472668153543035148129671780637227896828883674570146631828063729227565220680273886310420447806993094465120661723638298979276426691433045140017985773632549338390380340396692086076003078307209964551532743296155764470390648416875838344876991429878332212389335501256146672808645328493800240125334916044183275525225993477167464519040624798633909800878074028151672590056335993290686297728531922667548195715707495768601196044973818096086236559328398842971889596087524600836404390366444484615998547444678565837064271690996411173862332256");

        for (int prec = 25000; prec <= 50000; prec += 1000)
        {
            Apfloat tmpX = x.precision(prec),
                    expectedInvX = invX.precision(prec),
                    expectedInvSqrtX = invSqrtX.precision(prec),
                    actualInvX = ApfloatMath.inverseRoot(tmpX, 1),
                    actualInvSqrtX = ApfloatMath.inverseRoot(tmpX, 2);

            assertEquals("inv prec " + prec + " precision", prec, actualInvX.precision());
            assertEquals("inv prec " + prec + " value", expectedInvX, actualInvX, new Apfloat("5e-" + prec));
            assertEquals("invsqrt prec " + prec + " precision", prec, actualInvSqrtX.precision());
            assertEquals("invsqrt prec " + prec + " value", expectedInvSqrtX, actualInvSqrtX, new Apfloat("5e-" + prec));
        }

        assertEquals("inverseRoot(9.78675172784960122038e10773, 7912)", Apfloat.ONE.divide(new Apfloat(23, 21)), ApfloatMath.inverseRoot(new Apfloat("9.78675172784960122038e10773"), 7912), new Apfloat(1e-20));

        assertEquals("1", new Apfloat(1), ApfloatMath.inverseRoot(Apfloat.ONE, 5));
        assertEquals("sqrt(2)", new Apfloat(1.4142135623730950488016887242097), ApfloatMath.inverseRoot(new Apfloat(2.0), -2), new Apfloat(1e-15));
        //assertEquals("inv 0x7FFFFFFFFFFFFFFFth root of 2", new Apfloat("0.999999999999999999924848832098"), ApfloatMath.inverseRoot(new Apfloat(2, 30), 0x7FFFFFFFFFFFFFFFL), new Apfloat(2e-29));
        //assertEquals("inv 0x8000000000000000th root of 2", new Apfloat("1.0000000000000000000751511679"), ApfloatMath.inverseRoot(new Apfloat(2, 30), 0x8000000000000000L), new Apfloat(2e-29));

        try
        {
            ApfloatMath.inverseRoot(x, 0);
            fail("inverse zeroth root accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: inverse zeroth root
        }

        try
        {
            ApfloatMath.inverseRoot(new Apfloat(-2), 2);
            fail("inverse sqrt of -2 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: result would be imaginary
        }

        try
        {
            ApfloatMath.inverseRoot(new Apfloat(0), 2);
            fail("inverse sqrt of 0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: result would be infinite
        }

        try
        {
            ApfloatMath.inverseRoot(new Apfloat(3), 2, 0);
            fail("precision 0 accepted");
        }
        catch (IllegalArgumentException iae)
        {
            // OK: invalid input
        }

        try
        {
            ApfloatMath.inverseRoot(new Apfloat(3), 2);
            fail("infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK: can't have infinite memory
        }
    }

    public static void testRoot()
    {
        assertEquals("0", new Apfloat(0), ApfloatMath.root(Apfloat.ZERO, 3));
        assertEquals("1", new Apfloat(1), ApfloatMath.root(Apfloat.ONE, 5));
        assertEquals("1st root", new Apfloat(2), ApfloatMath.root(new Apfloat(2), 1));
        assertEquals("root(2, 2)", new Apfloat(1.4142135623730950488016887242097), ApfloatMath.root(new Apfloat(2.0), 2), new Apfloat(1e-15));
        assertEquals("root(2, -2)", new Apfloat(0.70710678118654752440084436210485), ApfloatMath.root(new Apfloat(2.0), -2), new Apfloat(5e-16));
        //assertEquals("0x7FFFFFFFFFFFFFFFth root of 2", new Apfloat("1.0000000000000000000751511679015"), ApfloatMath.root(new Apfloat(2, 30), 0x7FFFFFFFFFFFFFFFL), new Apfloat(2e-29));
        //assertEquals("0x8000000000000000th root of 2", new Apfloat("0.99999999999999999992484883209847"), ApfloatMath.root(new Apfloat(2, 30), 0x8000000000000000L), new Apfloat(2e-29));
        assertEquals("sqrt(2)", new Apfloat(1.4142135623730950488016887242097), ApfloatMath.sqrt(new Apfloat(2.0)), new Apfloat(1e-15));
        assertEquals("cbrt(2)", new Apfloat(1.2599210498948731647672106072782), ApfloatMath.cbrt(new Apfloat(2.0)), new Apfloat(1e-15));

        assertEquals("root(2^1048576,1048576)", new Apfloat(2), ApfloatMath.root(ApfloatMath.pow(new Apfloat(2, 30), 1048576), 1048576), new Apfloat(1e-29));
        assertEquals("root(2^100000000000,100000000000)", new Apfloat(2), ApfloatMath.root(ApfloatMath.pow(new Apfloat(2, 30), 100000000000L), 100000000000L), new Apfloat(1e-29));
        //assertEquals("root(2^100000000000000000,100000000000000000)", new Apfloat(2), ApfloatMath.root(ApfloatMath.pow(new Apfloat(2, 30), 100000000000000000L), 100000000000000000L), new Apfloat(1e-29));

        try
        {
            ApfloatMath.root(new Apfloat(2), 0);
            fail("zeroth root accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: zeroth root
        }

        try
        {
            ApfloatMath.root(new Apfloat(-2), 2);
            fail("sqrt of -2 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: result would be imaginary
        }

        try
        {
            ApfloatMath.root(new Apfloat(0), 0);
            fail("0th root of 0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK: result would be undefined
        }

        try
        {
            ApfloatMath.root(new Apfloat(3), 2);
            fail("infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK: can't have infinite memory
        }
    }

    public static void testAbs()
    {
        Apfloat x = new Apfloat(2);
        assertEquals("2", new Apfloat(2), ApfloatMath.abs(x));

        x = new Apfloat(-2);
        assertEquals("-2", new Apfloat(2), ApfloatMath.abs(x));

        x = new Apfloat(0);
        assertEquals("0", new Apfloat(0), ApfloatMath.abs(x));
    }

    public static void testCopySign()
    {
        assertEquals("2, 1", new Apfloat(2), ApfloatMath.copySign(new Apfloat(2), new Apfloat(1)));
        assertEquals("2, -1", new Apfloat(-2), ApfloatMath.copySign(new Apfloat(2), new Apfloat(-1)));
        assertEquals("-2, 1", new Apfloat(2), ApfloatMath.copySign(new Apfloat(-2), new Apfloat(1)));
        assertEquals("-2, -1", new Apfloat(-2), ApfloatMath.copySign(new Apfloat(-2), new Apfloat(-1)));

        assertEquals("0, 0", new Apfloat(0), ApfloatMath.copySign(new Apfloat(0), new Apfloat(0)));
        assertEquals("0, 1", new Apfloat(0), ApfloatMath.copySign(new Apfloat(0), new Apfloat(1)));
        assertEquals("0, -1", new Apfloat(0), ApfloatMath.copySign(new Apfloat(0), new Apfloat(-1)));
        assertEquals("1, 0", new Apfloat(0), ApfloatMath.copySign(new Apfloat(1), new Apfloat(0)));
        assertEquals("-1, 0", new Apfloat(0), ApfloatMath.copySign(new Apfloat(-1), new Apfloat(0)));
    }

    public static void testFloor()
    {
        Apfloat x = new Apfloat("1.1");
        assertEquals("1.1", new Apfloat(1), ApfloatMath.floor(x));
    }

    public static void testCeil()
    {
        Apfloat x = new Apfloat("1.1");
        assertEquals("1.1", new Apfloat(2), ApfloatMath.ceil(x));
    }

    public static void testTruncate()
    {
        Apfloat x = new Apfloat("-1.1");
        assertEquals("-1.1", new Apfloat(-1), ApfloatMath.truncate(x));
    }

    public static void testFrac()
    {
        Apfloat x = new Apfloat("-1.1");
        assertEquals("-1.1", new Apfloat("-0.1"), ApfloatMath.frac(x));
    }

    public static void testRound()
    {
        Apfloat[] inputs = { new Apfloat("5.5"),
                             new Apfloat("2.5"),
                             new Apfloat("1.6"),
                             new Apfloat("1.1"),
                             new Apfloat("1.0"),
                             new Apfloat("-1.0"),
                             new Apfloat("-1.1"),
                             new Apfloat("-1.6"),
                             new Apfloat("-2.5"),
                             new Apfloat("-5.5") };
        RoundingMode[] roundingModes = { UP,            DOWN,          CEILING,       FLOOR,         HALF_UP,       HALF_DOWN,     HALF_EVEN,     UNNECESSARY   };
        Apfloat[][] expected =       { { new Apint(6),  new Apint(5),  new Apint(6),  new Apint(5),  new Apint(6),  new Apint(5),  new Apint(6),  null          },
                                       { new Apint(3),  new Apint(2),  new Apint(3),  new Apint(2),  new Apint(3),  new Apint(2),  new Apint(2),  null          },
                                       { new Apint(2),  new Apint(1),  new Apint(2),  new Apint(1),  new Apint(2),  new Apint(2),  new Apint(2),  null          },
                                       { new Apint(2),  new Apint(1),  new Apint(2),  new Apint(1),  new Apint(1),  new Apint(1),  new Apint(1),  null          },
                                       { new Apint(1),  new Apint(1),  new Apint(1),  new Apint(1),  new Apint(1),  new Apint(1),  new Apint(1),  new Apint(1)  },
                                       { new Apint(-1), new Apint(-1), new Apint(-1), new Apint(-1), new Apint(-1), new Apint(-1), new Apint(-1), new Apint(-1) },
                                       { new Apint(-2), new Apint(-1), new Apint(-1), new Apint(-2), new Apint(-1), new Apint(-1), new Apint(-1), null          },
                                       { new Apint(-2), new Apint(-1), new Apint(-1), new Apint(-2), new Apint(-2), new Apint(-2), new Apint(-2), null          },
                                       { new Apint(-3), new Apint(-2), new Apint(-2), new Apint(-3), new Apint(-3), new Apint(-2), new Apint(-2), null          },
                                       { new Apint(-6), new Apint(-5), new Apint(-5), new Apint(-6), new Apint(-6), new Apint(-5), new Apint(-6), null          } };

        for (int i = 0; i < inputs.length; i++)
        {
            Apfloat input = inputs[i];
            for (int j = 0; j < roundingModes.length; j++)
            {
                RoundingMode roundingMode = roundingModes[j];
                Object result;
                try
                {
                    result = ApfloatMath.round(input, 1, roundingMode);
                }
                catch (ArithmeticException ae)
                {
                    result = null;
                }
                assertEquals(expected[i][j], result);
            }
        }

        assertEquals(new Apfloat("555.5600"), ApfloatMath.round(new Apfloat("555.5555"), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("555.5560"), ApfloatMath.round(new Apfloat("555.5555"), 6, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("555.5555"), ApfloatMath.round(new Apfloat("555.5555"), 7, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("555.5555"), ApfloatMath.round(new Apfloat("555.5555"), Apfloat.INFINITE, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("0.0005555600"), ApfloatMath.round(new Apfloat("0.0005555555"), 5, RoundingMode.HALF_EVEN));

        assertEquals(new Apfloat("1.1010", Apfloat.DEFAULT, 2), ApfloatMath.round(new Apfloat("1.10101", Apfloat.DEFAULT, 2), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1100", Apfloat.DEFAULT, 2), ApfloatMath.round(new Apfloat("1.10111", Apfloat.DEFAULT, 2), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1111", Apfloat.DEFAULT, 3), ApfloatMath.round(new Apfloat("1.11111", Apfloat.DEFAULT, 3), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1112", Apfloat.DEFAULT, 3), ApfloatMath.round(new Apfloat("1.11112", Apfloat.DEFAULT, 3), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1112", Apfloat.DEFAULT, 3), ApfloatMath.round(new Apfloat("1.11121", Apfloat.DEFAULT, 3), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1120", Apfloat.DEFAULT, 3), ApfloatMath.round(new Apfloat("1.11122", Apfloat.DEFAULT, 3), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1111", Apfloat.DEFAULT, 3), ApfloatMath.round(new Apfloat("1.1111111111111111111111111111", Apfloat.DEFAULT, 3), 5, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1112", Apfloat.DEFAULT, 3), ApfloatMath.round(new Apfloat("1.1111111111111111111111111112", Apfloat.DEFAULT, 3), 5, RoundingMode.HALF_EVEN));

        assertEquals(new Apfloat("1.1e9223372036853775808"), ApfloatMath.round(new Apfloat("1.1e9223372036853775808"), 100000000000L, RoundingMode.HALF_EVEN));
        assertEquals(new Apfloat("1.1e-9223372036853775808"), ApfloatMath.round(new Apfloat("1.1e-9223372036853775808"), 100000000000L, RoundingMode.HALF_EVEN));

        try
        {
            ApfloatMath.round(new Apfloat(1000), 0, RoundingMode.HALF_EVEN);
            fail("precision 0 accepted");
        }
        catch (IllegalArgumentException iae)
        {
            // OK; invalid precision
        }

        try
        {
            ApfloatMath.round(new Apfloat(1000), -1, RoundingMode.HALF_EVEN);
            fail("precision -1 accepted");
        }
        catch (IllegalArgumentException iae)
        {
            // OK; invalid precision
        }

        try
        {
            ApfloatMath.round(new Apfloat(1000), Long.MIN_VALUE, RoundingMode.HALF_EVEN);
            fail("precision Long.MIN_VALUE accepted");
        }
        catch (IllegalArgumentException iae)
        {
            // OK; invalid precision
        }
    }

    @SuppressWarnings("deprecation")
    public static void testNegate()
    {
        Apfloat x = new Apfloat(2);
        assertEquals("2", new Apfloat(-2), ApfloatMath.negate(x));

        x = new Apfloat(-2);
        assertEquals("-2", new Apfloat(2), ApfloatMath.negate(x));

        x = new Apfloat(0);
        assertEquals("0", new Apfloat(0), ApfloatMath.negate(x));
    }

    public static void testModf()
    {
        Apfloat[] f = ApfloatMath.modf(new Apfloat(2));
        assertEquals("2 i", new Apfloat(2), f[0]);
        assertEquals("2 f", new Apfloat(0), f[1]);

        f = ApfloatMath.modf(new Apfloat("1.5"));
        assertEquals("1.5 i", new Apfloat(1), f[0]);
        assertEquals("1.5 f", new Apfloat("0.5"), f[1]);
        assertEquals("1.5 f precision", 1, f[1].precision());

        f = ApfloatMath.modf(new Apfloat("-1.5", 3));
        assertEquals("-1.5 i", new Apfloat(-2), f[0]);
        assertEquals("-1.5 f", new Apfloat("0.5"), f[1]);
        assertEquals("-1.5 f precision", 2, f[1].precision());
    }

    public static void testFmod()
    {
        assertEquals("0, 1", new Apfloat(0), ApfloatMath.fmod(new Apfloat(0), Apfloat.ONE));
        assertEquals("1, 0", new Apfloat(0), ApfloatMath.fmod(new Apfloat(1), Apfloat.ZERO));

        assertEquals("2, 3", new Apfloat(2), ApfloatMath.fmod(new Apfloat(2), new Apfloat(3)));
        assertEquals("2, -3", new Apfloat(2), ApfloatMath.fmod(new Apfloat(2), new Apfloat(-3)));
        assertEquals("-2, 3", new Apfloat(-2), ApfloatMath.fmod(new Apfloat(-2), new Apfloat(3)));
        assertEquals("-2, -3", new Apfloat(-2), ApfloatMath.fmod(new Apfloat(-2), new Apfloat(-3)));

        assertEquals("200, 3 no precision", new Apfloat(0), ApfloatMath.fmod(new Apfloat(200, 1), new Apfloat(3)));
        assertEquals("20, 3 no precision", new Apfloat(0), ApfloatMath.fmod(new Apfloat(20, 1), new Apfloat(3)));

        assertEquals("20, 2", new Apfloat(0), ApfloatMath.fmod(new Apfloat(20, 2), new Apfloat(2)));

        Apfloat a = ApfloatMath.fmod(new Apfloat(20, 2), new Apfloat(3));
        assertEquals("20, 3 precision 1, precision", 1, a.precision());
        assertEquals("20, 3 precision 1, value", new Apfloat(2), a);

        a = ApfloatMath.fmod(new Apfloat("11257801564015978614078912465097"), new Apfloat("12590784789234078934789457778"));
        assertEquals("11257801564015978614078912465097, 12590784789234078934789457778, precision", 28, a.precision());
        assertEquals("11257801564015978614078912465097, 12590784789234078934789457778, value", new Apfloat("1639962440712046377137211565"), a);

        a = ApfloatMath.fmod(new Apfloat("11257801564015978614078912465097"), new Apfloat(13));
        assertEquals("11257801564015978614078912465097, 13, precision", 1, a.precision());
        assertEquals("11257801564015978614078912465097, 13, value", new Apfloat(4), a);

        a = ApfloatMath.fmod(new Apfloat("11257801564015978614078912465097"), new Apfloat(19));
        assertEquals("11257801564015978614078912465097, 19, precision", 2, a.precision());
        assertEquals("11257801564015978614078912465097, 19, value", new Apfloat(18), a);

        a = ApfloatMath.fmod(new Apfloat("123"), new Apfloat("13"));
        assertEquals("123, 13, precision", 1, a.precision());
        assertEquals("123, 13, value", new Apfloat(6), a);

        a = ApfloatMath.fmod(new Apfloat("123"), new Apfloat("14"));
        assertEquals("123, 14, precision", 2, a.precision());
        assertEquals("123, 14, value", new Apfloat(11), a);

        a = ApfloatMath.fmod(new Apfloat("123"), new Apfloat("61"));
        assertEquals("123, 61, precision", 1, a.precision());
        assertEquals("123, 61, value", new Apfloat(1), a);

        a = ApfloatMath.fmod(new Apfloat("123"), new Apfloat("99"));
        assertEquals("123, 99, precision", 2, a.precision());
        assertEquals("123, 99, value", new Apfloat(24), a);

        for (int i = 5; i <= 150; i++)
        {
            StringBuilder buffer = new StringBuilder();

            for (int j = 0; j < i; j++)
            {
                buffer.append((char) ('1' - (j & 1)));
            }

            String value = buffer.toString();

            Apfloat x = new Apfloat(value, Apfloat.DEFAULT, 9),
                    y = new Apfloat(value, Apfloat.DEFAULT, 9),
                    one = new Apfloat(1, Apfloat.DEFAULT, 9);

            assertEquals(x + ", " + y, new Apfloat(0), ApfloatMath.fmod(x, y));
            assertEquals("-" + x + ", " + y, new Apfloat(0), ApfloatMath.fmod(x.negate(), y));
            assertEquals(x + "+1, " + y, new Apfloat(1, Apfloat.DEFAULT, 9), ApfloatMath.fmod(x.add(one), y));
            assertEquals("-1-" + x + ", " + y, new Apfloat(-1, Apfloat.DEFAULT, 9), ApfloatMath.fmod(one.add(x).negate(), y));
        }

        assertEquals("very long, very long", new Apfloat("6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057150"), ApfloatMath.fmod(new Apfloat("610195307303654352666518121441737650909904045487886269771428999892641560353287146586044778246190438153093623532301738649952435264069156612677948549906937800453397951028502213195775639617773794408985442106591307693813584425141736150542412855661591984057451813085814343390198354303417947862296970200397097469208547"), new Apfloat("6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057151")));
        assertEquals("very long, -very long", new Apfloat("6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057150"), ApfloatMath.fmod(new Apfloat("610195307303654352666518121441737650909904045487886269771428999892641560353287146586044778246190438153093623532301738649952435264069156612677948549906937800453397951028502213195775639617773794408985442106591307693813584425141736150542412855661591984057451813085814343390198354303417947862296970200397097469208547"), new Apfloat("-6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057151")));
        assertEquals("-very long, very long", new Apfloat("-6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057150"), ApfloatMath.fmod(new Apfloat("-610195307303654352666518121441737650909904045487886269771428999892641560353287146586044778246190438153093623532301738649952435264069156612677948549906937800453397951028502213195775639617773794408985442106591307693813584425141736150542412855661591984057451813085814343390198354303417947862296970200397097469208547"), new Apfloat("6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057151")));
        assertEquals("-very long, -very long", new Apfloat("-6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057150"), ApfloatMath.fmod(new Apfloat("-610195307303654352666518121441737650909904045487886269771428999892641560353287146586044778246190438153093623532301738649952435264069156612677948549906937800453397951028502213195775639617773794408985442106591307693813584425141736150542412855661591984057451813085814343390198354303417947862296970200397097469208547"), new Apfloat("-6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057151")));
    }

    public static void testMultiplyAdd()
    {
        Apfloat a = ApfloatMath.multiplyAdd(new Apfloat("2"), new Apfloat("3"), new Apfloat("4"), new Apfloat("5"));
        assertEquals("2 * 3 + 4 * 5 precision", 1, a.precision());
        assertEquals("2 * 3 + 4 * 5 value", new Apfloat(26), a.precision(2));

        a = ApfloatMath.multiplyAdd(new Apfloat("2", 2), new Apfloat("3", 2), new Apfloat("4", 2), new Apfloat("5", 2));
        assertEquals("2 * 3 + 4 * 5, prec 2, precision", 2, a.precision());
        assertEquals("2 * 3 + 4 * 5, prec 2, value", new Apfloat(26), a);

        a = ApfloatMath.multiplyAdd(new Apfloat(20, 1), new Apfloat(30, 1), new Apfloat("4"), new Apfloat("5"));
        assertEquals("20 * 30 + 4 * 5, prec 1, precision", 1, a.precision());
        assertEquals("20 * 30 + 4 * 5, prec 1, value", new Apfloat(600), a);

        a = ApfloatMath.multiplyAdd(new Apfloat(200, 2), new Apfloat(300, 2), new Apfloat(4), new Apfloat(5));
        assertEquals("200 * 300 + 4 * 5, prec 2, precision", 2, a.precision());
        assertEquals("200 * 300 + 4 * 5, prec 2, value", new Apfloat(60000), a.precision(Apfloat.INFINITE));

        a = ApfloatMath.multiplyAdd(new Apfloat("4"), new Apfloat("5"), new Apfloat(20, 1), new Apfloat(30, 1));
        assertEquals("4 * 5 + 20 * 30, prec 1, precision", 1, a.precision());
        assertEquals("4 * 5 + 20 * 30, prec 1, value", new Apfloat(600), a);

        a = ApfloatMath.multiplyAdd(new Apfloat(4), new Apfloat(5), new Apfloat(200, 2), new Apfloat(300, 2));
        assertEquals("4 * 5 + 200 * 300, prec 2, precision", 2, a.precision());
        assertEquals("4 * 5 + 200 * 300, prec 2, value", new Apfloat(60000), a.precision(Apfloat.INFINITE));

        // xxxxx000 * xxxxx + xxx00000 * xxx0 =
        // xxxxxx0000000 + xxxx00000000 =
        // xxxxx00000000 + xxxx00000000 =
        // xxxx000000000 / xxxx00000000
        a = ApfloatMath.multiplyAdd(new Apfloat(12345678, 8), new Apfloat(12345, 5), new Apfloat(12345678, 5), new Apfloat(1234, 3));
        assertEquals("5-4-4 precision", 4, a.precision());
        assertEquals("5-4-4 value", new Apfloat(167641961562L), a, new Apfloat(100000000));
    }

    public static void testMultiplySubtract()
    {
        Apfloat a = ApfloatMath.multiplySubtract(new Apfloat(2), new Apfloat(3), new Apfloat(4), new Apfloat(5));
        assertEquals("2 * 3 - 4 * 5 precision", Apfloat.INFINITE, a.precision());
        assertEquals("2 * 3 - 4 * 5 value", new Apfloat(-14), a.precision(2));
    }

    public static void testAgm()
    {
        Apfloat a = ApfloatMath.agm(new Apfloat(1, 100), new Apfloat(2, 100));
        assertEquals("1, 2 precision", 100, a.precision());
        assertEquals("1, 2 value", new Apfloat("1.4567910310469068691864323832650819749738639432213055907941723832679264545802509002574737128184484443281894"), a, new Apfloat("5e-99"));

        a = ApfloatMath.agm(new Apfloat(1, 1), new Apfloat(2, 1));
        assertEquals("1, 2, prec 1, precision", 1, a.precision());
        assertEquals("1, 2, prec 1, value", new Apfloat(1), a, new Apfloat(1));

        a = ApfloatMath.agm(new Apfloat(0), new Apfloat(1));
        assertEquals("0, 1", new Apfloat(0), a);

        a = ApfloatMath.agm(Apfloat.ONE, Apfloat.ZERO);
        assertEquals("1, 0", new Apfloat(0), a);

        a = new Apfloat("1.456791031046906869186432383265081974973863943221305590794172383267926454580250900257473712818448444328189401816036799935576243074340124511691213249952279376897021197672689372826666678270743290207238456460096313336749441664951640082693223908626337673838241025488726264513659066040887588510046672813094743978935512911720175447186956416035641113070606125170400972745374521370401420144157682323238964502909132239229201863020459196677536211529560998432049400961861338863911084030381488628159073170114235547302303533626208986835613080075985703121250813571733533606272496417145565136129415437696905495272776402217189832840401938243495416339663411171247074920049399475823655320274233156954218768925951056191034134712504572955839404827707329984173302332020201906541083764475690954512308594220997449412380273230046465841574004772512701790771147617828666064344158947341035545499540170260305012929701470776236465507485850428931202947542598396287345703761265310456809232764193204759624931172723676784849010063883831645335627155765372880260543270126668904548807658246837332956745606320439206000827315925297924120517572792956898069837182018081118012502131089972469511003170367875430017874462279301921060156857761490839367431915105478717272782446538831715921363968336746689231345994523668360452657260101103397053499527132362563007397454373813873045156390854348724120700844774879469375150443446048584280930172395926036732129188875719856402864926298810995160417385214470404976503137921156910217010840121652176385776278443131535045190731074843750437867090838446698767945050890489992429995490314062282068159093045161404523458248697227150619981888378435664415174711160595006902423143459075966810454416997061373268370421830924936517791683419258027937814913005585514983905421612991836639607353242591728408919130405601743611335886762255281130983568838120661186537684120574342592819568100284858774281240119689820354838043041113162808407169939503577633814675423251711145297625856010709698328986771681300270753462124431438249079287242299178307352064301908595916689642113630953871478324579363224590054821977720752752434856153962211487647114809252498114236281605836155826301649837304049554058938232508056924063922852025132566569884400145216494747221972532532860069992021237959974546155685028445119102475386567214984215767810803373267199219113161724471869212162578104306058776030926578404910389652192404389117677295835067825022753770320621321736011850304863716204309012040569800028743530750682957857187725797496332806514717356341562788595136219969639265644529365016344351116338283641806458971781098774751480098405354570139783225722233163294120124683588116225556912157809787231837370485212940457115577099959109605532398430813959714213851721154337178339537086640233608391562229893503493649411031401362724533210284028590050377655433375682840029673698157498020943869571637004045326975420227362140674697142027390581797343320575190963092149403833594112151706423118043310372943975783109168287709984495991399458167906285118753395839483377129729549566614285242117269351913253207745257819406658932664711556189413");

        for (int prec = 500; prec <= 3000; prec += 100)
        {
            Apfloat agm = ApfloatMath.agm(new Apfloat(1, prec), new Apfloat(2, prec));

            assertEquals("agm prec " + prec + " precision", prec, agm.precision());
            assertEquals("agm prec " + prec + " value", a.precision(prec), agm, new Apfloat("5e-" + prec));
        }

        for (int prec = 901; prec < 1000; prec++)
        {
            Apfloat agm = ApfloatMath.agm(new Apfloat(1, prec), new Apfloat(2, prec));

            assertEquals("agm prec " + prec + " precision", prec, agm.precision());
            assertEquals("agm prec " + prec + " value", a.precision(prec), agm, new Apfloat("5e-" + prec));
        }

        try
        {
            ApfloatMath.agm(new Apfloat(1), new Apfloat(2));
            fail("AGM to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testFactorial()
    {
        Apfloat result = ApfloatMath.factorial(100, 10);
        assertEquals("value", new Apfloat(9.332621544e157), result, new Apfloat("5e148"));
        assertEquals("precision", 10, result.precision());

        result = ApfloatMath.factorial(10, Apfloat.INFINITE, 16);
        assertEquals("value 16", new Apfloat(0x375F00, Apfloat.DEFAULT, 16), result);
        assertEquals("precision 16", Apfloat.INFINITE, result.precision());

        try
        {
            ApfloatMath.factorial(7, 10, 1);
            fail("Radix 1 accepted");
        }
        catch (NumberFormatException nfe)
        {
            // OK; invalid radix
        }
    }

    public static void testPi()
    {
        Apfloat pi10 = new Apfloat("3.14159265358979323846264338327950288419716939937510582097494459230781640628620899862803482534211706798214808651328230664709384460955058223172535940812848111745028410270193852110555964462294895493038196442881097566593344612847564823378678316527120190914564856692346034861045432664821339360726024914127372458700660631558817488152092096282925409171536436789259036001133053054882046652138414695194151160943305727036575959195309218611738193261179310511854807446237996274956735188575272489122793818301194912983367336244065664308602139494639522473719070217986094370277053921717629317675238467481846766940513200056812714526356082778577134275778960917363717872146844090122495343014654958537105079227968925892354201995611212902196086403441815981362977477130996051870721134999999837297804995105973173281609631859502445945534690830264252230825334468503526193118817101000313783875288658753320838142061717766914730359825349042875546873115956286388235378759375195778185778053217122680661300192787661119590921642019893809525720106548586327886593615338182796823030195203530185296899577362259941389124972177528347913151");

        assertEquals("prec 100", pi10.precision(100), ApfloatMath.pi(100), new Apfloat(5e-98));
        assertEquals("prec 1000", pi10, ApfloatMath.pi(1000), new Apfloat("5e-998"));
        assertEquals("prec 500 precision", 500, ApfloatMath.pi(500).precision());
        assertEquals("prec 500 value", pi10.precision(500), ApfloatMath.pi(500));
        assertEquals("prec 50 precision", 50, ApfloatMath.pi(50).precision());
        assertEquals("prec 50 value", pi10.precision(50), ApfloatMath.pi(50));

        assertEquals("base 16", new Apfloat("3.243F6A8885A308D313198A2E03707344A4093822299F31D00", Apfloat.DEFAULT, 16), ApfloatMath.pi(50, 16), new Apfloat(5e-100, 1, 16));

        for (int prec = 1000; prec <= 1100; prec++)
        {
            Apfloat pi = ApfloatMath.pi(prec);

            assertEquals("pi prec " + prec + " precision", prec, pi.precision());
            assertEquals("pi prec " + prec + " value", pi10.precision(prec), pi, new Apfloat("5e-" + (prec - 2)));
        }

        try
        {
            ApfloatMath.pi(0);
            fail("Precision 0 accepted");
        }
        catch (IllegalArgumentException e)
        {
            // OK; invalid precision
        }

        try
        {
            ApfloatMath.pi(Apfloat.INFINITE);
            fail("Infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; invalid precision
        }

        try
        {
            ApfloatMath.pi(50, 1);
            fail("Radix 1 accepted");
        }
        catch (NumberFormatException nfe)
        {
            // OK; invalid radix
        }
    }

    public static void testLog()
    {
        Apfloat a = ApfloatMath.log(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision());
        assertEquals("2, 100 value", new Apfloat("0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687"), a, new Apfloat("5e-99"));

        a = ApfloatMath.log(new Apfloat(3, 150, 16));
        assertEquals("3, 150 precision", 150, a.precision());
        assertEquals("3, 150 value", new Apfloat("1.193ea7aad030a976a4198d55053b7cb5be1442d9b7e08df03d97eeea5149358caa9782d20cc698505071f733039a8ed5625c15071ea7bca1cf37d8f11024c66486d094e21e74d0a547df6", Apfloat.DEFAULT, 16), a, ApfloatMath.scale(new Apfloat(5, 1, 16), -148));

        a = ApfloatMath.log(new Apfloat(1));
        assertEquals("1", new Apfloat(0), a);

        a = ApfloatMath.log(new Apfloat("2.71828182845904523536028747135266249775724709369995957496696762772407663035354759457138217852516642"));
        assertEquals("e, 99, precision", 99, a.precision());
        assertEquals("e, 99, value", new Apfloat(1), a, new Apfloat("5e-97"));

        a = ApfloatMath.log(new Apfloat("2.718281828459045235360287471352662497757247093699959574966967627724076630353547594571382178525166427"));
        assertEquals("e, 100, precision", 100, a.precision());
        assertEquals("e, 100, value", new Apfloat(1), a, new Apfloat("5e-98"));

        a = ApfloatMath.log(new Apfloat("7.38905609893065022723042746057500781318031557055184732408712782252257379607905776338431248507912179"));
        assertEquals("e^2, 99, precision", 99, a.precision());
        assertEquals("e^2, 99, value", new Apfloat(2), a, new Apfloat("5e-98"));

        a = ApfloatMath.log(new Apfloat("7.389056098930650227230427460575007813180315570551847324087127822522573796079057763384312485079121794"));
        assertEquals("e^2, 100, precision", 100, a.precision());
        assertEquals("e^2, 100, value", new Apfloat(2), a, new Apfloat("5e-98"));

        a = ApfloatMath.log(new Apfloat("1.1"));
        assertEquals("1.1, precision", 1, a.precision());

        a = ApfloatMath.log(new Apfloat("1.01"));
        assertEquals("1.01, precision", 1, a.precision());

        a = ApfloatMath.log(new Apfloat("1.1", 3));
        assertEquals("1.1, 3, precision", 2, a.precision());

        a = ApfloatMath.log(new Apfloat("2.567534329783818000500533029709932117494558299117260796421579784603311989535237879614227625064708170e2171472", 100));
        assertEquals("e^5000000, 100 precision", 106, a.precision());
        assertEquals("e^5000000, 100 value", new Apfloat(5000000), a, new Apfloat("5e-99"));

        a = ApfloatMath.log(new Apfloat("3.894787261069254290252471079176073765185215930321023940272302235080330044710873765250549604329029361e-2171473", 100));
        assertEquals("-e^5000000, 100 precision", 100, a.precision());
        assertEquals("-e^5000000, 100 value", new Apfloat(-5000000), a, new Apfloat("5e-92"));

        Apfloat x = new Apfloat("0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687542001481020570685733685520235758130557032670751635075961930727570828371435190307038623891673471123350115364497955239120475172681574932065155524734139525882950453007095326366642654104239157814952043740430385500801944170641671518644712839968171784546957026271631064546150257207402481637773389638550695260668341137273873722928956493547025762652098859693201965058554764703306793654432547632744951250406069438147104689946506220167720424524529612687946546193165174681392672504103802546259656869144192871608293803172714367782654877566485085674077648451464439940461422603193096735402574446070308096085047486638523138181676751438667476647890881437141985494231519973548803751658612753529166100071053558249879414729509293113897155998205654392871700072180857610252368892132449713893203784393530887748259701715591070882368362758984258918535302436342143670611892367891923723146723217205340164925687274778234453534");
        a = new Apfloat("-0.3665129205816643270124391582326694694542634478371052630536777136705616153193527385494558228566989083583025230453648347655663425171940646634814655030562792138730255618922699717672286041800832613019942189532855463393890461583132806371040809369384417114396747846899170526422674184027391035004631190422189323680484095772276875435803559664871149188455813202333163965976808249936452933584668462357426343756525369861095695215917735720106501342222449994153507078142297833029637774311970293974350362633156244687655088608994135490709908943039494582566844946811522717063194325489044697564936223469819270466722804100902961886880218201749101992361055490535324877585339998097074377916278218193441996319021050429330124241505927926809754056140292099176158224802342090188412736812269321284034933052507728909167575791393588626671908786468115956639594429979580206395481395923027937867469315592027193309322415124007731716247058993690703827533513739231800701041955722130456332586628947326070184882339110039410732806653401");

        for (int prec = 500; prec <= 800; prec += 100)
        {
            Apfloat log = ApfloatMath.log(x.precision(prec));

            assertEquals("log(log2) prec " + prec + " precision", prec, log.precision());
            assertEquals("log(log2) prec " + prec + " value", a.precision(prec), log, new Apfloat("5e-" + prec));
        }

        for (int prec = 900; prec <= 1000; prec++)
        {
            Apfloat log = ApfloatMath.log(x.precision(prec));

            // NOTE: EXTRA ALLOWED ERROR NEEDED FOR CASE prec=1000
            assertEquals("log(log2) prec " + prec + " precision", prec, log.precision());
            assertEquals("log(log2) prec " + prec + " value", a.precision(prec), log, new Apfloat("11e-" + prec));
        }

        a = new Apfloat("0.6931471805599453094172321214581765680755001343602552541206800094933936219696947156058633269964186875420014810205706857336855202357581305570326707516350759619307275708283714351903070386238916734711233501153644979552391204751726815749320651555247341395258829504530070953263666426541042391578149520437404303855008019441706416715186447128399681717845469570262716310645461502572074024816377733896385506952606683411372738737229289564935470257626520988596932019650585547647033067936544325476327449512504060694381471046899465062201677204245245296126879465461931651746813926725041038025462596568691441928716082938031727143677826548775664850856740776484514644399404614226031930967354025744460703080960850474866385231381816767514386674766478908814371419854942315199735488037516586127535291661000710535582498794147295092931138971559982056543928717000721808576102523688921324497138932037843935308877482597017155910708823683627589842589185353024363421436706118923678919237231467232172053401649256872747782344535347648114941864238677677440606956265737960086707625719918473402265146283790488306203306114463007371949");

        for (int prec = 1000; prec <= 1100; prec++)
        {
            Apfloat log = ApfloatMath.log(new Apfloat(2, prec));

            assertEquals("log(2) prec " + prec + " precision", prec, log.precision());
            assertEquals("log(2) prec " + prec + " value", a.precision(prec), log, new Apfloat("5e-" + prec));
        }

        try
        {
            ApfloatMath.log(new Apfloat(0));
            fail("log of zero accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be -infinite
        }

        try
        {
            ApfloatMath.log(new Apfloat(-1));
            fail("log of -1 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be complex
        }

        try
        {
            ApfloatMath.log(new Apfloat(2));
            fail("log to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testLogBase()
    {
        Apfloat a = ApfloatMath.log(new Apfloat(4, 100), new Apfloat(2, 100));
        assertEquals("4 base 2, precision 100", 100, a.precision());
        assertEquals("4 base 2, value", new Apfloat(2), a, new Apfloat("5e-99"));

        a = ApfloatMath.log(new Apfloat(8, 400), new Apfloat(2, 200));
        assertEquals("4 base 2, precision 200", 200, a.precision());
        assertEquals("4 base 2, value", new Apfloat(3), a, new Apfloat("5e-199"));

        a = ApfloatMath.log(new Apfloat(16, 300), new Apfloat(2, 600));
        assertEquals("4 base 2, precision 300", 300, a.precision());
        assertEquals("4 base 2, value", new Apfloat(4), a, new Apfloat("5e-299"));

        a = ApfloatMath.log(new Apfloat(16, 400), new Apfloat(2));
        assertEquals("4 base 2, precision 400", 400, a.precision());
        assertEquals("4 base 2, value", new Apfloat(4), a, new Apfloat("5e-399"));

        a = ApfloatMath.log(new Apfloat(16), new Apfloat(2, 500));
        assertEquals("4 base 2, precision 500", 500, a.precision());
        assertEquals("4 base 2, value", new Apfloat(4), a, new Apfloat("5e-499"));

        a = ApfloatMath.log(new Apfloat(2, 5), new Apfloat("1.01", 7));
        assertEquals("2 base 1.01, precision", 5, a.precision());
        assertEquals("2 base 1.01, value", new Apfloat("69.661"), a, new Apfloat("5e-3"));

        a = ApfloatMath.log(new Apfloat("1.01", 7), new Apfloat(2, 5));
        assertEquals("1.01 base 2, precision", 5, a.precision());
        assertEquals("1.01 base 2, value", new Apfloat("0.014355"), a, new Apfloat("5e-6"));
    }

    public static void testExp()
    {
        Apfloat a = ApfloatMath.exp(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision());
        assertEquals("2, 100 value", new Apfloat("7.389056098930650227230427460575007813180315570551847324087127822522573796079057763384312485079121794"), a, new Apfloat("5e-98"));

        a = ApfloatMath.exp(new Apfloat(3, 150, 16));
        assertEquals("3, 150 precision", 150, a.precision());
        assertEquals("3, 150 value", new Apfloat("14.15E5BF6FB105F2D4BDFC53744C3A390585839728AA90A12389790C837E6FF2A68ABFE2D58DAFA5273C74EB175B9D0BD5584DFC81BE96F62CE66F428A2A50FFCBDF4083E8EE811BE42F", Apfloat.DEFAULT, 16), a, ApfloatMath.scale(new Apfloat(5, 1, 16), -147));

        a = ApfloatMath.exp(new Apfloat(0, 1, 17));
        assertEquals("1 radix", 17, a.radix());
        assertEquals("1 prec", Apfloat.INFINITE, a.precision());
        assertEquals("1 value", new Apfloat(1), a);

        a = ApfloatMath.exp(new Apfloat("0.1"));
        assertEquals("0.1, precision", 2, a.precision());

        a = ApfloatMath.exp(new Apfloat("0.01"));
        assertEquals("0.01, precision", 3, a.precision());

        a = ApfloatMath.exp(new Apfloat("0.1", 2));
        assertEquals("0.1, 2, precision", 3, a.precision());

        a = ApfloatMath.exp(new Apfloat("1e-10", 32));
        assertEquals("1e-10", new Apfloat("1.0000000001000000000050000000002"), a, new Apfloat("5e-31"));

        //a = ApfloatMath.exp(new Apfloat("1e" + Long.MIN_VALUE / 3 * 2));
        //assertEquals("1e-2/3MIN", new Apfloat(1), a);

        a = ApfloatMath.exp(new Apfloat(5000000, 21));
        assertEquals("5000000, 15 precision", 15, a.precision());
        assertEquals("5000000, 15 value", new Apfloat("2.56753432978382e2171472"), a, new Apfloat("5e2171458"));

        a = ApfloatMath.exp(new Apfloat(5000000, 106));
        assertEquals("5000000, 100 precision", 100, a.precision());
        assertEquals("5000000, 100 value", new Apfloat("2.567534329783818000500533029709932117494558299117260796421579784603311989535237879614227625064708170e2171472"), a, new Apfloat("5e2171373"));

        a = ApfloatMath.exp(new Apfloat(-5000000, 106));
        assertEquals("-5000000, 100 precision", 100, a.precision());
        assertEquals("-5000000, 100 value", new Apfloat("3.894787261069254290252471079176073765185215930321023940272302235080330044710873765250549604329029361e-2171473"), a, new Apfloat("5e-2171572"));

        a = ApfloatMath.exp(new Apfloat("-1e25", 50));
        assertEquals("-1e25, 50 value", new Apfloat(0), a);

        a = ApfloatMath.exp(new Apfloat("-21237598959199934509.830775042768"));
        assertEquals("-21237598959199934509.830775042768 value", new Apfloat(0), a);

        // How to compute this with some other tool:
        // exp(-2.123759895919992e19) = exp(-21237598959199920000) = exp(-2737330 * 2304964 * 3366000)
        // = (exp(-2737330) ^ 2304964) ^ 3366000
        // = ((4.8512289541615048587 * 10 ^ -1188808) ^ 2304964) ^ 3366000
        // = (4.8512289541615048587 ^ 2304964 * 10 ^ (-1188808 * 2304964)) ^ 3366000
        // = (4.3958838520520181922 * 10 ^ 1580863 * 10 ^ (-1188808 * 2304964)) ^ 3366000
        // = (4.3958838520520181922 * 10 ^ (1580863 + (-1188808 * 2304964)) ^ 3366000
        // = 4.3958838520520181922 ^ 3366000 * 10 ^ ((1580863 + (-1188808 * 2304964)) * 3366000)
        // = 3.4628920318050121308 * 10 ^ 2164493 * 10 ^ ((1580863 + (-1188808 * 2304964)) * 3366000)
        // = 3.4628920318050121308 * 10 ^ (2164493 + (1580863 + (-1188808 * 2304964)) * 3366000)
        // = 3.4628920318050121308 * 10 ^ -9223372036854769507
        a = ApfloatMath.exp(new Apfloat("-2.1237598959199920000000000000000000e19"));
        assertEquals("-2.123759895919992e19 precision", 16, a.precision());
        assertEquals("-2.123759895919992e19 value", new Apfloat("3.462892031805012e-9223372036854769507"), a, new Apfloat("5e-9223372036854769491"));

        a = new Apfloat("2.718281828459045235360287471352662497757247093699959574966967627724076630353547594571382178525166427427466391932003059921817413596629043572900334295260595630738132328627943490763233829880753195251019011573834187930702154089149934884167509244761460668082264800168477411853742345442437107539077744992069551702761838606261331384583000752044933826560297606737113200709328709127443747047230696977209310141692836819025515108657463772111252389784425056953696770785449969967946864454905987931636889230098793127736178215424999229576351482208269895193668033182528869398496465105820939239829488793320362509443117301238197068416140397019837679320683282376464804295311802328782509819455815301756717361332069811250996181881593041690351598888519345807273866738589422879228499892086805825749279610484198444363463244968487560233624827041978623209002160990235304369941849146314093431738143640546253152096183690888707016768396424378140592714563549061303107208510383750510115747704171898610687396965521267154688957035035");

        for (int prec = 500; prec <= 800; prec += 100)
        {
            Apfloat exp = ApfloatMath.exp(new Apfloat(1, prec));

            assertEquals("exp prec " + prec + " precision", prec, exp.precision());
            assertEquals("exp prec " + prec + " value", a.precision(prec), exp, new Apfloat("5e-" + (prec - 1)));
        }

        for (int prec = 900; prec <= 1000; prec++)
        {
            Apfloat exp = ApfloatMath.exp(new Apfloat(1, prec));

            assertEquals("exp prec " + prec + " precision", prec, exp.precision());
            assertEquals("exp prec " + prec + " value", a.precision(prec), exp, new Apfloat("5e-" + (prec - 1)));
        }

        try
        {
            ApfloatMath.exp(new Apfloat("50000000000000000000", 500000000));
            fail("Overflow should have occurred");
        }
        catch (OverflowException oe)
        {
            // OK; result would overflow
        }

        try
        {
            ApfloatMath.exp(new Apfloat(2));
            fail("exp to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testPow()
    {
        Apfloat a = ApfloatMath.pow(new Apfloat(2, 100), new Apfloat("0.5", 100));
        assertEquals("2, 100 precision", 100, a.precision(), 1);
        assertEquals("2, 100 value", new Apfloat("1.414213562373095048801688724209698078569671875376948073176679737990732478462107038850387534327641573"), a, new Apfloat("5e-98"));

        a = new Apfloat("1.193EA7AAD030A976A4198D55053B7CB5BE1442D9B7E08DF03D97EEEA5149358CAA9782D20CC698505071F733039A8ED5625C15071EA7BCA1CF37D8F11024C66486D094E21E74D0A547DF6", 150, 16);
        a = ApfloatMath.pow(a, a);
        assertEquals("3, 150 precision", 150, a.precision(), 1);
        assertEquals("3, 150 value", new Apfloat("1.1BDD7EF9BE61C2CAB36ED12BED65238AFAC02791331A39420E85923B550F899BC353D7408E2B07B374E294888A03E98DCBF9B55433D064C6D28E005F42ED02FAD35AC14C5E4A2E1F30AD8", Apfloat.DEFAULT, 16), a, ApfloatMath.scale(new Apfloat(5, 1, 16), -148));

        a = ApfloatMath.pow(new Apfloat(2), new Apfloat(0));
        assertEquals("2^0", new Apfloat(1), a);

        a = ApfloatMath.pow(new Apfloat(2), new Apfloat(1));
        assertEquals("2^1", new Apfloat(2), a);

        a = ApfloatMath.pow(new Apfloat(0), new Apfloat(2));
        assertEquals("0^2", new Apfloat(0), a);

        a = ApfloatMath.pow(new Apfloat(1), new Apfloat(2));
        assertEquals("1^2", new Apfloat(1), a);

        try
        {
            ApfloatMath.pow(new Apfloat("1.2"), new Apfloat("57"));
        }
        catch (LossOfPrecisionException lope)
        {
            // OK
        }

        a = ApfloatMath.pow(new Apfloat("1.02"), new Apfloat("57"));
        assertEquals("1.02^57, precision", 1, a.precision());

        a = ApfloatMath.pow(new Apfloat("1.2"), new Apfloat("1.0"));
        assertEquals("1.2^1.0, precision", 2, a.precision());

        a = ApfloatMath.pow(new Apfloat("1.2"), new Apfloat("1"));
        assertEquals("1.2^1, precision", 1, a.precision());

        a = ApfloatMath.pow(new Apfloat("1.02"), new Apfloat("1.00"));
        assertEquals("1.02^1.00, precision", 3, a.precision());

        a = ApfloatMath.pow(new Apfloat("1.02"), new Apfloat("1"));
        assertEquals("1.02^1, precision", 1, a.precision());

        a = ApfloatMath.pow(new Apfloat("1.2"), new Apfloat("0.01"));
        assertEquals("1.2^0.01, precision", 4, a.precision());

        a = ApfloatMath.pow(new Apfloat("1.02"), new Apfloat("0.01"));
        assertEquals("1.02^0.01, precision", 5, a.precision());

        a = ApfloatMath.pow(new Apfloat("12312312342343245234234.4554123", 100), new Apfloat("432.243245234", 100));
        assertEquals("12312312342343245234234.4554123^432.243245234, precision", 96, a.precision());
        assertEquals("12312312342343245234234.4554123^432.243245234, value", new Apfloat("2.51239266531220414856835532497217396556903456254971203012293241477748612273092084132746823767646e9548"), a, new Apfloat("5e9453"));

        Apfloat x = new Apfloat("0.693147180559945309417232121458176568075500134360255254120680009493393621969694715605863326996418687542001481020570685733685520235758130557032670751635075961930727570828371435190307038623891673471123350115364497955239120475172681574932065155524734139525882950453007095326366642654104239157814952043740430385500801944170641671518644712839968171784546957026271631064546150257207402481637773389638550695260668341137273873722928956493547025762652098859693201965058554764703306793654432547632744951250406069438147104689946506220167720424524529612687946546193165174681392672504103802546259656869144192871608293803172714367782654877566485085674077648451464439940461422603193096735402574446070308096085047486638523138181676751438667476647890881437141985494231519973548803751658612753529166100071053558249879414729509293113897155998205654392871700072180857610252368892132449713893203784393530887748259701715591070882368362758984258918535302436342143670611892367891923723146723217205340164925687274778234453534"),
                y = new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930381964428810975665933446128475648233786783165271201909145648566923460348610454326648213393607260249141273724587006606315588174881520920962829254091715364367892590360011330530548820466521384146951941511609433057270365759591953092186117381932611793105118548074462379962749567351885752724891227938183011949129833673362440656643086021394946395224737190702179860943702770539217176293176752384674818467669405132000568127145263560827785771342757789609173637178721468440901224953430146549585371050792279689258923542019956112129021960864034418159813629774771309960518707211349999998372978049951059731732816096318595024459455346908302642522308253344685035261931188171010003137838752886587533208381420617177669147303598253490428755468731159562863882353787593751957781857780532171226806613001927876611195909216420198");
        a = new Apfloat("0.316182943197044690614169268497792885452164361693051710732667324268823185192218337501840769380849175934963329367241486177119992719972036535656516370698214957622438377784029981033223873271141421939813138295037738564766790036552697116686100077566696580000839312315156298008831180831582540185037993133224966864508908869785121857911552089207739673945009998248291357701939332415732827372723670347306280056437522519983428006474751126018076658937443328965140553449786853150070311258642464494464377283951772289329403314138401969945114622833848835745934222479165579350121531744358768058822541639234759395945505389376644705555314322615751906758924387381898831546019383916458560077972932084527603542042873539826316621369800590919207282064653521206479549258209266902697520830020862834814623634525867254396928645227883965555367571830849867301421792625669502862980792359748616959118702882311710152741878652879534809041570410154075983958637237380877034358236840420087080482174290845507248144579739353309032206160873");

        for (int prec = 500; prec <= 800; prec += 100)
        {
            Apfloat pow = ApfloatMath.pow(x.precision(prec), y.precision(prec));

            assertEquals("pow prec " + prec + " precision", prec, pow.precision());
            assertEquals("pow prec " + prec + " value", a.precision(prec), pow, new Apfloat("5e-" + prec));
        }

        for (int prec = 900; prec <= 1000; prec++)
        {
            Apfloat pow = ApfloatMath.pow(x.precision(prec), y.precision(prec));

            // NOTE: EXTRA ALLOWED ERROR NEEDED FOR CASE prec=1000
            assertEquals("pow prec " + prec + " precision", prec, pow.precision());
            assertEquals("pow prec " + prec + " value", a.precision(prec), pow, new Apfloat("10e-" + prec));
        }

        try
        {
            ApfloatMath.pow(new Apfloat(0), new Apfloat(0));
            fail("0^0 accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be undefined
        }

        try
        {
            ApfloatMath.pow(new Apfloat(100, 100), new Apfloat(5000000000000000000L, 100));
            fail("Overflow should have occurred");
        }
        catch (OverflowException oe)
        {
            // OK; result would overflow
        }

        try
        {
            ApfloatMath.pow(new Apfloat(-2), new Apfloat("1.3"));
            fail("pow of negative number accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be complex
        }

        try
        {
            ApfloatMath.pow(new Apfloat(3), new Apfloat(-1));
            fail("pow to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; can't calculate this to infinite precision
        }
    }

    public static void testAcosh()
    {
        Apfloat a = ApfloatMath.acosh(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision(), 1);
        assertEquals("2, 100 value", new Apfloat("1.316957896924816708625046347307968444026981971467516479768472256920460185416443976074219013450101783"), a, new Apfloat("5e-98"));

        a = ApfloatMath.acosh(new Apfloat("1e5000000", 94));
        assertEquals("1e5000000, 100 precision", 100, a.precision(), 1);
        assertEquals("1e5000000, 100 value", new Apfloat("11512926.158117408980035266690653942496182075518643999240421893625517872541780384370874701631311318488"), a, new Apfloat("5e-91"));

        a = ApfloatMath.acosh(new Apfloat(1, 100));
        assertEquals("1, 100 value", new Apfloat(0), a);

        try
        {
            ApfloatMath.acosh(new Apfloat("0.9"));
            fail("acosh(0.9) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be complex
        }

        try
        {
            ApfloatMath.acosh(new Apfloat("-0.5"));
            fail("acosh(-0.5) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be complex
        }
    }

    public static void testAsinh()
    {
        Apfloat a = ApfloatMath.asinh(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision(), 1);
        assertEquals("2, 100 value", new Apfloat("1.443635475178810342493276740273105269405553003156981558983054506520491602824665323236028287368170425"), a, new Apfloat("5e-98"));

        a = ApfloatMath.asinh(new Apfloat(-2, 100));
        assertEquals("-2, 100 precision", 100, a.precision(), 1);
        assertEquals("-2, 100 value", new Apfloat("-1.443635475178810342493276740273105269405553003156981558983054506520491602824665323236028287368170425"), a, new Apfloat("5e-98"));

        a = ApfloatMath.asinh(new Apfloat("1e5000000", 94));
        assertEquals("1e5000000, 100 precision", 100, a.precision(), 1);
        assertEquals("1e5000000, 100 value", new Apfloat("11512926.158117408980035266690653942496182075518643999240421893625517872541780384370874701631311318488"), a, new Apfloat("5e-91"));

        a = ApfloatMath.asinh(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testAtanh()
    {
        Apfloat a = ApfloatMath.atanh(new Apfloat("0.5", 100));
        assertEquals("0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("0.5, 100 value", new Apfloat("0.5493061443340548456976226184612628523237452789113747258673471668187471466093044834368078774068660444"), a, new Apfloat("5e-99"));

        a = ApfloatMath.atanh(new Apfloat("-0.5", 100));
        assertEquals("-0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("-0.5, 100 value", new Apfloat("-0.5493061443340548456976226184612628523237452789113747258673471668187471466093044834368078774068660444"), a, new Apfloat("5e-99"));

        a = ApfloatMath.atanh(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApfloatMath.atanh(new Apfloat("1"));
            fail("atanh(1) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be infinite
        }

        try
        {
            ApfloatMath.atanh(new Apfloat("-1"));
            fail("atanh(-1) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be infinite
        }
    }

    public static void testCosh()
    {
        Apfloat a = ApfloatMath.cosh(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision(), 1);
        assertEquals("2, 100 value", new Apfloat("3.7621956910836314595622134777737461082939735582307116027776433475883235850902727266607053037848894217"), a, new Apfloat("5e-98"));

        a = ApfloatMath.cosh(new Apfloat(-2, 100));
        assertEquals("-2, 100 precision", 100, a.precision(), 1);
        assertEquals("-2, 100 value", new Apfloat("3.7621956910836314595622134777737461082939735582307116027776433475883235850902727266607053037848894217"), a, new Apfloat("5e-98"));

        a = ApfloatMath.cosh(new Apfloat(5000000, 106));
        assertEquals("5000000, 100 precision", 100, a.precision(), 1);
        assertEquals("5000000, 100 value", new Apfloat("1.283767164891909000250266514854966058747279149558630398210789892301655994767618939807113812532354085e2171472"), a, new Apfloat("5e2171372"));

        a = ApfloatMath.cosh(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(1), a);
    }

    public static void testSinh()
    {
        Apfloat a = ApfloatMath.sinh(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision(), 1);
        assertEquals("2, 100 value", new Apfloat("3.626860407847018767668213982801261704886342012321135721309484474934250210988785036723607181294232373"), a, new Apfloat("5e-98"));

        a = ApfloatMath.sinh(new Apfloat(-2, 100));
        assertEquals("-2, 100 precision", 100, a.precision(), 1);
        assertEquals("-2, 100 value", new Apfloat("-3.626860407847018767668213982801261704886342012321135721309484474934250210988785036723607181294232373"), a, new Apfloat("5e-98"));

        a = ApfloatMath.sinh(new Apfloat(5000000, 106));
        assertEquals("5000000, 100 precision", 100, a.precision(), 1);
        assertEquals("5000000, 100 value", new Apfloat("1.283767164891909000250266514854966058747279149558630398210789892301655994767618939807113812532354085e2171472"), a, new Apfloat("5e2171372"));

        a = ApfloatMath.sinh(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testTanh()
    {
        Apfloat a = ApfloatMath.tanh(new Apfloat(2, 100));
        assertEquals("2, 100 precision", 100, a.precision(), 1);
        assertEquals("2, 100 value", new Apfloat("0.96402758007581688394641372410092315025502997624093477604826321741310794631761020255947485004520768915"), a, new Apfloat("5e-99"));

        a = ApfloatMath.tanh(new Apfloat(-2, 100));
        assertEquals("-2, 100 precision", 100, a.precision(), 1);
        assertEquals("-2, 100 value", new Apfloat("-0.96402758007581688394641372410092315025502997624093477604826321741310794631761020255947485004520768915"), a, new Apfloat("5e-99"));

        a = ApfloatMath.tanh(new Apfloat(5000000, 100));
        assertEquals("5000000, 100 precision", 4343037, a.precision(), 1);
        assertEquals("5000000, 100 value", new Apfloat(1).subtract(new Apfloat("3.03387356179746871517892756452203249280217116298171287391802587502345779082540217305405121824e-4342945")), a, new Apfloat("5e-4343037"));

        a = ApfloatMath.tanh(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testAcos()
    {
        Apfloat a = ApfloatMath.acos(new Apfloat("0.5", 100));
        assertEquals("0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("0.5, 100 value", new Apfloat("1.04719755119659774615421446109316762806572313312503527365831486410260546876206966620934494178070569"), a, new Apfloat("5e-98"));

        a = ApfloatMath.acos(new Apfloat("-0.5", 100));
        assertEquals("-0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("-0.5, 100 value", new Apfloat("2.09439510239319549230842892218633525613144626625007054731662972820521093752413933241868988356141138"), a, new Apfloat("5e-98"));

        a = ApfloatMath.acos(new Apfloat("1e-100"));
        assertEquals("0, 100 precision", 100, a.precision(), 1);
        assertEquals("0, 100 value", new Apfloat("1.57079632679489661923132169163975144209858469968755291048747229615390820314310449931401741267105853"), a, new Apfloat("5e-98"));

        a = ApfloatMath.acos(new Apfloat(1, 100));
        assertEquals("1, 100 value", new Apfloat(0), a);

        try
        {
            ApfloatMath.acos(new Apfloat(0));
            fail("acos(0) accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; result would have infinite precision
        }

        try
        {
            ApfloatMath.acos(new Apfloat("1.1"));
            fail("acos(1.1) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be complex
        }
    }

    public static void testAsin()
    {
        Apfloat a = ApfloatMath.asin(new Apfloat("0.5", 100));
        assertEquals("0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("0.5, 100 value", new Apfloat("0.5235987755982988730771072305465838140328615665625176368291574320513027343810348331046724708903528447"), a, new Apfloat("5e-99"));

        a = ApfloatMath.asin(new Apfloat("-0.5", 100));
        assertEquals("-0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("-0.5, 100 value", new Apfloat("-0.5235987755982988730771072305465838140328615665625176368291574320513027343810348331046724708903528447"), a, new Apfloat("5e-99"));

        a = ApfloatMath.asin(new Apfloat("0.00001", 100));
        assertEquals("0.00001, 100 precision", 100, a.precision(), 1);
        assertEquals("0.00001, 100 value", new Apfloat("0.00001000000000016666666667416666666711309523812562003968477689844894697641769350752329383472276925426623"), a, new Apfloat("5e-103"));

        a = ApfloatMath.asin(new Apfloat(1, 100));
        assertEquals("1, 100 precision", 100, a.precision(), 1);
        assertEquals("1, 100 value", new Apfloat("1.57079632679489661923132169163975144209858469968755291048747229615390820314310449931401741267105853"), a, new Apfloat("5e-98"));

        a = ApfloatMath.asin(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApfloatMath.asin(new Apfloat("1.1"));
            fail("asin(1.1) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be complex
        }
    }

    public static void testAtan()
    {
        Apfloat a = ApfloatMath.atan(new Apfloat("0.5", 100));
        assertEquals("0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("0.5, 100 value", new Apfloat("0.4636476090008061162142562314612144020285370542861202638109330887201978641657417053006002839848878926"), a, new Apfloat("5e-99"));

        a = ApfloatMath.atan(new Apfloat("-0.5", 100));
        assertEquals("-0.5, 100 precision", 100, a.precision(), 1);
        assertEquals("-0.5, 100 value", new Apfloat("-0.4636476090008061162142562314612144020285370542861202638109330887201978641657417053006002839848878926"), a, new Apfloat("5e-99"));

        a = ApfloatMath.atan(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);
    }

    public static void testAtan2()
    {
        Apfloat a = ApfloatMath.atan2(new Apfloat(0), new Apfloat(2, 20));
        assertEquals("0,2 value", new Apfloat(0), a);

        a = ApfloatMath.atan2(new Apfloat(2, 20), new Apfloat(2, 20));
        assertEquals("2,2 precision", 20, a.precision(), 1);
        assertEquals("2,2 value", new Apfloat("0.78539816339744830962"), a, new Apfloat("5e-20"));

        a = ApfloatMath.atan2(new Apfloat(2, 20), new Apfloat(0));
        assertEquals("2,0 precision", 20, a.precision());
        assertEquals("2,0 value", new Apfloat("1.5707963267948966192"), a, new Apfloat("5e-19"));

        a = ApfloatMath.atan2(new Apfloat(2, 20), new Apfloat(-2, 20));
        assertEquals("2,-2 precision", 20, a.precision(), 1);
        assertEquals("2,-2 value", new Apfloat("2.3561944901923449288"), a, new Apfloat("5e-19"));

        a = ApfloatMath.atan2(new Apfloat(0), new Apfloat(-2, 20));
        assertEquals("0,-2 precision", 20, a.precision());
        assertEquals("0,-2 value", new Apfloat("3.1415926535897932385"), a, new Apfloat("5e-19"));

        a = ApfloatMath.atan2(new Apfloat(-2, 20), new Apfloat(2, 20));
        assertEquals("-2,2 precision", 20, a.precision(), 1);
        assertEquals("-2,2 value", new Apfloat("-0.78539816339744830962"), a, new Apfloat("5e-20"));

        a = ApfloatMath.atan2(new Apfloat(-2, 20), new Apfloat(0));
        assertEquals("-2,0 precision", 20, a.precision(), 1);
        assertEquals("-2,0 value", new Apfloat("-1.5707963267948966192"), a, new Apfloat("5e-19"));

        a = ApfloatMath.atan2(new Apfloat(-2, 20), new Apfloat(-2, 20));
        assertEquals("-2,-2 precision", 20, a.precision(), 1);
        assertEquals("-2,-2 value", new Apfloat("-2.3561944901923449288"), a, new Apfloat("5e-19"));

        a = ApfloatMath.atan2(new Apfloat(2, 20), new Apfloat(3, 20));
        assertEquals("2,3 precision", 20, a.precision(), 1);
        assertEquals("2,3 value", new Apfloat("0.58800260354756755125"), a, new Apfloat("5e-20"));

        a = ApfloatMath.atan2(new Apfloat("6.618533197939224e-13"), new Apfloat("0.9999999966650702"));
        assertEquals("6.618533197939224e-13,0.9999999966650702 precision", 16, a.precision(), 1);
        assertEquals("6.618533197939224e-13,0.9999999966650702 value", new Apfloat("6.61853322001157e-13"), a, new Apfloat("5e-27"));

        a = ApfloatMath.atan2(new Apfloat("-6.618533197939224e-13"), new Apfloat("0.9999999966650702"));
        assertEquals("-6.618533197939224e-13,0.9999999966650702 precision", 16, a.precision(), 1);
        assertEquals("-6.618533197939224e-13,0.9999999966650702 value", new Apfloat("-6.61853322001157e-13"), a, new Apfloat("5e-27"));

        try
        {
            ApfloatMath.atan2(new Apfloat(0), new Apfloat(0));
            fail("atan2(0,0) accepted");
        }
        catch (ArithmeticException ae)
        {
            // OK; result would be undefined
        }

        try
        {
            ApfloatMath.atan2(new Apfloat(2), new Apfloat(3));
            fail("atan2 to infinite precision accepted");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK; result would have infinite precision
        }
    }

    public static void testCos()
    {
        Apfloat a = ApfloatMath.cos(new Apfloat(1, 100));
        assertEquals("1, 100 precision", 100, a.precision(), 1);
        assertEquals("1, 100 value", new Apfloat("0.5403023058681397174009366074429766037323104206179222276700972553811003947744717645179518560871830893"), a, new Apfloat("5e-99"));

        a = ApfloatMath.cos(new Apfloat(-1, 100));
        assertEquals("-1, 100 precision", 100, a.precision(), 1);
        assertEquals("-1, 100 value", new Apfloat("0.5403023058681397174009366074429766037323104206179222276700972553811003947744717645179518560871830893"), a, new Apfloat("5e-99"));

        a = ApfloatMath.cos(new Apfloat("3141592653589793239.462643383"));
        assertEquals("1e18pi + 1 precision", 10, a.precision(), 1);
        assertEquals("1e18pi + 1 value", new Apfloat("0.5403023058"), a, new Apfloat("5e-10"));

        a = ApfloatMath.cos(new Apfloat("1e-10", 32));
        assertEquals("1e-10 prec 32 value", new Apfloat("0.99999999999999999999500000000000"), a, new Apfloat("5e-32"));

        a = ApfloatMath.cos(new Apfloat("1e-10", 3));
        assertEquals("1e-10 prec 3 value", new Apfloat("0.99999999999999999999500"), a, new Apfloat("5e-23"));

        a = ApfloatMath.cos(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(1), a);

        try
        {
            ApfloatMath.cos(new Apfloat(1000, 3));
            fail("cos(1000 prec 3) accepted");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK; loss of precision
        }
    }

    public static void testSin()
    {
        Apfloat a = ApfloatMath.sin(new Apfloat(1, 100));
        assertEquals("1, 100 precision", 100, a.precision(), 1);
        assertEquals("1, 100 value", new Apfloat("0.8414709848078965066525023216302989996225630607983710656727517099919104043912396689486397435430526959"), a, new Apfloat("5e-99"));

        a = ApfloatMath.sin(new Apfloat(-1, 100));
        assertEquals("-1, 100 precision", 100, a.precision(), 1);
        assertEquals("-1, 100 value", new Apfloat("-0.8414709848078965066525023216302989996225630607983710656727517099919104043912396689486397435430526959"), a, new Apfloat("5e-99"));

        a = ApfloatMath.sin(new Apfloat("3141592653589793239.462643383"));
        assertEquals("1e18pi + 1 precision", 10, a.precision(), 1);
        assertEquals("1e18pi + 1 value", new Apfloat("0.8414709848"), a, new Apfloat("5e-10"));

        a = ApfloatMath.sin(new Apfloat("1e-10", 32));
        assertEquals("1e-10 prec 32 value", new Apfloat("0.99999999999999999999833333333333e-10"), a, new Apfloat("5e-42"));

        a = ApfloatMath.sin(new Apfloat("1e-10", 3));
        assertEquals("1e-10 prec 3 value", new Apfloat("0.99999999999999999999833e-10"), a, new Apfloat("5e-33"));

        a = ApfloatMath.sin(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApfloatMath.sin(new Apfloat(1000, 3));
            fail("sin(1000 prec 3) accepted");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK; loss of precision
        }
    }

    public static void testTan()
    {
        Apfloat a = ApfloatMath.tan(new Apfloat(1, 100));
        assertEquals("1, 100 precision", 100, a.precision(), 1);
        assertEquals("1, 100 value", new Apfloat("1.55740772465490223050697480745836017308725077238152003838394660569886139715172728955509996520224298"), a, new Apfloat("5e-98"));

        a = ApfloatMath.tan(new Apfloat(-1, 100));
        assertEquals("-1, 100 precision", 100, a.precision(), 1);
        assertEquals("-1, 100 value", new Apfloat("-1.55740772465490223050697480745836017308725077238152003838394660569886139715172728955509996520224298"), a, new Apfloat("5e-98"));

        a = ApfloatMath.tan(new Apfloat(0));
        assertEquals("0, 100 value", new Apfloat(0), a);

        try
        {
            ApfloatMath.tan(new Apfloat(1000, 3));
            fail("tan(1000 prec 3) accepted");
        }
        catch (LossOfPrecisionException lope)
        {
            // OK; loss of precision
        }
    }

    public static void testW()
    {
        Apfloat a = new Apfloat(3, 55, 7);
        Apfloat w = ApfloatMath.w(a);
        assertEquals("value", new Apfloat("1.023055505635040031000066002051603260046346014422202623", 55, 7), w, new Apfloat("5e-54", 1, 7));
        assertEquals("radix", 7, w.radix());
        assertEquals("precision", 55, w.precision());
    }

    public static void testToRadians()
    {
        Apfloat a = ApfloatMath.toRadians(new Apfloat(180, 100));
        assertEquals("180 precision", 100, a.precision());
        assertEquals("180 value", new Apfloat("3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068"), a, new Apfloat("5e-99"));

        a = ApfloatMath.toRadians(new Apfloat(90, 50, 16));
        assertEquals("90 precision", 50, a.precision());
        assertEquals("90 value", new Apfloat("1.921fb54442d18469898cc51701b839a252049c1114cf98e80", 50, 16), a, new Apfloat(8, 1, 16).scale(-49));
    }

    public static void testToDegrees()
    {
        Apfloat a = ApfloatMath.toDegrees(ApfloatMath.pi(100, 10));
        assertEquals("180 precision", 100, a.precision());
        assertEquals("180 value", new Apfloat(180), a, new Apfloat("5e-97"));

        a = ApfloatMath.toDegrees(ApfloatMath.pi(50, 16).divide(new Apfloat(2, 50, 16)));
        assertEquals("90 precision", 50, a.precision());
        assertEquals("90 value", new Apfloat(90, 50, 16), a, new Apfloat(8, 1, 16).scale(-48));
    }

    public static void testProduct()
    {
        Apfloat a = ApfloatMath.product(new Apfloat(2, 5), new Apfloat(3, 2));
        assertEquals("5-2 precision", 2, a.precision());
        assertEquals("5-2 value", new Apfloat(6), a);

        a = ApfloatMath.product(new Apfloat(2), new Apfloat(3, 5), new Apfloat(4, 8));
        assertEquals("MAX-5-8 precision", 5, a.precision());
        assertEquals("MAX-5-8 value", new Apfloat(24), a);

        a = ApfloatMath.product(Apfloat.ZERO, new Apfloat(12345));
        assertEquals("0 precision", Apfloat.INFINITE, a.precision());
        assertEquals("0 value", new Apfloat(0), a);

        a = ApfloatMath.product(new Apfloat("1"),
                                new Apfloat("2.50000000000000000000000000000000000000000000000000000000000000000000001"),
                                new Apfloat("2.0000000000000000000000000000000000001"),
                                new Apfloat(2));
        assertEquals("Many precision", 1, a.precision());
        assertEquals("Many value", new Apfloat(10), a);

        Apfloat[] x = new Apfloat[] { new Apfloat(1000000000000L), new Apfloat(1) };
        ApfloatMath.product(x);
        assertEquals("Array product 1 [0]", new Apfloat(1000000000000L), x[0]);
        assertEquals("Array product 1 [1]", new Apfloat(1), x[1]);

        x = new Apfloat[] { new Apfloat(1), new Apfloat(1000000000000L) };
        ApfloatMath.product(x);
        assertEquals("Array product 2 [0]", new Apfloat(1), x[0]);
        assertEquals("Array product 2 [1]", new Apfloat(1000000000000L), x[1]);

        assertEquals("Empty product", new Apfloat("1"), ApfloatMath.product());

        Apfloat[] numbers = new Apfloat[100000];
        for (int i = 0; i < numbers.length; i++)
        {
            numbers[i] = new Apfloat(i + 1);
        }
        Apfloat factorial = ApintMath.factorial(numbers.length);
        assertEquals("Factorial", factorial, ApfloatMath.product(numbers));

        numbers[0] = factorial;
        assertEquals("Factorial squared", factorial.multiply(factorial), ApfloatMath.product(numbers));
    }

    public static void testSum()
    {
        Apfloat a = ApfloatMath.sum(new Apfloat(12345000, 5), new Apfloat(12345, 5));
        assertEquals("5-2 precision", 5, a.precision());
        assertEquals("5-2 value", new Apfloat(12357000), a);

        a = ApfloatMath.sum(new Apfloat(12345678, 10), new Apfloat(12345, 5));
        assertEquals("8-5 precision", 8, a.precision());
        assertEquals("8-5 value", new Apfloat(12358023), a);

        a = ApfloatMath.sum(new Apfloat(0), new Apfloat(12345, 5));
        assertEquals("0-0 precision", 5, a.precision());
        assertEquals("0-0 value", new Apfloat(12345), a);

        a = ApfloatMath.sum(new Apfloat(12345678, 6), new Apfloat(12, 2));
        assertEquals("6-0 precision", 6, a.precision());
        assertEquals("6-0 value", new Apfloat(12345600), a);

        a = ApfloatMath.sum(new Apfloat(12345678, 5), new Apfloat(12, 2));
        assertEquals("5-0 precision", 5, a.precision());
        assertEquals("5-0 value", new Apfloat(12345000), a);

        a = ApfloatMath.sum(new Apfloat(10), new Apfloat(100, 2));
        assertEquals("10, 100 2 precision", 2, a.precision());
        assertEquals("10, 100 2 value", new Apfloat(110), a);

        a = ApfloatMath.sum(new Apfloat(10), new Apfloat(100, 3));
        assertEquals("10, 100 3 precision", 3, a.precision());
        assertEquals("10, 100 3 value", new Apfloat(110), a);

        a = ApfloatMath.sum(new Apfloat(100, 2), new Apfloat(10));
        assertEquals("100, 10 2 precision", 2, a.precision());
        assertEquals("100, 10 2 value", new Apfloat(110), a);

        a = ApfloatMath.sum(new Apfloat(10), new Apfloat(1, 2));
        assertEquals("10, 1 2 precision", 3, a.precision());
        assertEquals("10, 1 2 value", new Apfloat(11), a);

        a = ApfloatMath.sum(new Apfloat(10), new Apfloat(1, 3));
        assertEquals("10, 1 3 precision", 4, a.precision());
        assertEquals("10, 1 3 value", new Apfloat(11), a);

        a = ApfloatMath.sum(new Apfloat(1, 2), new Apfloat(10));
        assertEquals("1, 10 2 precision", 3, a.precision());
        assertEquals("1, 10 2 value", new Apfloat(11), a);

        a = ApfloatMath.sum(new Apfloat(10), new Apfloat(10));
        assertEquals("10, 10 MAX precision", Apfloat.INFINITE, a.precision());
        assertEquals("10, 10 MAX value", new Apfloat(20), a);

        a = ApfloatMath.sum(new Apfloat(10), new Apfloat(100));
        assertEquals("10, 100 MAX precision", Apfloat.INFINITE, a.precision());
        assertEquals("10, 100 MAX value", new Apfloat(110), a);

        a = ApfloatMath.sum(new Apfloat(50, 2), new Apfloat(50, 2));
        assertEquals("50, 50 2 precision", 3, a.precision());
        assertEquals("50, 50 2 value", new Apfloat(100), a);

        a = ApfloatMath.sum(new Apfloat("2"));
        assertEquals("2 precision", 1, a.precision());
        assertEquals("2 value", new Apfloat(2), a);

        a = ApfloatMath.sum(new Apfloat("1"),
                            new Apfloat("3333333.33"),
                            new Apfloat("2.2"),
                            new Apfloat("444444444400000000000000000000000000"),
                            new Apfloat("6666.6"),
                            new Apfloat("555"),
                            new Apfloat(7));
        assertEquals("Many precision", 36, a.precision());
        assertEquals("Many value", new Apfloat("444444444400000000000000000003340564"), a);

        Apfloat[] x = new Apfloat[] { new Apfloat(1000000000000L), new Apfloat(1) };
        ApfloatMath.sum(x);
        assertEquals("Array sum 1 [0]", new Apfloat(1000000000000L), x[0]);
        assertEquals("Array sum 1 [1]", new Apfloat(1), x[1]);

        x = new Apfloat[] { new Apfloat(1), new Apfloat(1000000000000L) };
        ApfloatMath.sum(x);
        assertEquals("Array sum 2 [0]", new Apfloat(1), x[0]);
        assertEquals("Array sum 2 [1]", new Apfloat(1000000000000L), x[1]);

        assertEquals("Empty sum", new Apcomplex("0"), ApfloatMath.sum());

        Apfloat[] numbers = new Apfloat[100000];
        for (int i = 0; i < numbers.length; i++)
        {
            numbers[i] = new Apfloat(i + 1);
        }
        assertEquals("Big sum", new Apfloat(5000050000L), ApfloatMath.sum(numbers));

        numbers[0] = numbers[0].add(new Apfloat("1e10000000", Apfloat.INFINITE));
        assertEquals("Big number big sum", new Apfloat(5000050000L).add(new Apfloat("1e10000000", Apfloat.INFINITE)), ApfloatMath.sum(numbers));
    }

    public static void testGamma()
    {
        Apfloat a = ApfloatMath.gamma(new Apfloat("0.50000000"));
        assertEquals("0.5 precision", 8, a.precision());
        assertEquals("0.5 value", new Apfloat("1.7724539"), a, new Apfloat("5e-7"));
        a = ApfloatMath.gamma(new Apfloat("-0.50000000"));
        assertEquals("-0.5 precision", 8, a.precision());
        assertEquals("-0.5 value", new Apfloat("-3.5449077"), a, new Apfloat("5e-7"));
        a = ApfloatMath.gamma(new Apfloat(1, 8, 5));
        assertEquals("1 precision", 8, a.precision());
        assertEquals("1 radix", 5, a.radix());
        assertEquals("1 value", new Apfloat("1"), a);
        a = ApfloatMath.gamma(new Apfloat(2, 100, 15));
        assertEquals("2 precision", 100, a.precision());
        assertEquals("2 radix", 15, a.radix());
        assertEquals("2 value", new Apfloat("1"), a);
        a = ApfloatMath.gamma(new Apfloat("3.0000"));
        assertEquals("3 precision", 5, a.precision());
        assertEquals("3 value", new Apfloat("2"), a);
        a = ApfloatMath.gamma(new Apfloat(4));
        assertEquals("4 precision", Apfloat.INFINITE, a.precision());
        assertEquals("4 value", new Apfloat(6), a);
        a = ApfloatMath.gamma(new Apfloat("1.6", 50, 12));
        assertEquals("1.5 precision", 50, a.precision());
        assertEquals("1.5 radix", 12, a.radix());
        assertEquals("1.5 value", new Apfloat("0.a77497505445a57663a7a6a27293557aa7636b52055b106267", 50, 12), a, new Apfloat("6e-50", 1, 12));
        a = ApfloatMath.gamma(new Apfloat("100.1", 200));
        assertEquals("100.1 precision", 198, a.precision());
        assertEquals("100.1 radix", 10, a.radix());
        assertEquals("100.1 value", new Apfloat("1.47845449465151367987473964370058459815616330531263071222863411464877072133812707322505189203455965790627278052589675722784367103797434916544534856714784021130076746063941341581673323445817944799010e156"), a, new Apfloat("5e-41"));
        a = ApfloatMath.gamma(new Apfloat("1000000000000.1", 200));
        assertEquals("1000000000000.1 precision", 187, a.precision());
        assertEquals("1000000000000.1 radix", 10, a.radix());
        assertEquals("1000000000000.1 value", new Apfloat("2.224653017598333183967851025493808141494831993379408189875623062445371016679508334427856798243992091090588534116479723793172610167211719300756575577203942719906409747242019348186775610123e11565705518092"), a, new Apfloat("5e11565705517906"));
        a = ApfloatMath.gamma(new Apfloat("1000000000000.1", 40));
        assertEquals("1000000000000.1 precision", 27, a.precision());
        assertEquals("1000000000000.1 radix", 10, a.radix());
        assertEquals("1000000000000.1 value", new Apfloat("2.22465301759833318396785103e11565705518092"), a, new Apfloat("5e11565705518066"));
        a = ApfloatMath.gamma(new Apfloat("1.110100011010100101001010001000000000000000110011001100110011001100110011001100110011001100110011001100110011001100110011001100110011e39", 133, 2));
        assertEquals("1000000000000.1 radix 2 precision", 88, a.precision());
        assertEquals("1000000000000.1 radix 2 radix", 2, a.radix());
        assertEquals("1000000000000.1 radix 2 value", new Apfloat("1.101100110001111000000001111011100111100110011010110000111001101101000111111101101001101e38420442097744", Apfloat.DEFAULT, 2), a, new Apfloat("1e38420442097657", 1, 2));
        a = ApfloatMath.gamma(new Apfloat("cre66i9s.3lllllllllllllllm", 25, 36));
        assertEquals("1000000000000.1 radix 36 precision", 17, a.precision());
        assertEquals("1000000000000.1 radix 36 radix", 36, a.radix());
        assertEquals("1000000000000.1 radix 36 value", ApfloatMath.scale(new Apfloat("3.nbkxpiq7zeevaryz", Apfloat.DEFAULT, 36), 7431527940352L), a, ApfloatMath.scale(new Apfloat("h", 1, 36), 7431527940336L));
        a = ApfloatMath.gamma(new Apfloat("100000000000000000.1", 100));
        assertEquals("100000000000000000.1 precision", 82, a.precision());
        assertEquals("100000000000000000.1 radix", 10, a.radix());
        assertEquals("100000000000000000.1 value", new Apfloat("6.823026252477337409854801263715027688793307758143854258250527972341074891269582940e1656570551809674810"), a, new Apfloat("5e1656570551809674729"));
        try
        {
            ApfloatMath.gamma(new Apfloat("0"));
            fail("Gamma of zero");
        }
        catch (ArithmeticException ae)
        {
            // OK
        }
        try
        {
            ApfloatMath.gamma(new Apfloat("-1"));
            fail("Gamma of -1");
        }
        catch (ArithmeticException ae)
        {
            // OK
        }
        try
        {
            ApfloatMath.gamma(new Apfloat("1e100", 100));
            fail("Gamma overflow");
        }
        catch (OverflowException are)
        {
            // OK
        }
        try
        {
            ApfloatMath.gamma(new Apfloat("4.5", Apfloat.INFINITE));
            fail("Gamma infinite expansion");
        }
        catch (InfiniteExpansionException iee)
        {
            // OK
        }
    }

    public static void testRandom()
    {
        long maxScale = 0;
        long minScale = Long.MAX_VALUE;
        for (int i = 0; i < 1000; i++)
        {
            Apfloat a = ApfloatMath.random(1000);
            maxScale = Math.max(maxScale, a.scale());
            minScale = Math.min(minScale, a.scale());
        }
        assertEquals("random max scale", 0, maxScale);
        assertTrue("random min scale", minScale < 0);

        for (int radix = Character.MIN_RADIX; radix <= Character.MAX_RADIX; radix++)
        {
            for (int i = 0; i < 100; i++)
            {
                Apfloat a = ApfloatMath.random(1, radix);
                assertEquals("value < 1", -1, a.compareTo(new Apfloat(1, Apfloat.INFINITE, radix)));
                assertTrue("value >= 0", a.signum() >= 0);
            }
        }
    }

    public static void testRandomGaussian()
    {
        Apfloat sum = Apfloat.ZERO,
                squareSum = Apfloat.ZERO;
        final int N = 1000;
        for (int i = 0; i < N; i++)
        {
            Apfloat a = ApfloatMath.randomGaussian(N);
            sum = sum.add(a);
            squareSum = squareSum.add(a.multiply(a));
        }
        Apfloat n = new Apfloat(N),
                mean = sum.divide(n),
                standardDeviation = ApfloatMath.sqrt(squareSum.divide(n).subtract(mean));
        assertEquals("random mean", Apfloat.ZERO, mean, new Apfloat(0.15));
        assertEquals("random standard deviation", Apfloat.ONE, standardDeviation, new Apfloat(0.2));

        for (int radix = Character.MIN_RADIX; radix <= Character.MAX_RADIX; radix++)
        {
            for (int i = 0; i < 100; i++)
            {
                Apfloat a = ApfloatMath.randomGaussian(1, radix);
                assertEquals("abs value <= 5", Apfloat.ZERO, a, new Apfloat(5, Apfloat.INFINITE, radix));
            }
        }

        Apfloat a = ApfloatMath.randomGaussian(100);
        assertEquals("precision 1", 100, a.precision(), 7);
        a = ApfloatMath.randomGaussian(10);
        assertEquals("precision 2", 10, a.precision(), 7);
        a = ApfloatMath.randomGaussian(100);
        assertEquals("precision 3", 100, a.precision(), 7);
        a = ApfloatMath.randomGaussian(200);
        assertEquals("precision 4", 200, a.precision(), 7);
    }

    public static void testMax()
    {
        assertEquals("max of 1 and 1", new Apfloat(1), ApfloatMath.max(new Apfloat(1), new Apfloat(1)));
        assertEquals("max of 1 and 2", new Apfloat(2), ApfloatMath.max(new Apfloat(1), new Apfloat(2)));
        assertEquals("max of 2 and 1", new Apfloat(2), ApfloatMath.max(new Apfloat(2), new Apfloat(1)));
    }

    public static void testMin()
    {
        assertEquals("min of 1 and 1", new Apfloat(1), ApfloatMath.min(new Apfloat(1), new Apfloat(1)));
        assertEquals("min of 1 and 2", new Apfloat(1), ApfloatMath.min(new Apfloat(1), new Apfloat(2)));
        assertEquals("min of 2 and 1", new Apfloat(1), ApfloatMath.min(new Apfloat(2), new Apfloat(1)));
    }
}
