/*
 *	UNICODE/ISO-10646 functions for JOE
 *	Copyright
 *		(C) 1992 Joseph H. Allen
 *
 *	This file is part of JOE (Joe's Own Editor)
 *
 * This is basically the "i18n" file in C
 *
 * There really should be a program to convert from i18n to c, but this
 * was very easy to do with just the text editor.
 */

#include "charmap.h"
#include "utf8.h"
#include "i18n.h"

/*
 * This is an implementation of wcwidth() and wcswidth() (defined in
 * IEEE Std 1002.1-2001) for Unicode.
 *
 * http://www.opengroup.org/onlinepubs/007904975/functions/wcwidth.html
 * http://www.opengroup.org/onlinepubs/007904975/functions/wcswidth.html
 *
 * In fixed-width output devices, Latin characters all occupy a single
 * "cell" position of equal width, whereas ideographic CJK characters
 * occupy two such cells. Interoperability between terminal-line
 * applications and (teletype-style) character terminals using the
 * UTF-8 encoding requires agreement on which character should advance
 * the cursor by how many cell positions. No established formal
 * standards exist at present on which Unicode character shall occupy
 * how many cell positions on character terminals. These routines are
 * a first attempt of defining such behavior based on simple rules
 * applied to data provided by the Unicode Consortium.
 *
 * For some graphical characters, the Unicode standard explicitly
 * defines a character-cell width via the definition of the East Asian
 * FullWidth (F), Wide (W), Half-width (H), and Narrow (Na) classes.
 * In all these cases, there is no ambiguity about which width a
 * terminal shall use. For characters in the East Asian Ambiguous (A)
 * class, the width choice depends purely on a preference of backward
 * compatibility with either historic CJK or Western practice.
 * Choosing single-width for these characters is easy to justify as
 * the appropriate long-term solution, as the CJK practice of
 * displaying these characters as double-width comes from historic
 * implementation simplicity (8-bit encoded characters were displayed
 * single-width and 16-bit ones double-width, even for Greek,
 * Cyrillic, etc.) and not any typographic considerations.
 *
 * Much less clear is the choice of width for the Not East Asian
 * (Neutral) class. Existing practice does not dictate a width for any
 * of these characters. It would nevertheless make sense
 * typographically to allocate two character cells to characters such
 * as for instance EM SPACE or VOLUME INTEGRAL, which cannot be
 * represented adequately with a single-width glyph. The following
 * routines at present merely assign a single-cell width to all
 * neutral characters, in the interest of simplicity. This is not
 * entirely satisfactory and should be reconsidered before
 * establishing a formal standard in this area. At the moment, the
 * decision which Not East Asian (Neutral) characters should be
 * represented by double-width glyphs cannot yet be answered by
 * applying a simple rule from the Unicode database content. Setting
 * up a proper standard for the behavior of UTF-8 character terminals
 * will require a careful analysis not only of each Unicode character,
 * but also of each presentation form, something the author of these
 * routines has avoided to do so far.
 *
 * http://www.unicode.org/unicode/reports/tr11/
 *
 * Markus Kuhn -- 2003-05-20 (Unicode 4.0)
 *
 * Permission to use, copy, modify, and distribute this software
 * for any purpose and without fee is hereby granted. The author
 * disclaims all warranties with regard to this software.
 *
 * Latest version: http://www.cl.cam.ac.uk/~mgk25/ucs/wcwidth.c
 */

struct interval {
	int first;
	int last;
};

static int bisearch(int ucs, const struct interval *table, int max)
{
	int min = 0;
	int mid;

	if (ucs < table[0].first || ucs > table[max].last)
		return -1;
	while (max >= min) {
		mid = (min + max) / 2;
		if (ucs > table[mid].last)
			min = mid + 1;
		else if (ucs < table[mid].first)
			max = mid - 1;
		else
			return mid;
	}

	return -1;
}

/* Macro for generating joe_iswXXX functions */

#define MAKE_ISW(x) \
	int joe_isw##x(struct charmap *foo,int c) \
	{ \
		if (-1!=bisearch(c, data_wctype_##x, sizeof(data_wctype_##x)/sizeof(struct interval) - 1)) \
			return 1; \
		else \
			return 0; \
	}

/* The following two functions define the column width of an ISO 10646
 * character as follows:
 *
 *    - The null character (U+0000) has a column width of 0.
 *
 *    - Other C0/C1 control characters and DEL will lead to a return
 *      value of -1.
 *
 *    - Non-spacing and enclosing combining characters (general
 *      category code Mn or Me in the Unicode database) have a
 *      column width of 0.
 *
 *    - SOFT HYPHEN (U+00AD) has a column width of 1.
 *
 *    - Other format characters (general category code Cf in the Unicode
 *      database) and ZERO WIDTH SPACE (U+200B) have a column width of 0.
 *
 *    - Hangul Jamo medial vowels and final consonants (U+1160-U+11FF)
 *      have a column width of 0.
 *
 *    - Spacing characters in the East Asian Wide (W) or East Asian
 *      Full-width (F) category as defined in Unicode Technical
 *      Report #11 have a column width of 2.
 *
 *    - All remaining characters (including all printable
 *      ISO 8859-1 and WGL4 characters, Unicode control characters,
 *      etc.) have a column width of 1.
 *
 * This implementation assumes that wchar_t characters are encoded
 * in ISO 10646.
 */

/* Modified for JOE: returns printed width of control and other non-printable
   characters */

int joe_wcwidth(int wide,int ucs)
{
	int x;

	/* sorted list of non-overlapping intervals of non-spacing characters */
	/* generated by "uniset +cat=Me +cat=Mn +cat=Cf -00AD +1160-11FF +200B c" */
	static const struct interval combining[] = {
		{ 0x0300, 0x0357 }, { 0x035D, 0x036F }, { 0x0483, 0x0486 },
		{ 0x0488, 0x0489 }, { 0x0591, 0x05A1 }, { 0x05A3, 0x05B9 },
		{ 0x05BB, 0x05BD }, { 0x05BF, 0x05BF }, { 0x05C1, 0x05C2 },
		{ 0x05C4, 0x05C4 }, { 0x0600, 0x0603 }, { 0x0610, 0x0615 },
		{ 0x064B, 0x0658 }, { 0x0670, 0x0670 }, { 0x06D6, 0x06E4 },
		{ 0x06E7, 0x06E8 }, { 0x06EA, 0x06ED }, { 0x070F, 0x070F },
		{ 0x0711, 0x0711 }, { 0x0730, 0x074A }, { 0x07A6, 0x07B0 },
		{ 0x0901, 0x0902 }, { 0x093C, 0x093C }, { 0x0941, 0x0948 },
		{ 0x094D, 0x094D }, { 0x0951, 0x0954 }, { 0x0962, 0x0963 },
		{ 0x0981, 0x0981 }, { 0x09BC, 0x09BC }, { 0x09C1, 0x09C4 },
		{ 0x09CD, 0x09CD }, { 0x09E2, 0x09E3 }, { 0x0A01, 0x0A02 },
		{ 0x0A3C, 0x0A3C }, { 0x0A41, 0x0A42 }, { 0x0A47, 0x0A48 },
		{ 0x0A4B, 0x0A4D }, { 0x0A70, 0x0A71 }, { 0x0A81, 0x0A82 },
		{ 0x0ABC, 0x0ABC }, { 0x0AC1, 0x0AC5 }, { 0x0AC7, 0x0AC8 },
		{ 0x0ACD, 0x0ACD }, { 0x0AE2, 0x0AE3 }, { 0x0B01, 0x0B01 },
		{ 0x0B3C, 0x0B3C }, { 0x0B3F, 0x0B3F }, { 0x0B41, 0x0B43 },
		{ 0x0B4D, 0x0B4D }, { 0x0B56, 0x0B56 }, { 0x0B82, 0x0B82 },
		{ 0x0BC0, 0x0BC0 }, { 0x0BCD, 0x0BCD }, { 0x0C3E, 0x0C40 },
		{ 0x0C46, 0x0C48 }, { 0x0C4A, 0x0C4D }, { 0x0C55, 0x0C56 },
		{ 0x0CBC, 0x0CBC }, { 0x0CBF, 0x0CBF }, { 0x0CC6, 0x0CC6 },
		{ 0x0CCC, 0x0CCD }, { 0x0D41, 0x0D43 }, { 0x0D4D, 0x0D4D },
		{ 0x0DCA, 0x0DCA }, { 0x0DD2, 0x0DD4 }, { 0x0DD6, 0x0DD6 },
		{ 0x0E31, 0x0E31 }, { 0x0E34, 0x0E3A }, { 0x0E47, 0x0E4E },
		{ 0x0EB1, 0x0EB1 }, { 0x0EB4, 0x0EB9 }, { 0x0EBB, 0x0EBC },
		{ 0x0EC8, 0x0ECD }, { 0x0F18, 0x0F19 }, { 0x0F35, 0x0F35 },
		{ 0x0F37, 0x0F37 }, { 0x0F39, 0x0F39 }, { 0x0F71, 0x0F7E },
		{ 0x0F80, 0x0F84 }, { 0x0F86, 0x0F87 }, { 0x0F90, 0x0F97 },
		{ 0x0F99, 0x0FBC }, { 0x0FC6, 0x0FC6 }, { 0x102D, 0x1030 },
		{ 0x1032, 0x1032 }, { 0x1036, 0x1037 }, { 0x1039, 0x1039 },
		{ 0x1058, 0x1059 }, { 0x1160, 0x11FF }, { 0x1712, 0x1714 },
		{ 0x1732, 0x1734 }, { 0x1752, 0x1753 }, { 0x1772, 0x1773 },
		{ 0x17B4, 0x17B5 }, { 0x17B7, 0x17BD }, { 0x17C6, 0x17C6 },
		{ 0x17C9, 0x17D3 }, { 0x17DD, 0x17DD }, { 0x180B, 0x180D },
		{ 0x18A9, 0x18A9 }, { 0x1920, 0x1922 }, { 0x1927, 0x1928 },
		{ 0x1932, 0x1932 }, { 0x1939, 0x193B }, { 0x200B, 0x200F },
		{ 0x202A, 0x202E }, { 0x2060, 0x2063 }, { 0x206A, 0x206F },
		{ 0x20D0, 0x20EA }, { 0x302A, 0x302F }, { 0x3099, 0x309A },
		{ 0xFB1E, 0xFB1E }, { 0xFE00, 0xFE0F }, { 0xFE20, 0xFE23 },
		{ 0xFEFF, 0xFEFF }, { 0xFFF9, 0xFFFB }, { 0x1D167, 0x1D169 },
		{ 0x1D173, 0x1D182 }, { 0x1D185, 0x1D18B }, { 0x1D1AA, 0x1D1AD },
		{ 0xE0001, 0xE0001 }, { 0xE0020, 0xE007F }, { 0xE0100, 0xE01EF }
	};

	/* If terminal is not UTF-8 or file is not UTF-8: width is 1 */
	/* FIXME */
	if (!locale_map->type || !wide)
		return 1;

	/* Control characters are one column wide in JOE */
	if (ucs < 32 || ucs == 0x7F)
		return 1;

	/* More control characters... */
	if (ucs>=0x80 && ucs<=0x9F)
		return 4;

	/* More control characters... */
	if (ucs>=0x200b && ucs<=0x206f) {
		if (ucs<=0x200f) return 6;
		if (ucs>=0x2028 && ucs<=0x202E) return 6;
		if (ucs>=0x2060 && ucs<=0x2063) return 6;
		if (ucs>=0x206a) return 6;
	}

	/* More control characters... */
	if (ucs>=0xFDD0 && ucs<=0xFDEF)
		return 6;

	if (ucs==0xFEFF)
		return 6;

	if (ucs>=0xFFF9 && ucs<=0xFFFB)
		return 6;

	if (ucs>=0xFFFE && ucs<=0xFFFF)
		return 6;

	/* 0 Width Combining characters */
	if (-1!=bisearch(ucs, combining, sizeof(combining) / sizeof(struct interval) - 1))
		return 0;

	/* Double-wide characters */
	return 1 + 
		(ucs >= 0x1100 &&
		(ucs <= 0x115f ||                    /* Hangul Jamo init. consonants */
		ucs == 0x2329 || ucs == 0x232a ||
		(ucs >= 0x2e80 && ucs <= 0xa4cf &&
		ucs != 0x303f) ||                  /* CJK ... Yi */
		(ucs >= 0xac00 && ucs <= 0xd7a3) || /* Hangul Syllables */
		(ucs >= 0xf900 && ucs <= 0xfaff) || /* CJK Compatibility Ideographs */
		(ucs >= 0xfe30 && ucs <= 0xfe6f) || /* CJK Compatibility Forms */
		(ucs >= 0xff00 && ucs <= 0xff60) || /* Fullwidth Forms */
		(ucs >= 0xffe0 && ucs <= 0xffe6) ||
		(ucs >= 0x20000 && ucs <= 0x2fffd) ||
		(ucs >= 0x30000 && ucs <= 0x3fffd)));
}

/* MAKE_ISW functions... */

static struct interval data_wctype_upper[]=
{
	{ 0x0041, 0x005A },
	{ 0x00C0, 0x00D6 },
	{ 0x00D8, 0x00DE },
	{ 0x0100, 0x0136 },
	{ 0x0139, 0x0147 },
	{ 0x014A, 0x0178 },
	{ 0x0179, 0x017D },
	{ 0x0181, 0x0181 },
	{ 0x0182, 0x0186 },
	{ 0x0187, 0x0187 },
	{ 0x0189, 0x018B },
	{ 0x018E, 0x0191 },
	{ 0x0193, 0x0193 },
	{ 0x0194, 0x0194 },
	{ 0x0196, 0x0198 },
	{ 0x019C, 0x019C },
	{ 0x019D, 0x019D },
	{ 0x019F, 0x019F },
	{ 0x01A0, 0x01A4 },
	{ 0x01A6, 0x01A6 },
	{ 0x01A7, 0x01A7 },
	{ 0x01A9, 0x01A9 },
	{ 0x01AC, 0x01AC },
	{ 0x01AE, 0x01AE },
	{ 0x01AF, 0x01AF },
	{ 0x01B1, 0x01B3 },
	{ 0x01B5, 0x01B5 },
	{ 0x01B7, 0x01B7 },
	{ 0x01B8, 0x01B8 },
	{ 0x01BC, 0x01BC },
	{ 0x01C4, 0x01C4 },
	{ 0x01C5, 0x01C5 },
	{ 0x01C7, 0x01C7 },
	{ 0x01C8, 0x01C8 },
	{ 0x01CA, 0x01CA },
	{ 0x01CB, 0x01CB },
	{ 0x01CD, 0x01DB },
	{ 0x01DE, 0x01EE },
	{ 0x01F1, 0x01F1 },
	{ 0x01F2, 0x01F2 },
	{ 0x01F4, 0x01F4 },
	{ 0x01F6, 0x01F8 },
	{ 0x01FA, 0x01FE },
	{ 0x0200, 0x0232 },
	{ 0x0386, 0x0386 },
	{ 0x0388, 0x038A },
	{ 0x038C, 0x038C },
	{ 0x038E, 0x038E },
	{ 0x038F, 0x038F },
	{ 0x0391, 0x03A1 },
	{ 0x03A3, 0x03AB },
	{ 0x03D8, 0x03DE },
	{ 0x03E0, 0x03EE },
	{ 0x03F4, 0x03F4 },
	{ 0x0400, 0x042F },
	{ 0x0460, 0x047E },
	{ 0x0480, 0x0480 },
	{ 0x048A, 0x04BE },
	{ 0x04C1, 0x04CD },
	{ 0x04D0, 0x04F4 },
	{ 0x04F8, 0x04F8 },
	{ 0x0500, 0x050E },
	{ 0x0531, 0x0556 },
	{ 0x1E00, 0x1E7E },
	{ 0x1E80, 0x1E94 },
	{ 0x1EA0, 0x1EF8 },
	{ 0x1F08, 0x1F0F },
	{ 0x1F18, 0x1F1D },
	{ 0x1F28, 0x1F2F },
	{ 0x1F38, 0x1F3F },
	{ 0x1F48, 0x1F4D },
	{ 0x1F59, 0x1F5F },
	{ 0x1F68, 0x1F6F },
	{ 0x1F88, 0x1F8F },
	{ 0x1F98, 0x1F9F },
	{ 0x1FA8, 0x1FAF },
	{ 0x1FB8, 0x1FBC },
	{ 0x1FC8, 0x1FCC },
	{ 0x1FD8, 0x1FDB },
	{ 0x1FE8, 0x1FEC },
	{ 0x1FF8, 0x1FFC },
	{ 0x2126, 0x2126 },
	{ 0x212A, 0x212B },
	{ 0x2160, 0x216F },
	{ 0x24B6, 0x24CF },
	{ 0xFF21, 0xFF3A },
	{ 0x00010400, 0x00010425 }
};

MAKE_ISW(upper)

static struct interval data_wctype_lower[]=
{
	{ 0x0061, 0x007A },
	{ 0x00B5, 0x00B5 },
	{ 0x00DF, 0x00F6 },
	{ 0x00F8, 0x00FF },
	{ 0x0101, 0x0137 },
	{ 0x013A, 0x0148 },
	{ 0x014B, 0x0177 },
	{ 0x017A, 0x017E },
	{ 0x017F, 0x017F },
	{ 0x0183, 0x0183 },
	{ 0x0185, 0x0185 },
	{ 0x0188, 0x0188 },
	{ 0x018C, 0x018C },
	{ 0x0192, 0x0192 },
	{ 0x0195, 0x0195 },
	{ 0x0199, 0x0199 },
	{ 0x019E, 0x019E },
	{ 0x01A1, 0x01A1 },
	{ 0x01A3, 0x01A3 },
	{ 0x01A5, 0x01A5 },
	{ 0x01A8, 0x01A8 },
	{ 0x01AD, 0x01AD },
	{ 0x01B0, 0x01B0 },
	{ 0x01B4, 0x01B4 },
	{ 0x01B6, 0x01B6 },
	{ 0x01B9, 0x01B9 },
	{ 0x01BD, 0x01BD },
	{ 0x01BF, 0x01BF },
	{ 0x01C5, 0x01C5 },
	{ 0x01C6, 0x01C6 },
	{ 0x01C8, 0x01C8 },
	{ 0x01C9, 0x01C9 },
	{ 0x01CB, 0x01CB },
	{ 0x01CC, 0x01DC },
	{ 0x01DD, 0x01EF },
	{ 0x01F2, 0x01F2 },
	{ 0x01F3, 0x01F3 },
	{ 0x01F5, 0x01F5 },
	{ 0x01F9, 0x01FF },
	{ 0x0201, 0x021F },
	{ 0x0223, 0x0233 },
	{ 0x0253, 0x0253 },
	{ 0x0254, 0x0254 },
	{ 0x0256, 0x0256 },
	{ 0x0257, 0x0257 },
	{ 0x0259, 0x0259 },
	{ 0x025B, 0x025B },
	{ 0x0260, 0x0260 },
	{ 0x0263, 0x0263 },
	{ 0x0268, 0x0268 },
	{ 0x0269, 0x0269 },
	{ 0x026F, 0x026F },
	{ 0x0272, 0x0272 },
	{ 0x0275, 0x0275 },
	{ 0x0280, 0x0280 },
	{ 0x0283, 0x0283 },
	{ 0x0288, 0x0288 },
	{ 0x028A, 0x028A },
	{ 0x028B, 0x028B },
	{ 0x0292, 0x0292 },
	{ 0x0345, 0x0345 },
	{ 0x03AC, 0x03AF },
	{ 0x03B1, 0x03CE },
	{ 0x03D0, 0x03D0 },
	{ 0x03D1, 0x03D1 },
	{ 0x03D5, 0x03D5 },
	{ 0x03D6, 0x03D6 },
	{ 0x03D9, 0x03EF },
	{ 0x03F0, 0x03F2 },
	{ 0x03F5, 0x03F5 },
	{ 0x0430, 0x045F },
	{ 0x0461, 0x047F },
	{ 0x0481, 0x0481 },
	{ 0x048B, 0x04BF },
	{ 0x04C2, 0x04CE },
	{ 0x04D1, 0x04F5 },
	{ 0x04F9, 0x04F9 },
	{ 0x0501, 0x050F },
	{ 0x0561, 0x0586 },
	{ 0x1E01, 0x1E95 },
	{ 0x1E9B, 0x1E9B },
	{ 0x1EA1, 0x1EF9 },
	{ 0x1F00, 0x1F07 },
	{ 0x1F10, 0x1F15 },
	{ 0x1F20, 0x1F27 },
	{ 0x1F30, 0x1F37 },
	{ 0x1F40, 0x1F45 },
	{ 0x1F51, 0x1F57 },
	{ 0x1F60, 0x1F67 },
	{ 0x1F70, 0x1F7D },
	{ 0x1F80, 0x1F87 },
	{ 0x1F90, 0x1F97 },
	{ 0x1FA0, 0x1FA7 },
	{ 0x1FB0, 0x1FB0 },
	{ 0x1FB1, 0x1FB1 },
	{ 0x1FB3, 0x1FB3 },
	{ 0x1FBE, 0x1FBE },
	{ 0x1FC3, 0x1FC3 },
	{ 0x1FD0, 0x1FD0 },
	{ 0x1FD1, 0x1FD1 },
	{ 0x1FE0, 0x1FE0 },
	{ 0x1FE1, 0x1FE1 },
	{ 0x1FE5, 0x1FE5 },
	{ 0x1FF3, 0x1FF3 },
	{ 0x2170, 0x217F },
	{ 0x24D0, 0x24E9 },
	{ 0xFF41, 0xFF5A },
	{ 0x00010428, 0x0001044D }
};

MAKE_ISW(lower)

struct interval data_wctype_alpha[]=
{
	{ 0x0041, 0x005A },
	{ 0x005F, 0x005F },	/* Include _ for joe */
	{ 0x0061, 0x007A },
	{ 0x00AA, 0x00AA },
	{ 0x00B5, 0x00B5 },
	{ 0x00BA, 0x00BA },
	{ 0x00C0, 0x00D6 },
	{ 0x00D8, 0x00F6 },
	{ 0x00F8, 0x00FF },
	{ 0x0100, 0x017F },
	{ 0x0180, 0x0220 },
	{ 0x0222, 0x0233 },
	{ 0x0250, 0x02AD },
	{ 0x02B0, 0x02B8 },
	{ 0x02BB, 0x02C1 },
	{ 0x02D0, 0x02D0 },
	{ 0x02D1, 0x02D1 },
	{ 0x02E0, 0x02E4 },
	{ 0x02EE, 0x02EE },
	{ 0x0345, 0x0345 },
	{ 0x037A, 0x037A },
	{ 0x0386, 0x0386 },
	{ 0x0388, 0x038A },
	{ 0x038C, 0x038C },
	{ 0x038E, 0x03A1 },
	{ 0x03A3, 0x03CE },
	{ 0x03D0, 0x03F5 },
	{ 0x0400, 0x0481 },
	{ 0x048A, 0x04CE },
	{ 0x04D0, 0x04F5 },
	{ 0x04F8, 0x04F9 },
	{ 0x0500, 0x050F },
	{ 0x0531, 0x0556 },
	{ 0x0559, 0x0559 },
	{ 0x0561, 0x0587 },
	{ 0x05D0, 0x05EA },
	{ 0x05F0, 0x05F2 },
	{ 0x0621, 0x063A },
	{ 0x0640, 0x064A },
	{ 0x0660, 0x0669 },
	{ 0x066E, 0x066F },
	{ 0x0671, 0x06D3 },
	{ 0x06D5, 0x06D5 },
	{ 0x06E5, 0x06E6 },
	{ 0x06F0, 0x06F9 },
	{ 0x06FA, 0x06FC },
	{ 0x0710, 0x0710 },
	{ 0x0712, 0x072C },
	{ 0x0780, 0x07A5 },
	{ 0x07B1, 0x07B1 },
	{ 0x0905, 0x0939 },
	{ 0x093D, 0x093D },
	{ 0x0950, 0x0950 },
	{ 0x0958, 0x0961 },
	{ 0x0966, 0x096F },
	{ 0x0985, 0x098C },
	{ 0x098F, 0x098F },
	{ 0x0990, 0x0990 },
	{ 0x0993, 0x09A8 },
	{ 0x09AA, 0x09B0 },
	{ 0x09B2, 0x09B2 },
	{ 0x09B6, 0x09B9 },
	{ 0x09DC, 0x09DC },
	{ 0x09DD, 0x09DD },
	{ 0x09DF, 0x09E1 },
	{ 0x09E6, 0x09EF },
	{ 0x09F0, 0x09F1 },
	{ 0x0A05, 0x0A0A },
	{ 0x0A0F, 0x0A0F },
	{ 0x0A10, 0x0A10 },
	{ 0x0A13, 0x0A28 },
	{ 0x0A2A, 0x0A30 },
	{ 0x0A32, 0x0A32 },
	{ 0x0A33, 0x0A33 },
	{ 0x0A35, 0x0A35 },
	{ 0x0A36, 0x0A36 },
	{ 0x0A38, 0x0A38 },
	{ 0x0A39, 0x0A39 },
	{ 0x0A59, 0x0A5C },
	{ 0x0A5E, 0x0A5E },
	{ 0x0A66, 0x0A6F },
	{ 0x0A72, 0x0A74 },
	{ 0x0A85, 0x0A8B },
	{ 0x0A8D, 0x0A8D },
	{ 0x0A8F, 0x0A91 },
	{ 0x0A93, 0x0AA8 },
	{ 0x0AAA, 0x0AB0 },
	{ 0x0AB2, 0x0AB2 },
	{ 0x0AB3, 0x0AB3 },
	{ 0x0AB5, 0x0AB9 },
	{ 0x0ABD, 0x0ABD },
	{ 0x0AD0, 0x0AD0 },
	{ 0x0AE0, 0x0AE0 },
	{ 0x0AE6, 0x0AEF },
	{ 0x0B05, 0x0B0C },
	{ 0x0B0F, 0x0B0F },
	{ 0x0B10, 0x0B10 },
	{ 0x0B13, 0x0B28 },
	{ 0x0B2A, 0x0B30 },
	{ 0x0B32, 0x0B32 },
	{ 0x0B33, 0x0B33 },
	{ 0x0B36, 0x0B39 },
	{ 0x0B3D, 0x0B3D },
	{ 0x0B5C, 0x0B5C },
	{ 0x0B5D, 0x0B5D },
	{ 0x0B5F, 0x0B61 },
	{ 0x0B66, 0x0B6F },
	{ 0x0B83, 0x0B83 },
	{ 0x0B85, 0x0B8A },
	{ 0x0B8E, 0x0B90 },
	{ 0x0B92, 0x0B95 },
	{ 0x0B99, 0x0B99 },
	{ 0x0B9A, 0x0B9A },
	{ 0x0B9C, 0x0B9C },
	{ 0x0B9E, 0x0B9E },
	{ 0x0B9F, 0x0B9F },
	{ 0x0BA3, 0x0BA3 },
	{ 0x0BA4, 0x0BA4 },
	{ 0x0BA8, 0x0BAA },
	{ 0x0BAE, 0x0BB5 },
	{ 0x0BB7, 0x0BB9 },
	{ 0x0BE7, 0x0BEF },
	{ 0x0C05, 0x0C0C },
	{ 0x0C0E, 0x0C10 },
	{ 0x0C12, 0x0C28 },
	{ 0x0C2A, 0x0C33 },
	{ 0x0C35, 0x0C39 },
	{ 0x0C60, 0x0C61 },
	{ 0x0C66, 0x0C6F },
	{ 0x0C85, 0x0C8C },
	{ 0x0C8E, 0x0C90 },
	{ 0x0C92, 0x0CA8 },
	{ 0x0CAA, 0x0CB3 },
	{ 0x0CB5, 0x0CB9 },
	{ 0x0CDE, 0x0CDE },
	{ 0x0CE0, 0x0CE1 },
	{ 0x0CE6, 0x0CEF },
	{ 0x0D05, 0x0D0C },
	{ 0x0D0E, 0x0D10 },
	{ 0x0D12, 0x0D28 },
	{ 0x0D2A, 0x0D39 },
	{ 0x0D60, 0x0D61 },
	{ 0x0D66, 0x0D6F },
	{ 0x0D85, 0x0D96 },
	{ 0x0D9A, 0x0DB1 },
	{ 0x0DB3, 0x0DBB },
	{ 0x0DBD, 0x0DBD },
	{ 0x0DC0, 0x0DC6 },
	{ 0x0E01, 0x0E2E },
	{ 0x0E30, 0x0E3A },
	{ 0x0E40, 0x0E45 },
	{ 0x0E47, 0x0E4E },
	{ 0x0E50, 0x0E59 },
	{ 0x0E81, 0x0E82 },
	{ 0x0E84, 0x0E84 },
	{ 0x0E87, 0x0E88 },
	{ 0x0E8A, 0x0E8A },
	{ 0x0E8D, 0x0E8D },
	{ 0x0E94, 0x0E97 },
	{ 0x0E99, 0x0E9F },
	{ 0x0EA1, 0x0EA3 },
	{ 0x0EA5, 0x0EA5 },
	{ 0x0EA7, 0x0EA7 },
	{ 0x0EAA, 0x0EAB },
	{ 0x0EAD, 0x0EB0 },
	{ 0x0EB2, 0x0EB3 },
	{ 0x0EBD, 0x0EBD },
	{ 0x0EC0, 0x0EC4 },
	{ 0x0EC6, 0x0EC6 },
	{ 0x0ED0, 0x0ED9 },
	{ 0x0EDC, 0x0EDD },
	{ 0x0F00, 0x0F00 },
	{ 0x0F20, 0x0F29 },
	{ 0x0F40, 0x0F47 },
	{ 0x0F49, 0x0F6A },
	{ 0x0F88, 0x0F8B },
	{ 0x1000, 0x1021 },
	{ 0x1023, 0x1027 },
	{ 0x1029, 0x1029 },
	{ 0x102A, 0x102A },
	{ 0x1040, 0x1049 },
	{ 0x1050, 0x1055 },
	{ 0x10A0, 0x10C5 },
	{ 0x10D0, 0x10F8 },
	{ 0x1100, 0x1159 },
	{ 0x115F, 0x11A2 },
	{ 0x11A8, 0x11F9 },
	{ 0x1200, 0x1206 },
	{ 0x1208, 0x1246 },
	{ 0x1248, 0x1248 },
	{ 0x124A, 0x124D },
	{ 0x1250, 0x1256 },
	{ 0x1258, 0x1258 },
	{ 0x125A, 0x125D },
	{ 0x1260, 0x1286 },
	{ 0x1288, 0x1288 },
	{ 0x128A, 0x128D },
	{ 0x1290, 0x12AE },
	{ 0x12B0, 0x12B0 },
	{ 0x12B2, 0x12B5 },
	{ 0x12B8, 0x12BE },
	{ 0x12C0, 0x12C0 },
	{ 0x12C2, 0x12C5 },
	{ 0x12C8, 0x12CE },
	{ 0x12D0, 0x12D6 },
	{ 0x12D8, 0x12EE },
	{ 0x12F0, 0x130E },
	{ 0x1310, 0x1310 },
	{ 0x1312, 0x1315 },
	{ 0x1318, 0x131E },
	{ 0x1320, 0x1346 },
	{ 0x1348, 0x135A },
	{ 0x1369, 0x1371 },
	{ 0x13A0, 0x13F4 },
	{ 0x1401, 0x166C },
	{ 0x166F, 0x1676 },
	{ 0x1681, 0x169A },
	{ 0x16A0, 0x16EA },
	{ 0x16EE, 0x16F0 },
	{ 0x1700, 0x170C },
	{ 0x170E, 0x1711 },
	{ 0x1720, 0x1731 },
	{ 0x1740, 0x1751 },
	{ 0x1760, 0x176C },
	{ 0x176E, 0x1770 },
	{ 0x1780, 0x17B3 },
	{ 0x17D7, 0x17D7 },
	{ 0x17DC, 0x17DC },
	{ 0x17E0, 0x17E9 },
	{ 0x1810, 0x1819 },
	{ 0x1820, 0x1877 },
	{ 0x1880, 0x18A8 },
	{ 0x1E00, 0x1E9B },
	{ 0x1EA0, 0x1EF9 },
	{ 0x1F00, 0x1F15 },
	{ 0x1F18, 0x1F1D },
	{ 0x1F20, 0x1F45 },
	{ 0x1F48, 0x1F4D },
	{ 0x1F50, 0x1F57 },
	{ 0x1F59, 0x1F59 },
	{ 0x1F5B, 0x1F5B },
	{ 0x1F5D, 0x1F5D },
	{ 0x1F5F, 0x1F7D },
	{ 0x1F80, 0x1FB4 },
	{ 0x1FB6, 0x1FBC },
	{ 0x1FBE, 0x1FBE },
	{ 0x1FC2, 0x1FC4 },
	{ 0x1FC6, 0x1FCC },
	{ 0x1FD0, 0x1FD3 },
	{ 0x1FD6, 0x1FDB },
	{ 0x1FE0, 0x1FEC },
	{ 0x1FF2, 0x1FF4 },
	{ 0x1FF6, 0x1FFC },
	{ 0x2071, 0x2071 },
	{ 0x207F, 0x207F },
	{ 0x2102, 0x2102 },
	{ 0x2107, 0x2107 },
	{ 0x210A, 0x2113 },
	{ 0x2115, 0x2115 },
	{ 0x2119, 0x211D },
	{ 0x2124, 0x2124 },
	{ 0x2126, 0x2126 },
	{ 0x2128, 0x212D },
	{ 0x212F, 0x2131 },
	{ 0x2133, 0x2139 },
	{ 0x213D, 0x213F },
	{ 0x2145, 0x2149 },
	{ 0x2160, 0x2183 },
	{ 0x249C, 0x24E9 },
	{ 0x3005, 0x3007 },
	{ 0x3021, 0x3029 },
	{ 0x3031, 0x3035 },
	{ 0x3038, 0x303C },
	{ 0x3041, 0x3096 },
	{ 0x309D, 0x309F },
	{ 0x30A1, 0x30FA },
	{ 0x30FC, 0x30FF },
	{ 0x3105, 0x312C },
	{ 0x3131, 0x318E },
	{ 0x31A0, 0x31B7 },
	{ 0x31F0, 0x31FF },
	{ 0x3400, 0x4DB5 },
	{ 0x4E00, 0x9FA5 },
	{ 0xA000, 0xA48C },
	{ 0xAC00, 0xD7A3 },
	{ 0xF900, 0xFA2D },
	{ 0xFA30, 0xFA6A },
	{ 0xFB00, 0xFB06 },
	{ 0xFB13, 0xFB17 },
	{ 0xFB1D, 0xFB1D },
	{ 0xFB1F, 0xFB28 },
	{ 0xFB2A, 0xFB36 },
	{ 0xFB38, 0xFB3C },
	{ 0xFB3E, 0xFB3E },
	{ 0xFB40, 0xFB40 },
	{ 0xFB41, 0xFB41 },
	{ 0xFB43, 0xFB43 },
	{ 0xFB44, 0xFB44 },
	{ 0xFB46, 0xFB4F },
	{ 0xFB50, 0xFBB1 },
	{ 0xFBD3, 0xFD3D },
	{ 0xFD50, 0xFD8F },
	{ 0xFD92, 0xFDC7 },
	{ 0xFDF0, 0xFDFB },
	{ 0xFE70, 0xFE74 },
	{ 0xFE76, 0xFEFC },
	{ 0xFF10, 0xFF19 },
	{ 0xFF21, 0xFF3A },
	{ 0xFF41, 0xFF5A },
	{ 0xFF66, 0xFFBE },
	{ 0xFFC2, 0xFFC7 },
	{ 0xFFCA, 0xFFCF },
	{ 0xFFD2, 0xFFD7 },
	{ 0xFFDA, 0xFFDC },
	{ 0x00010300, 0x0001031E },
	{ 0x00010330, 0x0001034A },
	{ 0x00010400, 0x00010425 },
	{ 0x00010428, 0x0001044D },
	{ 0x0001D400, 0x0001D454 },
	{ 0x0001D456, 0x0001D49C },
	{ 0x0001D49E, 0x0001D49F },
	{ 0x0001D4A2, 0x0001D4A2 },
	{ 0x0001D4A5, 0x0001D4A6 },
	{ 0x0001D4A9, 0x0001D4AC },
	{ 0x0001D4AE, 0x0001D4B9 },
	{ 0x0001D4BB, 0x0001D4BB },
	{ 0x0001D4BD, 0x0001D4C0 },
	{ 0x0001D4C2, 0x0001D4C3 },
	{ 0x0001D4C5, 0x0001D505 },
	{ 0x0001D507, 0x0001D50A },
	{ 0x0001D50D, 0x0001D514 },
	{ 0x0001D516, 0x0001D51C },
	{ 0x0001D51E, 0x0001D539 },
	{ 0x0001D53B, 0x0001D53E },
	{ 0x0001D540, 0x0001D544 },
	{ 0x0001D546, 0x0001D546 },
	{ 0x0001D54A, 0x0001D550 },
	{ 0x0001D552, 0x0001D6A3 },
	{ 0x0001D6A8, 0x0001D6C0 },
	{ 0x0001D6C2, 0x0001D6DA },
	{ 0x0001D6DC, 0x0001D6FA },
	{ 0x0001D6FC, 0x0001D714 },
	{ 0x0001D716, 0x0001D734 },
	{ 0x0001D736, 0x0001D74E },
	{ 0x0001D750, 0x0001D76E },
	{ 0x0001D770, 0x0001D788 },
	{ 0x0001D78A, 0x0001D7A8 },
	{ 0x0001D7AA, 0x0001D7C2 },
	{ 0x0001D7C4, 0x0001D7C9 },
	{ 0x0001D7CE, 0x0001D7FF },
	{ 0x00020000, 0x0002A6D6 },
	{ 0x0002F800, 0x0002FA1D }
};

MAKE_ISW(alpha)

int joe_iswalpha_(struct charmap *foo,int c)
{
	if (c==0x5F)
		return 1;
	else
		return joe_iswalpha(foo,c);
}

int joe_iswalnum_(struct charmap *foo,int c)
{
	if (c>=0x30 && c<=0x39 || c==0x5F)
		return 1;
	else
		return joe_iswalpha(foo,c);
}

struct interval data_wctype_digit[]=
{
	{ 0x0030, 0x0039 }
};

MAKE_ISW(digit)

struct interval data_wctype_space[]=
{
	{ 0x0009, 0x000D },
	{ 0x0020, 0x0020 },
	{ 0x1680, 0x1680 },
	{ 0x2000, 0x2006 },
	{ 0x2008, 0x200B },
	{ 0x2028, 0x2028 },
	{ 0x2029, 0x2029 },
	{ 0x205F, 0x205F },
	{ 0x3000, 0x3000 }
};

MAKE_ISW(space)

struct interval data_wctype_ctrl[]=
{
	{ 0x0000, 0x001F },
	{ 0x007F, 0x009F },
	{ 0x2028, 0x2028 },
	{ 0x2029, 0x2029 }
};

MAKE_ISW(ctrl)

struct interval data_wctype_punct[]=
{
	{ 0x0021, 0x002F },
	{ 0x003A, 0x0040 },
	{ 0x005B, 0x0060 },
	{ 0x007B, 0x007E },
	{ 0x00A0, 0x00A9 },
	{ 0x00AB, 0x00B4 },
	{ 0x00B6, 0x00B9 },
	{ 0x00BB, 0x00BF },
	{ 0x00D7, 0x00D7 },
	{ 0x00F7, 0x00F7 },
	{ 0x02B9, 0x02BA },
	{ 0x02C2, 0x02CF },
	{ 0x02D2, 0x02DF },
	{ 0x02E5, 0x02ED },
	{ 0x0300, 0x0344 },
	{ 0x0346, 0x034F },
	{ 0x0360, 0x036F },
	{ 0x0374, 0x0375 },
	{ 0x037E, 0x037E },
	{ 0x0384, 0x0385 },
	{ 0x0387, 0x0387 },
	{ 0x03F6, 0x03F6 },
	{ 0x0482, 0x0486 },
	{ 0x0488, 0x0489 },
	{ 0x055A, 0x055F },
	{ 0x0589, 0x058A },
	{ 0x0591, 0x05A1 },
	{ 0x05A3, 0x05B9 },
	{ 0x05BB, 0x05C4 },
	{ 0x05F3, 0x05F4 },
	{ 0x060C, 0x060C },
	{ 0x061B, 0x061B },
	{ 0x061F, 0x061F },
	{ 0x064B, 0x0655 },
	{ 0x066A, 0x066D },
	{ 0x0670, 0x0670 },
	{ 0x06D4, 0x06D4 },
	{ 0x06D6, 0x06E4 },
	{ 0x06E7, 0x06ED },
	{ 0x06FD, 0x06FE },
	{ 0x0700, 0x070D },
	{ 0x070F, 0x070F },
	{ 0x0711, 0x0711 },
	{ 0x0730, 0x074A },
	{ 0x07A6, 0x07B0 },
	{ 0x0901, 0x0903 },
	{ 0x093C, 0x093C },
	{ 0x093E, 0x094D },
	{ 0x0951, 0x0954 },
	{ 0x0962, 0x0965 },
	{ 0x0970, 0x0970 },
	{ 0x0981, 0x0983 },
	{ 0x09BC, 0x09BC },
	{ 0x09BE, 0x09C4 },
	{ 0x09C7, 0x09C8 },
	{ 0x09CB, 0x09CD },
	{ 0x09D7, 0x09D7 },
	{ 0x09E2, 0x09E3 },
	{ 0x09F2, 0x09FA },
	{ 0x0A02, 0x0A02 },
	{ 0x0A3C, 0x0A3C },
	{ 0x0A3E, 0x0A42 },
	{ 0x0A47, 0x0A48 },
	{ 0x0A4B, 0x0A4D },
	{ 0x0A70, 0x0A71 },
	{ 0x0A81, 0x0A83 },
	{ 0x0ABC, 0x0ABC },
	{ 0x0ABE, 0x0AC5 },
	{ 0x0AC7, 0x0AC9 },
	{ 0x0ACB, 0x0ACD },
	{ 0x0B01, 0x0B03 },
	{ 0x0B3C, 0x0B3C },
	{ 0x0B3E, 0x0B43 },
	{ 0x0B47, 0x0B48 },
	{ 0x0B4B, 0x0B4D },
	{ 0x0B56, 0x0B57 },
	{ 0x0B70, 0x0B70 },
	{ 0x0B82, 0x0B82 },
	{ 0x0BBE, 0x0BC2 },
	{ 0x0BC6, 0x0BC8 },
	{ 0x0BCA, 0x0BCD },
	{ 0x0BD7, 0x0BD7 },
	{ 0x0BF0, 0x0BF2 },
	{ 0x0C01, 0x0C03 },
	{ 0x0C3E, 0x0C44 },
	{ 0x0C46, 0x0C48 },
	{ 0x0C4A, 0x0C4D },
	{ 0x0C55, 0x0C56 },
	{ 0x0C82, 0x0C83 },
	{ 0x0CBE, 0x0CC4 },
	{ 0x0CC6, 0x0CC8 },
	{ 0x0CCA, 0x0CCD },
	{ 0x0CD5, 0x0CD6 },
	{ 0x0D02, 0x0D03 },
	{ 0x0D3E, 0x0D43 },
	{ 0x0D46, 0x0D48 },
	{ 0x0D4A, 0x0D4D },
	{ 0x0D57, 0x0D57 },
	{ 0x0D82, 0x0D83 },
	{ 0x0DCA, 0x0DCA },
	{ 0x0DCF, 0x0DD4 },
	{ 0x0DD6, 0x0DD6 },
	{ 0x0DD8, 0x0DDF },
	{ 0x0DF2, 0x0DF4 },
	{ 0x0E2F, 0x0E2F },
	{ 0x0E3F, 0x0E3F },
	{ 0x0E46, 0x0E46 },
	{ 0x0E4F, 0x0E4F },
	{ 0x0E5A, 0x0E5B },
	{ 0x0EB1, 0x0EB1 },
	{ 0x0EB4, 0x0EB9 },
	{ 0x0EBB, 0x0EBC },
	{ 0x0EC8, 0x0ECD },
	{ 0x0F01, 0x0F1F },
	{ 0x0F2A, 0x0F3F },
	{ 0x0F71, 0x0F87 },
	{ 0x0F90, 0x0F97 },
	{ 0x0F99, 0x0FBC },
	{ 0x0FBE, 0x0FCC },
	{ 0x0FCF, 0x0FCF },
	{ 0x102C, 0x1032 },
	{ 0x1036, 0x1039 },
	{ 0x104A, 0x104F },
	{ 0x1056, 0x1059 },
	{ 0x10FB, 0x10FB },
	{ 0x1361, 0x1368 },
	{ 0x1372, 0x137C },
	{ 0x166D, 0x166E },
	{ 0x169B, 0x169C },
	{ 0x16EB, 0x16ED },
	{ 0x1712, 0x1714 },
	{ 0x1732, 0x1736 },
	{ 0x1752, 0x1753 },
	{ 0x1772, 0x1773 },
	{ 0x17B4, 0x17D6 },
	{ 0x17D8, 0x17DB },
	{ 0x1800, 0x180E },
	{ 0x18A9, 0x18A9 },
	{ 0x1FBD, 0x1FBD },
	{ 0x1FBF, 0x1FC1 },
	{ 0x1FCD, 0x1FCF },
	{ 0x1FDD, 0x1FDF },
	{ 0x1FED, 0x1FEF },
	{ 0x1FFD, 0x1FFE },
	{ 0x2007, 0x2007 },
	{ 0x200C, 0x2027 },
	{ 0x202A, 0x2052 },
	{ 0x2057, 0x2057 },
	{ 0x2060, 0x2063 },
	{ 0x206A, 0x2070 },
	{ 0x2074, 0x207E },
	{ 0x2080, 0x208E },
	{ 0x20A0, 0x20B1 },
	{ 0x20D0, 0x20EA },
	{ 0x2100, 0x2101 },
	{ 0x2103, 0x2106 },
	{ 0x2108, 0x2109 },
	{ 0x2114, 0x2114 },
	{ 0x2116, 0x2118 },
	{ 0x211E, 0x2123 },
	{ 0x2125, 0x2125 },
	{ 0x2127, 0x2127 },
	{ 0x212E, 0x212E },
	{ 0x2132, 0x2132 },
	{ 0x213A, 0x213A },
	{ 0x2140, 0x2144 },
	{ 0x214A, 0x214B },
	{ 0x2153, 0x215F },
	{ 0x2190, 0x23CE },
	{ 0x2400, 0x2426 },
	{ 0x2440, 0x244A },
	{ 0x2460, 0x249B },
	{ 0x24EA, 0x24FE },
	{ 0x2500, 0x2613 },
	{ 0x2616, 0x2617 },
	{ 0x2619, 0x267D },
	{ 0x2680, 0x2689 },
	{ 0x2701, 0x2704 },
	{ 0x2706, 0x2709 },
	{ 0x270C, 0x2727 },
	{ 0x2729, 0x274B },
	{ 0x274D, 0x274D },
	{ 0x274F, 0x2752 },
	{ 0x2756, 0x2756 },
	{ 0x2758, 0x275E },
	{ 0x2761, 0x2794 },
	{ 0x2798, 0x27AF },
	{ 0x27B1, 0x27BE },
	{ 0x27D0, 0x27EB },
	{ 0x27F0, 0x2AFF },
	{ 0x2E80, 0x2E99 },
	{ 0x2E9B, 0x2EF3 },
	{ 0x2F00, 0x2FD5 },
	{ 0x2FF0, 0x2FFB },
	{ 0x3001, 0x3004 },
	{ 0x3008, 0x3020 },
	{ 0x302A, 0x3030 },
	{ 0x3036, 0x3037 },
	{ 0x303D, 0x303F },
	{ 0x3099, 0x309C },
	{ 0x30A0, 0x30A0 },
	{ 0x30FB, 0x30FB },
	{ 0x3190, 0x319F },
	{ 0x3200, 0x321C },
	{ 0x3220, 0x3243 },
	{ 0x3251, 0x327B },
	{ 0x327F, 0x32CB },
	{ 0x32D0, 0x32FE },
	{ 0x3300, 0x3376 },
	{ 0x337B, 0x33DD },
	{ 0x33E0, 0x33FE },
	{ 0xA490, 0xA4C6 },
	{ 0xE000, 0xF8FF },
	{ 0xFB1E, 0xFB1E },
	{ 0xFB29, 0xFB29 },
	{ 0xFD3E, 0xFD3F },
	{ 0xFDFC, 0xFDFC },
	{ 0xFE00, 0xFE0F },
	{ 0xFE20, 0xFE23 },
	{ 0xFE30, 0xFE44 },
	{ 0xFE49, 0xFE52 },
	{ 0xFE54, 0xFE66 },
	{ 0xFE68, 0xFE6B },
	{ 0xFEFF, 0xFEFF },
	{ 0xFF01, 0xFF0F },
	{ 0xFF1A, 0xFF20 },
	{ 0xFF3B, 0xFF40 },
	{ 0xFF5B, 0xFF65 },
	{ 0xFFE0, 0xFFE6 },
	{ 0xFFE8, 0xFFEE },
	{ 0xFFF9, 0xFFFD },
	{ 0x00010320, 0x00010323 },
	{ 0x0001D000, 0x0001D0F5 },
	{ 0x0001D100, 0x0001D126 },
	{ 0x0001D12A, 0x0001D1DD },
	{ 0x0001D6C1, 0x0001D6C1 },
	{ 0x0001D6DB, 0x0001D6DB },
	{ 0x0001D6FB, 0x0001D6FB },
	{ 0x0001D715, 0x0001D715 },
	{ 0x0001D735, 0x0001D735 },
	{ 0x0001D74F, 0x0001D74F },
	{ 0x0001D76F, 0x0001D76F },
	{ 0x0001D789, 0x0001D789 },
	{ 0x0001D7A9, 0x0001D7A9 },
	{ 0x0001D7C3, 0x0001D7C3 },
	{ 0x000E0001, 0x000E0001 },
	{ 0x000E0020, 0x000E007F },
	{ 0x000F0000, 0x000FFFFD },
	{ 0x00100000, 0x0010FFFD }
};

MAKE_ISW(punct)

struct interval data_wctype_graph[]=
{
	{ 0x0021, 0x007E },
	{ 0x00A0, 0x0220 },
	{ 0x0222, 0x0233 },
	{ 0x0250, 0x02AD },
	{ 0x02B0, 0x02EE },
	{ 0x0300, 0x034F },
	{ 0x0360, 0x036F },
	{ 0x0374, 0x0375 },
	{ 0x037A, 0x037A },
	{ 0x037E, 0x037E },
	{ 0x0384, 0x038A },
	{ 0x038C, 0x038C },
	{ 0x038E, 0x03A1 },
	{ 0x03A3, 0x03CE },
	{ 0x03D0, 0x03F6 },
	{ 0x0400, 0x0486 },
	{ 0x0488, 0x04CE },
	{ 0x04D0, 0x04F5 },
	{ 0x04F8, 0x04F9 },
	{ 0x0500, 0x050F },
	{ 0x0531, 0x0556 },
	{ 0x0559, 0x055F },
	{ 0x0561, 0x0587 },
	{ 0x0589, 0x058A },
	{ 0x0591, 0x05A1 },
	{ 0x05A3, 0x05B9 },
	{ 0x05BB, 0x05C4 },
	{ 0x05D0, 0x05EA },
	{ 0x05F0, 0x05F4 },
	{ 0x060C, 0x060C },
	{ 0x061B, 0x061B },
	{ 0x061F, 0x061F },
	{ 0x0621, 0x063A },
	{ 0x0640, 0x0655 },
	{ 0x0660, 0x06ED },
	{ 0x06F0, 0x06FE },
	{ 0x0700, 0x070D },
	{ 0x070F, 0x072C },
	{ 0x0730, 0x074A },
	{ 0x0780, 0x07B1 },
	{ 0x0901, 0x0903 },
	{ 0x0905, 0x0939 },
	{ 0x093C, 0x094D },
	{ 0x0950, 0x0954 },
	{ 0x0958, 0x0970 },
	{ 0x0981, 0x0983 },
	{ 0x0985, 0x098C },
	{ 0x098F, 0x0990 },
	{ 0x0993, 0x09A8 },
	{ 0x09AA, 0x09B0 },
	{ 0x09B2, 0x09B2 },
	{ 0x09B6, 0x09B9 },
	{ 0x09BC, 0x09BC },
	{ 0x09BE, 0x09C4 },
	{ 0x09C7, 0x09C8 },
	{ 0x09CB, 0x09CD },
	{ 0x09D7, 0x09D7 },
	{ 0x09DC, 0x09DD },
	{ 0x09DF, 0x09E3 },
	{ 0x09E6, 0x09FA },
	{ 0x0A02, 0x0A02 },
	{ 0x0A05, 0x0A0A },
	{ 0x0A0F, 0x0A10 },
	{ 0x0A13, 0x0A28 },
	{ 0x0A2A, 0x0A30 },
	{ 0x0A32, 0x0A33 },
	{ 0x0A35, 0x0A36 },
	{ 0x0A38, 0x0A39 },
	{ 0x0A3C, 0x0A3C },
	{ 0x0A3E, 0x0A42 },
	{ 0x0A47, 0x0A48 },
	{ 0x0A4B, 0x0A4D },
	{ 0x0A59, 0x0A5C },
	{ 0x0A5E, 0x0A5E },
	{ 0x0A66, 0x0A74 },
	{ 0x0A81, 0x0A83 },
	{ 0x0A85, 0x0A8B },
	{ 0x0A8D, 0x0A8D },
	{ 0x0A8F, 0x0A91 },
	{ 0x0A93, 0x0AA8 },
	{ 0x0AAA, 0x0AB0 },
	{ 0x0AB2, 0x0AB3 },
	{ 0x0AB5, 0x0AB9 },
	{ 0x0ABC, 0x0AC5 },
	{ 0x0AC7, 0x0AC9 },
	{ 0x0ACB, 0x0ACD },
	{ 0x0AD0, 0x0AD0 },
	{ 0x0AE0, 0x0AE0 },
	{ 0x0AE6, 0x0AEF },
	{ 0x0B01, 0x0B03 },
	{ 0x0B05, 0x0B0C },
	{ 0x0B0F, 0x0B10 },
	{ 0x0B13, 0x0B28 },
	{ 0x0B2A, 0x0B30 },
	{ 0x0B32, 0x0B33 },
	{ 0x0B36, 0x0B39 },
	{ 0x0B3C, 0x0B43 },
	{ 0x0B47, 0x0B48 },
	{ 0x0B4B, 0x0B4D },
	{ 0x0B56, 0x0B57 },
	{ 0x0B5C, 0x0B5D },
	{ 0x0B5F, 0x0B61 },
	{ 0x0B66, 0x0B70 },
	{ 0x0B82, 0x0B83 },
	{ 0x0B85, 0x0B8A },
	{ 0x0B8E, 0x0B90 },
	{ 0x0B92, 0x0B95 },
	{ 0x0B99, 0x0B9A },
	{ 0x0B9C, 0x0B9C },
	{ 0x0B9E, 0x0B9F },
	{ 0x0BA3, 0x0BA4 },
	{ 0x0BA8, 0x0BAA },
	{ 0x0BAE, 0x0BB5 },
	{ 0x0BB7, 0x0BB9 },
	{ 0x0BBE, 0x0BC2 },
	{ 0x0BC6, 0x0BC8 },
	{ 0x0BCA, 0x0BCD },
	{ 0x0BD7, 0x0BD7 },
	{ 0x0BE7, 0x0BF2 },
	{ 0x0C01, 0x0C03 },
	{ 0x0C05, 0x0C0C },
	{ 0x0C0E, 0x0C10 },
	{ 0x0C12, 0x0C28 },
	{ 0x0C2A, 0x0C33 },
	{ 0x0C35, 0x0C39 },
	{ 0x0C3E, 0x0C44 },
	{ 0x0C46, 0x0C48 },
	{ 0x0C4A, 0x0C4D },
	{ 0x0C55, 0x0C56 },
	{ 0x0C60, 0x0C61 },
	{ 0x0C66, 0x0C6F },
	{ 0x0C82, 0x0C83 },
	{ 0x0C85, 0x0C8C },
	{ 0x0C8E, 0x0C90 },
	{ 0x0C92, 0x0CA8 },
	{ 0x0CAA, 0x0CB3 },
	{ 0x0CB5, 0x0CB9 },
	{ 0x0CBE, 0x0CC4 },
	{ 0x0CC6, 0x0CC8 },
	{ 0x0CCA, 0x0CCD },
	{ 0x0CD5, 0x0CD6 },
	{ 0x0CDE, 0x0CDE },
	{ 0x0CE0, 0x0CE1 },
	{ 0x0CE6, 0x0CEF },
	{ 0x0D02, 0x0D03 },
	{ 0x0D05, 0x0D0C },
	{ 0x0D0E, 0x0D10 },
	{ 0x0D12, 0x0D28 },
	{ 0x0D2A, 0x0D39 },
	{ 0x0D3E, 0x0D43 },
	{ 0x0D46, 0x0D48 },
	{ 0x0D4A, 0x0D4D },
	{ 0x0D57, 0x0D57 },
	{ 0x0D60, 0x0D61 },
	{ 0x0D66, 0x0D6F },
	{ 0x0D82, 0x0D83 },
	{ 0x0D85, 0x0D96 },
	{ 0x0D9A, 0x0DB1 },
	{ 0x0DB3, 0x0DBB },
	{ 0x0DBD, 0x0DBD },
	{ 0x0DC0, 0x0DC6 },
	{ 0x0DCA, 0x0DCA },
	{ 0x0DCF, 0x0DD4 },
	{ 0x0DD6, 0x0DD6 },
	{ 0x0DD8, 0x0DDF },
	{ 0x0DF2, 0x0DF4 },
	{ 0x0E01, 0x0E3A },
	{ 0x0E3F, 0x0E5B },
	{ 0x0E81, 0x0E82 },
	{ 0x0E84, 0x0E84 },
	{ 0x0E87, 0x0E88 },
	{ 0x0E8A, 0x0E8A },
	{ 0x0E8D, 0x0E8D },
	{ 0x0E94, 0x0E97 },
	{ 0x0E99, 0x0E9F },
	{ 0x0EA1, 0x0EA3 },
	{ 0x0EA5, 0x0EA5 },
	{ 0x0EA7, 0x0EA7 },
	{ 0x0EAA, 0x0EAB },
	{ 0x0EAD, 0x0EB9 },
	{ 0x0EBB, 0x0EBD },
	{ 0x0EC0, 0x0EC4 },
	{ 0x0EC6, 0x0EC6 },
	{ 0x0EC8, 0x0ECD },
	{ 0x0ED0, 0x0ED9 },
	{ 0x0EDC, 0x0EDD },
	{ 0x0F00, 0x0F47 },
	{ 0x0F49, 0x0F6A },
	{ 0x0F71, 0x0F8B },
	{ 0x0F90, 0x0F97 },
	{ 0x0F99, 0x0FBC },
	{ 0x0FBE, 0x0FCC },
	{ 0x0FCF, 0x0FCF },
	{ 0x1000, 0x1021 },
	{ 0x1023, 0x1027 },
	{ 0x1029, 0x102A },
	{ 0x102C, 0x1032 },
	{ 0x1036, 0x1039 },
	{ 0x1040, 0x1059 },
	{ 0x10A0, 0x10C5 },
	{ 0x10D0, 0x10F8 },
	{ 0x10FB, 0x10FB },
	{ 0x1100, 0x1159 },
	{ 0x115F, 0x11A2 },
	{ 0x11A8, 0x11F9 },
	{ 0x1200, 0x1206 },
	{ 0x1208, 0x1246 },
	{ 0x1248, 0x1248 },
	{ 0x124A, 0x124D },
	{ 0x1250, 0x1256 },
	{ 0x1258, 0x1258 },
	{ 0x125A, 0x125D },
	{ 0x1260, 0x1286 },
	{ 0x1288, 0x1288 },
	{ 0x128A, 0x128D },
	{ 0x1290, 0x12AE },
	{ 0x12B0, 0x12B0 },
	{ 0x12B2, 0x12B5 },
	{ 0x12B8, 0x12BE },
	{ 0x12C0, 0x12C0 },
	{ 0x12C2, 0x12C5 },
	{ 0x12C8, 0x12CE },
	{ 0x12D0, 0x12D6 },
	{ 0x12D8, 0x12EE },
	{ 0x12F0, 0x130E },
	{ 0x1310, 0x1310 },
	{ 0x1312, 0x1315 },
	{ 0x1318, 0x131E },
	{ 0x1320, 0x1346 },
	{ 0x1348, 0x135A },
	{ 0x1361, 0x137C },
	{ 0x13A0, 0x13F4 },
	{ 0x1401, 0x1676 },
	{ 0x1681, 0x169C },
	{ 0x16A0, 0x16F0 },
	{ 0x1700, 0x170C },
	{ 0x170E, 0x1714 },
	{ 0x1720, 0x1736 },
	{ 0x1740, 0x1753 },
	{ 0x1760, 0x176C },
	{ 0x176E, 0x1770 },
	{ 0x1772, 0x1773 },
	{ 0x1780, 0x17DC },
	{ 0x17E0, 0x17E9 },
	{ 0x1800, 0x180E },
	{ 0x1810, 0x1819 },
	{ 0x1820, 0x1877 },
	{ 0x1880, 0x18A9 },
	{ 0x1E00, 0x1E9B },
	{ 0x1EA0, 0x1EF9 },
	{ 0x1F00, 0x1F15 },
	{ 0x1F18, 0x1F1D },
	{ 0x1F20, 0x1F45 },
	{ 0x1F48, 0x1F4D },
	{ 0x1F50, 0x1F57 },
	{ 0x1F59, 0x1F59 },
	{ 0x1F5B, 0x1F5B },
	{ 0x1F5D, 0x1F5D },
	{ 0x1F5F, 0x1F7D },
	{ 0x1F80, 0x1FB4 },
	{ 0x1FB6, 0x1FC4 },
	{ 0x1FC6, 0x1FD3 },
	{ 0x1FD6, 0x1FDB },
	{ 0x1FDD, 0x1FEF },
	{ 0x1FF2, 0x1FF4 },
	{ 0x1FF6, 0x1FFE },
	{ 0x2007, 0x2007 },
	{ 0x200C, 0x2027 },
	{ 0x202A, 0x2052 },
	{ 0x2057, 0x2057 },
	{ 0x2060, 0x2063 },
	{ 0x206A, 0x2071 },
	{ 0x2074, 0x208E },
	{ 0x20A0, 0x20B1 },
	{ 0x20D0, 0x20EA },
	{ 0x2100, 0x213A },
	{ 0x213D, 0x214B },
	{ 0x2153, 0x2183 },
	{ 0x2190, 0x23CE },
	{ 0x2400, 0x2426 },
	{ 0x2440, 0x244A },
	{ 0x2460, 0x24FE },
	{ 0x2500, 0x2613 },
	{ 0x2616, 0x2617 },
	{ 0x2619, 0x267D },
	{ 0x2680, 0x2689 },
	{ 0x2701, 0x2704 },
	{ 0x2706, 0x2709 },
	{ 0x270C, 0x2727 },
	{ 0x2729, 0x274B },
	{ 0x274D, 0x274D },
	{ 0x274F, 0x2752 },
	{ 0x2756, 0x2756 },
	{ 0x2758, 0x275E },
	{ 0x2761, 0x2794 },
	{ 0x2798, 0x27AF },
	{ 0x27B1, 0x27BE },
	{ 0x27D0, 0x27EB },
	{ 0x27F0, 0x2AFF },
	{ 0x2E80, 0x2E99 },
	{ 0x2E9B, 0x2EF3 },
	{ 0x2F00, 0x2FD5 },
	{ 0x2FF0, 0x2FFB },
	{ 0x3001, 0x303F },
	{ 0x3041, 0x3096 },
	{ 0x3099, 0x30FF },
	{ 0x3105, 0x312C },
	{ 0x3131, 0x318E },
	{ 0x3190, 0x31B7 },
	{ 0x31F0, 0x321C },
	{ 0x3220, 0x3243 },
	{ 0x3251, 0x327B },
	{ 0x327F, 0x32CB },
	{ 0x32D0, 0x32FE },
	{ 0x3300, 0x3376 },
	{ 0x337B, 0x33DD },
	{ 0x33E0, 0x33FE },
	{ 0x3400, 0x4DB5 },
	{ 0x4E00, 0x9FA5 },
	{ 0xA000, 0xA48C },
	{ 0xA490, 0xA4C6 },
	{ 0xAC00, 0xD7A3 },
	{ 0xE000, 0xFA2D },
	{ 0xFA30, 0xFA6A },
	{ 0xFB00, 0xFB06 },
	{ 0xFB13, 0xFB17 },
	{ 0xFB1D, 0xFB36 },
	{ 0xFB38, 0xFB3C },
	{ 0xFB3E, 0xFB3E },
	{ 0xFB40, 0xFB41 },
	{ 0xFB43, 0xFB44 },
	{ 0xFB46, 0xFBB1 },
	{ 0xFBD3, 0xFD3F },
	{ 0xFD50, 0xFD8F },
	{ 0xFD92, 0xFDC7 },
	{ 0xFDF0, 0xFDFC },
	{ 0xFE00, 0xFE0F },
	{ 0xFE20, 0xFE23 },
	{ 0xFE30, 0xFE46 },
	{ 0xFE49, 0xFE52 },
	{ 0xFE54, 0xFE66 },
	{ 0xFE68, 0xFE6B },
	{ 0xFE70, 0xFE74 },
	{ 0xFE76, 0xFEFC },
	{ 0xFEFF, 0xFEFF },
	{ 0xFF01, 0xFFBE },
	{ 0xFFC2, 0xFFC7 },
	{ 0xFFCA, 0xFFCF },
	{ 0xFFD2, 0xFFD7 },
	{ 0xFFDA, 0xFFDC },
	{ 0xFFE0, 0xFFE6 },
	{ 0xFFE8, 0xFFEE },
	{ 0xFFF9, 0xFFFD },
	{ 0x00010300, 0x0001031E },
	{ 0x00010320, 0x00010323 },
	{ 0x00010330, 0x0001034A },
	{ 0x00010400, 0x00010425 },
	{ 0x00010428, 0x0001044D },
	{ 0x0001D000, 0x0001D0F5 },
	{ 0x0001D100, 0x0001D126 },
	{ 0x0001D12A, 0x0001D1DD },
	{ 0x0001D400, 0x0001D454 },
	{ 0x0001D456, 0x0001D49C },
	{ 0x0001D49E, 0x0001D49F },
	{ 0x0001D4A2, 0x0001D4A2 },
	{ 0x0001D4A5, 0x0001D4A6 },
	{ 0x0001D4A9, 0x0001D4AC },
	{ 0x0001D4AE, 0x0001D4B9 },
	{ 0x0001D4BB, 0x0001D4BB },
	{ 0x0001D4BD, 0x0001D4C0 },
	{ 0x0001D4C2, 0x0001D4C3 },
	{ 0x0001D4C5, 0x0001D505 },
	{ 0x0001D507, 0x0001D50A },
	{ 0x0001D50D, 0x0001D514 },
	{ 0x0001D516, 0x0001D51C },
	{ 0x0001D51E, 0x0001D539 },
	{ 0x0001D53B, 0x0001D53E },
	{ 0x0001D540, 0x0001D544 },
	{ 0x0001D546, 0x0001D546 },
	{ 0x0001D54A, 0x0001D550 },
	{ 0x0001D552, 0x0001D6A3 },
	{ 0x0001D6A8, 0x0001D7C9 },
	{ 0x0001D7CE, 0x0001D7FF },
	{ 0x00020000, 0x0002A6D6 },
	{ 0x0002F800, 0x0002FA1D },
	{ 0x000E0001, 0x000E0001 },
	{ 0x000E0020, 0x000E007F },
	{ 0x000F0000, 0x000FFFFD },
	{ 0x00100000, 0x0010FFFD }
};

MAKE_ISW(graph)

struct interval data_wctype_print[]=
{
	{ 0x0020, 0x007E },
	{ 0x00A0, 0x0220 },
	{ 0x0222, 0x0233 },
	{ 0x0250, 0x02AD },
	{ 0x02B0, 0x02EE },
	{ 0x0300, 0x034F },
	{ 0x0360, 0x036F },
	{ 0x0374, 0x0375 },
	{ 0x037A, 0x037A },
	{ 0x037E, 0x037E },
	{ 0x0384, 0x038A },
	{ 0x038C, 0x038C },
	{ 0x038E, 0x03A1 },
	{ 0x03A3, 0x03CE },
	{ 0x03D0, 0x03F6 },
	{ 0x0400, 0x0486 },
	{ 0x0488, 0x04CE },
	{ 0x04D0, 0x04F5 },
	{ 0x04F8, 0x04F9 },
	{ 0x0500, 0x050F },
	{ 0x0531, 0x0556 },
	{ 0x0559, 0x055F },
	{ 0x0561, 0x0587 },
	{ 0x0589, 0x058A },
	{ 0x0591, 0x05A1 },
	{ 0x05A3, 0x05B9 },
	{ 0x05BB, 0x05C4 },
	{ 0x05D0, 0x05EA },
	{ 0x05F0, 0x05F4 },
	{ 0x060C, 0x060C },
	{ 0x061B, 0x061B },
	{ 0x061F, 0x061F },
	{ 0x0621, 0x063A },
	{ 0x0640, 0x0655 },
	{ 0x0660, 0x06ED },
	{ 0x06F0, 0x06FE },
	{ 0x0700, 0x070D },
	{ 0x070F, 0x072C },
	{ 0x0730, 0x074A },
	{ 0x0780, 0x07B1 },
	{ 0x0901, 0x0903 },
	{ 0x0905, 0x0939 },
	{ 0x093C, 0x094D },
	{ 0x0950, 0x0954 },
	{ 0x0958, 0x0970 },
	{ 0x0981, 0x0983 },
	{ 0x0985, 0x098C },
	{ 0x098F, 0x0990 },
	{ 0x0993, 0x09A8 },
	{ 0x09AA, 0x09B0 },
	{ 0x09B2, 0x09B2 },
	{ 0x09B6, 0x09B9 },
	{ 0x09BC, 0x09BC },
	{ 0x09BE, 0x09C4 },
	{ 0x09C7, 0x09C8 },
	{ 0x09CB, 0x09CD },
	{ 0x09D7, 0x09D7 },
	{ 0x09DC, 0x09DD },
	{ 0x09DF, 0x09E3 },
	{ 0x09E6, 0x09FA },
	{ 0x0A02, 0x0A02 },
	{ 0x0A05, 0x0A0A },
	{ 0x0A0F, 0x0A10 },
	{ 0x0A13, 0x0A28 },
	{ 0x0A2A, 0x0A30 },
	{ 0x0A32, 0x0A33 },
	{ 0x0A35, 0x0A36 },
	{ 0x0A38, 0x0A39 },
	{ 0x0A3C, 0x0A3C },
	{ 0x0A3E, 0x0A42 },
	{ 0x0A47, 0x0A48 },
	{ 0x0A4B, 0x0A4D },
	{ 0x0A59, 0x0A5C },
	{ 0x0A5E, 0x0A5E },
	{ 0x0A66, 0x0A74 },
	{ 0x0A81, 0x0A83 },
	{ 0x0A85, 0x0A8B },
	{ 0x0A8D, 0x0A8D },
	{ 0x0A8F, 0x0A91 },
	{ 0x0A93, 0x0AA8 },
	{ 0x0AAA, 0x0AB0 },
	{ 0x0AB2, 0x0AB3 },
	{ 0x0AB5, 0x0AB9 },
	{ 0x0ABC, 0x0AC5 },
	{ 0x0AC7, 0x0AC9 },
	{ 0x0ACB, 0x0ACD },
	{ 0x0AD0, 0x0AD0 },
	{ 0x0AE0, 0x0AE0 },
	{ 0x0AE6, 0x0AEF },
	{ 0x0B01, 0x0B03 },
	{ 0x0B05, 0x0B0C },
	{ 0x0B0F, 0x0B10 },
	{ 0x0B13, 0x0B28 },
	{ 0x0B2A, 0x0B30 },
	{ 0x0B32, 0x0B33 },
	{ 0x0B36, 0x0B39 },
	{ 0x0B3C, 0x0B43 },
	{ 0x0B47, 0x0B48 },
	{ 0x0B4B, 0x0B4D },
	{ 0x0B56, 0x0B57 },
	{ 0x0B5C, 0x0B5D },
	{ 0x0B5F, 0x0B61 },
	{ 0x0B66, 0x0B70 },
	{ 0x0B82, 0x0B83 },
	{ 0x0B85, 0x0B8A },
	{ 0x0B8E, 0x0B90 },
	{ 0x0B92, 0x0B95 },
	{ 0x0B99, 0x0B9A },
	{ 0x0B9C, 0x0B9C },
	{ 0x0B9E, 0x0B9F },
	{ 0x0BA3, 0x0BA4 },
	{ 0x0BA8, 0x0BAA },
	{ 0x0BAE, 0x0BB5 },
	{ 0x0BB7, 0x0BB9 },
	{ 0x0BBE, 0x0BC2 },
	{ 0x0BC6, 0x0BC8 },
	{ 0x0BCA, 0x0BCD },
	{ 0x0BD7, 0x0BD7 },
	{ 0x0BE7, 0x0BF2 },
	{ 0x0C01, 0x0C03 },
	{ 0x0C05, 0x0C0C },
	{ 0x0C0E, 0x0C10 },
	{ 0x0C12, 0x0C28 },
	{ 0x0C2A, 0x0C33 },
	{ 0x0C35, 0x0C39 },
	{ 0x0C3E, 0x0C44 },
	{ 0x0C46, 0x0C48 },
	{ 0x0C4A, 0x0C4D },
	{ 0x0C55, 0x0C56 },
	{ 0x0C60, 0x0C61 },
	{ 0x0C66, 0x0C6F },
	{ 0x0C82, 0x0C83 },
	{ 0x0C85, 0x0C8C },
	{ 0x0C8E, 0x0C90 },
	{ 0x0C92, 0x0CA8 },
	{ 0x0CAA, 0x0CB3 },
	{ 0x0CB5, 0x0CB9 },
	{ 0x0CBE, 0x0CC4 },
	{ 0x0CC6, 0x0CC8 },
	{ 0x0CCA, 0x0CCD },
	{ 0x0CD5, 0x0CD6 },
	{ 0x0CDE, 0x0CDE },
	{ 0x0CE0, 0x0CE1 },
	{ 0x0CE6, 0x0CEF },
	{ 0x0D02, 0x0D03 },
	{ 0x0D05, 0x0D0C },
	{ 0x0D0E, 0x0D10 },
	{ 0x0D12, 0x0D28 },
	{ 0x0D2A, 0x0D39 },
	{ 0x0D3E, 0x0D43 },
	{ 0x0D46, 0x0D48 },
	{ 0x0D4A, 0x0D4D },
	{ 0x0D57, 0x0D57 },
	{ 0x0D60, 0x0D61 },
	{ 0x0D66, 0x0D6F },
	{ 0x0D82, 0x0D83 },
	{ 0x0D85, 0x0D96 },
	{ 0x0D9A, 0x0DB1 },
	{ 0x0DB3, 0x0DBB },
	{ 0x0DBD, 0x0DBD },
	{ 0x0DC0, 0x0DC6 },
	{ 0x0DCA, 0x0DCA },
	{ 0x0DCF, 0x0DD4 },
	{ 0x0DD6, 0x0DD6 },
	{ 0x0DD8, 0x0DDF },
	{ 0x0DF2, 0x0DF4 },
	{ 0x0E01, 0x0E3A },
	{ 0x0E3F, 0x0E5B },
	{ 0x0E81, 0x0E82 },
	{ 0x0E84, 0x0E84 },
	{ 0x0E87, 0x0E88 },
	{ 0x0E8A, 0x0E8A },
	{ 0x0E8D, 0x0E8D },
	{ 0x0E94, 0x0E97 },
	{ 0x0E99, 0x0E9F },
	{ 0x0EA1, 0x0EA3 },
	{ 0x0EA5, 0x0EA5 },
	{ 0x0EA7, 0x0EA7 },
	{ 0x0EAA, 0x0EAB },
	{ 0x0EAD, 0x0EB9 },
	{ 0x0EBB, 0x0EBD },
	{ 0x0EC0, 0x0EC4 },
	{ 0x0EC6, 0x0EC6 },
	{ 0x0EC8, 0x0ECD },
	{ 0x0ED0, 0x0ED9 },
	{ 0x0EDC, 0x0EDD },
	{ 0x0F00, 0x0F47 },
	{ 0x0F49, 0x0F6A },
	{ 0x0F71, 0x0F8B },
	{ 0x0F90, 0x0F97 },
	{ 0x0F99, 0x0FBC },
	{ 0x0FBE, 0x0FCC },
	{ 0x0FCF, 0x0FCF },
	{ 0x1000, 0x1021 },
	{ 0x1023, 0x1027 },
	{ 0x1029, 0x102A },
	{ 0x102C, 0x1032 },
	{ 0x1036, 0x1039 },
	{ 0x1040, 0x1059 },
	{ 0x10A0, 0x10C5 },
	{ 0x10D0, 0x10F8 },
	{ 0x10FB, 0x10FB },
	{ 0x1100, 0x1159 },
	{ 0x115F, 0x11A2 },
	{ 0x11A8, 0x11F9 },
	{ 0x1200, 0x1206 },
	{ 0x1208, 0x1246 },
	{ 0x1248, 0x1248 },
	{ 0x124A, 0x124D },
	{ 0x1250, 0x1256 },
	{ 0x1258, 0x1258 },
	{ 0x125A, 0x125D },
	{ 0x1260, 0x1286 },
	{ 0x1288, 0x1288 },
	{ 0x128A, 0x128D },
	{ 0x1290, 0x12AE },
	{ 0x12B0, 0x12B0 },
	{ 0x12B2, 0x12B5 },
	{ 0x12B8, 0x12BE },
	{ 0x12C0, 0x12C0 },
	{ 0x12C2, 0x12C5 },
	{ 0x12C8, 0x12CE },
	{ 0x12D0, 0x12D6 },
	{ 0x12D8, 0x12EE },
	{ 0x12F0, 0x130E },
	{ 0x1310, 0x1310 },
	{ 0x1312, 0x1315 },
	{ 0x1318, 0x131E },
	{ 0x1320, 0x1346 },
	{ 0x1348, 0x135A },
	{ 0x1361, 0x137C },
	{ 0x13A0, 0x13F4 },
	{ 0x1401, 0x1676 },
	{ 0x1680, 0x169C },
	{ 0x16A0, 0x16F0 },
	{ 0x1700, 0x170C },
	{ 0x170E, 0x1714 },
	{ 0x1720, 0x1736 },
	{ 0x1740, 0x1753 },
	{ 0x1760, 0x176C },
	{ 0x176E, 0x1770 },
	{ 0x1772, 0x1773 },
	{ 0x1780, 0x17DC },
	{ 0x17E0, 0x17E9 },
	{ 0x1800, 0x180E },
	{ 0x1810, 0x1819 },
	{ 0x1820, 0x1877 },
	{ 0x1880, 0x18A9 },
	{ 0x1E00, 0x1E9B },
	{ 0x1EA0, 0x1EF9 },
	{ 0x1F00, 0x1F15 },
	{ 0x1F18, 0x1F1D },
	{ 0x1F20, 0x1F45 },
	{ 0x1F48, 0x1F4D },
	{ 0x1F50, 0x1F57 },
	{ 0x1F59, 0x1F59 },
	{ 0x1F5B, 0x1F5B },
	{ 0x1F5D, 0x1F5D },
	{ 0x1F5F, 0x1F7D },
	{ 0x1F80, 0x1FB4 },
	{ 0x1FB6, 0x1FC4 },
	{ 0x1FC6, 0x1FD3 },
	{ 0x1FD6, 0x1FDB },
	{ 0x1FDD, 0x1FEF },
	{ 0x1FF2, 0x1FF4 },
	{ 0x1FF6, 0x1FFE },
	{ 0x2000, 0x2027 },
	{ 0x202A, 0x2052 },
	{ 0x2057, 0x2057 },
	{ 0x205F, 0x2063 },
	{ 0x206A, 0x2071 },
	{ 0x2074, 0x208E },
	{ 0x20A0, 0x20B1 },
	{ 0x20D0, 0x20EA },
	{ 0x2100, 0x213A },
	{ 0x213D, 0x214B },
	{ 0x2153, 0x2183 },
	{ 0x2190, 0x23CE },
	{ 0x2400, 0x2426 },
	{ 0x2440, 0x244A },
	{ 0x2460, 0x24FE },
	{ 0x2500, 0x2613 },
	{ 0x2616, 0x2617 },
	{ 0x2619, 0x267D },
	{ 0x2680, 0x2689 },
	{ 0x2701, 0x2704 },
	{ 0x2706, 0x2709 },
	{ 0x270C, 0x2727 },
	{ 0x2729, 0x274B },
	{ 0x274D, 0x274D },
	{ 0x274F, 0x2752 },
	{ 0x2756, 0x2756 },
	{ 0x2758, 0x275E },
	{ 0x2761, 0x2794 },
	{ 0x2798, 0x27AF },
	{ 0x27B1, 0x27BE },
	{ 0x27D0, 0x27EB },
	{ 0x27F0, 0x2AFF },
	{ 0x2E80, 0x2E99 },
	{ 0x2E9B, 0x2EF3 },
	{ 0x2F00, 0x2FD5 },
	{ 0x2FF0, 0x2FFB },
	{ 0x3000, 0x303F },
	{ 0x3041, 0x3096 },
	{ 0x3099, 0x30FF },
	{ 0x3105, 0x312C },
	{ 0x3131, 0x318E },
	{ 0x3190, 0x31B7 },
	{ 0x31F0, 0x321C },
	{ 0x3220, 0x3243 },
	{ 0x3251, 0x327B },
	{ 0x327F, 0x32CB },
	{ 0x32D0, 0x32FE },
	{ 0x3300, 0x3376 },
	{ 0x337B, 0x33DD },
	{ 0x33E0, 0x33FE },
	{ 0x3400, 0x4DB5 },
	{ 0x4E00, 0x9FA5 },
	{ 0xA000, 0xA48C },
	{ 0xA490, 0xA4C6 },
	{ 0xAC00, 0xD7A3 },
	{ 0xE000, 0xFA2D },
	{ 0xFA30, 0xFA6A },
	{ 0xFB00, 0xFB06 },
	{ 0xFB13, 0xFB17 },
	{ 0xFB1D, 0xFB36 },
	{ 0xFB38, 0xFB3C },
	{ 0xFB3E, 0xFB3E },
	{ 0xFB40, 0xFB41 },
	{ 0xFB43, 0xFB44 },
	{ 0xFB46, 0xFBB1 },
	{ 0xFBD3, 0xFD3F },
	{ 0xFD50, 0xFD8F },
	{ 0xFD92, 0xFDC7 },
	{ 0xFDF0, 0xFDFC },
	{ 0xFE00, 0xFE0F },
	{ 0xFE20, 0xFE23 },
	{ 0xFE30, 0xFE46 },
	{ 0xFE49, 0xFE52 },
	{ 0xFE54, 0xFE66 },
	{ 0xFE68, 0xFE6B },
	{ 0xFE70, 0xFE74 },
	{ 0xFE76, 0xFEFC },
	{ 0xFEFF, 0xFEFF },
	{ 0xFF01, 0xFFBE },
	{ 0xFFC2, 0xFFC7 },
	{ 0xFFCA, 0xFFCF },
	{ 0xFFD2, 0xFFD7 },
	{ 0xFFDA, 0xFFDC },
	{ 0xFFE0, 0xFFE6 },
	{ 0xFFE8, 0xFFEE },
	{ 0xFFF9, 0xFFFD },
	{ 0x00010300, 0x0001031E },
	{ 0x00010320, 0x00010323 },
	{ 0x00010330, 0x0001034A },
	{ 0x00010400, 0x00010425 },
	{ 0x00010428, 0x0001044D },
	{ 0x0001D000, 0x0001D0F5 },
	{ 0x0001D100, 0x0001D126 },
	{ 0x0001D12A, 0x0001D1DD },
	{ 0x0001D400, 0x0001D454 },
	{ 0x0001D456, 0x0001D49C },
	{ 0x0001D49E, 0x0001D49F },
	{ 0x0001D4A2, 0x0001D4A2 },
	{ 0x0001D4A5, 0x0001D4A6 },
	{ 0x0001D4A9, 0x0001D4AC },
	{ 0x0001D4AE, 0x0001D4B9 },
	{ 0x0001D4BB, 0x0001D4BB },
	{ 0x0001D4BD, 0x0001D4C0 },
	{ 0x0001D4C2, 0x0001D4C3 },
	{ 0x0001D4C5, 0x0001D505 },
	{ 0x0001D507, 0x0001D50A },
	{ 0x0001D50D, 0x0001D514 },
	{ 0x0001D516, 0x0001D51C },
	{ 0x0001D51E, 0x0001D539 },
	{ 0x0001D53B, 0x0001D53E },
	{ 0x0001D540, 0x0001D544 },
	{ 0x0001D546, 0x0001D546 },
	{ 0x0001D54A, 0x0001D550 },
	{ 0x0001D552, 0x0001D6A3 },
	{ 0x0001D6A8, 0x0001D7C9 },
	{ 0x0001D7CE, 0x0001D7FF },
	{ 0x00020000, 0x0002A6D6 },
	{ 0x0002F800, 0x0002FA1D },
	{ 0x000E0001, 0x000E0001 },
	{ 0x000E0020, 0x000E007F },
	{ 0x000F0000, 0x000FFFFD },
	{ 0x00100000, 0x0010FFFD }
};

MAKE_ISW(print)

struct interval data_wctype_xdigit[]=
{
	{ 0x0030, 0x0039 },
	{ 0x0041, 0x0046 },
	{ 0x0061, 0x0066 }
};

MAKE_ISW(xdigit)

struct interval data_wctype_blank[]=
{
	{ 0x0009, 0x0009 },
	{ 0x0020, 0x0020 },
	{ 0x1680, 0x1680 },
	{ 0x2000, 0x2006 },
	{ 0x2008, 0x200B },
	{ 0x205F, 0x205F },
	{ 0x3000, 0x3000 }
};

MAKE_ISW(blank)

/* Conversion functions */

static struct interval data_wctype_toupper[]=
{
	{ 0x0061, 0x0041 },
	{ 0x0062, 0x0042 },
	{ 0x0063, 0x0043 },
	{ 0x0064, 0x0044 },
	{ 0x0065, 0x0045 },
	{ 0x0066, 0x0046 },
	{ 0x0067, 0x0047 },
	{ 0x0068, 0x0048 },
	{ 0x0069, 0x0049 },
	{ 0x006A, 0x004A },
	{ 0x006B, 0x004B },
	{ 0x006C, 0x004C },
	{ 0x006D, 0x004D },
	{ 0x006E, 0x004E },
	{ 0x006F, 0x004F },
	{ 0x0070, 0x0050 },
	{ 0x0071, 0x0051 },
	{ 0x0072, 0x0052 },
	{ 0x0073, 0x0053 },
	{ 0x0074, 0x0054 },
	{ 0x0075, 0x0055 },
	{ 0x0076, 0x0056 },
	{ 0x0077, 0x0057 },
	{ 0x0078, 0x0058 },
	{ 0x0079, 0x0059 },
	{ 0x007A, 0x005A },
	{ 0x00B5, 0x039C },
	{ 0x00E0, 0x00C0 },
	{ 0x00E1, 0x00C1 },
	{ 0x00E2, 0x00C2 },
	{ 0x00E3, 0x00C3 },
	{ 0x00E4, 0x00C4 },
	{ 0x00E5, 0x00C5 },
	{ 0x00E6, 0x00C6 },
	{ 0x00E7, 0x00C7 },
	{ 0x00E8, 0x00C8 },
	{ 0x00E9, 0x00C9 },
	{ 0x00EA, 0x00CA },
	{ 0x00EB, 0x00CB },
	{ 0x00EC, 0x00CC },
	{ 0x00ED, 0x00CD },
	{ 0x00EE, 0x00CE },
	{ 0x00EF, 0x00CF },
	{ 0x00F0, 0x00D0 },
	{ 0x00F1, 0x00D1 },
	{ 0x00F2, 0x00D2 },
	{ 0x00F3, 0x00D3 },
	{ 0x00F4, 0x00D4 },
	{ 0x00F5, 0x00D5 },
	{ 0x00F6, 0x00D6 },
	{ 0x00F8, 0x00D8 },
	{ 0x00F9, 0x00D9 },
	{ 0x00FA, 0x00DA },
	{ 0x00FB, 0x00DB },
	{ 0x00FC, 0x00DC },
	{ 0x00FD, 0x00DD },
	{ 0x00FE, 0x00DE },
	{ 0x00FF, 0x0178 },
	{ 0x0101, 0x0100 },
	{ 0x0103, 0x0102 },
	{ 0x0105, 0x0104 },
	{ 0x0107, 0x0106 },
	{ 0x0109, 0x0108 },
	{ 0x010B, 0x010A },
	{ 0x010D, 0x010C },
	{ 0x010F, 0x010E },
	{ 0x0111, 0x0110 },
	{ 0x0113, 0x0112 },
	{ 0x0115, 0x0114 },
	{ 0x0117, 0x0116 },
	{ 0x0119, 0x0118 },
	{ 0x011B, 0x011A },
	{ 0x011D, 0x011C },
	{ 0x011F, 0x011E },
	{ 0x0121, 0x0120 },
	{ 0x0123, 0x0122 },
	{ 0x0125, 0x0124 },
	{ 0x0127, 0x0126 },
	{ 0x0129, 0x0128 },
	{ 0x012B, 0x012A },
	{ 0x012D, 0x012C },
	{ 0x012F, 0x012E },
	{ 0x0131, 0x0049 },
	{ 0x0133, 0x0132 },
	{ 0x0135, 0x0134 },
	{ 0x0137, 0x0136 },
	{ 0x013A, 0x0139 },
	{ 0x013C, 0x013B },
	{ 0x013E, 0x013D },
	{ 0x0140, 0x013F },
	{ 0x0142, 0x0141 },
	{ 0x0144, 0x0143 },
	{ 0x0146, 0x0145 },
	{ 0x0148, 0x0147 },
	{ 0x014B, 0x014A },
	{ 0x014D, 0x014C },
	{ 0x014F, 0x014E },
	{ 0x0151, 0x0150 },
	{ 0x0153, 0x0152 },
	{ 0x0155, 0x0154 },
	{ 0x0157, 0x0156 },
	{ 0x0159, 0x0158 },
	{ 0x015B, 0x015A },
	{ 0x015D, 0x015C },
	{ 0x015F, 0x015E },
	{ 0x0161, 0x0160 },
	{ 0x0163, 0x0162 },
	{ 0x0165, 0x0164 },
	{ 0x0167, 0x0166 },
	{ 0x0169, 0x0168 },
	{ 0x016B, 0x016A },
	{ 0x016D, 0x016C },
	{ 0x016F, 0x016E },
	{ 0x0171, 0x0170 },
	{ 0x0173, 0x0172 },
	{ 0x0175, 0x0174 },
	{ 0x0177, 0x0176 },
	{ 0x017A, 0x0179 },
	{ 0x017C, 0x017B },
	{ 0x017E, 0x017D },
	{ 0x017F, 0x0053 },
	{ 0x0183, 0x0182 },
	{ 0x0185, 0x0184 },
	{ 0x0188, 0x0187 },
	{ 0x018C, 0x018B },
	{ 0x0192, 0x0191 },
	{ 0x0195, 0x01F6 },
	{ 0x0199, 0x0198 },
	{ 0x019E, 0x0220 },
	{ 0x01A1, 0x01A0 },
	{ 0x01A3, 0x01A2 },
	{ 0x01A5, 0x01A4 },
	{ 0x01A8, 0x01A7 },
	{ 0x01AD, 0x01AC },
	{ 0x01B0, 0x01AF },
	{ 0x01B4, 0x01B3 },
	{ 0x01B6, 0x01B5 },
	{ 0x01B9, 0x01B8 },
	{ 0x01BD, 0x01BC },
	{ 0x01BF, 0x01F7 },
	{ 0x01C5, 0x01C4 },
	{ 0x01C6, 0x01C4 },
	{ 0x01C8, 0x01C7 },
	{ 0x01C9, 0x01C7 },
	{ 0x01CB, 0x01CA },
	{ 0x01CC, 0x01CA },
	{ 0x01CE, 0x01CD },
	{ 0x01D0, 0x01CF },
	{ 0x01D2, 0x01D1 },
	{ 0x01D4, 0x01D3 },
	{ 0x01D6, 0x01D5 },
	{ 0x01D8, 0x01D7 },
	{ 0x01DA, 0x01D9 },
	{ 0x01DC, 0x01DB },
	{ 0x01DD, 0x018E },
	{ 0x01DF, 0x01DE },
	{ 0x01E1, 0x01E0 },
	{ 0x01E3, 0x01E2 },
	{ 0x01E5, 0x01E4 },
	{ 0x01E7, 0x01E6 },
	{ 0x01E9, 0x01E8 },
	{ 0x01EB, 0x01EA },
	{ 0x01ED, 0x01EC },
	{ 0x01EF, 0x01EE },
	{ 0x01F2, 0x01F1 },
	{ 0x01F3, 0x01F1 },
	{ 0x01F5, 0x01F4 },
	{ 0x01F9, 0x01F8 },
	{ 0x01FB, 0x01FA },
	{ 0x01FD, 0x01FC },
	{ 0x01FF, 0x01FE },
	{ 0x0201, 0x0200 },
	{ 0x0203, 0x0202 },
	{ 0x0205, 0x0204 },
	{ 0x0207, 0x0206 },
	{ 0x0209, 0x0208 },
	{ 0x020B, 0x020A },
	{ 0x020D, 0x020C },
	{ 0x020F, 0x020E },
	{ 0x0211, 0x0210 },
	{ 0x0213, 0x0212 },
	{ 0x0215, 0x0214 },
	{ 0x0217, 0x0216 },
	{ 0x0219, 0x0218 },
	{ 0x021B, 0x021A },
	{ 0x021D, 0x021C },
	{ 0x021F, 0x021E },
	{ 0x0223, 0x0222 },
	{ 0x0225, 0x0224 },
	{ 0x0227, 0x0226 },
	{ 0x0229, 0x0228 },
	{ 0x022B, 0x022A },
	{ 0x022D, 0x022C },
	{ 0x022F, 0x022E },
	{ 0x0231, 0x0230 },
	{ 0x0233, 0x0232 },
	{ 0x0253, 0x0181 },
	{ 0x0254, 0x0186 },
	{ 0x0256, 0x0189 },
	{ 0x0257, 0x018A },
	{ 0x0259, 0x018F },
	{ 0x025B, 0x0190 },
	{ 0x0260, 0x0193 },
	{ 0x0263, 0x0194 },
	{ 0x0268, 0x0197 },
	{ 0x0269, 0x0196 },
	{ 0x026F, 0x019C },
	{ 0x0272, 0x019D },
	{ 0x0275, 0x019F },
	{ 0x0280, 0x01A6 },
	{ 0x0283, 0x01A9 },
	{ 0x0288, 0x01AE },
	{ 0x028A, 0x01B1 },
	{ 0x028B, 0x01B2 },
	{ 0x0292, 0x01B7 },
	{ 0x0345, 0x0399 },
	{ 0x03AC, 0x0386 },
	{ 0x03AD, 0x0388 },
	{ 0x03AE, 0x0389 },
	{ 0x03AF, 0x038A },
	{ 0x03B1, 0x0391 },
	{ 0x03B2, 0x0392 },
	{ 0x03B3, 0x0393 },
	{ 0x03B4, 0x0394 },
	{ 0x03B5, 0x0395 },
	{ 0x03B6, 0x0396 },
	{ 0x03B7, 0x0397 },
	{ 0x03B8, 0x0398 },
	{ 0x03B9, 0x0399 },
	{ 0x03BA, 0x039A },
	{ 0x03BB, 0x039B },
	{ 0x03BC, 0x039C },
	{ 0x03BD, 0x039D },
	{ 0x03BE, 0x039E },
	{ 0x03BF, 0x039F },
	{ 0x03C0, 0x03A0 },
	{ 0x03C1, 0x03A1 },
	{ 0x03C2, 0x03A3 },
	{ 0x03C3, 0x03A3 },
	{ 0x03C4, 0x03A4 },
	{ 0x03C5, 0x03A5 },
	{ 0x03C6, 0x03A6 },
	{ 0x03C7, 0x03A7 },
	{ 0x03C8, 0x03A8 },
	{ 0x03C9, 0x03A9 },
	{ 0x03CA, 0x03AA },
	{ 0x03CB, 0x03AB },
	{ 0x03CC, 0x038C },
	{ 0x03CD, 0x038E },
	{ 0x03CE, 0x038F },
	{ 0x03D0, 0x0392 },
	{ 0x03D1, 0x0398 },
	{ 0x03D5, 0x03A6 },
	{ 0x03D6, 0x03A0 },
	{ 0x03D9, 0x03D8 },
	{ 0x03DB, 0x03DA },
	{ 0x03DD, 0x03DC },
	{ 0x03DF, 0x03DE },
	{ 0x03E1, 0x03E0 },
	{ 0x03E3, 0x03E2 },
	{ 0x03E5, 0x03E4 },
	{ 0x03E7, 0x03E6 },
	{ 0x03E9, 0x03E8 },
	{ 0x03EB, 0x03EA },
	{ 0x03ED, 0x03EC },
	{ 0x03EF, 0x03EE },
	{ 0x03F0, 0x039A },
	{ 0x03F1, 0x03A1 },
	{ 0x03F2, 0x03A3 },
	{ 0x03F5, 0x0395 },
	{ 0x0430, 0x0410 },
	{ 0x0431, 0x0411 },
	{ 0x0432, 0x0412 },
	{ 0x0433, 0x0413 },
	{ 0x0434, 0x0414 },
	{ 0x0435, 0x0415 },
	{ 0x0436, 0x0416 },
	{ 0x0437, 0x0417 },
	{ 0x0438, 0x0418 },
	{ 0x0439, 0x0419 },
	{ 0x043A, 0x041A },
	{ 0x043B, 0x041B },
	{ 0x043C, 0x041C },
	{ 0x043D, 0x041D },
	{ 0x043E, 0x041E },
	{ 0x043F, 0x041F },
	{ 0x0440, 0x0420 },
	{ 0x0441, 0x0421 },
	{ 0x0442, 0x0422 },
	{ 0x0443, 0x0423 },
	{ 0x0444, 0x0424 },
	{ 0x0445, 0x0425 },
	{ 0x0446, 0x0426 },
	{ 0x0447, 0x0427 },
	{ 0x0448, 0x0428 },
	{ 0x0449, 0x0429 },
	{ 0x044A, 0x042A },
	{ 0x044B, 0x042B },
	{ 0x044C, 0x042C },
	{ 0x044D, 0x042D },
	{ 0x044E, 0x042E },
	{ 0x044F, 0x042F },
	{ 0x0450, 0x0400 },
	{ 0x0451, 0x0401 },
	{ 0x0452, 0x0402 },
	{ 0x0453, 0x0403 },
	{ 0x0454, 0x0404 },
	{ 0x0455, 0x0405 },
	{ 0x0456, 0x0406 },
	{ 0x0457, 0x0407 },
	{ 0x0458, 0x0408 },
	{ 0x0459, 0x0409 },
	{ 0x045A, 0x040A },
	{ 0x045B, 0x040B },
	{ 0x045C, 0x040C },
	{ 0x045D, 0x040D },
	{ 0x045E, 0x040E },
	{ 0x045F, 0x040F },
	{ 0x0461, 0x0460 },
	{ 0x0463, 0x0462 },
	{ 0x0465, 0x0464 },
	{ 0x0467, 0x0466 },
	{ 0x0469, 0x0468 },
	{ 0x046B, 0x046A },
	{ 0x046D, 0x046C },
	{ 0x046F, 0x046E },
	{ 0x0471, 0x0470 },
	{ 0x0473, 0x0472 },
	{ 0x0475, 0x0474 },
	{ 0x0477, 0x0476 },
	{ 0x0479, 0x0478 },
	{ 0x047B, 0x047A },
	{ 0x047D, 0x047C },
	{ 0x047F, 0x047E },
	{ 0x0481, 0x0480 },
	{ 0x048B, 0x048A },
	{ 0x048D, 0x048C },
	{ 0x048F, 0x048E },
	{ 0x0491, 0x0490 },
	{ 0x0493, 0x0492 },
	{ 0x0495, 0x0494 },
	{ 0x0497, 0x0496 },
	{ 0x0499, 0x0498 },
	{ 0x049B, 0x049A },
	{ 0x049D, 0x049C },
	{ 0x049F, 0x049E },
	{ 0x04A1, 0x04A0 },
	{ 0x04A3, 0x04A2 },
	{ 0x04A5, 0x04A4 },
	{ 0x04A7, 0x04A6 },
	{ 0x04A9, 0x04A8 },
	{ 0x04AB, 0x04AA },
	{ 0x04AD, 0x04AC },
	{ 0x04AF, 0x04AE },
	{ 0x04B1, 0x04B0 },
	{ 0x04B3, 0x04B2 },
	{ 0x04B5, 0x04B4 },
	{ 0x04B7, 0x04B6 },
	{ 0x04B9, 0x04B8 },
	{ 0x04BB, 0x04BA },
	{ 0x04BD, 0x04BC },
	{ 0x04BF, 0x04BE },
	{ 0x04C2, 0x04C1 },
	{ 0x04C4, 0x04C3 },
	{ 0x04C6, 0x04C5 },
	{ 0x04C8, 0x04C7 },
	{ 0x04CA, 0x04C9 },
	{ 0x04CC, 0x04CB },
	{ 0x04CE, 0x04CD },
	{ 0x04D1, 0x04D0 },
	{ 0x04D3, 0x04D2 },
	{ 0x04D5, 0x04D4 },
	{ 0x04D7, 0x04D6 },
	{ 0x04D9, 0x04D8 },
	{ 0x04DB, 0x04DA },
	{ 0x04DD, 0x04DC },
	{ 0x04DF, 0x04DE },
	{ 0x04E1, 0x04E0 },
	{ 0x04E3, 0x04E2 },
	{ 0x04E5, 0x04E4 },
	{ 0x04E7, 0x04E6 },
	{ 0x04E9, 0x04E8 },
	{ 0x04EB, 0x04EA },
	{ 0x04ED, 0x04EC },
	{ 0x04EF, 0x04EE },
	{ 0x04F1, 0x04F0 },
	{ 0x04F3, 0x04F2 },
	{ 0x04F5, 0x04F4 },
	{ 0x04F9, 0x04F8 },
	{ 0x0501, 0x0500 },
	{ 0x0503, 0x0502 },
	{ 0x0505, 0x0504 },
	{ 0x0507, 0x0506 },
	{ 0x0509, 0x0508 },
	{ 0x050B, 0x050A },
	{ 0x050D, 0x050C },
	{ 0x050F, 0x050E },
	{ 0x0561, 0x0531 },
	{ 0x0562, 0x0532 },
	{ 0x0563, 0x0533 },
	{ 0x0564, 0x0534 },
	{ 0x0565, 0x0535 },
	{ 0x0566, 0x0536 },
	{ 0x0567, 0x0537 },
	{ 0x0568, 0x0538 },
	{ 0x0569, 0x0539 },
	{ 0x056A, 0x053A },
	{ 0x056B, 0x053B },
	{ 0x056C, 0x053C },
	{ 0x056D, 0x053D },
	{ 0x056E, 0x053E },
	{ 0x056F, 0x053F },
	{ 0x0570, 0x0540 },
	{ 0x0571, 0x0541 },
	{ 0x0572, 0x0542 },
	{ 0x0573, 0x0543 },
	{ 0x0574, 0x0544 },
	{ 0x0575, 0x0545 },
	{ 0x0576, 0x0546 },
	{ 0x0577, 0x0547 },
	{ 0x0578, 0x0548 },
	{ 0x0579, 0x0549 },
	{ 0x057A, 0x054A },
	{ 0x057B, 0x054B },
	{ 0x057C, 0x054C },
	{ 0x057D, 0x054D },
	{ 0x057E, 0x054E },
	{ 0x057F, 0x054F },
	{ 0x0580, 0x0550 },
	{ 0x0581, 0x0551 },
	{ 0x0582, 0x0552 },
	{ 0x0583, 0x0553 },
	{ 0x0584, 0x0554 },
	{ 0x0585, 0x0555 },
	{ 0x0586, 0x0556 },
	{ 0x1E01, 0x1E00 },
	{ 0x1E03, 0x1E02 },
	{ 0x1E05, 0x1E04 },
	{ 0x1E07, 0x1E06 },
	{ 0x1E09, 0x1E08 },
	{ 0x1E0B, 0x1E0A },
	{ 0x1E0D, 0x1E0C },
	{ 0x1E0F, 0x1E0E },
	{ 0x1E11, 0x1E10 },
	{ 0x1E13, 0x1E12 },
	{ 0x1E15, 0x1E14 },
	{ 0x1E17, 0x1E16 },
	{ 0x1E19, 0x1E18 },
	{ 0x1E1B, 0x1E1A },
	{ 0x1E1D, 0x1E1C },
	{ 0x1E1F, 0x1E1E },
	{ 0x1E21, 0x1E20 },
	{ 0x1E23, 0x1E22 },
	{ 0x1E25, 0x1E24 },
	{ 0x1E27, 0x1E26 },
	{ 0x1E29, 0x1E28 },
	{ 0x1E2B, 0x1E2A },
	{ 0x1E2D, 0x1E2C },
	{ 0x1E2F, 0x1E2E },
	{ 0x1E31, 0x1E30 },
	{ 0x1E33, 0x1E32 },
	{ 0x1E35, 0x1E34 },
	{ 0x1E37, 0x1E36 },
	{ 0x1E39, 0x1E38 },
	{ 0x1E3B, 0x1E3A },
	{ 0x1E3D, 0x1E3C },
	{ 0x1E3F, 0x1E3E },
	{ 0x1E41, 0x1E40 },
	{ 0x1E43, 0x1E42 },
	{ 0x1E45, 0x1E44 },
	{ 0x1E47, 0x1E46 },
	{ 0x1E49, 0x1E48 },
	{ 0x1E4B, 0x1E4A },
	{ 0x1E4D, 0x1E4C },
	{ 0x1E4F, 0x1E4E },
	{ 0x1E51, 0x1E50 },
	{ 0x1E53, 0x1E52 },
	{ 0x1E55, 0x1E54 },
	{ 0x1E57, 0x1E56 },
	{ 0x1E59, 0x1E58 },
	{ 0x1E5B, 0x1E5A },
	{ 0x1E5D, 0x1E5C },
	{ 0x1E5F, 0x1E5E },
	{ 0x1E61, 0x1E60 },
	{ 0x1E63, 0x1E62 },
	{ 0x1E65, 0x1E64 },
	{ 0x1E67, 0x1E66 },
	{ 0x1E69, 0x1E68 },
	{ 0x1E6B, 0x1E6A },
	{ 0x1E6D, 0x1E6C },
	{ 0x1E6F, 0x1E6E },
	{ 0x1E71, 0x1E70 },
	{ 0x1E73, 0x1E72 },
	{ 0x1E75, 0x1E74 },
	{ 0x1E77, 0x1E76 },
	{ 0x1E79, 0x1E78 },
	{ 0x1E7B, 0x1E7A },
	{ 0x1E7D, 0x1E7C },
	{ 0x1E7F, 0x1E7E },
	{ 0x1E81, 0x1E80 },
	{ 0x1E83, 0x1E82 },
	{ 0x1E85, 0x1E84 },
	{ 0x1E87, 0x1E86 },
	{ 0x1E89, 0x1E88 },
	{ 0x1E8B, 0x1E8A },
	{ 0x1E8D, 0x1E8C },
	{ 0x1E8F, 0x1E8E },
	{ 0x1E91, 0x1E90 },
	{ 0x1E93, 0x1E92 },
	{ 0x1E95, 0x1E94 },
	{ 0x1E9B, 0x1E60 },
	{ 0x1EA1, 0x1EA0 },
	{ 0x1EA3, 0x1EA2 },
	{ 0x1EA5, 0x1EA4 },
	{ 0x1EA7, 0x1EA6 },
	{ 0x1EA9, 0x1EA8 },
	{ 0x1EAB, 0x1EAA },
	{ 0x1EAD, 0x1EAC },
	{ 0x1EAF, 0x1EAE },
	{ 0x1EB1, 0x1EB0 },
	{ 0x1EB3, 0x1EB2 },
	{ 0x1EB5, 0x1EB4 },
	{ 0x1EB7, 0x1EB6 },
	{ 0x1EB9, 0x1EB8 },
	{ 0x1EBB, 0x1EBA },
	{ 0x1EBD, 0x1EBC },
	{ 0x1EBF, 0x1EBE },
	{ 0x1EC1, 0x1EC0 },
	{ 0x1EC3, 0x1EC2 },
	{ 0x1EC5, 0x1EC4 },
	{ 0x1EC7, 0x1EC6 },
	{ 0x1EC9, 0x1EC8 },
	{ 0x1ECB, 0x1ECA },
	{ 0x1ECD, 0x1ECC },
	{ 0x1ECF, 0x1ECE },
	{ 0x1ED1, 0x1ED0 },
	{ 0x1ED3, 0x1ED2 },
	{ 0x1ED5, 0x1ED4 },
	{ 0x1ED7, 0x1ED6 },
	{ 0x1ED9, 0x1ED8 },
	{ 0x1EDB, 0x1EDA },
	{ 0x1EDD, 0x1EDC },
	{ 0x1EDF, 0x1EDE },
	{ 0x1EE1, 0x1EE0 },
	{ 0x1EE3, 0x1EE2 },
	{ 0x1EE5, 0x1EE4 },
	{ 0x1EE7, 0x1EE6 },
	{ 0x1EE9, 0x1EE8 },
	{ 0x1EEB, 0x1EEA },
	{ 0x1EED, 0x1EEC },
	{ 0x1EEF, 0x1EEE },
	{ 0x1EF1, 0x1EF0 },
	{ 0x1EF3, 0x1EF2 },
	{ 0x1EF5, 0x1EF4 },
	{ 0x1EF7, 0x1EF6 },
	{ 0x1EF9, 0x1EF8 },
	{ 0x1F00, 0x1F08 },
	{ 0x1F01, 0x1F09 },
	{ 0x1F02, 0x1F0A },
	{ 0x1F03, 0x1F0B },
	{ 0x1F04, 0x1F0C },
	{ 0x1F05, 0x1F0D },
	{ 0x1F06, 0x1F0E },
	{ 0x1F07, 0x1F0F },
	{ 0x1F10, 0x1F18 },
	{ 0x1F11, 0x1F19 },
	{ 0x1F12, 0x1F1A },
	{ 0x1F13, 0x1F1B },
	{ 0x1F14, 0x1F1C },
	{ 0x1F15, 0x1F1D },
	{ 0x1F20, 0x1F28 },
	{ 0x1F21, 0x1F29 },
	{ 0x1F22, 0x1F2A },
	{ 0x1F23, 0x1F2B },
	{ 0x1F24, 0x1F2C },
	{ 0x1F25, 0x1F2D },
	{ 0x1F26, 0x1F2E },
	{ 0x1F27, 0x1F2F },
	{ 0x1F30, 0x1F38 },
	{ 0x1F31, 0x1F39 },
	{ 0x1F32, 0x1F3A },
	{ 0x1F33, 0x1F3B },
	{ 0x1F34, 0x1F3C },
	{ 0x1F35, 0x1F3D },
	{ 0x1F36, 0x1F3E },
	{ 0x1F37, 0x1F3F },
	{ 0x1F40, 0x1F48 },
	{ 0x1F41, 0x1F49 },
	{ 0x1F42, 0x1F4A },
	{ 0x1F43, 0x1F4B },
	{ 0x1F44, 0x1F4C },
	{ 0x1F45, 0x1F4D },
	{ 0x1F51, 0x1F59 },
	{ 0x1F53, 0x1F5B },
	{ 0x1F55, 0x1F5D },
	{ 0x1F57, 0x1F5F },
	{ 0x1F60, 0x1F68 },
	{ 0x1F61, 0x1F69 },
	{ 0x1F62, 0x1F6A },
	{ 0x1F63, 0x1F6B },
	{ 0x1F64, 0x1F6C },
	{ 0x1F65, 0x1F6D },
	{ 0x1F66, 0x1F6E },
	{ 0x1F67, 0x1F6F },
	{ 0x1F70, 0x1FBA },
	{ 0x1F71, 0x1FBB },
	{ 0x1F72, 0x1FC8 },
	{ 0x1F73, 0x1FC9 },
	{ 0x1F74, 0x1FCA },
	{ 0x1F75, 0x1FCB },
	{ 0x1F76, 0x1FDA },
	{ 0x1F77, 0x1FDB },
	{ 0x1F78, 0x1FF8 },
	{ 0x1F79, 0x1FF9 },
	{ 0x1F7A, 0x1FEA },
	{ 0x1F7B, 0x1FEB },
	{ 0x1F7C, 0x1FFA },
	{ 0x1F7D, 0x1FFB },
	{ 0x1F80, 0x1F88 },
	{ 0x1F81, 0x1F89 },
	{ 0x1F82, 0x1F8A },
	{ 0x1F83, 0x1F8B },
	{ 0x1F84, 0x1F8C },
	{ 0x1F85, 0x1F8D },
	{ 0x1F86, 0x1F8E },
	{ 0x1F87, 0x1F8F },
	{ 0x1F90, 0x1F98 },
	{ 0x1F91, 0x1F99 },
	{ 0x1F92, 0x1F9A },
	{ 0x1F93, 0x1F9B },
	{ 0x1F94, 0x1F9C },
	{ 0x1F95, 0x1F9D },
	{ 0x1F96, 0x1F9E },
	{ 0x1F97, 0x1F9F },
	{ 0x1FA0, 0x1FA8 },
	{ 0x1FA1, 0x1FA9 },
	{ 0x1FA2, 0x1FAA },
	{ 0x1FA3, 0x1FAB },
	{ 0x1FA4, 0x1FAC },
	{ 0x1FA5, 0x1FAD },
	{ 0x1FA6, 0x1FAE },
	{ 0x1FA7, 0x1FAF },
	{ 0x1FB0, 0x1FB8 },
	{ 0x1FB1, 0x1FB9 },
	{ 0x1FB3, 0x1FBC },
	{ 0x1FBE, 0x0399 },
	{ 0x1FC3, 0x1FCC },
	{ 0x1FD0, 0x1FD8 },
	{ 0x1FD1, 0x1FD9 },
	{ 0x1FE0, 0x1FE8 },
	{ 0x1FE1, 0x1FE9 },
	{ 0x1FE5, 0x1FEC },
	{ 0x1FF3, 0x1FFC },
	{ 0x2170, 0x2160 },
	{ 0x2171, 0x2161 },
	{ 0x2172, 0x2162 },
	{ 0x2173, 0x2163 },
	{ 0x2174, 0x2164 },
	{ 0x2175, 0x2165 },
	{ 0x2176, 0x2166 },
	{ 0x2177, 0x2167 },
	{ 0x2178, 0x2168 },
	{ 0x2179, 0x2169 },
	{ 0x217A, 0x216A },
	{ 0x217B, 0x216B },
	{ 0x217C, 0x216C },
	{ 0x217D, 0x216D },
	{ 0x217E, 0x216E },
	{ 0x217F, 0x216F },
	{ 0x24D0, 0x24B6 },
	{ 0x24D1, 0x24B7 },
	{ 0x24D2, 0x24B8 },
	{ 0x24D3, 0x24B9 },
	{ 0x24D4, 0x24BA },
	{ 0x24D5, 0x24BB },
	{ 0x24D6, 0x24BC },
	{ 0x24D7, 0x24BD },
	{ 0x24D8, 0x24BE },
	{ 0x24D9, 0x24BF },
	{ 0x24DA, 0x24C0 },
	{ 0x24DB, 0x24C1 },
	{ 0x24DC, 0x24C2 },
	{ 0x24DD, 0x24C3 },
	{ 0x24DE, 0x24C4 },
	{ 0x24DF, 0x24C5 },
	{ 0x24E0, 0x24C6 },
	{ 0x24E1, 0x24C7 },
	{ 0x24E2, 0x24C8 },
	{ 0x24E3, 0x24C9 },
	{ 0x24E4, 0x24CA },
	{ 0x24E5, 0x24CB },
	{ 0x24E6, 0x24CC },
	{ 0x24E7, 0x24CD },
	{ 0x24E8, 0x24CE },
	{ 0x24E9, 0x24CF },
	{ 0xFF41, 0xFF21 },
	{ 0xFF42, 0xFF22 },
	{ 0xFF43, 0xFF23 },
	{ 0xFF44, 0xFF24 },
	{ 0xFF45, 0xFF25 },
	{ 0xFF46, 0xFF26 },
	{ 0xFF47, 0xFF27 },
	{ 0xFF48, 0xFF28 },
	{ 0xFF49, 0xFF29 },
	{ 0xFF4A, 0xFF2A },
	{ 0xFF4B, 0xFF2B },
	{ 0xFF4C, 0xFF2C },
	{ 0xFF4D, 0xFF2D },
	{ 0xFF4E, 0xFF2E },
	{ 0xFF4F, 0xFF2F },
	{ 0xFF50, 0xFF30 },
	{ 0xFF51, 0xFF31 },
	{ 0xFF52, 0xFF32 },
	{ 0xFF53, 0xFF33 },
	{ 0xFF54, 0xFF34 },
	{ 0xFF55, 0xFF35 },
	{ 0xFF56, 0xFF36 },
	{ 0xFF57, 0xFF37 },
	{ 0xFF58, 0xFF38 },
	{ 0xFF59, 0xFF39 },
	{ 0xFF5A, 0xFF3A },
	{ 0x00010428, 0x00010400 },
	{ 0x00010429, 0x00010401 },
	{ 0x0001042A, 0x00010402 },
	{ 0x0001042B, 0x00010403 },
	{ 0x0001042C, 0x00010404 },
	{ 0x0001042D, 0x00010405 },
	{ 0x0001042E, 0x00010406 },
	{ 0x0001042F, 0x00010407 },
	{ 0x00010430, 0x00010408 },
	{ 0x00010431, 0x00010409 },
	{ 0x00010432, 0x0001040A },
	{ 0x00010433, 0x0001040B },
	{ 0x00010434, 0x0001040C },
	{ 0x00010435, 0x0001040D },
	{ 0x00010436, 0x0001040E },
	{ 0x00010437, 0x0001040F },
	{ 0x00010438, 0x00010410 },
	{ 0x00010439, 0x00010411 },
	{ 0x0001043A, 0x00010412 },
	{ 0x0001043B, 0x00010413 },
	{ 0x0001043C, 0x00010414 },
	{ 0x0001043D, 0x00010415 },
	{ 0x0001043E, 0x00010416 },
	{ 0x0001043F, 0x00010417 },
	{ 0x00010440, 0x00010418 },
	{ 0x00010441, 0x00010419 },
	{ 0x00010442, 0x0001041A },
	{ 0x00010443, 0x0001041B },
	{ 0x00010444, 0x0001041C },
	{ 0x00010445, 0x0001041D },
	{ 0x00010446, 0x0001041E },
	{ 0x00010447, 0x0001041F },
	{ 0x00010448, 0x00010420 },
	{ 0x00010449, 0x00010421 },
	{ 0x0001044A, 0x00010422 },
	{ 0x0001044B, 0x00010423 },
	{ 0x0001044C, 0x00010424 },
	{ 0x0001044D, 0x00010425 }
};

static struct interval *data_wctype_toupper_i;
static int toupper_i_size;
static int *toupper_cvt;

int joe_towupper(struct charmap *foo,int c)
{
	int idx;

	if (c>=0x61 && c<=0x7A)
		return c+0x41-0x61;

	else if(c<128)
		return c;

	if (!data_wctype_toupper_i) {
		int x;
		int y = -1;
		data_wctype_toupper_i = (struct interval *)malloc(sizeof(data_wctype_toupper));
		toupper_cvt = (int *)malloc(sizeof(data_wctype_toupper)/sizeof(struct interval)*sizeof(int));

		for (x=0;x!=sizeof(data_wctype_toupper)/sizeof(struct interval);++x) {
			if (y == -1 || data_wctype_toupper_i[y].first + 1 != data_wctype_toupper[x].first ||
			    toupper_cvt[y] != data_wctype_toupper[x].last - data_wctype_toupper[x].first) {
				++y;
				data_wctype_toupper_i[y].first = data_wctype_toupper[x].first;
				data_wctype_toupper_i[y].last = data_wctype_toupper[x].first;
				toupper_cvt[y] = data_wctype_toupper[x].last - data_wctype_toupper[x].first;
			} else
				++data_wctype_toupper_i[y].last;
		}
		toupper_i_size = y;
	}

	idx = bisearch(c, data_wctype_toupper_i, toupper_i_size);

	if (idx==-1)
		return c;
	else
		return c+toupper_cvt[idx];
}

static struct interval data_wctype_tolower[]=
{
	{ 0x0041, 0x0061 },
	{ 0x0042, 0x0062 },
	{ 0x0043, 0x0063 },
	{ 0x0044, 0x0064 },
	{ 0x0045, 0x0065 },
	{ 0x0046, 0x0066 },
	{ 0x0047, 0x0067 },
	{ 0x0048, 0x0068 },
	{ 0x0049, 0x0069 },
	{ 0x004A, 0x006A },
	{ 0x004B, 0x006B },
	{ 0x004C, 0x006C },
	{ 0x004D, 0x006D },
	{ 0x004E, 0x006E },
	{ 0x004F, 0x006F },
	{ 0x0050, 0x0070 },
	{ 0x0051, 0x0071 },
	{ 0x0052, 0x0072 },
	{ 0x0053, 0x0073 },
	{ 0x0054, 0x0074 },
	{ 0x0055, 0x0075 },
	{ 0x0056, 0x0076 },
	{ 0x0057, 0x0077 },
	{ 0x0058, 0x0078 },
	{ 0x0059, 0x0079 },
	{ 0x005A, 0x007A },
	{ 0x00C0, 0x00E0 },
	{ 0x00C1, 0x00E1 },
	{ 0x00C2, 0x00E2 },
	{ 0x00C3, 0x00E3 },
	{ 0x00C4, 0x00E4 },
	{ 0x00C5, 0x00E5 },
	{ 0x00C6, 0x00E6 },
	{ 0x00C7, 0x00E7 },
	{ 0x00C8, 0x00E8 },
	{ 0x00C9, 0x00E9 },
	{ 0x00CA, 0x00EA },
	{ 0x00CB, 0x00EB },
	{ 0x00CC, 0x00EC },
	{ 0x00CD, 0x00ED },
	{ 0x00CE, 0x00EE },
	{ 0x00CF, 0x00EF },
	{ 0x00D0, 0x00F0 },
	{ 0x00D1, 0x00F1 },
	{ 0x00D2, 0x00F2 },
	{ 0x00D3, 0x00F3 },
	{ 0x00D4, 0x00F4 },
	{ 0x00D5, 0x00F5 },
	{ 0x00D6, 0x00F6 },
	{ 0x00D8, 0x00F8 },
	{ 0x00D9, 0x00F9 },
	{ 0x00DA, 0x00FA },
	{ 0x00DB, 0x00FB },
	{ 0x00DC, 0x00FC },
	{ 0x00DD, 0x00FD },
	{ 0x00DE, 0x00FE },
	{ 0x0100, 0x0101 },
	{ 0x0102, 0x0103 },
	{ 0x0104, 0x0105 },
	{ 0x0106, 0x0107 },
	{ 0x0108, 0x0109 },
	{ 0x010A, 0x010B },
	{ 0x010C, 0x010D },
	{ 0x010E, 0x010F },
	{ 0x0110, 0x0111 },
	{ 0x0112, 0x0113 },
	{ 0x0114, 0x0115 },
	{ 0x0116, 0x0117 },
	{ 0x0118, 0x0119 },
	{ 0x011A, 0x011B },
	{ 0x011C, 0x011D },
	{ 0x011E, 0x011F },
	{ 0x0120, 0x0121 },
	{ 0x0122, 0x0123 },
	{ 0x0124, 0x0125 },
	{ 0x0126, 0x0127 },
	{ 0x0128, 0x0129 },
	{ 0x012A, 0x012B },
	{ 0x012C, 0x012D },
	{ 0x012E, 0x012F },
	{ 0x0130, 0x0069 },
	{ 0x0132, 0x0133 },
	{ 0x0134, 0x0135 },
	{ 0x0136, 0x0137 },
	{ 0x0139, 0x013A },
	{ 0x013B, 0x013C },
	{ 0x013D, 0x013E },
	{ 0x013F, 0x0140 },
	{ 0x0141, 0x0142 },
	{ 0x0143, 0x0144 },
	{ 0x0145, 0x0146 },
	{ 0x0147, 0x0148 },
	{ 0x014A, 0x014B },
	{ 0x014C, 0x014D },
	{ 0x014E, 0x014F },
	{ 0x0150, 0x0151 },
	{ 0x0152, 0x0153 },
	{ 0x0154, 0x0155 },
	{ 0x0156, 0x0157 },
	{ 0x0158, 0x0159 },
	{ 0x015A, 0x015B },
	{ 0x015C, 0x015D },
	{ 0x015E, 0x015F },
	{ 0x0160, 0x0161 },
	{ 0x0162, 0x0163 },
	{ 0x0164, 0x0165 },
	{ 0x0166, 0x0167 },
	{ 0x0168, 0x0169 },
	{ 0x016A, 0x016B },
	{ 0x016C, 0x016D },
	{ 0x016E, 0x016F },
	{ 0x0170, 0x0171 },
	{ 0x0172, 0x0173 },
	{ 0x0174, 0x0175 },
	{ 0x0176, 0x0177 },
	{ 0x0178, 0x00FF },
	{ 0x0179, 0x017A },
	{ 0x017B, 0x017C },
	{ 0x017D, 0x017E },
	{ 0x0181, 0x0253 },
	{ 0x0182, 0x0183 },
	{ 0x0184, 0x0185 },
	{ 0x0186, 0x0254 },
	{ 0x0187, 0x0188 },
	{ 0x0189, 0x0256 },
	{ 0x018A, 0x0257 },
	{ 0x018B, 0x018C },
	{ 0x018E, 0x01DD },
	{ 0x018F, 0x0259 },
	{ 0x0190, 0x025B },
	{ 0x0191, 0x0192 },
	{ 0x0193, 0x0260 },
	{ 0x0194, 0x0263 },
	{ 0x0196, 0x0269 },
	{ 0x0197, 0x0268 },
	{ 0x0198, 0x0199 },
	{ 0x019C, 0x026F },
	{ 0x019D, 0x0272 },
	{ 0x019F, 0x0275 },
	{ 0x01A0, 0x01A1 },
	{ 0x01A2, 0x01A3 },
	{ 0x01A4, 0x01A5 },
	{ 0x01A6, 0x0280 },
	{ 0x01A7, 0x01A8 },
	{ 0x01A9, 0x0283 },
	{ 0x01AC, 0x01AD },
	{ 0x01AE, 0x0288 },
	{ 0x01AF, 0x01B0 },
	{ 0x01B1, 0x028A },
	{ 0x01B2, 0x028B },
	{ 0x01B3, 0x01B4 },
	{ 0x01B5, 0x01B6 },
	{ 0x01B7, 0x0292 },
	{ 0x01B8, 0x01B9 },
	{ 0x01BC, 0x01BD },
	{ 0x01C4, 0x01C6 },
	{ 0x01C5, 0x01C6 },
	{ 0x01C7, 0x01C9 },
	{ 0x01C8, 0x01C9 },
	{ 0x01CA, 0x01CC },
	{ 0x01CB, 0x01CC },
	{ 0x01CD, 0x01CE },
	{ 0x01CF, 0x01D0 },
	{ 0x01D1, 0x01D2 },
	{ 0x01D3, 0x01D4 },
	{ 0x01D5, 0x01D6 },
	{ 0x01D7, 0x01D8 },
	{ 0x01D9, 0x01DA },
	{ 0x01DB, 0x01DC },
	{ 0x01DE, 0x01DF },
	{ 0x01E0, 0x01E1 },
	{ 0x01E2, 0x01E3 },
	{ 0x01E4, 0x01E5 },
	{ 0x01E6, 0x01E7 },
	{ 0x01E8, 0x01E9 },
	{ 0x01EA, 0x01EB },
	{ 0x01EC, 0x01ED },
	{ 0x01EE, 0x01EF },
	{ 0x01F1, 0x01F3 },
	{ 0x01F2, 0x01F3 },
	{ 0x01F4, 0x01F5 },
	{ 0x01F6, 0x0195 },
	{ 0x01F7, 0x01BF },
	{ 0x01F8, 0x01F9 },
	{ 0x01FA, 0x01FB },
	{ 0x01FC, 0x01FD },
	{ 0x01FE, 0x01FF },
	{ 0x0200, 0x0201 },
	{ 0x0202, 0x0203 },
	{ 0x0204, 0x0205 },
	{ 0x0206, 0x0207 },
	{ 0x0208, 0x0209 },
	{ 0x020A, 0x020B },
	{ 0x020C, 0x020D },
	{ 0x020E, 0x020F },
	{ 0x0210, 0x0211 },
	{ 0x0212, 0x0213 },
	{ 0x0214, 0x0215 },
	{ 0x0216, 0x0217 },
	{ 0x0218, 0x0219 },
	{ 0x021A, 0x021B },
	{ 0x021C, 0x021D },
	{ 0x021E, 0x021F },
	{ 0x0220, 0x019E },
	{ 0x0222, 0x0223 },
	{ 0x0224, 0x0225 },
	{ 0x0226, 0x0227 },
	{ 0x0228, 0x0229 },
	{ 0x022A, 0x022B },
	{ 0x022C, 0x022D },
	{ 0x022E, 0x022F },
	{ 0x0230, 0x0231 },
	{ 0x0232, 0x0233 },
	{ 0x0386, 0x03AC },
	{ 0x0388, 0x03AD },
	{ 0x0389, 0x03AE },
	{ 0x038A, 0x03AF },
	{ 0x038C, 0x03CC },
	{ 0x038E, 0x03CD },
	{ 0x038F, 0x03CE },
	{ 0x0391, 0x03B1 },
	{ 0x0392, 0x03B2 },
	{ 0x0393, 0x03B3 },
	{ 0x0394, 0x03B4 },
	{ 0x0395, 0x03B5 },
	{ 0x0396, 0x03B6 },
	{ 0x0397, 0x03B7 },
	{ 0x0398, 0x03B8 },
	{ 0x0399, 0x03B9 },
	{ 0x039A, 0x03BA },
	{ 0x039B, 0x03BB },
	{ 0x039C, 0x03BC },
	{ 0x039D, 0x03BD },
	{ 0x039E, 0x03BE },
	{ 0x039F, 0x03BF },
	{ 0x03A0, 0x03C0 },
	{ 0x03A1, 0x03C1 },
	{ 0x03A3, 0x03C3 },
	{ 0x03A4, 0x03C4 },
	{ 0x03A5, 0x03C5 },
	{ 0x03A6, 0x03C6 },
	{ 0x03A7, 0x03C7 },
	{ 0x03A8, 0x03C8 },
	{ 0x03A9, 0x03C9 },
	{ 0x03AA, 0x03CA },
	{ 0x03AB, 0x03CB },
	{ 0x03D8, 0x03D9 },
	{ 0x03DA, 0x03DB },
	{ 0x03DC, 0x03DD },
	{ 0x03DE, 0x03DF },
	{ 0x03E0, 0x03E1 },
	{ 0x03E2, 0x03E3 },
	{ 0x03E4, 0x03E5 },
	{ 0x03E6, 0x03E7 },
	{ 0x03E8, 0x03E9 },
	{ 0x03EA, 0x03EB },
	{ 0x03EC, 0x03ED },
	{ 0x03EE, 0x03EF },
	{ 0x03F4, 0x03B8 },
	{ 0x0400, 0x0450 },
	{ 0x0401, 0x0451 },
	{ 0x0402, 0x0452 },
	{ 0x0403, 0x0453 },
	{ 0x0404, 0x0454 },
	{ 0x0405, 0x0455 },
	{ 0x0406, 0x0456 },
	{ 0x0407, 0x0457 },
	{ 0x0408, 0x0458 },
	{ 0x0409, 0x0459 },
	{ 0x040A, 0x045A },
	{ 0x040B, 0x045B },
	{ 0x040C, 0x045C },
	{ 0x040D, 0x045D },
	{ 0x040E, 0x045E },
	{ 0x040F, 0x045F },
	{ 0x0410, 0x0430 },
	{ 0x0411, 0x0431 },
	{ 0x0412, 0x0432 },
	{ 0x0413, 0x0433 },
	{ 0x0414, 0x0434 },
	{ 0x0415, 0x0435 },
	{ 0x0416, 0x0436 },
	{ 0x0417, 0x0437 },
	{ 0x0418, 0x0438 },
	{ 0x0419, 0x0439 },
	{ 0x041A, 0x043A },
	{ 0x041B, 0x043B },
	{ 0x041C, 0x043C },
	{ 0x041D, 0x043D },
	{ 0x041E, 0x043E },
	{ 0x041F, 0x043F },
	{ 0x0420, 0x0440 },
	{ 0x0421, 0x0441 },
	{ 0x0422, 0x0442 },
	{ 0x0423, 0x0443 },
	{ 0x0424, 0x0444 },
	{ 0x0425, 0x0445 },
	{ 0x0426, 0x0446 },
	{ 0x0427, 0x0447 },
	{ 0x0428, 0x0448 },
	{ 0x0429, 0x0449 },
	{ 0x042A, 0x044A },
	{ 0x042B, 0x044B },
	{ 0x042C, 0x044C },
	{ 0x042D, 0x044D },
	{ 0x042E, 0x044E },
	{ 0x042F, 0x044F },
	{ 0x0460, 0x0461 },
	{ 0x0462, 0x0463 },
	{ 0x0464, 0x0465 },
	{ 0x0466, 0x0467 },
	{ 0x0468, 0x0469 },
	{ 0x046A, 0x046B },
	{ 0x046C, 0x046D },
	{ 0x046E, 0x046F },
	{ 0x0470, 0x0471 },
	{ 0x0472, 0x0473 },
	{ 0x0474, 0x0475 },
	{ 0x0476, 0x0477 },
	{ 0x0478, 0x0479 },
	{ 0x047A, 0x047B },
	{ 0x047C, 0x047D },
	{ 0x047E, 0x047F },
	{ 0x0480, 0x0481 },
	{ 0x048A, 0x048B },
	{ 0x048C, 0x048D },
	{ 0x048E, 0x048F },
	{ 0x0490, 0x0491 },
	{ 0x0492, 0x0493 },
	{ 0x0494, 0x0495 },
	{ 0x0496, 0x0497 },
	{ 0x0498, 0x0499 },
	{ 0x049A, 0x049B },
	{ 0x049C, 0x049D },
	{ 0x049E, 0x049F },
	{ 0x04A0, 0x04A1 },
	{ 0x04A2, 0x04A3 },
	{ 0x04A4, 0x04A5 },
	{ 0x04A6, 0x04A7 },
	{ 0x04A8, 0x04A9 },
	{ 0x04AA, 0x04AB },
	{ 0x04AC, 0x04AD },
	{ 0x04AE, 0x04AF },
	{ 0x04B0, 0x04B1 },
	{ 0x04B2, 0x04B3 },
	{ 0x04B4, 0x04B5 },
	{ 0x04B6, 0x04B7 },
	{ 0x04B8, 0x04B9 },
	{ 0x04BA, 0x04BB },
	{ 0x04BC, 0x04BD },
	{ 0x04BE, 0x04BF },
	{ 0x04C1, 0x04C2 },
	{ 0x04C3, 0x04C4 },
	{ 0x04C5, 0x04C6 },
	{ 0x04C7, 0x04C8 },
	{ 0x04C9, 0x04CA },
	{ 0x04CB, 0x04CC },
	{ 0x04CD, 0x04CE },
	{ 0x04D0, 0x04D1 },
	{ 0x04D2, 0x04D3 },
	{ 0x04D4, 0x04D5 },
	{ 0x04D6, 0x04D7 },
	{ 0x04D8, 0x04D9 },
	{ 0x04DA, 0x04DB },
	{ 0x04DC, 0x04DD },
	{ 0x04DE, 0x04DF },
	{ 0x04E0, 0x04E1 },
	{ 0x04E2, 0x04E3 },
	{ 0x04E4, 0x04E5 },
	{ 0x04E6, 0x04E7 },
	{ 0x04E8, 0x04E9 },
	{ 0x04EA, 0x04EB },
	{ 0x04EC, 0x04ED },
	{ 0x04EE, 0x04EF },
	{ 0x04F0, 0x04F1 },
	{ 0x04F2, 0x04F3 },
	{ 0x04F4, 0x04F5 },
	{ 0x04F8, 0x04F9 },
	{ 0x0500, 0x0501 },
	{ 0x0502, 0x0503 },
	{ 0x0504, 0x0505 },
	{ 0x0506, 0x0507 },
	{ 0x0508, 0x0509 },
	{ 0x050A, 0x050B },
	{ 0x050C, 0x050D },
	{ 0x050E, 0x050F },
	{ 0x0531, 0x0561 },
	{ 0x0532, 0x0562 },
	{ 0x0533, 0x0563 },
	{ 0x0534, 0x0564 },
	{ 0x0535, 0x0565 },
	{ 0x0536, 0x0566 },
	{ 0x0537, 0x0567 },
	{ 0x0538, 0x0568 },
	{ 0x0539, 0x0569 },
	{ 0x053A, 0x056A },
	{ 0x053B, 0x056B },
	{ 0x053C, 0x056C },
	{ 0x053D, 0x056D },
	{ 0x053E, 0x056E },
	{ 0x053F, 0x056F },
	{ 0x0540, 0x0570 },
	{ 0x0541, 0x0571 },
	{ 0x0542, 0x0572 },
	{ 0x0543, 0x0573 },
	{ 0x0544, 0x0574 },
	{ 0x0545, 0x0575 },
	{ 0x0546, 0x0576 },
	{ 0x0547, 0x0577 },
	{ 0x0548, 0x0578 },
	{ 0x0549, 0x0579 },
	{ 0x054A, 0x057A },
	{ 0x054B, 0x057B },
	{ 0x054C, 0x057C },
	{ 0x054D, 0x057D },
	{ 0x054E, 0x057E },
	{ 0x054F, 0x057F },
	{ 0x0550, 0x0580 },
	{ 0x0551, 0x0581 },
	{ 0x0552, 0x0582 },
	{ 0x0553, 0x0583 },
	{ 0x0554, 0x0584 },
	{ 0x0555, 0x0585 },
	{ 0x0556, 0x0586 },
	{ 0x1E00, 0x1E01 },
	{ 0x1E02, 0x1E03 },
	{ 0x1E04, 0x1E05 },
	{ 0x1E06, 0x1E07 },
	{ 0x1E08, 0x1E09 },
	{ 0x1E0A, 0x1E0B },
	{ 0x1E0C, 0x1E0D },
	{ 0x1E0E, 0x1E0F },
	{ 0x1E10, 0x1E11 },
	{ 0x1E12, 0x1E13 },
	{ 0x1E14, 0x1E15 },
	{ 0x1E16, 0x1E17 },
	{ 0x1E18, 0x1E19 },
	{ 0x1E1A, 0x1E1B },
	{ 0x1E1C, 0x1E1D },
	{ 0x1E1E, 0x1E1F },
	{ 0x1E20, 0x1E21 },
	{ 0x1E22, 0x1E23 },
	{ 0x1E24, 0x1E25 },
	{ 0x1E26, 0x1E27 },
	{ 0x1E28, 0x1E29 },
	{ 0x1E2A, 0x1E2B },
	{ 0x1E2C, 0x1E2D },
	{ 0x1E2E, 0x1E2F },
	{ 0x1E30, 0x1E31 },
	{ 0x1E32, 0x1E33 },
	{ 0x1E34, 0x1E35 },
	{ 0x1E36, 0x1E37 },
	{ 0x1E38, 0x1E39 },
	{ 0x1E3A, 0x1E3B },
	{ 0x1E3C, 0x1E3D },
	{ 0x1E3E, 0x1E3F },
	{ 0x1E40, 0x1E41 },
	{ 0x1E42, 0x1E43 },
	{ 0x1E44, 0x1E45 },
	{ 0x1E46, 0x1E47 },
	{ 0x1E48, 0x1E49 },
	{ 0x1E4A, 0x1E4B },
	{ 0x1E4C, 0x1E4D },
	{ 0x1E4E, 0x1E4F },
	{ 0x1E50, 0x1E51 },
	{ 0x1E52, 0x1E53 },
	{ 0x1E54, 0x1E55 },
	{ 0x1E56, 0x1E57 },
	{ 0x1E58, 0x1E59 },
	{ 0x1E5A, 0x1E5B },
	{ 0x1E5C, 0x1E5D },
	{ 0x1E5E, 0x1E5F },
	{ 0x1E60, 0x1E61 },
	{ 0x1E62, 0x1E63 },
	{ 0x1E64, 0x1E65 },
	{ 0x1E66, 0x1E67 },
	{ 0x1E68, 0x1E69 },
	{ 0x1E6A, 0x1E6B },
	{ 0x1E6C, 0x1E6D },
	{ 0x1E6E, 0x1E6F },
	{ 0x1E70, 0x1E71 },
	{ 0x1E72, 0x1E73 },
	{ 0x1E74, 0x1E75 },
	{ 0x1E76, 0x1E77 },
	{ 0x1E78, 0x1E79 },
	{ 0x1E7A, 0x1E7B },
	{ 0x1E7C, 0x1E7D },
	{ 0x1E7E, 0x1E7F },
	{ 0x1E80, 0x1E81 },
	{ 0x1E82, 0x1E83 },
	{ 0x1E84, 0x1E85 },
	{ 0x1E86, 0x1E87 },
	{ 0x1E88, 0x1E89 },
	{ 0x1E8A, 0x1E8B },
	{ 0x1E8C, 0x1E8D },
	{ 0x1E8E, 0x1E8F },
	{ 0x1E90, 0x1E91 },
	{ 0x1E92, 0x1E93 },
	{ 0x1E94, 0x1E95 },
	{ 0x1EA0, 0x1EA1 },
	{ 0x1EA2, 0x1EA3 },
	{ 0x1EA4, 0x1EA5 },
	{ 0x1EA6, 0x1EA7 },
	{ 0x1EA8, 0x1EA9 },
	{ 0x1EAA, 0x1EAB },
	{ 0x1EAC, 0x1EAD },
	{ 0x1EAE, 0x1EAF },
	{ 0x1EB0, 0x1EB1 },
	{ 0x1EB2, 0x1EB3 },
	{ 0x1EB4, 0x1EB5 },
	{ 0x1EB6, 0x1EB7 },
	{ 0x1EB8, 0x1EB9 },
	{ 0x1EBA, 0x1EBB },
	{ 0x1EBC, 0x1EBD },
	{ 0x1EBE, 0x1EBF },
	{ 0x1EC0, 0x1EC1 },
	{ 0x1EC2, 0x1EC3 },
	{ 0x1EC4, 0x1EC5 },
	{ 0x1EC6, 0x1EC7 },
	{ 0x1EC8, 0x1EC9 },
	{ 0x1ECA, 0x1ECB },
	{ 0x1ECC, 0x1ECD },
	{ 0x1ECE, 0x1ECF },
	{ 0x1ED0, 0x1ED1 },
	{ 0x1ED2, 0x1ED3 },
	{ 0x1ED4, 0x1ED5 },
	{ 0x1ED6, 0x1ED7 },
	{ 0x1ED8, 0x1ED9 },
	{ 0x1EDA, 0x1EDB },
	{ 0x1EDC, 0x1EDD },
	{ 0x1EDE, 0x1EDF },
	{ 0x1EE0, 0x1EE1 },
	{ 0x1EE2, 0x1EE3 },
	{ 0x1EE4, 0x1EE5 },
	{ 0x1EE6, 0x1EE7 },
	{ 0x1EE8, 0x1EE9 },
	{ 0x1EEA, 0x1EEB },
	{ 0x1EEC, 0x1EED },
	{ 0x1EEE, 0x1EEF },
	{ 0x1EF0, 0x1EF1 },
	{ 0x1EF2, 0x1EF3 },
	{ 0x1EF4, 0x1EF5 },
	{ 0x1EF6, 0x1EF7 },
	{ 0x1EF8, 0x1EF9 },
	{ 0x1F08, 0x1F00 },
	{ 0x1F09, 0x1F01 },
	{ 0x1F0A, 0x1F02 },
	{ 0x1F0B, 0x1F03 },
	{ 0x1F0C, 0x1F04 },
	{ 0x1F0D, 0x1F05 },
	{ 0x1F0E, 0x1F06 },
	{ 0x1F0F, 0x1F07 },
	{ 0x1F18, 0x1F10 },
	{ 0x1F19, 0x1F11 },
	{ 0x1F1A, 0x1F12 },
	{ 0x1F1B, 0x1F13 },
	{ 0x1F1C, 0x1F14 },
	{ 0x1F1D, 0x1F15 },
	{ 0x1F28, 0x1F20 },
	{ 0x1F29, 0x1F21 },
	{ 0x1F2A, 0x1F22 },
	{ 0x1F2B, 0x1F23 },
	{ 0x1F2C, 0x1F24 },
	{ 0x1F2D, 0x1F25 },
	{ 0x1F2E, 0x1F26 },
	{ 0x1F2F, 0x1F27 },
	{ 0x1F38, 0x1F30 },
	{ 0x1F39, 0x1F31 },
	{ 0x1F3A, 0x1F32 },
	{ 0x1F3B, 0x1F33 },
	{ 0x1F3C, 0x1F34 },
	{ 0x1F3D, 0x1F35 },
	{ 0x1F3E, 0x1F36 },
	{ 0x1F3F, 0x1F37 },
	{ 0x1F48, 0x1F40 },
	{ 0x1F49, 0x1F41 },
	{ 0x1F4A, 0x1F42 },
	{ 0x1F4B, 0x1F43 },
	{ 0x1F4C, 0x1F44 },
	{ 0x1F4D, 0x1F45 },
	{ 0x1F59, 0x1F51 },
	{ 0x1F5B, 0x1F53 },
	{ 0x1F5D, 0x1F55 },
	{ 0x1F5F, 0x1F57 },
	{ 0x1F68, 0x1F60 },
	{ 0x1F69, 0x1F61 },
	{ 0x1F6A, 0x1F62 },
	{ 0x1F6B, 0x1F63 },
	{ 0x1F6C, 0x1F64 },
	{ 0x1F6D, 0x1F65 },
	{ 0x1F6E, 0x1F66 },
	{ 0x1F6F, 0x1F67 },
	{ 0x1F88, 0x1F80 },
	{ 0x1F89, 0x1F81 },
	{ 0x1F8A, 0x1F82 },
	{ 0x1F8B, 0x1F83 },
	{ 0x1F8C, 0x1F84 },
	{ 0x1F8D, 0x1F85 },
	{ 0x1F8E, 0x1F86 },
	{ 0x1F8F, 0x1F87 },
	{ 0x1F98, 0x1F90 },
	{ 0x1F99, 0x1F91 },
	{ 0x1F9A, 0x1F92 },
	{ 0x1F9B, 0x1F93 },
	{ 0x1F9C, 0x1F94 },
	{ 0x1F9D, 0x1F95 },
	{ 0x1F9E, 0x1F96 },
	{ 0x1F9F, 0x1F97 },
	{ 0x1FA8, 0x1FA0 },
	{ 0x1FA9, 0x1FA1 },
	{ 0x1FAA, 0x1FA2 },
	{ 0x1FAB, 0x1FA3 },
	{ 0x1FAC, 0x1FA4 },
	{ 0x1FAD, 0x1FA5 },
	{ 0x1FAE, 0x1FA6 },
	{ 0x1FAF, 0x1FA7 },
	{ 0x1FB8, 0x1FB0 },
	{ 0x1FB9, 0x1FB1 },
	{ 0x1FBA, 0x1F70 },
	{ 0x1FBB, 0x1F71 },
	{ 0x1FBC, 0x1FB3 },
	{ 0x1FC8, 0x1F72 },
	{ 0x1FC9, 0x1F73 },
	{ 0x1FCA, 0x1F74 },
	{ 0x1FCB, 0x1F75 },
	{ 0x1FCC, 0x1FC3 },
	{ 0x1FD8, 0x1FD0 },
	{ 0x1FD9, 0x1FD1 },
	{ 0x1FDA, 0x1F76 },
	{ 0x1FDB, 0x1F77 },
	{ 0x1FE8, 0x1FE0 },
	{ 0x1FE9, 0x1FE1 },
	{ 0x1FEA, 0x1F7A },
	{ 0x1FEB, 0x1F7B },
	{ 0x1FEC, 0x1FE5 },
	{ 0x1FF8, 0x1F78 },
	{ 0x1FF9, 0x1F79 },
	{ 0x1FFA, 0x1F7C },
	{ 0x1FFB, 0x1F7D },
	{ 0x1FFC, 0x1FF3 },
	{ 0x2126, 0x03C9 },
	{ 0x212A, 0x006B },
	{ 0x212B, 0x00E5 },
	{ 0x2160, 0x2170 },
	{ 0x2161, 0x2171 },
	{ 0x2162, 0x2172 },
	{ 0x2163, 0x2173 },
	{ 0x2164, 0x2174 },
	{ 0x2165, 0x2175 },
	{ 0x2166, 0x2176 },
	{ 0x2167, 0x2177 },
	{ 0x2168, 0x2178 },
	{ 0x2169, 0x2179 },
	{ 0x216A, 0x217A },
	{ 0x216B, 0x217B },
	{ 0x216C, 0x217C },
	{ 0x216D, 0x217D },
	{ 0x216E, 0x217E },
	{ 0x216F, 0x217F },
	{ 0x24B6, 0x24D0 },
	{ 0x24B7, 0x24D1 },
	{ 0x24B8, 0x24D2 },
	{ 0x24B9, 0x24D3 },
	{ 0x24BA, 0x24D4 },
	{ 0x24BB, 0x24D5 },
	{ 0x24BC, 0x24D6 },
	{ 0x24BD, 0x24D7 },
	{ 0x24BE, 0x24D8 },
	{ 0x24BF, 0x24D9 },
	{ 0x24C0, 0x24DA },
	{ 0x24C1, 0x24DB },
	{ 0x24C2, 0x24DC },
	{ 0x24C3, 0x24DD },
	{ 0x24C4, 0x24DE },
	{ 0x24C5, 0x24DF },
	{ 0x24C6, 0x24E0 },
	{ 0x24C7, 0x24E1 },
	{ 0x24C8, 0x24E2 },
	{ 0x24C9, 0x24E3 },
	{ 0x24CA, 0x24E4 },
	{ 0x24CB, 0x24E5 },
	{ 0x24CC, 0x24E6 },
	{ 0x24CD, 0x24E7 },
	{ 0x24CE, 0x24E8 },
	{ 0x24CF, 0x24E9 },
	{ 0xFF21, 0xFF41 },
	{ 0xFF22, 0xFF42 },
	{ 0xFF23, 0xFF43 },
	{ 0xFF24, 0xFF44 },
	{ 0xFF25, 0xFF45 },
	{ 0xFF26, 0xFF46 },
	{ 0xFF27, 0xFF47 },
	{ 0xFF28, 0xFF48 },
	{ 0xFF29, 0xFF49 },
	{ 0xFF2A, 0xFF4A },
	{ 0xFF2B, 0xFF4B },
	{ 0xFF2C, 0xFF4C },
	{ 0xFF2D, 0xFF4D },
	{ 0xFF2E, 0xFF4E },
	{ 0xFF2F, 0xFF4F },
	{ 0xFF30, 0xFF50 },
	{ 0xFF31, 0xFF51 },
	{ 0xFF32, 0xFF52 },
	{ 0xFF33, 0xFF53 },
	{ 0xFF34, 0xFF54 },
	{ 0xFF35, 0xFF55 },
	{ 0xFF36, 0xFF56 },
	{ 0xFF37, 0xFF57 },
	{ 0xFF38, 0xFF58 },
	{ 0xFF39, 0xFF59 },
	{ 0xFF3A, 0xFF5A },
	{ 0x00010400, 0x00010428 },
	{ 0x00010401, 0x00010429 },
	{ 0x00010402, 0x0001042A },
	{ 0x00010403, 0x0001042B },
	{ 0x00010404, 0x0001042C },
	{ 0x00010405, 0x0001042D },
	{ 0x00010406, 0x0001042E },
	{ 0x00010407, 0x0001042F },
	{ 0x00010408, 0x00010430 },
	{ 0x00010409, 0x00010431 },
	{ 0x0001040A, 0x00010432 },
	{ 0x0001040B, 0x00010433 },
	{ 0x0001040C, 0x00010434 },
	{ 0x0001040D, 0x00010435 },
	{ 0x0001040E, 0x00010436 },
	{ 0x0001040F, 0x00010437 },
	{ 0x00010410, 0x00010438 },
	{ 0x00010411, 0x00010439 },
	{ 0x00010412, 0x0001043A },
	{ 0x00010413, 0x0001043B },
	{ 0x00010414, 0x0001043C },
	{ 0x00010415, 0x0001043D },
	{ 0x00010416, 0x0001043E },
	{ 0x00010417, 0x0001043F },
	{ 0x00010418, 0x00010440 },
	{ 0x00010419, 0x00010441 },
	{ 0x0001041A, 0x00010442 },
	{ 0x0001041B, 0x00010443 },
	{ 0x0001041C, 0x00010444 },
	{ 0x0001041D, 0x00010445 },
	{ 0x0001041E, 0x00010446 },
	{ 0x0001041F, 0x00010447 },
	{ 0x00010420, 0x00010448 },
	{ 0x00010421, 0x00010449 },
	{ 0x00010422, 0x0001044A },
	{ 0x00010423, 0x0001044B },
	{ 0x00010424, 0x0001044C },
	{ 0x00010425, 0x0001044D }
};

static struct interval *data_wctype_tolower_i;
static int tolower_i_size;
static int *tolower_cvt;

int joe_towlower(struct charmap *foo,int c)
{
	int idx;

	if (c>=0x41 && c<=0x5A)
		return c + 0x61 - 0x41;

	else if(c<128)
		return c;

	if (!data_wctype_tolower_i) {
		int x;
		int y = -1;
		data_wctype_tolower_i = (struct interval *)malloc(sizeof(data_wctype_tolower));
		tolower_cvt = (int *)malloc(sizeof(data_wctype_tolower)/sizeof(struct interval)*sizeof(int));

		for (x=0;x!=sizeof(data_wctype_tolower)/sizeof(struct interval);++x) {
			if (y == -1 || data_wctype_tolower_i[y].last + 1 != data_wctype_tolower[x].first ||
			    tolower_cvt[y] != data_wctype_tolower[x].last - data_wctype_tolower[x].first) {
				++y;
				data_wctype_tolower_i[y].first = data_wctype_tolower[x].first;
				data_wctype_tolower_i[y].last = data_wctype_tolower[x].first;
				tolower_cvt[y] = data_wctype_tolower[x].last - data_wctype_tolower[x].first;
			} else
				++data_wctype_tolower_i[y].last;
		}
		tolower_i_size = y;
	}

	idx = bisearch(c, data_wctype_tolower_i, tolower_i_size);

	if (idx==-1)
		return c;
	else
		return c+tolower_cvt[idx];
}

/* Test the functions */

/*
main(int argc,char *argv[])
{
	int c;
	sscanf(argv[1],"%x",&c);
	printf("Properties of character %x:\n",c);
	printf("upper=%x\n",joe_iswupper(NULL,c));
	printf("lower=%x\n",joe_iswlower(NULL,c));
	printf("alpha=%x\n",joe_iswalpha(NULL,c));
	printf("digit=%x\n",joe_iswdigit(NULL,c));
	printf("ctrl=%x\n",joe_iswctrl(NULL,c));
	printf("punct=%x\n",joe_iswpunct(NULL,c));
	printf("graph=%x\n",joe_iswgraph(NULL,c));
	printf("print=%x\n",joe_iswprint(NULL,c));
	printf("xdigit=%x\n",joe_iswxdigit(NULL,c));
	printf("blank=%x\n",joe_iswblank(NULL,c));
	printf("width=%x\n",joe_wcwidth(1,c));
	printf("toupper=%x\n",joe_towupper(NULL,c));
	printf("tolower=%x\n",joe_towlower(NULL,c));
}
*/

/* Return true if c is a control character which should not be displayed */
/* This should match mk_wcwidth() */

int unictrl(int ucs)
{
	/* Control characters are one column wide in JOE */
	if (ucs < 32 || ucs == 0x7F)
		return 1;

	if (ucs >= 0x80 && ucs <= 0x9F)
		return 4;

	/* More control characters... */
	if (ucs>=0x200b && ucs<=0x206f) {
		if (ucs<=0x200f) return 6;
		if (ucs>=0x2028 && ucs<=0x202E) return 6;
		if (ucs>=0x2060 && ucs<=0x2063) return 6;
		if (ucs>=0x206a) return 6;
	}

	/* More control characters... */
	if (ucs>=0xFDD0 && ucs<=0xFDEF)
		return 6;

	if (ucs==0xFEFF)
		return 6;

	if (ucs>=0xFFF9 && ucs<=0xFFFB)
		return 6;

	if (ucs>=0xFFFE && ucs<=0xFFFF)
		return 6;

	return 0;
}
