"""Utility for calling pandoc"""
#-----------------------------------------------------------------------------
# Copyright (c) 2013 the IPython Development Team.
#
# Distributed under the terms of the Modified BSD License.
#
# The full license is in the file COPYING.txt, distributed with this software.
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Imports
#-----------------------------------------------------------------------------

from __future__ import print_function

# Stdlib imports
import subprocess
from io import TextIOWrapper, BytesIO

# IPython imports
from IPython.utils.py3compat import cast_bytes

from .exceptions import ConversionException

#-----------------------------------------------------------------------------
# Classes and functions
#-----------------------------------------------------------------------------

class PandocMissing(ConversionException):
    """Exception raised when Pandoc is missing. """
    pass


def pandoc(source, fmt, to, extra_args=None, encoding='utf-8'):
    """Convert an input string in format `from` to format `to` via pandoc.

    This function will raise an error if pandoc is not installed.
    Any error messages generated by pandoc are printed to stderr.

    Parameters
    ----------
    source : string
      Input string, assumed to be valid format `from`.
    fmt : string
      The name of the input format (markdown, etc.)
    to : string
      The name of the output format (html, etc.)

    Returns
    -------
    out : unicode
      Output as returned by pandoc.
    """
    command = ['pandoc', '-f', fmt, '-t', to]
    if extra_args:
        command.extend(extra_args)
    try:
        p = subprocess.Popen(command,
                             stdin=subprocess.PIPE, stdout=subprocess.PIPE
        )
    except OSError as e:
        raise PandocMissing(
            "The command '%s' returned an error: %s.\n" %(" ".join(command), e) +
            "Please check that pandoc is installed:\n" +
            "http://johnmacfarlane.net/pandoc/installing.html"
        )
    out, _ = p.communicate(cast_bytes(source, encoding))
    out = TextIOWrapper(BytesIO(out), encoding, 'replace').read()
    return out.rstrip('\n')

