// --------------------------------------------------------------------
// The image object.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeimage.h"
#include "ipevisitor.h"
#include "ipepainter.h"

// --------------------------------------------------------------------

/*! \class IpeImage
  \ingroup obj
  \brief The image object.
*/

//! Create from XML stream.
IpeImage::IpeImage(IpeRepository *rep, const IpeXmlAttributes &attr,
		   IpeString data)
  : IpeObject(rep, attr)
{
  Init(attr);
  iBitmap = IpeBitmap(attr, data);
}

//! Create from XML stream with given bitmap.
IpeImage::IpeImage(IpeRepository *rep, const IpeXmlAttributes &attr,
		   IpeBitmap bitmap)
  : IpeObject(rep, attr), iBitmap(bitmap)
{
  Init(attr);
}

void IpeImage::Init(const IpeXmlAttributes &attr)
{
  SetStroke(IpeAttribute()); // null stroke, so that it won't be saved
  // parse rect
  IpeLex st(attr["rect"]);
  IpeVector v;
  st >> v.iX >> v.iY;
  iRect.AddPoint(v);
  st >> v.iX >> v.iY;
  iRect.AddPoint(v);
}

//! Create a new image
IpeImage::IpeImage(const IpeRect &rect, IpeBitmap bitmap)
  : IpeObject()
{
  iRect = rect;
  iBitmap = bitmap;
  assert(!iBitmap.IsNull());
}

//! Clone object
IpeObject *IpeImage::Clone() const
{
  return new IpeImage(*this);
}

//! Return pointer to this object.
IpeImage *IpeImage::AsImage()
{
  return this;
}

//! Call VisitImage of visitor.
void IpeImage::Accept(IpeVisitor &visitor) const
{
  visitor.VisitImage(this);
}

//! Save image in XML stream.
void IpeImage::SaveAsXml(IpePainter &painter, IpeStream &stream,
			 IpeString layer) const
{
  stream << "<image";
  SaveAttributesAsXml(painter, stream, layer);
  stream << " rect=\"" << Rect() << "\"";
  stream << " bitmap=\"" << iBitmap.ObjNum() << "\"";
  stream << "/>\n";
}

//! Draw image.
void IpeImage::Draw(IpePainter &painter) const
{
  IpeMatrix m(iRect.Width(), 0, 0, iRect.Height(),
	      iRect.Min().iX, iRect.Min().iY);
  painter.Push();
  painter.Transform(Matrix());
  painter.Transform(m);
  painter.DrawBitmap(iBitmap);
  painter.Pop();
}

double IpeImage::Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const
{
  IpeMatrix m1 = m * Matrix();
  IpeVector u[5];
  u[0] = m1 * iRect.Min();
  u[1] = m1 * iRect.BottomRight();
  u[2] = m1 * iRect.Max();
  u[3] = m1 * iRect.TopLeft();
  u[4] = u[0];
  IpeRect box;
  for (int i = 0; i < 4; ++i)
    box.AddPoint(u[i]);
  if (box.CertainClearance(v, bound))
    return bound;
  double d = bound;
  double d1;
  for (int i = 0; i < 4; ++i) {
    if ((d1 = IpeSegment(u[i], u[i+1]).Distance(v, d)) < d)
      d = d1;
  }
  return d;
}

void IpeImage::AddToBBox(IpeRect &box, const IpeMatrix &m) const
{
  IpeMatrix m1 = m * Matrix();
  box.AddPoint(m1 * iRect.Min());
  box.AddPoint(m1 * iRect.BottomRight());
  box.AddPoint(m1 * iRect.Max());
  box.AddPoint(m1 * iRect.TopLeft());
}

void IpeImage::SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const
{
  IpeMatrix m1 = m * Matrix();
  SnapVertex(mouse, m1 * iRect.Min(), pos, bound);
  SnapVertex(mouse, m1 * iRect.BottomRight(), pos, bound);
  SnapVertex(mouse, m1 * iRect.Max(), pos, bound);
  SnapVertex(mouse, m1 * iRect.TopLeft(), pos, bound);
}

// --------------------------------------------------------------------
