/*
 *  PGGraphvizToolbarController.m
 *  graphviz
 *
 *  Created by Glen Low on Tue Dec 23 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#import "PGGraphvizToolbarController.h"
#import "PGGraphvizDocument.h"
#import "PGZoomImageView.h"

static NSString* DotSettings = @"DotSettings";
static NSString* DotEdit = @"DotEdit";
static NSString* DotStop = @"DotStop";
static NSString* DotRender = @"DotRender";
static NSString* DotZoom = @"DotZoom";

static NSToolbarItem* MakeToolbarItem (NSString* name)
	{
		NSToolbarItem* item = [[NSToolbarItem alloc] initWithItemIdentifier:
			[NSString stringWithFormat: @"Dot%@", name]];
			
		NSBundle* mainBundle = [NSBundle mainBundle];
		[item setLabel: [mainBundle localizedStringForKey: [NSString stringWithFormat: @"%@.label", name] value: name table: nil]];
		[item setPaletteLabel: [mainBundle localizedStringForKey: [NSString stringWithFormat: @"%@.paletteLabel", name] value: name table: nil]];
		[item setToolTip: [mainBundle localizedStringForKey: [NSString stringWithFormat: @"%@.toolTip", name] value: name table: nil]];
		[item setImage: [NSImage imageNamed: name]];
		
		return [item autorelease];
	}

@implementation PGGraphvizToolbarController

- (void) awakeFromNib
	{
		NSNotificationCenter* defaultCenter = [NSNotificationCenter defaultCenter];
		[defaultCenter
			addObserver: self
			selector: @selector (processDidChange:)
			name: DotDocumentProcessDidChange
			object: document];
		[defaultCenter
			addObserver: self
			selector: @selector (imagerZoomDidChange:)
			name: ZoomImageViewZoomDidChange
			object: imager];
			
		NSToolbar* toolbar = [[NSToolbar alloc] initWithIdentifier: @"DotToolbar"];
		[toolbar setAllowsUserCustomization: YES];
		[toolbar setDelegate: self];
		
		NSToolbarItem *settings = MakeToolbarItem (@"Settings");
		[settings setTarget: nil];
		[settings setAction: @selector (toggleSettings:)];

		NSToolbarItem *edit = MakeToolbarItem (@"Edit");
		[edit setTarget: nil];
		[edit setAction: @selector (editDocument:)];

		NSToolbarItem *render = MakeToolbarItem (@"Render");

		NSToolbarItem *stop = MakeToolbarItem (@"Stop");

		NSToolbarItem *zoom = MakeToolbarItem (@"Zoom");
		zoomer_ = [[NSSlider alloc] init];
		[zoomer_ setContinuous: YES];
		[zoomer_ setNumberOfTickMarks: 7];
		[zoomer_ setMinValue: -3.0];
		[zoomer_ setMaxValue: 3.0];
		[zoomer_ setDoubleValue: 0.0];
		[[zoomer_ cell] setControlSize: NSSmallControlSize];
		zoomFormat_ = [[zoom label] retain];
		[zoom setLabel: [NSString stringWithFormat: zoomFormat_, 100.0]];
		[zoom setView: zoomer_];
		[zoom setTarget: self];
		[zoom setAction: @selector (zoomerZoomDidChange:)];
		[zoom setMinSize: NSMakeSize (129, 18)];
		
		items_ = [[NSDictionary alloc] initWithObjectsAndKeys:
			settings, [settings itemIdentifier],
			edit, [edit itemIdentifier],
			render, [render itemIdentifier],
			stop, [stop itemIdentifier],
			zoom, [zoom itemIdentifier],
			nil];
			
		zooming_ = YES;
		
		[window setToolbar: toolbar];
		[toolbar release];
	}

- (NSToolbarItem*) toolbar: (NSToolbar*) toolbar itemForItemIdentifier: (NSString*) itemIdentifier willBeInsertedIntoToolbar: (BOOL) flag
	{
		return [items_ objectForKey: itemIdentifier];
	}
	
- (NSArray*) toolbarAllowedItemIdentifiers: (NSToolbar*) toolbar
	{
		return [NSArray arrayWithObjects: DotSettings,
			DotEdit, DotRender, DotStop, 
			DotZoom,
			NSToolbarShowColorsItemIdentifier,
			NSToolbarShowFontsItemIdentifier,
			NSToolbarCustomizeToolbarItemIdentifier,
			NSToolbarPrintItemIdentifier,
			NSToolbarSeparatorItemIdentifier,
			NSToolbarSpaceItemIdentifier,
			NSToolbarFlexibleSpaceItemIdentifier,
			nil];
	}
	
- (NSArray*) toolbarDefaultItemIdentifiers: (NSToolbar*) toolbar
	{
		return [NSArray arrayWithObjects: DotSettings, 
			NSToolbarSeparatorItemIdentifier,
			DotEdit, DotRender, DotStop, 
			NSToolbarFlexibleSpaceItemIdentifier,
			DotZoom, nil];
	}
	
- (void) processDidChange: (NSNotification*) notification
	{
		BOOL running;
		switch ([[notification object] process])
			{
				case ProcessRendering:
				case ProcessExporting:
					running = YES;
					break;
				default:
					running = NO;
					break;
			}
			
		[[items_ objectForKey: DotStop] setAction: (running ? @selector (stopDocument:) : nil)];
		[[items_ objectForKey: DotRender] setAction: (running ? nil : @selector (renderDocument:))];
		
	}
	
- (void) zoomerZoomDidChange: (id) sender
	{
		if (zooming_)
			{
				double zoom = pow (2.0, [zoomer_ doubleValue]);
				[[items_ objectForKey: DotZoom] setLabel:
					[NSString stringWithFormat: zoomFormat_, zoom * 100.0]];
				
				// inhibit indirect call of imagerZoomDidChange, which leads to infinite recursion
				zooming_ = NO;
				[imager setZoom: zoom];
				zooming_ = YES;
			}
	}

- (void) imagerZoomDidChange: (id) sender
	{
		if (zooming_)
			{
				double zoom = [imager zoom];
				[[items_ objectForKey: DotZoom] setLabel:
					[NSString stringWithFormat: zoomFormat_, zoom * 100.0]];
				
				// inhibit indirect call of zoomerZoomDidChange, which leads to infinite recursion
				zooming_ = NO;
				[zoomer_ setDoubleValue: log ([imager zoom]) / log (2.0)];
				zooming_ = YES;
			}
	}
	
- (void) dealloc
	{
		[[NSNotificationCenter defaultCenter] removeObserver: self];
		
		[zoomFormat_ release];
		[items_ release];
		[zoomer_ release];

		[super dealloc];
	}
	
@end
