/* 
 * Copyright (C) 1999 Robert Wilhelm
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include "debug.h"
#include "child.h"

int 
start_child (char *cmd, 
	     char **arg, 
	     GIOChannel **read_chan, 
	     GIOChannel **write_chan, 
	     pid_t *childpid)
{
	int pipe1[2], pipe2[2];

	if ((pipe (pipe1) <0) || (pipe (pipe2) < 0)) {
		perror ("pipe");
		exit (-1);
	}

	if ((*childpid = fork ()) < 0) {
		perror ("fork");
		exit (-1);
	} else if (*childpid > 0) /* Parent */ {
		close (pipe1[0]);
		close (pipe2[1]);

		*read_chan = g_io_channel_unix_new (pipe2[0]);
		*write_chan = g_io_channel_unix_new (pipe1[1]);

		return *childpid;
	} else   /* Child */ {
		close (pipe1[1]);
		close (pipe2[0]);

		dup2 (pipe1[0],0);
		dup2 (pipe2[1],1);

		close (pipe1[0]);
		close (pipe2[1]);

		if (execvp (cmd, arg) < 0)
			perror (cmd);
		_exit (1);
	}

	g_assert_not_reached ();

	return 0;
}

void 
write_child (GIOChannel *write_chan, char *format, ...) 
{
	GIOError err;
	va_list ap;
	char *buf;
	int len;	

	va_start (ap, format);

	buf = g_strdup_vprintf (format, ap);

	err = g_io_channel_write (write_chan, buf, strlen (buf), &len);
	if (err != G_IO_ERROR_NONE)
		g_warning ("Writing to child process failed");

	debug_print (DEBUG_NORMAL, buf);  

	va_end (ap);

	g_free (buf);
}

/* Kill Child */
int 
stop_child (pid_t childpid) 
{
	if (childpid && kill (childpid, SIGTERM) ) { 
		debug_print (DEBUG_NONE,"Failed to kill child!\n");
		return 1;
	}

	return 0;
}
