/* -*- mode: C++; c-basic-offset: 2; indent-tabs-mode: nil -*- */
/*
 *  Main authors:
 *     Christian Schulte <schulte@gecode.org>
 *
 *  Copyright:
 *     Christian Schulte, 2004
 *
 *  Last modified:
 *     $Date: 2009-09-08 21:10:29 +0200 (Tue, 08 Sep 2009) $ by $Author: schulte $
 *     $Revision: 9692 $
 *
 *  This file is part of Gecode, the generic constraint
 *  development environment:
 *     http://www.gecode.org
 *
 *  Permission is hereby granted, free of charge, to any person obtaining
 *  a copy of this software and associated documentation files (the
 *  "Software"), to deal in the Software without restriction, including
 *  without limitation the rights to use, copy, modify, merge, publish,
 *  distribute, sublicense, and/or sell copies of the Software, and to
 *  permit persons to whom the Software is furnished to do so, subject to
 *  the following conditions:
 *
 *  The above copyright notice and this permission notice shall be
 *  included in all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 *  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 *  LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 *  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 *  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

namespace Gecode { namespace Iter { namespace Ranges {

  /**
   * \brief Range iterator for pointwise minus of a range iterator
   *
   * This iterator in effect changes the order of how ranges
   * are iterated: the first range of the input iterator defines
   * the last range of the Minus iterator. Upon initialization
   * all ranges of the input iterator are stored in an array
   * which later allows iteration in inverse direction.
   *
   * \ingroup FuncIterRanges
   */

  template<class I>
  class Minus {
  private:
    /// Check that \a I is a range iterator;
    IsRangeIter<I> _checkI;
  private:
    /// %Range for storage in array
    class Range {
    public:
      int min; int max;
    };
    /// %Ranges stored
    SharedArray<Range> r;
    /// Current range
    int c;
  public:
    /// \name Constructors and initialization
    //@{
    /// Default constructor
    Minus(void);
    /// Initialize with ranges from \a i
    Minus(I& i);
    /// Initialize with ranges from \a i
    void init(I& i);
    //@}

    /// \name Iteration control
    //@{
    /// Test whether iterator is still at a range or done
    bool operator ()(void) const;
    /// Move iterator to next range (if possible)
    void operator ++(void);
    //@}

    /// \name Range access
    //@{
    /// Return smallest value of range
    int min(void) const;
    /// Return largest value of range
    int max(void) const;
    /// Return width of range (distance between minimum and maximum)
    unsigned int width(void) const;
    //@}
  };


  template<class I>
  forceinline
  Minus<I>::Minus(void) {}

  template<class I>
  inline void
  Minus<I>::init(I& i) {
    Support::DynamicArray<Range,Heap> d(heap);
    int n=0;
    while (i()) {
      d[n].min = -i.max(); d[n].max = -i.min();
      ++n; ++i;
    }
    r.init(n);
    for (int j=n; j--; )
      r[j]=d[j];
    c = n-1;
  }

  template<class I>
  inline
  Minus<I>::Minus(I& i) {
    init(i);
  }

  template<class I>
  forceinline void
  Minus<I>::operator ++(void) {
    c--;
  }
  template<class I>
  forceinline bool
  Minus<I>::operator ()(void) const {
    return c >= 0;
  }

  template<class I>
  forceinline int
  Minus<I>::min(void) const {
    return r[c].min;
  }
  template<class I>
  forceinline int
  Minus<I>::max(void) const {
    return r[c].max;
  }
  template<class I>
  forceinline unsigned int
  Minus<I>::width(void) const {
    return static_cast<unsigned int>(r[c].max-r[c].min+1);
  }

}}}

// STATISTICS: iter-any

