/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#include <glob.h>
#include <tiffio.h>
#include <png.h>
#include <lcms2.h>
#include "zfuncs.h"

//  Fotoxx definitions

#define Frelease     "Fotoxx 14.10.2"                                      //  Fotoxx release version
#define Flicense     "Free software - GNU General Public License v.3"
#define Fhomepage    "http://kornelix.com/fotoxx"
#define Fsoftware    "Software used: GNU, GTK, libtiff, libpng, liblcms, DCraw, exiftool"
#define Fcontact     "Questions and bugs: kornelix@posteo.de"
#define Ftranslators "Translators: \n"                                                                \
                     "   Doriano Blengino, Arthur Kalverboer, André Campos Rodovalho, \n"             \
                     "   Peter Landgren, J. A. Miralles Puignau, Pierrick Pinot "
                     //  Nikolay declined 
#define mega (1024 * 1024)                                                 //  1 million as 2**20
#define PI 3.141592654
#define PXMpix(PXM,px,py) (PXM->pixels+((py)*(PXM->ww)+(px))*3)            //  get PXM pixel[3] at (px,py) (float[3])
#define PXBpix(PXB,px,py) (PXB->pixels+((py)*(PXB->rs)+(px)*3))            //  get PXB pixel[3] at (px,py) (uint8[3])
#define pixbright(pix) (0.25*(pix)[0]+0.65*(pix)[1]+0.10*(pix)[2])         //  pixel brightness, 0-255.0
#define pixred(pix) (25*pix[0]/(pixbright(pix)+1))                         //  pixel redness, 0-100%

//  color match function for integer/float RGB colors 0-255                //  v.14.09
//  0.0 to 1.0 = zero to perfect color match
#define RGBMATCH(r1,g1,b1,r2,g2,b2)             \
      (   (1.0 - 0.0039 * fabsf(r1-r2))         \
        * (1.0 - 0.0039 * fabsf(g1-g2))         \
        * (1.0 - 0.0039 * fabsf(b1-b2)) )

#define PIXMATCH(pix1,pix2)                     \
      ( RGBMATCH(pix1[0],pix1[1],pix1[2],pix2[0],pix2[1],pix2[2]) )

#define tagdelims  ",;"                                                    //  valid tag delimiters (; obsolete)
#define tagdelim1  ','                                                     //  delimiter for new tags
#define tagdelimB  ", "                                                    //  delimiter + blank

//  compile time limits that could be increased

#define wwhh_limit 30000                                                   //  max. image width/height            v.14.07
#define max_threads 4                                                      //  max. process threads (hyperthreads useless)
#define thumbfilesize 256                                                  //  thumbnail image file size
#define maxtopdirks 100                                                    //  max. no. of top image directories
#define maxrecentfiles 100                                                 //  max. no. of recent files
#define image_index_max 2000                                               //  max. image files per image index file
#define indexrecl 2000                                                     //  max. image index rec. (>maxfcc >tagFcc)
#define maximages 500000                                                   //  max. image files supported 
#define maxgeotags 20000                                                   //  max. location/geotag records       v.14.02
#define maxtagcats 200                                                     //  max tag categories
#define tagcc  50                                                          //  max cc for one tag or category ID
#define tagFcc 1000                                                        //  max tag cc for one image file
#define tagntc 1000                                                        //  max no. tags for one category
#define tagGcc 50000                                                       //  max tag cc for one category
#define tagTcc 50000                                                       //  max tag cc for all defined tags
#define tagMcc 300                                                         //  max tag cc for batch tags
#define tagScc 200                                                         //  max tag cc for search tags
#define tagRcc 200                                                         //  max tag cc for recent tags

//  EXIF/IPTC keys for embedded image metadata

#define iptc_keywords_key "Keywords"                                       //  keywords (tags) (IPTC)
#define iptc_rating_key "Rating"                                           //  star rating (IPTC)
#define exif_size_key "ImageSize"                                          //  image size, NNNNxNNNN
#define exif_date_key "DateTimeOriginal"                                   //  date/time (EXIF)
#define exif_width_key "ImageWidth"                                        //  width, pixels (EXIF)
#define exif_height_key "ImageHeight"                                      //  height, pixels (EXIF)
#define exif_orientation_key "Orientation"                                 //  orientation (EXIF)
#define exif_editlog_key "ImageHistory"                                    //  edit history log (EXIF) 
#define exif_comment_key "UserComment"                                     //  user comments (EXIF)
#define iptc_caption_key "Caption-Abstract"                                //  image caption (IPTC)
#define exif_focal_length_key "FocalLengthIn35mmFormat"                    //  focal length (EXIF)
#define exif_city_key "City"                                               //  city name (geotags)
#define exif_country_key "Country"                                         //  country name
#define exif_latitude_key "GPSLatitude"                                    //  latitude in degrees (-180 to +180)
#define exif_longitude_key "GPSLongitude"                                  //  longitude in degrees (-180 to +180)
#define exif_maxcc 2000                                                    //  max. cc for exif/iptc text

//  GTK/GDK shortcuts

#define TEXTWIN GTK_TEXT_WINDOW_TEXT
#define NODITHER GDK_RGB_DITHER_NONE
#define ALWAYS GTK_POLICY_ALWAYS
#define NEVER GTK_POLICY_NEVER
#define GDKRGB GDK_COLORSPACE_RGB
#define LINEATTRIBUTES GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER
#define BILINEAR GDK_INTERP_BILINEAR
#define SCROLLWIN GTK_SCROLLED_WINDOW
#define VERTICAL GTK_ORIENTATION_VERTICAL
#define HORIZONTAL GTK_ORIENTATION_HORIZONTAL
#define MWIN GTK_WINDOW(Mwin)

//  externals from zfuncs module

namespace zfuncs {
   extern GdkDisplay        *display;                                      //  X11 workstation (KB, mouse, screen)
   extern GdkDeviceManager  *manager;                                      //  knows screen / mouse associations
   extern GdkScreen         *screen;                                       //  monitor
   extern GdkDevice         *mouse;                                        //  pointer device
   extern int     monitorDPI;                                              //  monitor dots/inch
   extern char    zappname[20];                                            //  app name/version
   extern char    zlang[8];                                                //  current language lc_RC
   extern char    zicondir[200];                                           //  where application icons live
   extern int     Nmalloc, Nstrdup, Nfree;                                 //  malloc() strdup() free() counters
   extern int     vmenuclickposn;                                          //  vert. menu icon click position, 0-100
   extern int     vmenuclickbutton;                                        //   "" button: 1/3 = left/right mouse
   extern int     zdialog_count;                                           //  zdialog count (new - free)
   extern int     zdialog_busy;                                            //  open zdialogs (run - destroy)
   extern int     open_popup_windows;                                      //  open popup window count
}

//  externals from f.file module

namespace navi {                                                           //  gallery() data
   extern char    *galleryname;                                            //  directory path or gallery type name
   extern int     gallerytype;                                             //  1-6 = dirk/search/metadata/coll/recent/new
   extern int     topfileposn;                                             //  gallery window target scroll position
   extern int     nfiles;                                                  //  gallery file count (incl. subdirks)
   extern int     nimages;                                                 //  gallery image file count
   extern char    **mdlist;                                                //  corresp. metadata list
   extern int     mdrows;                                                  //  text rows in metadata list
   extern int     xwinW, xwinH;                                            //  image gallery window size
   extern int     thumbsize;                                               //  curr. thumbnail display size
   extern int     genthumbs;                                               //  counts thumbnails generated
   
   int    gallery_paint(GtkWidget *, cairo_t *);                           //  gallery window paint function
   void   menufuncx(GtkWidget *win, cchar *menu);                          //  gallery menu buttons function
   void   changedirk(GtkWidget *widget, GdkEventButton *event);            //  gallery directory change function
   void   newtop(GtkWidget *widget, GdkEventButton *event);                //  gallery change top directory function
   void   mouse_event(GtkWidget *, GdkEvent *, void *);                    //  gallery window mouse event function
   int    KBrelease(GtkWidget *, GdkEventKey *, void *);                   //  gallery window key release event
}

EX GdkScreen      *screen;                                                 //  physical monitor
EX int            screenww, screenhh;                                      //  monitor screen size
EX GtkWidget      *Mwin, *Mvbox;                                           //  main window
EX GtkWidget      *Fhbox, *Fmenu, *Fvbox, *Fpanel, *Fpanlab, *Fdrawin;     //  F window widgets
EX GtkWidget      *Ghbox, *Gmenu, *Gvbox, *Gpanel, *Gtop, *Gpath;          //  G window widgets
EX GtkWidget      *Gscroll, *Gdrawin;
EX GtkAdjustment  *Gadjust;
EX GtkWidget      *Whbox, *Wmenu, *Wvbox, *Wdrawin;                        //  W window widgets
EX GtkWidget      *Cdrawin;                                                //  curr. image drawing window, Fdrawin/Wdrawin
EX GdkWindow      *gdkwin;                                                 //  corresp. GDK window
EX cairo_t        *mwcr;                                                   //  cairo context for drawing windows
EX char           FGW;                                                     //  curr. window mode: 'F' 'G' 'W'

EX GdkCursor         *arrowcursor;                                         //  main window cursors
EX GdkCursor         *dragcursor;
EX GdkCursor         *drawcursor;

EX uint8       BLACK[3], WHITE[3], RED[3], GREEN[3];                       //  RGB values 0-255
EX uint8       *FG_COLOR;                                                  //  foreground drawing color, one of the above
EX GdkRGBA     GDKbackground;                                              //  window background color

EX PangoFontDescription *panelfont;                                        //  font for window top panels
EX int panelfontchww;                                                      //  font pixel width per char.
EX float monitorscale;                                                     //  monitor magnification scale

#define ccc (maxfcc*2+200)                                                 //  max. command line size
EX char        command[ccc];                                               //  (command, parameters, 2 filespecs)

EX int         Nval[10];                                                   //  contains integer values 0-9

//  user settings

EX char        *menu_style;                                                //  menu style: icons/text/both
EX int         iconsize;                                                   //  menu icon size
EX char        *startdisplay;                                              //  display: recent/prev/blank/file/dirk
EX char        *startdirk;                                                 //  start directory (startdisplay=dirk)
EX char        *startfile;                                                 //  start image file (startdisplay=file)
EX int         Fdragopt;                                                   //  1/2 = 1x drag / magnified scroll
EX int         zoomcount;                                                  //  zoom count to reach 2x (1/2/3/4)
EX float       zoomratio;                                                  //  corresp. zoom ratio: 2**(1/zoomcount)
EX char        *DCrawcommand;                                              //  command, RAW to tiff using DCraw
EX char        *RAWfiletypes;                                              //  known RAW file types: .raw .rw2 ...
EX char        *myRAWtypes;                                                //  RAW types encountered

//  general parameters

EX char        *Prelease;                                                  //  prior fotoxx version
EX char        *Nrelease;                                                  //  notified release
EX int         Ffirsttime;                                                 //  first time startup
EX int         usagecounter;                                               //  continuously counts fotoxx startups
EX int         mwgeom[4];                                                  //  main window position and size
EX char        PIDstring[8];                                               //  process PID as string
EX int         NWT;                                                        //  working threads to use
EX int         wthreads_busy;                                              //  active thread count

typedef void   menufunc_t(GtkWidget *,cchar *);                            //  table of menu names and functions
typedef struct {
   GtkWidget   *topmenu;                                                   //  parent menu
   cchar        *menu;                                                     //  menu text
   cchar        *mtran;                                                    //  translation
   cchar        *icon;                                                     //  menu icon
   cchar        *desc;                                                     //  menu short description (tooltip)
   cchar        *dtran;                                                    //  translation
   menufunc_t   *func;                                                     //  menu function (GtkWidget *, cchar *)
   cchar        *arg;                                                      //  argument
}  menutab_t;

#define maxmenus 200
EX menutab_t   menutab[maxmenus];
EX int         Nmenus;

#define maxshortcuts 50
EX char        *shortcutkey[50];                                           //  KB shortcut keys (or combinations)
EX char        *shortcutmenu[50];                                          //  corresponding menu names
EX int         Nshortcuts;                                                 //  count of entries

EX int         Fshutdown;                                                  //  app shutdown underway
EX int         Fblankwindow;                                               //  set blank main window
EX int         Fdebug;                                                     //  debug flag
EX int         Fkillfunc;                                                  //  signal, kill running function
EX int         Findexbusy;                                                 //  file index function needed or running
EX int         Ffuncbusy;                                                  //  function is busy/working
EX int         Fthreadbusy;                                                //  thread is busy/working
EX int         Fmenulock;                                                  //  shared menu lock, block other funcs
EX int         Ffullscreen;                                                //  flag, window is fullscreen
EX int         Frefresh;                                                   //  full image refresh needed
EX int         Fblowup;                                                    //  zoom small images to window size
EX int         Fslideshow;                                                 //  flag, slide show is active
EX int         ss_escape;                                                  //  KB escape or F11 >> slide show process
EX int         ss_Larrow;                                                  //  KB left arrow >> slide show process
EX int         ss_Rarrow;                                                  //  KB right arrow >> slide show process
EX int         ss_spacebar;                                                //  KB space bar >> slide show process
EX int         ss_Bkey;                                                    //  KB B-key >> slide show process
EX int         Frecent;                                                    //  start with recent files gallery
EX int         Fnew;                                                       //  start with newly added files gallery
EX int         Fprev;                                                      //  start with previous file
EX int         Fblank;                                                     //  start with blank window
EX char        *topmenu;                                                   //  latest top-menu selection
EX char        *startmenu;                                                 //  command line, startup menu function
EX char        file_errmess[1000];                                         //  error message from low-level file funcs
EX cchar       *F1_help_topic;                                             //  current function help topic
EX int         Fmetachanged;                                               //  metadata changed and not saved to disk
EX int         Fcaptions;                                                  //  show image captions/comments
EX int         Fshowhidden;                                                //  gallery option, show hidden dirks  v.14.06
EX int         Fdcraw;                                                     //  flag, DCraw available
EX int         Fufraw;                                                     //  flag, UFraw available
EX int         Frawtherapee;                                               //  flag, Raw Therapee available
EX int         Fbrasero;                                                   //  fflag, Brasero available

volatile                                                                   //  stop optimizer removing code
EX double      Fbusy_goal, Fbusy_done;                                     //  BUSY message progress tracking
EX char        paneltext[200];                                             //  top panel application text
EX char        tempdir[200];                                               //  directory for temp files
EX char        *topdirks[maxtopdirks];                                     //  user top-level image directories
EX char        *thumbdirk;                                                 //  thumbnails directory
EX int         Ntopdirks;                                                  //  topdirk[] count
EX char        *curr_file, *curr_dirk;                                     //  current image file and directory
EX char        *initial_file;                                              //  initial file on command line
EX char        curr_file_type[8];                                          //  jpg / tif / png / other
EX int         curr_file_bpc;                                              //  image file bits/color, 8/16
EX int         curr_file_size;                                             //  image file size on disk
EX int         curr_file_posn;                                             //  position in current gallery, 0-last
EX int         curr_file_count;                                            //  count of images in current set
EX char        *clicked_file;                                              //  image file / thumbnail clicked
EX int         clicked_posn;                                               //  clicked gallery position (Nth)
EX int         clicked_width;                                              //  clicked thumbnail position
EX int         clicked_height;                                             //    normalized 0-100                 v.14.04
EX char        *rawfile;                                                   //  RAW file passed to raw edit program
EX char        *imagefiletypes;                                            //  supported image file types, .jpg .png etc.

EX char        f_load_type[8];                                             //  data set by f_load()
EX int         f_load_bpc, f_load_size;
EX char        *f_save_file;                                               //  data set by f_save()
EX char        f_save_type[8];
EX int         f_save_bpc, f_save_size;

//  files and directories in /.../.fotoxx/

EX char        index_dirk[200];                                            //  image index directory
EX char        tags_defined_file[200];                                     //  tags defined file
EX char        recentfiles_file[200];                                      //  file of recent image files 
EX char        newfiles_file[200];                                         //  file of newest image files
EX char        collections_dirk[200];                                      //  directory for saved image collections
EX char        collections_copy[200];                                      //  file for copying a collection being edited
EX char        saved_areas_dirk[200];                                      //  directory for saved select area files
EX char        saved_curves_dirk[200];                                     //  directory for saved curve data
EX char        retouch_combo_dirk[200];                                    //  directory for saved retouch combo settings
EX char        custom_kernel_dirk[200];                                    //  directory for saved custom kernel data
EX char        writetext_dirk[200];                                        //  directory for write_text files
EX char        geotags_dirk[200];                                          //  directory for geotags data files
EX char        worldmap_file[200];                                         //  world map image file 
EX char        favorites_dirk[200];                                        //  directory for favorites menu
EX char        locales_dirk[200];                                          //  directory for .po files
EX char        mashup_dirk[200];                                           //  directory for mashup projects
EX char        KBshortcuts[200];                                           //  keyboard shortcuts file
EX char        quickstart_file[200];                                       //  quick start image file
EX char        latest_release_file[200];                                   //  latest fotoxx release file
EX char        slideshow_dirk[200];                                        //  directory for slide show files
EX char        pattern_dirk[200];                                          //  directory for pattern files

//  fotoxx PXM and PXB pixmaps

typedef struct  {                                                          //  PXM pixmap, 3 x float per pixel
   char        wmi[8];                                                     //  self-identifier
   int         ww, hh;                                                     //  width, height
   float       *pixels;                                                    //  ww * hh * 3 * sizeof(float)
}  PXM;

typedef struct  {                                                          //  PXB pixmap, 3 x uint8/pixel
   char        wmi[8];                                                     //  self-identifier
   GdkPixbuf   *pixbuf;                                                    //  underlying pixbuf image
   int         ww, hh, rs;                                                 //  width, height, row-stride
   uint8       *pixels;                                                    //  ww * rs bytes (in pixbuf)
}  PXB;

//  parameters for F/W window painting (functions Fpaint() etc.)

typedef struct  {
   PXB         *fpxb;                                                      //  image PXB, size = 1x
   PXB         *mpxb;                                                      //  image PXB, size = Mscale
   int         morgx, morgy;                                               //  Dpxb origin in Mpxb image
   int         dorgx, dorgy;                                               //  Dpxb origin in drawing window
   float       fzoom;                                                      //  image zoom scale (0 = fit window)
   float       mscale;                                                     //  scale factor, 1x image to window image
   float       pscale;                                                     //  prior scale factor
}  FWstate;

EX FWstate    Fstate, Wstate;                                              //  parameters for F and W windows
EX FWstate    *Cstate;                                                     //  current parameters, F or W

#define Fpxb   Fstate.fpxb                                                 //  F window parameter equivalents
#define Mpxb   Fstate.mpxb
#define Morgx  Fstate.morgx
#define Morgy  Fstate.morgy
#define Dorgx  Fstate.dorgx
#define Dorgy  Fstate.dorgy
#define Fzoom  Fstate.fzoom
#define Mscale Fstate.mscale
#define Pscale Fstate.pscale

EX PXM         *E0pxm;                                                     //  edit pixmap, original image
EX PXM         *E1pxm;                                                     //  edit pixmap, base image for editing
EX PXM         *E3pxm;                                                     //  edit pixmap, edited image
EX PXM         *ERpxm;                                                     //  edit pixmap, undo/redo image
EX PXM         *E8pxm;                                                     //  scratch image for some functions
EX PXM         *E9pxm;                                                     //  scratch image for some functions

EX int         Dww, Dhh;                                                   //  main/drawing window size
EX int         dww, dhh;                                                   //  Dpxb size in drawing window, <= Dww, Dhh
EX int         zoomx, zoomy;                                               //  req. zoom center of window

EX mutex_tp    Fpixmap_lock;                                               //  lock for accessing PXM pixmaps
EX mutex_tp    preload_lock;                                               //  lock for image file preload function

EX int         Mbutton;                                                    //  mouse button, 1/3 = left/right
EX int         Mwxposn, Mwyposn;                                           //  mouse position, window space
EX int         Mxposn, Myposn;                                             //  mouse position, image space
EX int         LMclick, RMclick;                                           //  mouse left, right click
EX int         Mxclick, Myclick;                                           //  mouse click position, image space
EX int         Mdrag;                                                      //  mouse drag underway
EX int         Mwdragx, Mwdragy;                                           //  drag increment, window space
EX int         Mxdown, Mydown, Mxdrag, Mydrag;                             //  mouse drag vector, image space
EX int         Fmousemain;                                                 //  mouse acts on main window not dialog
EX int         Mcapture;                                                   //  mouse captured by edit function

EX int         KBcapture;                                                  //  KB key captured by edit function
EX int         KBkey;                                                      //  active keyboard key
EX int         KBcontrolkey;                                               //  keyboard key states, 1 = down
EX int         KBshiftkey;
EX int         KBaltkey;

//  lines, text and circles drawn over window image whenever repainted

struct topline_t {
   int      x1, y1, x2, y2;                                                //  endpoint coordinates in image space
   int      type;                                                          //  1/2 = solid/dotted
};

#define maxtoplines 8                                                      //  max. top lines
EX topline_t   toplines[8], ptoplines[8];                                  //  top lines, prior top lines
EX int         Ntoplines, Nptoplines;                                      //  current counts

struct toptext_t {
   int      ID, px, py;
   cchar    *text;
   cchar    *font;
};

#define maxtoptext 100                                                     //  max. text strings
EX toptext_t   toptext[100];                                               //  text strings
EX int         Ntoptext;                                                   //  current count

struct topcircle_t {
   int      px, py;
   int      radius;
};

#define maxtopcircles 100                                                  //  max. circles
EX topcircle_t    topcircles[100];                                         //  circles 
EX int            Ntopcircles;                                             //  current count

//  spline curve data

typedef void t_spcfunc(int spc);                                           //  callback function, spline curve edit

typedef struct {                                                           //  spline curve data 
   GtkWidget   *drawarea;                                                  //  drawing area for spline curves
   t_spcfunc   *spcfunc;                                                   //  callback function when curve changed
   int         Nspc;                                                       //  number of active curves, 1-10
   int         vert[10];                                                   //  curve is vert. (1) or horz. (0)
   int         nap[10];                                                    //  anchor points per curve
   float       apx[10][50], apy[10][50];                                   //  up to 50 anchor points per curve
   float       yval[10][1000];                                             //  y-values for x = 0 to 1 by 0.001
   int         Nscale;                                                     //  no. of fixed scale lines, 0-10
   float       xscale[2][10];                                              //  2 x-values for end points
   float       yscale[2][10];                                              //  2 y-values for end points
} spldat;

//  select area data

#define sa_initseq 10                  //  initial sequence number (0/1/2 reserved)
#define sa_maxseq 9999                 //  ultimate limit 64K
#define mode_rect       1              /* select rectangle by drag/click */
#define mode_ellipse    2              /* select ellipse by drag */
#define mode_draw       3              /* freehand draw by drag/click */
#define mode_follow     4              /* follow edge indicated by clicks */
#define mode_replace    5              /* freehand draw and replace nearby pixels */
#define mode_mouse      6              /* select area within mouse (radius) */
#define mode_onecolor   7              /* select one matching color within mouse */
#define mode_allcolors  8              /* select all matching colors within mouse */
#define mode_image      9              /* select whole image */

EX uint16      sa_endpx[10000], sa_endpy[10000];                           //  last pixel drawn per seqence no.
EX int         sa_thresh;                                                  //  mouse pixel distance threshold
EX int         sa_mouseradius;                                             //  mouse selection radius
EX int         sa_searchrange;                                             //  search range (* mouse radius)
EX int         sa_mousex, sa_mousey;                                       //  mouse position in image
EX float       sa_colormatch;                                              //  color range to match (0.001 to 1.0)
EX int         sa_firewall;                                                //  firewall: no search thru selected pixels
EX int         sa_calced;                                                  //  edge calculation done
EX int         sa_blend;                                                   //  edge blend width
EX int         sa_minx, sa_maxx;                                           //  enclosing rectangle for area
EX int         sa_miny, sa_maxy;
EX char        *sa_stackdirec;                                             //  pixel search stack
EX int         *sa_stackii;    
EX int         sa_maxstack;
EX int         sa_Nstack;
EX int         sa_stat;                                                    //  0/1/2/3/4 = none/edit/xx/fini./disab.
EX int         sa_mode;                                                    //  1-7 = curr. select area edit method
EX int         sa_finOK;                                                   //  finish area - success
EX int         sa_finhole;                                                 //  finish area - area has a hole
EX int         sa_fincancel;                                               //  finish area - user cancel
EX int         sa_currseq;                                                 //  current select sequence no.
EX int         sa_Ncurrseq;                                                //  current sequence pixel count
EX char        *sa_pixselc;                                                //  maps pixels selected in current cycle
EX uint16      *sa_pixmap;                                                 //  0/1/2+ = outside/edge/inside edge dist
EX int         sa_Npixel;                                                  //  total select area pixel count
EX int         sa_fww, sa_fhh;                                             //  valid image dimensions for select area
EX int         Fshowarea;                                                  //  show select area outline
EX int         areanumber;                                                 //  increasing sequential number

//  grid lines parameters, for each grid lines set [G] 

#define GON    0                                                           //  gridsettings[G][0]  grid lines on/off
#define GX     1                                                           //              [G][1]  x-lines on/off
#define GY     2                                                           //              [G][2]  y-lines on/off
#define GXS    3                                                           //              [G][3]  x-lines spacing
#define GYS    4                                                           //              [G][4]  y-lines spacing
#define GXC    5                                                           //              [G][5]  x-lines count
#define GYC    6                                                           //              [G][6]  y-lines count
#define GXF    7                                                           //              [G][7]  x-lines offset
#define GYF    8                                                           //              [G][8]  y-lines offset
#define G9     9                                                           //              [G][9]  unused
EX int         currgrid;                                                   //  current grid params set, 0-5
EX int         gridsettings[6][10];                                        //  settings for 6 sets of grid lines

//  image index file record. 
//  tags, comms, capt, gtags may have "null" (char) as a value.

struct sxrec_t {
   char        *file;                                                      //  image filespec
   char        fdate[16];                                                  //  file date, yyyymmddhhmmss
   char        pdate[16];                                                  //  EXIF (photo) date, yyyymmddhhmmss
   char        size[16];                                                   //  image size, "12345x9876"
   char        rating[4];                                                  //  IPTC rating, '0' to '5' stars
   char        *tags;                                                      //  IPTC tags
   char        *capt;                                                      //  IPTC caption
   char        *comms;                                                     //  EXIF comments
   char        *gtags;                                                     //  EXIF geotags
};
EX FILE        *sxrec_fid;                                                 //  open file ID - get_sxrec()

//  image edit function data

typedef struct {                                                           //  edit function data
   cchar       *funcname;                                                  //  function name, e.g. flatten
   int         FprevReq;                                                   //  request to use smaller image if poss.
   int         Farea;                                                      //  area: 0/1/2 = delete/ignore/useable
   int         Fmods;                                                      //  flag, image modifications by this func
   int         Fpreview;                                                   //  flag, smaller image for edits in use
   int         Frestart;                                                   //  flag, OK to restart with new file
   int         Fsaved;                                                     //  current mods are saved to disk
   zdialog     *zd;                                                        //  edit dialog
   spldat      *curves;                                                    //  edit curve data
   void        (*menufunc)(GtkWidget *, cchar *);                          //  edit menu function
   void *      (*threadfunc)(void *);                                      //  edit thread function
   void        (*mousefunc)();                                             //  edit mouse function
   int         thread_command, thread_status;
   int         thread_pend, thread_done, thread_hiwater;
}  editfunc;

EX editfunc    *CEF;                                                       //  current active edit function

//  undo/redo stack data

#define maxedits 100                                                       //  undo/redo stack size
EX char        URS_filename[200];                                          //  stack image filename template
EX int         URS_pos;                                                    //  stack position, 0-99
EX int         URS_max;                                                    //  stack max. position, 0-99
EX char        URS_funcs[100][20];                                         //  corresp. edit func. names
EX int         URS_saved[100];                                             //  corresp. edit saved to disk or not
EX int         URS_reopen_pos;                                             //  reopen last saved file, stack position

#define maxplugins 100
EX int         Nplugins;                                                   //  plugin menu items
EX char        *plugins[100];

struct textattr_t {                                                        //  attributes for gentext() function
   char     text[1000];                                                    //  text to generate image from
   char     font[80];                                                      //  font name
   int      size;                                                          //  font size
   int      tww, thh;                                                      //  generated image size, unrotated
   float    angle;                                                         //  text angle, degrees
   float    sinT, cosT;                                                    //  trig funcs for text angle
   char     color[4][20];                                                  //  text, backing, outline, shadow "R|G|B"
   int      transp[4];                                                     //  corresponding transparencies 0-255
   int      towidth;                                                       //  outline width, pixels
   int      shwidth;                                                       //  shadow width
   int      shangle;                                                       //  shadow angle -180...+180
   PXM      *pxm_text;                                                     //  image with text/outline/shadow
   PXM      *pxm_transp;                                                   //  image with transparency data
};

struct lineattr_t {                                                        //  attributes for genline() function
   int      length, width;                                                 //  line length and width, pixels
   int      larrow, rarrow;                                                //  left/right arrow head size (0 = no arrow)
   int      lww, lhh;                                                      //  generated image size, unrotated
   float    angle;                                                         //  line angle, degrees
   float    sinT, cosT;                                                    //  trig funcs for line angle
   char     color[4][20];                                                  //  line, background, outline, shadow "R|G|B"
   int      transp[4];                                                     //  corresponding transparencies 0-255
   int      towidth;                                                       //  outline width, pixels
   int      shwidth;                                                       //  shadow width
   int      shangle;                                                       //  shadow angle -180...+180
   PXM      *pxm_line;                                                     //  line/arrow image
   PXM      *pxm_transp;                                                   //  image with transparency data
};

//  dialogs with global visibility

EX zdialog     *zdbrdist;                                                  //  brightness distribution zdialog
EX zdialog     *zddarkbrite;                                               //  highlight dark/bright pixels zdialog
EX zdialog     *zdeditmeta;                                                //  edit metadata zdialog
EX zdialog     *zdbatchtags;                                               //  batch tags dialog
EX zdialog     *zdexifview;                                                //  view EXIF/IPTC zdialog
EX zdialog     *zdexifedit;                                                //  edit EXIF/IPTC zdialog
EX zdialog     *zdeditgeotags;                                             //  add geotags zdialog
EX zdialog     *zdfilesave;                                                //  file save zdialog
EX zdialog     *zdrename;                                                  //  rename file zdialog
EX zdialog     *zdcopymove;                                                //  copy/move file zdialog
EX zdialog     *zddelete;                                                  //  delete file zdialog
EX zdialog     *zdtrash;                                                   //  trash file zdialog
EX zdialog     *zdsela;                                                    //  select area zdialog
EX zdialog     *zd_gallery_getfiles;                                       //  gallery_getfiles zdialog
EX zdialog     *zd_edit_bookmarks;                                         //  bookmarks edit zdialog
EX zdialog     *zd_ss_imageprefs;                                          //  slide show image prefs zdialog     v.14.04

//  method for running thread to send an event to an active zdialog

EX zdialog     *zd_thread;                                                 //  active zdialog
EX cchar       *zd_thread_event;                                           //  event to send

EX int         trimrect[4];                                                //  trim rectangle, x1, y1, x2, y2
EX int         trimsize[2];                                                //  trim (crop) width, height
EX char        *trimbuttons[6];                                            //  trim dialog button labels
EX char        *trimratios[6];                                             //  corresponding aspect ratios
EX int         editresize[2];                                              //  edit resize width, height

EX int         jpeg_def_quality;                                           //  default jpeg save quality (user setting)
EX int         jpeg_1x_quality;                                            //  file save 1-time quality setting

EX float       lens_mm, lens_bow;                                          //  pano lens mm and bow settings

//  GTK functions (fotoxx main)

int   main(int argc, char * argv[]);                                       //  main program
int   initzfunc(void *);                                                   //  initializations
int   delete_event();                                                      //  window delete event function
int   destroy_event();                                                     //  window destroy event function
int   state_event(GtkWidget *, GdkEvent *event);                           //  window state event function
int   gtimefunc(void *arg);                                                //  periodic function
void  update_Fpanel();                                                     //  update F window information panel
void  paint_busy(cchar *message = 0);                                      //  paint "BUSY" on window if something busy
int   Fpaint(GtkWidget *, cairo_t *);                                      //  F and W drawing area paint function
void  Fpaint2();                                                           //  window repaint (thread callable)
void  Fpaintnow();                                                         //  window repaint synchronously
void  Fpaint3(int px, int py, int ww, int hh);                             //  update Mpxb area from updated E3 area
void  Fpaint4(int px, int py, int ww, int hh);                             //  update Dpxb area from updated Mpxb 
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
void  m_zoom(GtkWidget *, cchar *);                                        //  zoom image +/-
void  KBstate(GdkEventKey *event, int state);                              //  pass dialog KB events to main app
int   KBpress(GtkWidget *, GdkEventKey *, void *);                         //  KB key press event function
int   KBrelease(GtkWidget *, GdkEventKey *, void *);                       //  KB key release event
void  win_fullscreen(int hidemenu);                                        //  full screen without menu/panel
void  win_unfullscreen();                                                  //  restore to prior size with menu etc.

//  cairo drawing functions (fotoxx main)

void  draw_pixel(int px, int py);                                          //  draw pixel
void  erase_pixel(int px, int py);                                         //  erase pixel
void  draw_line(int x1, int y1, int x2, int y2, int type);                 //  draw line or dotted line (type 1/2)
void  erase_line(int x1, int y1, int x2, int y2);                          //  erase line
void  add_topline(int x1, int y1, int x2, int y2, int type);               //  add to list of pre-set overlay lines
void  draw_toplines(int arg);                                              //  draw all pre-set overlay lines
void  draw_gridlines();                                                    //  draw grid lines on image
void  add_toptext(int ID, int px, int py, cchar *text, cchar *font);       //  add text string with ID on window
void  draw_toptext();                                                      //  draw text strings when window repainted
void  erase_toptext(int ID);                                               //  remove all text strings with ID
void  draw_text(int px, int py, cchar *text, cchar *font);                 //  draw text on window
void  add_topcircle(int px, int py, int radius);                           //  draw circle on window
void  draw_topcircles();                                                   //  draw circles when window repainted
void  erase_topcircles();                                                  //  remove all circles
void  draw_mousearc(int cx, int cy, int ww, int hh, int Ferase);           //  draw circle/ellipse around pointer
void  draw_mousearc2(int cx, int cy, int ww, int hh, int Ferase);          //  2nd circle for paint/clone         v.14.02

//  spline curve edit functions (fotoxx main)

spldat * splcurve_init(GtkWidget *frame, void func(int spc));              //  initialize spline curves
int      splcurve_adjust(void *, GdkEventButton *event, spldat *);         //  curve editing function
int      splcurve_draw(GtkWidget *, cairo_t *, spldat *);                  //  spline curve draw signal function
int      splcurve_generate(spldat *, int spc);                             //  generate data from anchor points
float    splcurve_yval(spldat *, int spc, float xval);                     //  get curve y-value
int      splcurve_load(spldat *sd, cchar *file = 0);                       //  load curve from a file
int      splcurve_save(spldat *sd, cchar *file = 0);                       //  save curve to a file

//  edit support functions (fotoxx main)

int   edit_setup(editfunc &EF);                                            //  start new edit transaction
void  edit_cancel(int keep);                                               //  cancel edit (keep zdialog etc.)
void  edit_done(int keep);                                                 //  commit edit, add undo stack
void  edit_undo();                                                         //  undo edit, back to org. image
void  edit_redo();                                                         //  redo the edit after undo
void  edit_reset();                                                        //  reset to initial status
void  edit_fullsize();                                                     //  convert to full-size pixmaps
void  edit_load_widgets(editfunc *EF, int Freload);                        //  load zdialog widgets data from a file
void  edit_save_widgets(editfunc *EF);                                     //  save zdialog widgets data to a file
void  m_undo_redo(GtkWidget *, cchar *);                                   //  undo / redo if left / right mouse click
void  m_undo(GtkWidget *, cchar *);                                        //  undo one edit
void  m_redo(GtkWidget *, cchar *);                                        //  redo one edit
void  undo_all();                                                          //  undo all edits of current image
void  redo_all();                                                          //  redo all edits of current image
void  save_undo();                                                         //  undo/redo save function
void  load_undo();                                                         //  undo/redo load function
int   checkpend(cchar *list);                                              //  check if anything busy or pending

typedef  void mcbFunc();                                                   //  callback function type
EX mcbFunc   *mouseCBfunc;                                                 //  current edit mouse function
void  takeMouse(mcbFunc func, GdkCursor *);                                //  capture mouse for edit dialog
void  freeMouse();                                                         //  free mouse for main window

//  thread support functions (fotoxx main)

typedef void * threadfunc(void *);                                         //  edit thread function
void  start_thread(threadfunc func, void *arg);                            //  start edit thread
void  signal_thread();                                                     //  signal thread, work is pending
void  wait_thread_idle();                                                  //  wait for work complete
void  wrapup_thread(int command);                                          //  wait for exit or command exit
void  wrapup_thread_noblock(int command);                                  //  same, non-blocking
void  thread_idle_loop();                                                  //  thread: wait for work or exit command
void  thread_exit();                                                       //  thread: exit unconditionally
void  start_wthread(threadfunc func, void *arg);                           //  start working thread (per proc. core)
void  exit_wthread();                                                      //  exit working thread
void  wait_wthreads(int block = 1);                                        //  wait for working threads

//  other support functions (fotoxx main)

void  m_favorites(GtkWidget *, cchar *);                                   //  graphic popup menu, user favorites
void  m_help(GtkWidget *, cchar *menu);                                    //  various help menu functions
void  save_params();                                                       //  save parameters for next session
void  load_params();                                                       //  load parameters from prior session
void  free_resources(int fkeepundo = 0);                                   //  free all allocated resources
int   vpixel(PXM *pxm, float px, float py, float *vpix);                   //  get virtual pixel at (px,py)
int   sigdiff(float d1, float d2, float signf);                            //  test for significant difference

//  PXM and PXB pixmap functions (fotoxx main)

void  PXM_audit(PXM *pxm);                                                 //  audit contents of a PXM pixmap
PXM * PXM_make(int ww, int hh);                                            //  create a PXM pixmap
void  PXM_free(PXM *&pxm);                                                 //  free PXM pixmap
PXM * PXM_copy(PXM *pxm);                                                  //  copy PXM pixmap
PXM * PXM_copy_area(PXM *pxm, int orgx, int orgy, int ww, int hh);         //  copy section of PXM pixmap
void  PXM_fixblue(PXM *pxm);                                               //  set blue = 0 pixels to blue = 2
PXM * PXM_rescale(PXM *pxm, int ww, int hh);                               //  rescale PXM pixmap (ww/hh)
PXM * PXM_rotate(PXM *pxm, float angle);                                   //  rotate PXM pixmap

PXB * PXB_make(int ww, int hh);                                            //  create a PXB pixmap
PXB * PXB_make(GdkPixbuf *pixbuf);                                         //  create a PXB pixmap from a pixbuf
void  PXB_free(PXB *&pxb);                                                 //  free PXB pixmap
PXB * PXB_copy(PXB *pxb);                                                  //  copy PXB
PXB * PXB_rescale(PXB *pxb, int ww, int hh);                               //  rescale PXB pixmap (ww/hh)

PXB * PXM_PXB_copy(PXM *pxm);                                              //  PXM to pixbuf, same scale
void  PXM_PXB_update(PXM *, PXB *, int px3, int py3, int ww3, int hh3);    //  update PXB area from PXM, same scale
void  PXB_PXB_update(PXB *, PXB *, int px3, int py3, int ww3, int hh3);    //  update PXB area from PXB, any scale

//  window and menu builder (f.wdgets.cc)

void  build_widgets();                                                     //  build widgets and menus for F/G/W view modes
void  m_viewmode(GtkWidget *, cchar *fgw);                                 //  set current F/G/W view mode
void  popup_menufunc(GtkWidget *, cchar *menu);                            //  image/thumb right-click menu func
void  image_Rclick_popup(int Nth);                                         //  popup menu for image right-click
void  gallery_Lclick_func(int Nth);                                        //  function for thumbnail left-click
void  gallery_Rclick_popup(int Nth);                                       //  popup menu for thumbnail right-click

//  file menu functions (f.file.cc)

void  m_clone(GtkWidget *, cchar *);                                       //  start parallel fotoxx instance
void  m_open(GtkWidget *, cchar *);                                        //  open image file in same window
void  m_open_drag(int x, int y, char *file);                               //  open drag-drop file
void  m_previous(GtkWidget *, cchar *);                                    //  open previously accessed file
void  m_recentfiles(GtkWidget *, cchar *);                                 //  open recently accessed file
void  m_newfiles(GtkWidget *, cchar *);                                    //  open newly added file
void  m_prev(GtkWidget *, cchar *);                                        //  open previous file in gallery
void  m_next(GtkWidget *, cchar *);                                        //  open next file in gallery
void  m_prev_next(GtkWidget *, cchar *);                                   //  open prev/next if left/right icon click
void  m_ufraw(GtkWidget *, cchar *);                                       //  open, edit camera RAW file with ufraw
void  m_rawtherapee(GtkWidget *, cchar *);                                 //  open, edit camera RAW file with Raw Therapee
void  m_file_save(GtkWidget *, cchar *);                                   //  save modified image file to didk
void  m_rename(GtkWidget *, cchar *);                                      //  rename an image file (same location)
void  m_create(GtkWidget *, cchar *);                                      //  create new blank image file, dialog
int   create_blank_file(cchar *file, int ww, int hh, int RGB[3]);          //  create new blank image file, callable
void  m_copytoloc(GtkWidget *, cchar *);                                   //  copy an image file to a new location
void  m_movetoloc(GtkWidget *, cchar *);                                   //  move an image file to a new location
void  copy_move(cchar *menu);                                              //  copy/move shared function
void  m_trash(GtkWidget *, cchar *);                                       //  move an image file to trash
void  m_delete(GtkWidget *, cchar *);                                      //  delete an image file (undo not poss.)
void  m_print(GtkWidget *, cchar *);                                       //  print an image file
void  m_quit(GtkWidget *, cchar *);                                        //  exit application with checks
void  quitxx();                                                            //  exit unconditionally
void  copyto_clip(GtkWidget *, cchar *file);                               //  copy an image file to the clipboard
void  add_recent_file(cchar *file);                                        //  add file to recent file list
void  set_mwin_title();                                                    //  main window title = curr_file path
int   find_imagefiles(cchar *dirk, char **&flist, int &Nfiles,             //  find all image files within a given
                                   int Fthumb, int Finit = 1);             //    directory (thumbnails optional)

//  image file load and save functions (f.file.cc)

char * raw_to_tiff(cchar *rawfile);                                        //  get corresp. tiff file for a RAW file
void   f_preload(int next);                                                //  start preload of next file
int    f_open(cchar *file, int Nth = 0, int keep = 0,                      //  open new current image file
                           int Fack = 1, int zoom = 0);
int    f_open_saved();                                                     //  open saved file, retain undo/redo stack
int    f_save(char *outfile, cchar *type, int bpc, int Fack = 1);          //  save current image file to disk

PXB * PXB_load(cchar *filespec, int fack = 0);                             //  load file to PXB pixmap, 8 bpc
PXM * PXM_load(cchar *filespec, int fack = 0);                             //  load file to PXM pixmap, 8/16 bpc

PXB * TIFF_PXB_load(cchar *filespec);                                      //  TIFF file to PXB, 8 bpc
PXM * TIFF_PXM_load(cchar *filespec);                                      //  TIFF file to PXM, 8/16 bpc
int   PXM_TIFF_save(PXM *pxm, cchar *filespec, int bpc);                   //  PXM to TIFF file, 8/16 bpc

PXB * PNG_PXB_load(cchar *filespec);                                       //  PNG file to PXB, 8 bpc
PXM * PNG_PXM_load(cchar *filespec);                                       //  PNG file to PXM, 8/16 bpc
int   PXM_PNG_save(PXM *pxm, cchar *filespec, int bpc);                    //  PXM to PNG file, 8/16 bpc

PXB * ANY_PXB_load(cchar *filespec);                                       //  ANY file to PXB, 8 bpc
PXM * ANY_PXM_load(cchar *filespec);                                       //  ANY file to PXM, 8 bpc
int   PXM_ANY_save(PXM *pxm, cchar *filespec);                             //  PXM to ANY file, 8 bpc

PXB * RAW_PXB_load(cchar *filespec);                                       //  RAW file to PXB, 8 bpc
PXM * RAW_PXM_load(cchar *filespec);                                       //  RAW file to PXM, 16 bpc

//  thumbnail gallery and navigation functions (f.file.cc)

char * gallery(cchar *filez, cchar *action, int Nth = 0);                  //  display image gallery window, navigate
void   set_gwin_title();                                                   //  main window title = gallery name
char * gallery_find(int Nth);                                              //  return Nth file in curr. gallery
int    gallery_position(cchar *file, int Nth);                             //  get rel. position of file in gallery
int    image_file_type(cchar *file);                                       //  get file type 1-5: dirk/image/RAW/thumb/other
char * thumb2imagefile(cchar *thumbfile);                                  //  get image file for thumbnail file
char * image2thumbfile(cchar *imagefile);                                  //  get thumbnail file for image file
char * image_thumbfile(char *imagefile, int *ind = 0);                     //  get thumbnail filespec, create if missing
GdkPixbuf * image_thumbnail(char *imagefile, int size = 0);                //  get thumbnail pixbuf, create if nec.
GdkPixbuf * get_thumbnail_pixbuf(char *imagefile, int size);               //  make thumbnail pixbuf from image file
void   popimage(int Fnewin);                                               //  popup big image of clicked thumbnail
void   gallery_monitor(cchar *action);                                     //  start/stop gallery directory monitor
char ** gallery_getfiles(char **initfiles = 0);                            //  select files from gallery window
void   gallery_getfiles_Lclick_func(int Nth);                              //  get files, thumbnail left-click
void   gallery_getfiles_Rclick_func(int Nth);                              //  get files, thumbnail right-click

//  tools menu functions (f.tools.cc)

void  m_index(GtkWidget *, cchar *);                                       //  rebuild image index and thumbnails
void  index_rebuild(int menu);                                             //  index rebuild function
void  m_settings(GtkWidget *, cchar *);                                    //  user settings
void  m_edit_bookmarks(GtkWidget *, cchar *);                              //  edit bookmarks 
void  bookmarks_Lclick_func(int Nth);                                      //  thumbnail click response function
void  m_goto_bookmark(GtkWidget *, cchar *);                               //  select bookmarked image, goto gallery posn
void  m_KBshortcuts(GtkWidget *, cchar *);                                 //  edit KB shortcuts, update file
int   KBshortcuts_load();                                                  //  load KB shortcuts at startup time
void  m_manage_coll(GtkWidget *, cchar *);                                 //  create and edit image collections
void  conv_collections(char **, char **, int);                             //  convert after files renamed/moved  v.14.09
void  coll_drag_start(int Nth);                                            //  rearrange - get image to move      v.14.08
void  coll_drag_drop(int Nth);                                             //  rearrange - put image here
void  coll_drag_kill();                                                    //  abort drag/drop operation
void  coll_popup_remove(GtkWidget *, cchar *);                             //  remove image from collection
void  coll_popup_cutfile(GtkWidget *, cchar *);                            //  remove image from collection, add to cache
void  coll_popup_copyfile(GtkWidget *, cchar *);                           //  add image in collection to cache
void  coll_popup_pastecache(GtkWidget *, cchar *);                         //  paste image cache to collection position
void  m_slideshow(GtkWidget *, cchar *);                                   //  enter or leave slideshow mode
void  slideshow_Lclick_func(int Nth);                                      //  slideshow thumbnail click function
void  slideshow_next(cchar *);                                             //  show prev/next image
void  m_batch_convert(GtkWidget *, cchar *);                               //  rename/convert/resize/export image files
void  m_batch_upright(GtkWidget *, cchar *);                               //  upright rotated image files
void  m_batch_dcraw(GtkWidget *, cchar *);                                 //  convert RAW files to tiff using DCraw
void  m_batch_rawtherapee(GtkWidget *, cchar *);                           //  convert RAW files to tiff using Raw Therapee
int   batch_sharp_func(PXM *pxm, int amount, int thresh);                  //  callable sharpen func used by batch funcs
void  m_show_brdist(GtkWidget *, cchar *);                                 //  show brightness distribution graph
void  brdist_drawgraph(GtkWidget *drawin, cairo_t *cr, int *);             //  draw brightness distribution graph
void  brdist_drawscale(GtkWidget *drawarea, cairo_t *cr, int *);           //  draw brightness scale, black to white stripe
void  m_gridlines(GtkWidget *, cchar *);                                   //  grid lines setup dialog
void  toggle_grid(int action);                                             //  set grid off/on or toggle (0/1/2)
void  choose_grid(int grid);                                               //  choose current grid settings
void  m_fg_color(GtkWidget *, cchar *);                                    //  set foreground line color (area, mouse, trim ...)
void  m_show_RGB(GtkWidget *, cchar *);                                    //  show RGB values at mouse click
void  m_darkbrite(GtkWidget *, cchar *);                                   //  highlight the darkest and brightest pixels
void  darkbrite_paint();                                                   //  paint function called from Fpaint()
void  m_duplicates(GtkWidget *, cchar *);                                  //  find duplicate image files
void  m_moncheck(GtkWidget *, cchar *);                                    //  check monitor brightness and color
void  temp_image(cchar *file, cchar *title, cchar *message);               //  show temp. image, return to curr. image
void  m_mongamma(GtkWidget *, cchar *);                                    //  adjust monitor gamma
void  m_lang(GtkWidget *, cchar *);                                        //  change language
void  m_untranslated(GtkWidget *, cchar *);                                //  report missing translations
void  m_menu_launcher(GtkWidget *, cchar *);                               //  make desktop menu entry and launcher
void  m_burn(GtkWidget *, cchar *);                                        //  burn selected images to CD/DVD
void  m_resources(GtkWidget *, cchar *);                                   //  report CPU and memory usage
void  m_zappcrash(GtkWidget *, cchar *);                                   //  deliberate zappcrash (test function)

//  metadata menu functions (EXIF, IPTC, etc.) (f.meta.cc) 

void  m_meta_view_short(GtkWidget *, cchar *);                             //  view selected EXIF/IPTC data
void  m_meta_view_long(GtkWidget *, cchar *);                              //  view all EXIF/IPTC data
void  m_meta_view_capcomm(GtkWidget *, cchar *);                           //  view captions and comments only
void  meta_view(int length);                                               //  view EXIF/IPTC data in popup window
void  m_captions(GtkWidget *, cchar *);                                    //  show caption/comments at the top 
void  m_edit_metadata(GtkWidget *, cchar *);                               //  edit date/rating/tags dialog
void  manage_tags();                                                       //  manage tags dialog function
void  m_meta_edit_any(GtkWidget *, cchar *);                               //  add or change any EXIF/IPTC data
void  m_meta_delete(GtkWidget *, cchar *);                                 //  delete EXIF/IPTC data
void  m_batchTags(GtkWidget *, cchar *);                                   //  add/remove tags in multiple image files
void  load_filemeta(cchar *file);                                          //  EXIF/IPTC >> tags_* in memory
void  save_filemeta(cchar *file);                                          //  tags_* in memory >> EXIF/IPTC
void  update_image_index(cchar *file);                                     //  update image index file 
void  delete_image_index(cchar *file);                                     //  delete entry from image index file

void  m_download_geolocs(GtkWidget *, cchar *);                            //  download world map and geolocations files
int   init_geolocs();                                                      //  initz. geolocations data
int   load_worldmap();                                                     //  load world map image
void  free_worldmap();                                                     //  free huge memory for world map
void  m_worldmap_test(GtkWidget *, cchar *);                               //  test world map coordinate conversions
void  m_edit_geotags(GtkWidget *, cchar *);                                //  edit geotags in image file EXIF
void  m_batch_add_geotags(GtkWidget *, cchar *menu);                       //  batch add geotags
void  m_geotag_groups(GtkWidget *, cchar *);                               //  report images grouped by date, geotags
void  m_geotag_worldmap(GtkWidget *, cchar *);                             //  report images by clicking world map
void  m_search_images(GtkWidget *, cchar *);                               //  search image metadata

char ** exif_get(cchar *file, cchar **keys, int nkeys);                    //  get EXIF/IPTC data for given key(s)
int     exif_put(cchar *file, cchar **keys, cchar **text, int nkeys);      //  put EXIF/IPTC data for given key(s)
int     exif_copy(cchar *f1, cchar *f2, cchar **k, cchar **t, int nk);     //  copy EXIF/IPTC data from file to file
void    exif_tagdate(cchar *exifdate, char *tagdate);                      //  yyyy:mm:dd:hh:mm:ss to yyyymmddhhmmss
void    tag_exifdate(cchar *tagdate, char *exifdate);                      //  yyyymmddhhmmss to yyyy:mm:dd:hh:mm:ss
char ** exiftool_server(char **inputs);                                    //  run exiftool as a server process

int  get_sxrec(sxrec_t &sxrec, cchar *file);                               //  get image index rec.
int  get_sxrec_min(cchar *file, char *fdate, char *pdate, char *size);     //   " min. for gallery paint
int  put_sxrec(sxrec_t *sxrec, cchar *file);                               //  add/update image index rec.
int  read_sxrec_seq(sxrec_t &sxrec, int &ftf);                             //  read image index recs. 1-last
int  write_sxrec_seq(sxrec_t *sxrec, int &ftf);                            //  write image index recs. 1-last
int  image_fcomp(cchar *file1, cchar *file2);                              //  file name compare in image index sequence

//  select area menu functions (f.area.cc)

void  m_select(GtkWidget *, cchar *);                                      //  select area within image
void  m_select_show(GtkWidget *, cchar *);                                 //  enable area for subsequent edits
void  m_select_hide(GtkWidget *, cchar *);                                 //  show area outline
void  m_select_enable(GtkWidget *, cchar *);                               //  hide area outline
void  m_select_disable(GtkWidget *, cchar *);                              //  disable area
void  m_select_invert(GtkWidget *, cchar *);                               //  invert area 
void  m_select_unselect(GtkWidget *, cchar *);                             //  unselect area 
void  m_select_copy(GtkWidget *, cchar *);                                 //  copy area, save in memory
void  m_select_paste(GtkWidget *, cchar *);                                //  paste saved area into image
void  m_select_open(GtkWidget *, cchar *);                                 //  open an area file, save in memory
void  m_select_save(GtkWidget *, cchar *);                                 //  save area to a file
void  m_select_save_png(GtkWidget *, cchar *);                             //  save area to a PNG file with transparency

void  sa_geom_mousefunc();                                                 //  select rectangle or ellipse
void  sa_draw_mousefunc();                                                 //  line drawing functions
int   sa_nearpix(int mx, int my, int rad, int &px, int &py, int fx);       //  find nearest pixel
void  sa_draw_line(int px1, int py1, int px2, int py2);                    //  draw a connected line
void  sa_draw1pix(int px, int py);                                         //  add one pixel to select area
void  sa_mouse_select();                                                   //  select by mouse (opt. color match)
void  sa_nextseq();                                                        //  start next sequence number
void  sa_unselect_pixels();                                                //  remove current selection
void  sa_finish();                                                         //  finish - map interior pixels
void  sa_finish_auto();                                                    //  finish - interior pixels already known
void  sa_unfinish();                                                       //  set finished area back to edit mode
void  sa_map_pixels();                                                     //  map edge and interior pixels
void  sa_show(int flag);                                                   //  show or hide area outline
void  sa_show_rect(int px1, int py1, int ww, int hh);                      //  show area outline inside a rectangle
int   sa_validate();                                                       //  validate area for curr. image context
void  sa_enable();                                                         //  enable area
void  sa_disable();                                                        //  disable area
void  sa_invert();                                                         //  invert area
void  sa_unselect();                                                       //  unselect area
void  sa_edgecalc();                                                       //  calculate edge distances
void  sa_edgecreep(int);                                                   //  adjust area edge +/-
   
//  image edit functions - Image menu (f.edit.cc)

void  m_trimrotate(GtkWidget *, cchar *);                                  //  combined trim/rotate function
void  m_rotate(GtkWidget *, cchar *);                                      //  rotate image by any angle
void  m_trim(GtkWidget *, cchar *);                                        //  trim image
void  m_rotate90(GtkWidget *, cchar *);                                    //  rotate image +/-90 degrees
void  m_rotate90_thumb(GtkWidget *, cchar *);                              //  same for gallery thumb popup menu
void  m_autotrim(GtkWidget *, cchar *);                                    //  autotrim image after rotate, warp
void  m_voodoo(GtkWidget *, cchar *);                                      //  automatic image retouch
void  m_combo(GtkWidget *, cchar *);                                       //  combo function, brightness/contrast/color
void  m_tonemap(GtkWidget *, cchar *);                                     //  tone mapping
void  m_adjust_brdist(GtkWidget *, cchar *);                               //  adjust brightness distribution
void  m_resize(GtkWidget *, cchar *);                                      //  resize image
void  m_flip(GtkWidget *, cchar *);                                        //  flip horizontally or vertically
void  m_write_text(GtkWidget *, cchar *);                                  //  write text on image
void  load_text(zdialog *zd);                                              //  load text and attributes from a file
void  save_text(zdialog *zd);                                              //  save text and attributes to a file
int   gentext(textattr_t *attr);                                           //  generate text image from attributes
void  m_write_line(GtkWidget *, cchar *);                                  //  write line/arrow on image
int   genline(lineattr_t *attr);                                           //  generate line/arrow image from attributes
void  m_brightramp(GtkWidget *, cchar *);                                  //  ramp brightness across image
void  m_paint_edits(GtkWidget *, cchar *);                                 //  select and edit in parallel
void  m_lever_edits(GtkWidget *, cchar *);                                 //  leverage edits by pixel bright/color
void  m_edit_plugins(GtkWidget *, cchar *);                                //  add and remove plugin functions
void  m_run_plugin(GtkWidget *, cchar *);                                  //  perform a plugin edit function

//  image edit functions - Repair menu (f.repair.cc)

void  m_sharpen(GtkWidget *, cchar *);                                     //  sharpen image
void  m_blur(GtkWidget *, cchar *);                                        //  blur image
void  m_denoise(GtkWidget *, cchar *);                                     //  image noise reduction
void  m_smart_erase(GtkWidget *, const char *);                            //  smart erase object
void  m_redeye(GtkWidget *, cchar *);                                      //  red-eye removal
void  m_paint_clone(GtkWidget *, cchar *);                                 //  paint pixels with the mouse
void  m_dust(GtkWidget *, const char *);                                   //  remove dust
void  m_antialias(GtkWidget *, const char *);                              //  anti-alias
void  m_chromatic(GtkWidget *, const char *);                              //  reduce chromatic abberation
void  m_stuckpix(GtkWidget *, const char *);                               //  fix stuck pixels
void  m_shift_colors(GtkWidget *, cchar *);                                //  shift one color into another color
void  m_colormode(GtkWidget *, cchar *);                                   //  B+W/color, negative/positive, sepia
void  m_colorprof(GtkWidget *, cchar *);                                   //  convert to another color profile
void  m_match_color(GtkWidget *, cchar *);                                 //  set image2 colors to match image1
void  m_revise_RGB(GtkWidget *, cchar *);                                  //  revise RGB values
void  m_CMYK(GtkWidget *, cchar *);                                        //  brightness / color adjust with OD units

//  image edit functions - Bend menu (f.bend.cc)

void  m_unbend(GtkWidget *, cchar *);                                      //  fix perspective problems
void  m_perspective(GtkWidget *, cchar *);                                 //  warp tetragon into rectangle
void  m_warp_area(GtkWidget *, cchar *);                                   //  warp image within an area
void  m_warp_curved(GtkWidget *, cchar *);                                 //  warp image, curved transform
void  m_warp_linear(GtkWidget *, cchar *);                                 //  warp image, linear transform
void  m_warp_affine(GtkWidget *, cchar *);                                 //  warp image, affine transform
void  m_flatbook(GtkWidget *, cchar *);                                    //  flatten a photographed book page

//  image edit functions - Effects menu (f.effects.cc)

void  m_colordep(GtkWidget *, cchar *);                                    //  set color depth 1-16 bits/color
void  m_sketch(GtkWidget *, cchar *);                                      //  convert image to pencil sketch
void  m_linedraw(GtkWidget *, cchar *);                                    //  comvert image to outline drawing
void  m_colordraw(GtkWidget *, cchar *);                                   //  convert image to solid color drawing
void  m_gradblur(GtkWidget *, cchar *);                                    //  graduated blur tool
void  m_emboss(GtkWidget *, cchar *);                                      //  convert image to simulated embossing
void  m_tiles(GtkWidget *, cchar *);                                       //  convert image to tile array (pixelate)
void  m_dots(GtkWidget *, cchar *);                                        //  convert image to dot array (Roy Lichtenstein)
void  m_painting(GtkWidget *, cchar *);                                    //  convert image to simulated painting
void  m_vignette(GtkWidget *, cchar *);                                    //  vignette tool
void  m_texture(GtkWidget *, cchar *);                                     //  add texture to image
void  m_pattern(GtkWidget *, cchar *);                                     //  tile image with a pattern
void  m_mosaic(GtkWidget *, cchar *);                                      //  create mosaic using tiles made from images
void  m_anykernel(GtkWidget *, cchar *);                                   //  operate on image using a custom kernel
void  m_waves(GtkWidget *, cchar *);                                       //  warp image using a wave pattern

//  image edit functions - Combine menu (f.combine.cc)

void  m_HDR(GtkWidget *, cchar *);                                         //  make HDR composite image
void  m_HDF(GtkWidget *, cchar *);                                         //  make HDF composite image
void  m_STP(GtkWidget *, cchar *);                                         //  stack / paint image
void  m_STN(GtkWidget *, cchar *);                                         //  stack / noise reduction
void  m_pano(GtkWidget *, cchar *);                                        //  make panorama composite image
void  m_vpano(GtkWidget *, cchar *);                                       //  make vertical panorama
void  m_mashup(GtkWidget *, cchar *);                                      //  arrange images on a background image

//  translatable strings used in multiple dialogs

#define Badd ZTX("Add")
#define Baddall ZTX("Add All")
#define Ball ZTX("All")
#define Bamount ZTX("Amount")
#define Bangle ZTX("Angle")
#define Bapply ZTX("Apply")
#define Bautocomp ZTX("Autocomplete")
#define Bblack ZTX("Black")
#define Bblendwidth ZTX("Blend Width")
#define Bblue ZTX("Blue")
#define Bbottom ZTX("bottom")
#define Bbrightness ZTX("Brightness")
#define Bbrowse ZTX("Browse")
#define Bcancel ZTX("Cancel")
#define Bcenter ZTX("center")
#define Bchoose ZTX("Choose")
#define Bclear ZTX("Clear")
#define Bcolor ZTX("Color")
#define Bcontrast ZTX("Contrast")
#define Bcopy ZTX("Copy")
#define Bcreate ZTX("Create")
#define Bcurvefile ZTX("Curve File:")
#define Bcut ZTX("Cut")
#define Bdelete ZTX("Delete")
#define Bdisable ZTX("Disable")
#define Bdist ZTX("Dist")
#define Bdone ZTX("Done")
#define Bedge ZTX("edge")
#define Bedit ZTX("Edit")
#define Benable ZTX("Enable")
#define Berase ZTX("Erase")
#define Bfetch ZTX("Fetch")
#define Bfileselected ZTX("%d files selected")
#define Bfind ZTX("Find")
#define Bfinish ZTX("Finish")
#define Bfont ZTX("Font")
#define Bgeotags ZTX("Geotags")
#define Bgreen ZTX("Green")
#define Bgrid ZTX("Grid")
#define Bheight ZTX("Height")
#define Bhelp ZTX("Help")
#define Bhide ZTX("Hide")
#define Bimages ZTX("Images")
#define Binsert ZTX("Insert")
#define Binvert ZTX("Invert")
#define Bkeep ZTX("Keep") 
#define Bleft ZTX("left")
#define Blimit ZTX("limit")
#define Bload ZTX("Load")
#define Bmake ZTX("Make")
#define Bmanagetags ZTX("Manage Tags") 
#define Bmap ZTX("Map")
#define Bmax ZTX("Max")
#define Bnegative ZTX("Negative")
#define Bnew ZTX("New")
#define Bnext ZTX("Next")
#define Bno ZTX("No")
#define Bnoimages ZTX("no images")
#define Bnofileselected ZTX("no files selected")
#define Bnoselection ZTX("no selection")
#define BOK ZTX("OK")
#define Bopen ZTX("Open")
#define Bpaste ZTX("Paste")
#define Bpause ZTX("Pause")
#define Bpercent ZTX("Percent")
#define Bpower ZTX("Power")
#define Bpresets ZTX("Presets")
#define Bprev ZTX("Prev")
#define Bproceed ZTX("Proceed")
#define Bradius ZTX("Radius")
#define Brange ZTX("range")
#define Bred ZTX("Red")
#define Bredo ZTX("Redo")
#define Breduce ZTX("Reduce")
#define Bremove ZTX("Remove")
#define Brename ZTX("Rename")
#define Breset ZTX("Reset")
#define Bright ZTX("right")
#define Brotate ZTX("Rotate")
#define Bsave ZTX("Save")
#define Bsavetoedit ZTX("Unknown file type, save as tiff/jpeg/png to edit")
#define Bsearch ZTX("Search")
#define Bseconds ZTX("Seconds")
#define Bselect ZTX("Select")
#define Bselectfiles ZTX("Select Files")
#define Bshow ZTX("Show")
#define Bsize ZTX("Size")
#define Bstart ZTX("Start")
#define Bstrength ZTX("Strength")
#define Bthresh ZTX("Threshold")
#define Btoomanyfiles ZTX("exceed %d files")
#define Btrash ZTX("Trash")
#define Btrim ZTX("Trim")
#define Bundoall ZTX("Undo All")
#define Bundolast ZTX("Undo Last")
#define Bundo ZTX("Undo")
#define Bunfinish ZTX("Unfinish")
#define Bunselect ZTX("Unselect")
#define Bview ZTX("View")
#define Bweb ZTX("Web")
#define Bwhite ZTX("White")
#define Bwidth ZTX("Width")
#define Bxoffset ZTX("x-offset")
#define Byoffset ZTX("y-offset")
#define Byes ZTX("Yes")


