/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************

   Fotoxx image editor - image metadata functions.
   View and edit ANY metadata: EXIF, IPTC, etc. 

   m_meta_view_short          metadata short report
   m_meta_view_long           report all metadata
   m_meta_view_capcomm        report caption and comments
   meta_view                  callable function
   m_captions                 write captions and comments at top of current image
   m_edit_metadata            primary edit metadata dialog
   pdate_metadate             convert yyyy-mm-dd to yyyymmdd
   pdate_metatime             convert hh:mm:ss to hhmmss
   metadate_pdate             convert yyyymmddhhmmss to yyyy-mm-dd and hh:mm:ss
   manage_tags                maintain list of defined tags
   add_tag                    add tag to a tag list
   del_tag                    remove tag from a tag list
   add_recentag               add tag to recent tags list, remove oldest if needed
   load_deftags               load defined tags list from tags file and image index
   save_deftags               save defined tags list to tags file
   find_deftag                check if given tag is in defined tags list
   add_deftag                 add new tag to defined tags list or change category
   del_deftag                 remove tag from defined tags list
   deftags_stuff              stuff defined tags into dialog text widget
   tag_orphans                report tags defined and not used in any image file
   m_meta_edit_any            dialog to fetch and save any image file metadata by name
   m_meta_delete              dialog to delete any image file metadata by name
   m_batch_tags               batch add and delete tags - selected images
   load_filemeta              load image file metadata into memory (indexed data only)
   save_filemeta              save metadata to image file EXIF and to image index
   update_image_index         update index data for current image file
   delete_image_index         delete index record for deleted image file
   validate_latilongi         validate latitude/longitude data
   m_download_geolocs         download world city geolocations and world map
   init_geolocs               load geolocations from index file and geolocations file
   get_geolocs                get latitude/longitude for a city or location
   put_geolocs                save geolocation in image file EXIF and image index file
   load_worldmap              load world map into memory, add red dots for mapped images
   get_worldmap_coordinates   convert map pixel position to latitude/longitude
   get_worldmap_position      convert latitude/longitude to map pixel position
   click_worldmap             show world map and connect mouse clicks to caller function
   m_edit_geotags             edit image city/location, country, latitude, longitude
   m_batch_add_geotags        add given geotag to selected set of images
   geotags_choosecity         get possible location matches for partial input, choose one
   web_geocode                use web geocoding service to map location to latitude/longitude
   m_geotag_groups            list locations/image counts/time groups, choose one for gallery
   m_geotag_worldmap          show world map, show gallery of images at mouse click position
   m_search_images            general image search using file names and any metadata
   exif_get                   get image metadata from list of keys
   exif_put                   update image metadata from list of keys and data
   exif_copy                  copy metadata from one image to another, with revisions
   exiftool_server            start exiftool server process, send data requests
   init_image_index_map       initialization for image file index read/write
   get_sxrec                  get image index record for image file
   get_sxrec_min              get index data used for gallery view
   put_sxrec                  add or update index record for an image file
   read_sxrec_seq             read all index records sequentially, one per call
   write_sxrec_seq            write all index records sequentially

***************************************************************************/

#define EX extern                                                          //  enable extern declarations
#include "fotoxx.h"                                                        //  (variables in fotoxx.h are refs)

/**************************************************************************/


char  *pdate_metadate(cchar *pdate);                                       //  "yyyy-mm-dd" to "yyyymmdd"
char  *ptime_metatime(cchar *ptime);                                       //  "hh:mm" to "hhmm"
void  metadate_pdate(cchar *metadate, char *pdate, char *ptime);           //  "yyyymmddhhmm" to "yyyy-mm-dd" and "hh:mm"
int   get_mouse_tag(GtkTextView *, int px, int py, cchar *);               //  get tag selected by mouse
int   add_tag(char *tag, char *taglist, int maxcc);                        //  add tag if unique and enough space
int   del_tag(char *tag, char *taglist);                                   //  remove tag from tag list
int   add_recentag(char *tag);                                             //  add tag to recent tags, keep recent
void  load_deftags();                                                      //  tags_defined file >> tags_deftags[]
void  save_deftags();                                                      //  tags_deftags[] >> defined_tags file
int   find_deftag(char *tag);                                              //  find tag in tags_deftags[]
int   add_deftag(char *catg, char *tag);                                   //  add tag to tags_deftags[]
int   del_deftag(char *tag);                                               //  remove tag from tags_deftags[]
void  deftags_stuff(zdialog *zd);                                          //  tags_deftags[] >> dialog widget

char  meta_date[16] = "";                                                  //  image date, yyyymmddhhmmss
char  meta_prdate[16] = "";                                                //  previous image date read or set
char  meta_rating[4] = "0";                                                //  image rating in stars, "0" to "5"
char  meta_size[16] = "";                                                  //  image size, "NNNNxNNNN"
char  *tags_deftags[maxtagcats];                                           //  defined tags: catg: tag1, ... tagN,
char  tags_deftext[tagTcc] = "";                                           //  defined tags as flat text buffer
char  tags_imagetags[tagFcc] = "";                                         //  tags for current image file
char  tags_recentags[tagRcc] = "";                                         //  recently added tags list
char  tags_searchtags[tagScc] = "";                                        //  search tags list
char  tags_searchtext[tagScc] = "";                                        //  search comments & captions word list
char  meta_searchfiles[tagScc] = "";                                       //  search files list
char  meta_comments[exif_maxcc];                                           //  image comments            expanded
char  meta_caption[exif_maxcc];                                            //  image caption
char  meta_city[100], meta_country[100];                                   //  geolocs: city, country
char  meta_latitude[20], meta_longitude[20];                               //  geolocs: lati/longitude (-123.4567)


/**************************************************************************/

//  menu function and popup dialog to show EXIF/IPTC data
//  window is updated when navigating to another image

void m_meta_view_short(GtkWidget *, cchar *menu)
{
   if (menu) F1_help_topic = "view_meta";
   meta_view(1);
   return;
}

void m_meta_view_long(GtkWidget *, cchar *menu)
{
   if (menu) F1_help_topic = "view_meta";
   meta_view(2);
   return;
}

void m_meta_view_capcomm(GtkWidget *, cchar *menu)
{
   if (menu) F1_help_topic = "view_meta";
   meta_view(3);
   return;
}

void meta_view(int arg)
{
   int   meta_view_dialog_event(zdialog *zd, cchar *event);

   static int     reportype = 1;
   char           *buff;
   static char    *file = 0;
   int            contx = 0;
   GtkWidget      *widget;
   cchar          *DTformat = "%%Y-%%m-%%d %%H:%%M";                       //  exiftool date/time format

   F1_help_topic = "view_meta";
   
   if (file) zfree(file);
   file = 0;
   
   if (clicked_file) {                                                     //  use clicked file if present
      file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                     //  else current file
      file = zstrdup(curr_file);
   else return;

   if (arg > 0) reportype = arg;                                           //  change report type

   if (! zdexifview)                                                       //  popup dialog if not already
   {
      zdexifview = zdialog_new(ZTX("View Metadata"),Mwin,Bcancel,null);
      zdialog_add_widget(zdexifview,"scrwin","scroll","dialog",0,"expand");
      zdialog_add_widget(zdexifview,"text","exifdata","scroll",0,"expand|wrap");
      if (reportype == 1) zdialog_resize(zdexifview,400,400);
      else zdialog_resize(zdexifview,600,600);
      zdialog_run(zdexifview,meta_view_dialog_event);
   }

   widget = zdialog_widget(zdexifview,"exifdata");
   gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),0);                    //  disable widget editing
   wclear(widget);

   if (reportype == 1)                                                     //  short report
   {
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_NONE);    //  disable text wrap

      snprintf(command,ccc,"exiftool -S -d \"%s\" -common "                //  exiftool command
         "-%s -%s -%s -%s -%s -%s -%s -%s -%s -%s \"%s\" ", DTformat, 
         iptc_keywords_key, iptc_rating_key, exif_editlog_key, exif_comment_key, 
         iptc_caption_key, exif_focal_length_key, exif_city_key, exif_country_key, 
         exif_latitude_key, exif_longitude_key, file);
   }
   
   else if (reportype == 2)                                                //  long report
   {
      gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_NONE);
      snprintf(command,ccc,"exiftool -s -e -d \"%s\" \"%s\" ", DTformat, file);
   }
   
   else if (reportype == 3) goto report3;                                  //  captions/comments only
   
   else return;
   
   while ((buff = command_output(contx,command))) {                        //  run command, output into window
      wprintf(widget,"%s\n",buff);
      zfree(buff);
   }

   command_status(contx);                                                  //  free resources
   return;

// ------------------------------------------------------------------------

   report3:                                                                //  captions/comments report

   char     **keyvals;
   cchar    *keynames[2] = { iptc_caption_key, exif_comment_key };

   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_WORD);       //  enable word wrap

   keyvals = exif_get(file,keynames,2);                                    //  get captions and comments
   
   wprintf(widget,ZTX("Caption: \n"),0);
   if (keyvals[0]) wprintf(widget,"%s \n",keyvals[0]);
   wprintf(widget,ZTX("\nComment: \n"),0);
   if (keyvals[1]) wprintf(widget,"%s \n",keyvals[1]);

   return;
}


//  dialog event and completion callback function

int meta_view_dialog_event(zdialog *zd, cchar *event)                      //  kill dialog
{
   if (! zd->zstat) return 0;
   zdialog_free(zdexifview);
   return 0;
}


/**************************************************************************/

//  Show captions and comments on top of current image in main window.
//  Menu call (menu arg not null): toggle switch on/off.
//  Non-menu call: write caption/comment on image if switch is ON.

void m_captions(GtkWidget *, cchar *menu)
{
   cchar        *keynames[2] = { iptc_caption_key, exif_comment_key };
   char         **keyvals;
   char         caption[200], comment[200];
   static char  text[402];

   F1_help_topic = "show_captions";

   if (menu) {                                                             //  if menu call, flip toggle
      Fcaptions = 1 - Fcaptions;
      if (curr_file) f_open(curr_file);
      return;
   }
   
   if (! Fcaptions) return;
   
   if (! curr_file) return;
   *caption = *comment = 0;

   keyvals = exif_get(curr_file,keynames,2);                               //  get captions and comments metadata

   if (keyvals[0]) {
      strncpy0(caption,keyvals[0],200);
      zfree(keyvals[0]);
   }

   if (keyvals[1]) {
      strncpy0(comment,keyvals[1],200);
      zfree(keyvals[1]);
   }
   
   *text = 0;

   if (*caption) strcpy(text,caption);
   if (*caption && *comment) strcat(text,"\n");
   if (*comment) strcat(text,comment);

   if (*text) add_toptext(1,0,0,text,"Sans 10");

   return;
}


/**************************************************************************/

//  edit metadata menu function

char     pstars2[40];
char     cctext[exif_maxcc+50];
cchar    *pstars = 0;


void m_edit_metadata(GtkWidget *, cchar *menu)                             //  overhauled
{
   void  edit_imagetags_clickfunc(GtkWidget *widget, int line, int pos);
   void  edit_recentags_clickfunc(GtkWidget *widget, int line, int pos);
   void  edit_deftags_clickfunc(GtkWidget *widget, int line, int pos);
   int   editmeta_dialog_event(zdialog *zd, cchar *event);

   GtkWidget   *widget;
   zdialog     *zd;
   char        *ppv, pdate[12], ptime[8], psize[8];
   float       sizeMB, fMB = 1.0 / 1024.0 / 1024.0;

   pstars = ZTX("Rating: %c (stars)");

   if (menu) F1_help_topic = "edit_tags";
   if (! curr_file) return;
   if (checkpend("lock")) return;                                          //  check nothing pending 

/**
          ________________________________________________________
         |                Edit Metadata                           |
         |                                                        |
         |  Image File: xxxxxxxxxxxxxxxxxxx.jpg   Size: 1.23 MB   |
         |  Image Width: 2345  Height: 1234  Depth: 8 bits        |
         |  Image Date: [_________]  Time: [______]  [get prev]   |
         |  Rating: 0 (stars)  Choose: 0 1 2 3 4 5                |
         |  Caption [___________________________________________] |
         |  Comments [__________________________________________] |
         |  Image Tags [________________________________________] |
         |  Recent Tags [_______________________________________] |
         |  Defined Tags: ______________________________________  |
         |  |                                                   | |
         |  |                                                   | |
         |  |                                                   | |
         |  |                                                   | |
         |  |___________________________________________________| |
         |                                                        |
         |                [Geotags] [Manage Tags] [Apply] [Done]  |
         |________________________________________________________|

**/         

   if (! zdeditmeta)                                                       //  (re) start edit dialog 
   {
      zd = zdialog_new(ZTX("Edit Metadata"),Mwin,Bgeotags,Bmanagetags,Bapply,Bdone,null);
      zdeditmeta = zd;

      //  File: xxxxxxxxx.jpg  Size: nn.n MB
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labfile","hb1",ZTX("Image File:"),"space=3");
      zdialog_add_widget(zd,"label","file","hb1","filename.jpg","space=5");
      zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
      zdialog_add_widget(zd,"label","labtype","hb1",ZTX("Size:"),"space=3");
      zdialog_add_widget(zd,"label","size","hb1","12.3");
      zdialog_add_widget(zd,"label","labmb","hb1","MB","space=2");

      //  Image Width: xxxx  Height: xxxx  Depth nn
      zdialog_add_widget(zd,"hbox","hb2","dialog");
      zdialog_add_widget(zd,"label","labww","hb2",ZTX("Image Width:"),"space=3");
      zdialog_add_widget(zd,"label","width","hb2","2345","space=3");
      zdialog_add_widget(zd,"label","space","hb2",0,"space=5");
      zdialog_add_widget(zd,"label","labhh","hb2",ZTX("Height:"),"space=3");
      zdialog_add_widget(zd,"label","height","hb2","1234","space=3");
      zdialog_add_widget(zd,"label","space","hb2",0,"space=5");
      zdialog_add_widget(zd,"label","labdd","hb2",ZTX("Depth:"),"space=3");
      zdialog_add_widget(zd,"label","depth","hb2","8 bits","space=3");

      //  Image Date yyyy-mm-dd [__________]  Time hh:mm [_____]  [get prev]      
      zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labdate","hb3",ZTX("Image Date"),"space=3");
      zdialog_add_widget(zd,"entry","date","hb3",0,"scc=12");
      zdialog_add_widget(zd,"label","space","hb3",0,"space=5");
      zdialog_add_widget(zd,"label","labtime","hb3",ZTX("Time"),"space=3");
      zdialog_add_widget(zd,"entry","time","hb3",0,"scc=8");
      zdialog_add_widget(zd,"button","prdate","hb3",ZTX("get prev"),"space=8");
      
      //  Rating: 0 (stars)   Choose: 0  1  2  3  4  5
      zdialog_add_widget(zd,"hbox","hb6","dialog");
      zdialog_add_widget(zd,"label","rating","hb6",pstars,"space=3");
      zdialog_add_widget(zd,"label","labchoose","hb6",ZTX("Choose:"),"space=8");
      zdialog_add_widget(zd,"button","stars0","hb6","0","space=3");
      zdialog_add_widget(zd,"button","stars1","hb6","1","space=3");
      zdialog_add_widget(zd,"button","stars2","hb6","2","space=3");
      zdialog_add_widget(zd,"button","stars3","hb6","3","space=3");
      zdialog_add_widget(zd,"button","stars4","hb6","4","space=3");
      zdialog_add_widget(zd,"button","stars5","hb6","5","space=3");

      zdialog_add_widget(zd,"hbox","space","dialog",0,"space=3");
      
      //  Caption [__________________________________________] 
      zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=1");
      zdialog_add_widget(zd,"label","labcap","hb4",ZTX("Caption"),"space=3");
      zdialog_add_widget(zd,"frame","frame4","hb4",0,"space=3|expand");
      zdialog_add_widget(zd,"edit","caption","frame4",0,"wrap");    

      //  Comments [_________________________________________] 
      zdialog_add_widget(zd,"hbox","hb5","dialog",0,"space=1");
      zdialog_add_widget(zd,"label","labcomm","hb5",ZTX("Comments"),"space=3");
      zdialog_add_widget(zd,"frame","frame5","hb5",0,"space=3|expand");
      zdialog_add_widget(zd,"edit","comments","frame5",0,"wrap");    
      
      //  Image Tags [________________________________________]
      zdialog_add_widget(zd,"hbox","hb7","dialog",0,"space=1");
      zdialog_add_widget(zd,"label","labimtags","hb7",ZTX("Image Tags"),"space=3");
      zdialog_add_widget(zd,"frame","frame7","hb7",0,"space=3|expand");
      zdialog_add_widget(zd,"text","imagetags","frame7",0,"wrap");

      //  Recent Tags [________________________________________]
      zdialog_add_widget(zd,"hbox","hb8","dialog",0,"space=1");
      zdialog_add_widget(zd,"label","labrectags","hb8",ZTX("Recent Tags"),"space=3");
      zdialog_add_widget(zd,"frame","frame8","hb8",0,"space=3|expand");
      zdialog_add_widget(zd,"text","recentags","frame8",0,"wrap");

      //  Defined Tags (above big box)
      zdialog_add_widget(zd,"hbox","space","dialog",0,"space=3");          //  v.14.08
      zdialog_add_widget(zd,"hbox","hb9","dialog");
      zdialog_add_widget(zd,"label","labdeftags","hb9",ZTX("Defined Tags:"),"space=3");
      zdialog_add_widget(zd,"hbox","hb10","dialog",0,"expand");
      zdialog_add_widget(zd,"frame","frame10","hb10",0,"expand|space=3");
      zdialog_add_widget(zd,"scrwin","scrwin10","frame10",0,"expand");
      zdialog_add_widget(zd,"text","deftags","scrwin10",0,"wrap");

      load_deftags();                                                      //  stuff defined tags into dialog
      deftags_stuff(zd);

      widget = zdialog_widget(zd,"imagetags");                             //  tag widget mouse functions
      textwidget_set_clickfunc(widget,edit_imagetags_clickfunc);

      widget = zdialog_widget(zd,"recentags");
      textwidget_set_clickfunc(widget,edit_recentags_clickfunc);

      widget = zdialog_widget(zd,"deftags");
      textwidget_set_clickfunc(widget,edit_deftags_clickfunc);
      
      zdialog_resize(zd,400,500);                                          //  run dialog
      zdialog_run(zd,editmeta_dialog_event);
   }
   
   zd = zdeditmeta;

   load_filemeta(curr_file);                                               //  get EXIF/IPTC data

   ppv = (char *) strrchr(curr_file,'/');
   zdialog_stuff(zd,"file",ppv+1);                                         //  stuff dialog fields from curr. image file
   
   sizeMB = curr_file_size * fMB;   
   snprintf(psize,8,"%.2f",sizeMB);
   zdialog_stuff(zd,"size",psize);

   zdialog_stuff(zd,"width",Fpxb->ww);
   zdialog_stuff(zd,"height",Fpxb->hh);
   zdialog_stuff(zd,"depth",curr_file_bpc);
   
   metadate_pdate(meta_date,pdate,ptime);
   zdialog_stuff(zd,"date",pdate);
   zdialog_stuff(zd,"time",ptime);

   snprintf(pstars2,40,pstars,meta_rating[0]);
   zdialog_stuff(zd,"rating",pstars2);

   repl_1str(meta_caption,cctext,"\\n","\n");
   zdialog_stuff(zd,"caption",cctext);
   repl_1str(meta_comments,cctext,"\\n","\n");
   zdialog_stuff(zd,"comments",cctext);

   zdialog_stuff(zd,"imagetags",tags_imagetags);
   zdialog_stuff(zd,"recentags",tags_recentags);

   return;
}


//  mouse click functions for various text widgets for tags

void edit_imagetags_clickfunc(GtkWidget *widget, int line, int pos)        //  existing image tag was clicked
{
   char     *txline, *txtag, end = 0;
   cchar    *delims = tagdelims":";
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag) { zfree(txline); return; }
   
   del_tag(txtag,tags_imagetags);                                          //  remove tag from image
   zdialog_stuff(zdeditmeta,"imagetags",tags_imagetags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


void edit_recentags_clickfunc(GtkWidget *widget, int line, int pos)        //  recent tag was clicked
{
   char     *txline, *txtag, end = 0;
   cchar    *delims = tagdelims":";
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag) { zfree(txline); return; }
   
   add_tag(txtag,tags_imagetags,tagFcc);                                   //  add recent tag to image
   zdialog_stuff(zdeditmeta,"imagetags",tags_imagetags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


void edit_deftags_clickfunc(GtkWidget *widget, int line, int pos)          //  defined tag was clicked
{
   char     *txline, *txtag, end = 0;
   cchar    *delims = tagdelims":";
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag || end == ':') { zfree(txline); return; }                   //  tag category clicked, ignore
   
   add_tag(txtag,tags_imagetags,tagFcc);                                   //  add new tag to image
   zdialog_stuff(zdeditmeta,"imagetags",tags_imagetags);                   //    from defined tags list

   add_recentag(txtag);                                                    //  and add to recent tags
   zdialog_stuff(zdeditmeta,"recentags",tags_recentags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


//  dialog event and completion callback function

int editmeta_dialog_event(zdialog *zd, cchar *event)                       //  overhauled
{
   char     pdate[12], ptime[12];                                          //  yyyy-mm-dd  and  hh:mm:ss
   char     *metadate, *metatime;                                          //  yyyymmdd  and  hhmmss
   char     rating, *pp;
   int      nn = -1;
   
   if (! curr_file) return 1;
   
   if (strstr("date time caption comments",event))                         //  note change but process later
      Fmetachanged++;

   if (strEqu(event,"prdate")) {                                           //  repeat last date used
      if (*meta_prdate) {
         metadate_pdate(meta_prdate,pdate,ptime);
         zdialog_stuff(zd,"date",pdate);
         zdialog_stuff(zd,"time",ptime);
         Fmetachanged++;
         return 1;
      }
   }

   if (strnEqu(event,"stars",5)) {                                         //  event = stars0 to stars5
      rating = event[5];                                                   //  '0' to '5'
      snprintf(pstars2,40,pstars,rating);
      zdialog_stuff(zd,"rating",pstars2);
      Fmetachanged++;
      return 1;
   }

   if (strEqu(event,"enter")) zd->zstat = 3;                               //  enter = [apply]                    v.14.03

   if (! zd->zstat) return 1;                                              //  wait for completion
   
   if (zd->zstat == 1) {                                                   //  geotags
      zd->zstat = 0;                                                       //  keep dialog active
      m_edit_geotags(0,0);
      return 1;
   }   

   if (zd->zstat == 2) {                                                   //  manage tags
      zd->zstat = 0;                                                       //  keep dialog active
      zdialog_show(zd,0);                                                  //  hide parent dialog
      manage_tags();
      zdialog_show(zd,1);
      return 1;
   }

   if (zd->zstat != 3) {                                                   //  (not apply)
      zdialog_free(zdeditmeta);                                            //  cancel - kill dialog
      zdeditmeta = 0;
      return 1;
   }
   
   zd->zstat = 0;                                                          //  apply - keep dialog active
   gtk_window_present(MWIN);                                               //  keep focus on main window          v.14.09
   
   if (! Fmetachanged) return 1;                                           //  nothing changed
   if (checkpend("lock")) return 1;                                        //  check nothing pending

   zdialog_fetch(zd,"date",pdate,12);                                      //  get image date and time
   zdialog_fetch(zd,"time",ptime,12);
   if (*pdate) {                                                           //  date available
      metadate = pdate_metadate(pdate);                                    //  validate
      if (! metadate) return 1;                                            //  bad, re-input
      strcpy(meta_date,metadate);                                          //  convert to yyyymmdd
      if (*ptime) {                                                        //  time available
         metatime = ptime_metatime(ptime);                                 //  validate
         if (! metatime) return 1;                                         //  bad, re-input
         strcat(meta_date,metatime);                                       //  append hhmmss
      }
   }
   else *meta_date = 0;                                                    //  leave empty
   
   zdialog_fetch(zd,"rating",pstars2,40);                                  //  get new rating
   pp = strchr(pstars2,':');
   if (pp) nn = atoi(pp+1);
   if (nn < 0 || nn > 5) nn = 0;
   meta_rating[0] = '0' + nn;
   meta_rating[1] = 0;

   zdialog_fetch(zd,"caption",cctext,exif_maxcc);                          //  get new caption
   repl_1str(cctext,meta_caption,"\n","\\n");                              //  replace newlines with "\n"
   zdialog_fetch(zd,"comments",cctext,exif_maxcc);                         //  get new comments
   repl_1str(cctext,meta_comments,"\n","\\n");                             //  replace newlines with "\n"

   save_filemeta(curr_file);                                               //  save tag changes
   return 1;
}


//  Convert pretty date/time format from "yyyy-mm-dd" and "hh:mm:ss" to "yyyymmdd" and "hhmmss".
//  Missing month or day ("yyyy" or "yyyy-mm") is replaced with "-01".
//  Missing seconds ("hh:mm") are replaced with zero ("hh:mm:00").
//  Output user message and return null if not valid.

char * pdate_metadate(cchar *pdate)                                        //  "yyyy-mm-dd" >> "yyyymmdd"
{
   int         monlim[12] = { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
   int         cc, year, mon, day;
   char        pdate2[12];
   static char mdate[12];
   
   cc = strlen(pdate);
   if (cc > 10) goto badformat;

   strcpy(pdate2,pdate);

   if (cc == 4)                                                            //  conv. "yyyy" to "yyyy-01-01"
      strcat(pdate2,"-01-01");
   else if (cc == 7)                                                       //  conv. "yyyy-mm" to "yyyy-mm-01"
      strcat(pdate2,"-01");

   if (strlen(pdate2) != 10) goto badformat;
   if (pdate2[4] != '-' || pdate2[7] != '-') goto badformat;

   year = atoi(pdate2);
   mon = atoi(pdate2+5);
   day = atoi(pdate2+8);
   
   if (year < 0 || year > 2999) goto baddate;
   if (mon < 1 || mon > 12) goto baddate;
   if (day < 1 || day > monlim[mon-1]) goto baddate;
   if (mon == 2 && day == 29 && (year % 4)) goto baddate;
   
   memcpy(mdate,pdate2,4);                                                 //  return "yyyymmdd"
   memcpy(mdate+4,pdate2+5,2);
   memcpy(mdate+6,pdate2+8,3);
   return mdate;

badformat:
   zmessageACK(Mwin,0,ZTX("date format is YYYY-MM-DD"));
   return 0;

baddate:
   zmessageACK(Mwin,0,ZTX("date is invalid"));
   return 0;
}

char * ptime_metatime(cchar *ptime)                                        //  "hh:mm:ss" >> "hhmmss" 
{
   int         cc, hour, min, sec;
   char        ptime2[12];
   static char mtime[8];
   
   cc = strlen(ptime);
   if (cc > 8) goto badformat;

   strcpy(ptime2,ptime);
   
   if (cc == 5) strcat(ptime2,":00");                                      //  conv. "hh:mm" to "hh:mm:00"

   if (strlen(ptime2) != 8) goto badformat;
   if (ptime2[2] != ':' || ptime2[5] != ':') goto badformat;
   
   hour = atoi(ptime2);
   min = atoi(ptime2+3);
   sec = atoi(ptime2+6);
   if (hour < 0 || hour > 23) goto badtime;
   if (min < 0 || min > 59) goto badtime;
   if (sec < 0 || sec > 59) goto badtime;
   
   memcpy(mtime,ptime2,2);                                                 //  return "hhmmss"
   memcpy(mtime+2,ptime2+3,2);
   memcpy(mtime+4,ptime2+6,2);
   return mtime;

badformat:
   zmessageACK(Mwin,0,ZTX("time format is HH:MM [:SS]"));
   return 0;

badtime:
   zmessageACK(Mwin,0,ZTX("time is invalid"));
   return 0;
}


//  Convert metadata date/time "yyyymmddhhmmss" to pretty format "yyyy-mm-dd" and "hh:mm:ss"

void metadate_pdate(cchar *metadate, char *pdate, char *ptime)
{
   if (*metadate) {
      memcpy(pdate,metadate,4);                                            //  yyyymmdd to yyyy-mm-dd
      memcpy(pdate+5,metadate+4,2);
      memcpy(pdate+8,metadate+6,2);
      pdate[4] = pdate[7] = '-';
      pdate[10] = 0;

      memcpy(ptime,metadate+8,2);                                          //  hhmmss to hh:mm:ss
      memcpy(ptime+3,metadate+10,2);
      ptime[2] = ':';
      ptime[5] = 0;
      if (metadate[12] > '0' || metadate[13] > '0') {                      //  append :ss only if not :00
         memcpy(ptime+6,metadate+12,2);
         ptime[5] = ':';
         ptime[8] = 0;
      }
   }
   else *pdate = *ptime = 0;                                               //  missing
   return;
}


/**************************************************************************/

//  manage tags function - auxilliary dialog

zdialog  *zdmanagetags = 0;

void manage_tags()
{
   void  manage_deftags_clickfunc(GtkWidget *widget, int line, int pos);
   int   managetags_dialog_event(zdialog *zd, cchar *event);

   GtkWidget   *widget;
   zdialog     *zd;

   if (zdmanagetags) return;
   zd = zdialog_new(ZTX("Manage Tags"),Mwin,ZTX("orphan tags"),Bdone,null);
   zdmanagetags = zd;

   zdialog_add_widget(zd,"hbox","hb7","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labcatg","hb7",ZTX("category"),"space=5");
   zdialog_add_widget(zd,"entry","catg","hb7",0,"scc=12");
   zdialog_add_widget(zd,"label","space","hb7",0,"space=5");
   zdialog_add_widget(zd,"label","labtag","hb7",ZTX("tag"),"space=5");
   zdialog_add_widget(zd,"entry","tag","hb7",0,"scc=20|expand");
   zdialog_add_widget(zd,"label","space","hb7",0,"space=5");
   zdialog_add_widget(zd,"button","create","hb7",Bcreate);
   zdialog_add_widget(zd,"button","delete","hb7",Bdelete);

   zdialog_add_widget(zd,"hbox","hb8","dialog");
   zdialog_add_widget(zd,"label","labdeftags","hb8",ZTX("Defined Tags:"),"space=5");
   zdialog_add_widget(zd,"hbox","hb9","dialog",0,"expand");
   zdialog_add_widget(zd,"frame","frame8","hb9",0,"space=5|expand");
   zdialog_add_widget(zd,"scrwin","scrwin8","frame8",0,"expand");
   zdialog_add_widget(zd,"text","deftags","scrwin8",0,"expand|wrap");

   widget = zdialog_widget(zd,"deftags");                                  //  tag widget mouse function
   textwidget_set_clickfunc(widget,manage_deftags_clickfunc);

   load_deftags();                                                         //  stuff defined tags into dialog
   deftags_stuff(zd);

   zdialog_resize(zd,0,400);
   zdialog_run(zd,managetags_dialog_event);                                //  run dialog
   zdialog_wait(zd);
   zdialog_free(zd);                                                       //  v.14.06
   
   return;
}


//  mouse click functions for widget having tags

void manage_deftags_clickfunc(GtkWidget *widget, int line, int pos)        //  tag or tag category was clicked
{
   char     *txline, *txtag, end = 0;
   cchar    *delims = tagdelims":";
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag) { zfree(txline); return; }
   
   if (end == ':')                                                         //  tag category >> dialog widget
      zdialog_stuff(zdmanagetags,"catg",txtag);
   else 
      zdialog_stuff(zdmanagetags,"tag",txtag);                             //  selected tag >> dialog widget
   
   zfree(txline);
   zfree(txtag);
   return;
}


//  dialog event and completion callback function

int managetags_dialog_event(zdialog *zd, cchar *event)
{
   void tag_orphans();

   char        tag[tagcc], catg[tagcc];
   int         changed = 0;
   
   if (strEqu(event,"enter")) zd->zstat = 2;                               //  [done]  v.14.03
   
   if (zd->zstat) 
   {
      if (zd->zstat == 1) {                                                //  report orphan tags
         zd->zstat = 0;                                                    //  keep dialog active
         tag_orphans();
      }
     
      else {                                                               //  done or [x]
         zdmanagetags = 0;
         return 0;
      }
   }
   
   if (strEqu(event,"create")) {                                           //  add new tag to defined tags
      zdialog_fetch(zd,"catg",catg,tagcc);
      zdialog_fetch(zd,"tag",tag,tagcc);
      add_deftag(catg,tag);
      changed++;
   }

   if (strEqu(event,"delete")) {                                           //  remove tag from defined tags
      zdialog_fetch(zd,"tag",tag,tagcc);
      del_deftag(tag);
      changed++;
   }

   if (changed) {   
      save_deftags();                                                      //  save tag updates to file
      deftags_stuff(zd);                                                   //  update dialog "deftags" window
      if (zdeditmeta)                                                      //  and edit metadata dialog if active
         deftags_stuff(zdeditmeta);
      if (zdbatchtags)                                                     //  and batch tags dialog if active
         deftags_stuff(zdbatchtags);
   }

   return 0;
}


/**************************************************************************/

//  add input tag to output tag list if not already there and enough room
//  returns:   0 = added OK     1 = already there (case ignored)
//             2 = overflow     3 = bad utf8 characters     4 = null tag

int add_tag(char *tag, char *taglist, int maxcc)
{
   char     *pp1, *pp2, tag1[tagcc], tag2[tagcc];
   int      cc, cc1, cc2;

   strncpy0(tag1,tag,tagcc);                                               //  remove leading and trailing blanks
   cc = strTrim2(tag2,tag1);
   if (! cc) return 4;

   if (utf8_check(tag2)) {                                                 //  check for valid utf8 encoding
      printz("*** bad utf8 characters: %s \n",tag2);
      return 3;
   }
   
   while ((pp1 = strpbrk(tag2,tagdelims":"))) *pp1 = '-';                  //  replace problem characters

   strcpy(tag,tag2);                                                       //  replace tag with sanitized version
   
   pp1 = taglist;
   cc1 = strlen(tag);

   while (true)                                                            //  check if already in tag list
   {
      while (*pp1 == ' ' || *pp1 == tagdelim1) pp1++;
      if (! *pp1) break;
      pp2 = pp1 + 1;
      while (*pp2 && *pp2 != tagdelim1) pp2++;
      cc2 = pp2 - pp1;
      if (cc2 == cc1 && strncaseEqu(tag,pp1,cc1)) return 1;
      pp1 = pp2;
   }

   cc2 = strlen(taglist);                                                  //  append to tag list if space enough
   if (cc1 + cc2 + 3 > maxcc) return 2;
   strcpy(taglist + cc2,tag);
   strcpy(taglist + cc2 + cc1, tagdelimB);                                 //  add delimiter + space

   if (taglist == tags_imagetags)                                          //  image tags were changed
      Fmetachanged++;

   return 0;
}


//  remove tag from taglist, if present
//  returns: 0 if found and deleted, otherwise 1

int del_tag(char *tag, char *taglist)
{
   int         ii, ftcc, atcc, found;
   char        *temptags;
   cchar       *pp;
   
   temptags = zstrdup(taglist);
   
   *taglist = 0;
   ftcc = found = 0;
   
   for (ii = 1; ; ii++)
   {
      pp = strField(temptags,tagdelims,ii);                                //  next tag
      if (! pp) {
         zfree(temptags);
         if (found && taglist == tags_imagetags)                           //  image tags were changed
            Fmetachanged++;
         return 1-found;
      }
      if (*pp == ' ') continue;
      
      if (strcaseEqu(pp,tag)) {                                            //  skip matching tag
         found = 1;
         continue;
      }

      atcc = strlen(pp);                                                   //  copy non-matching tag
      strcpy(taglist + ftcc, pp);
      ftcc += atcc;
      strcpy(taglist + ftcc, tagdelimB);                                   //  + delim + blank
      ftcc += 2;
   }
}


//  add new tag to recent tags, if not already.
//  remove oldest to make space if needed.

int add_recentag(char *tag)
{
   int         err;
   char        *pp, temptags[tagRcc];

   err = add_tag(tag,tags_recentags,tagRcc);                               //  add tag to recent tags

   while (err == 2)                                                        //  overflow
   {
      strncpy0(temptags,tags_recentags,tagRcc);                            //  remove oldest to make room
      pp = strpbrk(temptags,tagdelims);
      if (! pp) return 0;
      strcpy(tags_recentags,pp+2);                                         //  delimiter + blank before tag
      err = add_tag(tag,tags_recentags,tagRcc);
   }

   return 0;
}


/**************************************************************************/

//  Load tags_defined file into tags_deftags[ii] => category: tag1, tag2, ...
//  Read image_index recs. and add unmatched tags: => nocatg: tag1, tag2, ...

void load_deftags()
{
   int tags_Ucomp(cchar *tag1, cchar *tag2);

   static int  Floaded = 0;
   FILE *      fid;
   sxrec_t     sxrec;
   int         ii, jj, ntags, err, cc, tcc, ftf;
   int         ncats, catoverflow;
   int         nocat, nocatcc;
   char        tag[tagcc], catg[tagcc];
   char        tagsbuff[tagGcc];
   char        *pp1, *pp2;
   char        ptags[tagntc][tagcc];
   
   if (Floaded) return;                                                    //  use memory tags if already there
   Floaded++;

   for (ii = 0; ii < maxtagcats; ii++)                                     //  clean memory
      tags_deftags[ii] = 0;
   
   ncats = catoverflow = 0;

   fid = fopen(tags_defined_file,"r");                                     //  read tags_defined file
   if (fid) {
      while (true) {
         pp1 = fgets_trim(tagsbuff,tagGcc,fid);
         if (! pp1) break;
         if (ncats == maxtagcats-1) goto toomanycats;
         pp2 = strchr(pp1,':');                                            //  isolate "category:"
         if (! pp2) continue;                                              //  reject bad data
         cc = pp2 - pp1 + 1;
         if (cc > tagcc-1) continue;
         strncpy0(catg,pp1,cc);                                            //  (for error message)
         if (strlen(pp1) > tagGcc-2) goto cattoobig;
         pp2++;
         while (*pp2 == ' ') pp2++;
         if (strlen(pp2) < 3) continue;
         while ((pp2 = strpbrk(pp2,tagdelims))) *pp2++ = tagdelim1;        //  force comma delimiter
         tags_deftags[ncats] = zstrdup(pp1);                               //  tags_deftags[ii]
         ncats++;                                                          //   = category: tag1, tag2, ... tagN,
      }
      err = fclose(fid);
      if (err) goto deftagserr;
   }

   nocat = ncats;                                                          //  make last category "nocatg" for
   ncats++;                                                                //   unmatched tags in image_index recs.
   tags_deftags[nocat] = (char *) zmalloc(tagGcc);
   strcpy(tags_deftags[nocat],"nocatg: ");
   nocatcc = 8;

   ftf = 1;                                                                //  read all image index recs.

   while (true)
   {
      err = read_sxrec_seq(sxrec,ftf);
      if (err) break;
      
      pp1 = sxrec.tags;                                                    //  may be "null,"
      
      while (true)
      {
         while (*pp1 && strchr(tagdelims" ",*pp1)) pp1++;                  //  next image tag start
         if (! *pp1) break;
         pp2 = strpbrk(pp1,tagdelims);                                     //  end
         if (! pp2) pp2 = pp1 + strlen(pp1);  
         cc = pp2 - pp1;
         if (cc > tagcc-1) {
            pp1 = pp2;
            continue;                                                      //  ignore huge tag
         }
         
         strncpy0(tag,pp1,cc+1);                                           //  look for tag in defined tags
         err = find_deftag(tag);
         if (! err) {                                                      //  found
            pp1 = pp2;
            continue;
         }

         if (nocatcc + cc + 2 > tagGcc-2) {
            catoverflow = 1;                                               //  nocatg: length limit reached
            break;
         }
         else {
            strcpy(tags_deftags[nocat] + nocatcc, tag);                    //  append tag to list
            nocatcc += cc;
            strcpy(tags_deftags[nocat] + nocatcc, tagdelimB);              //  + delim + blank
            nocatcc += 2;
         }

         pp1 = pp2;
      }

      zfree(sxrec.file);
      zfree(sxrec.tags);
      zfree(sxrec.comms);
      zfree(sxrec.capt);
      zfree(sxrec.gtags);
   }

   if (catoverflow) goto cattoobig;
   
//  parse all the tags in each category and sort in ascending order

   for (ii = 0; ii < ncats; ii++)
   {
      pp1 = tags_deftags[ii];
      pp2 = strchr(pp1,':');
      cc = pp2 - pp1 + 1;
      strncpy0(catg,pp1,cc);
      pp1 = pp2 + 1;
      while (*pp1 == ' ') pp1++;
      tcc = 0;

      for (jj = 0; jj < tagntc; jj++)
      {
         if (! *pp1) break;
         pp2 = strchr(pp1,tagdelim1);
         if (pp2) cc = pp2 - pp1;
         else cc = strlen(pp1);
         if (cc > tagcc-1) cc = tagcc-1;
         strncpy0(ptags[jj],pp1,cc+1);
         pp1 += cc + 1;
         tcc += cc;
         while (*pp1 == ' ') pp1++;
      }
      
      ntags = jj;
      if (ntags == tagntc) goto cattoobig;
      HeapSort((char *) ptags,tagcc,ntags,tags_Ucomp);

      pp1 = tags_deftags[ii];
      tcc += strlen(catg) + 2 + 2 * ntags + 2;                             //  category, all tags, delimiters
      pp2 = (char *) zmalloc(tcc);

      tags_deftags[ii] = pp2;                                              //  swap memory
      zfree(pp1);

      strcpy(pp2,catg);
      pp2 += strlen(catg);
      strcpy(pp2,": ");                                                    //  pp2 = "category: "
      pp2 += 2;

      for (jj = 0; jj < ntags; jj++)                                       //  add the sorted tags
      {
         strcpy(pp2,ptags[jj]);                                            //  append tag + delim + blank
         pp2 += strlen(pp2);
         strcpy(pp2,tagdelimB);
         pp2 += 2;
      }
      
      *pp2 = 0;
   }
   
//  sort the categories in ascending order
//  leave "nocatg" at the end

   for (ii = 0; ii < ncats-1; ii++)
   for (jj = ii+1; jj < ncats-1; jj++) 
   {
      pp1 = tags_deftags[ii];
      pp2 = tags_deftags[jj];
      if (strcasecmp(pp1,pp2) > 0) {
         tags_deftags[ii] = pp2;
         tags_deftags[jj] = pp1;
      }
   }

   return;

toomanycats:
   zmessLogACK(Mwin,"more than %d categories",maxtagcats);
   fclose(fid);
   return;
   
cattoobig:
   zmessLogACK(Mwin,"category %s is too big",catg);
   fclose(fid);
   return;

deftagserr:
   zmessLogACK(Mwin,"tags_defined file error: %s",strerror(errno));
   return;
}


//  compare function for tag sorting

int tags_Ucomp(cchar *tag1, cchar *tag2)
{
   return strcasecmp(tag1,tag2);
}


//  write tags_deftags[] memory data to the defined tags file if any changes were made

void save_deftags()
{
   int         ii, err;
   FILE        *fid;

   fid = fopen(tags_defined_file,"w");                                     //  write tags_defined file
   if (! fid) goto deftagserr;

   for (ii = 0; ii < maxtagcats; ii++)
   {
      if (! tags_deftags[ii+1]) break;                                     //  omit last category, "nocatg"
      err = fprintf(fid,"%s\n",tags_deftags[ii]);                          //  each record: 
      if (err < 0) goto deftagserr;                                        //    category: tag1, tag2, ... tagN,
   }

   err = fclose(fid);
   if (err) goto deftagserr;
   return;
   
deftagserr:
   zmessLogACK(Mwin,"tags_defined file error: %s",strerror(errno));
   return;
}


//  find a given tag in tags_deftags[]
//  return: 0 = found, 1 = not found

int find_deftag(char *tag)
{
   int      ii, cc;
   char     tag2[tagcc+4];
   char     *pp;

   if (! tag || *tag <= ' ') return 0;                                     //  bad tag

   strncpy0(tag2,tag,tagcc);                                               //  construct tag + delim + blank
   cc = strlen(tag2);
   strcpy(tag2+cc,tagdelimB);
   cc += 2;

   for (ii = 0; ii < maxtagcats; ii++)
   {
      pp = tags_deftags[ii];                                               //  category: tag1, tag2, ... tagN, 
      if (! pp) return 1;                                                  //  not found

      while (pp)
      {
         pp = strcasestr(pp,tag2);                                         //  look for delim + blank + tag + delim
         if (! pp) break;
         if (strchr(tagdelims":", pp[-2])) return 0;
         pp += cc;
      }
   }

   return 1;
}


//  add new tag to tags_deftags[] >> category: tag1, tag2, ... newtag,
//  returns:   0 = added OK     1 = not unique (case ignored)
//             2 = overflow     3 = bad utf8 characters     4 = null tag
//  if tag present under another category, it is moved to new category

int add_deftag(char *catg, char *tag)
{
   int         ii, cc, cc1, cc2;
   char        catg1[tagcc], tag1[tagcc];
   char        *pp1, *pp2;
   
   if (! catg || *catg <= ' ') catg = (char *) "nocatg";                   //  remove leading and trailing blanks
   strncpy0(catg1,catg,tagcc);
   cc = strTrim2(catg1);

   if (utf8_check(catg1)) {                                                //  check for valid utf8 encoding
      printz("*** bad utf8 characters: %s \n",catg1);
      return 3;
   }

   while ((pp1 = strpbrk(catg1,tagdelims":"))) *pp1 = '-';                 //  replace problem characters

   strncpy0(tag1,tag,tagcc);                                               //  remove leading and trailing blanks
   cc = strTrim2(tag1);
   if (! cc) return 4;

   if (utf8_check(tag1)) {                                                 //  check for valid utf8 encoding
      printz("*** bad utf8 characters: %s \n",tag1);
      return 3;
   }

   while ((pp1 = strpbrk(tag1,tagdelims":"))) *pp1 = '-';                  //  replace problem characters
   
   strcpy(tag,tag1);                                                       //  replace tag with sanitized version

   del_deftag(tag1);                                                       //  delete if already there

   cc1 = strlen(catg1);

   for (ii = 0; ii < maxtagcats; ii++)                                     //  look for given category
   {
      pp1 = tags_deftags[ii];
      if (! pp1) goto newcatg;
      if (! strnEqu(catg1,pp1,cc1)) continue;
      if (pp1[cc1] == ':') goto oldcatg;
   }

newcatg:
   if (ii == maxtagcats) goto toomanycats;
   cc1 = strlen(catg1) + strlen(tag1) + 6;
   pp1 = (char *) zmalloc(cc1);
   *pp1 = 0;
   strncatv(pp1,cc1,catg1,": ",tag1,tagdelimB,null);                       //  category: + tag + delim + blank
   tags_deftags[ii] = tags_deftags[ii-1];                                  //  move "nocatg" record to next slot
   tags_deftags[ii-1] = pp1;                                               //  insert new record before
   return 0;

oldcatg:
   cc1 = strlen(tag1);
   pp2 = pp1 + 2;
   while (true) {
      pp2 = strcasestr(pp2,tag1);                                          //  look for delim + blank + tag + delim
      if (! pp2) break;                                                    //        or colon + blank + tag + delim
      if (strchr(tagdelims,pp2[cc]) && strchr(tagdelims":", pp2[-2]))
         return 1;                                                         //  tag not unique
      pp2 += cc1;
   }
   cc2 = strlen(pp1);                                                      //  add new tag to old record
   if (cc1 + cc2 + 4 > tagGcc) goto cattoobig;
   pp2 = zstrdup(pp1,cc1+cc2+4);                                           //  expand string
   zfree(pp1);
   tags_deftags[ii] = pp2;
   strcpy(pp2+cc2,tag1);                                                   //  old record + tag + delim + blank
   strcpy(pp2+cc2+cc1,tagdelimB);
   return 0;

toomanycats:
   zmessLogACK(Mwin,"more than %d categories",maxtagcats);
   return 2;

cattoobig:
   zmessLogACK(Mwin,"category has too many tags");
   return 2;
}


//  delete tag from defined tags list, tags_deftags[] 
//  return: 0 = found and deleted, 1 = not found

int del_deftag(char *tag)
{
   int      ii, cc;
   char     tag2[tagcc+4];
   char     *pp, *pp1, *pp2;

   if (! tag || *tag <= ' ') return 1;                                     //  bad tag

   strncpy0(tag2,tag,tagcc);                                               //  construct tag + delim + blank
   cc = strlen(tag2);
   strcpy(tag2+cc,tagdelimB);
   cc += 2;

   for (ii = 0; ii < maxtagcats; ii++)
   {
      pp = tags_deftags[ii];
      if (! pp) return 1;                                                  //  not found

      while (pp)
      {
         pp = strcasestr(pp,tag2);                                         //  look for prior delim or colon
         if (! pp) break;
         if (strchr(tagdelims":", pp[-2])) goto found;
         pp += cc;
      }
   }
   
found:
   for (pp1 = pp, pp2 = pp+cc; *pp2; pp1++, pp2++)                         //  eliminate tag, delim, blank
      *pp1 = *pp2;
   *pp1 = 0;
   
   return 0;
}


//  stuff tags_deftags[] into given text widget and format by category
//  create tags_deftext with flat list of tags for mouse clicking

void deftags_stuff(zdialog *zd)
{
   GtkWidget      *widget;
   GtkTextBuffer  *textbuff;
   GtkTextIter    iter1, iter2;
   int            ii, cc;
   char           catgname[tagcc+3];
   char           *pp1, *pp2;
   
   widget = zdialog_widget(zd,"deftags");
   wclear(widget);

   for (ii = 0; ii < maxtagcats; ii++)
   {
      pp1 = tags_deftags[ii];
      if (! pp1) break;
      pp2 = strchr(pp1,':');
      if (! pp2) continue;
      if (pp2 > pp1 + tagcc-3) continue;
      pp2 += 2;
      cc = pp2 - pp1 + 1;
      strncpy0(catgname,pp1,cc);
      wprintx(widget,0,catgname,"monospace bold 9");
      if (*pp2) wprintx(widget,0,pp2,"monospace 9");
      wprintx(widget,0,"\n");
   }
   
   textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
   gtk_text_buffer_get_bounds(textbuff,&iter1,&iter2);
   pp1 = gtk_text_buffer_get_text(textbuff,&iter1,&iter2,0);
   if (strlen(pp1) > tagTcc-2) 
      zmessageACK(Mwin,0,"defined tags exceed %d characters",tagTcc);
   strncpy0(tags_deftext,pp1,tagTcc);

   return;
}


//  report tags defined and not used in any image file

void tag_orphans()
{
   FILE        *fid;
   sxrec_t     sxrec;
   int         ii, cc, err, ftf;
   int         Ndeftags;
   char        **deftags;
   char        usedtag[tagcc], tagsbuff[tagGcc];
   char        *pp1, *pp2, text[20];
  
   ii = tagTcc / 4;                                                        //  max. tags if average size is tiny
   deftags = (char **) zmalloc(ii * sizeof(char *));                       //  allocate memory
   Ndeftags = 0;
   
   fid = fopen(tags_defined_file,"r");                                     //  read tags_defined file
   if (fid) {
      while (true) {
         pp1 = fgets_trim(tagsbuff,tagGcc,fid);
         if (! pp1) break;
         pp1 = strchr(pp1,':');                                            //  skip over "category:"
         if (! pp1) continue;
         cc = pp1 - tagsbuff;
         if (cc > tagcc) continue;                                         //  reject bad data (manual edit?)
         pp1++;
         for (ii = 1; ; ii++) {                                            //  get tags: tag1, tag2, ...
            pp2 = (char *) strField(pp1,tagdelims,ii);
            if (! pp2) break;
            if (strlen(pp2) < 3) continue;                                 //  reject bad data
            if (strlen(pp2) > tagcc) continue;
            deftags[Ndeftags] = zstrdup(pp2);
            Ndeftags++;
         }
      }
      fclose(fid);
   }
   
   ftf = 1;                                                                //  read all image index recs.

   while (true)
   {
      err = read_sxrec_seq(sxrec,ftf);
      if (err) break;
      
      pp1 = sxrec.tags;                                                    //  image tags
      if (! pp1) continue;
      if (strnEqu(pp1,"null",4)) continue;

      while (true)
      {
         while (*pp1 && strchr(tagdelims" ",*pp1)) pp1++;                  //  next image tag start
         if (! *pp1) break;
         pp2 = strpbrk(pp1,tagdelims);                                     //  end
         if (! pp2) pp2 = pp1 + strlen(pp1);  
         cc = pp2 - pp1;
         if (cc > tagcc-1) {
            pp1 = pp2;
            continue;                                                      //  ignore huge tag
         }

         strncpy0(usedtag,pp1,cc+1);                                       //  used tag, without delimiter

         for (ii = 0; ii < Ndeftags; ii++)                                 //  find in defined tags
            if (strEqu(usedtag,deftags[ii])) break;
         
         if (ii < Ndeftags) {                                              //  found
            zfree(deftags[ii]);
            Ndeftags--;
            while (ii < Ndeftags) {                                        //  defined tag is in use
               deftags[ii] = deftags[ii+1];                                //  remove from list and pack down
               ii++;
            }
         }

         pp1 = pp2;
      }
   }

   write_popup_text("open","unused tags",200,200,Mwin);
   for (ii = 0; ii < Ndeftags; ii++)
      write_popup_text("write",deftags[ii]);
   snprintf(text,20,"%d unused tags",Ndeftags);
   write_popup_text("write",text);

   for (ii = 0; ii < Ndeftags; ii++)
      zfree(deftags[ii]);   
   zfree(deftags);

   return;
}


/**************************************************************************/

//  edit EXIF/IPTC data - add or change specified EXIF/IPTC/etc. key

void m_meta_edit_any(GtkWidget *, cchar *menu)
{
   int  meta_edit_any_dialog_event(zdialog *zd, cchar *event);

   char        keyname[40], keydata[exif_maxcc];
   cchar       *pp1[1];
   char        **pp2;

   if (menu) F1_help_topic = "edit_meta";
   if (! curr_file) return;
   if (checkpend("lock")) return;                                          //  check nothing pending

   if (! zdexifedit)                                                       //  popup dialog if not already
   {   
      zdexifedit = zdialog_new(ZTX("Edit Metadata"),Mwin,Bfetch,Bsave,Bdone,null);
      zdialog_add_widget(zdexifedit,"vbox","hb1","dialog");
      zdialog_add_widget(zdexifedit,"hbox","hbkey","dialog",0,"space=2");
      zdialog_add_widget(zdexifedit,"hbox","hbdata","dialog",0,"space=2");
      zdialog_add_widget(zdexifedit,"label","labkey","hbkey",ZTX("key name"));
      zdialog_add_widget(zdexifedit,"entry","keyname","hbkey",0,"scc=20");
      zdialog_add_widget(zdexifedit,"label","labdata","hbdata",ZTX("key value"));
      zdialog_add_widget(zdexifedit,"entry","keydata","hbdata",0,"expand");
      zdialog_run(zdexifedit,meta_edit_any_dialog_event);
   }

   zdialog_fetch(zdexifedit,"keyname",keyname,40);                         //  get key name from dialog
   strCompress(keyname);

   if (*keyname)                                                           //  update live dialog
   {
      pp1[0] = keyname;                                                    //  look for key data 
      pp2 = exif_get(curr_file,pp1,1);
      if (pp2[0]) {
         strncpy0(keydata,pp2[0],exif_maxcc);
         zfree(pp2[0]);
      }
      else *keydata = 0;
      zdialog_stuff(zdexifedit,"keydata",keydata);                         //  stuff into dialog
   }

   return;
}


//  dialog event and completion callback function

int  meta_edit_any_dialog_event(zdialog *zd, cchar *event)
{
   char        keyname[40], keydata[exif_maxcc];
   cchar       *pp1[1], *pp2[1];
   char        **pp3;
   int         err;

   if (! zd->zstat) return 1;
   if (! curr_file) return 1;
   
   zdialog_fetch(zd,"keyname",keyname,40);
   zdialog_fetch(zd,"keydata",keydata,exif_maxcc);
   strCompress(keyname);                                                   //  remove blanks

   if (zd->zstat == 1)                                                     //  fetch
   {
      zd->zstat = 0;
      if (! *keyname) return 0;
      pp1[0] = keyname;
      pp3 = exif_get(curr_file,pp1,1);
      if (pp3[0]) {
         strncpy0(keydata,pp3[0],exif_maxcc);
         zfree(pp3[0]);
      }
      else *keydata = 0;
      zdialog_stuff(zd,"keydata",keydata);
   }

   else if (zd->zstat == 2)                                                //  save
   {
      if (checkpend("lock")) return 1;                                     //  check nothing pending
      zd->zstat = 0;                                                       //  keep dialog active
      if (! *keyname) return 1;
      pp1[0] = keyname;
      pp2[0] = keydata;
      err = exif_put(curr_file,pp1,pp2,1);
      if (err) zmessageACK(Mwin,0,"error: %s",strerror(err));
      load_filemeta(curr_file);                                            //  update image index in case
      update_image_index(curr_file);                                       //    searchable metadata item updated
      if (zdexifview) meta_view(0);                                        //  update exif view if active
   }

   else zdialog_free(zdexifedit);                                          //  other

   return 1;
}


/**************************************************************************/

//  delete EXIF/IPTC data, specific key or all data

void m_meta_delete(GtkWidget *, cchar *menu)
{
   int   meta_delete_dialog_event(zdialog *zd, cchar *event);
   
   zdialog     *zd;

   F1_help_topic = "delete_meta";
   if (! curr_file) return;
   if (checkpend("lock")) return;                                          //  check nothing pending

   zd = zdialog_new(ZTX("Delete Metadata"),Mwin,Bapply,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"radio","kall","hb1",ZTX("All"),"space=5");
   zdialog_add_widget(zd,"radio","key1","hb1",ZTX("One Key:"));
   zdialog_add_widget(zd,"entry","keyname","hb1",0,"scc=20");
   zdialog_stuff(zd,"key1",1);

   zdialog_run(zd,meta_delete_dialog_event);
   return;
}


//  dialog event and completion callback function

int meta_delete_dialog_event(zdialog *zd, cchar *event)
{
   int         kall, key1;
   char        keyname[40];
   
   if (! zd->zstat) return 1;
   if (! curr_file) return 1;

   if (zd->zstat != 1) {                                                   //  canceled
      zdialog_free(zd);
      return 1;
   }
   
   zd->zstat = 0;                                                          //  dialog remains active
   gtk_window_present(MWIN);                                               //  keep focus on main window          v.14.09
   if (checkpend("lock")) return 1;                                        //  check nothing pending

   zdialog_fetch(zd,"kall",kall);
   zdialog_fetch(zd,"key1",key1);
   zdialog_fetch(zd,"keyname",keyname,40);
   strCompress(keyname);

   gallery_monitor("stop");                                                //  stop excess gallery inits

   if (kall)                                                               //  -P preserve date removed 
      shell_ack("exiftool -m -q -overwrite_original -all=  \"%s\"",curr_file);
   else if (key1)
      shell_ack("exiftool -m -q -overwrite_original -%s=  \"%s\"",keyname,curr_file);
   else return 1;
   
   load_filemeta(curr_file);                                               //  update image index in case a
   update_image_index(curr_file);                                          //    searchable metadata deleted

   gallery_monitor("start");
   
   if (zdexifview) meta_view(0);                                           //  update exif view if active

   return 1;
}


/**************************************************************************/

//  menu function - add and remove tags for many files at once

namespace batchtags 
{
   char        **filelist = 0;                                             //  files to process
   int         filecount = 0;                                              //  file count
   char        addtags[tagMcc];                                            //  tags to add, list
   char        deltags[tagMcc];                                            //  tags to remove, list
}


void m_batchTags(GtkWidget *, cchar *)                                     //  combine batch add/del tags
{
   using namespace batchtags;

   void  batch_addtags_clickfunc(GtkWidget *widget, int line, int pos);
   void  batch_deltags_clickfunc(GtkWidget *widget, int line, int pos);
   void  batch_deftags_clickfunc(GtkWidget *widget, int line, int pos);
   int   batchTags_dialog_event(zdialog *zd, cchar *event);
   
   char        *ptag, *file;
   int         zstat, ii, jj, err;
   zdialog     *zd;
   GtkWidget   *widget;

   F1_help_topic = "batch_tags";

   if (checkpend("all")) return;                                           //  check nothing pending
   Fmenulock = 1;

/***
               Batch Add/Remove Tags   

         [Select Files]  NN files selected 

         (o) tags to add    [_________________________]
         (o) tags to remove [_________________________]

          __ defined tags ____________________________
         |                                            |
         |                                            |
         |                                            |
         |                                            |
         |____________________________________________|

                      [manage tags] [proceed] [cancel]
***/

   zd = zdialog_new(ZTX("Batch Add/Remove Tags"),Mwin,Bmanagetags,Bproceed,Bcancel,null);
   zdbatchtags = zd;

   zdialog_add_widget(zd,"hbox","hbfiles","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","files","hbfiles",Bselectfiles,"space=5");
   zdialog_add_widget(zd,"label","labcount","hbfiles",Bnofileselected,"space=10");

   zdialog_add_widget(zd,"hbox","hbtags","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hbtags",0,"space=3|homog");
   zdialog_add_widget(zd,"vbox","vb2","hbtags",0,"space=3|homog|expand");
   
   zdialog_add_widget(zd,"radio","radadd","vb1",ZTX("tags to add"));
   zdialog_add_widget(zd,"radio","raddel","vb1",ZTX("tags to remove"));

   zdialog_add_widget(zd,"frame","fradd","vb2",0,"expand");
   zdialog_add_widget(zd,"text","addtags","fradd",0,"expand|wrap");
   zdialog_add_widget(zd,"frame","frdel","vb2",0,"expand");
   zdialog_add_widget(zd,"text","deltags","frdel",0,"expand|wrap");

   zdialog_add_widget(zd,"hbox","hbspace","dialog",0,"space=3");
   zdialog_add_widget(zd,"hbox","hbdef1","dialog");
   zdialog_add_widget(zd,"label","labdef1","hbdef1",ZTX("Defined Tags:"),"space=5");
   zdialog_add_widget(zd,"hbox","hbdef2","dialog",0,"expand");
   zdialog_add_widget(zd,"frame","frdef2","hbdef2",0,"space=5|expand");
   zdialog_add_widget(zd,"scrwin","scrdef2","frdef2",0,"expand");
   zdialog_add_widget(zd,"text","deftags","scrdef2",0,"expand|wrap");
   
   zdialog_stuff(zd,"radadd",1);
   zdialog_stuff(zd,"raddel",0);

   load_deftags();                                                         //  stuff defined tags into dialog
   deftags_stuff(zd);

   filelist = 0;
   filecount = 0;
   *addtags = *deltags = 0;

   widget = zdialog_widget(zd,"addtags");                                  //  tag widget mouse functions
   textwidget_set_clickfunc(widget,batch_addtags_clickfunc);

   widget = zdialog_widget(zd,"deltags");
   textwidget_set_clickfunc(widget,batch_deltags_clickfunc);

   widget = zdialog_widget(zd,"deftags");
   textwidget_set_clickfunc(widget,batch_deftags_clickfunc);

   zdialog_resize(zd,500,400);                                             //  run dialog
   zdialog_run(zd,batchTags_dialog_event);
   zstat = zdialog_wait(zd);                                               //  wait for dialog completion
   zdialog_free(zd);                                                       //  kill dialog

   zdbatchtags = 0;

   if (zstat != 2)                                                         //  cancel
   {
      if (filecount) {
         for (ii = 0; filelist[ii]; ii++) 
            zfree(filelist[ii]);
         zfree(filelist);
      }

      Fmenulock = 0;
      return;
   }

   write_popup_text("open","Batch Tags",500,200,Mwin);                     //  status monitor popup window

   for (ii = 0; filelist[ii]; ii++)                                        //  loop all selected files
   {
      file = filelist[ii];                                                 //  display image
      err = f_open(file,0,0,0);
      if (err) continue;

      write_popup_text("write",file);                                      //  report progress
      zmainloop();
      
      load_filemeta(file);                                                 //  load current file tags

      for (jj = 1; ; jj++)                                                 //  remove tags if present
      {
         ptag = (char *) strField(deltags,tagdelims,jj);
         if (! ptag) break;
         if (*ptag == ' ') continue;
         err = del_tag(ptag,tags_imagetags);
         if (err) continue;
      }

      for (jj = 1; ; jj++)                                                 //  add new tags unless already
      {
         ptag = (char *) strField(addtags,tagdelims,jj);
         if (! ptag) break;
         if (*ptag == ' ') continue;
         err = add_tag(ptag,tags_imagetags,tagFcc);
         if (err == 2) {
            zmessageACK(Mwin,0,ZTX("%s \n too many tags"),file);
            break;
         }
      }

      save_filemeta(file);                                                 //  save tag changes
   }

   write_popup_text("write","COMPLETED");
   
   for (ii = 0; filelist[ii]; ii++) 
      zfree(filelist[ii]);
   zfree(filelist);

   Fmenulock = 0;
   return;
}


//  mouse click functions for widgets holding tags

void batch_addtags_clickfunc(GtkWidget *widget, int line, int pos)         //  a tag in the add list was clicked
{ 
   using namespace batchtags;

   char     *txline, *txtag, end;
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,tagdelims,end);
   if (! txtag) { zfree(txline); return; }
   
   del_tag(txtag,addtags);                                                 //  remove tag from list
   zdialog_stuff(zdbatchtags,"addtags",addtags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


void batch_deltags_clickfunc(GtkWidget *widget, int line, int pos)         //  a tag in the remove list was clicked
{ 
   using namespace batchtags;

   char     *txline, *txtag, end;
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,tagdelims,end);
   if (! txtag) { zfree(txline); return; }
   
   del_tag(txtag,deltags);                                                 //  remove tag from list
   zdialog_stuff(zdbatchtags,"deltags",deltags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


void batch_deftags_clickfunc(GtkWidget *widget, int line, int pos)         //  a defined tag was clicked
{
   using namespace batchtags;

   char     *txline, *txtag, end;
   cchar    *delims = tagdelims":";
   int      radadd;
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag) { zfree(txline); return; }
   
   zdialog_fetch(zdbatchtags,"radadd",radadd);                             //  which radio button?

   if (radadd) {
      add_tag(txtag,addtags,tagMcc);                                       //  add defined tag to tag add list
      zdialog_stuff(zdbatchtags,"addtags",addtags);
   }
   else {
      add_tag(txtag,deltags,tagMcc);                                       //  add defined tag to tag remove list
      zdialog_stuff(zdbatchtags,"deltags",deltags);
   }
   
   zfree(txline);
   zfree(txtag);
   return;
}


//  batchTags dialog event function

int batchTags_dialog_event(zdialog *zd, cchar *event)
{
   using namespace batchtags;

   int      ii;
   char     countmess[50];

   if (zd->zstat) 
   {
      if (zd->zstat == 1) {                                                //  manage tags
         zd->zstat = 0;                                                    //  keep dialog active
         zdialog_show(zd,0);                                               //  hide parent dialog
         manage_tags();
         zdialog_show(zd,1);
      }
      
      if (zd->zstat == 2) {                                                //  proceed
         if (! filecount || (*addtags <= ' ' && *deltags <= ' ')) {
            zmessageACK(Mwin,0,ZTX("specify files and tags"));
            zd->zstat = 0;                                                 //  keep dialog active
         }
      }

      return 0;                                                            //  cancel
   }

   if (strEqu(event,"files"))                                              //  select images to process
   {
      if (filelist) {                                                      //  free prior list
         for (ii = 0; filelist[ii]; ii++) 
            zfree(filelist[ii]);
         zfree(filelist);
      }

      zdialog_show(zd,0);                                                  //  hide parent dialog
      filelist = gallery_getfiles();                                       //  get file list from user
      zdialog_show(zd,1);

      filelist = filelist;

      if (filelist)                                                        //  count files in list
         for (ii = 0; filelist[ii]; ii++);
      else ii = 0;
      filecount = ii;
      
      snprintf(countmess,50,Bfileselected,filecount);
      zdialog_stuff(zd,"labcount",countmess);
   }
   
   return 1;
}
   

/**************************************************************************/

//  image file EXIF/IPTC data >> memory data:
//    meta_date, meta_rating, tags_imagetags, meta_comments, meta_caption,
//    meta_city, meta_country, meta_latitude, meta_longitude

void load_filemeta(cchar *file)
{
   int      ii, jj, cc;
   char     *pp;
   cchar    *exifkeys[10] = { exif_date_key, iptc_keywords_key, 
                              iptc_rating_key, exif_size_key,
                              exif_comment_key, iptc_caption_key,
                              exif_city_key, exif_country_key, 
                              exif_latitude_key, exif_longitude_key };

   char     **ppv, *imagedate, *imagekeywords, *imagestars, *imagesize;
   char     *imagecomms, *imagecapt;
   char     *imagecity, *imagecountry, *imagelatitude, *imagelongitude;
   
   if (Fmetachanged) return;                                               //  do not revert changes

   *tags_imagetags = *meta_date = *meta_comments = *meta_caption = 0;
   strcpy(meta_rating,"0");
   *meta_city = *meta_country = *meta_latitude = *meta_longitude = 0;
   
   ppv = exif_get(file,exifkeys,10);                                       //  get metadata from image file
   imagedate = ppv[0];
   imagekeywords = ppv[1];
   imagestars = ppv[2];
   imagesize = ppv[3];
   imagecomms = ppv[4];
   imagecapt = ppv[5];
   imagecity = ppv[6];
   imagecountry = ppv[7];
   imagelatitude = ppv[8];
   imagelongitude = ppv[9];
   
   if (imagedate) {
      exif_tagdate(imagedate,meta_date);                                   //  EXIF date/time >> yyyymmddhhmmss
      strcpy(meta_prdate,meta_date);
      zfree(imagedate);
   }

   if (imagekeywords)
   {
      for (ii = 1; ; ii++)
      {
         pp = (char *) strField(imagekeywords,tagdelims,ii);
         if (! pp) break;
         if (*pp == ' ') continue;
         cc = strlen(pp);
         if (cc >= tagcc) continue;                                        //  reject tags too big
         for (jj = 0; jj < cc; jj++)
            if (pp[jj] > 0 && pp[jj] < ' ') break;                         //  reject tags with control characters
         if (jj < cc) continue;
         add_tag(pp,tags_imagetags,tagFcc);                                //  add to file tags if unique
      }

      zfree(imagekeywords);
   }

   if (imagestars) {
      meta_rating[0] = *imagestars;
      if (meta_rating[0] < '0' || meta_rating[0] > '5') meta_rating[0] = '0';
      meta_rating[1] = 0;
      zfree(imagestars);
   }
   
   if (imagesize) {
      strncpy0(meta_size,imagesize,15);
      zfree(imagesize);
   }

   if (imagecomms) {
      strncpy0(meta_comments,imagecomms,exif_maxcc);
      zfree(imagecomms);
   }

   if (imagecapt) {
      strncpy0(meta_caption,imagecapt,exif_maxcc);
      zfree(imagecapt);
   }
   
   if (imagecity) {                                                        //  geotags
      strncpy0(meta_city,imagecity,99);
      zfree(imagecity);
   }
   else strcpy(meta_city,"null");                                          //  replace missing data with "null"

   if (imagecountry) {
      strncpy0(meta_country,imagecountry,99);
      zfree(imagecountry);
   }
   else strcpy(meta_country,"null");

   if (imagelatitude) {
      strncpy0(meta_latitude,imagelatitude,12);
      zfree(imagelatitude);
   }
   else strcpy(meta_latitude,"null");

   if (imagelongitude) {
      strncpy0(meta_longitude,imagelongitude,12);
      zfree(imagelongitude);
   }
   else strcpy(meta_longitude,"null");
   
   Fmetachanged = 0;
   return;
}


//  add metadata in memory to image file EXIF/IPTC data and image_index recs.
//  update defined tags file if any changes

void save_filemeta(cchar *file)
{
   cchar    *exifkeys[10] = { exif_date_key, iptc_keywords_key, 
                              iptc_rating_key, exif_size_key,
                              exif_comment_key, iptc_caption_key,
                              exif_city_key, exif_country_key, 
                              exif_latitude_key, exif_longitude_key };
   cchar       *exifdata[10];
   char        imagedate[24];

   if (! Fmetachanged) return;                                             //  no changes to tags etc.

   *imagedate = 0;
   if (*meta_date) {
      tag_exifdate(meta_date,imagedate);                                   //  yyyymmddhhmmss >> EXIF date/time
      strcpy(meta_prdate,meta_date);
   }
   
   exifdata[0] = imagedate;                                                //  update file EXIF/IPTC data
   exifdata[1] = tags_imagetags;
   exifdata[2] = meta_rating;
   exifdata[3] = meta_size;
   exifdata[4] = meta_comments;
   exifdata[5] = meta_caption;

   if (strEqu(meta_city,"null")) exifdata[6] = "";                         //  geotags
   else exifdata[6] = meta_city;                                           //  if "null" erase EXIF
   if (strEqu(meta_country,"null")) exifdata[7] = "";
   else exifdata[7] = meta_country;
   if (strEqu(meta_latitude,"null") || strEqu(meta_longitude,"null"))
      exifdata[8] = exifdata[9] = "";
   else {
      exifdata[8] = meta_latitude;
      exifdata[9] = meta_longitude;
   }

   exif_put(file,exifkeys,exifdata,10);                                    //  write EXIF

   update_image_index(file);                                               //  update image index file

   if (zdexifview) meta_view(0);                                           //  live EXIF/IPTC update

   Fmetachanged = 0;                                                       //  all changes saved
   return;
}


//  update image index record (replace updated file data)

void update_image_index(cchar *file)                                       //  overhauled
{
   char     gtags[200];
   int      err;
   sxrec_t  sxrec;
   STATB    statb;

   err = stat(file,&statb);                                                //  build new metadata record to insert 
   if (err) {                                                              //    or replace
      zmessageACK(Mwin,0,ZTX("file not found"));
      return;
   }
   
   memset(&sxrec,0,sizeof(sxrec_t));
   
   sxrec.file = (char *) file;                                             //  image filespec

   compact_time(statb.st_mtime,sxrec.fdate);                               //  convert to "yyyymmddhhmmss"

   if (*meta_date)                                                         //  image (photo) date 
      strncpy0(sxrec.pdate,meta_date,15);

   sxrec.rating[0] = meta_rating[0];                                       //  rating '0' to '5' stars
   sxrec.rating[1] = 0;
   
   strncpy0(sxrec.size,meta_size,15);
   
   if (*tags_imagetags)                                                    //  tags
      sxrec.tags = tags_imagetags;
  
   if (*meta_caption)                                                      //  user caption
      sxrec.capt = meta_caption;
   
   if (*meta_comments)                                                     //  user comments
      sxrec.comms = meta_comments;

   if (! *meta_city) strcpy(meta_city,"null");                             //  geotags
   if (! *meta_country) strcpy(meta_country,"null");
   if (! *meta_latitude) strcpy(meta_latitude,"null");                     //  "null" for city/country is searchable
   if (! *meta_longitude) strcpy(meta_longitude,"null");
   
   snprintf(gtags,200,"%s^ %s^ %s^ %s",meta_city, meta_country, 
                                  meta_latitude, meta_longitude);
   sxrec.gtags = gtags;

   put_sxrec(&sxrec,file);
   return;
}


//  delete given image file from image index recs.

void delete_image_index(cchar *file)
{
   put_sxrec(null,file);
   return;
}


/**************************************************************************
            GEOTAG functions
***************************************************************************/

struct geolocs_t {                                                         //  geotag location data, memory DB
   char     *city, *country;
   char     *lati, *longi;
   float    flati, flongi;
};
geolocs_t   *geolocs;
int      Ngeolocs = 0;                                                     //  number of geotag location records
int      Ngeolocs2 = 0;                                                    //  those with images

zdialog  *zd_geotags;                                                      //  active zdialog to get geotags data
int      worldmap_range = 20;                                              //  default geotag search range, km
char     prevcity[100] = "", prevcountry[100] = "";                        //  last geotag data used
char     prevlatitude[20] = "", prevlongitude[20] = "";                    //  (recalled with [prev] button)

typedef void worldmap_callback_t(float flati, float flongi);
worldmap_callback_t  *worldmap_callback = 0;

int geotags_choosecity(char *location[2], char *coord[2]);                 //  choose one city from multiple options
cchar * web_geocode(char *location[2], char *coord[2]);                    //  find latitude/longitude via web service
int get_worldmap_coordinates(int mx, int my, float &flat, float &flong);   //  pixel position to latitude/longitude
int get_worldmap_position(float flat, float flong, int &mx, int &my);      //  latitude/longitude to pixel position


/**************************************************************************/

//  validate and convert earth coordinates, latitude and longitude
//  return: 0  OK
//          1  both are missing ("null" or "")
//          2  invalid data
//  if status is > 0, 0.0 is returned for both values

int validate_latilongi(char *latitude, char *longitude, float &flati, float &flongi)
{
   if (! latitude || *latitude == 0 || strEqu(latitude,"null"))
      if (! longitude || *longitude == 0 || strEqu(longitude,"null"))
         goto status1;                                                     //  both missing
   
   if (! latitude || *latitude == 0 || strEqu(latitude,"null"))
      goto status2;                                                        //  one missing
   if (! longitude || *longitude == 0 || strEqu(longitude,"null")) 
      goto status2;
   
   flati = atof(latitude);                                                 //  convert strings to float
   flongi = atof(longitude);
   if (flati < -90.0 || flati > +90.0) goto status2;                       //  check range
   if (flongi < -180.0 || flongi > +180.0) goto status2;
   if (flati == 0.0 && flongi == 0.0) goto status2;                        //  reject both = 0.0 
   return 0;

status1:
   flati = flongi = 0.0;                                                   //  both missing
   return 1;

status2:                                                                   //  either missing or invalid
   flati = flongi = 0.0;
   return 2;
}


/**************************************************************************/

//  download world-map-mercator.jpg and cities-geotags files.
//  save in /.../.fotoxx/geotags/*

void m_download_geolocs(GtkWidget *, cchar *)
{
   int      yn, err;
   char     scriptfile[200];
   char     destfile1[200], sourcefile1[200];
   char     destfile2[200], sourcefile2[200];
   FILE     *fid;

   F1_help_topic = "download_geolocs";

   if (Ngeolocs) zfree(geolocs);
   Ngeolocs = 0;

   yn = zmessageYN(Mwin,ZTX("Download geolocations data (18 megabytes). \n"
                            "Save in %s \n"
                            "Proceed?"),geotags_dirk);
   if (! yn) return;

   if (checkpend("all")) return;                                           //  check nothing pending

   snprintf(sourcefile1,200,"http://www.kornelix.com/uploads/1/3/0/3/13035936/world-map-20k.jpg");
   snprintf(destfile1,200,"%s/world-map-20K.jpg",geotags_dirk);
   snprintf(sourcefile2,200,"http://www.kornelix.com/uploads/1/3/0/3/13035936/cities-geotags");
   snprintf(destfile2,200,"%s/cities-geotags",geotags_dirk);
   snprintf(scriptfile,200,"%s/download.scr",geotags_dirk);
   
   fid = fopen(scriptfile,"w");
   if (! fid) { 
      zmessageACK(Mwin,0,"cannot create script file \n %s",strerror(errno)); 
      return;
   }
   
   fprintf(fid,"#! /bin/bash \n");                                         //  build bash script
   fprintf(fid,"#  download geotags data \n");
   fprintf(fid,"# \n");
   fprintf(fid,"echo DOWNLOAD WORLD MAP \n");
   fprintf(fid,"wget --progress=dot:mega -O %s %s 2>&1 \n",destfile1,sourcefile1);
   fprintf(fid,"err=$? \n");
   fprintf(fid,"if [ $err -ne 0 ]; then exit $err; fi \n");
   fprintf(fid,"echo DOWNLOAD GEOTAGS FILE \n");
   fprintf(fid,"wget -O %s %s 2>&1 \n",destfile2,sourcefile2);
   fprintf(fid,"err=$? \n");
   fprintf(fid,"if [ $err -ne 0 ]; then exit $err; fi \n");
   fprintf(fid,"echo DONE \n");
   fprintf(fid,"exit 0 \n");
   fclose(fid);

   chmod(scriptfile,0744);
   
   Ffuncbusy++;  
   snprintf(command,ccc,"bash %s",scriptfile);                             //  run the script
   err = popup_command(command,800,600,Mwin);
   if (err) zmessageACK(Mwin,0,ZTX("download failed"));
   Ffuncbusy--;  

   return;
}


/**************************************************************************/

//  Initialize for geotag functions.
//  Load geolocations data into memory.
//  Returns no. geolocations or 0 if downloads are needed.
//  (user is informed with popup message)

int init_geolocs()
{
   int  init_glocs_comp(cchar *rec1, cchar *rec2);
   
   char     geotagsfile[200];
   char     buff[200], city[100], country[100];
   char     latitude[20], longitude[20];
   char     *filename, *gtags, *pp;
   float    flati, flongi;
   int      err, ftf, cc, ii, jj;
   int      found, keep, Ngeotags;
   FILE     *fid;
   sxrec_t  sxrec;
   STATB    statbuf;

   if (Ngeolocs) return Ngeolocs;                                          //  already done
   if (checkpend("lock")) return 0;                                        //  v.14.09

   snprintf(geotagsfile,200,"%s/cities-geotags",geotags_dirk);             //  read cities-geotags file
   err = stat(geotagsfile,&statbuf);
   if (err) {
      zmessageACK(Mwin,0,ZTX("please download geolocations data"));
      return 0;
   }

   Ffuncbusy = 1;
   Fmenulock = 1;

   cc = maxgeotags * sizeof(geolocs_t);                                    //  get memory for geotag locations DB 
   geolocs = (geolocs_t *) zmalloc(cc);
   
   //  populate geolocs[] from search-index file (= image EXIF data)

   Ngeotags = 0;                                                           //  images with geotags

   ftf = 1;
   while (true)
   {
      zmainloop(100);

      err = read_sxrec_seq(sxrec,ftf);                                     //  read image index recs.
      if (err) break;
      
      filename = sxrec.file;
      gtags = sxrec.gtags;

      strcpy(city,"null");
      strcpy(country,"null");
      strcpy(latitude,"null");
      strcpy(longitude,"null");
      
      found = 0;
      
      pp = (char *) strField(gtags,'^',1);                                 //  city name or "null"
      if (pp) {
         strncpy0(city,pp,99);
         found++;
      }
      
      pp = (char *) strField(gtags,'^',2);                                 //  country
      if (pp) { 
         strncpy0(country,pp,99);
         found++;
      }

      pp = (char *) strField(gtags,'^',3);                                 //  latitude
      if (pp) strncpy0(latitude,pp,19);

      pp = (char *) strField(gtags,'^',4);                                 //  longitude
      if (pp) strncpy0(longitude,pp,19);
      
      err = validate_latilongi(latitude,longitude,flati,flongi);           //  validate and replace bad data
      if (err) {
         strcpy(latitude,"null");                                          //  missing (1) or bad (2) data
         strcpy(longitude,"null");
         if (err == 2) {                                                   //  exclude invalid
            printz("*** bad geotag in image_index: %s \n",filename);
            found = 0;
         }
      }
      
      if (strNeq(city,"null") || strNeq(country,"null"))                   //  count images with geotags
         Ngeotags++;

      if (found > 0 && Ngeolocs) {                                         //  at least city or country was present
         ii = Ngeolocs - 1;
         if (strEqu(city,geolocs[ii].city))                                //  ignore same city in sequence (frequent)
            if (strEqu(country,geolocs[ii].country)) found = 0;            //    to reduce subsequent sort
      }

      if (found > 0) {  
         ii = Ngeolocs;
         geolocs[ii].city = zstrdup(city);
         geolocs[ii].country = zstrdup(country);
         geolocs[ii].lati = zstrdup(latitude);
         geolocs[ii].longi = zstrdup(longitude);
         geolocs[ii].flati = atof(latitude);
         geolocs[ii].flongi = atof(longitude);
         Ngeolocs++;
      }
   
      zfree(sxrec.file);
      zfree(sxrec.tags);
      zfree(sxrec.capt);
      zfree(sxrec.comms);
      zfree(sxrec.gtags);

      if (Ngeolocs < maxgeotags-1) continue;                               //  -1 necessary, see below
      
      HeapSort((char *) geolocs, sizeof(geolocs_t), Ngeolocs, init_glocs_comp);

      for (ii = 0, jj = 1; jj < Ngeolocs; jj++)                            //  eliminate duplicate cities
      {
         keep = 0;
         if (strNeq(geolocs[jj].country,geolocs[ii].country)) keep = 1;
         if (! keep && strNeq(geolocs[jj].city,geolocs[ii].city)) keep = 1;
         if (keep) {
            ii++;
            geolocs[ii] = geolocs[jj];
         }
         else {
            zfree(geolocs[jj].country);
            zfree(geolocs[jj].city);
            zfree(geolocs[jj].lati);
            zfree(geolocs[jj].longi);
         }
      }
      
      Ngeolocs = ii + 1;
      if (Ngeolocs < maxgeotags-10) continue;
      zmessLogACK(Mwin,"max. geotags %d exceeded",maxgeotags);
      break;
   }
   
   if (Ngeolocs > 1)                                                       //  sort by country, city
      HeapSort((char *) geolocs, sizeof(geolocs_t), Ngeolocs, init_glocs_comp);

   for (ii = 0, jj = 1; jj < Ngeolocs; jj++)                               //  eliminate duplicate cities
   {
      keep = 0;
      if (strNeq(geolocs[jj].country,geolocs[ii].country)) keep = 1;
      if (! keep && strNeq(geolocs[jj].city,geolocs[ii].city)) keep = 1;
      if (keep) {
         ii++;
         geolocs[ii] = geolocs[jj];
      }
      else {
         zfree(geolocs[jj].country);
         zfree(geolocs[jj].city);
         zfree(geolocs[jj].lati);
         zfree(geolocs[jj].longi);
      }
   }
   
   Ngeolocs = ii + 1;
   Ngeolocs2 = Ngeolocs;                                                   //  geolocs from images 

   printz("image files with geotags: %d  locations: %d \n",Ngeotags,Ngeolocs);

   //  add data from cities-geotags file to geolocs[] data
   //  (after data from image files)

   snprintf(geotagsfile,200,"%s/cities-geotags",geotags_dirk);             //  read cities-geotags file
   fid = fopen(geotagsfile,"r");
   if (! fid) {
      zmessageACK(Mwin,0,strerror(errno));
      Ngeolocs = 0;
      Ffuncbusy = 0;
      Fmenulock = 0;
      return 0;
   }
   
   while (true)
   {
      pp = fgets_trim(buff,200,fid,1);                                     //  read each city record into memory
      if (! pp) break;                                                     //  (remove trailing \n)
      if (strlen(pp) < 2) continue;
      
      ii = Ngeolocs;
      
      pp = (char *) strField(buff,'^',1);                                  //  get latitude
      if (! pp) goto badrec;
      strTrim2(pp);
      geolocs[ii].lati = zstrdup(pp);
      geolocs[ii].flati = atof(pp);

      pp = (char *) strField(buff,'^',2);                                  //  longitude
      if (! pp) goto badrec;
      strTrim2(pp);
      geolocs[ii].longi = zstrdup(pp);
      geolocs[ii].flongi = atof(pp);
      
      err = validate_latilongi(geolocs[ii].lati,geolocs[ii].longi,flati,flongi);
      if (err) goto badrec;

      pp = (char *) strField(buff,'^',3);                                  //  city
      if (! pp) goto badrec;
      strTrim2(pp);
      geolocs[ii].city = zstrdup(pp);
         
      pp = (char *) strField(buff,'^',4);                                  //  country
      if (! pp) goto badrec;
      strTrim2(pp);
      geolocs[ii].country = zstrdup(pp);
      
      Ngeolocs++;                                                          //  count good recs found
      if (Ngeolocs < maxgeotags) continue;
      zmessLogACK(Mwin,"max. geotags %d exceeded",maxgeotags);
      break;
   
   badrec:
      printz("*** bad cities-geotags record: %s \n",buff);
      continue;
   }
   
   fclose(fid);
   printz("total geolocations %d  with images: %d \n",Ngeolocs,Ngeolocs2);
   
   Ffuncbusy = 0;
   Fmenulock = 0;
   return Ngeolocs;
}


//  Compare 2 geolocs records by country and city
//  return  <0  0  >0   for   rec1  <  ==  >  rec2.

int  init_glocs_comp(cchar *rec1, cchar *rec2)
{
   int      ii;

   char * country1 = ((geolocs_t *) rec1)->country;                        //  compare countries
   char * country2 = ((geolocs_t *) rec2)->country;
   ii = strcmp(country1,country2);
   if (ii) return ii;

   char * city1 = ((geolocs_t *) rec1)->city;                              //  compare cities
   char * city2 = ((geolocs_t *) rec2)->city;
   ii = strcmp(city1,city2);
   return ii;
}


/**************************************************************************/

//  get latitude/longitude data for a city [ country ]
//  inputs:        location[0] = city
//                 location[1] = country (opt)
//  outputs:       coord[2] = latitude -90 to +90, longitude -180 to +180
//                 matches[20][2] up to 20 matching city/country locations
//  returns:       no. matches for input city [ country ]
//                 (max. 20)
//
//  use null or "" for missing city or country input (no NULL pointer)
//  coordinates are returned for the first match only, if any
//

int get_geolocs(char *location[2], char *coord[2], char *matches[20][2])
{
   int      cc, ii, jj, Nmatch;
   int      fcity = 0, fcountry = 0;
   
   fcity = location[0] && *location[0];                                    //  city or country must be present
   fcountry = location[1] && *location[1];
   if (! fcity && ! fcountry) return 0;   
   
   for (ii = Nmatch = 0; ii < Ngeolocs; ii++)                              //  search for exact city [ country ]
   {
      if (fcity && strcasecmp(location[0],geolocs[ii].city) != 0) continue;
      if (fcountry && strcasecmp(location[1],geolocs[ii].country) != 0) continue;

      for (jj = 0; jj < Nmatch; jj++) {                                    //  look for duplicate match 
         if (strEqu(geolocs[ii].city,matches[jj][0]))                      //  (EXIF and cities-geotag file)
            if (strEqu(geolocs[ii].country,matches[jj][1])) break;
      }
      if (jj < Nmatch) continue;                                           //  discard duplicates

      matches[Nmatch][0] = geolocs[ii].city;
      matches[Nmatch][1] = geolocs[ii].country;
      Nmatch++;
      if (Nmatch == 20) break;                                             //  no more than 20 matches are reported
      if (Nmatch == 1) {
         coord[0] = geolocs[ii].lati;                                      //  return geolat/long of 1st match
         coord[1] = geolocs[ii].longi;
         StripZeros(coord[0]);
         StripZeros(coord[1]);
      }
   }
   
   if (Nmatch) return Nmatch;                                              //  exact match found

   for (ii = Nmatch  = 0; ii < Ngeolocs; ii++)                             //  search for partial city [ country ]
   {
      if (strEqu("null",geolocs[ii].city)) continue; 

      if (fcity) {
         cc = strlen(location[0]);
         if (strncasecmp(location[0],geolocs[ii].city,cc) != 0) continue;
      }
      if (fcountry) {
         cc = strlen(location[1]);
         if (strncasecmp(location[1],geolocs[ii].country,cc) != 0) continue;
      }
      matches[Nmatch][0] = geolocs[ii].city;
      matches[Nmatch][1] = geolocs[ii].country;
      Nmatch++;
      if (Nmatch == 20) break;                                             //  no more than 20 matches are reported
      if (Nmatch == 1) {
         coord[0] = geolocs[ii].lati;                                      //  return geolat/long of 1st match
         coord[1] = geolocs[ii].longi;
         StripZeros(coord[0]);
         StripZeros(coord[1]);
      }
   }

   return Nmatch;
}


/**************************************************************************/

//  Save geotags in image file EXIF and image index file.
//  location[2] = city and country
//  coord[2] = latitude and longitude
//  return value:  0    OK, no geotag revision (incomplete data)
//                 1    OK, no geotag revision (matches existing data)
//                 2    OK, geotag lat/long updated
//                 3    OK, geotag new location added
//                -1    error, lat/long bad
//  (cities-geotags-new is no longer used)

int put_geolocs(char *location[2], char *coord[2])
{
   char        acoord[2][20];
   float       flati, flongi;
   int         ii, err, Fnew = 0, Fchange = 0;
   int         px, py, mx, my;
   uint8       *pixel;
   int         retval;
   
   if (! curr_file) return 0;

   err = validate_latilongi(coord[0],coord[1],flati,flongi);
   if (err) {
      if (err == 2) goto badcoord;                                         //  reject bad data
      strcpy(acoord[0],"null");                                            //  replace missing data with "null"
      strcpy(acoord[1],"null");
      flati = flongi = 0;                                                  //  lati/longi missing value 
   }
   else {
      snprintf(acoord[0],20,"%.4f",flati);                                 //  reformat with std. precision
      snprintf(acoord[1],20,"%.4f",flongi);
      StripZeros(acoord[0]);
      StripZeros(acoord[1]);
   }
   
   if (strNeq(location[0],"null"))                                         //  unless null,
      *location[0] = toupper(*location[0]);                                //  force capitalization
   if (strNeq(location[1],"null"))
      *location[1] = toupper(*location[1]);

   strncpy0(meta_city,location[0],99);                                     //  save geotags in image file EXIF
   strncpy0(meta_country,location[1],99);                                  //    and in search-index file
   strncpy0(meta_latitude,acoord[0],12);
   strncpy0(meta_longitude,acoord[1],12);

   Fmetachanged++;                                                         //  update file EXIF data
   save_filemeta(curr_file);
   
   if (! *location[0] || strEqu(location[0],"null")) return 0;             //  quit here if city data not complete
   if (! *location[1] || strEqu(location[1],"null")) return 0;
   
   for (ii = 0; ii < Ngeolocs; ii++) {                                     //  search geotags for city, country
      if (strcasecmp(location[0],geolocs[ii].city) != 0) continue;         //  (case-insensitive compare)
      if (strcasecmp(location[1],geolocs[ii].country) == 0) break;
   }

   if (ii < Ngeolocs) {                                                    //  found, check for revised lat/long
      if (strNeq(geolocs[ii].lati,acoord[0])) Fchange = 1;
      if (strNeq(geolocs[ii].longi,acoord[1])) Fchange = 1;
      if (strNeq(geolocs[ii].city,location[0])) Fchange = 1;               //  or revised capitalization
      if (strNeq(geolocs[ii].country,location[1])) Fchange = 1;
   }
   else Fnew = 1;                                                          //  a new city, country

   if (Fnew + Fchange == 0) return 1;                                      //  no change

   if (Fchange) 
   {
      zfree(geolocs[ii].city);                                             //  change geotag data in memory
      geolocs[ii].city = zstrdup(location[0]);                             //  (to be used subsequently)
      zfree(geolocs[ii].country);
      geolocs[ii].country = zstrdup(location[1]);                          //  presense in image EXIF will make
      zfree(geolocs[ii].lati);                                             //    this the preferred version
      geolocs[ii].lati = zstrdup(acoord[0]);
      zfree(geolocs[ii].longi);
      geolocs[ii].longi = zstrdup(acoord[1]);
      geolocs[ii].flati = flati;
      geolocs[ii].flongi = flongi;
      retval = 2;
   }

   else if (Fnew)
   {
      if (Ngeolocs == maxgeotags) {
         zmessLogACK(Mwin,"max. geotags %d exceeded",maxgeotags);
         return -1;
      }
      for (ii = Ngeolocs; ii > 0; ii--)                                    //  shift all geotag data up
         geolocs[ii] = geolocs[ii-1];
      Ngeolocs++;

      ii = 0;
      geolocs[ii].city = zstrdup(location[0]);                             //  new geotag is now first
      geolocs[ii].country = zstrdup(location[1]);                          //  (find again faster)
      geolocs[ii].lati = zstrdup(acoord[0]);
      geolocs[ii].longi = zstrdup(acoord[1]);
      geolocs[ii].flati = flati;
      geolocs[ii].flongi = flongi;
      retval = 3;
   }

   else return -1;                                                         //  should not happen

   if (flati && flongi && Wstate.fpxb) {                                   //  put new red dot on map    v.14.08
      err = get_worldmap_position(flati,flongi,mx,my);
      if (! err) {
         for (px = -1; px <= 1; px++)                                      //  paint 3x3 block of pixels
         for (py = -1; py <= 1; py++) 
         {
            pixel = PXBpix(Wstate.fpxb,mx+px,my+py);
            pixel[0] = 255;
            pixel[1] = pixel[2] = 0;
         }
      }
   }

   return retval;

badcoord:
   zmessageACK(Mwin,0,ZTX("bad latitude/longitude: %s %s"),coord[0],coord[1]);
   return -1;
}


/**************************************************************************/

//  initialize for world map functions
//  load and initialize the world map image

int  load_worldmap()                                                       //  v.14.08
{
   int         mx, my, px, py;
   int         ii, err;
   float       flati, flongi;
   uint8       *pixel;
   zdialog     *zdbusy;

   if (Wstate.fpxb) return 1;                                              //  world map already loaded
   if (! init_geolocs() ) return 0;                                        //  insure geolocations are loaded
   if (checkpend("all")) return 0;

   printz("loading %s \n",worldmap_file);

   Ffuncbusy = 1;
   Fmenulock = 1;

   zdbusy = zmessage_post(Mwin,0,"Loading world map ...",null);
   for (ii = 0; ii < 10; ii++) {                                           //  required to get message out        v.14.08
      zmainloop();                                                         //  (time for x11 to react?)
      zsleep(0.01);
   }

   Wstate.fpxb = PXB_load(worldmap_file,0);                                //  load world map image, 20K x 13.3K
   if (! Wstate.fpxb || Wstate.fpxb->ww != 20000) {                        //  (800 MB in memory)
      zmessageACK(Mwin,0,"please download geolocations data");
      free_worldmap();
      Ffuncbusy = 0;
      Fmenulock = 0;
      return 0;
   }
   
   for (ii = 0; ii < Ngeolocs2; ii++)                                      //  paint red dots on world map
   {                                                                       //    where images are present
      flati = geolocs[ii].flati;
      flongi = geolocs[ii].flongi;
      err = get_worldmap_position(flati,flongi,mx,my);
      if (err) continue;
      for (px = -1; px <= 1; px++)                                         //  paint 3x3 block of pixels
      for (py = -1; py <= 1; py++) 
      {
         pixel = PXBpix(Wstate.fpxb,mx+px,my+py);
         pixel[0] = 255;
         pixel[1] = pixel[2] = 0;
      }
   }

   zdialog_free(zdbusy);                                                   //  kill the message                   v.14.08

   Ffuncbusy = 0;
   Fmenulock = 0;
   return 1;
}


//  free memory used for world map (800 MB)
//  used by edit_setup() to maximize available memory

void free_worldmap()                                                       //  v.14.08
{
   if (Wstate.fpxb) PXB_free(Wstate.fpxb);
   Wstate.fpxb = 0;
   return;
}


/**************************************************************************/

//  World map functions.
//  See the Wikipedia article on Mercator Projection for the math.

namespace worldmap                                                         //  based on 20K x 13.3K world map     v.14.07
{
   int         xmargin = 166;
   int         ymargin = 133;
   int         mapw = 19619;                                               //  map width less margins
   int         maph = 13077;                                               //  map height less margins
   int         mape = 9211;                                                //  equator less margins
   double      px, py;
   double      R, rad = 180.0 / PI;
}


//  Adjust latitude based on a correction table.
//  Input is latitude ranged 0 to +90.

double worldmap_fudgelat(double flat)
{
   int      ii, jj;
   double   span;
   double   fudge[10] = { 0.00, 0.07, 0.12, 0.16, 0.18, 0.19, 0.17, 0.12, 0.07, 0.0 };

   flat = 0.1 * flat;
   for (ii = 1; ii < 10; ii++)
      if (flat < ii) break;
   if (ii > 9) ii = 9;
   jj = ii - 1;
   span = flat - jj;
   return fudge[jj] + span * (fudge[ii] - fudge[jj]);
}  


//  Convert a world map position mx/my into latitude and longitude.
//  Return 0 if OK, +N if error (off the map).

int get_worldmap_coordinates(int mx, int my, float &flat, float &flong)
{
   using namespace worldmap;
   
   flat = flong = 0;
   
   mx = mx - xmargin;                                                      //  remove margins
   my = my - ymargin;

   if (mx < 0 || mx > mapw-1) return 1;                                    //  check if on the map
   if (my < 0 || my > maph-1) return 1;
   
   px = (2.0 * mx - mapw) / mapw;                                          //  -1 to +1
   flong = 180.0 * px;                                                     //  longitude -180 to +180

   py = fabs(mape - my);                                                   //  displacement from equator
   R = 2.0 * PI * py / mapw;
   R = exp(R);
   R = 2.0 * atan(R) * rad;
   flat = R - 90.0;                                                        //  latitude 0 to +90

   flat += worldmap_fudgelat(flat);                                        //  add map correction

   if (my > mape) flat = -flat;                                            //  S. latitude 0 to -90
   
   return 0;
}


//  Convert latitude and longitude into a world map position mx/my;
//  Return 0 if OK, +N if error (off the map).

int get_worldmap_position(float flat, float flong, int &mx, int &my)
{
   using namespace worldmap;
   
   mx = my = 0;

   if (flat < -90 || flat > 90) return 1;                                  //  check if out of bounds
   if (flong < -180 || flong > 180) return 1;
   
   mx = mapw * (0.5 + flong / 360.0) + 0.5;                                //  0 to mapw

   if (flat >= 0) flat -= worldmap_fudgelat(flat);                         //  subtract map correction
   else flat += worldmap_fudgelat(-flat);

   flat = 0.5 * flat / rad + PI / 4.0;
   py = 0.5 * mapw / PI * log(tan(flat));                                  //  displacement from equator
   my = mape - py + 0.5;                                                   //  0 to maph
   
   if (mx < 0 || mx > mapw) return 1;
   if (my < 0 || my > maph) return 1;
   
   mx += xmargin;                                                          //  add margins
   my += ymargin;

   return 0;
}


/**************************************************************************/

//  show a world map and get geotag location from mouse click on map

int click_worldmap()
{
   void click_worldmap_mousefunc();
   
   if (! load_worldmap()) return 0;                                        //  load geolocs and world map
   m_viewmode(0,"W");                                                      //  set view mode W
   takeMouse(click_worldmap_mousefunc,dragcursor);                         //  connect mouse 
   return 1;
}


//  Respond to mouse movement and left clicks on world map.
//  Set longitude and latitude, and city and country.

void click_worldmap_mousefunc()
{
   int         err, mx, my, ii, minii;
   char        *city, *country;
   float       flati, flongi, glati, glongi;
   float       dist, mindist;
   static      char  *pcity = 0;
   
   if (FGW != 'W') {                                                       //  no longer in view W
      poptext_window(0,0,0,0,0,0);
      freeMouse();
      return;
   }

   if (LMclick && Cstate->fzoom < 1.0) {                                   //  zoom to 100% in one step
      m_zoom(null,"100");
      return;
   }

   if ((Mxdrag || Mydrag)) return;                                         //  pan/scroll - handle normally
   if (RMclick) return;                                                    //  zoom - fit window, handle normally

   mx = Mxposn;                                                            //  mouse position, image space
   my = Myposn;

   err = get_worldmap_coordinates(mx,my,flati,flongi);                     //  convert map location to lat/long
   if (err) return;

   dist = mindist = 999999;
   minii = 0;

   for (ii = 0; ii < Ngeolocs; ii++)                                       //  find nearest city/country
   {
      glati = geolocs[ii].flati;
      dist = (flati - glati) * (flati - glati);
      if (dist > mindist) continue;
      glongi = geolocs[ii].flongi;
      dist += (flongi - glongi) * (flongi - glongi);                       //  degrees**2
      if (dist > mindist) continue;
      mindist = dist;
      minii = ii;
   }

   mindist = 111 * sqrtf(mindist);                                         //  degrees**2 to km at earth surface

   if (mindist <= worldmap_range) {                                        //  within range of a known place
      ii = minii;
      city = geolocs[ii].city;
      country = geolocs[ii].country;
      flati = geolocs[ii].flati;
      flongi = geolocs[ii].flongi;
   }
   else  city = country = 0;
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      poptext_window(0,0,0,0,0,0);                                         //  erase popup text
      
      if (zd_geotags) {
         zdialog_stuff(zd_geotags,"city",city);                            //  stuff calling dialog
         zdialog_stuff(zd_geotags,"country",country);
         zdialog_stuff(zd_geotags,"latitude",flati);
         zdialog_stuff(zd_geotags,"longitude",flongi);
         zdialog_send_event(zd_geotags,"worldmap");                        //  activate calling dialog
      }
      
      else if (worldmap_callback && city) {                                //  callback function
         freeMouse();                                                      //  disconnect mouse
         worldmap_callback(flati,flongi);                                  //  do callback function
      }
   }

   else if (city) {                                                        //  mouse movement, no click
      if (! pcity || strNeq(city,pcity)) {
         poptext_window(city,MWIN,Mwxposn,Mwyposn,0.1,2);                  //  popup the city name at mouse
         pcity = city;
      }
   }

   else if (pcity) {
      poptext_window(0,0,0,0,0,0);                                         //  erase popup text
      pcity = 0;
   }

   return;
}


/**************************************************************************/

//  World map test function.
//  Convert clicked position into earth coordinates and 
//  then the earth coordinates back into the clicked position.
//  Results: accuracy is about 0.02 degrees or 2 km.

void m_worldmap_test(GtkWidget *, cchar *menu)
{
   void worldmap_test_mousefunc();
   
   if (! load_worldmap()) return;                                          //  initialize geotags

   m_viewmode(0,"W");                                                      //  set W view mode

   takeMouse(worldmap_test_mousefunc,dragcursor);                          //  connect mouse 

   return;
}


//  Respond to mouse movement and clicks on world map.
//  Set longitude and latitude, and city and country.

void worldmap_test_mousefunc()
{
   int         err, mx1, my1, mx2, my2;
   float       flati, flongi;
   char        text[100];
   
   if (FGW != 'W') {                                                       //  no longer in view mode W
      poptext_window(0,0,0,0,0,0);
      freeMouse();
      return;
   }

   if (! LMclick) return;
   LMclick = 0;
   
   mx1 = Mxposn;                                                           //  mouse position, image space
   my1 = Myposn;

   err = get_worldmap_coordinates(mx1,my1,flati,flongi);                   //  convert map location to lat/long
   if (err) return;

   err = get_worldmap_position(flati,flongi,mx2,my2);                      //  convert lat/long to map location
   if (err) return;
   
   snprintf(text,100,"%d/%d  %.2f/%.2f  %d/%d",mx1,my1,flati,flongi,mx2,my2);
   poptext_window(text,MWIN,Mwxposn,Mwyposn,0.1,5);

   return;
}


/**************************************************************************/

//  add or edit image geotags - city, country, latitude, longitude

void m_edit_geotags(GtkWidget *, cchar *menu)
{
   int  edit_geotags_dialog_event(zdialog *zd, cchar *event);

/**
                  Edit Geotags

      city [______________]  country [______________]
      latitude [_______] longitude [_______]
      
      [find] [web] [prev] [map] [apply] [clear] [done]

**/

   cchar    *title = ZTX("Edit Geotags");
   cchar    *mapquest1 = ZTX("Geocoding web service courtesy of");
   cchar    *mapquest2 = "http://www.mapquest.com";
   
   zdialog  *zd;

   F1_help_topic = "edit_geotags";

   if (! init_geolocs()) return;                                           //  initialize geotags
   if (! curr_file) return;
   if (checkpend("lock")) return;                                          //  check nothing pending

   if (! zdeditgeotags)                                                    //  start dialog if not already
   {   
      zdeditgeotags = zdialog_new(title,Mwin,Bfind,Bweb,Bprev,Bmap,Bapply,Bclear,Bdone,null);
      zd = zdeditgeotags;
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labcity","hb1",ZTX("city"),"space=5");
      zdialog_add_widget(zd,"entry","city","hb1",0,"expand");
      zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
      zdialog_add_widget(zd,"label","labcountry","hb1",ZTX("country"),"space=5");
      zdialog_add_widget(zd,"entry","country","hb1",0,"expand");
      zdialog_add_widget(zd,"hbox","hb4","dialog");
      zdialog_add_widget(zd,"label","lablat","hb4","Latitude","space=3");
      zdialog_add_widget(zd,"entry","latitude","hb4",0,"scc=10");
      zdialog_add_widget(zd,"label","space","hb4",0,"space=5");
      zdialog_add_widget(zd,"label","lablong","hb4","Longitude","space=3");
      zdialog_add_widget(zd,"entry","longitude","hb4",0,"scc=10");
      zdialog_add_widget(zd,"hbox","hbmq","dialog");
      zdialog_add_widget(zd,"label","labmq","hbmq",mapquest1,"space=3");
      zdialog_add_widget(zd,"link","MapQuest","hbmq",mapquest2);

      zdialog_run(zd,edit_geotags_dialog_event);
   }
   
   load_filemeta(curr_file);                                               //  get current image geotags (EXIF)
   zd = zdeditgeotags;

   if (strEqu(meta_city,"null")) zdialog_stuff(zd,"city","");
   else zdialog_stuff(zd,"city",meta_city);

   if (strEqu(meta_country,"null")) zdialog_stuff(zd,"country","");
   else zdialog_stuff(zd,"country",meta_country);

   if (strEqu(meta_latitude,"null")) zdialog_stuff(zd,"latitude","");
   else zdialog_stuff(zd,"latitude",meta_latitude);

   if (strEqu(meta_longitude,"null")) zdialog_stuff(zd,"longitude","");
   else zdialog_stuff(zd,"longitude",meta_longitude);

   return;
}


//  dialog event and completion callback function

int edit_geotags_dialog_event(zdialog *zd, cchar *event)
{
   int          zstat, Nmatch;
   char         *location[2], *coord[2], *matches[20][2];
   char         city[100], country[100];
   char         latitude[20], longitude[20], temp[20];
   float        fcoord[2];
   cchar        *errmess;

   if (strEqu(event,"worldmap"))                                           //  have geotags data from world map
      return 1;                                                            //  do nothing
   
   if (! curr_file) return 1;

   if (strEqu(event,"enter")) zd->zstat = 5;                               //  [apply]                            v.14.03
   
   if (strstr("city country latitude longitude",event))                    //  dialog inputs changed              v.14.09
      Fmetachanged++;

   if (! zd->zstat) return 1;                                              //  wait for action button
   zstat = zd->zstat;
   zd->zstat = 0;                                                          //  keep dialog active
   m_viewmode(0,"F");                                                      //  back to image

   zdialog_fetch(zd,"city",city,99);                                       //  get city [country] from dialog
   zdialog_fetch(zd,"country",country,99);
   location[0] = city;
   location[1] = country;
   zdialog_fetch(zd,"latitude",temp,12);                                   //  and latitude, longitude
   repl_1str(temp,latitude,",",".");                                       //  (replace comma decimal)
   zdialog_fetch(zd,"longitude",temp,12);
   repl_1str(temp,longitude,",",".");
   coord[0] = latitude;
   coord[1] = longitude;

   if (zstat == 1)                                                         //  [find]
   {
      Nmatch = get_geolocs(location,coord,matches);                        //  find in city-geotags file
      if (Nmatch == 0)                                                     //  no matches
         zmessageACK(Mwin,0,ZTX("city not found"));

      else if (Nmatch == 1) {                                              //  one match
         zdialog_stuff(zd,"city",matches[0][0]);                           //  stuff matching city data into dialog
         zdialog_stuff(zd,"country",matches[0][1]);
         zdialog_stuff(zd,"latitude",coord[0]);
         zdialog_stuff(zd,"longitude",coord[1]);
      }

      else {                                                               //  multiple matching cities
         zstat = geotags_choosecity(location,coord);                       //  ask user to choose one
         if (zstat == 1) {                                                 //  response is available
            zdialog_stuff(zd,"city",location[0]);                          //  stuff matching city data into dialog
            zdialog_stuff(zd,"country",location[1]);
            zdialog_stuff(zd,"latitude",coord[0]);
            zdialog_stuff(zd,"longitude",coord[1]);
         }
      }
   }

   else if (zstat == 2)                                                    //  [web]
   {
      errmess = web_geocode(location,coord);                               //  look-up in web service
      if (errmess) 
         zmessageACK(Mwin,0,errmess);                                      //  fail
      else {
         zdialog_stuff(zd,"city",location[0]);                             //  success, return all data
         zdialog_stuff(zd,"country",location[1]);                          //  (location may have been completed)
         zdialog_stuff(zd,"latitude",coord[0]);
         zdialog_stuff(zd,"longitude",coord[1]);
      }
   }

   else if (zstat == 3)                                                    //  [prev] 
   {
      zdialog_stuff(zd,"city",prevcity);                                   //  get last-used geotags
      zdialog_stuff(zd,"country",prevcountry);
      zdialog_stuff(zd,"latitude",prevlatitude);
      zdialog_stuff(zd,"longitude",prevlongitude);
   }

   else if (zstat == 4)                                                    //  [map]
   {
      zd_geotags = zd;                                                     //  open world map and get geotags
      click_worldmap();                                                    //    via mouse click
   }
   
   else if (zstat == 5)                                                    //  [apply]
   {
      gtk_window_present(MWIN);                                            //  keep focus on main window          v.14.09

      if (strEqu(coord[0],"null")) *coord[0] = 0;                          //  replace "null" with ""
      if (strEqu(coord[1],"null")) *coord[1] = 0;

      if (*coord[0] || *coord[1]) {                                        //  if coordinates present, validate
         if (! *coord[0] || ! *coord[1]) goto badcoord;
         fcoord[0] = atof(coord[0]);
         if (fcoord[0] < -90 || fcoord[0] > 90) goto badcoord;
         fcoord[1] = atof(coord[1]);
         if (fcoord[1] < -180 || fcoord[1] > 180) goto badcoord;
         if (fcoord[0] == 0 && fcoord[1] == 0) goto badcoord;
      }

      if (checkpend("lock")) return 1;                                     //  check nothing pending

      put_geolocs(location,coord);                                         //  update EXIF and geotags city data

      strcpy(prevcity,city);                                               //  save data for later use as [prev]
      strcpy(prevcountry,country);
      strcpy(prevlatitude,latitude);
      strcpy(prevlongitude,longitude);
   }

   else if (zstat == 6)                                                    //  [clear]
   {
      zdialog_stuff(zd,"city","");                                         //  erase dialog fields
      zdialog_stuff(zd,"country","");
      zdialog_stuff(zd,"latitude","");
      zdialog_stuff(zd,"longitude","");
   }

   else {                                                                  //  cancel
      zdialog_free(zd);
      zdeditgeotags = 0;
      zd_geotags = 0;                                                      //  deactivate world map clicks
      m_viewmode(0,"F"); 
   }

   return 1;

badcoord:
   zmessageACK(Mwin,0,ZTX("bad latitude/longitude: %s %s"),coord[0],coord[1]);
   return 1;
}


/**************************************************************************/

//  batch add geotags - set geotags for multiple image files

char     **batch_add_geotags_filelist = 0;
int      batch_add_geotags_filecount = 0;

void m_batch_add_geotags(GtkWidget *, cchar *menu)
{
   int   batch_add_geotags_dialog_event(zdialog *zd, cchar *event);

   cchar       *title = ZTX("Batch Add Geotags");
   char        *file, **flist;
   zdialog     *zd;
   int         ii, err;
   char        *location[2], *coord[2];
   char        city[100], country[100];
   char        latitude[20], longitude[20];
   cchar       *mapquest1 = "Geocoding web service courtesy of";
   cchar       *mapquest2 = "http://www.mapquest.com";

   F1_help_topic = "batch_add_geotags";

   if (! init_geolocs()) return;                                           //  initialize geotags
   
   if (checkpend("all")) return;                                           //  check nothing pending
   Fmenulock = 1;
   
/**
                Batch Add Geotags

      [select files]  NN files selected
      city [______________]  country [______________]
      latitude [_______] longitude [_______]
      
      [find] [web] [prev] [map] [proceed] [cancel]

**/

   zd = zdialog_new(title,Mwin,Bfind,Bweb,Bprev,Bmap,Bproceed,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","files","hb1",Bselectfiles,"space=10");
   zdialog_add_widget(zd,"label","labcount","hb1",Bnofileselected,"space=10");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labcity","hb2",ZTX("city"),"space=5");
   zdialog_add_widget(zd,"entry","city","hb2",0,"expand");
   zdialog_add_widget(zd,"label","space","hb2",0,"space=5");
   zdialog_add_widget(zd,"label","labcountry","hb2",ZTX("country"),"space=5");
   zdialog_add_widget(zd,"entry","country","hb2",0,"expand");
   zdialog_add_widget(zd,"hbox","hb3","dialog");
   zdialog_add_widget(zd,"label","lablat","hb3","Latitude","space=3");
   zdialog_add_widget(zd,"entry","latitude","hb3",0,"scc=10");
   zdialog_add_widget(zd,"label","space","hb3",0,"space=5");
   zdialog_add_widget(zd,"label","lablong","hb3","Longitude","space=3");
   zdialog_add_widget(zd,"entry","longitude","hb3",0,"scc=10");
   zdialog_add_widget(zd,"hbox","hbmq","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmq","hbmq",mapquest1,"space=3");
   zdialog_add_widget(zd,"link","MapQuest","hbmq",mapquest2);

   batch_add_geotags_filelist = 0;
   batch_add_geotags_filecount = 0;
   flist = 0;

   zdialog_run(zd,batch_add_geotags_dialog_event);                         //  run dialog
   zdialog_wait(zd);                                                       //  wait for dialog completion

   if (zd->zstat != 5) goto cleanup;                                       //  status not [proceed]
   if (! batch_add_geotags_filecount) goto cleanup;                        //  no files selected

   zdialog_fetch(zd,"city",city,99);                                       //  get city [country] from dialog
   zdialog_fetch(zd,"country",country,99);
   location[0] = city;
   location[1] = country;
   zdialog_fetch(zd,"latitude",latitude,12);                               //  and latitude, longitude
   zdialog_fetch(zd,"longitude",longitude,12);
   coord[0] = latitude;
   coord[1] = longitude;
   
   zdialog_free(zd);                                                       //  kill dialog
   zd_geotags = 0;                                                         //  deactivate world map clicks

   flist = batch_add_geotags_filelist;                                     //  selected files
   if (! flist) goto cleanup;

   write_popup_text("open","Adding Geotags",500,200,Mwin);                 //  status monitor popup window

   for (ii = 0; flist[ii]; ii++)                                           //  loop all selected files
   {
      file = flist[ii];                                                    //  display image
      err = f_open(file,0,0,0);
      if (err) continue;

      write_popup_text("write",file);                                      //  report progress
      zmainloop();
      
      load_filemeta(file);                                                 //  load current file tags
      put_geolocs(location,coord);                                         //  update geotags part
   }

   write_popup_text("write","COMPLETED");
   
cleanup:

   if (zd) zdialog_free(zd);
   zd_geotags = 0;                                                         //  deactivate world map clicks

   if (flist) {
      for (ii = 0; flist[ii]; ii++) 
         zfree(flist[ii]);
      zfree(flist);
   }

   Fmenulock = 0;
   return;
}


//  batch_add_geotags dialog event function

int batch_add_geotags_dialog_event(zdialog *zd, cchar *event)
{
   int      ii, yn, zstat, Nmatch;
   char     **flist = batch_add_geotags_filelist;
   char     countmess[50];
   char     *location[2], *coord[2], *matches[20][2];
   char     city[100], country[100];
   char     latitude[20], longitude[20];
   cchar    *errmess;
   float    fcoord[2];

   if (strEqu(event,"files"))                                              //  select images to add tags
   {
      if (flist) {                                                         //  free prior list
         for (ii = 0; flist[ii]; ii++) 
            zfree(flist[ii]);
         zfree(flist);
      }

      zdialog_show(zd,0);                                                  //  hide parent dialog
      flist = gallery_getfiles();                                          //  get file list from user
      zdialog_show(zd,1);

      batch_add_geotags_filelist = flist;

      if (flist)                                                           //  count files in list
         for (ii = 0; flist[ii]; ii++);
      else ii = 0;
      batch_add_geotags_filecount = ii;
      
      snprintf(countmess,50,Bfileselected,batch_add_geotags_filecount);
      zdialog_stuff(zd,"labcount",countmess);
   }

   if (! zd->zstat) return 1;                                              //  wait for action button
   zstat = zd->zstat;
   zd->zstat = 0;                                                          //  keep dialog active

   zdialog_fetch(zd,"city",city,99);                                       //  get city [country] from dialog
   zdialog_fetch(zd,"country",country,99);
   location[0] = city;
   location[1] = country;
   zdialog_fetch(zd,"latitude",latitude,12);                               //  and latitude, longitude
   zdialog_fetch(zd,"longitude",longitude,12);
   coord[0] = latitude;
   coord[1] = longitude;

   if (zstat == 1)                                                         //  [find]
   {
      Nmatch = get_geolocs(location,coord,matches);                        //  find in city-geotags file
      if (Nmatch == 0)                                                     //  no matches
         zmessageACK(Mwin,0,ZTX("city not found"));

      else if (Nmatch == 1) {                                              //  one match
         zdialog_stuff(zd,"city",matches[0][0]);                           //  stuff matching city data into dialog
         zdialog_stuff(zd,"country",matches[0][1]);
         zdialog_stuff(zd,"latitude",coord[0]);
         zdialog_stuff(zd,"longitude",coord[1]);
      }

      else {                                                               //  multiple matching cities
         zstat = geotags_choosecity(location,coord);                       //  ask user to choose one
         if (zstat == 1) {                                                 //  response is available
            zdialog_stuff(zd,"city",location[0]);                          //  stuff matching city data into dialog
            zdialog_stuff(zd,"country",location[1]);
            zdialog_stuff(zd,"latitude",coord[0]);
            zdialog_stuff(zd,"longitude",coord[1]);
         }
      }
   }

   else if (zstat == 2)                                                    //  [web]
   {
      errmess = web_geocode(location,coord);                               //  look-up in web service
      if (errmess) 
         zmessageACK(Mwin,0,errmess);                                      //  fail
      else {
         zdialog_stuff(zd,"city",location[0]);                             //  success, return all data
         zdialog_stuff(zd,"country",location[1]);                          //  (location may have been completed)
         zdialog_stuff(zd,"latitude",coord[0]);
         zdialog_stuff(zd,"longitude",coord[1]);
      }
   }

   else if (zstat == 3)                                                    //  [prev] 
   {
      zdialog_stuff(zd,"city",prevcity);                                   //  get last-used geotags
      zdialog_stuff(zd,"country",prevcountry);
      zdialog_stuff(zd,"latitude",prevlatitude);
      zdialog_stuff(zd,"longitude",prevlongitude);
   }

   else if (zstat == 4)                                                    //  [map]
   {
      zd_geotags = zd;                                                     //  open world map and get geotags
      click_worldmap();                                                    //    via mouse click
   }
   
   else if (zstat == 5)                                                    //  [proceed]
   {
      if (strEqu(coord[0],"null")) *coord[0] = 0;                          //  replace "null" with ""
      if (strEqu(coord[1],"null")) *coord[1] = 0;

      if (*coord[0] || *coord[1])                                          //  if coordinates present, validate
      {
         if (! *coord[0] || ! *coord[1]) goto badcoord;
         fcoord[0] = atof(coord[0]);
         if (fcoord[0] < -90 || fcoord[0] > 90) goto badcoord;
         fcoord[1] = atof(coord[1]);
         if (fcoord[1] < -180 || fcoord[1] > 180) goto badcoord;
         if (fcoord[0] == 0 && fcoord[1] == 0) goto badcoord;
      }

      if (! batch_add_geotags_filecount) goto nofiles;

      if (! *city || ! *country || ! *latitude || ! *longitude) {          //  check data is complete
         yn = zmessageYN(Mwin,ZTX("data is incomplete \n proceed?"));
         if (! yn) return 1;
      }

      strcpy(prevcity,city);                                               //  save data for later use as [prev]
      strcpy(prevcountry,country);
      strcpy(prevlatitude,latitude);
      strcpy(prevlongitude,longitude);

      zd->zstat = 5;                                                       //  OK to proceed
      zdialog_destroy(zd);
   }

   else {                                                                  //  cancel
      zdialog_destroy(zd);
      zd_geotags = 0;                                                      //  deactivate world map clicks
      m_viewmode(0,"F");
   }

   return 1;

badcoord:
   zmessageACK(Mwin,0,ZTX("bad latitude/longitude: %s %s"),coord[0],coord[1]);
   return 1;

nofiles:
   zmessageACK(Mwin,0,Bnofileselected);
   return 1;
}


//  dialog to choose one city from multiple options
//  location[2] is input city and optional country
//  (may be substrings, may have multiple matches in city geotags data)
//  location[2] is output unique city and country after user choice
//  coord[2] is output latitude, longitude

char     *geotags_chosenlocation[2];
char     *geotags_chosencoord[2];

int geotags_choosecity(char *location[2], char *coord[2])
{
   int  geotags_choosecity_event(zdialog *zd, cchar *event);

   char     *matches[20][2], text[200];
   int      Nmatch, ii, zstat;
   zdialog  *zd;

   Nmatch = get_geolocs(location,coord,matches);                           //  get matching city geotags data

   if (Nmatch == 0) return 0;                                              //  no match

   if (Nmatch == 1) {                                                      //  one match, done
      location[0] = matches[0][0];
      location[1] = matches[0][1];
      return 1;
   }

   zd = zdialog_new(ZTX("choose city"),Mwin,BOK,Bcancel,null);             //  multiple matches, start dialog
   zdialog_add_widget(zd,"comboE","cities","dialog",0,"space=5");
   for (ii = 0; ii < Nmatch; ii++) {                                       //  list matching cities to choose from
      snprintf(text,200,"%s | %s",matches[ii][0],matches[ii][1]);
      zdialog_cb_app(zd,"cities",text);                                    //  duplicates are removed
   }

   zdialog_resize(zd,300,100);
   zdialog_run(zd,geotags_choosecity_event);                               //  run dialog, wait for completion
   zstat = zdialog_wait(zd);
   zdialog_free(zd);

   if (zstat == 1) {                                                       //  valid response available
      location[0] = geotags_chosenlocation[0];
      location[1] = geotags_chosenlocation[1];
      coord[0] = geotags_chosencoord[0];
      coord[1] = geotags_chosencoord[1];
      return 1;
   }
   
   return 0;
}


//  dialog event function - get chosen city/country from multiple choices

int geotags_choosecity_event(zdialog *zd, cchar *event)
{
   char           text[200];
   static char    city[100], country[100];
   char           *location[2], *coord[2];
   char           *matches[20][2];
   cchar          *pp;
   int            nn;
   static int     ftf = 1;
   
   if (ftf) {
      zdialog_cb_popup(zd,"cities");                                       //  first time, open combo box list
      ftf = 0;
   }

   if (strEqu(event,"cities")) {                                           //  OK
      zdialog_fetch(zd,"cities",text,200);

      pp = strField(text,'|',1);
      if (pp) strncpy0(city,pp,99);
      pp = strField(text,'|',2);
      if (pp) strncpy0(country,pp,99);
      strTrim2(city);
      strTrim2(country);
      location[0] = city;
      location[1] = country;

      nn = get_geolocs(location,coord,matches);                            //  find in city geotags data
      if (nn) {
         geotags_chosenlocation[0] = location[0];                          //  use 1st match if > 1
         geotags_chosenlocation[1] = location[1];
         geotags_chosencoord[0] = coord[0];
         geotags_chosencoord[1] = coord[1];
         zd->zstat = 1;
      }
      else zd->zstat = 2;                                                  //  bad status
   }

   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [OK]  v.14.03

   if (zd->zstat) ftf = 1;

   return 1;
}


/**************************************************************************/

//  Convert a city [country] to latitude/longitude using the
//    MapQuest geocoding service.
//  (incomplete names may be completed with a bad guess) 

cchar * web_geocode(char *location[2], char *coord[2])                     //  overhaul 13.05, 13.05.1
{
   int         err;
   static char latitude[20], longitude[20];
   char        outfile[200], URI[300];
   char        *pp1, *pp2, buffer[200];
   float       flati, flongi;
   FILE        *fid;
   cchar       *notfound = ZTX("not found");
   cchar       *badinputs = ZTX("city and country required");
   cchar       *query = "http://open.mapquestapi.com/geocoding/v1/address?"
                        "&key=Fmjtd%7Cluub2qa72d%2C20%3Do5-9u700a"
                        "&maxResults=1"
                        "&outFormat=csv";
   
   *coord[0] = *coord[1] = 0;                                              //  null outputs
   *latitude = *longitude = 0;
   
   if (*location[0] < ' ' || *location[1] < ' ')
      return badinputs;
   
   snprintf(outfile,199,"%s/web-data",geotags_dirk);
   snprintf(URI,299,"\"%s&location=%s,%s\"",query,location[0],location[1]);

   err = shell_ack("wget -T 10 -o /dev/null -O %s %s",outfile,URI);
   if (err) return wstrerror(err);
   
   fid = fopen(outfile,"r");                                               //  get response
   if (! fid) return notfound;
   pp1 = fgets(buffer,200,fid);
   pp1 = fgets(buffer,200,fid);
   fclose(fid);
   if (! pp1) return notfound;
   
   pp2 = (char *) strField(pp1,",",7);
   if (! pp2) return notfound;
   strncpy0(latitude,pp2,20);
   
   pp2 = (char *) strField(pp1,",",8);
   if (! pp2) return notfound;
   strncpy0(longitude,pp2,20);
   
   err = validate_latilongi(latitude,longitude,flati,flongi);
   if (err) return notfound;

   pp1 = strchr(latitude,'.');                                             //  keep max. 4 decimal digits
   if (pp1) *(pp1+5) = 0;
   pp1 = strchr(longitude,'.');
   if (pp1) *(pp1+5) = 0;
   
   coord[0] = latitude;
   coord[1] = longitude;
   return 0;
}


/**************************************************************************/

//  Group images by location and date, with a count of images in each group. 
//  Click on a group to get a thumbnail gallery of all images in the group.

int   ggroups_comp(cchar *rec1, cchar *rec2);
void  ggroups_click(GtkWidget *widget, int line, int pos); 
int   ggroups_getdays(cchar *date);

struct grec_t  {                                                           //  image geotags data
   char        *city, *country;                                            //  group location
   char        pdate[12];                                                  //  nominal group date, yyyymmdd
   int         lodate, hidate;                                             //  range, days since 0 CE
   int         count;                                                      //  images in group
};

grec_t   *grec = 0;
int      Ngrec = 0;
int      ggroups_groupby, ggroups_daterange;


void m_geotag_groups(GtkWidget *, cchar *)
{
   zdialog        *zd;
   int            zstat, ftf, err, cc, cc1, cc2;
   int            iix, iig, newgroup;
   char           country[100], city[100], buff[300], pdate[12];
   cchar          *pp;
   sxrec_t        sxrec;
   GtkWidget      *textwin;

   F1_help_topic = "geotag_groups";
   if (checkpend("all")) return;                                           //  check nothing pending

/***
            Report Geotag Groups
         
         (o) Group by country
         (o) Group by country/city
         (o) Group by country/city/date
             Combine within [ xx |-|+] days
          
                        [proceed]  [cancel]
***/

   zd = zdialog_new(ZTX("Report Geotag Groups"),Mwin,Bproceed,Bcancel,null);
   zdialog_add_widget(zd,"radio","country","dialog",ZTX("Group by country"));
   zdialog_add_widget(zd,"radio","city","dialog",ZTX("Group by country/city"));
   zdialog_add_widget(zd,"radio","date","dialog",ZTX("Group by country/city/date"));
   zdialog_add_widget(zd,"hbox","hbr","dialog");
   zdialog_add_widget(zd,"label","space","hbr",0,"space=10");
   zdialog_add_widget(zd,"label","labr1","hbr",ZTX("Combine within"),"space=10");
   zdialog_add_widget(zd,"spin","range","hbr","0|999|1|1");
   zdialog_add_widget(zd,"label","labr2","hbr",ZTX("days"),"space=10");
   
   zdialog_stuff(zd,"country",0);
   zdialog_stuff(zd,"city",1);
   zdialog_stuff(zd,"date",0);
   
   zdialog_resize(zd,300,0);
   zdialog_run(zd);
   zstat = zdialog_wait(zd);
   if (zstat != 1) {
      zdialog_free(zd);
      return;
   }
   
   zdialog_fetch(zd,"country",iix);
   if (iix) ggroups_groupby = 1;                                           //  group by country
   zdialog_fetch(zd,"city",iix);
   if (iix) ggroups_groupby = 2;                                           //  group by country/city
   zdialog_fetch(zd,"date",iix);
   if (iix) ggroups_groupby = 3;                                           //  group by country/city/date (range)
   zdialog_fetch(zd,"range",ggroups_daterange);

   zdialog_free(zd);
   
   if (Ngrec) {                                                            //  free prior memory
      for (iix = 0; iix < Ngrec; iix++) {
         if (grec[iix].city) zfree(grec[iix].city);
         if (grec[iix].country) zfree(grec[iix].country);
      }
      zfree(grec);
   }

   cc = maximages * sizeof(grec_t);                                        //  allocate memory
   grec = (grec_t *) zmalloc(cc);
   memset(grec,0,cc);

   Ngrec = 0;
   ftf = 1;

   while (true)
   {
      err = read_sxrec_seq(sxrec,ftf);                                     //  read image index recs.
      if (err) break;
      
      iix = Ngrec;

      pp = strField(sxrec.gtags,'^',1);                                    //  get city
      if (pp) grec[iix].city = zstrdup(pp);
      else grec[iix].city = zstrdup("null");

      pp = strField(sxrec.gtags,'^',2);                                    //  country
      if (pp) grec[iix].country = zstrdup(pp);
      else grec[iix].country = zstrdup("null");

      strncpy0(grec[iix].pdate,sxrec.pdate,9);                             //  photo date, truncate to yyyymmdd
      grec[iix].lodate = ggroups_getdays(sxrec.pdate);                     //  days since 0 CE
      grec[iix].hidate = grec[iix].lodate;

      if (++Ngrec == maximages) {
         zmessageACK(Mwin,0,"too many image files");
         return;
      }

      zfree(sxrec.file);
      zfree(sxrec.tags);
      zfree(sxrec.capt);
      zfree(sxrec.comms);
      zfree(sxrec.gtags);
   }

   if (! Ngrec) {
      zmessageACK(Mwin,0,"no geotags data found");
      return;
   }

   if (Ngrec > 1)                                                          //  sort index by country/city/date
      HeapSort((char *) grec, sizeof(grec_t), Ngrec, ggroups_comp);

   iig = 0;                                                                //  1st group from grec[0]
   grec[iig].count = 1;                                                    //  group count = 1

   for (iix = 1; iix < Ngrec; iix++)                                       //  scan following grecs
   {
      newgroup = 0;

      if (strNeq(grec[iix].country,grec[iig].country)) 
         newgroup = 1;                                                     //  new country >> new group
      
      if (ggroups_groupby >= 2) 
         if (strNeq(grec[iix].city,grec[iig].city)) newgroup = 1;          //  new city >> new group if group by city

      if (ggroups_groupby >= 3) 
         if (grec[iix].lodate - grec[iig].hidate > ggroups_daterange)      //  new date >> new group if group by date
            newgroup = 1;                                                  //    and date out of range

      if (newgroup)
      {
         iig++;                                                            //  new group
         if (iix > iig) {
            grec[iig] = grec[iix];                                         //  copy and pack down
            grec[iix].city = grec[iix].country = 0;                        //  no zfree()
         }
         grec[iig].count = 1;                                              //  group count = 1
      }
      else
      {
         zfree(grec[iix].city);                                            //  same group
         zfree(grec[iix].country);                                         //  free memory
         grec[iix].city = grec[iix].country = 0;
         grec[iig].hidate = grec[iix].lodate;                              //  expand group date range
         grec[iig].count++;                                                //  increment group count
      }
   }

   Ngrec = iig + 1;                                                        //  unique groups count

   textwin = write_popup_text("open",ZTX("geotag groups"),620,400,Mwin);   //  write groups to popup window
   
   if (ggroups_groupby == 1)                                               //  group by country
   {
      snprintf(buff,300,"%-30s  %5s ","Country","Count");
      write_popup_text("write",buff);

      for (iig = 0; iig < Ngrec; iig++)
      {
         utf8substring(country,grec[iig].country,0,30);
         cc1 = 30 + strlen(country) - utf8len(country);
         snprintf(buff,300,"%-*s  %5d ",cc1,country,grec[iig].count);
         write_popup_text("write",buff);
      }
   }
   
   if (ggroups_groupby == 2)                                               //  group by country/city
   {
      snprintf(buff,300,"%-30s  %-30s  %5s ","Country","City","Count");
      write_popup_text("write",buff);

      for (iig = 0; iig < Ngrec; iig++)
      {
         utf8substring(country,grec[iig].country,0,30);
         cc1 = 30 + strlen(country) - utf8len(country);
         utf8substring(city,grec[iig].city,0,30);
         cc2 = 30 + strlen(city) - utf8len(city);
         snprintf(buff,300,"%-*s  %-*s  %5d ",
                  cc1,country,cc2,city,grec[iig].count);
         write_popup_text("write",buff);
      }
   }
   
   if (ggroups_groupby == 3)                                               //  group by country/city/date (range)
   {
      snprintf(buff,300,"%-30s  %-30s  %-10s  %5s ","Country","City","Date","Count");
      write_popup_text("write",buff);

      for (iig = 0; iig < Ngrec; iig++)
      {
         utf8substring(country,grec[iig].country,0,30);                    //  get graphic cc for UTF-8 names
         cc1 = 30 + strlen(country) - utf8len(country);
         utf8substring(city,grec[iig].city,0,30);
         cc2 = 30 + strlen(city) - utf8len(city);
         
         strncpy(pdate,grec[iig].pdate,8);                                 //  date, yyyymmdd
         if (strNeq(pdate,"null")) {
            memcpy(pdate+8,pdate+6,2);                                     //  convert to yyyy-mm-dd
            memcpy(pdate+5,pdate+4,2);
            pdate[4] = pdate[7] = '-';
            pdate[10] = 0;
         }
         
         snprintf(buff,300,"%-*s  %-*s  %-10s  %5d ",
                  cc1,country,cc2,city,pdate,grec[iig].count);
         write_popup_text("write",buff);
      }
   }

   write_popup_text("top");
   textwidget_set_clickfunc(textwin,ggroups_click);                        //  response function for mouse click
   return;
}


//  Compare 2 grec records by geotags and date, 
//  return < 0  = 0  > 0   for   rec1  <  =  >  rec2.

int ggroups_comp(cchar *rec1, cchar *rec2)
{
   int      ii;

   char * country1 = ((grec_t *) rec1)->country;                           //  compare countries
   char * country2 = ((grec_t *) rec2)->country;
   ii = strcmp(country1,country2);
   if (ii) return ii;

   char * city1 = ((grec_t *) rec1)->city;                                 //  compare cities
   char * city2 = ((grec_t *) rec2)->city;
   ii = strcmp(city1,city2);
   if (ii) return ii;

   int date1 = ((grec_t *) rec1)->lodate;                                  //  compare dates
   int date2 = ((grec_t *) rec2)->lodate;
   ii = date1 - date2;
   return ii;
}


//  convert yyyymmdd date into days from 0 C.E.
//  "null" date returns 999999 (year 2737)

int ggroups_getdays(cchar *date)
{
   int   CEdays(int year, int mon, int day);

   int      year, month, day;
   char     temp[8];

   year = month = day = 0;
   
   strncpy0(temp,date,5);
   year = atoi(temp);
   
   strncpy0(temp,date+4,3);
   month = atoi(temp);
   
   strncpy0(temp,date+6,3);
   day = atoi(temp);

   return CEdays(year,month,day);
}


//   convert year/month/day into days since Jan 1, 0001 (day 0 CE)
//   year is 0001 to 9999, month is 1-12, day is 1-31
     
int CEdays(int year, int month, int day)
{
   int    montab[12] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };
   int    elaps;
   
   elaps = 365 * (year-1) + (year-1) / 4;                                  //  elapsed days in prior years
   elaps += montab[month-1];                                               //  + elapsed days in prior months
   if (year % 4 == 0 && month > 2) elaps += 1;                             //  + 1 for Feb. 29 
   elaps += day-1;                                                         //  + elapsed days in month
   return elaps;
}


//  Receive clicks on report window and generate gallery of images
//  matching the selected country/city/date

void ggroups_click(GtkWidget *widget, int line, int pos)
{
   int      iix, ftf, err, lodate, hidate, datex;
   cchar    *pp;
   char     city[100], country[100];
   char     resultsfile[100];
   FILE     *fid;
   sxrec_t  sxrec;
   
   if (checkpend("all")) return;                                           //  check nothing pending

   textwidget_get_line(widget,line,1);                                     //  hilite clicked line

   iix = line - 1;                                                         //  clicked grec[iix]
   if (iix < 0 || iix > Ngrec-1) return;

   strncpy0(country,grec[iix].country,99);                                 //  selected country/city/date range
   strncpy0(city,grec[iix].city,99);
   lodate = grec[iix].lodate;
   hidate = grec[iix].hidate;
   
   snprintf(resultsfile,100,"%s/search_results",tempdir);                  //  output scratch file
   fid = fopen(resultsfile,"w");
   if (! fid) goto filerror;
   
   ftf = 1;

   while (true)                                                            //  read image index recs.
   {
      err = read_sxrec_seq(sxrec,ftf);
      if (err) break;
      
      pp = strField(sxrec.gtags,'^',2);
      if (! pp) pp = "null";                                               //  enable search for "null"

      if (strNeq(pp,country)) goto freemem;                                //  no country match
      
      if (ggroups_groupby >= 2) {
         pp = strField(sxrec.gtags,'^',1);
         if (! pp) pp = "null";
         if (strNeq(pp,city)) goto freemem;                                //  no city match
      }

      if (ggroups_groupby == 3) {
         datex = ggroups_getdays(sxrec.pdate);
         if (datex < lodate || datex > hidate) goto freemem;               //  no date match
      }

      fprintf(fid,"%s\n",sxrec.file);                                      //  output matching file
   
   freemem:
      zfree(sxrec.file);
      zfree(sxrec.tags);
      zfree(sxrec.capt);
      zfree(sxrec.comms);
      zfree(sxrec.gtags);
   }

   fclose(fid);

   free_resources();
   navi::gallerytype = 2;                                                  //  search results
   gallery(resultsfile,"initF");                                           //  generate gallery of matching files
   gallery(0,"paint",0);
   m_viewmode(0,"G");
   return;
   
filerror:
   zmessLogACK(Mwin,"file error: %s",strerror(errno));
   return;
}


/**************************************************************************/

//  Show locations of geotagged images on a world map.
//  Click on world map to get a gallery of images at/near the location.

void m_geotag_worldmap(GtkWidget *, cchar *menu)
{
   void geotag_worldmap(float flati, float flongi);
   
   char     trange[12], *pp;
   int      irange = 0;

   F1_help_topic = "geotag_worldmap";

   worldmap_callback = geotag_worldmap;                                    //  set worldmap click callback function
   click_worldmap();                                                       //  wait for worldmap click

   if (menu) {                                                             //  if user menu (not view W selection)
      while (true) {                                                       //  get a new search range if wanted
         snprintf(trange,12,"%d",worldmap_range);
         pp = zdialog_text(Mwin,ZTX("search range (km)"),trange);
         if (pp) irange = atoi(pp);
         if (pp) zfree(pp);
         if (irange > 0 && irange < 1000) break;
      }
      worldmap_range = irange;
   }

   return;
}


//  click_worldmap callback function - receive clicked geotag location
//  and generate gallery of images geotagged with this location

void geotag_worldmap(float flati, float flongi)
{
   int            ftf, err, nn = 0;
   char           imagefile[maxfcc], resultsfile[100];
   float          glati, glongi, grange;
   cchar          *pp;
   FILE           *fid;
   sxrec_t        sxrec;

   snprintf(resultsfile,100,"%s/search_results",tempdir);                  //  output scratch file 

   fid = fopen(resultsfile,"w");
   if (! fid) {
      zmessLogACK(Mwin,"output file error: %s",strerror(errno));
      return;
   }
   
   if (checkpend("all")) return;                                           //  check nothing pending 

   ftf = 1;

   while (true)                                                            //  read image index recs.
   {
      err = read_sxrec_seq(sxrec,ftf);
      if (err) break;

      strncpy0(imagefile,sxrec.file,maxfcc);                               //  save filespec
      
      pp = strField(sxrec.gtags,'^',3);                                    //  latitude
      if (! pp) goto freemem;
      if (strEqu(pp,"null")) goto freemem;
      glati = atof(pp);

      pp = strField(sxrec.gtags,'^',4);                                    //  longitude
      if (! pp) goto freemem;
      if (strEqu(pp,"null")) goto freemem;
      glongi = atof(pp);
      
      grange = (flati - glati) * (flati - glati);
      grange += (flongi - glongi) * (flongi - glongi);
      grange = 111.0 * sqrt(grange);

      if (grange <= worldmap_range) {                                      //  within distance limit, select
         fprintf(fid,"%s\n",imagefile);                                    //  output matching file
         nn++;
      }

   freemem:
      zfree(sxrec.file);
      zfree(sxrec.tags);
      zfree(sxrec.capt);
      zfree(sxrec.comms);
      zfree(sxrec.gtags);
   }

   fclose(fid);

   if (! nn) {
      zmessageACK(Mwin,0,ZTX("No matching images found"));
      click_worldmap();                                                    //  try again
      return;
   }
   
   free_resources();
   navi::gallerytype = 2;                                                  //  search results
   gallery(resultsfile,"initF");                                           //  generate gallery of matching files
   m_viewmode(0,"G");

   return;
}


/**************************************************************************/

//  Search image tags, geotags, dates, stars, comments, captions 
//  to find matching images. This is fast using the image index. 
//  Search also any other metadata, but relatively slow.

namespace search_images
{
   zdialog  *zdsearchimages = 0;                                           //  search images dialog

   char     searchDateFrom[16] = "";                                       //  search images
   char     searchDateTo[16] = "";
   char     searchStarsFrom[4] = "";
   char     searchStarsTo[4] = "";

   char     searchCity[100] = "";                                          //  search geotags
   char     searchCountry[100] = "";
   char     searchLatitude[20] = "";
   char     searchLongitude[20] = "";
   float    searchRange = 100;
   float    flatitude = 0, flongitude = 0;
   
   int      Fscanall, Fscancurr, Fnewset, Faddset, Fremset;
   int      Fdates, Ftext, Ffiles, Ftags, Fstars, Fgtags;
   int      Flastver, Fsearchmeta;
   int      Falltags, Falltext, Fallfiles;
   int      Frepgallery, Frepmeta;
   int      Nsearchkeys = 0;
   char     *searchkeys[5];                                                //  search metadata keys and match data
   char     *searchkeydata[5];
   char     searchkeyx[8], searchkeydatax[8];
}

using namespace search_images;


void m_search_images(GtkWidget *, cchar *)                                 //  overhauled
{
   void  search_searchtags_clickfunc(GtkWidget *widget, int line, int pos);
   void  search_deftags_clickfunc(GtkWidget *widget, int line, int pos);
   int   searchimages_dialog_event(zdialog*, cchar *event);

   zdialog     *zd;
   GtkWidget   *widget;

   F1_help_topic = "search_images";

   if (checkpend("all")) return;                                           //  check nothing pending 

/***
                  Search Image Metadata

         images to search: (o) all  (o) current set only
         matching images: (o) new set  (o) add to set  (o) remove 
         report type: (o) gallery  (o) metadata
         date range   [___________] [___________] (yyyymmdd)
         stars range  [__] [__]   (o) last version         all/any
         search tags  [__________________________________] (o) (o)
         search text  [__________________________________] (o) (o)
         search files [__________________________________] (o) (o)
         other criteria: [geotags] (*)  [other] (*)

         defined tags
          --------------------------------------------------------
         |                                                        |
         |                                                        |
         |                                                        |
         |                                                        |
         |                                                        |
          --------------------------------------------------------
                                               [proceed] [cancel]
***/

   zd = zdialog_new(ZTX("Search Image Metadata"),Mwin,Bproceed,Bcancel,null);
   zdsearchimages = zd;
   
   zdialog_add_widget(zd,"hbox","hbs1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labs1","hbs1",ZTX("images to search:"),"space=5");
   zdialog_add_widget(zd,"radio","allimages","hbs1",ZTX("all"),"space=3");
   zdialog_add_widget(zd,"radio","currset","hbs1",ZTX("current set only"),"space=5");

   zdialog_add_widget(zd,"hbox","hbm1","dialog");
   zdialog_add_widget(zd,"label","labs1","hbm1",ZTX("matching images:"),"space=5");
   zdialog_add_widget(zd,"radio","newset","hbm1",ZTX("new set"),"space=5");
   zdialog_add_widget(zd,"radio","addset","hbm1",ZTX("add to set"),"space=5");
   zdialog_add_widget(zd,"radio","remset","hbm1",ZTX("remove"),"space=5");
   
   zdialog_add_widget(zd,"hbox","hbrt","dialog");
   zdialog_add_widget(zd,"label","labrt","hbrt",ZTX("report type:"),"space=5");
   zdialog_add_widget(zd,"radio","repgallery","hbrt",ZTX("gallery"),"space=5");
   zdialog_add_widget(zd,"radio","repmeta","hbrt",ZTX("metadata"),"space=5");

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|expand");

   zdialog_add_widget(zd,"label","labD","vb1",ZTX("date range"));
   zdialog_add_widget(zd,"label","labS","vb1",ZTX("stars range"));
   zdialog_add_widget(zd,"label","labT","vb1",ZTX("search tags"));
   zdialog_add_widget(zd,"label","labT","vb1",ZTX("search text"));
   zdialog_add_widget(zd,"label","labF","vb1",ZTX("search files"));

   zdialog_add_widget(zd,"hbox","hbD","vb2",0,"space=1");
   zdialog_add_widget(zd,"entry","datefrom","hbD",0,"scc=12");
   zdialog_add_widget(zd,"entry","dateto","hbD",0,"scc=12");
   zdialog_add_widget(zd,"label","labD","hbD",ZTX("(yyyymmdd)"),"space=5");

   zdialog_add_widget(zd,"hbox","hbS","vb2",0,"space=1");
   zdialog_add_widget(zd,"entry","starsfrom","hbS",0,"scc=2");
   zdialog_add_widget(zd,"entry","starsto","hbS",0,"scc=2");
   zdialog_add_widget(zd,"check","lastver","hbS",ZTX("last version only"),"space=10");
   zdialog_add_widget(zd,"label","space","hbS",0,"expand");
   zdialog_add_widget(zd,"label","all-any","hbS",ZTX("all/any"),"space=2");

   zdialog_add_widget(zd,"hbox","hbT","vb2",0,"space=1");
   zdialog_add_widget(zd,"frame","frameT","hbT",0,"expand");
   zdialog_add_widget(zd,"text","searchtags","frameT",0,"expand|wrap");
   zdialog_add_widget(zd,"radio","alltags","hbT",0);
   zdialog_add_widget(zd,"radio","anytags","hbT",0);

   zdialog_add_widget(zd,"hbox","hbC","vb2",0,"space=1|expand");
   zdialog_add_widget(zd,"entry","searchtext","hbC",0,"expand");
   zdialog_add_widget(zd,"radio","alltext","hbC",0);
   zdialog_add_widget(zd,"radio","anytext","hbC",0);

   zdialog_add_widget(zd,"hbox","hbF","vb2",0,"space=1|expand");
   zdialog_add_widget(zd,"entry","searchfiles","hbF",0,"expand");
   zdialog_add_widget(zd,"radio","allfiles","hbF",0);
   zdialog_add_widget(zd,"radio","anyfiles","hbF",0);
   
   zdialog_add_widget(zd,"hbox","hbO","dialog","space=3");
   zdialog_add_widget(zd,"label","labO1","hbO",ZTX("other criteria"),"space=5");
   zdialog_add_widget(zd,"button","geotags","hbO",Bgeotags,"space=8");
   zdialog_add_widget(zd,"label","geotags#","hbO","( )");
   zdialog_add_widget(zd,"label","space","hbO",0,"space=5");
   zdialog_add_widget(zd,"button","other","hbO",ZTX("other"),"space=8");
   zdialog_add_widget(zd,"label","other#","hbO","( )");
   
   zdialog_add_widget(zd,"hbox","space","dialog",0,"space=3");
   zdialog_add_widget(zd,"hbox","hbA1","dialog");
   zdialog_add_widget(zd,"label","labdeftags","hbA1",ZTX("Defined Tags:"),"space=5");
   zdialog_add_widget(zd,"hbox","hbA2","dialog",0,"expand");
   zdialog_add_widget(zd,"frame","frameA","hbA2",0,"space=5|expand");
   zdialog_add_widget(zd,"scrwin","scrwinA","frameA",0,"expand");
   zdialog_add_widget(zd,"text","deftags","scrwinA",0,"expand|wrap"); 

   widget = zdialog_widget(zd,"searchtags");                               //  tag widget mouse functions
   textwidget_set_clickfunc(widget,search_searchtags_clickfunc);

   widget = zdialog_widget(zd,"deftags");
   textwidget_set_clickfunc(widget,search_deftags_clickfunc);
   
   zdialog_stuff(zd,"allimages",1);                                        //  defaults
   zdialog_stuff(zd,"currset",0);
   zdialog_stuff(zd,"newset",1);
   zdialog_stuff(zd,"addset",0);
   zdialog_stuff(zd,"remset",0);
   zdialog_stuff(zd,"repgallery",1);
   zdialog_stuff(zd,"repmeta",0);
   zdialog_stuff(zd,"lastver",0);                                          //  v.14.09
   zdialog_stuff(zd,"alltags",0);
   zdialog_stuff(zd,"anytags",1);
   zdialog_stuff(zd,"alltext",0);
   zdialog_stuff(zd,"anytext",1);
   zdialog_stuff(zd,"allfiles",0);
   zdialog_stuff(zd,"anyfiles",1);

   zdialog_restore_inputs(zd);                                             //  preload prior user inputs

   load_deftags();                                                         //  stuff defined tags into dialog
   deftags_stuff(zd);

   zdialog_resize(zd,0,600);                                               //  start dialog
   zdialog_run(zd,searchimages_dialog_event,"save");
   zdialog_wait(zd);                                                       //  wait for dialog completion
   zdialog_free(zd);

   return;
}

//  mouse click functions for search tags and defined tags widgets

void search_searchtags_clickfunc(GtkWidget *widget, int line, int pos)     //  search tag clicked
{
   char     *txline, *txtag, end = 0;
   cchar    *delims = tagdelims":";
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag) { zfree(txline); return; }
   
   del_tag(txtag,tags_searchtags);                                         //  remove from search list
   zdialog_stuff(zdsearchimages,"searchtags",tags_searchtags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


void search_deftags_clickfunc(GtkWidget *widget, int line, int pos)        //  defined tag clicked
{
   char     *txline, *txtag, end = 0;
   cchar    *delims = tagdelims":";
   
   txline = textwidget_get_line(widget,line,0);
   if (! txline) return;

   txtag = textwidget_get_word(txline,pos,delims,end);
   if (! txtag) { zfree(txline); return; }
   
   add_tag(txtag,tags_searchtags,tagScc);                                  //  add to search tag list
   zdialog_stuff(zdsearchimages,"searchtags",tags_searchtags);
   
   zfree(txline);
   zfree(txtag);
   return;
}


//  search images dialog event and completion callback function

int searchimages_dialog_event(zdialog *zd, cchar *event)                   //  overhauled
{
   using namespace navi;

   int  searchimages_select(sxrec_t &sxrec);
   int  searchimages_geotags_dialog(zdialog *zd);
   int  searchimages_metadata_dialog(zdialog *zd);
   int  searchimages_metadata_report();

   cchar    dateLoDefault[16] = "000001010000";                            //  date: 0000/01/01  time: 00:00
   cchar    dateHiDefault[16] = "999912312359";                            //  date: 9999/12/31  time: 23:59

   char     *file, resultsfile[100];
   char     **flist, *pp, buffer[maxfcc];
   int      ftf, match, ii, cc, err;
   int      Nadded, Nremoved, Nleft, Npver;
   sxrec_t  sxrec;
   FILE     *fid;

   if (strEqu(event,"geotags"))                                            //  get geotags search criteria
      searchimages_geotags_dialog(zd);

   if (strEqu(event,"other"))                                              //  get other metadata criteria
      searchimages_metadata_dialog(zd);

   Fgtags = 0;
   if (*searchCity || *searchCountry ||                                    //  search geotags was given
       *searchLatitude || *searchLongitude) Fgtags = 1;
   
   Fsearchmeta = 0;
   if (Nsearchkeys) Fsearchmeta = 1;                                       //  search other metadata was given
   
   if (Fgtags) zdialog_stuff(zd,"geotags#","(*)");                         //  add (*) visual flags to main dialog if
   else zdialog_stuff(zd,"geotags#","( )");                                //    geotags or metadata selection active
   if (Fsearchmeta) zdialog_stuff(zd,"other#","(*)");
   else zdialog_stuff(zd,"other#","( )");
   
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  [proceed]  v.14.03

   if (! zd->zstat) return 1;                                              //  wait for dialog completion

   if (zd->zstat != 1) return 1;                                           //  cancel

   zdialog_fetch(zd,"allimages",Fscanall);                                 //  search all images
   zdialog_fetch(zd,"currset",Fscancurr);                                  //  search current set (gallery)
   zdialog_fetch(zd,"newset",Fnewset);                                     //  matching images --> new set
   zdialog_fetch(zd,"addset",Faddset);                                     //  add matching image to set
   zdialog_fetch(zd,"remset",Fremset);                                     //  remove matching images from set

   if (Fremset && Fscanall) {                                              //  illogical search
      zmessageACK(Mwin,0,ZTX("to remove images from current set, \n"
                             "search current set"));
      zd->zstat = 0;                                                       //  keep dialog active
      return 1;
   }
   
   if (Faddset && Fscancurr) {
      zmessageACK(Mwin,0,ZTX("to add images to current set, \n"
                             "search all images"));
      zd->zstat = 0;                                                       //  keep dialog active
      return 1;
   }

   zdialog_fetch(zd,"repgallery",Frepgallery);                             //  gallery report                     v.14.02
   zdialog_fetch(zd,"repmeta",Frepmeta);                                   //  metadata report
   zdialog_fetch(zd,"lastver",Flastver);                                   //  get last versions only             v.14.09

   zdialog_fetch(zd,"datefrom",searchDateFrom,15);                         //  get search date range
   zdialog_fetch(zd,"dateto",searchDateTo,15);
   zdialog_fetch(zd,"starsfrom",searchStarsFrom,2);                        //  get search stars range
   zdialog_fetch(zd,"starsto",searchStarsTo,2);
   zdialog_fetch(zd,"searchtags",tags_searchtags,tagScc);                  //  get search tags
   zdialog_fetch(zd,"searchtext",tags_searchtext,tagScc);                  //  get search text*
   zdialog_fetch(zd,"searchfiles",meta_searchfiles,tagScc);                //  get search /path*/file*

   zdialog_fetch(zd,"alltags",Falltags);                                   //  get match all/any options
   zdialog_fetch(zd,"alltext",Falltext);
   zdialog_fetch(zd,"allfiles",Fallfiles);

   Fdates = 0;
   if (*searchDateFrom) Fdates++;                                          //  search date from was given
   else strcpy(searchDateFrom,"000001010000");                             //  else search from begining of time

   if (*searchDateTo) Fdates++;                                            //  search date to was given
   else strcpy(searchDateTo,"999912312359");                               //  else search to end of time

   if (Fdates) {                                                           //  complete partial date/time data
      cc = strlen(searchDateFrom);
      for (ii = cc; ii < 12; ii++)                                         //  default date from:
         searchDateFrom[ii] = dateLoDefault[ii];                           //    date: 0000/01/01 time: 00:00
      cc = strlen(searchDateTo);
      for (ii = cc; ii < 12; ii++)                                         //  default date to:
         searchDateTo[ii] = dateHiDefault[ii];                             //    date: 9999/12/31 time: 23:59
   }
   
   Fstars = 0;
   if (*searchStarsFrom || *searchStarsTo) Fstars = 1;                     //  stars was given
   
   Ffiles = 0;
   if (! blank_null(meta_searchfiles)) Ffiles = 1;                         //  search path / file (fragment) was given

   Ftext = 0;
   if (! blank_null(tags_searchtext)) Ftext = 1;                           //  search text was given

   Ftags = 0;
   if (! blank_null(tags_searchtags)) Ftags = 1;                           //  search tags was given
   
   if (Ffiles) strToLower(meta_searchfiles);                               //  all comparisons in lower case
   if (Ftags) strToLower(tags_searchtags);
   if (Ftext) strToLower(tags_searchtext);
   
   if (checkpend("all")) return 1;                                         //  check nothing pending
   Ffuncbusy++;  

   Nadded = Nremoved = Npver = Nleft = 0;                                  //  image counts

   //  search all images and keep those meeting search criteria
   //  result is gallery of images meeting criteria
   
   if (Fscanall && Fnewset)
   {
      snprintf(resultsfile,100,"%s/search_results",tempdir);               //  open file to save search results
      fid = fopen(resultsfile,"w");
      if (! fid) goto filerror;
      
      ftf = 1;

      while (true)
      {
         zmainloop(20);

         err = read_sxrec_seq(sxrec,ftf);                                  //  scan all index recs.
         if (err) break;
         
         match = searchimages_select(sxrec);                               //  test against select criteria
         if (match) {                                                      //  all criteria passed
            Nadded++;                                                      //  count matches
            fprintf(fid,"%s\n",sxrec.file);                                //  save matching filename
         }

         zfree(sxrec.file);                                                //  free allocated strings
         zfree(sxrec.tags);
         zfree(sxrec.capt);
         zfree(sxrec.comms);
         zfree(sxrec.gtags);
      }

      fclose(fid);
      Nleft = Nadded;
   }
   
   //  search all images and add those meeting search criteria
   //    to current image set (gallery)

   if (Fscanall && Faddset)
   {
      snprintf(resultsfile,100,"%s/search_results",tempdir);               //  open file to save search results
      fid = fopen(resultsfile,"w");
      if (! fid) goto filerror;

      for (ii = 0; ii < navi::nfiles; ii++)                                //  scan current gallery
      {
         file = gallery(0,"find",ii);
         if (! file) break;
         if (*file != '!') {                                               //  skip directories
            fprintf(fid,"%s\n",file);                                      //  add image files to output
            Nleft++;
         }
         zfree(file);                                                      //  free memory
      }

      ftf = 1;

      while (true)
      {
         zmainloop(20);
         
         err = read_sxrec_seq(sxrec,ftf);                                  //  scan all index recs.
         if (err) break;
         
         match = searchimages_select(sxrec);                               //  test against select criteria
         if (match) {                                                      //  all criteria passed
            Nadded++;                                                      //  count matches
            fprintf(fid,"%s\n",sxrec.file);                                //  save matching filename
         }

         zfree(sxrec.file);                                                //  free memory
         zfree(sxrec.tags);
         zfree(sxrec.capt);
         zfree(sxrec.comms);
         zfree(sxrec.gtags);
      }

      fclose(fid);
      Nleft += Nadded;
   }

   //  search current image set and keep only those meeting search criteria
   
   if (Fscancurr && Fnewset)
   {
      snprintf(resultsfile,100,"%s/search_results",tempdir);               //  open file to save search results
      fid = fopen(resultsfile,"w");
      if (! fid) goto filerror;

      for (ii = 0; ii < navi::nfiles; ii++)                                //  scan current gallery
      {
         zmainloop(20);

         file = gallery(0,"find",ii);
         if (! file) break;
         if (*file == '!') {                                               //  skip directories
            zfree(file);
            continue;
         }

         err = get_sxrec(sxrec,file);
         if (err) {                                                        //  no metadata rec?
            zfree(file);
            continue;
         }

         match = searchimages_select(sxrec);                               //  test against select criteria
         if (match) {                                                      //  passed
            Nleft++;                                                       //  count retained images
            fprintf(fid,"%s\n",file);                                      //  save retained filename
         }
         else Nremoved++;
         
         zfree(file);                                                      //  free memory

         zfree(sxrec.file);
         zfree(sxrec.tags);
         zfree(sxrec.capt);
         zfree(sxrec.comms);
         zfree(sxrec.gtags);
      }

      fclose(fid);
   }
   
   //  search current image set and remove those meeting search criteria
   
   if (Fscancurr && Fremset)
   {
      snprintf(resultsfile,100,"%s/search_results",tempdir);               //  open file to save search results 
      fid = fopen(resultsfile,"w");
      if (! fid) goto filerror;

      for (ii = 0; ii < navi::nfiles; ii++)                                //  scan current gallery
      {
         zmainloop(20);

         file = gallery(0,"find",ii);
         if (! file) break;
         if (*file == '!') {                                               //  skip directories
            zfree(file);
            continue;
         }

         err = get_sxrec(sxrec,file);
         if (err) {                                                        //  no metadata rec?
            zfree(file);
            continue;
         }

         match = searchimages_select(sxrec);                               //  test against select criteria
         if (! match) {                                                    //  failed
            Nleft++;
            fprintf(fid,"%s\n",file);                                      //  save retained filename
         }
         else Nremoved++;
         
         zfree(file);                                                      //  free memory

         zfree(sxrec.file);
         zfree(sxrec.tags);
         zfree(sxrec.capt);
         zfree(sxrec.comms);
         zfree(sxrec.gtags);
      }

      fclose(fid);
   }

   if (Flastver)                                                           //  remove all but latest versions     v.14.09
   {
      cc = Nleft * sizeof(char *);
      flist = (char **) zmalloc(cc);
      
      fid = fopen(resultsfile,"r");                                        //  read file of selected image files
      if (! fid) goto filerror;
      
      for (ii = 0; ii < Nleft; ii++)                                       //  build file list in memory
      {
         file = fgets_trim(buffer,maxfcc,fid);
         if (! file) break;
         flist[ii] = zstrdup(file);
      }

      fclose(fid);
      
      for (ii = 1; ii < Nleft; ii++)                                       //  scan file list in memory
      {
         pp = strrchr(flist[ii],'/');                                      //  /directory.../filename.v...
         if (! pp) continue;                                               //  |                     |
         pp = strstr(pp,".v");                                             //  flist[ii]             pp
         if (! pp) continue;
         cc = pp - flist[ii] + 1;
         if (strnEqu(flist[ii],flist[ii-1],cc)) {                          //  compare each filespec with prior
            zfree(flist[ii-1]);                                            //  match: remove prior from list
            flist[ii-1] = 0;
         }
      }
      
      fid = fopen(resultsfile,"w");                                        //  write remaining file list
      if (! fid) goto filerror;                                            //    to results file
      
      Npver = 0;
      for (ii = 0; ii < Nleft; ii++)
      {
         file = flist[ii];
         if (file) {
            fprintf(fid,"%s\n",file);
            zfree(file);
         }
         else Npver++;
      }

      fclose(fid);
      zfree(flist);
      
      if (Nadded) Nadded -= Npver;                                         //  update counts
      Nremoved += Npver;
      Nleft -= Npver;
   }

   Ffuncbusy--;  
   
   zmessageACK(Mwin,0,ZTX("images added: %d  removed: %d  new count: %d"),
                           Nadded, Nremoved, Nleft);
   if (Nleft == 0) {
      zmessageACK(Mwin,0,ZTX("no changes made"));
      return 1;
   }

   free_resources();
   navi::gallerytype = 2;                                                  //  search results
   gallery(resultsfile,"initF");                                           //  generate gallery of matching files

   if (Frepmeta)                                                           //  metadata report format
      searchimages_metadata_report();

   m_viewmode(0,"G");
   gallery(0,"paint",0);                                                   //  position at top 
   return 1;

filerror:
   zmessLogACK(Mwin,"file error: %s",strerror(errno));
   Ffuncbusy--;  
   return 1;
}


//  test a given image against selection criteria, return match status

int searchimages_select(sxrec_t &sxrec) 
{
   int  searchimages_geotags_select(char *gtagsrec);
   int  searchimages_metadata_select(char *imagefile);

   cchar    *pps, *ppf;
   int      iis, iif;
   int      Nmatch, Nnomatch;

   if (Ffiles)                                                             //  file name match is wanted
   {
      Nmatch = Nnomatch = 0;

      for (iis = 1; ; iis++)
      {
         pps = strField(meta_searchfiles,' ',iis);                         //  step thru search file names
         if (! pps) break;
         if (strcasestr(sxrec.file,pps)) Nmatch++;
         else Nnomatch++;
      }
      
      if (Nmatch == 0) return 0;                                           //  no match any file
      if (Fallfiles && Nnomatch) return 0;                                 //  no match all files (dir & file names)
   }

   if (Fdates)                                                             //  date match is wanted
   {
      if (strcmp(sxrec.pdate,searchDateFrom) < 0) return 0;
      if (strcmp(sxrec.pdate,searchDateTo) > 0) return 0;
   }

   if (Ftags)                                                              //  tags match is wanted
   {
      Nmatch = Nnomatch = 0;
      strToLower(sxrec.tags);
      
      for (iis = 1; ; iis++)                                               //  step thru search tags
      {
         pps = strField(tags_searchtags,tagdelims,iis);                    //  (delimited, wildcards) 
         if (! pps) break;
         if (*pps == ' ') continue;

         for (iif = 1; ; iif++)                                            //  step thru file tags (delimited)
         {
            ppf = strField(sxrec.tags,tagdelims,iif);
            if (! ppf) { Nnomatch++; break; }                              //  count matches and fails
            if (*ppf == ' ') continue;
            if (MatchWild(pps,ppf) == 0) { Nmatch++; break; }              //  wildcard match 
         }
      }
      
      if (Nmatch == 0) return 0;                                           //  no match to any tag
      if (Falltags && Nnomatch) return 0;                                  //  no match to all tags
   }

   if (Fstars)                                                             //  rating (stars) match is wanted
   {
      if (*searchStarsFrom && sxrec.rating[0] < *searchStarsFrom) return 0;
      if (*searchStarsTo && sxrec.rating[0] > *searchStarsTo) return 0;
   }

   if (Ftext)                                                              //  text match is wanted
   {
      Nmatch = Nnomatch = 0;

      for (iis = 1; ; iis++)                                               //  step through search words
      {
         pps = strField(tags_searchtext,' ',iis);
         if (! pps) break;
         if (*pps == ' ') continue;
         if (strcasestr(sxrec.capt,pps)) Nmatch++;                         //  search captions for word
         else if (strcasestr(sxrec.comms,pps)) Nmatch++;                   //  search comments for word
         else Nnomatch++;
      }
                  
      if (Nmatch == 0) return 0;                                           //  no match to any word
      if (Falltext && Nnomatch) return 0;                                  //  no match to all words
   }

   if (Fgtags)                                                             //  geotags match is wanted
      if (! searchimages_geotags_select(sxrec.gtags)) return 0;
   
   if (Fsearchmeta)                                                        //  other metadata match
      if (! searchimages_metadata_select(sxrec.file)) return 0;
   
   return 1;
}


/**************************************************************************/

//  dialog to get geotags search criteria

int searchimages_geotags_dialog(zdialog *zdp)
{
   int searchimages_geotags_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   int         zstat;

/***
            Add Geotags Search Criteria

         city [__________]  country [__________]
         latitude [_______]  longitude [_______] 
         [find]  [map]   range (km) [___]
         
                      [clear] [apply] [cancel]
***/

   if (! init_geolocs()) return 0;

   zd = zdialog_new(ZTX("Add Geotags Search Criteria"),Mwin,Bclear,Bapply,Bcancel,null);

   zdialog_add_widget(zd,"hbox","hbg1","dialog",0);
   zdialog_add_widget(zd,"label","labcity","hbg1",ZTX("city"),"space=3");
   zdialog_add_widget(zd,"entry","city","hbg1",0,"expand");
   zdialog_add_widget(zd,"label","space","hbg1",0,"space=5");
   zdialog_add_widget(zd,"label","labcountry","hbg1",ZTX("country"),"space=3");
   zdialog_add_widget(zd,"entry","country","hbg1",0,"expand");

   zdialog_add_widget(zd,"hbox","hbg2","dialog",0);
   zdialog_add_widget(zd,"label","lablat","hbg2","latitude","space=3");
   zdialog_add_widget(zd,"entry","latitude","hbg2",0,"scc=10");
   zdialog_add_widget(zd,"label","space","hbg2",0,"space=5");
   zdialog_add_widget(zd,"label","lablong","hbg2","longitude","space=3");
   zdialog_add_widget(zd,"entry","longitude","hbg2",0,"scc=10");

   zdialog_add_widget(zd,"hbox","hbg3","dialog",0);
   zdialog_add_widget(zd,"button","find","hbg3",Bfind,"space=10");
   zdialog_add_widget(zd,"button","map","hbg3",Bmap,"space=10");
   zdialog_add_widget(zd,"label","labkm","hbg3",ZTX("range (km)"),"space=10");
   zdialog_add_widget(zd,"entry","range","hbg3","100","scc=5");

   zdialog_show(zdp,0);                                                    //  hide parent dialog
   zdialog_restore_inputs(zd);                                             //  preload prior user inputs
   zdialog_run(zd,searchimages_geotags_dialog_event);                      //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion
   zdialog_free(zd);
   zd_geotags = 0;                                                         //  deactivate world map clicks
   zdialog_show(zdp,1);                                                    //  restore parent dialog
   
   if (zstat == 1) return 1;                                               //  search criteria were entered
   else return 0;                                                          //  not
}


//  dialog event and completion function

int searchimages_geotags_dialog_event(zdialog *zd, cchar *event)
{
   int         fgtags, err, nn;
   char        city[100], country[100];
   char        *location[2], *coord[2];

   if (strEqu(event,"find")) {                                             //  get geotags data from 
      zdialog_fetch(zd,"city",city,99);                                    //    (partial) city [country] names
      zdialog_fetch(zd,"country",country,99);
      location[0] = city;
      location[1] = country;
      nn = geotags_choosecity(location,coord);                             //  find matching city geotags data
      if (nn == 1) {                                                       //    and ask user to choose
         zdialog_stuff(zd,"city",location[0]);
         zdialog_stuff(zd,"country",location[1]);
         zdialog_stuff(zd,"latitude",coord[0]);
         zdialog_stuff(zd,"longitude",coord[1]);
      }
   }

   if (strEqu(event,"map"))                                                //  get geotags data from world map
   {
      zd_geotags = zd;
      click_worldmap();
      return 0;
   }
   
   if (strEqu(event,"enter")) zd->zstat = 2;                               //  [apply]  v.14.03

   if (! zd->zstat) return 0;                                              //  wait for completion
   
   if (zd->zstat == 1) {
      zdialog_stuff(zd,"city","");                                         //  clear
      zdialog_stuff(zd,"country","");
      zdialog_stuff(zd,"latitude","");
      zdialog_stuff(zd,"longitude","");
      zd->zstat = 0;                                                       //  keep dialog active
      return 0;
   }

   if (zd->zstat != 2) {                                                   //  cancel
      *searchCity = *searchCountry = 0;                                    //  clear criteria
      *searchLatitude = *searchLongitude = 0;
      zd_geotags = 0;
      return 0;
   }

   zdialog_fetch(zd,"city",searchCity,99);                                 //  apply - get search geotags
   zdialog_fetch(zd,"country",searchCountry,99);
   zdialog_fetch(zd,"latitude",searchLatitude,20);
   zdialog_fetch(zd,"longitude",searchLongitude,20);
   zdialog_fetch(zd,"range",searchRange);

   if (*searchCity || *searchCountry)                                      //  if city or country supplied,
      if (strEqu(searchLatitude,"null"))                                   //    remove "null" lati/longi search
         *searchLatitude = *searchLongitude = 0;

   strToLower(searchCity);
   strToLower(searchCountry);

   err = 0;
   if (*searchLatitude || *searchLongitude) {                              //  sanity check for latitude/longitude
      if (! *searchLatitude || ! *searchLongitude) err++;
      flatitude = atof(searchLatitude);
      flongitude = atof(searchLongitude);
      if (flatitude < -90 || flatitude > 90) err++;
      if (flongitude < -180 || flongitude > 180) err++;
      if (searchRange < 1 || searchRange > 1000) err++;
      if (strEqu(searchLatitude,"null")) err = 0;                          //  search for null lat/long 
   }

   if (err) {
      zmessageACK(Mwin,0,ZTX("error in latitude/longitude/range"));
      zd->zstat = 0;
      return 1;
   }

   fgtags = 0;
   if (! blank_null(searchCity)) fgtags = 1;                               //  look for inputs
   if (! blank_null(searchCountry)) fgtags = 1;
   if (! blank_null(searchLatitude)) fgtags = 1;
   if (! blank_null(searchLongitude)) fgtags = 1;

   if (fgtags) zd->zstat = 1;                                              //  search criteria was given
   else zd->zstat = 2;                                                     //  no inputs were made

   return 1;
}


//  test the image geotags data against the search criteria   

int searchimages_geotags_select(char *gtags)
{
   float       glatitude, glongitude, distance;
   cchar       *pps;

   if (*searchLatitude) {                                                  //  if latitude/longitude present,
      pps = strField(gtags,'^',3);                                         //    ignore city/country data
      if (! pps) return 0;
      if (strEqu(pps,"null"))                                              //  no latitude/longitude data
         if (flatitude == 0 && flongitude == 0) return 1;                  //  "no data" was wanted
      glatitude = atof(pps);
      pps = strField(gtags,'^',4);
      if (! pps) return 0;
      if (strEqu(pps,"null")) return 0;
      glongitude = atof(pps);
      distance = (flatitude - glatitude) * (flatitude - glatitude);
      distance += (flongitude - glongitude) * (flongitude - glongitude);
      distance = 111.0 * sqrt(distance);
      if (distance <= searchRange) return 1;                               //  within distance limit, match
      return 0;                                                            //  outside, no match
   }

   if (*searchCity) {                                                      //  check city and country criteria
      pps = strField(gtags,'^',1);
      if (pps && strcasestr(pps,searchCity) == 0) return 0;
   }
   
   if (*searchCountry) {
      pps = strField(gtags,'^',2);
      if (pps && strcasestr(pps,searchCountry) == 0) return 0;
   }

   return 1;
}


/**************************************************************************/

//  dialog to get metadata search criteria

int searchimages_metadata_dialog(zdialog *zdp)
{
   int searchimages_metadata_dialog_event(zdialog *zd, cchar *event);

   cchar *metamess = ZTX("These items are always reported: \n"
                         "date, stars, tags, caption, comment");
   zdialog  *zd;
   int      zstat;

/***
         Search and Report Metadata

         These items are always reported: 
         date, stars, tags, caption, comment

            Additional Items for Report
            Keyword      Match Criteria
         [__________] [__________________]
         [__________] [__________________]
         [__________] [__________________]
         [__________] [__________________]
         [__________] [__________________]

                   [clear] [apply] [cancel]
***/

   zd = zdialog_new("Search and Report Metadata",Mwin,Bclear,Bapply,Bcancel,null);
   zdialog_add_widget(zd,"label","labmeta","dialog",metamess,"space=3");
   zdialog_add_widget(zd,"label","labopts","dialog",ZTX("Additional Items for Report"));

   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5|expand");
   
   zdialog_add_widget(zd,"label","lab1","vb1",ZTX("Keyword"));
   zdialog_add_widget(zd,"entry","key0","vb1");
   zdialog_add_widget(zd,"entry","key1","vb1");
   zdialog_add_widget(zd,"entry","key2","vb1");
   zdialog_add_widget(zd,"entry","key3","vb1");
   zdialog_add_widget(zd,"entry","key4","vb1");

   zdialog_add_widget(zd,"label","lab2","vb2",ZTX("Match Criteria"));
   zdialog_add_widget(zd,"entry","match0","vb2",0,"expand");
   zdialog_add_widget(zd,"entry","match1","vb2",0,"expand");
   zdialog_add_widget(zd,"entry","match2","vb2",0,"expand");
   zdialog_add_widget(zd,"entry","match3","vb2",0,"expand");
   zdialog_add_widget(zd,"entry","match4","vb2",0,"expand");
   
   strcpy(searchkeyx,"keyx");
   strcpy(searchkeydatax,"matchx");
   
   if (! searchkeys[0])                                                    //  first call initialization
   {
      for (int ii = 0; ii < 5; ii++) {
         searchkeys[ii] = (char *) zmalloc(40);
         searchkeydata[ii] = (char *) zmalloc(100);
         *searchkeys[ii] = *searchkeydata[ii] = 0;
      }
   }   

   zdialog_show(zdp,0);                                                    //  hide parent dialog
   zdialog_resize(zd,400,300);
   zdialog_restore_inputs(zd);                                             //  preload prior user inputs
   zdialog_run(zd,searchimages_metadata_dialog_event);                     //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for completion
   zdialog_free(zd);
   zdialog_show(zdp,1);                                                    //  restore parent dialog
   
   if (zstat == 1) return 1;                                               //  search criteria were entered
   else return 0;                                                          //  not
}


//  dialog event and completion callback function

int searchimages_metadata_dialog_event(zdialog *zd, cchar *event)
{
   int      ii, jj;
   char     keyx[8] = "keyx", matchx[8] = "matchx";

   if (strEqu(event,"enter")) zd->zstat = 2;                               //  [apply]  v.14.03

   if (! zd->zstat) return 1;                                              //  wait for completion

   if (zd->zstat == 1) {
      for (ii = 0; ii < 5; ii++) {                                         //  clear
         keyx[3] = '0' + ii;
         matchx[5] = '0' + ii;
         zdialog_stuff(zd,keyx,"");
         zdialog_stuff(zd,matchx,"");
         zd->zstat = 0;                                                    //  keep dialog active
      }
      return 0;
   }

   if (zd->zstat != 2) {
      Nsearchkeys = 0;                                                     //  no search keys
      return 1;
   }
   
   Nsearchkeys = 0;                                                        //  apply

   for (ii = jj = 0; ii < 5; ii++)                                         //  get metadata keys
   {
      searchkeyx[3] = '0' + ii;
      zdialog_fetch(zd,searchkeyx,searchkeys[ii],40);
      strCompress(searchkeys[ii]);                                         //  remove all blanks from key names
      if (*searchkeys[ii] <= ' ') continue;
      memmove(searchkeys[jj],searchkeys[ii],40);                           //  repack blank keys
      searchkeydatax[5] = '0' + ii;
      zdialog_fetch(zd,searchkeydatax,searchkeydata[ii],100);              //  get corresp. match value if any
      strTrim2(searchkeydata[jj],searchkeydata[ii]);                       //  trim leading and trailing blanks
      if (ii > jj) *searchkeys[ii] = *searchkeydata[ii] = 0;
      jj++;
   }

   Nsearchkeys = jj;                                                       //  keys found, no blanks
   
   if (Nsearchkeys) zd->zstat = 1;                                         //  keys were entered
   else zd->zstat = 2;                                                     //  no keys were entered

   return 1;
}


//  test image metadata against metadata select criteria

int searchimages_metadata_select(char *file)
{
   char           **kvals;
   int            ii, nth;
   cchar          *pp;

   kvals = exif_get(file,(cchar **) searchkeys,Nsearchkeys);               //  get the image metadata

   for (ii = 0; ii < Nsearchkeys; ii++) {                                  //  loop all metadata search keys
      if (*searchkeydata[ii] > ' ') {                                      //  key match value(s) are present
         if (kvals[ii]) {                                                  //  key values present in metadata
            for (nth = 1; ; nth++) {                                       //  loop all match values
               pp = strField(searchkeydata[ii],' ',nth);                   //  get each match value
               if (! pp) return 0;                                         //  no more, no match found
               if (strcasestr(kvals[ii],pp)) break;                        //  found match (substring also)
            }
         }
         else return 0;                                                    //  empty metadata, no match
      }
   }

   return 1;
}


//  Report the selected metadata using a gallery window layout
//  with image thumbnails and selected metadata text.

int searchimages_metadata_report()
{
   using namespace navi;

   cchar    *keys1[7] = { exif_date_key, iptc_rating_key, 
                          iptc_keywords_key,                               //  v.14.02
                          exif_city_key, exif_country_key, 
                          iptc_caption_key, exif_comment_key };
   cchar    *keys[20];
   char     *file, **kvals;
   char     text1[2*indexrecl], text2[200];
   int      Nkeys, ii, jj, cc;
   
   if (! mdlist) {
      cc = nfiles * sizeof(char *);                                        //  allocate metadata list
      mdlist = (char **) zmalloc(cc);                                      //  nfiles = curr. gallery files
      memset(mdlist,0,cc);                                                 //  mdlist = corresp. metadata list
   }                                                                       //  (zfree() in image_navigate)
   
   for (ii = 0; ii < 7; ii++)                                              //  set first 6 key names, fixed
      keys[ii] = keys1[ii];
   
   for (ii = 0; ii < Nsearchkeys; ii++)                                    //  remaining key names from user
      keys[ii+7] = searchkeys[ii];

   Nkeys = 7 + Nsearchkeys;                                                //  total keys to extract              v.14.02
   mdrows = Nkeys - 1;                                                     //  report rows (city/country 1 row)

   for (ii = 0; ii < nfiles; ii++)                                         //  loop image gallery files
   {
      file = gallery(0,"find",ii);
      if (! file) continue;

      kvals = exif_get(file,(cchar **) keys,Nkeys);                        //  get the metadata
      
      snprintf(text2,200,"%s  %s",kvals[3],kvals[4]);                      //  combine city and country
      if (kvals[3]) zfree(kvals[3]);                                       //  bugfix                             v.14.06
      if (kvals[4]) zfree(kvals[4]);
      kvals[3] = zstrdup(text2);
      kvals[4] = 0;
      
      for (cc = jj = 0; jj < Nkeys; jj++)                                  //  add metadata to report
      {
         if (jj == 4) continue;                                            //  skip country
         if (jj == 0 && kvals[0]) kvals[0][4] = kvals[0][7] = '-';         //  conv. yyyy:mm:dd to yyyy-mm-dd
         snprintf(text2,200,"key: %s  value: %s \n",keys[jj], kvals[jj]);
         strcpy(text1+cc,text2);
         cc += strlen(text2);
      }

      if (mdlist[ii]) zfree(mdlist[ii]);                                   //  attach metadata for gallery paint
      mdlist[ii] = zstrdup(text1);

      for (jj = 0; jj < Nkeys; jj++)                                       //  free memory
         if (kvals[jj]) zfree(kvals[jj]);

      zfree(file);
   }

   gallerytype = 3;                                                        //  gallery type = search results/metadata

   return 0;
}


/**************************************************************************
   Functions to read and write exif/iptc or other metadata
***************************************************************************/

//  get EXIF/IPTC metadata for given image file and EXIF/IPTC key(s)
//  returns array of pointers to corresponding key values
//  if a key is missing, corresponding pointer is null
//  returned strings belong to caller, are subject for zfree()
//  up to 20 keynames may be requested per call
//  use -fast not -fast2 which can lose geotag data

char ** exif_get(cchar *file, cchar **keys, int nkeys)                     //  revised
{
   char           *pp, **outputs;
   char           *inputs[30];
   static char    *keyvals[20];
   int            cc, ii, jj;

   if (nkeys < 1 || nkeys > 20) zappcrash("exif_get nkeys: %d",nkeys);

   inputs[0] = (char *) "-m";                                              //  options for exiftool
   inputs[1] = (char *) "-s2";
   inputs[2] = (char *) "-n";
   inputs[3] = (char *) "-fast2";                                          //  -fast2 reinstated                  v.14.09
   jj = 4;

   for (ii = 0; ii < nkeys; ii++)                                          //  build exiftool inputs
   {
      cc = strlen(keys[ii]);                                               //  -keyname
      inputs[jj] = (char *) zmalloc(cc+2);
      inputs[jj][0] = '-';
      strcpy(inputs[jj]+1,keys[ii]);
      jj++;
   }

   inputs[jj] = zstrdup(file);                                             //  filename last
   jj++;

   inputs[jj] = 0;                                                         //  EOL

   cc = 20 * sizeof(char *);
   memset(keyvals,0,cc);                                                   //  set all outputs to null

   outputs = exiftool_server(inputs);                                      //  get exif outputs
   if (! outputs) return 0; 
   
   for (ii = 4; ii < jj; ii++)                                             //  free memory
      zfree(inputs[ii]);

   for (ii = 0; outputs[ii]; ii++)                                         //  search outputs
   {
      pp = outputs[ii];                                                    //  keyname: keyvalue

      for (jj = 0; jj < nkeys; jj++)
      {
         uint cc = strlen(keys[jj]);                                       //  look for matching input keyname
         if (strncasecmp(pp,keys[jj],cc) == 0)
            if (strlen(pp) > cc+2)                                         //  if not empty, 
               keyvals[jj] = zstrdup(pp+cc+2);                             //    return keyvalue alone
      }
   }
   
   return keyvals;
}


/**************************************************************************/

//  create or change EXIF/IPTC metadata for given image file and key(s)
//  up to 20 keys may be processed
//  command: 
//    exiftool -m -overwrite_original -keyname="keyvalue" ... "file"
//
//  NOTE: exiftool replaces \n (newline) in keyvalue with . (period).

int exif_put(cchar *file, cchar **keys, cchar **text, int nkeys)           //  revised
{
   int      ii, jj, cc;
   char     *inputs[30];
   
   if (nkeys < 1 || nkeys > 20) zappcrash("exif_put nkeys: %d",nkeys);

   inputs[0] = (char *) "-m";                                              //  exiftool options
   inputs[1] = (char *) "-overwrite_original";                             //  -P preserve date removed 
   jj = 2;

   for (ii = 0; ii < nkeys; ii++)                                          //  build exiftool inputs
   {
      cc = strlen(keys[ii]) + strlen(text[ii]) + 3;
      inputs[jj] = (char *) zmalloc(cc);
      inputs[jj][0] = '-';                                                 //  -keyname=value
      strcpy(inputs[jj]+1,keys[ii]);
      cc = strlen(keys[ii]);
      inputs[jj][cc+1] = '=';
      strcpy(inputs[jj]+cc+2,text[ii]);
      jj++;
      
      if (strcasecmp(keys[ii],"GPSLatitude") == 0) {                       //  take care of latitude N/S 
         if (*text[ii] == '-')
            inputs[jj] = zstrdup("-GPSLatitudeRef=S");
         else 
            inputs[jj] = zstrdup("-GPSLatitudeRef=N");
         jj++;
      }

      if (strcasecmp(keys[ii],"GPSLongitude") == 0) {                      //  and longitude E/W
         if (*text[ii] == '-')
            inputs[jj] = zstrdup("-GPSLongitudeRef=W");
         else 
            inputs[jj] = zstrdup("-GPSLongitudeRef=E");
         jj++;
      }
   }

   inputs[jj] = zstrdup(file);                                             //  last input is filename
   jj++;
   
   inputs[jj] = 0;                                                         //  EOL

   exiftool_server(inputs);
   
   for (ii = 2; ii < jj-1; ii++)                                           //  free memory
       zfree(inputs[ii]);

   return 0;
}


/**************************************************************************/

//  copy EXIF/IPTC data from one image file to new (edited) image file
//  if nkeys > 0, up to 20 keys may be replaced with new values
//  exiftool -m -tagsfromfile file1 -all -xmp -icc_profile [-keyname=newvalue ...] 
//                file2 -overwrite_original

int exif_copy(cchar *file1, cchar *file2, cchar **keys, cchar **text, int nkeys)
{
   char     *inputs[30];                                                   //  revised
   int      cc, ii, jj;
   
   if (nkeys > 20) zappcrash("exif_copy() nkeys %d",nkeys);

   inputs[0] = (char *) "-m";                                              //  -m               (suppress warnings)
   inputs[1] = (char *) "-tagsfromfile";                                   //  -tagsfromfile
   inputs[2] = zstrdup(file1);                                             //  file1
   inputs[3] = (char *) "-all";                                            //  -all
   inputs[4] = (char *) "-xmp";                                            //  -xmp             added v.14.01
   inputs[5] = (char *) "-icc_profile";                                    //  -icc_profile
   
   jj = 6;                                                                 //  count of inputs so far

   for (int ii = 0; ii < nkeys; ii++)                                      //  -keyname=keyvalue
   {
      cc = strlen(keys[ii]) + strlen(text[ii]) + 3;
      inputs[jj] = (char *) zmalloc(cc);
      *inputs[jj] = 0;                                                     //  v.14.03
      strncatv(inputs[jj],cc,"-",keys[ii],"=",text[ii],null);
      jj++;
   }

   inputs[jj++] = zstrdup(file2);                                          //  file2
   inputs[jj++] = (char *) "-overwrite_original";                          //  -overwrite_original
   inputs[jj] = 0;                                                         //  EOL

   exiftool_server(inputs);

   zfree(inputs[2]);                                                       //  free memory
   for (ii = 6; ii < jj-1; ii++)
   zfree(inputs[ii]);

   return 0;
}


/**************************************************************************/

//  convert between EXIF and fotoxx tag date formats
//  EXIF date: yyyy:mm:dd hh:mm:ss        20 chars.
//  tag date: yyyymmddhhmmss              16 chars.
//  

void exif_tagdate(cchar *exifdate, char *tagdate)
{
   int      cc;
   
   memset(tagdate,0,15);
   cc = strlen(exifdate);
   
   if (cc > 3) strncpy(tagdate+0,exifdate+0,4);
   if (cc > 6) strncpy(tagdate+4,exifdate+5,2);
   if (cc > 9) strncpy(tagdate+6,exifdate+8,2);
   if (cc > 12) strncpy(tagdate+8,exifdate+11,2);
   if (cc > 15) strncpy(tagdate+10,exifdate+14,2);
   if (cc > 18) strncpy(tagdate+12,exifdate+17,2);
   tagdate[14] = 0;
   return;
}

void tag_exifdate(cchar *tagdate, char *exifdate)
{
   int      cc;
   
   memset(exifdate,0,20);
   cc = strlen(tagdate);
   
   strcpy(exifdate,"1900:01:01 00:00:00"); 
   if (cc > 3) strncpy(exifdate+0,tagdate+0,4);
   if (cc > 5) strncpy(exifdate+5,tagdate+4,2);
   if (cc > 7) strncpy(exifdate+8,tagdate+6,2);
   if (cc > 9) strncpy(exifdate+11,tagdate+8,2);
   if (cc > 11) strncpy(exifdate+14,tagdate+10,2);
   if (cc > 13) strncpy(exifdate+17,tagdate+12,2);
   exifdate[19] = 0;
   return;
}


/**************************************************************************

   char ** exiftool_server(char **inputs)
   
   Server wrapper for exiftool for put/get exif/iptc data.
   This saves perl startup overhead for each call (0.1 >> 0.01 secs).

   Input records: -opt1
                  -opt2
                    ...
                  -keyword1=value1
                  -keyword2=value2
                    ... 
                  filename.jpg
                  (null pointer)

   Returned: list of pointers to resulting output records.
   The last output record is a null pointer.
   Returns null if there was a fatal error.
   Prior output records are freed automatically with each 
   new call, so the caller need not worry about this.

   First call: 
      Starts exiftool with pipe input and output files.
      Sends input record to exiftool and returns output records.
   Subsequent calls: 
      The existing exiftool process is re-used so that the 
      substantial startup overhead is avoided.
   To kill the exiftool process: exiftool_server(null).

***************************************************************************/

char ** exiftool_server(char **inputs)                                     //  revised API
{
   int            ii, cc, err;
   static int     fcf = 1;                                                 //  first call flag
   static FILE    *fid1 = 0, *fid2 = 0;                                    //  exiftool input, output files
   char           *pp, command[100];
   static char    exiftool_input[100] = "";
   static char    *outrecs[100];                                           //  pointers for up to 99 output recs.
   char           outrec[exif_maxcc];                                      //  single output record

   if (! *exiftool_input)                                                  //  exiftool_server input file
      snprintf(exiftool_input,99,"%s/exiftool_input",tempdir);

   if (! inputs)                                                           //  kill exiftool process
   {                                                                       //  (also orphaned process)
      fid1 = fopen(exiftool_input,"a");
      if (fid1) {
         fprintf(fid1,"-stay_open\nFalse\n");                              //  tell it to exit 
         fclose(fid1);                                                     //  bugfix: fflush after fclose 
      }
      remove(exiftool_input);
      if (fid2) pclose(fid2);
      fid2 = 0;
      fcf = 1;                                                             //  start exiftool process if called again
      return 0;
   }

   if (fcf)                                                                //  first call only
   {
      fid1 = fopen(exiftool_input,"w");                                    //  start exiftool input file
      if (! fid1) {
         zmessLogACK(Mwin,"exiftool_server error: %s \n",strerror(errno));
         return 0;
      }

      snprintf(command,100,"exiftool -stay_open True -@ %s",exiftool_input);

      fid2 = popen(command,"r");                                           //  start exiftool and output file
      if (! fid2) {
         zmessLogACK(Mwin,"exiftool_server error: %s \n",strerror(errno));
         fclose(fid1);
         return 0;
      }

      cc = 100 * sizeof(char *);                                           //  initz. no outputs
      memset(outrecs,0,cc);

      fcf = 0;                                                             //  keep last
   }

   for (ii = 0; ii < 99; ii++)                                             //  free memory from prior call
   {
      if (! outrecs[ii]) break;
      zfree(outrecs[ii]);
      outrecs[ii] = 0;
   }

   gallery_monitor("stop");                                                //  stop excess gallery inits
   
   for (ii = 0; inputs[ii]; ii++)
      fprintf(fid1,"%s\n",inputs[ii]);                                     //  write to exiftool, 1 per record

   err = fprintf(fid1,"-execute\n");                                       //  tell exiftool to process
   if (err < 0) {
      zmessLogACK(Mwin,"exiftool_server error: %s \n",strerror(errno));
      return 0;
   }

   fflush(fid1);                                                           //  flush buffer

   for (ii = 0; ii < 99; ii++)                                             //  get exiftool outputs
   {
      pp = fgets_trim(outrec,exif_maxcc,fid2,1);
      if (! pp) break;
      if (strncmp(outrec,"{ready}",7) == 0) break;                         //  look for output end
      outrecs[ii] = zstrdup(outrec);                                       //  add to returned records
   }

   outrecs[ii] = 0;                                                        //  mark output end
   if (ii == 99) zmessLogACK(Mwin,"exiftool_server >99 output recs \n");

   gallery_monitor("start");

   return outrecs;                                                         //  return outputs to caller
}


/**************************************************************************
   Functions to read and write image index records
***************************************************************************/

//  Initialize for reading/writing to the image index.
//  Build a memory map of the first image file in each image index subfile.
//  Set refresh = 1 to initialize after an index file has been split 
//    or the first image file in an image index file has changed.
//  Returns 0 if OK, +N otherwise.
//
//    /.../.fotoxx/image_index/index_001
//    /.../.fotoxx/image_index/index_002
//    /.../.fotoxx/image_index/index_...         up to 999
//
//  Each of these files contains up to 'image_index_max' image filespecs.

char     **image_index_map = 0;                                            //  1st image file in each index file

int init_image_index_map(int refresh)
{
   int      ii, cc;
   char     *pp;
   char     indexfile[200];
   char     buff[indexrecl];
   FILE     *fid;

   if (image_index_map && ! refresh) return 0;                             //  already initialized

   if (image_index_map) {                                                  //  free memory
      for (ii = 0; image_index_map[ii]; ii++)
         zfree(image_index_map[ii]);
      zfree(image_index_map);
   }

   cc = 1001 * sizeof(char *);                                             //  allocate memory
   image_index_map = (char **) zmalloc(cc);
   memset(image_index_map,0,cc);
   
   for (ii = 0; ii < 999; ii++)                                            //  read image index files 001 to 999
   {
      snprintf(indexfile,200,"%s/index_%03d",index_dirk,ii+1);             //  image_index_map[ii] is for
      fid = fopen(indexfile,"r");                                          //    image_index_file_(ii+1)
      if (! fid) break;
      pp = fgets_trim(buff,indexrecl,fid);
      if (pp && strnEqu(pp,"file: ",6)) 
         pp = zstrdup(pp+6);                                               //  map first image file
      else pp = zstrdup("empty");                                          //  if none, map "empty"
      image_index_map[ii] = pp;
      fclose(fid);
   }

   image_index_map[ii] = 0;                                                //  mark EOL
   
   if (ii == 0) {
      zmessLogACK(Mwin,ZTX("image index is missing"));                     //  v.14.02
      m_quit(0,0);
   }

   if (ii == 999) {
      zmessLogACK(Mwin,"too many image index files");
      m_quit(0,0);
   }

   return 0;
}


//  Return which image index file a given image file belongs in.

int find_image_index_file(cchar *file)
{
   char     *pp;
   int      ii, nn;
   
   init_image_index_map(0);

   for (ii = 0; (pp = image_index_map[ii]); ii++)
   {
      if (strEqu(pp,"empty")) continue;                                    //  same sort as image_index_compare()
      nn = strcasecmp(file,pp);
      if (nn == 0) nn = strcmp(file,pp);
      if (nn < 0) break;                                                   //  file < 1st index file, go back 1
   }
   
   if (ii == 0) ii = 1;                                                    //  file < 1st file in 1st index file
   
   while (ii > 1 && strEqu(image_index_map[ii-1],"empty")) ii--;
   return ii;                                                              //  (map ii is for index file ii+1)
}


/**************************************************************************/

//  Split an index file when the number of entries > image_index_max
//  All following index files are renumbered +1.
//  Index_index file is refreshed.
//  Returns 0 if OK, else +N.

int split_image_index_file(int Nth)
{
   int      ii, err, nn, last, filecount;
   char     indexfile1[200], indexfile2[200], indexfile3[200];
   char     buff[indexrecl], *pp;
   FILE     *fid1 = 0, *fid2 = 0, *fid3 = 0;
   STATB    statb;

   if (sxrec_fid) {                                                        //  stop get_sxrec()
      fclose(sxrec_fid);
      sxrec_fid = 0;
   }

   for (last = Nth; last < 999; last++)                                    //  find last index file
   {
      snprintf(indexfile1,200,"%s/index_%03d",index_dirk,last);            //  /.../.fotoxx/image_index/index_NNN
      err = stat(indexfile1,&statb);
      if (err) break;
   }

   last--;

   if (last == 999) {
      zmessLogACK(Mwin,"too many image index files");
      return 1;
   }

   for (ii = last; ii > Nth; ii--)                                         //  rename files following Nth
   {                                                                       //    index_012 >> index_013 etc.
      snprintf(indexfile1,200,"%s/index_%03d",index_dirk,ii);
      snprintf(indexfile2,200,"%s/index_%03d",index_dirk,ii+1);
      err = rename(indexfile1,indexfile2);
      if (err) goto file_err;
   }

   snprintf(indexfile1,200,"%s/index_%03d",index_dirk,Nth);                //  read Nth file
   snprintf(indexfile2,200,"%s/index_%03d_temp2",index_dirk,Nth);          //  write 2 temp. files
   snprintf(indexfile3,200,"%s/index_%03d_temp3",index_dirk,Nth);
   
   fid1 = fopen(indexfile1,"r");
   if (! fid1) goto file_err;
   fid2 = fopen(indexfile2,"w");
   if (! fid2) goto file_err;
   fid3 = fopen(indexfile3,"w");
   if (! fid3) goto file_err;

   filecount = 0;   

   while (true)                                                            //  copy half the entries
   {                                                                       //    to temp2 file
      pp = fgets_trim(buff,indexrecl,fid1);
      if (! pp) break;
      if (strnEqu(pp,"file: ",6)) filecount++;
      if (filecount > image_index_max/2) break;
      nn = fprintf(fid2,"%s\n",pp);
      if (! nn) goto file_err;
   }

   if (pp) {
      nn = fprintf(fid3,"%s\n",pp);                                        //  copy remaining record sets
      if (! nn) goto file_err;                                             //    to temp3 file
   }

   while (true)
   {
      pp = fgets_trim(buff,indexrecl,fid1);
      if (! pp) break;
      nn = fprintf(fid3,"%s\n",pp);
      if (! nn) goto file_err;
   }
   
   err = fclose(fid1);
   err = fclose(fid2);
   err = fclose(fid3);
   fid1 = fid2 = fid3 = 0;
   if (err) goto file_err;
   
   err = rename(indexfile2,indexfile1);                                    //  temp2 file replaces Nth index file
   if (err) goto file_err;
   
   snprintf(indexfile1,200,"%s/index_%03d",index_dirk,Nth+1);              //  temp3 file >> Nth+1 index file
   err = rename(indexfile3,indexfile1);
   if (err) goto file_err;
   
   err = init_image_index_map(1);                                          //  update image index map
   return err;

file_err:
   zmessLogACK(Mwin,"split_image_index error \n %s",strerror(errno));
   if (fid1) fclose(fid1);
   if (fid2) fclose(fid2);
   if (fid3) fclose(fid3);
   return 3;
}


/**************************************************************************/

//  Get the image index record for given image file.
//  Returns 0 if OK, 1 if not found, >1 if error (diagnosed).
//  Returned sxrec_t data has allocated fields subject to zfree().
//
//  Index file is kept open across calls to reduce overhead for the 
//  normal case of records being accessed in image file sequence
//  (gallery paint). Random access works, but is slower.

int get_sxrec(sxrec_t &sxrec, cchar *file)
{
   FILE           *fid = 0;
   char           indexfile[200];
   static char    buff[indexrecl];
   static char    pfile[maxfcc];
   int            err, Nth, nn = 0, Fcontinue;
   cchar          *pp, *pp2;
   static int     pNth, logerr = 0;
   STATB          statb;

   memset(&sxrec,0,sizeof(sxrec));                                         //  clear output record
   err = stat(file,&statb);                                                //  check image file exists
   if (err) return 1;

   Fcontinue = 1;                                                          //  test if prior search can continue
   if (! sxrec_fid) Fcontinue = 0;                                         //  no prior search still open
   Nth = find_image_index_file(file);                                      //  get index file for this image file
   if (Nth != pNth) Fcontinue = 0;                                         //  index file not the same
   if (image_fcomp(file,pfile) <= 0) Fcontinue = 0;                        //  gallery file sequence

   if (Fcontinue) fid = sxrec_fid;
   else {
      if (sxrec_fid) fclose(sxrec_fid);
      snprintf(indexfile,200,"%s/index_%03d",index_dirk,Nth);              //  /.../.fotoxx/image_index/index_NNN
      fid = fopen(indexfile,"r");                                          //  open index_NNN file
      sxrec_fid = fid;
      if (! fid) goto file_err;
      *buff = 0;                                                           //  no prior data in buffer
   }

   if (! strnEqu(buff,"file: ",6) || strNeq(buff+6,file))                  //  file in buffer not my file
   {
      while (true)                                                         //  loop until my file found
      {
         pp = fgets_trim(buff,indexrecl,fid);                              //  read existing records
         if (! pp) break;                                                  //  EOF
         if (! strnEqu(pp,"file: ",6)) continue;                           //  to start of next file set
         nn = image_fcomp(pp+6,file);
         if (nn >= 0) break;                                               //  same or after my file
      }

      if (! pp || nn > 0) {                                                //  EOF or after, file not found
         fclose(fid);
         sxrec_fid = 0;
         return 1;
      }
   }

   pNth = Nth;                                                             //  set up for poss. continuation
   strcpy(pfile,file);
   
   sxrec.file = zstrdup(file);                                             //  build sxrec
   
   while (true)                                                            //  get recs following "file" rec.
   {
      pp = fgets_trim(buff,indexrecl,fid);
      if (! pp) break;
      if (strnEqu(pp,"file: ",6)) break;                                   //  ran into next file rec.

      if (strnEqu(pp,"date: ",6)) {                                        //  EXIF (photo) date
         pp += 6;
         pp2 = strField(pp,' ',1);                                         //  EXIF (photo) date, yyyymmddhhmmss
         if (pp2) strncpy0(sxrec.pdate,pp2,15);
         pp2 = strField(pp,' ',2);
         if (pp2) strncpy0(sxrec.fdate,pp2,15);                            //  file date, yyyymmddhhmmss          v.14.02
      }
      
      else if (strnEqu(pp,"stars: ",7)) {                                  //  rating, '0' to '5' stars
         sxrec.rating[0] = *(pp+7);
         sxrec.rating[1] = 0;
      }
      
      else if (strnEqu(pp,"size: ",6))                                     //  size, "NNNNxNNNN"
         strncpy0(sxrec.size,pp+6,15);

      else if (strnEqu(pp,"tags: ",6))                                     //  tags
         sxrec.tags = zstrdup(pp+6);

      else if (strnEqu(pp,"capt: ",6))                                     //  caption
         sxrec.capt = zstrdup(pp+6);

      else if (strnEqu(pp,"comms: ",7))                                    //  comments
         sxrec.comms = zstrdup(pp+7);

      else if (strnEqu(pp,"gtags: ",7))                                    //  geotags
         sxrec.gtags = zstrdup(pp+7);
   }

   if (! sxrec.pdate[0])                                                   //  supply defaults for missing items
      strcpy(sxrec.pdate,"null");

   if (! sxrec.rating[0])
      strcpy(sxrec.rating,"0");

   if (! sxrec.size[0]) 
      strcpy(sxrec.size,"null");

   if (! sxrec.tags) 
      sxrec.tags = zstrdup("null"tagdelimB);

   if (! sxrec.capt) 
      sxrec.capt = zstrdup("null");

   if (! sxrec.comms) 
      sxrec.comms = zstrdup("null");

   if (! sxrec.gtags) 
      sxrec.gtags = zstrdup("null^ null^ null^ null");

   return 0;

file_err:
   if (! logerr) printz("image index read error: %s \n",strerror(errno));
   logerr++;
   if (fid) fclose(fid);
   sxrec_fid = 0;
   return 3;
}


/**************************************************************************/

//  minimized version of get_sxrec() for use by gallery window.
//  fdate[16], pdate[16] and size[16] are provided by caller for returned data.
//  returns 0 if found, >0 if error (diagnosed)

int get_sxrec_min(cchar *file, char *fdate, char *pdate, char *size)
{
   FILE           *fid = 0;
   char           indexfile[200];
   static char    buff[indexrecl];
   static char    pfile[maxfcc];
   int            Nth, nn = 0, Fcontinue;
   cchar          *pp;
   static int     pNth, logerr = 0;

   strcpy(fdate,"");                                                       //  outputs = missing
   strcpy(pdate,"undated");
   strcpy(size,"");

   Fcontinue = 1;                                                          //  test if prior search can continue
   if (! sxrec_fid) Fcontinue = 0;                                         //  no prior search still open
   Nth = find_image_index_file(file);                                      //  get index file for this image file
   if (Nth != pNth) Fcontinue = 0;                                         //  index file not the same
   if (image_fcomp(file,pfile) <= 0) Fcontinue = 0;                        //  req. file not after prior file

   if (Fcontinue) fid = sxrec_fid;
   else {
      if (sxrec_fid) fclose(sxrec_fid);
      snprintf(indexfile,200,"%s/index_%03d",index_dirk,Nth);              //  /.../.fotoxx/image_index/index_NNN
      fid = fopen(indexfile,"r");                                          //  open index_NNN file
      if (! fid) goto file_err;
      sxrec_fid = fid;
      *buff = 0;                                                           //  no prior data in buffer
   }

   if (! strnEqu(buff,"file: ",6) || strNeq(buff+6,file))                  //  file in buffer not my file
   {
      while (true)                                                         //  loop until my file found
      {
         pp = fgets_trim(buff,indexrecl,fid);                              //  read existing records
         if (! pp) break;                                                  //  EOF
         if (! strnEqu(pp,"file: ",6)) continue;                           //  to start of next file set
         nn = image_fcomp(pp+6,file);
         if (nn >= 0) break;                                               //  same or after my file
      }

      if (! pp || nn > 0) {                                                //  EOF or after, file not found
         fclose(fid);
         sxrec_fid = 0;
         return 1;
      }
   }

   pNth = Nth;                                                             //  set up for poss. continuation
   strcpy(pfile,file);
   
   while (true)                                                            //  get recs following "file" rec.
   {
      pp = fgets_trim(buff,indexrecl,fid);
      if (! pp) break;
      if (strnEqu(pp,"file: ",6)) break;                                   //  ran into next file rec.

      if (strnEqu(pp,"date: ",6))                                          //  date record
      {
         pp = strField(buff,' ',2);
         if (pp) {
            if (*pp != 'n') strncpy0(pdate,pp,15);                         //  photo date unless "null"
            pp = strField(buff,' ',3);      
            if (pp) strncpy0(fdate,pp,15);                                 //  file date
         }
      }
      
      else if (strnEqu(pp,"size: ",6))                                     //  size, "NNNNxNNNN"
         strncpy0(size,pp+6,15);
   }

   return 0;

file_err:
   if (! logerr) printz("image index read error: %s \n",strerror(errno));
   logerr++;
   if (fid) fclose(fid);
   sxrec_fid = 0;
   return 3;
}


/**************************************************************************/

//  Add or update image index record for given image file.
//  If sxrec is null, delete index record.
//  Return 0 if success, +N if error.

int put_sxrec(sxrec_t *sxrec, cchar *file)
{
   FILE     *fid1 = 0, *fid2 = 0;
   char     indexfile[200], tempfile[200];
   char     buff[indexrecl];
   char     *pp;
   int      err, nn, Nth;
   int      filecount, Fcopy, Finsert, Finserted, Fnewfirst;
   STATB    statb;

   if (sxrec_fid) {                                                        //  stop get_sxrec()
      fclose(sxrec_fid);
      sxrec_fid = 0;
   }

   err = stat(file,&statb);                                                //  check image file exists
   if (err && sxrec) return 1;

   Nth = find_image_index_file(file);                                      //  construct index file
   snprintf(indexfile,200,"%s/index_%03d",index_dirk,Nth);                 //  /.../.fotoxx/image_index/index_NNN
   
   err = stat(indexfile,&statb);                                           //  check if missing
   if (err) {
      nn = creat(indexfile,0640);                                          //  create if needed
      if (nn < 0) goto file_err;
      close(nn);
   }

   fid1 = fopen(indexfile,"r");                                            //  open index_NNN file
   if (! fid1) goto file_err;

   strcpy(tempfile,indexfile);                                             //  temp image index file
   strcat(tempfile,"_temp");

   fid2 = fopen(tempfile,"w");                                             //  open for output
   if (! fid2) goto file_err;

   Finsert = Finserted = Fcopy = Fnewfirst = 0;
   filecount = 0;

   while (true)                                                            //  copy input to output
   {
      pp = fgets_trim(buff,indexrecl,fid1);                                //  read existing records

      if (pp && strnEqu(pp,"file: ",6))                                    //  start of a file record set
      {
         Finsert = Fcopy = Fnewfirst = 0;

         nn = strcasecmp(file,pp+6);                                       //  compare input file to index file
         if (nn == 0) nn = strcmp(file,pp+6);
         if (nn <= 0 && sxrec && ! Finserted) Finsert = 1;                 //  input <= index file, insert sxrec here
         if (nn != 0) Fcopy = 1;                                           //  input != index file, copy to output
         if (filecount == 0) {
            if (Finsert && nn < 0) Fnewfirst = 1;                          //  detect if first image file in index
            if (! Fcopy) Fnewfirst = 1;                                    //    file will be changed or deleted
         }
         filecount += Fcopy + Finsert;
      }

      if (! pp && sxrec && ! Finserted) {                                  //  input EOF, insert at the end
         Finsert = 1;
         if (filecount == 0) Fnewfirst = 1;
      }

      if (! pp && (Finserted || ! sxrec)) break;                           //  done

      if (Finsert && ! Finserted)
      {
         Finserted = 1;                                                    //  new index file recs. go here
         
         nn = fprintf(fid2,"file: %s\n",file);                             //  write new index file recs.
         if (! nn) goto file_err;

         compact_time(statb.st_mtime,sxrec->fdate);                        //  convert to "yyyymmddhhmmss"

         if (! sxrec->pdate[0]) strcpy(sxrec->pdate,"null");               //  EXIF (photo) date, yyyy:mm:dd

         nn = fprintf(fid2,"date: %s  %s\n",sxrec->pdate,sxrec->fdate);    //  photo and file date rec.
         if (! nn) goto file_err;

         if (sxrec->rating[0]) 
            nn = fprintf(fid2,"stars: %s\n",sxrec->rating);                //  rating rec.
         else nn = fprintf(fid2,"stars: 0\n"); 
         if (! nn) goto file_err;
         
         if (sxrec->size[0]) 
            nn = fprintf(fid2,"size: %s\n",sxrec->size);
         else nn = fprintf(fid2,"size: null\n");

         if (sxrec->tags) 
            nn = fprintf(fid2,"tags: %s\n",sxrec->tags);                   //  tags rec.
         else nn = fprintf(fid2,"tags: null"tagdelimB"\n");
         if (! nn) goto file_err;

         if (sxrec->capt) 
            nn = fprintf(fid2,"capt: %s\n",sxrec->capt);                   //  caption rec.
         else nn = fprintf(fid2,"capt: null\n");
         if (! nn) goto file_err;
         
         if (sxrec->comms) 
            nn = fprintf(fid2,"comms: %s\n",sxrec->comms);                 //  comments rec.
         else nn = fprintf(fid2,"comms: null\n");
         if (! nn) goto file_err;

         if (sxrec->gtags) 
            nn = fprintf(fid2,"gtags: %s\n",sxrec->gtags);                 //  geotags rec.
         else nn = fprintf(fid2,"gtags: null^ null^ null^ null\n");
         if (! nn) goto file_err;

         nn = fprintf(fid2,"\n");                                          //  EOL blank rec.
         if (! nn) goto file_err;
      }
      
      if (pp && Fcopy) {                                                   //  copy input to output
         nn = fprintf(fid2,"%s\n",pp);                                     //    unless replaced by sxrec
         if (! nn) goto file_err;
      }
   }

   err = fclose(fid1);                                                     //  close input file
   err = fclose(fid2);                                                     //  close output file
   fid1 = fid2 = 0;
   if (err) goto file_err;

   err = rename(tempfile,indexfile);                                       //  replace index file with temp file
   if (err) goto file_err;

   if (filecount > image_index_max)                                        //  if index file too big, split
      split_image_index_file(Nth);
   else if (Fnewfirst) 
      init_image_index_map(1);                                             //  update image index map

   return 0;

file_err:
   zmessLogACK(Mwin,"image index write error 3\n %s",strerror(errno));
   if (fid1) fclose(fid1);
   if (fid2) fclose(fid2);
   return 3;
}


/**************************************************************************/

//  Read image index files sequentially, return one index rec. per call.
//  Set ftf = 1 for first read, will be reset to 0.
//  Returns 0 if OK, 1 if EOF, 2 if error (diagnosed).
//  Returned sxrec_t data has allocated fields subject to zfree().

int read_sxrec_seq(sxrec_t &sxrec, int &ftf)
{
   char           indexfile[200];
   static FILE    *fid = 0;
   static char    buff[indexrecl];
   static int     Nth;
   cchar          *pp, *pp2;
   int            err;
   STATB          statb;
   
   if (ftf)                                                                //  initial call
   {
      ftf = 0;
      snprintf(indexfile,200,"%s/index_001",index_dirk);                   //  first index file
      fid = fopen(indexfile,"r");
      if (! fid) return 2;                                                 //  no index file ?
      Nth = 1;
      *buff = 0;
   }
   
   while (true)
   {
      if (! strnEqu(buff,"file: ",6))                                      //  next file rec. may be already there
      {
         while (true)                                                      //  get start of next record set
         {
            pp = fgets_trim(buff,indexrecl,fid);                                 
            if (pp && strnEqu(buff,"file: ",6)) break;
            if (pp) continue;
            fclose(fid);                                                   //  EOF, start next index file
            Nth++;
            snprintf(indexfile,200,"%s/index_%03d",index_dirk,Nth);
            fid = fopen(indexfile,"r");
            if (! fid) return 1;                                           //  no more, final EOF
         }
      }

      *buff = 0;                                                           //  no longer match "file: "
      err = stat(buff+6,&statb);                                           //  check image file exists
      if (err) continue;
      if (S_ISREG(statb.st_mode)) break;
   }

   memset(&sxrec,0,sizeof(sxrec));                                         //  clear record
   
   sxrec.file = zstrdup(buff+6);                                           //  image file name

   while (true)                                                            //  get recs following "file" rec.
   {
      pp = fgets_trim(buff,indexrecl,fid);
      if (! pp) break;

      if (strnEqu(pp,"file: ",6)) break;                                   //  ran into next file rec.

      else if (strnEqu(pp,"date: ",6))
      {
         pp += 6;
         pp2 = strField(pp,' ',1);                                         //  EXIF (photo) date, yyyymmddhhmmss
         if (pp2) strncpy0(sxrec.pdate,pp2,15);
         pp2 = strField(pp,' ',2);
         if (pp2) strncpy0(sxrec.fdate,pp2,15);                            //  file mod date, yyyymmddhhmmss      v.14.02
      }
      
      else if (strnEqu(pp,"stars: ",7)) {                                  //  rating, '0' to '5' stars
         sxrec.rating[0] = *(pp+7);
         sxrec.rating[1] = 0;
      }

      else if (strnEqu(pp,"size: ",6))                                     //  size, "NNNNxNNNN"
         strncpy0(sxrec.size,pp+6,15);

      else if (strnEqu(pp,"tags: ",6))                                     //  tags
         sxrec.tags = zstrdup(pp+6);

      else if (strnEqu(pp,"capt: ",6))                                     //  caption
         sxrec.capt = zstrdup(pp+6);

      else if (strnEqu(pp,"comms: ",7))                                    //  comments
         sxrec.comms = zstrdup(pp+7);

      else if (strnEqu(pp,"gtags: ",7))                                    //  geotags
         sxrec.gtags = zstrdup(pp+7);
   }

   if (! sxrec.pdate[0])                                                   //  supply defaults for missing items
      strcpy(sxrec.pdate,"null");
   
   if (! sxrec.rating[0])
      strcpy(sxrec.rating,"0");

   if (! sxrec.size[0])
      strcpy(sxrec.size,"null");

   if (! sxrec.tags) 
      sxrec.tags = zstrdup("null"tagdelimB);
   
   if (! sxrec.capt) 
      sxrec.capt = zstrdup("null");
   
   if (! sxrec.comms) 
      sxrec.comms = zstrdup("null");
   
   if (! sxrec.gtags) 
      sxrec.gtags = zstrdup("null^ null^ null^ null");
   
   return 0;
}


/**************************************************************************/

//  Write the image index files sequentially, 1 rec. per call
//  Set ftf = 1 for first call, will be reset to 0.
//  Set sxrec = 0 to close file after last write.
//  Returns 0 if OK, otherwise +N.
//  Used by index image files function.

int write_sxrec_seq(sxrec_t *sxrec, int &ftf)
{
   static int     Nth, filecount;
   static FILE    *fid = 0;
   char           indexfile[200], oldirk[200];
   int            nn, err;
   STATB          statb;

   if (sxrec_fid) {                                                        //  stop get_sxrec()
      fclose(sxrec_fid);
      sxrec_fid = 0;
   }

   if (ftf)                                                                //  first call 
   {
      ftf = 0;
      err = stat(index_dirk,&statb);
      if (! err && S_ISREG(statb.st_mode)) {                               //  rename old image_index
         strcpy(oldirk,index_dirk);
         strcat(oldirk,"_old");
         rename(index_dirk,oldirk);
      }

      err = stat(index_dirk,&statb);                                       //  create new image_index directory
      if (err) err = mkdir(index_dirk,0750);                               //    if not already there
      if (err) goto file_err;

      err = shell_quiet("rm -f -v %s/index_* > /dev/null",index_dirk);     //  delete all image index files
      Nth = 1;
      snprintf(indexfile,200,"%s/index_%03d",index_dirk,Nth);              //  create initial index file
      fid = fopen(indexfile,"w");
      if (! fid) goto file_err;
      filecount = 0;
   }
   
   if (! sxrec) {                                                          //  EOF call
      if (fid) {
         err = fclose(fid);
         fid = 0;
         if (err) goto file_err;
      }
      err = init_image_index_map(1);                                       //  initz. image index map
      return err;
   }
   
   err = stat(sxrec->file,&statb);                                         //  check image file exists
   if (err || ! S_ISREG(statb.st_mode)) {
      printz("file %s not found \n",sxrec->file);
      return 0;
   }

   compact_time(statb.st_mtime,sxrec->fdate);                              //  convert to "yyyymmddhhmmss"

   if (filecount > 0.8 * image_index_max) {                                //  if 80% max reached,
      err = fclose(fid);                                                   //    start new index file
      fid = 0;
      if (err) goto file_err;
      if (++Nth > 999) {
         zmessLogACK(Mwin,"too many image index files");
         return 2;
      }
      snprintf(indexfile,200,"%s/index_%03d",index_dirk,Nth);              //  open/write next index file 
      fid = fopen(indexfile,"w");
      if (! fid) goto file_err;
      filecount = 0;                                                       //  is empty
   }

   filecount++;                                                            //  new image file record set

   nn = fprintf(fid,"file: %s\n",sxrec->file);                             //  output: filename rec.
   if (! nn) goto file_err;

   if (! sxrec->pdate[0]) strcpy(sxrec->pdate,"null");                     //  EXIF (photo) date, yyyymmddhhmmss

   nn = fprintf(fid,"date: %s  %s\n",sxrec->pdate,sxrec->fdate);           //  photo and file date rec.
   if (! nn) goto file_err;

   if (sxrec->rating[0]) 
      nn = fprintf(fid,"stars: %c\n",sxrec->rating[0]);                    //  rating rec.
   else nn = fprintf(fid,"stars: 0\n"); 
   if (! nn) goto file_err;

   if (sxrec->size[0])
      nn = fprintf(fid,"size: %s\n",sxrec->size);
   else nn = fprintf(fid,"size: null\n");

   if (sxrec->tags) 
      nn = fprintf(fid,"tags: %s\n",sxrec->tags);                          //  tags rec.
   else nn = fprintf(fid,"tags: null"tagdelimB"\n");
   if (! nn) goto file_err;

   if (sxrec->capt) 
      nn = fprintf(fid,"capt: %s\n",sxrec->capt);                          //  caption rec.
   else nn = fprintf(fid,"capt: null\n");
   if (! nn) goto file_err;
   
   if (sxrec->comms) 
      nn = fprintf(fid,"comms: %s\n",sxrec->comms);                        //  comments rec.
   else nn = fprintf(fid,"comms: null\n");
   if (! nn) goto file_err;

   if (sxrec->gtags) 
      nn = fprintf(fid,"gtags: %s\n",sxrec->gtags);                        //  geotags rec.
   else nn = fprintf(fid,"gtags: null^ null^ null^ null\n");
   if (! nn) goto file_err;

   nn = fprintf(fid,"\n");                                                 //  EOL blank rec.
   if (! nn) goto file_err;

   return 0;

file_err:
   zmessLogACK(Mwin,"image index write error 4\n %s",strerror(errno));
   if (fid) fclose(fid);
   fid = 0;
   return 3;
}


//  file name compare in image index sequence
//  use case blind compare first, then normal compare as a tiebreaker

int image_fcomp(cchar *file1, cchar *file2)
{
   int      nn;
   nn = strcasecmp(file1,file2);                                           //  compare case blind
   if (nn != 0) return nn;
   nn = strcmp(file1,file2);                                               //  if equal, use utf8 compare
   return nn;
}



