/*----------------------------------------------------------------------
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: objects.c
 *---------------------------------------------------------------------*/

/*----------------------------------------------------------------------
 * Change History:
 *
 * 12/2001  John Stiles   Initial version.
 *
 *---------------------------------------------------------------------*/


/* Identify this file. */
#define OBJECTS_C    1

/*----------------------------------------------------------------------
 * Necessary include files
 *---------------------------------------------------------------------*/
#include <sys/types.h>
#include <string.h>                    /* strcat, strcpy                   */
#include <ctype.h>                     /* toupper                          */
#include <stdio.h>                     /* printf                           */
#include <stdlib.h>                    /* malloc, free                     */

#include "evmsn.h"

/*----------------------------------------------------------------------
 * Global Variables
 *---------------------------------------------------------------------*/

static char work_buffer[] = "                                                ";


/*----------------------------------------------------------------------
 * There are No Private Constants
 * There are No Private Type Definitions
 * There are No Private Global Variables.
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Local Functions
 *---------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * Public Functions
 *---------------------------------------------------------------------*/

int topmost_actions( object_handle_t topmost )
{
  int i, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES, removx = 99, dispext = 99,
      destrx = 99, xpandx = 99, shrnkx = 99, crevms = 99, crcompat = 99,
      treevx = 99, setinfx = 99;
  handle_object_info_t *object_info;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  if ( topmost != 0 ) {
    rc = evms_get_info( topmost, &object_info );
    if ( rc == 0 ) {
      sprintf( holdbuf, "%s:  %s", object_name_col, object_info->info.object.name );
      evms_free( object_info );
    }
    else {
      log_error( "%s: %s OBJECT %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
    }
  }
  print_string( XPOS_BOX3TEXT - 1, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );

  drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

  do {

    i = 0;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, object_createnew, strlen( object_createnew ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

    if ( topmost != 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, object_displaydetsel, strlen( object_displaydetsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      dispext = i;
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, object_treeviewsel, strlen( object_treeviewsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      treevx = i;

      if ( evms_can_remove( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_removesel, strlen( object_removesel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        removx = i;
      }
      if ( evms_can_destroy( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_destroysel, strlen( object_destroysel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        destrx = i;
      }
  //    if ( evms_can_move( topmost ) == 0 ) {
  //      i++;
  //      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, "Move topmost" );
  //    }
      if ( evms_can_expand( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_expandsel, strlen( object_expandsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        xpandx = i;
      }
      if ( evms_can_shrink( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_shrinksel, strlen( object_shrinksel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        shrnkx = i;
      }
      if ( evms_can_create_volume( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_createevmssel, strlen( object_createevmssel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crevms = i;
      }
      if ( evms_can_create_compatibility_volume( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_createcompatsel, strlen( object_createcompatsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crcompat = i;
      }
      if ( evms_can_set_info( topmost ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object_setinfsel, strlen( object_setinfsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        setinfx = i;
      }      
    }

    draw_status_line( action_selectmsg );

    keyval = key_scroller( &start_index, &select_index, i + 1, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        if ( select_index == removx ) {
          rc = remove_object( topmost, _( "Remove the Storage Object?" ));
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        else  if ( select_index == destrx ) {
          rc = destroy_object( topmost, _( "Destroy the Storage Object?" ));
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        else  if ( select_index == xpandx ) {
          rc = resize_object( topmost, EVMS_Task_Expand );
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        else  if ( select_index == shrnkx ) {
          rc = resize_object( topmost, EVMS_Task_Shrink );
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        else  if ( select_index == crevms ) {
          rc = object_create_evms( topmost );
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        else  if ( select_index == crcompat ) {
          rc = object_create_compat( topmost );
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        else  if ( select_index == setinfx ) {
          rc = set_object_info( topmost );
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }        
        else  if ( select_index == treevx ) {
          rc = tree_object_view( topmost, 0 );
        }
        else  if ( select_index == dispext ) {
          rc = object_extended_info( topmost );
        }
        else {
          sprintf( work_buffer, " %s ", object_createnew );
          // a plug_code of 0 means all features...
          rc = create_new_object( EVMS_FEATURE, 0, work_buffer );
          if ( rc == 0 )
            rc = KEY_OBJECTS;
        }
        break;

      default:
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int draw_topmost( int start_index, int *select_index, handle_array_t *top_objects )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object, *plugobject;
  char *holdbuf, *col1, *col2, *col3, *col4;

  if ( top_objects == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // topmost objects view

  draw_box2( title_availobjs );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( top_objects->count ) {
    BLANK_BUFFER( holdbuf, screen_cols + 2)
    memcpy( holdbuf, object_name_col, strlen( object_name_col ));
    col1 = holdbuf + X_OBJECT_SIZE;
    memcpy( col1, object_size_col, strlen( object_size_col ));
    col2 = holdbuf + X_OBJECT_PLUGIN;
    memcpy( col2, object_plugin_col, strlen( object_plugin_col ));
    col3 = holdbuf + X_OBJECT_DIRTY;
    memcpy( col3, object_dirty_col, strlen( object_dirty_col ));
    col4 = holdbuf + X_OBJECT_BIOSR;
    memcpy( col4, object_biosr_col, strlen( object_biosr_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > top_objects->count )
      item_count = top_objects->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( top_objects->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.object.name, strlen( object->info.object.name ));
        draw_sector_size( object->info.object.size, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));
        if ( object->info.object.plugin ) {
          rc = evms_get_info( object->info.object.plugin, &plugobject );
          if ( rc == 0 ) {
            memcpy( col2, plugobject->info.plugin.short_name, strlen( plugobject->info.plugin.short_name ));
            evms_free( plugobject );
          }
        }
        if( object->info.object.flags & SOFLAG_DIRTY )
          *col3 = ITEM_SELECTED_CHAR;
        if( object->info.object.flags & SOFLAG_BIOS_READABLE )
          *col4 = ITEM_SELECTED_CHAR;

        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.object.name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( object_selectmsg );
  }
  // else no topmost objects
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_availobjs, strlen( info_no_availobjs ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}


int region_actions( object_handle_t region )
{
  int i, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES, removx = 99, dispext = 99,
      destrx = 99, xpandx = 99, shrnkx = 99, crevms = 99, crcompat = 99,
      treevx = 99, rmconx = 99, setinfx = 99;
  handle_object_info_t *object_info;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  if ( region != 0 ) {
    rc = evms_get_info( region, &object_info );
    if ( rc == 0 ) {
      sprintf( holdbuf, "%s:  %s", region_name_col, object_info->info.region.name );
      evms_free( object_info );
    }
    else {
      log_error( "%s: %s REGION %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
    }
  }
  print_string( XPOS_BOX3TEXT - 1, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );

  drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

  do {

    i = 0;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, region_createnew, strlen( region_createnew ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

    if ( region != 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, region_displaydetsel, strlen( region_displaydetsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      dispext = i;
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, object_treeviewsel, strlen( object_treeviewsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      treevx = i;

      if ( evms_can_remove( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_removesel, strlen( region_removesel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        removx = i;
      }
      if ( evms_can_destroy( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_destroysel, strlen( region_destroysel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        destrx = i;
      }
  //    if ( evms_can_move( region ) == 0 ) {
  //      i++;
  //      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, "Move region" );
  //    }
      if ( evms_can_expand( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_expandsel, strlen( region_expandsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        xpandx = i;
      }
      if ( evms_can_shrink( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_shrinksel, strlen( region_shrinksel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        shrnkx = i;
      }
      if ( evms_can_create_volume( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_createevmssel, strlen( region_createevmssel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crevms = i;
      }
      if ( evms_can_create_compatibility_volume( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_createcompatsel, strlen( region_createcompatsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crcompat = i;
      }
      if ( evms_can_remove_from_container( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_contremove, strlen( region_contremove ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        rmconx = i;
      }      
      if ( evms_can_set_info( region ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, region_setinfsel, strlen( region_setinfsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        setinfx = i;
      }            
    }

    draw_status_line( action_selectmsg );

    keyval = key_scroller( &start_index, &select_index, i + 1, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        if ( select_index == removx ) {
          rc = remove_object( region, _( "Remove the Region?" ));
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        else  if ( select_index == destrx ) {
          rc = destroy_object( region, _( "Destroy the Region?" ));
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        else  if ( select_index == xpandx ) {
          rc = resize_object( region, EVMS_Task_Expand );
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        else  if ( select_index == shrnkx ) {
          rc = resize_object( region, EVMS_Task_Shrink );
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        else  if ( select_index == crevms ) {
          rc = object_create_evms( region );
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        else  if ( select_index == crcompat ) {
          rc = object_create_compat( region );
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        else  if ( select_index == setinfx ) {
          rc = set_object_info( region );
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }        
        else  if ( select_index == treevx ) {
          rc = tree_object_view( region, 0 );
        }
        else  if ( select_index == dispext ) {
          rc = object_extended_info( region );
        }
        else  if ( select_index == rmconx ) {
          rc = remove_object_from_container( region, _("Remove Region from Consuming Container?") );
          if ( rc == 0 )
            rc = KEY_REGIONS;  
        }        
        else {
          sprintf( work_buffer, " %s ", region_createnew );
          rc = create_new_object( EVMS_REGION_MANAGER, 0, work_buffer );
          if ( rc == 0 )
            rc = KEY_REGIONS;
        }
        break;

      default:
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int draw_regions( int start_index, int *select_index, handle_array_t *regions )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object, *plugobject;
  char *holdbuf, *col1, *col2, *col3, *col4, *col5;

  if ( regions == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // regions view

  draw_box2( title_regions );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( regions->count ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, region_name_col, strlen( region_name_col ));
    col1 = holdbuf + X_REGION_SIZE;
    memcpy( col1, region_size_col, strlen( region_size_col ));
    col2 = holdbuf + X_REGION_PLUGIN;
    memcpy( col2, region_plugin_col, strlen( region_plugin_col ));
    col3 = holdbuf + X_REGION_ALLOC;
    memcpy( col3, region_alloc_col, strlen( region_alloc_col ));
    col4 = holdbuf + X_REGION_DIRTY;
    memcpy( col4, region_dirty_col, strlen( region_dirty_col ));
    col5 = holdbuf + X_REGION_BIOSR;
    memcpy( col5, region_biosr_col, strlen( region_biosr_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > regions->count )
      item_count = regions->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( regions->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.region.name, strlen( object->info.region.name ));
        draw_sector_size( object->info.region.size, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));
        if ( object->info.region.plugin ) {
          rc = evms_get_info( object->info.region.plugin, &plugobject );
          if ( rc == 0 ) {
            memcpy( col2, plugobject->info.plugin.short_name, strlen( plugobject->info.plugin.short_name ));
            evms_free( plugobject );
          }
        }
        if( object->info.region.data_type == DATA_TYPE )
          *col3 = ITEM_SELECTED_CHAR;
        if( object->info.region.flags & SOFLAG_DIRTY )
          *col4 = ITEM_SELECTED_CHAR;
        if( object->info.region.flags & SOFLAG_BIOS_READABLE )
          *col5 = ITEM_SELECTED_CHAR;

        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.region.name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( region_selectmsg );
  }
  // else no regions
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_regions, strlen( info_no_regions ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}


int container_actions( object_handle_t container )
{
  int i, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES, removx = 99, dispext = 99,
      destrx = 99, xpandx = 99, shrnkx = 99, crevms = 99, crcompat = 99, setinfx = 99;
  handle_object_info_t *object_info;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  if ( container != 0 ) {
    rc = evms_get_info( container, &object_info );
    if ( rc == 0 ) {
      sprintf( holdbuf, "%s:  %s", contain_name_col, object_info->info.container.name );
      evms_free( object_info );
    }
    else {
      log_error( "%s: %s CONTAINER %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
    }
  }
  print_string( XPOS_BOX3TEXT - 1, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );

  drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

  do {

    i = 0;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, contain_createnew, strlen( contain_createnew ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

    if ( container != 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, contain_displaydetsel, strlen( contain_displaydetsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      dispext = i;

      if ( evms_can_remove( container ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_removesel, strlen( contain_removesel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        removx = i;
      }
      if ( evms_can_destroy( container ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_destroysel, strlen( contain_destroysel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        destrx = i;
      }
  //    if ( evms_can_move( container ) == 0 ) {
  //      i++;
  //      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, "Move container" );
  //    }
    /*  if ( evms_can_expand( container ) == 0 ) */{
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_expandsel, strlen( contain_expandsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        xpandx = i;
      }
      if ( evms_can_shrink( container ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_shrinksel, strlen( contain_shrinksel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        shrnkx = i;
      }
      if ( evms_can_create_volume( container ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_createevmssel, strlen( contain_createevmssel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crevms = i;
      }
      if ( evms_can_create_compatibility_volume( container ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_createcompatsel, strlen( contain_createcompatsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crcompat = i;
      }
      if ( evms_can_set_info( container ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, contain_setinfsel, strlen( contain_setinfsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        setinfx = i;
      }            
    }

    draw_status_line( action_selectmsg );

    keyval = key_scroller( &start_index, &select_index, i + 1, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        if ( select_index == removx ) {
          rc = remove_object( container, _( "Remove the Container?" ));
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        else  if ( select_index == destrx ) {
          rc = destroy_object( container, _( "Destroy the Container?" ));
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        else  if ( select_index == xpandx ) {
          rc = resize_object( container, EVMS_Task_Expand_Container );
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        else  if ( select_index == shrnkx ) {
          rc = resize_object( container, EVMS_Task_Shrink );
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        else  if ( select_index == crevms ) {
          rc = object_create_evms( container );
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        else  if ( select_index == crcompat ) {
          rc = object_create_compat( container );
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        else  if ( select_index == setinfx ) {
          rc = set_object_info( container );
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }        
        else  if ( select_index == dispext ) {
          rc = object_extended_info( container );
        }
        else {
          sprintf( work_buffer, " %s ", contain_createnew );
          rc = create_new_object( EVMS_REGION_MANAGER, 1, work_buffer );
          if ( rc == 0 )
            rc = KEY_CONTAINERS;
        }
        break;

      default:
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int draw_containers( int start_index, int *select_index, handle_array_t *containers )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object, *plugobject;
  char *holdbuf, *col1, *col2, *col3;

  if ( containers == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // containers view

  draw_box2( title_containers );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( containers->count ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, contain_name_col, strlen( contain_name_col ));
    col1 = holdbuf + X_CONTAIN_SIZE;
    memcpy( col1, contain_size_col, strlen( contain_size_col ));
    col2 = holdbuf + X_CONTAIN_PLUGIN;
    memcpy( col2, contain_plugin_col, strlen( contain_plugin_col ));
    col3 = holdbuf + X_CONTAIN_DIRTY;
    memcpy( col3, contain_dirty_col, strlen( contain_dirty_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > containers->count )
      item_count = containers->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( containers->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.container.name, strlen( object->info.container.name ));
        draw_sector_size( object->info.container.size, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));
        if ( object->info.container.plugin ) {
          rc = evms_get_info( object->info.container.plugin, &plugobject );
          if ( rc == 0 ) {
            memcpy( col2, plugobject->info.plugin.short_name, strlen( plugobject->info.plugin.short_name ));
            evms_free( plugobject );
          }
        }
        if( object->info.container.flags & SOFLAG_DIRTY )
          *col3 = ITEM_SELECTED_CHAR;

        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.container.name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( contain_selectmsg );
  }
  // else no containers
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_containers, strlen( info_no_containers ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}


int segment_actions( object_handle_t segment )
{
  int i, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES, removx = 99, dispext = 99,
      destrx = 99, xpandx = 99, shrnkx = 99, crevms = 99, crcompat = 99,
      treevx = 99, rmconx = 99, setinfx = 99;
  handle_object_info_t *object_info;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  if ( segment != 0 ) {
    rc = evms_get_info( segment, &object_info );
    if ( rc == 0 ) {
      sprintf( holdbuf, "%s:  %s", seg_name_col, object_info->info.segment.name );
      evms_free( object_info );
    }
    else {
      log_error( "%s: %s OBJECT %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
    }
  }
  print_string( XPOS_BOX3TEXT - 1, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );

  drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

  do {

    i = 0;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, seg_createnew, strlen( seg_createnew ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

    if ( segment != 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, seg_displaydetsel, strlen( seg_displaydetsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      dispext = i;
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, object_treeviewsel, strlen( object_treeviewsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      treevx = i;

      if ( evms_can_remove( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_removesel, strlen( seg_removesel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        removx = i;
      }
      if ( evms_can_destroy( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_destroysel, strlen( seg_destroysel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        destrx = i;
      }
  //    if ( evms_can_move( segment ) == 0 ) {
  //      i++;
  //      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, "Move segment" );
  //    }
      if ( evms_can_expand( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_expandsel, strlen( seg_expandsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        xpandx = i;
      }
      if ( evms_can_shrink( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_shrinksel, strlen( seg_shrinksel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        shrnkx = i;
      }
      if ( evms_can_create_volume( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_createevmssel, strlen( seg_createevmssel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crevms = i;
      }
      if ( evms_can_create_compatibility_volume( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_createcompatsel, strlen( seg_createcompatsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        crcompat = i;
      }
      if ( evms_can_remove_from_container( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_contremove, strlen( seg_contremove ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        rmconx = i;
      }      
      if ( evms_can_set_info( segment ) == 0 ) {
        i++;
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, seg_setinfsel, strlen( seg_setinfsel ));
        if ( select_index == i )
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
        setinfx = i;
      }                  
    }

    draw_status_line( action_selectmsg );

    keyval = key_scroller( &start_index, &select_index, i + 1, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        if ( select_index == removx ) {
          rc = remove_object( segment, _( "Remove the Segment?" ));
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        else  if ( select_index == destrx ) {
          rc = destroy_object( segment, _( "Destroy the Segment?" ));
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        else  if ( select_index == xpandx ) {
          rc = resize_object( segment, EVMS_Task_Expand );
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        else  if ( select_index == shrnkx ) {
          rc = resize_object( segment, EVMS_Task_Shrink );
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        else  if ( select_index == crevms ) {
          rc = object_create_evms( segment );
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        else  if ( select_index == crcompat ) {
          rc = object_create_compat( segment );
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        else  if ( select_index == treevx ) {
          rc = tree_object_view( segment, 0 );
        }
        else  if ( select_index == dispext ) {
          rc = object_extended_info( segment );
        }
        else  if ( select_index == rmconx ) {
          rc = remove_object_from_container( segment , _("Remove Segment from Consuming Container?"));
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }        
        else  if ( select_index == setinfx ) {
          rc = set_object_info( segment );
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }        
        else {
          sprintf( work_buffer, " %s ", seg_createnew );
          rc = create_new_object( EVMS_SEGMENT_MANAGER, 0, work_buffer );
          if ( rc == 0 )
            rc = KEY_SEGMENTS;
        }
        break;

      default:
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int draw_segments( int start_index, int *select_index, handle_array_t *segments )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object;
  char *holdbuf, *col1, *col2, *col3, *col4, *col5;

  if ( segments == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // segments view

  draw_box2( title_segments );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( segments->count ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, seg_name_col, strlen( seg_name_col ));
    col1 = holdbuf + X_SEG_SIZE;
    memcpy( col1, seg_size_col, strlen( seg_size_col ));
    col2 = holdbuf + X_SEG_STLBA;
    memcpy( col2, seg_stlba_col, strlen( seg_stlba_col ));
    col3 = holdbuf + X_SEG_TYPE;
    memcpy( col3, seg_type_col, strlen( seg_type_col ));
    col4 = holdbuf + X_SEG_DIRTY;
    memcpy( col4, seg_dirty_col, strlen( seg_dirty_col ));
    col5 = holdbuf + X_SEG_BIOSR;
    memcpy( col5, seg_biosr_col, strlen( seg_biosr_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > segments->count )
      item_count = segments->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( segments->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.segment.name, strlen( object->info.segment.name ));
        draw_sector_size( object->info.segment.size, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));
        sprintf( work_buffer, "%lld", object->info.segment.start );
        memcpy( col2, work_buffer, strlen( work_buffer ));
        switch ( object->info.region.data_type ) {
          case  META_DATA_TYPE  :
            memcpy( col3, seg_type_meta, strlen( seg_type_meta ));
            break;
          case  FREE_SPACE_TYPE  :
            memcpy( col3, seg_type_free, strlen( seg_type_free ));
            break;
          default:  //  DATA_TYPE
              memcpy( col3, seg_type_data, strlen( seg_type_data ));
            break;
        }
        if( object->info.segment.flags & SOFLAG_DIRTY )
          *col4 = ITEM_SELECTED_CHAR;
        if( object->info.segment.flags & SOFLAG_BIOS_READABLE )
          *col5 = ITEM_SELECTED_CHAR;

        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.segment.name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( seg_selectmsg );
  }
  // else no segments
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_segments, strlen( info_no_segments ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}


int disk_actions( object_handle_t disk )
{
  int i, rc = 0, keyval = 0, start_index = 0, select_index = 0,
      screen_cols, screen_lines = CALC_SCREEN3_LINES,
      crevms = 99, crcompat = 99, assgndsk = 99, treevx = 99, rmconx = 99,
      setinfx = 99;
  handle_object_info_t *object_info;
  char *holdbuf;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  BLANK_BUFFER( holdbuf, screen_cols )
  rc = evms_get_info( disk, &object_info );
  if ( rc == 0 ) {
    sprintf( holdbuf, "%s:  %s", disk_name_col, object_info->info.disk.name );
    print_string( XPOS_BOX3TEXT - 1, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );
    evms_free( object_info );
  }
  else {
    log_error( "%s: %s DISK %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
  }

  drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

  do {

    i = 0;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, disk_displaydetsel, strlen( disk_displaydetsel ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT, COLOR_TITLE, holdbuf );

    i++;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, object_treeviewsel, strlen( object_treeviewsel ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
    treevx = i;

    if ( evms_can_create_volume( disk ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, disk_createevmssel, strlen( disk_createevmssel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      crevms = i;
    }
    if ( evms_can_create_compatibility_volume( disk ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, disk_createcompatsel, strlen( disk_createcompatsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      crcompat = i;
    }
    if ( evms_can_remove_from_container( disk ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, disk_contremove, strlen( disk_contremove ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      rmconx = i;
    }
    if ( evms_can_set_info( disk ) == 0 ) {
      i++;
      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, disk_setinfsel, strlen( disk_setinfsel ));
      if ( select_index == i )
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
      else
        print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
      setinfx = i;
    }            
        
    i++;
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, disk_assignpluginsel, strlen( disk_assignpluginsel ));
    if ( select_index == i )
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_BAR, holdbuf );
    else
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT + i, COLOR_TITLE, holdbuf );
    assgndsk = i;

    draw_status_line( action_selectmsg );

    keyval = key_scroller( &start_index, &select_index, i + 1, screen_lines );

    switch ( keyval ) {

      case KEY_ESC :
        break;

      case KEY_F1  :
      case '1'     :
        draw_help_panel( 5 );
        keyval = KEY_CURSORACT;
        break;

      case KEY_ENTER :
        if ( select_index == crevms ) {
          rc = object_create_evms( disk );
          if ( rc == 0 )
            rc = KEY_DISKS;
        }
        else  if ( select_index == crcompat ) {
          rc = object_create_compat( disk );
          if ( rc == 0 )
            rc = KEY_DISKS;
        }
        else  if ( select_index == rmconx ) {
          rc = remove_object_from_container( disk, _( "Remove Disk from Consuming Container?" ) );
          if ( rc == 0 )
            rc = KEY_DISKS;
        }        
        else  if ( select_index == setinfx ) {
          rc = set_object_info( disk );
          if ( rc == 0 )
            rc = KEY_DISKS;
        }        
        else  if ( select_index == assgndsk ) {
          sprintf( work_buffer, " %s ", disk_assignpluginsel );
          rc = assign_object( EVMS_SEGMENT_MANAGER, 0, work_buffer );
          if ( rc == 0 )
            rc = KEY_DISKS;
        }
        else  if ( select_index == treevx ) {
          rc = tree_object_view( disk, 1 );
        }
        else {
          rc = object_extended_info( disk );
        }
        break;

      default:
        break;
    }

  }  while ( keyval == KEY_CURSORACT );

  free( holdbuf );
  return rc;
}


int draw_disks( int start_index, int *select_index, handle_array_t *disks )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object;
  char *holdbuf, *col1, *col2, *col3, *col4, *col5;

  if ( disks == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // disks view

  draw_box2( title_disks );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( disks->count ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, disk_name_col, strlen( disk_name_col ));
    col1 = holdbuf + X_DISK_SIZE;
    memcpy( col1, disk_size_col, strlen( disk_size_col ));
    col2 = holdbuf + X_DISK_GEO;
    memcpy( col2, disk_geo_col, strlen( disk_geo_col ));
    col3 = holdbuf + X_DISK_DIRTY;
    memcpy( col3, disk_dirty_col, strlen( disk_dirty_col ));
    col4 = holdbuf + X_DISK_ERRORS;
    memcpy( col4, disk_error_col, strlen( disk_error_col ));
    col5 = holdbuf + X_DISK_CORRUPT;
    memcpy( col5, disk_corrupt_col, strlen( disk_corrupt_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > disks->count )
      item_count = disks->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( disks->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.disk.name, strlen( object->info.disk.name ));
        draw_sector_size( object->info.disk.size, work_buffer );
        memcpy( col1, work_buffer, strlen( work_buffer ));
        sprintf( work_buffer, "(%lld,%d,%d)", object->info.disk.geometry.cylinders,
                                              object->info.disk.geometry.heads,
                                              object->info.disk.geometry.sectors_per_track );
        memcpy( col2, work_buffer, strlen( work_buffer ));

        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.disk.name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( disk_selectmsg );
  }
  // else no disks
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_disks, strlen( info_no_disks ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}


int plugin_information( object_handle_t plugin )
{
  int i, j, rc = 0, screen_cols;
  extended_info_array_t *extended_info;
  char *holdbuf, *col1;
  size_t  title_size;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    draw_status_line( "malloc error." );
    return 0;
  }

  rc = evms_get_extended_info( plugin, NULL, &extended_info );
  if ( rc == 0 ) {
    log_debug( "%s: %s %i %s\n", __FUNCTION__, info_msg_get_extinfo, extended_info->count, info_msg_getitems );
    if ( extended_info->count ) {

      for ( i = 0, j = YPOS_BOX3TOP + 2; i < extended_info->count; i++, j++ ) {
        log_debug( "%s: ExtInfo item %i is called %s\n", __FUNCTION__, i, extended_info->info[i].title );
        title_size = strlen( extended_info->info[i].title );
        col1 = holdbuf + title_size + 3;
        sprintf( holdbuf, "%s:  ", extended_info->info[i].title );
        draw_value_type( extended_info->info[i].type, extended_info->info[i].format,
                         &( extended_info->info[i].value ), screen_cols - title_size - 3, col1 );
        print_string( XPOS_BOX3TEXT, j, COLOR_TITLE, holdbuf );
      }
      draw_status_line( ext_info_when_done );
    }
    else {
      draw_status_line( info_no_ext_info );
    }
    evms_free( extended_info );
  }
  else {
    sprintf( work_buffer, "%s %i", err_msg_get_extinfo, rc );
    log_error( "%s: %s\n", __FUNCTION__, err_msg_get_info, work_buffer );
    draw_status_line( work_buffer );
  }
  i = get_inputc();

  free( holdbuf );
  return 0;
}


int draw_plugins( int start_index, int *select_index, handle_array_t *plugins )
{
  int i, j, rc = 0, item_count, screen_cols,
      screen_lines = CALC_SCREEN2_LINES;
  handle_object_info_t *object;
  char *holdbuf, *col2, *col3, *col4, *col5;

  if ( plugins == NULL )                               /* nothing to do*/
    return 1;

  screen_cols = GL_screen_cols - XPOS_BOX2TEXT - XPOS_BOX2RIGHT - 2;

  holdbuf = malloc( screen_cols + 2 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return 1;
  }

  // plugin view

  draw_box2( title_plugins );

  drawhline( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 1, screen_cols );

  if ( plugins->count ) {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, plug_name_col, strlen( plug_name_col ));
    col2 = holdbuf + X_PLUG_ID;
    memcpy( col2, plug_id_col, strlen( plug_id_col ));
    col3 = holdbuf + X_PLUG_VERS;
    memcpy( col3, plug_vers_col, strlen( plug_vers_col ));
    col4 = holdbuf + X_PLUG_DESC;
    memcpy( col4, plug_desc_col, strlen( plug_desc_col ));
    col5 = holdbuf + X_PLUG_OEM;
    memcpy( col5, plug_oem_col, strlen( plug_oem_col ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT - 2, COLOR_TITLE, holdbuf );

    if ( start_index + screen_lines > plugins->count )
      item_count = plugins->count - start_index + YPOS_BOX2TEXT;
    else
      item_count = screen_lines - 1 + YPOS_BOX2TEXT;

    for ( i = YPOS_BOX2TEXT, j = start_index; i < item_count; i++, j++ ) {

      rc = evms_get_info( plugins->handle[j], &object );
      if ( rc == 0 ) {
        BLANK_BUFFER( holdbuf, screen_cols )
        memcpy( holdbuf, object->info.plugin.short_name, strlen( object->info.plugin.short_name ));
        sprintf( work_buffer, "%X", object->info.plugin.id );
        memcpy( col2, work_buffer, strlen( work_buffer ));
        sprintf( work_buffer, " %d.%d.%d ", object->info.plugin.version.major,
                                            object->info.plugin.version.minor,
                                            object->info.plugin.version.patchlevel );
        memcpy( col3, work_buffer, strlen( work_buffer ));
        memcpy( col4, object->info.plugin.long_name, strlen( object->info.plugin.long_name ));
        memcpy( col5, object->info.plugin.oem_name, strlen( object->info.plugin.oem_name ));

        if ( j == *select_index )
          print_string( XPOS_BOX2TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX2TEXT, i, COLOR_TITLE, holdbuf );

        log_debug( "%s: entry %i is: %s \n", __FUNCTION__, i, object->info.plugin.short_name );
        evms_free( object );
      }
      else {
        log_error( "%s: %s %i %s %i.\n", __FUNCTION__, err_msg_get_info, i, err_msg_rc, rc );
      }
    }
    draw_status_line( plugins_selectmsg );
  }
  // else no plugins
  else {
    BLANK_BUFFER( holdbuf, screen_cols )
    memcpy( holdbuf, info_no_plugins, strlen( info_no_plugins ));
    print_string( XPOS_BOX2TEXT, YPOS_BOX2TEXT, COLOR_BAR, holdbuf );
    draw_status_line( NULL );                        /* redraw status line.*/
    free( holdbuf );
    return 1;
  }

  free( holdbuf );
  return 0;
}


int tree_object_view( object_handle_t object, int need_parents )
{
  int i, j, rc = 0, keyval = 0, start_index = 0, select_index = 0, child_count,
      item_count, screen_cols, screen_lines = CALC_SCREEN3_LINES;
  handle_object_info_t *object_info, *child_object_info, *plugobject;
  char *holdbuf, *col1, *col2;

  screen_cols = GL_screen_cols - XPOS_BOX3TEXT - XPOS_BOX3RIGHT - 2;

  holdbuf = malloc( screen_cols + 4 );
  if ( !holdbuf ) {
    log_error( "%s: %s\n", __FUNCTION__, err_msg_malloc_blanks );
    return ENOMEM;
  }

  rc = evms_get_info( object, &object_info );
  if ( rc == 0 ) {

    do {
      draw_box3( title_treeview );

      BLANK_BUFFER( holdbuf, screen_cols + 2 )
      for ( i = YPOS_BOX3TOP + 1; i < YPOS_BOX3TOP + 2 + screen_lines; i++ )
        print_string( XPOS_BOX3TEXT - 1, i, COLOR_DEFAULT, holdbuf );

      BLANK_BUFFER( holdbuf, screen_cols )
      memcpy( holdbuf, object_name_col, strlen( object_name_col ));
      col1 = holdbuf + X_OBJECT_SIZE;
      memcpy( col1, object_size_col, strlen( object_size_col ));
      col2 = holdbuf + X_OBJECT_PLUGIN;
      memcpy( col2, object_plugin_col, strlen( object_plugin_col ));
      print_string( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 2, COLOR_TITLE, holdbuf );

      drawhline( XPOS_BOX3TEXT, YPOS_BOX3TEXT - 1, screen_cols );

      if ( need_parents )
        child_count = object_info->info.object.parent_objects->count;
      else
        child_count = object_info->info.object.child_objects->count;

      if ( start_index + screen_lines - 1 > child_count )
        item_count = child_count - start_index + YPOS_BOX3TEXT + 1;
      else
        item_count = screen_lines - 1 + YPOS_BOX3TEXT;

      for ( i = YPOS_BOX3TEXT, j = start_index; i < item_count; i++, j++ ) {

        BLANK_BUFFER( holdbuf, screen_cols )
        if ( j == 0 ) {
          memcpy( holdbuf, object_info->info.object.name, strlen( object_info->info.object.name ));
          draw_sector_size( object_info->info.object.size, work_buffer );
          memcpy( col1, work_buffer, strlen( work_buffer ));
          if ( object_info->info.object.plugin ) {
            rc = evms_get_info( object_info->info.object.plugin, &plugobject );
            if ( rc == 0 ) {
              memcpy( col2, plugobject->info.plugin.short_name, strlen( plugobject->info.plugin.short_name ));
              evms_free( plugobject );
            }
          }
        }
        else {
          if ( need_parents )
            rc = evms_get_info( object_info->info.object.parent_objects->handle[j - 1], &child_object_info );
          else
            rc = evms_get_info( object_info->info.object.child_objects->handle[j - 1], &child_object_info );
          if ( rc == 0 ) {
            memcpy( holdbuf + 2, child_object_info->info.object.name, strlen( child_object_info->info.object.name ));
            draw_sector_size( child_object_info->info.object.size, work_buffer );
            memcpy( col1, work_buffer, strlen( work_buffer ));
            if ( child_object_info->info.object.plugin ) {
              rc = evms_get_info( child_object_info->info.object.plugin, &plugobject );
              if ( rc == 0 ) {
                memcpy( col2, plugobject->info.plugin.short_name, strlen( plugobject->info.plugin.short_name ));
                evms_free( plugobject );
              }
            }
            evms_free( child_object_info );
          }
        }

        if ( j == select_index )
          print_string( XPOS_BOX3TEXT, i, COLOR_BAR, holdbuf );
        else
          print_string( XPOS_BOX3TEXT, i, COLOR_TITLE, holdbuf );
      }

      draw_status_line( tree_selectmsg );

      keyval = key_scroller( &start_index, &select_index, child_count + 1, screen_lines - 2 );

      switch ( keyval ) {

        case KEY_ESC :
          break;

        case KEY_F1  :
        case '1'     :
          draw_help_panel( 5 );
          keyval = KEY_CURSORACT;
          break;

        case KEY_ENTER :
          if ( select_index == 0 )
            rc = object_extended_info( object );
          else {
            if ( need_parents )
              rc = object_extended_info( object_info->info.object.parent_objects->handle[select_index - 1] );
            else
              rc = object_extended_info( object_info->info.object.child_objects->handle[select_index - 1] );
          }

          keyval = KEY_CURSORACT;
          break;

        default:
          break;
      }

    }  while ( keyval == KEY_CURSORACT );
    evms_free( object_info );
  }
  else {
    log_error( "%s: %s OBJECT %s %i.\n", __FUNCTION__, err_msg_get_info, err_msg_rc, rc );
    draw_status_line( _( "Could not get Object info, press any key to continue." ));
    get_inputc();
  }

  free( holdbuf );
  return 0;
}

