/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: revert.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "revert.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "help.h"
/*
 *
 *   void on_revert_thing_button_clicked (GtkButton *, gchar *, gchar *)
 *
 *   Description:
 *      This routine initiates the revert API call using the handle
 *      of the thing associated with the last row selected. It then
 *      creates and displays the results popup which destroys both
 *      windows when dismissed.
 * 
 *   Entry:
 *      button      - address of the GtkButton widget
 *      error_msg   - string to use for additional error message on failure
 *      success_msg - string to use as additional message on success
 *
 *   Exit:
 *      evms_remove() is invoked on handle and results window is displayed afterwards
 *
 */
void on_revert_thing_button_clicked (GtkButton *button, gchar *error_msg, gchar *success_msg)
{
    GtkCList       *clist;
    object_handle_t handle;
        
    clist = GTK_CLIST (lookup_widget (GTK_WIDGET (button), "selection_window_clist"));

    handle = GPOINTER_TO_UINT (get_single_select_current_row_data (clist));

    if (handle != 0)
    {
        gint rc;

        rc = evms_remove (handle);

        display_selection_window_results (GTK_WIDGET (button), rc, error_msg, success_msg);
    }
}

/*
 *
 *   void on_revert_thing_clist_realize (GtkWidget *, object_type_t)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes or storage objects that can be removed from
 *      the top of storage object stack.
 * 
 *   Entry:
 *      widget - address of the selections GtkCList widget
 *      type   - specifies whether to populate the list with volumes or objects
 *
 *   Exit:
 *      Selection list populated with volumes or storage objects
 *
 */
void on_revert_thing_clist_realize (GtkWidget *widget, object_type_t type)
{
    gint            rc;
    GtkCList       *clist = GTK_CLIST (widget);
    handle_array_t *things;

    if (type == VOLUME)
        rc = evms_get_volume_list (0, &things);
    else
        rc = evms_get_object_list (type, DATA_TYPE, 0, TOPMOST, &things);

    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_xxxxx_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint    i;
        gboolean is_selected = (things->count == 1);

        set_selection_window_clist_column_titles (clist, _("Size"), 
                                                  make_object_type_readable_string (type), NULL);
        
        for (i=0; i < things->count; i++)
        {
            if (evms_can_remove (things->handle[i]) == 0)
                add_thing_to_selection_list (clist, things->handle[i], is_selected);
        }

        if (clist->rows == 1)
            gtk_clist_select_row (clist, 0, 0);
        
        evms_free (things);
    }
}

/*
 *
 *   void on_revert_thing_generic_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates reverting a "thing" (volume or feature).
 *      The handle to the thing is received as the user data. We 
 *      select the proper messages for the results screen and invoke 
 *      the routine that actually reverts the thing.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_revert_thing_generic_button_clicked (GtkButton *button, gpointer user_data)
{
    object_type_t type;

    if (evms_get_handle_object_type (GPOINTER_TO_UINT (user_data), &type) == SUCCESS)
    {
        gchar *error_msg;
        gchar *success_msg;

        switch (type)
        {
            case VOLUME:
                error_msg   = _("An error was encountered attempting to revert the volume.");
                success_msg = _("The volume was successfully reverted to a storage object.");
                break;

            case EVMS_OBJECT:
                error_msg   = _("An error was encountered attempting to remove the toplevel feature.");
                success_msg = _("The toplevel feature was successfully removed from the storage object stack.");
                break;

            case REGION:
                error_msg   = _("An error was encountered attempting to remove the storage region.");
                success_msg = _("The storage region was successfully removed from the storage object stack.");
                break;
            
            case SEGMENT:
                error_msg   = _("An error was encountered attempting to remove the disk segment.");
                success_msg = _("The disk segment was successfully removed from the storage object stack.");
                break;
            
            default:
                error_msg   = _("The remove operation encountered an error.");
                success_msg = _("The remove operation completed successfully.");
                break;
        }
   
        on_revert_thing_button_clicked (button, error_msg, success_msg);
    }
}

/*
 *
 *   void on_revert_thing_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to revert and
 *      allow the user to confirm the operation on the thing.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_revert_thing_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    object_type_t type;

    if (evms_get_handle_object_type (GPOINTER_TO_UINT (user_data), &type) == SUCCESS)
    {
        gchar     *window_title;
        gchar     *help_text;
        GtkWidget *window;

        switch (type)
        {
            case VOLUME:
                window_title = _("Remove Volume from Storage Object");
                help_text = remove_volume_help_text;
                break;

            case EVMS_OBJECT:
                window_title = _("Remove Feature from Storage Object");
                help_text = remove_feature_help_text;            
                break;

            case REGION:
                window_title = _("Remove Storage Region from Storage Object");
                help_text = remove_region_help_text;            
                break;

            case SEGMENT:
                window_title = _("Remove Disk Segment from Storage Object");
                help_text = remove_segment_help_text;            
                break;
            
            default:
                window_title = _("Remove");
                help_text = NULL;
                break;
        }

        window = create_standard_selection_window (window_title,
                                                   _("Remove"),
                                                   help_text,
                                                   add_thing_as_selected_list_item,
                                                   on_revert_thing_generic_button_clicked,
                                                   NULL, NULL, NULL, NULL, user_data);

        gtk_widget_show (window);
    }
}
