/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: snapshot.c
 *
 *
 *   This module provides support for snapshotting volumes.
 *   This function is implemented as an associative feature so that it can be applied
 *   at any time to an existing volume.  The 'snapshot' volume is created from freespace
 *   and is an EVMS volume.
 *
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <unistd.h>
#include "snapshot.h"
#include <sys/ioctl.h>
#include <linux/evms/evms_snapshot.h>
#include <linux/genhd.h>

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/
#define READ(object, start, len, buffer)  object->plugin->functions.plugin->read(object, start, len, buffer)
#define WRITE(object, start, len, buffer) object->plugin->functions.plugin->write(object, start, len, buffer)
#define KILLSECTORS(object, start, len)   object->plugin->functions.plugin->add_sectors_to_kill_list(object, start, len)
#define DELETE(object) object->plugin->functions.plugin->delete(object)
#define CANDELETE(object) object->plugin->functions.plugin->can_delete(object)
#define SET_STRING_FIELD(a,b) a = EngFncs->engine_alloc( strlen(b)+1 );if (a ) { strcpy(a, b); } else { rc = ENOMEM; RETURN(rc);}
#define MESSAGE(msg, args...)		EngFncs->user_message(my_plugin_rec, NULL, NULL, msg, ##args)

#define EVMS_IBM_SNAPSHOT_FEATURE       104
static engine_mode_t                    engine_mode;
static engine_functions_t             *EngFncs=NULL;

/* Macro: SET_POWER2_LIST
 */
// Set a contraint list that increments each item by a power of 2.
// b is the starting value, c is ending value (both must be power of 2).
#define SET_POWER2_LIST(a, b, c) \
{ \
	if ( ((b) & ((b)-1)) || ((c) & ((c)-1)) ) { \
		RETURN(EINVAL); \
	} \
	(a) = EngFncs->engine_alloc(calc_log2(c)*sizeof(value_t)+1); \
	if (a) { \
		long i = b; \
		(a)->count = 0; \
		while ( i <= c ) { \
			(a)->value[(a)->count].ui32 = i; \
			(a)->count++; \
			i <<= 1; \
		} \
	} else { \
		RETURN(ENOMEM); \
	} \
}

/*
 * Function: calc_log2
 *
 *	Calculate the log-base2 of the argument
 */
static inline long calc_log2( long arg )
{
	long	result = -1;
	long	tmp;

	if ( arg ) {
		tmp = arg;
		result++;
		while ( ! (tmp & 1) ) {
			result++;
			tmp >>= 1;
		}
		if ( tmp != 1 ) {
			// arg isn't a power of 2!
			result = -2;
		}
	}
	RETURN(result);
}



/*
 * Function: convert_metadata
 *
 *	Performs endian conversion on metadata sector.
 */
static int convert_metadata( snapshot_metadata_t * metadata ){

	LOG_ENTRY;
	metadata->chunk_size = DISK_TO_CPU32(metadata->chunk_size);
	metadata->lba_of_COW_table = DISK_TO_CPU64(metadata->lba_of_COW_table);
	metadata->lba_of_first_chunk = DISK_TO_CPU64(metadata->lba_of_first_chunk);
	metadata->original_size = DISK_TO_CPU64(metadata->original_size);
        metadata->signature = DISK_TO_CPU32(metadata->signature);
	metadata->total_chunks = DISK_TO_CPU32(metadata->total_chunks);
        metadata->version.major = DISK_TO_CPU32(metadata->version.major);
        metadata->version.minor = DISK_TO_CPU32(metadata->version.minor);
        metadata->version.patchlevel = DISK_TO_CPU32(metadata->version.patchlevel);
	metadata->CRC = DISK_TO_CPU32(metadata->CRC);

	RETURN(0);
}

/*
 * Function: remove_snapshot_from_chain
 *
 *	Remove the specified snapshot volume from its original's chain of
 *	snapshots.
 */
static int remove_snapshot_from_chain( snapshot_volume_t * snap_volume )
{
	snapshot_volume_t * org_volume = snap_volume->snapshot_org;

	LOG_ENTRY;
	if ( org_volume ) {
		while ( org_volume->snapshot_next && org_volume->snapshot_next != snap_volume ) {
			org_volume = org_volume->snapshot_next;
		}
		if ( org_volume->snapshot_next ) {
			org_volume->snapshot_next = org_volume->snapshot_next->snapshot_next;
		}
	}
	snap_volume->snapshot_org = NULL;
	snap_volume->snapshot_next = NULL;
	RETURN(0);
}

// passed in the pointer to the snapshot object.  Free the snapshot object and RETURN it's child.
// Also, if this is the last snapshot of the original, delete the original object, else just
// remove the snapshot_volume structure from the chain on the original.

static int delete_objects(storage_object_t * snapshot_object, storage_object_t ** child){

	storage_object_t *  snapshot_child;
        storage_object_t *  original_object;
	storage_object_t * original_child;
	snapshot_volume_t * snapshot_volume;
	snapshot_volume_t * original_volume;
	int rc=0;

	LOG_ENTRY;
	rc = ExtractObject(snapshot_object->child_objects,
			sizeof(storage_object_t),
			EVMS_OBJECT_TAG,
			NULL,			   // get current
			(void**)&snapshot_child);
	if (rc) {
		LOG_ERROR("DLIST error %d die die die....\n",rc);
		RETURN(rc);
	}

	snapshot_volume = snapshot_object->private_data;
	original_volume = snapshot_volume->snapshot_org;
	original_object = original_volume->object;

	remove_snapshot_from_chain(snapshot_volume);
	// if this was the last snapshot, remove the object from the original stack.
	if (original_volume->snapshot_next == NULL) {
		rc = ExtractObject(original_object->child_objects,
				sizeof(storage_object_t),
				EVMS_OBJECT_TAG,
				NULL,			   // get current
				(void**)&original_child);
		if (rc) {
			LOG_ERROR("DLIST error %d die die die....\n",rc);
			RETURN(rc);
		}
		// fix up the object stack on the original now that we are gone.
		EngFncs->unregister_name(original_object->name);
		rc = DeleteObject(original_child->parent_objects, original_object);
		if (rc) {
			LOG_ERROR("DLIST error die die die....\n");
			RETURN(rc);
		}
		original_child->volume->object = original_child;

		free(original_volume);
		EngFncs->free_evms_object(original_object);
	}
	

	// kill metadata header and feature header
	rc = KILLSECTORS(snapshot_child, snapshot_child->size - 3, 3);
	if (rc) {
		LOG_ERROR("Error deleting metadata sectors rc = %d....\n",rc);
	}
	// kill first cow sector to be safe
	rc = KILLSECTORS(snapshot_child, snapshot_volume->meta_data.lba_of_COW_table, 1);
	if (rc) {
		LOG_ERROR("Error deleting metadata sectors rc = %d....\n",rc);
	}

	free(snapshot_volume);

	EngFncs->unregister_name(snapshot_object->name);
	rc = DeleteObject(snapshot_child->parent_objects, snapshot_object);
	EngFncs->free_evms_object(snapshot_object);

	EngFncs->engine_free(snapshot_child->feature_header);
	snapshot_child->feature_header = NULL;
	*child = snapshot_child;  //RETURN the snapshot child object

	RETURN(rc);
}


// allocate the objects for both the new snapshot and the object to be inserted
// into the stack of the original volume. If we already have a snapshot on the
// original, then we do not need to create a new node, just RETURN the already
// created one.

static int allocate_snap_objects(storage_object_t * * snap_object,
				storage_object_t  * snapshot_child,
				storage_object_t * * original_object,
				storage_object_t  * original_child,
				snapshot_metadata_t * metadata){

	snapshot_volume_t * snapshot_volume;
	snapshot_volume_t * original_volume;
	int rc, tmp;

	LOG_ENTRY;
	rc = EngFncs->allocate_evms_object(NULL, snap_object);
	if (rc) {
		MESSAGE("Error allocating memory = %d....\n",rc);
		RETURN(rc);
	}
	(*snap_object)->private_data = EngFncs->engine_alloc(sizeof(snapshot_volume_t));
	if (!(*snap_object)->private_data ) {
		MESSAGE("Error allocating memory = %d....\n",rc);
		EngFncs->free_evms_object(*snap_object);
		RETURN(ENOMEM);
	}

	snapshot_volume = (snapshot_volume_t *)(*snap_object)->private_data;
	snapshot_volume->meta_data =  *metadata;	// copy the metadata
	snapshot_volume->child_object = snapshot_child;
	snapshot_volume->object = *snap_object;

	// if we do not already have a snapshot on this original
	if (original_child->plugin != my_plugin_rec) {
		rc = EngFncs->allocate_evms_object(NULL, original_object);
		if (rc) {
			MESSAGE("Error allocating memory = %d....\n",rc);
			EngFncs->engine_free((*snap_object)->private_data );
			EngFncs->free_evms_object(*snap_object);
			RETURN(rc);
		}
		(*original_object)->private_data = EngFncs->engine_alloc(sizeof(snapshot_volume_t));
		if (!(*original_object)->private_data) {
			MESSAGE("Error allocating memory = %d....\n",rc);
			EngFncs->engine_free((*snap_object)->private_data );
			EngFncs->free_evms_object(*snap_object);
			EngFncs->free_evms_object(*original_object);
			rc = ENOMEM;
			RETURN(rc);
		}
		original_volume = (snapshot_volume_t *)(*original_object)->private_data;
		original_volume->meta_data =  *metadata;	// copy the metadata
		original_volume->child_object = original_child;
		original_volume->object = *original_object;
		InsertObject((*original_object)->child_objects, sizeof(storage_object_t), original_child, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&tmp);
		original_volume->snapshot_org = NULL;
		(*original_object)->flags |= SOFLAG_MUST_BE_TOP;
		(*original_object)->plugin = my_plugin_rec;
		(*original_object)->size = original_child->size;
		(*original_object)->volume = original_child->volume;
		(*original_object)->volume->object = *original_object;
		// copy volume name of original, minus the /dev/evms prefix.
		if (strstr(original_child->volume->name, EVMS_DEV_NODE_PATH)) {
			strcpy((*original_object)->name, original_child->volume->name+strlen(EVMS_DEV_NODE_PATH));
		} else{ // volume name doesn't start with /dev/evms, oh well.  just copy it
			strcpy((*original_object)->name, original_child->volume->name);
		}
		strncat((*original_object)->name,"#snapshot#",EVMS_NAME_SIZE-strlen((*original_object)->name));
		if (EngFncs->register_name((*original_object)->name)){
			LOG_WARNING("Error registering object %s, continuing \n",(*original_object)->name, rc );
		}
		InsertObject(original_child->parent_objects, sizeof(storage_object_t), *original_object, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&tmp);
	}else{
		*original_object = original_child;  // we are already in the stack
		original_volume = (snapshot_volume_t *)(*original_object)->private_data;
		snapshot_volume->snapshot_next = original_volume->snapshot_next;
	}

	// REGISTER SNAPSHOT OBJECT NAME
	EngFncs->register_name(snapshot_child->feature_header->object_name);
	original_volume->snapshot_next = snapshot_volume;

	snapshot_volume->snapshot_org = original_volume;
	(*snap_object)->associated_object = *original_object;
	(*snap_object)->plugin = my_plugin_rec;
	(*snap_object)->flags |= SOFLAG_MUST_BE_TOP;
	if (!(metadata->flags & EVMS_SNAPSHOT_WRITEABLE)) { //mark object read only if not writable
		(*snap_object)->flags |= SOFLAG_READ_ONLY;
	}
	(*snap_object)->size = metadata->total_chunks * metadata->chunk_size;
	strcpy((*snap_object)->name, snapshot_child->feature_header->object_name);

	InsertObject(snapshot_child->parent_objects, sizeof(storage_object_t), *snap_object, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&tmp);
	InsertObject((*snap_object)->child_objects, sizeof(storage_object_t), snapshot_child, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&tmp);

	RETURN(rc);
}

/* get the list of volumes, search for the one requested.  If found, RETURN the top object
  in the volume, otherwise RETURN NULL */
static storage_object_t * find_original(char * original_name) {

	int rc = 0;
	logical_volume_t * check_volume;
	storage_object_t * original_object= NULL;
	dlist_t global_volumes;

	LOG_ENTRY;
	if (!original_name) {
		RETURN(NULL);
	}

	// get the list of objects to search for original volumes
	rc = EngFncs->get_volume_list(NULL, &global_volumes);
	if (!rc) {
		while (ExtractObject(global_volumes, sizeof(logical_volume_t), VOLUME_TAG, NULL, (void**)&check_volume)==0) {
			if (!strncmp(check_volume->name + strlen(EVMS_DEV_NODE_PATH),original_name,EVMS_VOLUME_NAME_SIZE)) {
				LOG("Found a Snapshot Original  %s\n",check_volume->name);
				original_object = check_volume->object;
			}
		}
		DestroyList(&global_volumes, FALSE);
	}else {
		LOG_ERROR("Error getting volume list = %d....\n",rc);
	}
	RETURN(original_object);
}

// Get the list of volumes on the system that we can snapshot
static int get_volume_list(value_list_t ** value_list) {

	int rc = 0;
	logical_volume_t * volume;
	dlist_t global_volumes;
	int count, i;

	LOG_ENTRY;
	// get the list of objects to search for original volumes
	rc = EngFncs->get_volume_list(NULL, &global_volumes);
	if (rc) {
		RETURN(rc);
	}
	GetListSize(global_volumes, &count);
	*value_list = malloc(count * sizeof(value_t) + sizeof(value_list_t));  // yeah it's too big, but so what
	if (*value_list) {
		(*value_list)->count = count;
		i = 0;
		while (ExtractObject(global_volumes, sizeof(logical_volume_t), VOLUME_TAG, NULL, (void**)&volume)==0) {
			(*value_list)->value[i].s = malloc(strlen(volume->name) + 1);
			if (!(*value_list)->value[i].s) {
				RETURN(ENOMEM);
			}
			strcpy((*value_list)->value[i].s, volume->name);
			i++;
		}
	}
	DestroyList(&global_volumes, FALSE);

	RETURN (rc);
}

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static int Snap_SetupEVMSPlugin ( engine_mode_t   mode,
				  engine_functions_t * engine_function_table)
{
	/* make a copy of the evms setup parms */
	engine_mode = mode;
	EngFncs     = engine_function_table;
	RETURN(0);
}


static  int Snap_CanSetVolume(storage_object_t * object,
				BOOLEAN            flag){
	int rc = 0;
	LOG_ENTRY;
	if ( ((snapshot_volume_t *)(object->private_data))->snapshot_org == NULL) {
		 // either this is an original with an active snapshot, or a bad object
		rc = 1;
	}
	RETURN(rc);
}
static void Snap_SetVolume(storage_object_t * object,
                           BOOLEAN            creating){
	snapshot_volume_t * snapshot_volume = (snapshot_volume_t *)object->private_data;
	storage_object_t  * original_object;

	LOG_ENTRY;

	if (snapshot_volume->snapshot_org == NULL) {
		// We were called because the name of the original changed.
		// Display warning(s) to the user that all the snapshots
		// of this volume are broken.

		char * msg = malloc(4096);  // Lots of memory for the message.
		BOOLEAN need_header = TRUE;

		if (msg) {
			*msg = '\0';
			while ((snapshot_volume = snapshot_volume->snapshot_next)) {
				if (!(snapshot_volume->flags & EVMS_SNAPSHOT_DISABLED) &&
				    (snapshot_volume->object->volume != NULL)) {
					if (need_header) {
						sprintf(msg, "The name of the snapshot target volume was changed from %s%s to %s.  "
							"All data on the following snapshot volume(s) will be lost due to this operation.  "
							"If you do not want to lose the snapshot data, you must exit this program and not save the changes.\n"
							"\nSnapshot volume(s):\n",
							EVMS_DEV_NODE_PATH, snapshot_volume->meta_data.original_volume, object->volume->name);

						need_header = FALSE;
					}
					strcat(msg, snapshot_volume->object->volume->name);
					strcat(msg, "\n");
				}
			}

			if (!need_header) {
				char * choices[] = {"OK", NULL};
				int answer = 0;

				EngFncs->user_message(my_plugin_rec, &answer, choices, msg);
			}
		}

		LOG_EXIT(0);
		return;
	}

	original_object = snapshot_volume->snapshot_org->object;
	object->flags |= SOFLAG_DIRTY;
	// since this snapshot might have been discovered in the full or disabled state,
	// make sure the full and disabled flags are off.
	snapshot_volume->meta_data.flags &= ~(EVMS_SNAPSHOT_DISABLED | EVMS_SNAPSHOT_FULL);
	
	// we only need to do the rediscovery for volume creation (TRUE) if
	// volume revert, then the Engine will use the assiciative pointer in the
	// volume structure to do the soft delete of our original.
	// for each snapshot in the chain, call for a rediscover.
	// don't worry about filtering out this snapshot as the new flag in the volume will
	// keep it from being added to the list in the rediscover_volume function.
	if (creating) {
		snapshot_volume =(snapshot_volume_t *)original_object->private_data ;
		while ((snapshot_volume = snapshot_volume->snapshot_next)) {
			// for now, check to be sure it is not the object we are now making into a volume
			if ((snapshot_volume->object != object) && (snapshot_volume->object->volume)) {
				EngFncs->rediscover_volume(snapshot_volume->object->volume, FALSE); // make the Engine delete and rediscover the snapshot volume in the kernel
			}
		}
		EngFncs->rediscover_volume(original_object->volume, TRUE); // make the Engine delete and rediscover the original volume in the kernel
	} else { // unset
		// if we are the first and only snapshot in the list.
		if (snapshot_volume->snapshot_org->snapshot_next == snapshot_volume &&
		       snapshot_volume->snapshot_next == NULL) {
			EngFncs->rediscover_volume(original_object->volume, FALSE); // make the Engine delete and rediscover the original volume in the kernel
		}
	}
	LOG_EXIT(0);
	return;
}

//  Only allow deletes of snapshot volumes, not originals.
static int Snap_CanDelete( storage_object_t * object)
{
	int rc = 0;
	LOG_ENTRY;
	if ( ((snapshot_volume_t *)(object->private_data))->snapshot_org == NULL ||
	      object->volume != NULL ) {
		// We only allow delete on actual snapshot, original
		// must be deleted after snapshot, and by then we will be removed from
		// the object stack and don't need to worry about it.
		rc = EINVAL;
	}

	RETURN(rc);
}

// No, can't expand yet
static int Snap_CanExpand( storage_object_t * object,
			   u_int64_t        * expand_limit,      // a delta size
			   dlist_t            expantion_points)
                                                            // tag = EXPAND_OBJECT_TAG
{
	int rc = 0;
	// BUGBUG for now, don't allow expand.
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}


static int Snap_CanExpandBy(storage_object_t * object,
			    u_int64_t        * size){
	int rc = 0;
	// BUGBUG for now, don't allow expand.
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}

// No, can't shrink SNAPSHOTS
static int Snap_CanShrink( storage_object_t * object,
                           u_int64_t        * shrink_limit,      // a delta size
			   dlist_t            shrink_points )
{
	int rc = 0;
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}

static int Snap_CanShrinkBy(storage_object_t * object,
			     u_int64_t       *  size){
	int rc = 0;
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}

// Not us, but should allow underneath
static int Snap_CanMove( storage_object_t * object )
{
	//BUGBUG definitely allow move of snapshot
	// Concern with moving of original is if compatibilityy, can I ensure
	// name will not change as a result.  If so allow it.
	int rc = 0;
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);  //BUGBUG  not supported yet
}

/*
 *  Called to run discovery code on the object list that
 *  the disk manager found.
 *
 */
static int Snap_Discover(dlist_t object_list,
			dlist_t output_objects,
			BOOLEAN final_call)
{
	char                data[EVMS_VSECTOR_SIZE];
	snapshot_metadata_t * metadata = (snapshot_metadata_t *) data;
	storage_object_t * snapshot_child;
	storage_object_t * snap_object;
	storage_object_t * original_object;
	storage_object_t * original_child;
	int  rc, size, org_crc, new_crc;

	LOG_ENTRY;

	//Only the target object should be in the list, if there are more, this is an error.
	GetListSize(object_list, &size);
	if (size != 1) {
		LOG_ERROR("Error: more than 1 object in list\n");
		rc = EINVAL;
		RETURN(rc);
	}
	
	rc = ExtractObject(object_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, NULL, (void**)&snapshot_child);
	if (rc) {
		LOG_ERROR("Error: Input list corrupt\n");
	}
	else{
		rc = READ(snapshot_child, snapshot_child->size - 3, 1, (void *)data);
		if (rc) {
			LOG_ERROR("Error reading object %s rc=%d\n",snapshot_child->name, rc );
		}
		else{
			if ( DISK_TO_CPU32(metadata->signature) == EVMS_SNAPSHOT_SIGNATURE ) {
				LOG("Found a Snapshot on  %s\n",snapshot_child->name);
				org_crc = DISK_TO_CPU32(metadata->CRC);
				metadata->CRC = 0;
				new_crc = EngFncs->calculate_CRC(EVMS_INITIAL_CRC, metadata, sizeof(snapshot_metadata_t));
				if (new_crc != org_crc) {
					LOG_ERROR("SnapShot:Error in crc on object %s \n",snapshot_child->name);
					rc = EVMS_FEATURE_FATAL_ERROR;
				}else {	
					convert_metadata(metadata);
					if (metadata->version.major > my_plugin_rec->version.major ) {
						LOG_ERROR("SnapShot:Error Metadata at higher version than plugin %s \n",snapshot_child->name);
						rc = EVMS_FEATURE_FATAL_ERROR;
					} else {
						original_child = find_original(metadata->original_volume);
						if (!original_child) {
							LOG_ERROR("Error finding original volume %s \n",metadata->original_volume);
							rc = EVMS_FEATURE_FATAL_ERROR;
						}
						else{
							rc = allocate_snap_objects(&snap_object, snapshot_child, &original_object, original_child, metadata);
						}
					}
				}

			}
			else{
				rc = EINVAL;
				LOG_ERROR("Error No Snapshot Metadata found on object %s rc=%d\n",snapshot_child->name, rc );
			}
		}

	}
	if (rc == 0) {
		InsertObject(output_objects, sizeof(storage_object_t), snap_object, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&size);
		if (metadata->flags & EVMS_SNAPSHOT_FULL) {
			MESSAGE("SnapShot object \"%s\" has been disabled because it is full.\n",snap_object->name);
		} else if (metadata->flags & EVMS_SNAPSHOT_DISABLED) {
			MESSAGE("SnapShot object \"%s\" has been disabled by the kernel. See syslog for details.\n",snap_object->name);
		}
	}

	if (rc) {
		rc = EVMS_FEATURE_FATAL_ERROR;
	}
	RETURN(rc);
}

static int get_create_options( option_array_t * options,
                              char       **     name,
                              char       **     org_volume,
			      u_int32_t  *      chunk_size,
			      unsigned char	 * 	writeable)
{
    int i;
    int rc = 0;

    LOG_ENTRY;

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            switch (options->option[i].number) {

	    case SNAP_OPTION_SNAPSHOT_INDEX:
		    rc = EngFncs->validate_name(options->option[i].value.s);
		    if (rc == 0){
			    *name = options->option[i].value.s;
		    }
                    break;

	    case SNAP_OPTION_ORG_VOLUME_INDEX:
		    // Not worth validation, will catch when we try to find the original
                    *org_volume = options->option[i].value.s;
                    break;

	    case SNAP_OPTION_CHUNKSIZE_INDEX:
		    // option is in 512 byte sectors
                    *chunk_size = options->option[i].value.ui32;
                    break;

	    case SNAP_OPTION_WRITEABLE_INDEX:
		    // option true is for writeable snapshot
                    *writeable = options->option[i].value.bool;
                    break;

                default:
                    break;

            }

        } else {

            if (strcmp(options->option[i].name, SNAP_OPTION_SNAPSHOT_NAME) == 0) {
		    rc = EngFncs->validate_name(options->option[i].value.s);
		    if (rc == 0){
			    *name = options->option[i].value.s;
		    }
            }
            else if (strcmp(options->option[i].name, SNAP_OPTION_ORG_VOLUME_NAME) == 0) {
                *org_volume = options->option[i].value.s;
            }
            else if (strcmp(options->option[i].name, SNAP_OPTION_CHUNKSIZE_NAME) == 0) {
                // option is in 512 byte sector
                *chunk_size = options->option[i].value.ui32;
            }
            else if (strcmp(options->option[i].name, SNAP_OPTION_WRITEABLE_NAME) == 0) {
                *writeable = options->option[i].value.bool;
	    }
        }
    }

    RETURN(rc);

}


/*
 * Create a new snapshot.
 * input: objects - a list which contians one object to be the target of the new snapshot
 * options: object name, volume to snapshot, and chunk size.
 * output: new object for the snapshot.
 * also we have inserted an object into the stack of the original
 */
static int Snap_Create(dlist_t                objects,
                       option_array_t     * options,
                       dlist_t new_objects)
{
	int size, rc, tmp;
	TAG tag;
	storage_object_t * original_object;
	storage_object_t * original_child;
	storage_object_t * snapshot_object;
	storage_object_t * snapshot_child;  // this is the object to be consumed for the snapshot
	char * name;
	char * org_vol_name;
	u_int32_t chunk_size = SNAPSHOT_DEFAULT_CHUNK_SIZE;
	u_int32_t data_size, chunks;
	unsigned long remainder;
	snapshot_metadata_t * metadata;
	evms_feature_header_t * feature_header;
	unsigned char writeable = 0;

	LOG_ENTRY;

	//Only the target object should be in the list, if there are more, this is an error.
	GetListSize(objects, &size);
	if (size != 1) {
		LOG_ERROR( "Error: %d objects in list, expected 1\n",size);
		rc = EINVAL;
		RETURN(rc);
	}

	rc = get_create_options(options, &name, &org_vol_name, &chunk_size, &writeable);
	if (rc) {
		LOG_ERROR("Error: Bad Parameter\n");
		rc = EINVAL;
		RETURN(rc);
	}


	rc = BlindExtractObject(objects,
				&size,
				&tag,
				NULL,			   // get current
				(void**)&snapshot_child);
	if (rc) {
		LOG_ERROR("Error: Input list corrupt\n");
		rc = EINVAL;
		RETURN(rc);
	}

	original_child = find_original(org_vol_name+strlen(EVMS_DEV_NODE_PATH));
	if (!original_child) {
		LOG_ERROR("Error: Can't find the original volume %s\n", org_vol_name);
		rc = EINVAL;
		RETURN(rc);
	}

	feature_header = (evms_feature_header_t *)EngFncs->engine_alloc(SECTOR_SIZE);
	metadata = (snapshot_metadata_t *)EngFncs->engine_alloc(SECTOR_SIZE);

	if (!feature_header || !metadata) {
		rc = ENOMEM;
		RETURN(rc);		// don't bother cleaning up, no mem we are dead.
	}



	chunks = (snapshot_child->size - 3) / chunk_size;	// find possible chunks  (3 = 2*feature header + 1*metadata)
	remainder = (snapshot_child->size - 3) % chunk_size;	// find possible chunks  (3 = 2*feature header + 1*metadata)
	data_size = (((chunks * 8)+511) / SECTOR_SIZE) + 3;	// calc size of cow table ;8 bytes per entry  + 1 for rounding + medadataand FHs
	chunks = chunks - (data_size / chunk_size); 	 	// remove whole chunks need for cow table
	if ((data_size % chunk_size) > remainder) {		// see if extra fits partial chunk at end
		chunks--;
	}

	feature_header->signature = EVMS_FEATURE_HEADER_SIGNATURE;
	feature_header->feature_id = SetPluginID(
				   IBM_OEM_ID,
				   EVMS_ASSOCIATIVE_FEATURE,         //Associative FEATURE class
				   EVMS_SNAPSHOT_FEATURE_ID);
  	feature_header->flags = EVMS_FEATURE_ACTIVE;            // Set active, but not volume.
								// will get a SetVolume call to make real volume
	feature_header->feature_data1_size = data_size;
	feature_header->feature_data1_start_lsn= snapshot_child->size - data_size;
	feature_header->feature_data2_size = 0;
	feature_header->feature_data2_start_lsn= 0;
        strcpy (feature_header->object_name, name);
	snapshot_child->feature_header = feature_header;

	metadata->signature = EVMS_SNAPSHOT_SIGNATURE;
	strcpy (metadata->original_volume,org_vol_name + strlen(EVMS_DEV_NODE_PATH));
	metadata->original_size = original_child->volume->vol_size;
	metadata->lba_of_COW_table = feature_header->feature_data1_start_lsn; //= feature_header, metadata
	metadata->lba_of_first_chunk = 0;
	metadata->chunk_size = chunk_size;
	metadata->total_chunks = chunks;
	metadata->version.major = MAJOR_VERSION;
	metadata->version.minor = MINOR_VERSION;
	metadata->version.patchlevel = PATCH_LEVEL;
	if (writeable) {
		metadata->flags = EVMS_SNAPSHOT_WRITEABLE;
	}

	rc = allocate_snap_objects(&snapshot_object, snapshot_child, &original_object, original_child, metadata);
	if (!rc) {
		snapshot_object->flags |= SOFLAG_DIRTY;
		original_object->flags &= ~SOFLAG_DIRTY;
		InsertObject(new_objects, sizeof(storage_object_t), snapshot_object, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&tmp);
	}

	EngFncs->engine_free(metadata);
	
	RETURN(rc);
}

static int Snap_Delete(storage_object_t * object,  dlist_t            child_objects){

	int rc=0;
	int tmp;
	storage_object_t * child;

	LOG_ENTRY;
	// check if valid object, mine, and not the original.
	if ( ( !object ) ||
	     ( object->plugin != my_plugin_rec) ||
	     ( ((snapshot_volume_t *)(object->private_data))->snapshot_org == NULL)) {
		rc = EINVAL;
		RETURN(rc);
	}

	delete_objects(object, &child);
	rc = InsertObject(child_objects, sizeof(storage_object_t), child, EVMS_OBJECT_TAG, NULL, InsertAtStart , 0, (void**)&tmp);

	RETURN(rc);
}


// Move
static int Snap_Move(storage_object_t * source,
		     storage_object_t * target,
		     option_array_t   * options)
{
	int rc = 0;
	//BUGBUG, if we allow moving, fill in real code here
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}

static int Snap_Expand(storage_object_t * object,
			storage_object_t * expand_object,
			dlist_t              objects,
			option_array_t   * options )
{
	
	int rc = 0;
	//BUGBUG, if we allow expanding, fill in real code here
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}

static int Snap_Shrink( storage_object_t * object,
			storage_object_t * shrink_object,
			dlist_t            objects,
			option_array_t   * options)
{
	int rc = 0;
	// No thanks.
	LOG_ENTRY;
	rc = ENOSYS;
	RETURN(rc);
}


static int Snap_AddSectorsToKillList( storage_object_t * object,
				      lsn_t              lsn,
				      sector_count_t     count)
{
	int rc = ENOSYS;
	LOG_ENTRY;
	// we don't really expect to get called on this API.
	LOG_WARNING("WARNING!!! Called for Killsectors\n");
	RETURN(rc);

}


static int Snap_CommitChanges( storage_object_t * object, uint phase)
{
	int rc = 0;
	char buffer[EVMS_VSECTOR_SIZE];
	snapshot_volume_t * snapshot_volume = (snapshot_volume_t *)object->private_data;
	storage_object_t * snapshot_child = snapshot_volume->child_object;

	LOG_ENTRY;

	if (!(object->flags & SOFLAG_DIRTY) || (phase != 1)) {
		LOG_DETAILS("Nothing to do on Commit object=%s phase =%d\n",object->name,phase);
		RETURN(rc);
	}
	// if this is an original node, skip it as we will be called on the snapshot object
	// to do the real work.
	if (snapshot_volume->snapshot_org == NULL) {
		object->flags &= ~SOFLAG_DIRTY;
		rc = 0;
	} else {
		memset(buffer, 0x00, sizeof(buffer));
		convert_metadata(&snapshot_volume->meta_data);
		snapshot_volume->meta_data.CRC = 0;
		snapshot_volume->meta_data.CRC = DISK_TO_CPU32(EngFncs->calculate_CRC(EVMS_INITIAL_CRC,&(snapshot_volume->meta_data), sizeof(snapshot_metadata_t)));
		memcpy(buffer, &(snapshot_volume->meta_data), sizeof(snapshot_metadata_t));
         	rc = WRITE(snapshot_child, snapshot_child->size - 3, 1, buffer);

		convert_metadata(&snapshot_volume->meta_data); // convert it back for engine use.
		memset(buffer, 0xff, sizeof(buffer));
         	rc = WRITE(snapshot_child, snapshot_volume->meta_data.lba_of_COW_table, 1, buffer);

		// turn off the dirty flag
		object->flags &= ~SOFLAG_DIRTY;
	}

	RETURN(rc);
}

static int Snap_GetInfo(storage_object_t        * object,
			char                    * name,
			extended_info_array_t * * info_array){

	extended_info_array_t * info=NULL;
	snapshot_volume_t * volume = object->private_data;
	snapshot_volume_t * tmp_volume;
	int i, percent_full = 0;
	evms_plugin_ioctl_t	arg;
	int			handle = 0;
	int rc = 0;
	LOG_ENTRY;
       		if (volume->snapshot_org) {		   // this is a snapshot
       			info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + sizeof(extended_info_t) * 4);
			if (!info) {
				RETURN(ENOMEM);
			}

       			if (volume->object->volume) {
       				SET_STRING_FIELD( info->info[0].name, "Original" );
       				SET_STRING_FIELD( info->info[0].title, "SnapShot of:" );
       				SET_STRING_FIELD( info->info[0].desc, "Indicates which volume this volume is a snapshot of.");
       				info->info[0].type               = EVMS_Type_String;
       				info->info[0].unit               = EVMS_Unit_None;
       				SET_STRING_FIELD( info->info[0].value.s, volume->snapshot_org->object->volume->name );
       				info->info[0].collection_type    = EVMS_Collection_None;
       				memset( &info->info[0].group, 0, sizeof(group_info_t));
       			}else{
       				SET_STRING_FIELD( info->info[0].name, "iOriginal" );
       				SET_STRING_FIELD( info->info[0].title, "Inactive SnapShot of:" );
       				SET_STRING_FIELD( info->info[0].desc, "The volume which this object is a snapshot of.  Making this object a volume will reset and activate the snapshot.");
       				info->info[0].type               = EVMS_Type_String;
       				info->info[0].unit               = EVMS_Unit_None;
       				SET_STRING_FIELD( info->info[0].value.s, volume->snapshot_org->object->volume->name );
       				info->info[0].collection_type    = EVMS_Collection_None;
       				memset( &info->info[0].group, 0, sizeof(group_info_t));
       			}

       			SET_STRING_FIELD( info->info[1].name, "ChunkSize" );
       			SET_STRING_FIELD( info->info[1].title, "Chunk Size:" );
       			SET_STRING_FIELD( info->info[1].desc, "The size in Kilobyes of the chunks which are copied to this snapshot");
       			info->info[1].type               = EVMS_Type_Unsigned_Int32;
       			info->info[1].unit               = EVMS_Unit_Kilobytes;
       			// option is kbytes, convert from 512 byte sector and store
       			info->info[1].value.ui32 = volume->meta_data.chunk_size / 2;
       			info->info[1].collection_type    = EVMS_Collection_None;
       			memset( &info->info[1].group, 0, sizeof(group_info_t));

       			// Calculate the % used.
       			if ( !object->volume || (handle = open (object->volume->name, O_RDWR )) < 0 ) {
       				LOG_WARNING("Error getting handle for volume %d.\n",errno);
       				percent_full = 0;
       			}else{
       				arg.feature_id = my_plugin_rec->id;
       				arg.feature_command = SNAPSHOT_QUERY_PERCENT_FULL;
       				arg.status = 0;
       				arg.feature_ioctl_data = &percent_full;

       			       	if ( (rc = ioctl(handle, EVMS_PLUGIN_IOCTL, &arg)) ||
       				     (rc = arg.status) ) {
       				       	LOG_WARNING("Ioctl error (%d).\n", rc);
       				       	LOG_WARNING("Could not get status of snapshot from kernel\n");
       				       	arg.status = 0;
       				}
       				close(handle);
				if (percent_full == -1) {
					volume->meta_data.flags |= EVMS_SNAPSHOT_FULL;
					percent_full = 100;
				} else if (percent_full == -2) {
					volume->meta_data.flags |= EVMS_SNAPSHOT_DISABLED;
					percent_full = 0;
				}

       			}

       			SET_STRING_FIELD( info->info[2].name, "Percent" );
       			SET_STRING_FIELD( info->info[2].title, "Percent Full:" );
       			SET_STRING_FIELD( info->info[2].desc, "The percentage of the snapshot already filled.");
       			info->info[2].type               = EVMS_Type_Unsigned_Int32;
       			info->info[2].unit               = EVMS_Unit_Percent;
       			info->info[2].value.ui32 = percent_full;
       			info->info[2].collection_type    = EVMS_Collection_None;
       			memset( &info->info[2].group, 0, sizeof(group_info_t));

       			SET_STRING_FIELD( info->info[3].name, "Writeable" );
       			SET_STRING_FIELD( info->info[3].title, "Writeable" );
       			SET_STRING_FIELD( info->info[3].desc, "Whether snapshot is writeable or not");
       			info->info[3].type               = EVMS_Type_Boolean;
       			info->info[3].unit               = EVMS_Unit_None;
       			if (volume->meta_data.flags & EVMS_SNAPSHOT_WRITEABLE) {
       				info->info[3].value.bool = TRUE;
       			}else{
       				info->info[3].value.bool = FALSE;
       			}
       			info->info[3].collection_type    = EVMS_Collection_None;
       			memset( &info->info[3].group, 0, sizeof(group_info_t));

       			SET_STRING_FIELD( info->info[4].name, "Status" );
       			SET_STRING_FIELD( info->info[4].title, "SnapShot Status:" );
       			SET_STRING_FIELD( info->info[4].desc, "The current status of the SnapShot.");
       			info->info[4].type               = EVMS_Type_String;
       			info->info[4].unit               = EVMS_Unit_None;
			if (volume->meta_data.flags & EVMS_SNAPSHOT_FULL) {
				SET_STRING_FIELD( info->info[4].value.s, "Full, Disabled" );
			} else if (volume->meta_data.flags & EVMS_SNAPSHOT_DISABLED) {
				SET_STRING_FIELD( info->info[4].value.s, "Disabled" );
			} else if (object->volume) {
				SET_STRING_FIELD( info->info[4].value.s, "Active" );
			} else {
				SET_STRING_FIELD( info->info[4].value.s, "Inactive" );
			}
       			info->info[4].collection_type    = EVMS_Collection_None;
       			memset( &info->info[4].group, 0, sizeof(group_info_t));


       			info->count = 5;
       		} else // snapshot original
       		{
       			// count number of snapshots on original
       			for (i=0, tmp_volume = volume;tmp_volume->snapshot_next != NULL;i++,tmp_volume = tmp_volume->snapshot_next) {
       			}

       			info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + sizeof(extended_info_t) * (i+1));
			if (!info) {
				RETURN(ENOMEM);
			}

       			for (i=0, tmp_volume = volume->snapshot_next;tmp_volume != NULL;i++,tmp_volume = tmp_volume->snapshot_next) {
       				if (tmp_volume->object->volume) {
       					SET_STRING_FIELD( info->info[i].name, "SnapShot" );
       					SET_STRING_FIELD( info->info[i].title, "SnapShotted on:" );
       					SET_STRING_FIELD( info->info[i].desc, "Snapshots of this volume");
       					info->info[i].type               = EVMS_Type_String;
       					info->info[i].unit               = EVMS_Unit_None;
       					SET_STRING_FIELD( info->info[i].value.s, tmp_volume->object->volume->name );
       					info->info[i].collection_type    = EVMS_Collection_None;
       					memset( &info->info[i].group, 0, sizeof(group_info_t));
       				}else {
       					SET_STRING_FIELD( info->info[i].name, "iSnapShot" );
       					SET_STRING_FIELD( info->info[i].title, "Inactive Snapshot on:" );
       					SET_STRING_FIELD( info->info[i].desc, "This object is set up to snapshot this volume. If the object is made into a volume the snapshot will be reset and activated");
       					info->info[i].type               = EVMS_Type_String;
       					info->info[i].unit               = EVMS_Unit_None;
       					SET_STRING_FIELD( info->info[i].value.s, tmp_volume->object->name );
       					info->info[i].collection_type    = EVMS_Collection_None;
       					memset( &info->info[i].group, 0, sizeof(group_info_t));
       				}
       			}

       			info->count = i;
		}
	if (rc == 0) {
		*info_array=info;
	}
	RETURN (rc);
}

static int Snap_GetOptionCount(task_context_t * task)
{
	int count = 0;
	LOG_ENTRY;
	switch (task->action) {
	case EVMS_Task_Create:
		count = 4;
		break;
	default:
		count = -1;
	}
	
	RETURN(count);

}
static int Snap_InitTask(task_context_t * context)
{
	dlist_t tmp_list;
	storage_object_t * tmp_object;
	int rc = 0;
	unsigned long tmp, tag;
	int size;
	LOG_ENTRY;

	// set the number of objects that must/can be selected.
	context->min_selected_objects = 1;
	context->max_selected_objects = 1;

	// get a list of all valid input objects.
	EngFncs->get_object_list(0,          // all types
				DATA_TYPE,
				NULL,
				VALID_INPUT_OBJECT,
				&tmp_list);

	// extract each object and add to list if appropriate for action
	while (!(rc = BlindExtractObject(tmp_list, &size, &tag, NULL, (void **)&tmp_object))) {

		switch (context->action) {
		case EVMS_Task_Create:
			// as long as we aren't on top we can use it.
			if (tmp_object->plugin != my_plugin_rec) {
				rc = InsertObject(context->acceptable_objects, sizeof(storage_object_t), tmp_object, EVMS_OBJECT_TAG, NULL, InsertAtStart, TRUE, (void **)&tmp);
			}
			break;
		default:
			rc = EINVAL;
			
			RETURN(rc);

		}

	}
	if (rc == DLIST_EMPTY || rc == DLIST_END_OF_LIST) {
		rc = 0;
	}

	switch (context->action) {
	case EVMS_Task_Create:

		context->option_descriptors->count = 4;	     // must set count
		// option 0 is original volume name
		context->option_descriptors->option[0].flags =EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		// get the list of volume that can be snapshotted.
		get_volume_list((value_list_t **)&context->option_descriptors->option[0].constraint.list);
		context->option_descriptors->option[0].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[0].help = NULL;
		context->option_descriptors->option[0].name = strdup(SNAP_OPTION_ORG_VOLUME_NAME);
		context->option_descriptors->option[0].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING_FIELD(context->option_descriptors->option[0].tip, "This is the volume you wish to take a snapshot of." );
		SET_STRING_FIELD(context->option_descriptors->option[0].title, "Volume to be Snapshotted" );
		context->option_descriptors->option[0].type = EVMS_Type_String;
		context->option_descriptors->option[0].unit = EVMS_Unit_None;
		context->option_descriptors->option[0].value.s = calloc(1,EVMS_VOLUME_NAME_SIZE);

		// option 1 is original snapshot name
		context->option_descriptors->option[1].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		context->option_descriptors->option[1].constraint.list = NULL;
		context->option_descriptors->option[1].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[1].help = NULL;
		context->option_descriptors->option[1].name = strdup(SNAP_OPTION_SNAPSHOT_NAME);
		context->option_descriptors->option[1].size = EVMS_VOLUME_NAME_SIZE;
		SET_STRING_FIELD(context->option_descriptors->option[1].tip, "This is the name you wish to assign the the object being created." );
		SET_STRING_FIELD(context->option_descriptors->option[1].title, "Snapshot Object Name:" );
		context->option_descriptors->option[1].type = EVMS_Type_String;
		context->option_descriptors->option[1].unit = EVMS_Unit_None;
		context->option_descriptors->option[1].value.s = calloc(1,EVMS_VOLUME_NAME_SIZE);

		// option 2 is chunk size
		context->option_descriptors->option[2].flags = 0;
		SET_POWER2_LIST(context->option_descriptors->option[2].constraint.list, SNAPSHOT_MIN_CHUNK_SIZE, SNAPSHOT_MAX_CHUNK_SIZE);
		context->option_descriptors->option[2].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[2].help = NULL;
		context->option_descriptors->option[2].name = strdup(SNAP_OPTION_CHUNKSIZE_NAME);
		context->option_descriptors->option[2].size = sizeof(u_int32_t);
		SET_STRING_FIELD(context->option_descriptors->option[2].tip, "Chunksize is the unit size of the data that will be copied to the snapshot." );
		SET_STRING_FIELD(context->option_descriptors->option[2].title, "Chunk Size:" );
		context->option_descriptors->option[2].type = EVMS_Type_Unsigned_Int32;
		context->option_descriptors->option[2].unit = EVMS_Unit_Sectors;
		context->option_descriptors->option[2].value.ui32 = SNAPSHOT_DEFAULT_CHUNK_SIZE;

		// option 3 is Writeable flag
		context->option_descriptors->option[3].flags = 0;
		context->option_descriptors->option[3].constraint.list = NULL;
		context->option_descriptors->option[3].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[3].help = NULL;
		context->option_descriptors->option[3].name = strdup(SNAP_OPTION_WRITEABLE_NAME);
		context->option_descriptors->option[3].size = sizeof(int);
		SET_STRING_FIELD(context->option_descriptors->option[3].tip, "Enableing this option allows writes to the snapshot.." );
		SET_STRING_FIELD(context->option_descriptors->option[3].title, "Writeable:" );
		context->option_descriptors->option[3].type = EVMS_Type_Boolean;
		context->option_descriptors->option[3].unit = EVMS_Unit_None;
		context->option_descriptors->option[3].value.bool = 0;

		break;
	default:
		break;
	}


	
	RETURN(0);
}

static int Snap_SetOption(task_context_t * context,
                     u_int32_t index,
                     value_t * value,
                     u_int32_t * info)
{
	int rc =0;
	int i;

	LOG_ENTRY;
	switch (context->action){
	case EVMS_Task_Create:

		switch (index) {
		    case SNAP_OPTION_SNAPSHOT_INDEX:
			    rc = EngFncs->validate_name(value->s);
			    if (rc == 0) {
				    strcpy(context->option_descriptors->option[index].value.s, value->s);
                                    context->option_descriptors->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
			    }
			break;

		case SNAP_OPTION_ORG_VOLUME_INDEX:
			if (strlen(value->s) > EVMS_VOLUME_NAME_SIZE) {
				rc = EINVAL;
			}
			else {
				rc = EINVAL;
				// compare input with predetermind list of volumes in constraint list
				for (i = 0; i < context->option_descriptors->option[index].constraint.list->count; i++) {
					if (!strcmp(value->s,context->option_descriptors->option[index].constraint.list->value[i].s)) {
						strcpy(context->option_descriptors->option[index].value.s, value->s);
						LOG_DETAILS("Setting original volume to %s\n",value->s);
                                                context->option_descriptors->option[index].flags &= ~EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
						rc = 0;
                                                break;
					}
				}
			}
			break;

		case SNAP_OPTION_CHUNKSIZE_INDEX:
			if ( value->ui32 >= SNAPSHOT_MIN_CHUNK_SIZE &&
			     value->ui32 <= SNAPSHOT_MAX_CHUNK_SIZE &&
			     ( value->ui32 & ( value->ui32 - 1)) == 0)  {
			    context->option_descriptors->option[index].value.ui32 = value->ui32;
			}else{
				rc = EINVAL;
			}
			break;

		case SNAP_OPTION_WRITEABLE_INDEX:
			if ( value->bool ) {
				context->option_descriptors->option[index].value.bool = 1;
			}else{
				context->option_descriptors->option[index].value.bool = 0;
			}
			break;

		    default:
			break;
		}
		break;
	default:
	}
	
	RETURN(rc);
}

static int Snap_SetObjects(task_context_t * context,
			   dlist_t declined_objects,       // of type declined_handle_t
			   task_effect_t * effect)
{
	int rc = 0;
	LOG_ENTRY;

	// BUGBUG validate selected_object list in the task context.
	// put unaccaptable objects on the declined_object list.

	
	RETURN(rc);
}

static int Snap_DirectPluginCommunication(void          * thing,
					  BOOLEAN         target_kernel_plugin,
					  void          * arg)
{
	int rc = 0;
	LOG_ENTRY;
	rc = ENOSYS;
	
	RETURN(rc);
}

static int Snap_Read( storage_object_t * object,
			lsn_t	          lsn,
			sector_count_t    count,
			void            * buffer)
{
	int  rc = EINVAL;
	snapshot_volume_t * snapshot_volume = (snapshot_volume_t *)object->private_data;
	LOG_ENTRY;

	// only allow reads from original
	if (snapshot_volume->snapshot_org == NULL) {
		rc = READ(snapshot_volume->child_object, lsn, count, buffer);
	}
	
	RETURN(rc);
}

static int Snap_Write( storage_object_t * object,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer)
{
	int  rc = 0;

	LOG_ENTRY;
	// DON'T ALLOW WRITES
	rc = ENOSYS;
	
	RETURN(rc);
}


/* Function: Snap_get_plugin_info
 *
 *	Return information about the Snapshot plugin. There is no "extra"
 *	information about Snapshot, so "name" should always be NULL.
 */
static int Snap_get_plugin_info(	char			* name,
				extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[50] = {0};
	int			i = 0;
	int rc = 0;
	
	LOG_ENTRY;

	if ( ! name ) {
		// Get memory for the info array
		if ( ! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*4)) ) {
			LOG_ERROR("Error allocating memory for info array\n");
			RETURN(ENOMEM);
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, my_plugin_rec->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, my_plugin_rec->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Associative Feature");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required API Version
		SET_STRING(info->info[i].name, "Required_Version");
		SET_STRING(info->info[i].title, "Required Plugin API Version");
		SET_STRING(info->info[i].desc, "This is the version of the engine that the plugin requires.It will not run on older versions of the Engine.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", my_plugin_rec->required_api_version.major, my_plugin_rec->required_api_version.minor, my_plugin_rec->required_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	}
	else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		
		RETURN(EINVAL);
	}

	info->count = i;
	*info_array = info;
	
	RETURN(0);
}

// Cleanup routine
// Get the list of all object owned by this pluging from the Engine, delete our
// private data (snapshot_volume_t structure) and free the node.
// Do NOT try to look at parents or children as they may already be destroyed, so
// we can't free our feature headers, someone else will get them.

void Snap_plugin_cleanup()
{
	int rc = 0;
	dlist_t tmp_list;
	storage_object_t * object;

	LOG_ENTRY;
	EngFncs->get_object_list(0,          // all types
				DATA_TYPE,
				my_plugin_rec,
				0,
				&tmp_list);
	if (!rc) {
		while (ExtractObject(tmp_list, sizeof(storage_object_t), EVMS_OBJECT_TAG, NULL, (void**)&object)==0) {
			EngFncs->engine_free(object->private_data );
		}
		DestroyList(&tmp_list, FALSE);
	}
	LOG_EXIT(0);
	return;
}

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static plugin_functions_t fft={
	setup_evms_plugin:                      Snap_SetupEVMSPlugin,
	cleanup_evms_plugin:                    Snap_plugin_cleanup,
	can_delete:                             Snap_CanDelete,
	can_expand:                             Snap_CanExpand,
	can_expand_by:                          Snap_CanExpandBy,
	can_shrink:                             Snap_CanShrink,
	can_shrink_by:                          Snap_CanShrinkBy,
	can_move:                               Snap_CanMove,
	can_set_volume:				Snap_CanSetVolume,
	discover:                       	Snap_Discover,
	create:                        		Snap_Create,
	delete: 	  	                Snap_Delete,
	expand:                 	        Snap_Expand,
	shrink:                         	Snap_Shrink,
	move:                               	Snap_Move,
	add_sectors_to_kill_list:               Snap_AddSectorsToKillList,
	commit_changes:                       	Snap_CommitChanges,
	get_option_count:                       Snap_GetOptionCount,
	init_task:                              Snap_InitTask,
	set_option:                             Snap_SetOption,
	get_info:                               Snap_GetInfo,
	get_plugin_info:			Snap_get_plugin_info,
	set_objects:                            Snap_SetObjects,
	direct_plugin_communication:           	Snap_DirectPluginCommunication,
	read:                                	Snap_Read,
	write:                               	Snap_Write,
	set_volume: 				Snap_SetVolume
};

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                 EXPORTED SUBROUTINES ( Init & Cleanup )                              +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t plugin_record = {
	id:		SetPluginID(IBM_OEM_ID, EVMS_ASSOCIATIVE_FEATURE, EVMS_IBM_SNAPSHOT_FEATURE),

	version:          {major:      MAJOR_VERSION,
                           minor:      MINOR_VERSION,
                           patchlevel: PATCH_LEVEL},

	required_api_version: {major:      3,
				minor:      0,
				patchlevel: 0},

    short_name:           	"Snapshot",
    long_name:           	"Snapshot Feature",
    oem_name:             	"IBM",

    functions:             {plugin:   &fft},

    container_functions:   NULL
};

static plugin_record_t                *my_plugin_rec = &plugin_record;
plugin_record_t * evms_plugin_records[] = {&plugin_record, NULL};
