/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: orm_io.c
 *
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <linux/evms/evms_os2.h>
#include <plugin.h>
#include "os2regmgr.h"
#include "orm_options.h"
#include "orm_io.h"


/*
 *  Special call for LVM Volumes...
 *    Walk the drive links until the proper object is found.
 *
 */
static int LVM_add_sectors_to_kill_list( storage_object_t   * object,
                                         lsn_t                lsn,
                                         sector_count_t       count,
                                         orm_private_data_t  *pdata )
{
    int rc = EINVAL;
    int i;

    lsn_t           io_lsn                = lsn;
    sector_count_t  io_sector_count,
                    sectors_left_to_write = count,
                    max_sector_count = 0;

    storage_object_t          *child;
    struct plugin_functions_s  *Fncs;
    os2_drivelink_t         *curlink = pdata->drive_link;

    LOGENTRY();

    if ( count > 0 ) {

        for ( i = 0; i < pdata->drive_link_count; i++ ) {

            if ( io_lsn > curlink->start_sector + curlink->sector_count )
                curlink = curlink->next;
            else {

                child = curlink->object;

                Fncs = ( struct plugin_functions_s * )child->plugin->functions.plugin;

                max_sector_count = curlink->start_sector + curlink->sector_count - io_lsn + 1;

                if ( max_sector_count >= sectors_left_to_write ) {
                    io_sector_count = sectors_left_to_write;
                }
                else {
                    io_sector_count = max_sector_count;
                }

                rc = Fncs->add_sectors_to_kill_list( child, io_lsn, io_sector_count );

                io_lsn                += io_sector_count;
                sectors_left_to_write -= io_sector_count;

                if (( sectors_left_to_write == 0 ) || ( rc ))  break;
            }
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Passes the API call down to the appropriate drive link child objects.
 *
 *  I check that:
 *
 *  - I own the object
 *  - the logical sector is found in the drive link
 *
 *  I do it by walking the drive link and determining the child storage
 *    object that contains the specified sectors and then forwarding the
 *    API call to that object.
 *
 *  Returns:  RC == 0 if call was successful.
 *
 */
int orm_add_sectors_to_kill_list( storage_object_t   * region,
                                  lsn_t              lsn,
                                  sector_count_t     count )
{
    int  rc = EINVAL;
    struct plugin_functions_s  *Fncs;
    storage_object_t  *seg;

    LOGENTRY();

    if ( orm_ican_modify( region ) == TRUE  &&
         lsn + count <= region->size ) {

        if ( (( orm_private_data_t * )region->private_data )->flags & COMPATIBILITY_VOLUME ) {

            seg  = orm_get_first_object_in_list( region->child_objects );

            if ( seg ) {

                Fncs = ( struct plugin_functions_s * )seg->plugin->functions.plugin;
                rc   = Fncs->add_sectors_to_kill_list( seg, lsn, count );
            }

        }
        else if ( (( orm_private_data_t * )region->private_data )->flags & LVM_VOLUME ) {

            rc = LVM_add_sectors_to_kill_list( region, lsn, count, ( orm_private_data_t * )region->private_data );
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Special call for LVM Volumes...
 *    Walk the drive links until the proper object is found.
 *
 */
static int LVM_Read( storage_object_t   * object,
                     lsn_t                lsn,
                     sector_count_t       count,
                     void               * buffer,
                     orm_private_data_t * pdata )
{
    int rc = EINVAL;
    int i;

    char      *io_buffer_ptr     = ( char * )buffer;

    lsn_t           io_lsn                = lsn;
    sector_count_t  io_sector_count,
                    sectors_left_to_write = count,
                    max_sector_count = 0;

    storage_object_t           *child;
    struct plugin_functions_s  *Fncs;
    os2_drivelink_t         *curlink = pdata->drive_link;


    LOGENTRY();
    LOG_DEBUG( "\tobject->size = %08d  LSN= %d  count= %08d\n", object->size, lsn, count );

    if ( count > 0 ) {

        for ( i = 0; i < pdata->drive_link_count; i++ ) {

            if ( io_lsn > curlink->start_sector + curlink->sector_count )
                curlink = curlink->next;
            else {

                LOG_DEBUG( "\tlsn is in link %d cux link has end_lsn of %d\n", i, ( u_int32_t )( curlink->start_sector + curlink->sector_count ));

                child = curlink->object;

                Fncs  = ( struct plugin_functions_s * )child->plugin->functions.plugin;

                max_sector_count = curlink->start_sector + curlink->sector_count - io_lsn + 1;

                if ( max_sector_count >= sectors_left_to_write ) {
                    io_sector_count = sectors_left_to_write;
                }
                else {
                    io_sector_count = max_sector_count;
                }

                rc = Fncs->read( child,
                                 io_lsn - curlink->start_sector,
                                 io_sector_count,
                                 ( void * )io_buffer_ptr );

                io_lsn                += io_sector_count;
                io_buffer_ptr         += io_sector_count * EVMS_VSECTOR_SIZE;
                sectors_left_to_write -= io_sector_count;

                if (( sectors_left_to_write == 0 ) || ( rc ))  break;
            }
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Passes the API call down to the appropriate drive link child objects.
 *
 *  I check that:
 *
 *  - I own the object
 *  - the logical sector is found in the drive link
 *
 *  I do it by walking the drive link and determining the child storage
 *    object that contains the specified sectors and then forwarding the
 *    API call to that object.
 *
 *  Returns:  RC == 0 if call was successful.
 *
 */
int orm_read( storage_object_t   * region,
              lsn_t                lsn,
              sector_count_t       count,
              void               * buffer )
{
    int  rc = EINVAL;
    struct plugin_functions_s  *Fncs;
    storage_object_t *seg;

    LOGENTRY();

    if (( buffer ) &&
       ( orm_ican_modify( region ) == TRUE ) &&
       ( lsn + count <= region->size )) {

        if ( (( orm_private_data_t * )region->private_data )->flags & COMPATIBILITY_VOLUME ) {

            seg  = orm_get_first_object_in_list( region->child_objects );

            if ( seg ) {

                Fncs = ( struct plugin_functions_s * )seg->plugin->functions.plugin;
                rc   = Fncs->read( seg, lsn, count, buffer );
            }
        }
        else if ( (( orm_private_data_t * )region->private_data )->flags & LVM_VOLUME ) {

            rc = LVM_Read( region, lsn, count, buffer, ( orm_private_data_t * )region->private_data );
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Special call for LVM Volumes...
 *    Walk the drive links until the proper object is found.
 *
 */
static int LVM_Write( storage_object_t  * object,
                      lsn_t               lsn,
                      sector_count_t      count,
                      void              * buffer,
                      orm_private_data_t *pdata )
{
    int rc = EINVAL;
    int i;

    char      *io_buffer_ptr     = ( char * )buffer;

    lsn_t           io_lsn                = lsn;
    sector_count_t  io_sector_count,
                    sectors_left_to_write = count,
                    max_sector_count = 0;

    storage_object_t           *child;
    struct plugin_functions_s  *Fncs;
    os2_drivelink_t         *curlink = pdata->drive_link;

    LOGENTRY();
    LOG_DEBUG( "\tobject->size = %08d  LSN= %d  count= %08d\n", object->size, lsn, count );

    if ( count > 0 ) {

        for ( i = 0; i < pdata->drive_link_count; i++ ) {

            if ( io_lsn > curlink->start_sector + curlink->sector_count )
                curlink = curlink->next;
            else {

                LOG_DEBUG( "\tlsn is in link %d cux link has end_lsn of %d\n", i, ( u_int32_t )( curlink->start_sector + curlink->sector_count ));

                child = curlink->object;

                Fncs  = ( struct plugin_functions_s * )child->plugin->functions.plugin;

                max_sector_count = curlink->start_sector + curlink->sector_count - io_lsn + 1;

                if ( max_sector_count >= sectors_left_to_write ) {
                    io_sector_count = sectors_left_to_write;
                }
                else {
                    io_sector_count = max_sector_count;
                }

                rc=Fncs->write( child,
                                io_lsn - curlink->start_sector,
                                io_sector_count,
                                ( void * )io_buffer_ptr );

                io_lsn                += io_sector_count;
                io_buffer_ptr         += io_sector_count * EVMS_VSECTOR_SIZE;
                sectors_left_to_write -= io_sector_count;

                if (( sectors_left_to_write == 0 ) || ( rc ))  break;
            }
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *  Passes the API call down to the appropriate drive link child objects.
 *
 *  I check that:
 *
 *  - I own the object
 *  - the logical sector is found in the drive link
 *
 *  I do it by walking the drive link and determining the child storage
 *    object that contains the specified sectors and then forwarding the
 *    API call to that object.
 *
 *  Returns:  RC == 0 if call was successful.
 *
 */
int orm_write( storage_object_t  * region,
               lsn_t               lsn,
               sector_count_t      count,
               void              * buffer )
{
    int  rc = EINVAL;
    struct plugin_functions_s  *Fncs;
    storage_object_t  *seg;


    LOGENTRY();

    if (( buffer ) &&
       ( orm_ican_modify( region ) == TRUE ) &&
       ( lsn + count <= region->size )) {

        if ( (( orm_private_data_t * )region->private_data )->flags & COMPATIBILITY_VOLUME ) {

            seg  = orm_get_first_object_in_list( region->child_objects );

            if ( seg ) {

                Fncs = ( struct plugin_functions_s * )seg->plugin->functions.plugin;
                rc   = Fncs->write( seg, lsn, count, buffer );
            }

        }
        else if ( (( orm_private_data_t * )region->private_data )->flags & LVM_VOLUME ) {

            rc = LVM_Write( region, lsn, count, buffer, ( orm_private_data_t * )region->private_data );
        }
    }

    LOGEXITRC();
    return rc;
}

