/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_create.c
 *
 * Description: This file contains all functions related to the common creation
 *              of MD volumes.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#define MY_PLUGIN my_plugin
#include "md.h"


u_int32_t get_random(void) {
	u_int32_t num;
	int fd;

	fd = open("/dev/urandom", O_RDONLY);
	if (fd != -1 && read(fd, &num, 4) == 4) {
		LOG_DEBUG("raid set magic: %x\n", num);
		close(fd);
	} else {
		num = rand();
		LOG_DEBUG("raid set magic (pseudo-random): %x\n", num);
	}
	close(fd);
	return num;
}


// Get the legacy device major and minor for the object to be used in the md array.
// If there is no legacy device number available, then just create a unique
// device number using the MD major 58.
u_int32_t get_legacy_dev(md_volume_t * volume,char * name,u_int32_t *major_num, u_int32_t *minor_num) {

	struct stat statbuf;
	int i,j;
	char devnode_name[EVMS_VOLUME_NAME_SIZE+1];
	LOG_ENTRY;

	sprintf(devnode_name,"/dev/%s",name);
	if (!(stat(devnode_name, &statbuf))) {
		*major_num = major(statbuf.st_rdev);
		*minor_num = minor(statbuf.st_rdev);
	} else {
		// if no legacy device, check if device is in the array
		// and use its major/minor
		for (i = 0; i < MAX_MD_DEVICES; i++) {
			if (volume->child_object[i] &&
			    (strcmp(volume->child_object[i]->name, name) == 0) &&
				 volume->super_block->disks[i].major !=0 ){
				*major_num = volume->super_block->disks[i].major;
				*minor_num = volume->super_block->disks[i].minor;
				break;
			}
		}
		
		if (i >= MAX_MD_DEVICES) {
			// create a number.  find an unused minor.
			*major_num = 9;
			*minor_num = 0;

			for (i = 0; i < MAX_MD_MINORS; i++ ) {
				for (j = 0; j < MAX_MD_DEVICES; j++ ) {
					if (volume->super_block->disks[j].major == 9 &&
					    volume->super_block->disks[j].minor == i) {
						break;
					}
				}
				if (j >= MAX_MD_DEVICES) {
					// found an unused minor
					*minor_num = i;
					break;
				}
			}
		}
	}

	RETURN(0);
}


int md_clone_superblock( md_volume_t *volume,int index) {
	int rc = 0;
	LOG_ENTRY;
	if (md_allocate_memory((void **)&(volume->super_array[index]),MD_SB_BYTES)) {
		RETURN(ENOMEM);
	}
	*(volume->super_array[index]) = *(volume->super_block);
	volume->super_array[index]->this_disk = volume->super_array[index]->disks[index];

	RETURN(rc);
}





/*
 md_create_first_superblock :
		Allocate and initialize a super block, filling in all of the common fields.
	Caller is responsible for filling in the following before cloning SB:


*/


int md_create_first_superblock( md_volume_t *volume,
				mdp_disk_t first_disk,
				__u32 level,
				__u32 chunk_size,
				__u32 size,
				__u32 nr_disks,
				__u32 spare_disks,
				__u32 state) {

	int rc = 0;
	int i,j;
	mdp_super_t * super;
	md_volume_t * tmp_volume = volume_list_head;
	char minor_array[MAX_MD_DEVICES];
	LOG_ENTRY;

	if (md_allocate_memory((void**)&super, MD_SB_BYTES )) {
		LOG_CRITICAL("Memory error creating buffer for new super block.\n");
		RETURN(ENOMEM);
	}

	memset(minor_array, 0x0, sizeof(minor_array));
	while (tmp_volume != NULL) {
		minor_array[tmp_volume->super_block->md_minor] = 1;
		tmp_volume = tmp_volume->next;
	}
	for (i = 0; (i < MAX_MD_DEVICES) && (minor_array[i] == 1); i++);
	if (i >= MAX_MD_DEVICES) {
		LOG_ERROR("No more MD devices avaliable, creation failing");
		RETURN(ENODEV);
	}

	volume->super_block = super;
	sprintf(volume->name, "md/md%d",i);

	super->active_disks = nr_disks - spare_disks;
	super->chunk_size = chunk_size * BLOCK_SIZE;
	super->ctime = super->utime =  time(NULL);
	for (j = 0; j < nr_disks; j++) {
		get_legacy_dev(volume, volume->child_object[j]->name, &first_disk.major, &first_disk.minor);
		first_disk.number = j;
		first_disk.raid_disk = j;
		super->disks[j] = first_disk;
	}
	super->events_lo = 1;
	super->events_hi = 0;
	super->failed_disks = 0;
//	super->gstate_creserved -- Padding space, not used
//	super->gstate_sreserved -- Padding space, not used
//	super->gvalid_words     -- does not appear to ever be filled in
	super->layout = 0;	   // parity layout for raid 5 see *parity_algorithm_table[] in parser.c
	super->level = level;
	super->major_version = 0;
	super->md_magic = MD_SB_MAGIC;
	super->md_minor = i;
	super->minor_version = 90;
	super->not_persistent = 0;
	super->nr_disks = nr_disks;
	super->patch_version = 0;
//	super->pstate_reserved-- Padding space, not used
	super->raid_disks = nr_disks - spare_disks;
//	super->reserved = -- Padding space, not used
//	super->root_block -- Does not appear to be set
//	super->root_pv -- Does not appear to be set
	super->sb_csum = 0;	  // Set on commit
	super->set_uuid0 = get_random();
	super->set_uuid1 = get_random();
	super->set_uuid2 = get_random();
	super->set_uuid3 = get_random();
	super->size = size;
	super->spare_disks = spare_disks;
	super->state = state;
	super->this_disk = super->disks[0];
//	super->utime -- filled in above
	super->working_disks = nr_disks;

	RETURN(rc);
}



