/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_volumes.c
 *
 * Description: This file contains all functions related to the discovery,
 *              creation and management of logical volumes in the LVM region
 *              manager.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <linux/kdev_t.h>
#include <linux/major.h>
#include <plugin.h>
#include "lvmregmgr.h"


// Array for keeping track of which minor numbers are used by LVM LVs.
static int minor_in_use[MAX_LV] = {FALSE};


/* Function: lvm_find_volume_by_name
 *
 *	Find the volume for the specified name. The name given can start
 *	with "/dev/evms/", but must conform to the "lvm/vg_name/lv_name"
 *	scheme for LVM regions.
 */
int lvm_find_volume_by_name(	char			* name,
				lvm_volume_group_t	* group,
				lvm_logical_volume_t	** volume )
{
	char	region_name[EVMS_NAME_SIZE+1] = {0};
	int	i;

	LOG_ENTRY;

	lvm_translate_dev_name(name, region_name);

	for ( i = 1; i <= MAX_LV; i++ ) {
		if ( group->volume_list[i] &&
		     ! strcmp(region_name, group->volume_list[i]->region->name) ) {
			*volume = group->volume_list[i];
			RETURN(0);
		}
	}
	
	LOG_ERROR("Could not find region %s in container %s\n", name, group->container->name);
	*volume = NULL;

	RETURN(EINVAL);
}



/****** Volume Memory Allocation/Deallocation Functions ******/


/* Function: lvm_allocate_le_map
 *
 *	Allocates memory to hold a volume's LE-to-PE mapping. The
 *	lv_allocated_le field of the lv must be filled in.
 */
static int lvm_allocate_le_map( lvm_logical_volume_t * volume )
{
	LOG_ENTRY;

	// A freespace region could potentially have no allocated LEs.
	if ( volume->lv->lv_allocated_le == 0 ) {
		volume->le_map = NULL;
		RETURN(0);
	}

	volume->le_map = lvm_engine->engine_alloc(volume->lv->lv_allocated_le * sizeof(le_table_entry_t));
	if ( ! volume->le_map ) {
		RETURN(ENOMEM);
	}
	RETURN(0);
}


/* Function: lvm_deallocate_le_map
 *
 *	Releases memory used for a volume's LE-to-PE mapping.
 */
static int lvm_deallocate_le_map( lvm_logical_volume_t * volume )
{
	LOG_ENTRY;

	if ( volume->le_map ) {
		lvm_engine->engine_free(volume->le_map);
		volume->le_map = NULL;
	}
	RETURN(0);
}


/* Function: lvm_allocate_logical_volume
 *
 *	Allocate all necessary memory structures for a new logical volume.
 *	Initialize based on the lv and group parameters.
 */
lvm_logical_volume_t * lvm_allocate_logical_volume(	lv_disk_t		* lv,
							lvm_volume_group_t	* group )
{
	lvm_logical_volume_t	* new_volume;
	char			region_name[EVMS_NAME_SIZE+1] = {0};
	int			rc;

	LOG_ENTRY;

	// Allocate the logical volume itself
	new_volume = lvm_engine->engine_alloc(sizeof(lvm_logical_volume_t));
	if ( ! new_volume ) {
		LOG_CRITICAL("Memory error creating new logical volume %s.\n", lv->lv_name);
		RETURN(NULL);
	}

	// Initialize the basic fields
	new_volume->lv		= lv;
	new_volume->group	= group;
	new_volume->number	= lv->lv_number + 1;	// Need the +1 to match the PE Map entries on the PV
	new_volume->minor	= MINOR(lv->lv_dev);
	new_volume->flags	= LVM_LV_FLAG_DIRTY;

	// Space for the LE-to-PE mapping table
	rc = lvm_allocate_le_map(new_volume);
	if (rc) {
		lvm_deallocate_logical_volume(new_volume);
		RETURN(NULL);
	}

	// Translate the LV name to the region name. Should have checked for
	// uniqueness when initializing the LV.
	rc = lvm_translate_lv_name_to_region_name(lv->lv_name, region_name);
	if (rc) {
		lvm_deallocate_logical_volume(new_volume);
		RETURN(NULL);
	}

	// An EVMS region to represent this volume. Add this new region
	// to the "produced" list in the container.
	rc = lvm_engine->allocate_region(region_name, &new_volume->region);
	if (rc) {
		lvm_deallocate_logical_volume(new_volume);
		RETURN(NULL);
	}

	rc = lvm_append_region_to_container(new_volume->region, group->container);
	if (rc) {
		lvm_deallocate_logical_volume(new_volume);
		RETURN(NULL);
	}

	new_volume->region->object_type		= REGION;
	new_volume->region->data_type		= DATA_TYPE;
	new_volume->region->plugin		= lvm_plugin;
	new_volume->region->flags		= ((lv->lv_access & (LV_SNAPSHOT|LV_SNAPSHOT_ORG)) ? (SOFLAG_READ_ONLY | SOFLAG_MUST_BE_VOLUME) : 0) |
						  ((!(lv->lv_access & LV_WRITE)) ? SOFLAG_READ_ONLY : 0);
	new_volume->region->size		= lv->lv_size;
	new_volume->region->private_data	= new_volume;

	minor_in_use[new_volume->minor]		= TRUE;

	LOG("Created region %s\n", new_volume->region->name);

	RETURN(new_volume);
}


/* Function: lvm_deallocate_logical_volume
 *
 *	Free all memory usage for this volume. Remove it from its group's
 *	list of volumes if necessary.
 */
int lvm_deallocate_logical_volume( lvm_logical_volume_t * volume )
{
	lvm_volume_group_t	* group = volume->group;
	storage_object_t	* region = volume->region;

	LOG_ENTRY;

	// Delete the LE map
	if ( volume->le_map ) {
		lvm_engine->engine_free(volume->le_map);
		volume->le_map = NULL;
	}

	// Remove the region from the container and delete
	if ( region ) {
		lvm_clear_child_list(region);
		lvm_remove_region_from_container(region);
		lvm_engine->free_region(region);
		volume->region = NULL;
	}

	// Only the freespace volume has an lv field that was allocated. All
	// other volumes point to entries in the group's lv_array.
	if ( volume->number == 0 && volume->lv ) {
		lvm_engine->engine_free(volume->lv);
	}

	// Remove this volume from the group's list
	if ( group && group->volume_list[volume->number] == volume ) {
		group->volume_list[volume->number] = NULL;
		group->volume_count--;
	}

	// Delete the volume itself
	minor_in_use[volume->minor]	= FALSE;
	volume->number			= 0;
	volume->flags			= 0;
	volume->minor			= 0;
	volume->group			= NULL;
	volume->lv			= NULL;
	volume->snapshot_next		= NULL;
	volume->snapshot_org		= NULL;

	lvm_engine->engine_free(volume);

	RETURN(0);
}


/* Function: lvm_mark_volume_dirty
 *
 *	When an LV is newly created or modified, it will need to be marked
 *	dirty to indicate it requires some special processing. This will
 *	include deleting all associations to its underlying segments, and
 *	setting the dirty flag so its LE maps will be rebuilt.
 */
static int lvm_mark_volume_dirty( lvm_logical_volume_t * volume )
{
	int rc;

	LOG_ENTRY;

	if ( (rc = lvm_clear_child_list(volume->region)) ) {
		LOG_ERROR("Error marking region %s dirty\n", volume->region->name);
		RETURN(rc);
	}

	volume->flags |= LVM_LV_FLAG_DIRTY;

	RETURN(0);
}


/* Function: lvm_create_freespace_volume
 *
 *	Every group must have one volume that represents the unused space
 *	in that group. This is done by having a "dummy" volume that is
 *	separate from the group's regular volume list. The size of the
 *	freespace volume is determined by subtracting the number of allocated
 *	PEs in the entire group from the total number of PEs in the group. Then
 *	when the LE maps are built, any PE found with an LV number of zero will
 *	be added to the LE map of the freespace volume.
 */
int lvm_create_freespace_volume( lvm_volume_group_t * group )
{
	lv_disk_t		* free_lv;
	lvm_logical_volume_t	* free_volume;

	LOG_ENTRY;

	if ( group->freespace ) {
		RETURN(0);
	}

	// Create a dummy lv_disk_t structure and initialize
	free_lv = lvm_engine->engine_alloc(sizeof(lv_disk_t));
	if ( ! free_lv ) {
		LOG_CRITICAL("Memory error creating LV structure for Freespace region for container %s.\n", group->container->name);
		RETURN(ENOMEM);
	}

	// Need to make the freespace's name look like a legal LVM lv name.
	lvm_make_lv_name("Freespace", group, free_lv->lv_name);

	free_lv->lv_allocated_le = group->vg->pe_total - group->vg->pe_allocated;
	free_lv->lv_size	= free_lv->lv_allocated_le * group->vg->pe_size;
	free_lv->lv_number	= -1;

	// Create the volume structure
	if ( ! (free_volume = lvm_allocate_logical_volume(free_lv, group)) ) {
		LOG_CRITICAL("Memory error creating region %s.\n", free_lv->lv_name);
		RETURN(ENOMEM);
	}
	free_volume->region->data_type = FREE_SPACE_TYPE;

	group->freespace = free_volume;

	RETURN(0);
}


/* Function: lvm_update_freespace_volume
 *
 *	Update the freespace volume after a change in the volume group. Reset
 *	the freespace size accordingly, delete and recreate the LE map, and
 *	rebuild the LE maps for the group.
 */
int lvm_update_freespace_volume( lvm_volume_group_t * group )
{
	lvm_logical_volume_t	* freespace = group->freespace;
	int			rc;

	LOG_ENTRY;

	freespace->lv->lv_allocated_le	= group->vg->pe_total - group->vg->pe_allocated;
	freespace->lv->lv_size		= freespace->lv->lv_allocated_le * group->vg->pe_size;
	freespace->region->size		= freespace->lv->lv_size;

	lvm_mark_volume_dirty(freespace);

	lvm_deallocate_le_map(freespace);
	if ( (rc = lvm_allocate_le_map(freespace)) ) {
		LOG_CRITICAL("Memory error creating LE map for region %s\n", freespace->region->name);
		RETURN(rc);
	}
	lvm_build_le_maps(group);

	RETURN(0);
}


/* Function: lvm_update_expanded_volume
 */
int lvm_update_expanded_volume( lvm_logical_volume_t	* volume,
				lvm_lv_expand_options_t	* lv_opts )
{
	int rc;

	LOG_ENTRY;

	volume->lv->lv_allocated_le += lv_opts->add_extents;
	volume->lv->lv_size += lv_opts->add_size;
	volume->region->size = volume->lv->lv_size;

	lvm_mark_volume_dirty(volume);

	lvm_deallocate_le_map(volume);
	if ( (rc = lvm_allocate_le_map(volume)) ) {
		LOG_CRITICAL("Memory error creating LE map for region %s\n", volume->region->name);
		RETURN(rc);
	}

	RETURN(0);
}


/* Function: lvm_update_shrunk_volume
 */
int lvm_update_shrunk_volume(	lvm_logical_volume_t	* volume,
				u_int32_t		remove_extents )
{
	int rc;

	LOG_ENTRY;

	volume->lv->lv_allocated_le -= remove_extents;
	volume->lv->lv_size -= remove_extents * volume->group->vg->pe_size;
	volume->region->size = volume->lv->lv_size;

	lvm_mark_volume_dirty(volume);

	lvm_deallocate_le_map(volume);
	rc = lvm_allocate_le_map(volume);
	if (rc) {
		LOG_CRITICAL("Memory error creating LE map for region %s\n", volume->region->name);
		RETURN(rc);
	}

	RETURN(0);
}


/* Function: lvm_build_le_maps
 *
 *	After all logical volumes have been discovered, the mappings from
 *	logical extents to physical extents must be constructed. Each PV
 *	contains an on-disk-map of its PEs. Each PE map entry contains the
 *	logical volume number and the logical extent number on that volume.
 *	The maps in the logical volumes are the reverse. Each entry contains
 *	a pointer to an entry in the group's pv_list, and a sector offset of
 *	the PE that the LE maps to.
 *	In the PE map, an LV number of zero indicates an unused PE. This PE
 *	will be added as an LE for the freespace volume. A running count is
 *	kept of the current LE for the freespace volume to figure out where
 *	each free PE should be mapped.
 */
int lvm_build_le_maps( lvm_volume_group_t * group )
{
	pv_disk_t	* pv;
	pe_disk_t	* pe_map;
	lsn_t		first_pe_sector;
	lsn_t		offset;
	u_int32_t	lv_number;
	u_int32_t	le_number;
	u_int32_t	freespace_current_le = 0;
	int		i, j;

	LOG_ENTRY;

	LOG_DETAILS("Building LE maps for container %s\n", group->container->name);

	// For every PV in this group.
	for ( i = 1; i <= MAX_PV; i++ ) {
		if ( ! group->pv_list[i] ) {
			continue;
		}
		pv	= group->pv_list[i]->pv;
		pe_map	= group->pv_list[i]->pe_map;

		// Calculate the sector number of the first PE on the PV.
		first_pe_sector = lvm_get_pe_start(pv, group);

		// For every entry in the PE map, calculate the PE's sector
		// offset and update the correct LV's LE map. LV number of 0
		// marks an unused PE. Also, use this opportunity to set up
		// the child and parent lists in the segments and regions.
		for ( j = 0; j < pv->pe_total; j++ ) {
			lv_number = pe_map[j].lv_num;
			offset = j * pv->pe_size + first_pe_sector;
			if ( lv_number ) {
				if ( group->volume_list[lv_number] &&
				     group->volume_list[lv_number]->flags & LVM_LV_FLAG_DIRTY ) {
					le_number = pe_map[j].le_num;
					group->volume_list[lv_number]->le_map[le_number].owning_pv = group->pv_list[i];
					group->volume_list[lv_number]->le_map[le_number].pe_number = j;
					group->volume_list[lv_number]->le_map[le_number].pe_sector_offset = offset;
					lvm_append_region_to_segment(group->volume_list[lv_number]->region, group->pv_list[i]->segment);
				}
			}
			else  {
				if ( group->freespace->lv->lv_allocated_le == 0 ) {
					LOG_SERIOUS("Found a free PE, but freespace is supposed to be empty!\n");
					LOG_SERIOUS("Container %s, PV %s, PE %d\n",
						group->container->name, group->pv_list[i]->segment->name, j);
				}
				else {
					// Unused PEs are added to the freespace region.
					group->freespace->le_map[freespace_current_le].owning_pv = group->pv_list[i];
					group->freespace->le_map[freespace_current_le].pe_number = j;
					group->freespace->le_map[freespace_current_le].pe_sector_offset = offset;
					lvm_append_region_to_segment(group->freespace->region, group->pv_list[i]->segment);
					freespace_current_le++;
				}
			}
		}
	}

	RETURN(0);
}


/* Function: lvm_check_le_maps
 */
int lvm_check_le_maps(	lvm_volume_group_t	* group,
			boolean			final_call )
{
	lvm_logical_volume_t	* volume;
	int			i, j, count;

	LOG_ENTRY;
	LOG_DETAILS("Verifying LE maps for container %s.\n", group->container->name);

	for ( i = 1; i <= MAX_LV; i++ ) {
		if ( ! group->volume_list[i] ) {
			continue;
		}
		volume = group->volume_list[i];

		for ( j = 0, count = 0; j < volume->lv->lv_allocated_le; j++ ) {
			if ( ! volume->le_map[j].owning_pv ||
			     ! volume->le_map[j].pe_sector_offset ) {
				count++;
			}
		}
		if ( count ) {
			if ( final_call ) {
				LOG_ERROR("Region %s has an incomplete LE map.\n", volume->region->name);
				LOG_ERROR("     Missing %d out of %d LEs.\n", count, volume->lv->lv_allocated_le);
			}
			volume->flags |= LVM_LV_FLAG_INCOMPLETE;
		}
		else {
			volume->flags &= (~LVM_LV_FLAG_INCOMPLETE & ~LVM_LV_FLAG_DIRTY);
		}
	}

	RETURN(0);
}


/* Function: lvm_export_logical_volumes
 *
 *	Find any new LVM logical volumes and place them on the discovery
 *	output list. If the final_call flag is set, also place any incomplete
 *	volumes on the discovery list, and mark them as read-only.
 */
int lvm_export_logical_volumes( dlist_t	regions,
				boolean	final_call )
{
	lvm_volume_group_t	* group;
	lvm_logical_volume_t	* volume;
	int			count = 0;
	int			i, rc;

	LOG_ENTRY;

	// Examine every volume group
	FOR_EACH(group, lvm_group_list) {

		// First export the freespace volumes.
		volume = group->freespace;
		if ( ! (volume->flags & LVM_LV_FLAG_EXPORTED) ) {
			if ( ! lvm_add_object_to_list(volume->region, regions) ) {
				volume->flags |= LVM_LV_FLAG_EXPORTED;
				count++;
				LOG("Exporting region %s\n", volume->region->name);
			}
		}

		// Now examine every regular LV in this group.
		for ( i = 0; i <= MAX_LV; i++ ) {
			volume = group->volume_list[i];

			// Only export volumes without the EXPORTED flag set.
			// Only export incomplete volumes on final discovery.
			if ( ! volume ||
			     volume->flags & LVM_LV_FLAG_EXPORTED ||
			     (!final_call && volume->flags & LVM_LV_FLAG_INCOMPLETE) ) {
				continue;
			}

			if ( lvm_add_object_to_list(volume->region, regions) ) {
				continue;
			}

			volume->flags |= LVM_LV_FLAG_EXPORTED;
			count++;

			if ( volume->flags & LVM_LV_FLAG_INCOMPLETE ) {
				volume->region->flags |= SOFLAG_READ_ONLY;
			}

			LOG("Exporting region %s\n", volume->region->name);
		}
	}

	RETURN(count);
}


/* Function: lvm_get_freespace_volume
 *
 *	Get the first (and only) item from the input list, which should be
 *	the freespace region specified for the create. Verify that it is
 *	owned by LVM and is actually freespace.
 */
int lvm_get_freespace_volume(	dlist_t			freespace_region_list,
				lvm_logical_volume_t	** freespace_volume )
{
	storage_object_t	* freespace_region;
	lvm_logical_volume_t	* free_vol;
	long			tag;
	int			size, count = 0;

	LOG_ENTRY;

	// Extract the specified freespace region from the input list. There
	// must be only a single item on this list.
	GetListSize(freespace_region_list, &count);
	if ( count != 1 ) {
		LOG_ERROR("Must specify exactly one freespace region\n");
		RETURN(EINVAL);
	}

	GoToStartOfList(freespace_region_list);
	BlindGetObject(freespace_region_list, &size, &tag, NULL, FALSE, (void**)&freespace_region);

	// The selected object must be owned by LVM,
	// and must be a freespace region.
	if ( freespace_region->plugin != lvm_plugin ) {
		LOG_ERROR("Region %s does not belong to LVM\n", freespace_region->name);
		RETURN(EINVAL);
	}

	free_vol = freespace_region->private_data;
	if ( free_vol->group->freespace != free_vol ) {
		LOG_ERROR("Region %s is not a freespace region\n", freespace_region->name);
		RETURN(EINVAL);
	}

	*freespace_volume = free_vol;

	RETURN(0);
}


/* Function: lvm_check_lv_name
 *
 *	Run through the specified volume group and check if the specified LV
 *	name is already in use. It is assumed that "name" is ONLY the volume
 *	name (no lvm/group_name).
 */
int lvm_check_lv_name(	char			* name,
			lvm_volume_group_t	* group )
{
	char	tmp_name[NAME_LEN] = {0};
	int	i;

	LOG_ENTRY;

	// No empty strings allowed.
	if ( name[0] == 0 ) {
		LOG_ERROR("Must specify a name for the new region\n");
		RETURN(EINVAL);
	}

	// Convert "name" to "lvm/vg_name/name".
        strncpy(tmp_name, group->container->name, NAME_LEN-1);
	strncat(tmp_name, "/", NAME_LEN-strlen(tmp_name)-1);
	strncat(tmp_name, name, NAME_LEN-strlen(tmp_name));

	// Search through all LVs in this group.
	for ( i = 1; i <= MAX_LV; i++ ) {
		if ( group->volume_list[i] &&
		     ! strncmp(tmp_name, group->volume_list[i]->region->name, NAME_LEN) ) {
			LOG_ERROR("LV name %s already exists in container %s\n", name, group->container->name);
			RETURN(EEXIST);
		}
	}

	RETURN(0);
}


/* Function: lvm_check_lv_size
 *
 *	Make sure that lv_size is a multiple of the group's PE size. If it
 *	isn't, round up to the next PE size multiple.
 */
int lvm_check_lv_size(	u_int32_t	* lv_size,
			u_int32_t	pe_size )
{
	int rc = 0;

	LOG_ENTRY;

	if ( *lv_size % pe_size ) {
		LOG_WARNING("LV Size (%ld) is not a multiple of the PE size %ld\n", *lv_size, pe_size);
		*lv_size = round_up(*lv_size, pe_size);
		LOG_WARNING("Rounding LV Size up to %ld\n", *lv_size);
		rc = -1;
	}

	RETURN(rc);
}


/* Function: lvm_compare_lv_size_and_extents
 *
 *	Verify that the specified lv_size and extents match. If one is not
 *	set, calculate it from the other. If neither are set, return error.
 *
 *	The lvm_check_lv_size function should be called before this to
 *	make sure lv_size is a multiple of the group's pe_size.
 */
int lvm_compare_lv_size_and_extents(	u_int32_t	* lv_size,
					u_int32_t	* extents,
					u_int32_t	pe_size )
{
	int rc = 0;

	LOG_ENTRY;

	if ( *lv_size ) {
		if ( *extents ) {
			if ( *lv_size != *extents * pe_size ) {
				LOG_ERROR("Mismatch in LV Size and Extents\n");
				LOG_ERROR("LV Size: %ld\n", *lv_size);
				LOG_ERROR("Extents: %ld (total size: %ld)\n", *extents, *extents * pe_size);
				LOG_ERROR("Please specify only LV Size or only Extents\n");
				rc = EINVAL;
			}
		}
		else {
			*extents = *lv_size / pe_size;
		}
	}
	else if ( *extents ) {
		*lv_size = *extents * pe_size;
	}
	else {
		LOG_ERROR("Must specify either LV Size or Extents\n");
		rc = EINVAL;
	}

	RETURN(rc);
}


/* Function: lvm_check_stripe_size
 *
 *	During creation of a striped LV, the user-specified stripe size must
 *	be a power of 2 betweeen 8k and 512k, and also less than the PE size
 *	for that group. If it is out of range, reset to the nearest correct
 *	value. If it is not a power of 2, round down until a suitable value
 *	is found.
 */
int lvm_check_stripe_size(	u_int32_t	* stripe_size,
				u_int32_t	pe_size )
{
	unsigned long mask = 1;

	LOG_ENTRY;

	if ( *stripe_size ) {
		// Check lower bound
		if ( *stripe_size < LVM_MIN_STRIPE_SIZE ) {
			LOG_WARNING("Stripe size %d is below lower limit.\n", *stripe_size);
			*stripe_size = LVM_MIN_STRIPE_SIZE;
			LOG_WARNING("Resetting stripe size to %d.\n", *stripe_size);
		}
		// Check upper bound
		else if ( *stripe_size > LVM_MAX_STRIPE_SIZE ) {
			LOG_WARNING("Stripe size %d is above upper limit.\n", *stripe_size);
			*stripe_size = LVM_MAX_STRIPE_SIZE;
			LOG_WARNING("Resetting stripe size to %d.\n", *stripe_size);
		}
		// Check that stripe_size is a power of 2.
		else if ( (*stripe_size & (*stripe_size - 1)) ) {
			LOG_WARNING("Stripe size %d not a power of 2.\n", *stripe_size);
			while ( (*stripe_size & (*stripe_size - 1)) ) {
				*stripe_size =  *stripe_size & ~mask;
				mask = mask << 1;
			}
			LOG_WARNING("Rounding stripe size down to %d.\n", *stripe_size);
		}
	}
	else {
		// Default stripe-size is 16k
		*stripe_size = LVM_DEFAULT_STRIPE_SIZE;
	}

	// Check against PE size.
	if ( *stripe_size > pe_size ) {
		LOG_WARNING("Stripe size %d is greater than PE size %d.\n", *stripe_size, pe_size);
		*stripe_size = pe_size;
		LOG_WARNING("Resetting stripe size to %d.\n", *stripe_size);
	}

	RETURN(0);
}


/* Function: lvm_check_chunk_size
 *
 *	During creation of a snapshot LV, the user-specified chunk size must
 *	be a power of 2 betweeen 8k and 1M, and also less than the PE size
 *	for that group. If it is out of range, reset to the nearest correct
 *	value. If it is not a power of 2, round down until a suitable value
 *	is found.
 */
int lvm_check_chunk_size(u_int32_t	* chunk_size,
			u_int32_t	pe_size )
{
	unsigned long mask = 1;

	LOG_ENTRY;

	if ( *chunk_size ) {
		// Check lower bound
		if ( *chunk_size < LVM_SNAPSHOT_MIN_CHUNK ) {
			LOG_WARNING("Chunk size %d is below lower limit.\n", *chunk_size);
			*chunk_size = LVM_SNAPSHOT_MIN_CHUNK;
			LOG_WARNING("Resetting chunk size to %d.\n", *chunk_size);
		}
		// Check upper bound
		else if ( *chunk_size > LVM_SNAPSHOT_MAX_CHUNK ) {
			LOG_WARNING("Chunk size %d is above upper limit.\n", *chunk_size);
			*chunk_size = LVM_SNAPSHOT_MAX_CHUNK;
			LOG_WARNING("Resetting chunk size to %d.\n", *chunk_size);
		}
		// Check that chunk_size is a power of 2.
		else if ( (*chunk_size & (*chunk_size - 1)) ) {
			LOG_WARNING("Chunk size %d not a power of 2.\n", *chunk_size);
			while ( (*chunk_size & (*chunk_size - 1)) ) {
				*chunk_size =  *chunk_size & ~mask;
				mask = mask << 1;
			}
			LOG_WARNING("Rounding chunk size down to %d.\n", *chunk_size);
		}
	}
	else {
		// Default stripe-size is 64k
		*chunk_size = LVM_SNAPSHOT_DEF_CHUNK;
	}

	// Check against PE size.
	if ( *chunk_size > pe_size ) {
		LOG_WARNING("Chunk size %d is greater than PE size (%d).\n", *chunk_size, pe_size);
		*chunk_size = pe_size;
		LOG_WARNING("Resetting chunk size to %d.\n", *chunk_size);
	}

	RETURN(0);
}


/* Function: lvm_find_free_lv_number
 *
 *	Search through the list of LVs in this group and return the first
 *	unused LV number.
 */
static int lvm_find_free_lv_number( lvm_volume_group_t * group )
{
	int i;

	LOG_ENTRY;

	for ( i = 1; i <= MAX_LV; i++ ) {
		if ( ! group->volume_list[i] ) {
			RETURN(i);
		}
	}
	LOG_ERROR("Container %s has maximum number of regions.\n", group->container->name);
	RETURN(-1);
}


/* Function: lvm_find_free_minor_number
 *
 *	Determine the next minor number that is available for an LV.
 */
static int lvm_find_free_minor_number(void)
{
	int	minor = 0;

	LOG_ENTRY;

	for ( minor = 0; minor < MAX_LV; minor++ ) {
		if ( ! minor_in_use[minor] ) {
			RETURN(minor);
		}
	}

	LOG_ERROR("All LVM minor numbers in use.\n");
	RETURN(-1);
}


/* Function: lvm_initialize_new_lv
 *
 *	During creation of a new logical volume, we need to fill in an LV disk
 *	structure with appropriate values based on the user's input.
 */
int lvm_initialize_new_lv(	lvm_lv_create_options_t	* lv_opts,
				lvm_volume_group_t	* group,
				lv_disk_t		** lv )
{
	int		lv_number;
	int		minor;

	LOG_ENTRY;

	// Get an LV number
	lv_number = lvm_find_free_lv_number(group);
	if ( lv_number <= 0 ) {
		LOG_ERROR("Could not initialize LV metadata\n");
		RETURN(ENOSPC);
	}

	// Get a minor number
	minor = lvm_find_free_minor_number();
	if ( minor < 0 ) {
		LOG_ERROR("Could not initialize LV metadata\n");
		RETURN(ENOSPC);
	}

	*lv = &(group->lv_array[lv_number - 1]);
	lvm_clear_lv(*lv);

	// Copy the LV and VG names
	lvm_make_lv_name(lv_opts->lv_name, group, (*lv)->lv_name);
	lvm_translate_container_name_to_vg_name(group->container->name, (*lv)->vg_name);

	// Set access flags
	(*lv)->lv_access = LV_READ;
	if ( ! lv_opts->read_only ) {
		(*lv)->lv_access |= LV_WRITE;
	}
	if ( lv_opts->snapshot ) {
		(*lv)->lv_access |= LV_SNAPSHOT;
	}

	// Fill in remaining fields.
	(*lv)->lv_status		= LV_ACTIVE;
	(*lv)->lv_open			= 0;
	(*lv)->lv_dev			= MKDEV(LVM_BLK_MAJOR, minor);
	(*lv)->lv_number		= lv_number - 1;
	(*lv)->lv_mirror_copies		= 0;
	(*lv)->lv_recovery		= 0;
	(*lv)->lv_schedule		= 0;
	(*lv)->lv_size			= lv_opts->lv_size;
	(*lv)->lv_snapshot_minor	= (lv_opts->snapshot) ? lv_opts->snapshot_org->minor : 0;
	(*lv)->lv_chunk_size		= lv_opts->chunk_size;
	(*lv)->dummy			= 0;
	(*lv)->lv_allocated_le		= lv_opts->extents;
	(*lv)->lv_stripes		= lv_opts->stripes;
	(*lv)->lv_stripesize		= lv_opts->stripe_size;
	(*lv)->lv_badblock		= 0;
	(*lv)->lv_allocation		= lv_opts->contiguous;
	(*lv)->lv_io_timeout		= 0;
	(*lv)->lv_read_ahead		= LVM_MAX_READ_AHEAD;

	RETURN(0);
}


/* Function: lvm_clear_lv
 *
 *	Erases the contents of the specified lv_disk_t structure.
 */
int lvm_clear_lv( lv_disk_t * lv )
{
	LOG_ENTRY;
	memset(lv, 0, sizeof(lv_disk_t));
	RETURN(0);
}



/****** Checking Available Extents for Volumes ******/



/* Function: lvm_check_available_extents_simple
 *
 *	Check that the specified group has the specified number of unallocated
 *	extents. If any PVs have been specified, only check available PEs on
 *	those PVs. The pv_entries parameter must be a NULL-terminated array of
 *	PV pointers.
 */
static int lvm_check_available_extents_simple(	lvm_volume_group_t	* group,
						u_int32_t		desired_extents,
						lvm_physical_volume_t	* pv_entries[] )
{
	int available_extents = 0;
	int rc = 0;
	int i = 0;

	LOG_ENTRY;

	if ( ! pv_entries[0] ) {
		// No PVs specified.
		available_extents = group->freespace->lv->lv_allocated_le;
	}
	else {
		// Sum up available PEs from specified PVs.
		while ( pv_entries[i] ) {
			available_extents += pv_entries[i]->pv->pe_total - pv_entries[i]->pv->pe_allocated;
			i++;
		}
	}

	if ( available_extents < desired_extents ) {
		LOG_ERROR("Requested %d extents.\n", desired_extents);
		LOG_ERROR("Container %s only has %d extents available.\n", group->container->name, available_extents);
		rc = ENOSPC;
	}

	RETURN(rc);
}


/* Function: lvm_check_available_extents_striped
 *
 *	Check that the specified group has enough available extents to
 *	allocate to a striped volume with the specified number of stripes.
 *	The options must have already been verified before calling this
 *	function, so that extents is a multiple of stripes.
 */
static int lvm_check_available_extents_striped( lvm_volume_group_t	* group,
						u_int32_t		extents,
						u_int32_t		stripes,
						lvm_physical_volume_t	* pv_list[] )
{
	u_int32_t	extents_per_stripe = extents / stripes;
	u_int32_t	stripes_found = 0;
	int		i;

	LOG_ENTRY;

	if ( ! pv_list[0] ) {
		// No PVs specified. Use the group's list.
		pv_list = group->pv_list;
	}

	// Check each PV for sufficient extents_per_stripe. If found, increment
	// stripes_found. When stripes_found == stripes, we know we have enough
	// extents.
	for ( i = 0; i <= MAX_PV && stripes_found < stripes; i++ ) {
		if ( pv_list[i] &&
		     pv_list[i]->pv->pe_total - pv_list[i]->pv->pe_allocated >= extents_per_stripe ) {
			stripes_found++;
		}
	}

	if ( stripes_found < stripes ) {
		LOG_ERROR("Requested %d extents on %d stripes (%d extents per stripe)\n", extents, stripes, extents_per_stripe);
		LOG_ERROR("Only have %d stripes available with %d extents each.\n", stripes_found, extents_per_stripe);
		RETURN(ENOSPC);
	}

	RETURN(0);
}


/* Function: lvm_check_available_extents_contiguous
 *
 *	In the LVM plugin, "contiguous" means a volume that resides on a single
 *	PV, with LEs that map to contiguous PEs. To check for available
 *	contiguous extents, just make sure there is a single PV that has enough
 *	contiguous extents to fulfil the desired number.
 */
static int lvm_check_available_extents_contiguous(lvm_volume_group_t	* group,
						u_int32_t		extents,
						lvm_physical_volume_t	* pv_list[],
						lvm_physical_volume_t	** result_pv,
						u_int32_t		* result_pe )
{
	int con_extents = 0;
	int i, j, k, rc = ENOSPC;

	LOG_ENTRY;

	if ( ! pv_list[0] ) {
		// No PVs specified. Use the group's list.
		pv_list = group->pv_list;
	}

	for ( i = 0; rc && i <= MAX_PV; i++ ) {
		if ( pv_list[i] ) {
			// Start with simple check for available extents
			if ( extents > pv_list[i]->pv->pe_total - pv_list[i]->pv->pe_allocated ) {
				continue;
			}

			for ( j = 0; rc && j < pv_list[i]->pv->pe_total; j++ ) {
				if ( pv_list[i]->pe_map[j].lv_num == 0 ) {

					// See if this PE begins a consecutive
					// run of suitable size.
					for ( con_extents = 1, k = j + 1;
					      con_extents < extents && pv_list[i]->pe_map[k].lv_num == 0;
					      con_extents++, k++ ) ;

					// Found a suitable consecutive run.
					if ( con_extents == extents ) {
						if ( result_pv ) {
							*result_pv = pv_list[i];
						}
						if ( result_pe ) {
							*result_pe = j;
						}
						rc = 0;
					}
					else {
						j = k;
					}
				}
			}
		}
	}

	if (rc) {
		LOG_ERROR("Requested %d extents.\n", extents);
		LOG_ERROR("Not enough contiguous extents available for request.\n");
	}

	RETURN(rc);
}


/* Function: lvm_check_available_expand_extents_striped
 *
 *	Striped volumes can only be expanded if all of the underlying PVs
 *	each have the appropriate number of extents available.
 */
static int lvm_check_available_expand_extents_striped(	lvm_logical_volume_t	* volume,
							u_int32_t		add_extents)
{
	storage_object_t	* segment;
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		extents_per_stripe = add_extents / volume->lv->lv_stripes;
	int			rc;

	LOG_ENTRY;

	// Check each PV in this volume's child-list to see if it has
	// enough free extents for the expand.
	for ( rc = GoToStartOfList(volume->region->child_objects);
	      !rc && (segment = lvm_get_list_item(volume->region->child_objects));
	      rc = NextItem(volume->region->child_objects) ) {
		pv_entry = lvm_get_pv_for_segment(segment);
		if ( pv_entry->pv->pe_total - pv_entry->pv->pe_allocated < extents_per_stripe ) {
			LOG_ERROR("Not enough available extents on PV %s to expand region %s.\n",
				pv_entry->segment->name, volume->region->name);
			RETURN(EINVAL);
		}
	}

	RETURN(0);
}


/* Function: lvm_check_available_expand_extents_contiguous
 *
 *	Contiguous volumes can only be expanded if there is contiguous space
 *	available immediately following the volume.
 */
static int lvm_check_available_expand_extents_contiguous( lvm_logical_volume_t	* volume,
							u_int32_t		add_extents)
{
	lvm_physical_volume_t	* pv_entry;
	int			count, i;

	LOG_ENTRY;

	pv_entry = volume->le_map[volume->lv->lv_allocated_le - 1].owning_pv;
	i = volume->le_map[volume->lv->lv_allocated_le - 1].pe_number + 1;

	for ( count = 0; count < add_extents; count++ ) {
		if ( i >= pv_entry->pv->pe_total ||
		     pv_entry->pe_map[i].lv_num ) {
			LOG_ERROR("Not enough contiguous extents to extend region %s\n", volume->region->name);
			RETURN(EINVAL);	
		}
		i++;
	}

	RETURN(0);
}


/* Function: lvm_check_available_extents
 *
 *	Determine which kind of allocation is going to be used, and call the
 *	appropriate function to check for available space.
 */
int lvm_check_available_extents(lvm_volume_group_t	* group,
				lvm_lv_create_options_t	* lv_opts )
{
	int rc = 0;

	LOG_ENTRY;

	if ( lv_opts->stripe_size > 1 ) {
		rc = lvm_check_available_extents_striped(group, lv_opts->extents, lv_opts->stripes, lv_opts->pv_entries);
	}
	else if ( lv_opts->contiguous ) {
		rc = lvm_check_available_extents_contiguous(group, lv_opts->extents, lv_opts->pv_entries, NULL, NULL);
	}
	else {
		rc = lvm_check_available_extents_simple(group, lv_opts->extents, lv_opts->pv_entries);
	}

	RETURN(rc);
}


/* Function: lvm_check_available_expand_extents
 *
 *	Determine which kind of allocation the existing volume uses, and call
 *	the appropriate function to check for available space.
 */
int lvm_check_available_expand_extents(	lvm_logical_volume_t	* volume,
					lvm_lv_expand_options_t	* lv_opts )
{
	int rc;

	LOG_ENTRY;

	if ( volume->lv->lv_stripes > 1 ) {
		rc = lvm_check_available_expand_extents_striped(volume, lv_opts->add_extents);
	}
	else if ( volume->lv->lv_allocation & LV_CONTIGUOUS ) {
		rc = lvm_check_available_expand_extents_contiguous(volume, lv_opts->add_extents);
	}
	else {
		rc = lvm_check_available_extents_simple(volume->group, lv_opts->add_extents, lv_opts->pv_entries);
	}

	RETURN(rc);
}



/****** Allocating Extents to Volumes ******/



/* Function: lvm_allocate_extents_simple
 *
 *	Run through each PV in this volume's group and look for unused PEs.
 *	Assign unused PEs to this volume until total is reached. This scheme
 *	implements a simple first-available allocation of PEs. The current_le
 *	parameter is the first new LE of the LV. For new LVs, this should be
 *	zero. For LV expands, this should be lv_allocated_le + 1. le_total
 *	should be the total number of extents after the create/expand.
 *	For PV-specific allocation, pv_entries should be a NULL-terminated
 *	array of PV pointers.
 *
 *	This function must NOT be called unless lvm_check_available_extents has
 *	been called to verify that ample extents are available.
 */
static int lvm_allocate_extents_simple(	lvm_logical_volume_t	* volume,
					u_int32_t		le_total,
					lvm_physical_volume_t	* pv_list[] )
{
	lvm_volume_group_t	* group = volume->group;
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		current_le = 0;
	int			used_this_pv;
	int			i,j;

	LOG_ENTRY;

	if ( ! pv_list[0] ) {
		// No PVs specified. Use the group's list.
		pv_list = group->pv_list;
	}

	// Go through the list of PVs. For each PV, go through the list of PEs.
	// For every PE that does not map to an LV, record the LV and LE
	// numbers, and updated the appropriate PV entries.
	for ( i = 0; i <= MAX_PV && current_le < le_total; i++ ) {
		if ( (pv_entry = pv_list[i]) ) {
			used_this_pv = FALSE;
			for ( j = 0; j < pv_entry->pv->pe_total && current_le < le_total; j++ ) {
				if ( ! pv_entry->pe_map[j].lv_num ) {
					pv_entry->pe_map[j].lv_num = volume->number;
					pv_entry->pe_map[j].le_num = current_le;
					pv_entry->pv->pe_allocated++;
					current_le++;
					if ( ! used_this_pv ) {
						pv_entry->pv->lv_cur++;
						used_this_pv = TRUE;
					}
				}
			}
		}
	}

	if ( current_le != le_total ) {
		// This should never happen as long as we've checked for
		// available space ahead of time.
		LOG_SERIOUS("Could not allocate enough extents for region %s\n", volume->region->name);
		RETURN(ENOSPC);
	}

	RETURN(0);
}


/* Function: lvm_allocate_expand_extents_simple
 *
 *	Expanding simple volumes now has its own function for allocating
 *	the extents.
 */
static int lvm_allocate_expand_extents_simple(	lvm_logical_volume_t	* volume,
						u_int32_t		expand_extents,
						lvm_physical_volume_t	* pv_list[] )
{
	lvm_volume_group_t	* group = volume->group;
	lvm_physical_volume_t	* pv_entry;
	lvm_physical_volume_t	* other_pv;
	storage_object_t	* segment;
	u_int32_t		current_le = volume->lv->lv_allocated_le;
	u_int32_t		le_total = current_le + expand_extents;
	int			used_this_pv;
	int			i, j, rc;

	LOG_ENTRY;

	if ( ! pv_list[0] ) {
		// No PVs specified. Use the group's list.
		pv_list = group->pv_list;
	}

	// Go through the list of PVs. For each PV, go through the list of PEs.
	// For every PE that does not map to an LV, record the LV and LE
	// numbers, and updated the appropriate PV entries.
	for ( i = 0; i <= MAX_PV && current_le < le_total; i++ ) {
		if ( (pv_entry = pv_list[i]) ) {

			// Determine if the volume already has extents on
			// this PV.
			used_this_pv = FALSE;
			for ( rc = GoToStartOfList(volume->region->child_objects);
			      !rc && (segment = lvm_get_list_item(volume->region->child_objects));
			      rc = NextItem(volume->region->child_objects) ) {
				other_pv = lvm_get_pv_for_segment(segment);
				if ( other_pv == pv_entry ) {
					used_this_pv = TRUE;
					break;
				}
			}

			for ( j = 0; j < pv_entry->pv->pe_total && current_le < le_total; j++ ) {
				if ( ! pv_entry->pe_map[j].lv_num ) {
					pv_entry->pe_map[j].lv_num = volume->number;
					pv_entry->pe_map[j].le_num = current_le;
					pv_entry->pv->pe_allocated++;
					current_le++;
					if ( ! used_this_pv ) {
						pv_entry->pv->lv_cur++;
						used_this_pv = TRUE;
					}
				}
			}
		}
	}

	if ( current_le != le_total ) {
		// This should never happen as long as we've checked for
		// available space ahead of time.
		LOG_SERIOUS("Could not allocate enough extents for region %s\n", volume->region->name);
		RETURN(ENOSPC);
	}

	RETURN(0);
}


static int lvm_allocate_extents_contiguous(	lvm_logical_volume_t	* volume,
						u_int32_t		extents,
						lvm_physical_volume_t	* pv_list[] )
{
	lvm_physical_volume_t	* target_pv;
	u_int32_t		target_pe;
	u_int32_t		current_le;
	int			i, rc;

	LOG_ENTRY;

	if ( ! pv_list[0] ) {
		// No PVs specified. Use the group's list.
		pv_list = volume->group->pv_list;
	}

	if ( (rc = lvm_check_available_extents_contiguous(volume->group, extents, pv_list, &target_pv, &target_pe)) ) {
		// This should never happen as long as we check for available
		// extents before trying to allocate.
		LOG_SERIOUS("Could not allocate enough extents for region %s\n", volume->region->name);
		RETURN(rc);
	}

	// Now we have the target PV and starting PE, and we know that there is
	// a contiguous run starting there. So just walk through and allocate
	// the extents.
	for ( i = target_pe, current_le = 0; current_le < extents; i++, current_le++ ) {
		if ( target_pv->pe_map[i].lv_num ) {
			LOG_SERIOUS("Found an allocated PE in a set that was checked for contiguous!\n");
			LOG_SERIOUS("PV %s, PE %d\n", target_pv->segment->name, i);
			RETURN(ENOSPC);
		}
		target_pv->pe_map[i].lv_num = volume->number;
		target_pv->pe_map[i].le_num = current_le;
		target_pv->pv->pe_allocated++;
	}
	target_pv->pv->lv_cur++;

	RETURN(0);
}


/* Function: lvm_allocate_expand_extents_contiguous
 *
 *	Contiguous volumes can only be expanded into the freespace on the PV
 *	immediately following the existing volume.
 */
static int lvm_allocate_expand_extents_contiguous( lvm_logical_volume_t	* volume,
							u_int32_t	add_extents )
{
	lvm_physical_volume_t	* pv_entry;
	int			le_count, current_pe;

	LOG_ENTRY;

	pv_entry = volume->le_map[volume->lv->lv_allocated_le - 1].owning_pv;
	current_pe = volume->le_map[volume->lv->lv_allocated_le - 1].pe_number + 1;

	for ( le_count = 0; le_count < add_extents; le_count++ ) {
		if ( current_pe < pv_entry->pv->pe_total &&
		     pv_entry->pe_map[current_pe].lv_num == 0 ) {
			pv_entry->pe_map[current_pe].lv_num = volume->number;
			pv_entry->pe_map[current_pe].le_num = volume->lv->lv_allocated_le + le_count;
			pv_entry->pv->pe_allocated++;
			current_pe++;
		}
		else {
			// This should never happen as long as we check for
			// available space before trying to allocate.
			LOG_SERIOUS("Not enough contiguous extents to extend region %s\n", volume->region->name);
			RETURN(EINVAL);
		}
	}

	RETURN(0);
}


/* Function: lvm_allocate_extents_striped
 *
 *	This function implements a striped allocation scheme. 
 *
 *	This function must NOT be called unless lvm_check_available_extents has
 *	been called to verify that ample extents are available.
 */
static int lvm_allocate_extents_striped( lvm_logical_volume_t	* volume,
					u_int32_t		extents,
					u_int32_t		stripes,
					lvm_physical_volume_t	* pv_list[] )
{
	lvm_volume_group_t	* group = volume->group;
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		extents_per_stripe = extents / stripes;
	u_int32_t		stripes_found = 0;
	u_int32_t		current_le = 0;
	int			i, j, k;

	LOG_ENTRY;

	if ( ! pv_list[0] ) {
		// No PVs specified. Use the group's list.
		pv_list = group->pv_list;
	}

	// Run through the list of PVs. For each one that has enough extents
	// available, look for extents to allocate. When one is found, assign
	// it to the specified volume.
	for ( i = 0; i <= MAX_PV && stripes_found < stripes; i++ ) {
		if ( (pv_entry = pv_list[i]) &&
		     pv_entry->pv->pe_total - pv_entry->pv->pe_allocated >= extents_per_stripe ) {
			for ( j = 0, k = 0;
			      j < pv_entry->pv->pe_total && k < extents_per_stripe;
			      j++ ) {
				if ( ! pv_entry->pe_map[j].lv_num ) {
					pv_entry->pe_map[j].lv_num = volume->number;
					pv_entry->pe_map[j].le_num = current_le;
					pv_entry->pv->pe_allocated++;
					current_le++;
					k++;
				}
			}
			pv_entry->pv->lv_cur++;
			stripes_found++;
		}
	}

	if ( stripes_found < stripes ) {
		// This should never happen as long as we've checked for
		// available space ahead of time.
		LOG_SERIOUS("Requested %d extents on %d stripes (%d extents per stripe)\n", extents, stripes, extents_per_stripe);
		LOG_SERIOUS("Only have %d stripes available with %d extents each.\n", stripes_found, extents_per_stripe);
		RETURN(ENOSPC);
	}

	RETURN(0);
}


/* Function: lvm_allocate_expand_extents_striped
 *
 *	Striped volumes can only be expanded if all underlying PVs for the
 *	volume have enough available extents.
 */
static int lvm_allocate_expand_extents_striped(	lvm_logical_volume_t	* volume,
						u_int32_t		add_extents )
{
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		extents_per_stripe = add_extents / volume->lv->lv_stripes;
	u_int32_t		pe, le, new_le;
	int			i, j;

	LOG_ENTRY;

	// Rewrite all PE map entries in all the PVs for this LV.
	for ( le = new_le = 0; le < volume->lv->lv_allocated_le; le++ ) {
		pe = volume->le_map[le].pe_number;
		volume->le_map[le].owning_pv->pe_map[pe].le_num = new_le;
		new_le++;

		// When we get to the last PE for each PV, allocate new
		// extents on that PV.
		if ( le < volume->lv->lv_allocated_le &&
		     volume->le_map[le].owning_pv != volume->le_map[le+1].owning_pv ) {
			pv_entry = volume->le_map[le].owning_pv;
			for ( i = j = 0; i < pv_entry->pv->pe_total && j < extents_per_stripe; i++ ) {
				if ( ! pv_entry->pe_map[i].lv_num ) {
					pv_entry->pe_map[i].lv_num = volume->number;
					pv_entry->pe_map[i].le_num = new_le;
					pv_entry->pv->pe_allocated++;
					new_le++;
					j++;
				}
			}
		}
	}

	if ( new_le < volume->lv->lv_allocated_le + add_extents ) {
		// This should never happen as long as we check for
		// available space before trying to allocate.
		LOG_SERIOUS("Not enough extents to extend striped region %s\n", volume->region->name);
		RETURN(EINVAL);
	}

	RETURN(0);
}


/* Function: lvm_allocate_extents_to_volume
 *
 *	Decide what kind of allocation is needed, and call the appropriate
 *	function to perform the allocation of extents.
 */
int lvm_allocate_extents_to_volume(	lvm_logical_volume_t	* volume,
					lvm_lv_create_options_t	* lv_opts )
{
	int rc = 0;

	LOG_ENTRY;

	if ( lv_opts->stripes > 1 ) {
		rc = lvm_allocate_extents_striped(volume, lv_opts->extents, lv_opts->stripes, lv_opts->pv_entries);
	}
	else if ( lv_opts->contiguous ) {
		rc = lvm_allocate_extents_contiguous(volume, lv_opts->extents, lv_opts->pv_entries);
	}
	else {
		rc = lvm_allocate_extents_simple(volume, lv_opts->extents, lv_opts->pv_entries);
	}

	RETURN(rc);
}


/* Function: lvm_allocate_expand_extents_to_volume
 *
 *	Decide what kind of allocation is needed, and call the appropriate
 *	function to perform the allocation of extents.
 */
int lvm_allocate_expand_extents_to_volume(	lvm_logical_volume_t	* volume,
						lvm_lv_expand_options_t	* lv_opts )
{
	int rc;

	LOG_ENTRY;

	if ( volume->lv->lv_stripes > 1 ) {
		rc = lvm_allocate_expand_extents_striped(volume, lv_opts->add_extents);
	}
	else if ( volume->lv->lv_allocation & LV_CONTIGUOUS ) {
		rc = lvm_allocate_expand_extents_contiguous(volume, lv_opts->add_extents);
	}
	else {
		rc = lvm_allocate_expand_extents_simple(volume,
							lv_opts->add_extents,
							lv_opts->pv_entries );
	}

	RETURN(rc);
}


/* Function: lvm_deallocate_extents_from_volume
 *
 *	Traverse the LE map of the specified volume, and free each extent by
 *	clearing the appropriate PE map entry on the appropriate PV.
 */
int lvm_deallocate_extents_from_volume( lvm_logical_volume_t * volume )
{
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		pe_number;
	int			i;

	LOG_ENTRY;

	// For every logical extent
	for ( i = 0; i < volume->lv->lv_allocated_le; i++ ) {
		// Check that this LE points to a valid PV.
		if ( (pv_entry = volume->le_map[i].owning_pv) ) {
			pe_number = volume->le_map[i].pe_number;
			// Check that the LE entry and the PE entry match.
			if ( pv_entry->pe_map[pe_number].lv_num == volume->number ) {
				pv_entry->pe_map[pe_number].lv_num = 0;
				pv_entry->pe_map[pe_number].le_num = 0;
				pv_entry->pv->pe_allocated--;
				if ( ! (pv_entry->flags & LVM_PV_FLAG_LV_CUR_UPDATED) ) {
					// Only update the lv_cur variable once
					// per PV.
					pv_entry->pv->lv_cur--;
					pv_entry->flags |= LVM_PV_FLAG_LV_CUR_UPDATED;
				}
			}
			else {
				LOG_SERIOUS("LE map inconsistency in region %s (%d)\n", volume->region->name, volume->number);
				LOG_SERIOUS("LE %d maps to PV %s:PE %d\n", i, pv_entry->segment->name, pe_number);
				LOG_SERIOUS("PV %s:PE %s maps to LV %d:LE %d\n", pv_entry->segment->name, pe_number,
					pv_entry->pe_map[pe_number].lv_num, pv_entry->pe_map[pe_number].le_num);
			}
		}
	}

	// Clear all the LV_CUR flags in the PVs.
	for ( i = 1; i <= MAX_PV; i++ ) {
		if ( volume->group->pv_list[i] ) {
			volume->group->pv_list[i]->flags &= ~LVM_PV_FLAG_LV_CUR_UPDATED;
		}
	}

	RETURN(0);
}


/* Function: lvm_deallocate_shrink_extents_striped
 */
static int lvm_deallocate_shrink_extents_striped(lvm_logical_volume_t	* volume,
						u_int32_t		remove_extents )
{
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		current_extents_per_stripe;
	u_int32_t		new_extents_per_stripe;
	u_int32_t		pe, le, new_le, i;
	
	LOG_ENTRY;

	current_extents_per_stripe = volume->lv->lv_allocated_le / volume->lv->lv_stripes;
	new_extents_per_stripe = current_extents_per_stripe - remove_extents / volume->lv->lv_stripes;

	for ( le = new_le = i = 0; le < volume->lv->lv_allocated_le; ) {
		if ( i < new_extents_per_stripe ) {
			pv_entry = volume->le_map[le].owning_pv;
			pe = volume->le_map[le].pe_number;
			pv_entry->pe_map[pe].le_num = new_le;
			new_le++;
			le++;
			i++;
		}
		else {
			for ( ; i < current_extents_per_stripe; i++, le++ ) {
				pv_entry = volume->le_map[le].owning_pv;
				pe = volume->le_map[le].pe_number;

				pv_entry->pe_map[pe].lv_num = 0;
				pv_entry->pe_map[pe].le_num = 0;
				pv_entry->pv->pe_allocated--;
			}
			i = 0;
		}
	}

	RETURN(0);
}


/* Function: lvm_deallocate_shrink_extents_simple
 */
static int lvm_deallocate_shrink_extents_simple(lvm_logical_volume_t	* volume,
						u_int32_t		remove_extents )
{
	lvm_physical_volume_t	* pv_entry;
	u_int32_t		pe, le = volume->lv->lv_allocated_le - 1;
	int			i, j;

	LOG_ENTRY;

	for ( i = 0; i < remove_extents; i++, le-- ) {
		// Check that this LE points to a valid PV.
		pv_entry = volume->le_map[le].owning_pv;
		if ( pv_entry ) {
			pe = volume->le_map[le].pe_number;

			// Check that the LE entry and the PE entry match.
			if ( pv_entry->pe_map[pe].lv_num == volume->number ) {
				pv_entry->pe_map[pe].lv_num = 0;
				pv_entry->pe_map[pe].le_num = 0;
				pv_entry->pv->pe_allocated--;

				// See if we need to decrement lv_cur for
				// this PV.
				for ( j = 0; j < pv_entry->pv->pe_total; j++ ) {
					if ( pv_entry->pe_map[j].lv_num == volume->number ) {
						break;
					}
				}
				if ( j == pv_entry->pv->pe_total ) {
					pv_entry->pv->lv_cur--;
				}
			}
			else {
				LOG_SERIOUS("LE map inconsistency in region %s (%d)\n", volume->region->name, volume->number);
				LOG_SERIOUS("LE %d maps to PV %s:PE %d\n", le, pv_entry->segment->name, pe);
				LOG_SERIOUS("PV %s:PE %s maps to LV %d:LE %d\n", pv_entry->segment->name, pe,
					pv_entry->pe_map[pe].lv_num, pv_entry->pe_map[pe].le_num);
			}
		}
	}

	RETURN(0);
}


/* Function: lvm_deallocate_shrink_extents_from_volume
 *
 *	Traverse the LE map of the specified volume, and free the specified
 *	number of extents by clearing the appropriate PE map entry on the
 *	appropriate PV.
 */
int lvm_deallocate_shrink_extents_from_volume(	lvm_logical_volume_t	* volume,
						u_int32_t		remove_extents )
{
	int rc;

	LOG_ENTRY;

	if ( volume->lv->lv_stripes > 1 ) {
		rc = lvm_deallocate_shrink_extents_striped(volume, remove_extents);
	}
	else {
		rc = lvm_deallocate_shrink_extents_simple(volume, remove_extents);
	}

	RETURN(rc);
}

