/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvmregmgr
 * File: lvm_dlist.c
 *
 * Description: This file contains all functions related to manipulating the
 *              dlists that are used by the LVM region manager.
 */ 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"


/* Function: lvm_append_segment_to_container
 *
 *	Append the specified new_segment to the end of the "consumed" dlist in
 *	the specified container.
 */
int lvm_append_segment_to_container(	storage_object_t	* new_segment,
					storage_container_t	* container )
{
	int	rc;
	void	* handle;

	LOG_ENTRY;

	rc = InsertObject(container->objects_consumed,
			sizeof(storage_object_t),
			new_segment,	
			new_segment->object_type,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (!rc) {
		// Mark the segment as belonging to this container.
		new_segment->consuming_container = container;
	}
	else {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			new_segment->name, container->name );
	}

	RETURN(rc);
}


/* Function: lvm_remove_segment_from_container
 *
 *	Remove the specified segment from its "consuming" container.
 */
int lvm_remove_segment_from_container( storage_object_t * segment )
{
	int rc = 0;

	LOG_ENTRY;

	if ( ! segment->consuming_container ) {
		LOG_WARNING("Object %s is not in a container\n", segment->name);
		RETURN(EINVAL);
	}

	rc = DeleteObject(segment->consuming_container->objects_consumed, segment);
	if (!rc) {
		// Mark the segment as "unconsumed".
		segment->consuming_container = NULL;
	}
	else {
		LOG_SERIOUS("Error removing object %s from container %s\n",
			segment->name, segment->consuming_container->name);
	}

	RETURN(rc);
}


/* Function: lvm_append_region_to_container
 *
 *	Append the specified new_region at the end of the "producing" dlist in
 *	the specified container.
 */
int lvm_append_region_to_container(	storage_object_t	* new_region,
					storage_container_t	* container )
{
	int	rc;
	void	* handle;

	LOG_ENTRY;

	rc = InsertObject(container->objects_produced,
			sizeof(storage_object_t),
			new_region,
			REGION_TAG,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (!rc) {
		// Mark the region as belonging to this container.
		new_region->producing_container = container;
	}
	else {
		LOG_SERIOUS("Error adding region %s to container %s\n",
			new_region->name, container->name);
	}

	RETURN(rc);
}


/* Function: lvm_remove_region_from_container
 *
 *	Remove the specified region from its "producing" container.
 */
int lvm_remove_region_from_container( storage_object_t * region )
{
	int rc = 0;

	LOG_ENTRY;

	if ( ! region->producing_container ) {
		LOG_WARNING("Region %s not in a container\n", region->name);
		RETURN(EINVAL);
	}

	rc = DeleteObject(region->producing_container->objects_produced, region);
	if (!rc) {
		// Mark the region as "unproduced".
		region->producing_container = NULL;
	}
	else {
		LOG_SERIOUS("Error removing region %s from container %s\n",
			region->name, region->producing_container->name);
	}

	RETURN(rc);
}


/* Function: lvm_append_region_to_segment
 *
 *	Associate the specified region and segment as parent/child. Add the
 *	region to the segment's "parent" dlist, and add the segment to the
 *	region's "child" dlist. Also need to check for duplicates, so the
 *	same region/segment don't get associated more than once.
 */
int lvm_append_region_to_segment(	storage_object_t	* region,
					storage_object_t	* segment )
{
	int	rc;
	void	* handle;

	// No LOG_ENTRY or RETURN calls. Writes too many messages to the log.

	rc = ExclusiveInsertObject(segment->parent_objects,
				sizeof(storage_object_t),
				region,
				REGION_TAG,
				NULL,
				AppendToList,
				TRUE,
				&handle);
	if (rc) {
		LOG_SERIOUS("Error adding region %s as a parent to object %s\n",
			region->name, segment->name);
		return rc;
	}

	rc = ExclusiveInsertObject(region->child_objects,
				sizeof(storage_object_t),
				segment,
				segment->object_type,
				NULL,
				AppendToList,
				TRUE,
				&handle);
	if (rc) {
		LOG_SERIOUS("Error adding object %s as a child to region %s\n",
			segment->name, region->name);
		DeleteObject(segment->parent_objects, region);
	}

	return rc;
}


/* Function: lvm_remove_region_from_segment
 *
 *	Remove the parent/child association between the specified region
 *	and segment. Remove the region from the segment's "parent" list,
 *	and remove the segment from the region's "child" list.
 */
int lvm_remove_region_from_segment(	storage_object_t	* region,
					storage_object_t	* segment )
{
	int	rc;

	LOG_ENTRY;

	rc = DeleteObject(segment->parent_objects, region);
	if (rc) {
		LOG_SERIOUS("Error removing region %s from object %s\n",
			region->name, segment->name);
	}

	rc = DeleteObject(region->child_objects, segment);
	if (rc) {
		LOG_SERIOUS("Error removing object %s from region %s\n",
			segment->name, region->name);
	}

	RETURN(rc);
}


/* Function: lvm_clear_child_list
 *
 *	Remove all segments from this region's child list.
 */
int lvm_clear_child_list( storage_object_t * region )
{
	storage_object_t	* segment = NULL;
	TAG			tag;
	int			size;

	LOG_ENTRY;

	GoToStartOfList(region->child_objects);

	while ( ! BlindGetObject(region->child_objects, &size, &tag, NULL, FALSE, (void**)&segment) ) {
		if ( segment ) {
			if ( lvm_remove_region_from_segment(region, segment) ) {
				LOG_SERIOUS("Could not clear all objects from child list in region %s\n", region->name);
			}
		}
	}

	RETURN(0);
}


/* Function: lvm_add_object_to_list
 *
 *	Add the specified object to the specified list.
 */
int lvm_add_object_to_list(	storage_object_t	* object,
				dlist_t			list )
{
	int	rc;
	void	* handle;

	LOG_ENTRY;

	rc = InsertObject(list,
			sizeof(storage_object_t),
			object,
			object->object_type,
			NULL,
			AppendToList,
			TRUE,
			&handle);

	if (rc) {
		LOG_SERIOUS("Error adding object %s to list\n", object->name);
	}

	RETURN(rc);
}


/* Function: lvm_add_group_to_list
 *
 *	Add the specified group to the global list.
 */
int lvm_add_group_to_list( lvm_volume_group_t * group )
{
	void	* handle;
	int	rc;

	LOG_ENTRY;

	rc = InsertObject(lvm_group_list,
			sizeof(lvm_volume_group_t),
			group,
			0, NULL,
			AppendToList,
			TRUE,
			&handle);

	if (rc) {
		LOG_SERIOUS("Error adding container %s to global list\n", group->container->name);
	}

	RETURN(rc);
}


/* Function: lvm_get_list_item
 *
 *	Get the current item from the specified list. This is just
 *	a simple wrapper for BlindGetObject, but will make dlist
 *	loops easier to read and understand.
 */
void * lvm_get_list_item( dlist_t list )
{
	void	* object;
	TAG	tag;
	int	size;

	LOG_ENTRY;
	BlindGetObject(list, &size, &tag, NULL, TRUE, (void**)&object);
	RETURN(object);
}


