/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: liblocaldskmgr.so
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>

#include <plugin.h>

#include "localdskmgr.h"

plugin_record_t LD_PluginRecord;



/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                         PRIVATE DATA AREAS AND SUBROUTINES                           +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static engine_functions_t *EngFncs;           /* evms engine function table     */


/*
 *  Called to ask the evms kernel runtime for all the block devices (logical disks)
 *  that it has discovered.
 *
 *  Input:  The master logical disk list, kept by the engine
 *
 *  Output: rc = 0 for success, rc = errno if any errors occur
 *
 *          if successful then storage_object_t structs will be added to the
 *          MasterDiskList Dlist.
 */

static int get_logical_disk_list( dlist_t   disk_list ) {
    int                    rc = 0;
    evms_user_disk_t       KernelDisk;
    evms_user_disk_info_t  KernelDiskInfo;
    int                    status;
    int                    error;
    int                    dev_handle;
    ADDRESS                handle;
    storage_object_t      *ld;
    ld_private_data_t     *private_data;
    char                  *pName;

    LOG_PROC_ENTRY();

    if (disk_list==NULL) {
        LOG_ERROR("Disk list pointer in NULL.\n");
        LOG_PROC_EXIT_INT(EINVAL);
        return EINVAL;
    }

    KernelDisk.command = EVMS_FIRST_DISK;
    KernelDisk.status  = EVMS_DISK_VALID;

    /* Loop while we have valid device info.          */
    /* Get logical disks and insert into our dlist_t. */
    while (KernelDisk.status == EVMS_DISK_VALID) {

        /* Get a handle to a logical disk from the evms device driver */
        status = EngFncs->ioctl_evms_kernel(EVMS_GET_LOGICAL_DISK, &KernelDisk);
        if (status) {
            LOG_WARNING("Status from ioctl to kernel is %d  Error code is %d.\n", status, errno);
            LOG_PROC_EXIT_INT(-errno);
            return -errno;
        }

        if (KernelDisk.status == EVMS_DISK_INVALID) break;  /* DONE?  kernel says no more disks */

        KernelDisk.command = EVMS_NEXT_DISK;

        /* Ask the kernel for info about this logical disk. */
        KernelDiskInfo.disk_handle = KernelDisk.disk_handle;
        status = EngFncs->ioctl_evms_kernel(EVMS_GET_LOGICAL_DISK_INFO, &KernelDiskInfo);
        if ((status != 0) || (KernelDiskInfo.status != 0)) {
            LOG_WARNING("rc from ioctl to kernel = %d  Errno = %d.  KernelInfoStatus = %d.\n", status, errno,KernelDiskInfo.status);
            continue;
        }

        /* If the kernel name has the EVMS dev node prefix, skip over it. */
        /* Don't use the prefix as part of the storage object's name. */
        pName = KernelDiskInfo.disk_name;
        if (strncmp(pName, EVMS_DEV_NODE_PATH, strlen(EVMS_DEV_NODE_PATH)) == 0) {
            pName += strlen(EVMS_DEV_NODE_PATH);
        }

        /* If the disk is removable, try opening the device.  This does two
         * things.  One is that some devices that have removable media report
         * a disk even if there is no medium in the drive.  The open will fail
         * in that case.  If there is a medium in the drive, the open will lock
         * the medium in the drive so that it can't be removed while the Engine
         * is open.
         */
        dev_handle = 0;
        if (KernelDiskInfo.flags & EVMS_DEVICE_REMOVABLE) {
            char dev_name[EVMS_NAME_SIZE+1];

            /* Use the standard dev node name, not the EVMS name, to open the
             * device.
             */
            strcpy(dev_name, "/dev/");
            strcat(dev_name, pName);
            dev_handle = open(dev_name, O_RDWR);

            if (dev_handle < 0) {
                /* Open failed.  Device must not have any medium in it. */
                /* Skip it. */
                continue;
            }
        }

        /* Get a logical_disk struct */
        rc = EngFncs->allocate_logical_disk(pName, &ld);
        if (rc != 0) {
            if (dev_handle > 0) {
                close(dev_handle);
            }
            LOG_CRITICAL("Error code %d from Engine's allocate_logical_disk.\n", rc);
            LOG_PROC_EXIT_INT(rc);
            return rc;
        }

        ld->plugin = &LD_PluginRecord;

        /* All disk space is allocated data. */
        ld->data_type = DATA_TYPE;

        private_data = malloc(sizeof(ld_private_data_t));
        if (private_data == NULL) {
            EngFncs->free_logical_disk(ld);
            if (dev_handle > 0) {
                close(dev_handle);
            }
            LOG_CRITICAL("Error allocating memory for private data.\n");
            LOG_PROC_EXIT_INT(ENOMEM);
            return ENOMEM;
        }

        /* Keep a copy of the kernel disk handle in the private data. */
        private_data->kernel_disk_handle = KernelDisk.disk_handle;
        private_data->dev_handle = dev_handle;
        ld->private_data = private_data;

        ld->geometry.cylinders           = (u_int64_t) KernelDiskInfo.geometry.cylinders;
        ld->geometry.heads               = (u_int32_t) KernelDiskInfo.geometry.heads;
        ld->geometry.bytes_per_sector    = (u_int32_t) KernelDiskInfo.hardsect_size;
        ld->geometry.block_size          = KernelDiskInfo.block_size;
        ld->size                         = (u_int64_t) KernelDiskInfo.total_sectors;

        /* Kernel defines a different set of flags so dont pick them up! */
//      ld->flags                        = (u_int32_t )KernelDiskInfo.flags;

        /* We cannot calculate track size for very large scsi drives cuz
         * they tend to report max geometry (n,255,63) and then report a
         * capacity that is much greater than C * H * S.
         */
//      ld->geometry.sectors_per_track    = ld->size / (ld->geometry.heads*ld->geometry.cylinders);
        ld->geometry.sectors_per_track    = (u_int32_t) KernelDiskInfo.geometry.sectors;

        /* LBA of 1st sector above boot cylinder limit for this drive  */
        if (ld->geometry.cylinders < 1024) {                  // if drive is too small
            ld->geometry.boot_cylinder_limit = ld->size;      // then the limit is the size of the drive
        }                                                     // else the limit is calculated
        else {
            ld->geometry.boot_cylinder_limit = ld->geometry.heads * ld->geometry.sectors_per_track * 1023;
        }

        /* Insert the logical disk into our logical disk dlist_t. */
        error = InsertObject ( (dlist_t)          disk_list,
                               (uint)             sizeof( storage_object_t ),
                               (ADDRESS)          ld,
                               (TAG)              DISK_TAG,
                               (ADDRESS)          NULL,
                               (Insertion_Modes)  AppendToList,
                               (BOOLEAN)          TRUE,
                               (ADDRESS)          &handle );

        if (error) {
            LOG_SERIOUS("Error code %d when inserting new logical disk into the disk list.\n", error);
            LOG_PROC_EXIT_INT(error);
            return error;
        }

        LOG_DETAILS("NEW LOGICAL DEVICE ...........:\n");
        LOG_DETAILS("                          Name: %s\n", ld->name);
        LOG_DETAILS("                          size: %lld\n", ld->size);
        LOG_DETAILS("\n");
        LOG_DETAILS("        GEOMETRY .............:\n");
        LOG_DETAILS("                     cylinders: %d\n", ld->geometry.cylinders );
        LOG_DETAILS("                         heads: %d\n", ld->geometry.heads);
        LOG_DETAILS("                       sectors: %d\n", ld->geometry.sectors_per_track);
        LOG_DETAILS("           sector size (bytes): %d\n", ld->geometry.bytes_per_sector);
    }

    LOG_PROC_EXIT_INT(0);
    return 0;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                            Start Of EVMS Plugin Functions                            +
+                        (exported to engine via function table)                       +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

static int LD_setup(engine_mode_t mode, engine_functions_t *engine_function_table) {
    /* save info we get from the engine */
    EngFncs = engine_function_table;

    return 0;
}


static int close_dev(ADDRESS object,
                     TAG     object_tag,
                     uint    object_tize,
                     ADDRESS object_tandle,
                     ADDRESS parameters) {

    LOG_PROC_ENTRY();

    /* Safety check. */
    if (object_tag == DISK_TAG) {
        storage_object_t * disk = (storage_object_t *) object;
        ld_private_data_t * private_data = (ld_private_data_t *) disk->private_data;

        if (private_data->dev_handle) {
            close(private_data->dev_handle);
        }
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


static void LD_cleanup() {

    /* Find any disks that have removable media and close the device that
       was opended during discovery. */

    int rc;
    dlist_t disk_list;

    /* Get a list of disks that are managed by this plug-in. */
    rc = EngFncs->get_object_list(DISK,0, &LD_PluginRecord, 0, &disk_list);

    if (rc == 0) {
        /* Close and dev handles that might be open. */
        ForEachItem(disk_list, close_dev, NULL, TRUE);

        DestroyList(&disk_list, FALSE);
    }
}


static int LD_can_delete(storage_object_t * disk) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_can_expand(storage_object_t * object,
                         sector_count_t   * expand_limit,
                         dlist_t            expand_points) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(0);
    return 0;
}


static int LD_can_expand_by(storage_object_t * object,
                            sector_count_t   * size) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_can_shrink(storage_object_t * object,
                         sector_count_t   * shrink_limit,
                         dlist_t            shrink_points) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(0);
    return 0;
}


static int LD_can_shrink_by(storage_object_t * object,
                            sector_count_t   * size) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_can_move(storage_object_t * object) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_can_set_volume(storage_object_t * object,
                             BOOLEAN            flag) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(0);
    return 0;
}


static int LD_discover(dlist_t input_list, dlist_t output_list, BOOLEAN final_call ) {
    int                    rc;

    LOG_PROC_ENTRY();

    /* Note:  Device Managers ignore the input_list parameter. */
    /* Device Managers only generate output.                   */

    if (output_list==NULL) {
        LOG_ERROR("The output list is NULL.\n");
        LOG_PROC_EXIT_INT(EINVAL);
        return EINVAL;
    }

    /* Get the kernel's list of local disks. */
    rc = get_logical_disk_list(output_list);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int LD_create_logical_disk(dlist_t          input_objects,
                                  option_array_t * options,
                                  dlist_t          output_objects) {
    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_delete_logical_disk(storage_object_t * object,
                                  dlist_t            child_objects) {
    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_expand(storage_object_t * object,
                     storage_object_t * expand_object,
                     dlist_t            input_objects,
                     option_array_t   * options) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_shrink(storage_object_t * object,
                     storage_object_t * shrink_object,
                     dlist_t            input_objects,
                     option_array_t   * options) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_move(storage_object_t * source,
                   storage_object_t * target,
                   option_array_t   * options) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static void LD_set_volume(storage_object_t * object,
                         BOOLEAN            flag) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_VOID();
}


static int LD_add_sectors_to_kill_list( storage_object_t * ld, lsn_t lsn, sector_count_t count) {

    int rc = 0;

    if ( (ld==NULL) ||
         (lsn+count > ld->size) ) {
        if (ld == NULL) {
            LOG_ERROR("Pointer to disk structure is NULL.\n");
        }
        if (lsn+count > ld->size) {
            LOG_ERROR("Sector offset plus count goes beyond the end of the disk.\n");
        }
        LOG_PROC_EXIT_INT(EINVAL);
        return EINVAL;
    }

    rc = EngFncs->add_sectors_to_kill_list(ld, lsn, count );

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int LD_commit_changes( storage_object_t *ld, uint phase) {
    /* Even though we don't do anything we'll return success so that the engine
     * will continue with the commit process.
     */

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(0);
    return 0;
}


static int LD_get_option_count(task_context_t * context) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(0);
    return 0;
}


static int LD_init_task(task_context_t * context) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_set_option(task_context_t * context,
                         u_int32_t        index,
                         value_t        * value,
                         task_effect_t  * effect) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


static int LD_set_objects(task_context_t * context,
                          dlist_t          declined_objects,
                          task_effect_t  * effect) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


/*
 * Returns DISK specific information
 */
static int LD_get_info( storage_object_t * object,  char * descriptor_name, extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    uint     count = 0;

    LOG_PROC_ENTRY();

    if (info) {

        if (descriptor_name == NULL) {
            *info = NULL;     // init to no info returned

            if ( object->object_type == DISK ) {

                Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( (LD_INFO_COUNT - 1) * sizeof(extended_info_t) ) );
                if (Info) {

                    Info->count = LD_INFO_COUNT;

                    SET_STRING_FIELD( Info->info[LD_INFO_NAME_INDEX].name, "Name" );
                    SET_STRING_FIELD( Info->info[LD_INFO_NAME_INDEX].title, "Name" );
                    SET_STRING_FIELD( Info->info[LD_INFO_NAME_INDEX].desc, "EVMS name for the DISK storage object");
                    Info->info[LD_INFO_NAME_INDEX].type               = EVMS_Type_String;
                    Info->info[LD_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                    SET_STRING_FIELD( Info->info[LD_INFO_NAME_INDEX].value.s, object->name );
                    Info->info[LD_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_SIZE_INDEX].name, "Size" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SIZE_INDEX].title, "Size" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SIZE_INDEX].desc, "Size of the disk in sectors");
                    Info->info[LD_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                    Info->info[LD_INFO_SIZE_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_SIZE_INDEX].value.ui64         = object->size;
                    Info->info[LD_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_CYLINDERS_INDEX].name, "Cyl" );
                    SET_STRING_FIELD( Info->info[LD_INFO_CYLINDERS_INDEX].title, "Cylinders" );
                    SET_STRING_FIELD( Info->info[LD_INFO_CYLINDERS_INDEX].desc, "Drive geometry -- number of cylinders");
                    Info->info[LD_INFO_CYLINDERS_INDEX].type               = EVMS_Type_Unsigned_Int64;
                    Info->info[LD_INFO_CYLINDERS_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_CYLINDERS_INDEX].value.ui64         = object->geometry.cylinders;
                    Info->info[LD_INFO_CYLINDERS_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_CYLINDERS_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_HEADS_INDEX].name, "Heads" );
                    SET_STRING_FIELD( Info->info[LD_INFO_HEADS_INDEX].title, "Heads" );
                    SET_STRING_FIELD( Info->info[LD_INFO_HEADS_INDEX].desc, "Drive geometry -- number of heads");
                    Info->info[LD_INFO_HEADS_INDEX].type               = EVMS_Type_Unsigned_Int32;
                    Info->info[LD_INFO_HEADS_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_HEADS_INDEX].value.ui32         = object->geometry.heads;
                    Info->info[LD_INFO_HEADS_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_HEADS_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_SECTORS_INDEX].name, "Sectors" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SECTORS_INDEX].title, "Sectors" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SECTORS_INDEX].desc, "Drive geometry -- sectors per track");
                    Info->info[LD_INFO_SECTORS_INDEX].type               = EVMS_Type_Unsigned_Int32;
                    Info->info[LD_INFO_SECTORS_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_SECTORS_INDEX].value.ui32         = object->geometry.sectors_per_track;
                    Info->info[LD_INFO_SECTORS_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_SECTORS_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_SECTOR_SIZE_INDEX].name, "SectorSize" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SECTOR_SIZE_INDEX].title, "Sector Size" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SECTOR_SIZE_INDEX].desc, "Number of bytes per sector");
                    Info->info[LD_INFO_SECTOR_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                    Info->info[LD_INFO_SECTOR_SIZE_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_SECTOR_SIZE_INDEX].value.ui32         = object->geometry.bytes_per_sector;
                    Info->info[LD_INFO_SECTOR_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_SECTOR_SIZE_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_BLOCK_SIZE_INDEX].name, "BlockSize" );
                    SET_STRING_FIELD( Info->info[LD_INFO_BLOCK_SIZE_INDEX].title, "Block Size" );
                    SET_STRING_FIELD( Info->info[LD_INFO_BLOCK_SIZE_INDEX].desc, "Number of bytes per block");
                    Info->info[LD_INFO_BLOCK_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                    Info->info[LD_INFO_BLOCK_SIZE_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_BLOCK_SIZE_INDEX].value.ui64         = object->geometry.block_size;
                    Info->info[LD_INFO_BLOCK_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_BLOCK_SIZE_INDEX].group, 0, sizeof(group_info_t));

                    SET_STRING_FIELD( Info->info[LD_INFO_BOOT_LIMIT_INDEX].name, "BootLimit" );
                    SET_STRING_FIELD( Info->info[LD_INFO_BOOT_LIMIT_INDEX].title, "Boot Cylinder Limit" );
                    SET_STRING_FIELD( Info->info[LD_INFO_BOOT_LIMIT_INDEX].desc, "LBA of the first sector above the boot cylinder limit for this drive");
                    Info->info[LD_INFO_BOOT_LIMIT_INDEX].type               = EVMS_Type_Unsigned_Int64;
                    Info->info[LD_INFO_BOOT_LIMIT_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_BOOT_LIMIT_INDEX].value.ui64         = object->geometry.boot_cylinder_limit;
                    Info->info[LD_INFO_BOOT_LIMIT_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_BOOT_LIMIT_INDEX].group, 0, sizeof(group_info_t));

                    GetListSize(object->parent_objects, &count);

                    SET_STRING_FIELD( Info->info[LD_INFO_SEGMENTS_INDEX].name, "Segments" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SEGMENTS_INDEX].title, "Segments" );
                    SET_STRING_FIELD( Info->info[LD_INFO_SEGMENTS_INDEX].desc, "Number of segments discovered on the drive (metadata, data, freespace)");
                    Info->info[LD_INFO_SEGMENTS_INDEX].type               = EVMS_Type_Unsigned_Int32;
                    Info->info[LD_INFO_SEGMENTS_INDEX].unit               = EVMS_Unit_None;
                    Info->info[LD_INFO_SEGMENTS_INDEX].value.ui32         = count;
                    Info->info[LD_INFO_SEGMENTS_INDEX].collection_type    = EVMS_Collection_None;
                    memset( &Info->info[LD_INFO_SEGMENTS_INDEX].group, 0, sizeof(group_info_t));

                    *info = Info;

                    rc = 0;
                }
                else {
                    rc = ENOMEM;
                }
            }

        } else {
            /* There is no more information about any of the descriptors. */
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


static int LD_set_info(storage_object_t * object,
                      option_array_t   * options) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}


/*
 *  Returns plug-in specific information
 */
static int LD_get_plugin_info( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_version_string[64];

    LOG_PROC_ENTRY();

    if (info) {

        if (descriptor_name == NULL) {

            *info = NULL;     // init to no info returned

            Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (6*sizeof(extended_info_t))  );
            if (Info) {

                Info->count = 5;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_version_string, "%d.%d.%d",
                        LD_PluginRecord.required_api_version.major,
                        LD_PluginRecord.required_api_version.minor,
                        LD_PluginRecord.required_api_version.patchlevel );

                SET_STRING_FIELD( Info->info[0].name, "Short Name" );
                SET_STRING_FIELD( Info->info[0].title, "Short Name" );
                SET_STRING_FIELD( Info->info[0].desc, "A short name given to this plugin");
                Info->info[0].type               = EVMS_Type_String;
                Info->info[0].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[0].value.s, LD_PluginRecord.short_name );
                Info->info[0].collection_type    = EVMS_Collection_None;
                memset( &Info->info[0].group, 0, sizeof(group_info_t));

                SET_STRING_FIELD( Info->info[1].name, "Long Name" );
                SET_STRING_FIELD( Info->info[1].title, "Long Name" );
                SET_STRING_FIELD( Info->info[1].desc, "A long name given to this plugin");
                Info->info[1].type               = EVMS_Type_String;
                Info->info[1].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[1].value.s, LD_PluginRecord.long_name );
                Info->info[1].collection_type    = EVMS_Collection_None;
                memset( &Info->info[1].group, 0, sizeof(group_info_t));

                SET_STRING_FIELD( Info->info[2].name, "Type" );
                SET_STRING_FIELD( Info->info[2].title, "Plug-in Type" );
                SET_STRING_FIELD( Info->info[2].desc, "There are various types of plugins, each responsible for some kind of storage object.");
                Info->info[2].type               = EVMS_Type_String;
                Info->info[2].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[2].value.s, "Device Manager" );
                Info->info[2].collection_type    = EVMS_Collection_None;
                memset( &Info->info[2].group, 0, sizeof(group_info_t));

                SET_STRING_FIELD( Info->info[3].name, "Version" );
                SET_STRING_FIELD( Info->info[3].title, "Plugin Version" );
                SET_STRING_FIELD( Info->info[3].desc, "Version number of this plug-in");
                Info->info[3].type               = EVMS_Type_String;
                Info->info[3].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[3].value.s, version_string );
                Info->info[3].collection_type    = EVMS_Collection_None;
                memset( &Info->info[3].group, 0, sizeof(group_info_t));

                SET_STRING_FIELD( Info->info[4].name, "Required Version" );
                SET_STRING_FIELD( Info->info[4].title, "Required Engine Version" );
                SET_STRING_FIELD( Info->info[4].desc, "Version of the Engine that the plug-in requires. It will not run on older versions of the Engine.");
                Info->info[4].type               = EVMS_Type_String;
                Info->info[4].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[4].value.s, required_version_string );
                Info->info[4].collection_type    = EVMS_Collection_None;
                memset( &Info->info[4].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;
            }
            else {
                rc = ENOMEM;
            }

        } else {
            /* There is no more information on any of the descriptors. */
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 *
 *   Function reads sectors from a disk.
 *
 *   Returns: rc=0 if successful, else rc=errno of cause of failure.
 *
 */
static int LD_read( storage_object_t *ld, lsn_t sector_offset, sector_count_t sector_count, void * buffer) {
    int                    status;
    evms_sector_io_t       Sector_IO_Struct;

    LOG_PROC_ENTRY();

    if ( ( ld==NULL ) ||
         ( ld->private_data==NULL ) ||
         ( ((ld_private_data_t *)ld->private_data)->kernel_disk_handle==0) ||
         ( sector_offset+sector_count > ld->size) ) {

        if (ld == NULL) {
            LOG_ERROR("The pointer to the logical disk is NULL.\n");
        }
        if (ld->private_data==NULL) {
            LOG_SERIOUS("The logical disk has no private data structure.\n");
        }
        if (((ld_private_data_t *)ld->private_data)->kernel_disk_handle==0) {
            LOG_SERIOUS("The logical disk's private data does not have a kernel handle for the disk.\n");
        }
        if (sector_offset+sector_count > ld->size) {
            LOG_ERROR("Sector offset plus count goes beyond the end of the disk.\n");
        }

        LOG_PROC_EXIT_INT(EINVAL);
        return EINVAL;
    }

    Sector_IO_Struct.disk_handle     = ((ld_private_data_t *)ld->private_data)->kernel_disk_handle;
    Sector_IO_Struct.io_flag         = EVMS_SECTOR_IO_READ;
    Sector_IO_Struct.starting_sector = sector_offset;
    Sector_IO_Struct.sector_count    = sector_count;
    Sector_IO_Struct.buffer_address  = (u_char *) buffer;
    Sector_IO_Struct.status          = 0;

    LOG_DEBUG("Read %lld sectors from disk %s at sector %lld.\n", sector_count, ld->name, sector_offset);

    status=EngFncs->ioctl_evms_kernel(EVMS_SECTOR_IO, &Sector_IO_Struct);

    if ( (status) || (Sector_IO_Struct.status) ) {
        LOG_PROC_EXIT_INT(EIO);
        return EIO;
    }
    else {
        LOG_PROC_EXIT_INT(0);
        return 0;
    }
}


/*
 *
 *   Function writes sectors to a disk.
 *
 *   Returns: rc=0 if successful, else rc=errno of cause of failure.
 *
 */
static int LD_write( storage_object_t *ld, lsn_t sector_offset, sector_count_t sector_count, void * buffer) {
    int                status;
    evms_sector_io_t   Sector_IO_Struct;

    LOG_PROC_ENTRY();

    if ( ( ld==NULL ) ||
         ( ld->private_data==NULL ) ||
         ( ((ld_private_data_t *)ld->private_data)->kernel_disk_handle==0) ||
         ( sector_offset+sector_count > ld->size) ) {

        if (ld == NULL) {
            LOG_ERROR("The pointer to the logical disk is NULL.\n");
        }
        if (ld->private_data==NULL) {
            LOG_SERIOUS("The logical disk has no private data structure.\n");
        }
        if (((ld_private_data_t *)ld->private_data)->kernel_disk_handle==0) {
            LOG_SERIOUS("The logical disk's private data does not have a kernel handle for the disk.\n");
        }
        if (sector_offset+sector_count > ld->size) {
            LOG_ERROR("Sector offset plus count goes beyond the end of the disk.\n");
        }

        LOG_PROC_EXIT_INT(EINVAL);
        return EINVAL;
    }

    Sector_IO_Struct.disk_handle     = ((ld_private_data_t *)ld->private_data)->kernel_disk_handle;
    Sector_IO_Struct.io_flag         = EVMS_SECTOR_IO_WRITE;
    Sector_IO_Struct.starting_sector = sector_offset;
    Sector_IO_Struct.sector_count    = sector_count;
    Sector_IO_Struct.buffer_address  = (u_char *) buffer;
    Sector_IO_Struct.status          = 0;

    LOG_DEBUG("Write %lld sectors to disk %s at sector %lld.\n", sector_count, ld->name, sector_offset);

    status=EngFncs->ioctl_evms_kernel(EVMS_SECTOR_IO, &Sector_IO_Struct);

    if ( (status) || (Sector_IO_Struct.status) ) {
        LOG_PROC_EXIT_INT(EIO);
        return EIO;
    }
    else {
        LOG_PROC_EXIT_INT(0);
        return 0;
    }
}


static int LD_direct_plugin_communication(void  * object,
                                          BOOLEAN target_kernel_plugin,
                                          void  * arg) {
    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_INT(ENOSYS);
    return ENOSYS;
}




/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                PLUGIN FUNCTION TABLE                                 +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static plugin_functions_t dft={

    setup_evms_plugin:          LD_setup,
    cleanup_evms_plugin:        LD_cleanup,
    can_delete:                 LD_can_delete,
    can_expand:                 LD_can_expand,
    can_expand_by:              LD_can_expand_by,
    can_shrink:                 LD_can_shrink,
    can_shrink_by:              LD_can_shrink_by,
    can_move:                   LD_can_move,
    can_set_volume:             LD_can_set_volume,
    discover:                   LD_discover,
    create:                     LD_create_logical_disk,
    delete:                     LD_delete_logical_disk,
    expand:                     LD_expand,
    shrink:                     LD_shrink,
    move:                       LD_move,
    set_volume:                 LD_set_volume,
    add_sectors_to_kill_list:   LD_add_sectors_to_kill_list,
    commit_changes:             LD_commit_changes,
    get_option_count:           LD_get_option_count,
    init_task:                  LD_init_task,
    set_option:                 LD_set_option,
    set_objects:                LD_set_objects,
    get_info:                   LD_get_info,
    set_info:                   LD_set_info,
    get_plugin_info:            LD_get_plugin_info,
    read:                       LD_read,
    write:                      LD_write,
    direct_plugin_communication:LD_direct_plugin_communication
};


plugin_record_t LD_PluginRecord = {
    id:                    SetPluginID(EVMS_OEM_IBM, EVMS_DEVICE_MANAGER, 1),

    version:              {major:      MAJOR_VERSION,
                           minor:      MINOR_VERSION,
                           patchlevel: PATCH_LEVEL},

    required_api_version: {major:      3,
                           minor:      0,
                           patchlevel: 0},

    short_name:            "LocalDskMgr",
    long_name:             "Local Disk Manager",
    oem_name:              "IBM",

    functions:             {plugin:    (ADDRESS) &dft},

    container_functions:   NULL
};


plugin_record_t * evms_plugin_records[] = {&LD_PluginRecord, NULL};

