/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: bbroptions.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>
#include <linux/evms/evms_drivelink.h>
#include <linux/genhd.h>

#include "drivelink.h"
#include "dloptions.h"



/*
 *  Copied from drivelink.c
 */
static BOOLEAN i_can_modify_object( storage_object_t *object )
{

    if (object) {

        if (object->plugin == DL_PluginRecord_Ptr) {

            if ( object->private_data ) {

                if ( ((Drive_Link_Private_Data *)object->private_data)->signature==EVMS_DRIVELINK_SIGNATURE) {

                    return TRUE;

                }

            }

        }

    }

    return FALSE;
}




/*
 *  Returns the current object in the specified DLIST.  This call should
 *  be preceeded by the get_first_object_in_list() call or the caller
 *  should position the DLIST current pointer themselves.  Note, this call
 *  will not modify anything in the DLIST, including the CURRENT_ITEM ptr
 *  and so it will not advance in the DLIST.
 */
static  storage_object_t *  get_object_from_list( dlist_t list )
{
    int rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint size;
    TAG  tag;

    LOGENTRY();

    rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
    if (rc == DLIST_SUCCESS) {
        object_from_list = obj;
    }

    LOGEXIT();
    return  object_from_list;
}



/*
 *   Called to build a reverse ordered child_objects dlist from a list
 *   of child objects.
 *
 *   Returns DLIST_SUCCESS if successful and the child_objects
 *   dlist should contain all the objects from the drive link
 *   in the proper order.
 *
 *   Returns
 */
static int build_reverse_ordered_child_object_list( storage_object_t *parent, dlist_t list )
{
    void                     *handle=NULL;
    int                       rc=EINVAL;
    int                       i;
    dlist_t                   reverse_ordered_list;
    Drive_Link_Private_Data  *pdata = (Drive_Link_Private_Data  *) parent->private_data;
    uint                      starting_count=0;
    uint                      ending_count=0;
    uint                      size;
    TAG                       tag;
    storage_object_t         *obj;

    LOGENTRY();

    if ( parent && list ) {

        GetListSize(list, &starting_count);

        reverse_ordered_list = CreateList();

        if ( reverse_ordered_list != NULL ) {

            rc = DLIST_SUCCESS;
            for(i=pdata->drive_link_count-1; i>=0; i-- ) {

                if ( pdata->drive_link[i].object != NULL ) {

                    rc = GoToStartOfList(list);

                    while (rc == DLIST_SUCCESS) {

                        rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
                        if (rc == DLIST_SUCCESS) {

                            if (obj == pdata->drive_link[i].object) {

                                rc = ExclusiveInsertObject ( reverse_ordered_list,
                                                             sizeof(storage_object_t),
                                                             obj,
                                                             obj->object_type,
                                                             NULL,
                                                             AppendToList,
                                                             TRUE,
                                                             &handle );
                                if (rc == DLIST_SUCCESS) {
                                    ++ending_count;
                                }

                                break;
                            }

                            rc = NextItem(list);
                        }
                    }

                }

            }

        }

        // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
        if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
            rc = DLIST_SUCCESS;
        }

        // if I finished with a reversed ordered list that contains
        // the same number of objects that I started with ... then
        // it means that the drive link ordering table matches the
        // input list exactly.
        //
        // So, now copy the reversed ordered list back to the callers
        // list so we can return the reordered drive link objects.
        if ( rc == DLIST_SUCCESS ) {

            if (starting_count == ending_count) {

                // first delete all objects in the caller list
                rc = DeleteAllItems ( list, FALSE );

                // then copy reordered objects to callers list
                if (rc == DLIST_SUCCESS) {
                    rc = CopyList(list, reverse_ordered_list, AppendToList);
                }

            }
            else {
                rc = EPERM;
            }

        }

    }
    else {
        LOG_DEBUG("bad parms\n");
    }

    // dont forget to toss working dlist
    if (reverse_ordered_list != NULL) DestroyList(&reverse_ordered_list, FALSE);

    LOGEXITRC();
    return rc;
}





/*
 *  Returns count of options for specified task
 */
int DL_GetOptionCount(task_context_t * task)
{
    int count;

    LOGENTRY();

    switch (task->action) {

    case EVMS_Task_Create:
        count = DL_CREATE_OPTION_COUNT;
        break;

    case EVMS_Task_Expand:
        count = DL_EXPAND_OPTION_COUNT;
        break;

    case EVMS_Task_Shrink:
        count = DL_SHRINK_OPTION_COUNT;
        break;

    default:
        count = 0;
        break;
    }

    LOGEXIT();
    return count;
}




static int allocate_create_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = DL_CREATE_OPTION_COUNT ;      // must set count

        // DriveLink Name
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].constraint.list = NULL;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].name, DL_CREATE_LINKNAME_NAME );
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].size = EVMS_VOLUME_NAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].tip, "The name to be given to the drivelink object.");
        SET_STRING_FIELD( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].title, "Name" );
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].value.s = DLEngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
        if (context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].value.s) {
            memset(context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].value.s, 0, EVMS_VOLUME_NAME_SIZE);
        }
        else {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

        rc = 0;
    }


    LOGEXITRC();
    return rc;
}

static int initialize_create_option_descriptors(  task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    if (context) {

        // there should be exactly DL_CREATE_OPTION_COUNT options
        if ( context->option_descriptors->count == DL_CREATE_OPTION_COUNT ) {

            // DriveLink name
            context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;

            rc = 0;

        }
        else {

            LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);

        }

    }

    LOGEXITRC();
    return rc;
}


static int allocate_expand_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();


    if ( context ) {

        context->option_descriptors->count = 0;  // no options

        rc = 0;
    }


    LOGEXITRC();
    return rc;
}



static int allocate_shrink_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    if ( context ) {
        rc = 0;
    }


    LOGEXITRC();
    return rc;
}



/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the BOOLEAN flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int   rc;
    uint  linkcount=0;


    LOGENTRY();

    rc = GetListSize(context->selected_objects, &linkcount );
    if (rc == DLIST_SUCCESS) {

        if ( (linkcount > 0) &&
             (linkcount < EVMS_DRIVELINK_MAX_ENTRIES)) {

            rc = initialize_create_option_descriptors( context );

            if (rc == DLIST_SUCCESS) {
                *effect |=  EVMS_Effect_Reload_Options;
            }
        }

    }
    else {
        rc = EINVAL;
    }


    LOGEXITRC();
    return rc;
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int set_expand_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int                      rc=EINVAL;
    storage_object_t        *obj=NULL;
    void                    *handle;
    declined_object_t       *declined_object=NULL;
    int                      selected_objects_count=0;
    int                      max_objects_allowed=0;
    Drive_Link_Private_Data *pdata;
    int                      declined_object_count=0;


    LOGENTRY();

    // determine how many links we can add
    if (context) {
        if (i_can_modify_object(context->object)==TRUE) {

            pdata = ( Drive_Link_Private_Data * )context->object->private_data;

            max_objects_allowed = EVMS_DRIVELINK_MAX_ENTRIES - pdata->drive_link_count;

            if (max_objects_allowed > 0) {
                rc = GoToStartOfList( context->selected_objects );
                if (rc) {
                    LOG_ERROR("error, selected objects list is corrupt\n");
                    rc = EINVAL;  // change soft dlist error to serious invalid selection error
                }
            }

        }
    }

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            if ( max_objects_allowed > selected_objects_count ) {

                ++selected_objects_count;
                *effect |=  EVMS_Effect_Reload_Options;
                rc = NextItem(context->selected_objects);

            }
            else {

                LOG_ERROR("declining an object found in context->selected_objects, object name= %s\n", obj->name);

                ++declined_object_count;

                declined_object = DLEngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = obj;
                    declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );

                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Objects;
                        rc = NextItem(context->selected_objects);
                    }
                    else {
                        DLEngFncs->engine_free(declined_object);
                        LOG_ERROR("error, unable to insert declined object in engine dlist\n");
                    }

                }
                else {
                    LOG_ERROR("unable to malloc a declined object struct\n");
                    rc = ENOMEM;
                }

            }


        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    if (declined_object_count) {
        *effect = EVMS_Effect_Reload_Objects;
    }
    else if ( (rc==0) &&                        // didnt decline any of the objects and the result looks
              (selected_objects_count > 0) ) {  // good so now call the engine to see if we can expand
                                                // by the specified amount.

        sector_count_t   expand_sectors = 0;
        sector_count_t   child_sectors  = 0;

        rc = GoToStartOfList( context->selected_objects );

        while (rc == DLIST_SUCCESS) {

            obj = get_object_from_list( context->selected_objects );
            if (obj) {

                child_sectors = obj->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);
                expand_sectors += child_sectors;

                rc = DLEngFncs->can_expand_by( obj, &expand_sectors );

                if (rc) {

                    expand_sectors -= child_sectors;

                    LOG_DEBUG("declining object, object name= %s ... engine wont allow expanding\n", obj->name);

                    declined_object = DLEngFncs->engine_alloc( sizeof(declined_object_t));

                    if (declined_object) {

                        declined_object->object = obj;
                        declined_object->reason = -1; // ENOCANEXPAND

                        // remove object from selectable/acceptable object lists
                        DeleteObject( context->selected_objects, obj );
                        DeleteObject( context->acceptable_objects, obj );

                        // place object in declined list
                        rc = InsertObject ( declined_objects,
                                            sizeof(declined_object_t),
                                            obj,
                                            DECLINED_OBJECT_TAG,
                                            NULL,
                                            AppendToList,
                                            TRUE,
                                            &handle );

                        if (rc) {
                            DLEngFncs->engine_free(declined_object);
                            LOG_ERROR("error, unable to insert declined object in the engine declined object list\n");
                        }

                    }
                    else {
                        rc = ENOMEM;
                        LOG_ERROR("error, unable to malloc a declined object.\n");
                    }

                }

                rc = NextItem(context->selected_objects);
            }
            else {
                rc = DLIST_END_OF_LIST;
            }
        }
    }

    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int set_shrink_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int                rc = EINVAL;
    storage_object_t  *obj=NULL;
    storage_object_t  *child;
    storage_object_t  *last_good=NULL;
    void              *handle;
    BOOLEAN            invalidate_objects=FALSE;
    declined_object_t *declined_object=NULL;
    uint               selected_objects_count=0;
    int                search_index=0;
    int                declined_object_count=0;
    int                i;
    Drive_Link_Private_Data *pdata=NULL;


    LOGENTRY();

    if ((context) &&
        (declined_objects) &&
        (effect) &&
        (context->object) &&
        (context->object->private_data) &&
        (context->selected_objects) &&
        (context->acceptable_objects) ) {

        pdata = (Drive_Link_Private_Data *) context->object->private_data;

        // reorder the selected objects
        rc = build_reverse_ordered_child_object_list( context->object, context->selected_objects );
        if ( rc ) {
            LOG_ERROR("error, unable to build ordered list of selected objects\n");
            LOGEXITRC();
            return rc;
        }

        rc = GetListSize(context->selected_objects, &selected_objects_count);

        if ( (pdata) &&
             (rc == 0) &&
             (selected_objects_count > 0) &&
             (selected_objects_count < pdata->drive_link_count) ){

            declined_object_count = 0;
            search_index          = pdata->drive_link_count - 1;

            rc = GoToStartOfList( context->selected_objects );

            while (rc == DLIST_SUCCESS) {

                obj = get_object_from_list( context->selected_objects );
                if ( ( obj != NULL ) &&
                     ( obj != last_good ) ) {

                    if (obj == pdata->drive_link[search_index].object) {
                        last_good = obj;
                        rc = NextItem(context->selected_objects);
                    }
                    else {
                        invalidate_objects = TRUE;
                    }

                    --search_index;

                    if ( invalidate_objects == TRUE ) {

                        ++declined_object_count;

                        declined_object = DLEngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                            declined_object->object = obj;
                            declined_object->reason = -1; // EOUTOFORDER;

                            // remove object from selectable/acceptable object lists
                            DeleteObject( context->selected_objects, obj );
                            DeleteObject( context->acceptable_objects, obj );

                            // place it on the declined object list
                            rc = InsertObject ( declined_objects,
                                                sizeof(declined_object_t),
                                                declined_object,
                                                DECLINED_OBJECT_TAG,
                                                NULL,
                                                AppendToList,
                                                TRUE,
                                                &handle );

                            if (rc) {
                                DLEngFncs->engine_free(declined_object);
                            }

                        }
                        else {
                            rc = ENOMEM;
                            LOGEXITRC();
                            return rc;
                        }

                    }


                }
                else {
                    rc = DLIST_END_OF_LIST;;
                }

            }

            // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
            if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
                rc = DLIST_SUCCESS;
            }

        }

    }

    if (declined_object_count == 0) {

        // didnt decline any of the objects so call engine to see
        // if we can actually shrink by the resulting sector count.

        sector_count_t   shrink_sectors=0;

        for( i=pdata->drive_link_count-1; (i>=0)&&(selected_objects_count); i--, selected_objects_count-- ) {

            shrink_sectors += pdata->drive_link[i].sector_count;

            rc = DLEngFncs->can_shrink_by( context->object, &shrink_sectors );
            if (rc) {

                shrink_sectors -= pdata->drive_link[i].sector_count;

                ++declined_object_count;

                declined_object = DLEngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = pdata->drive_link[i].object;
                    declined_object->reason = -1; // ENOCANSHRINK

                    // remove object from selectable/acceptable object lists
                    DeleteObject( context->selected_objects, obj );
                    DeleteObject( context->acceptable_objects, obj );

                    // place it on the declined object list
                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );

                    if (rc) {
                        DLEngFncs->engine_free(declined_object);
                    }

                }
                else {
                    rc = ENOMEM;
                    LOGEXITRC();
                    return rc;
                }

            }

        }


        // if we can shrink the drivelink Ok then add another link
        // to the acceptable objects list so the user can further
        // shrink the drivelink
        if (declined_object_count == 0) {

            if ( selected_objects_count+1 < pdata->drive_link_count ) {

                child = pdata->drive_link[pdata->drive_link_count-selected_objects_count-1].object;

                rc = ExclusiveInsertObject ( context->acceptable_objects,
                                             sizeof(storage_object_t),
                                             child,
                                             EVMS_OBJECT_TAG,
                                             NULL,
                                             AppendToList,
                                             TRUE,
                                             &handle );

                if (rc == 0) {
                    GetListSize(context->acceptable_objects, &context->max_selected_objects);
                    *effect |= EVMS_Effect_Reload_Objects;
                }

                rc = 0;  // dont stop because we couldn't do the insert here
            }

       }


    }


    // if we declined some objects then we need to fixup the shrink task
    // context so that we only show the good selected objects PLUS the
    // next object that would be Ok to select.

    DeleteAllItems ( context->acceptable_objects, FALSE );
    CopyList(context->acceptable_objects, context->selected_objects, AppendToList );

    GetListSize(context->selected_objects, &context->max_selected_objects);

    if ( context->max_selected_objects+1 < pdata->drive_link_count ) {

        child = pdata->drive_link[pdata->drive_link_count-context->max_selected_objects-1].object;

        rc = ExclusiveInsertObject ( context->acceptable_objects,
                                     sizeof(storage_object_t),
                                     child,
                                     EVMS_OBJECT_TAG,
                                     NULL,
                                     AppendToList,
                                     TRUE,
                                     &handle );

        if (rc == 0) {
            ++context->max_selected_objects;
        }

    }

    *effect |= EVMS_Effect_Reload_Objects;



    LOGEXITRC();
    return rc;
}




/*
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int DL_SetObjects( task_context_t * context,
                   dlist_t          declined_objects,
                   task_effect_t  * effect )
{

    int rc = EINVAL;

    LOGENTRY();

    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            rc = set_create_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Expand:

            rc = set_expand_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Shrink:

            rc = set_shrink_object( context, declined_objects, effect );
            break;

        default:

            LOG_ERROR("error, context->action is unknown or unsupported\n");
            break;
        }
    }

    LOGEXITRC();
    return rc;
}




static int SetCreateOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOGENTRY();

    if (index == DL_CREATE_LINKNAME_INDEX ) {


        if (value) {

            if (value->s) {

                if ( (strlen(value->s) > 0) &&
                     (strlen(value->s) <= EVMS_VOLUME_NAME_SIZE) ){

                    rc = DLEngFncs->validate_name( value->s );

                    if (rc == 0) {
                        strcpy( context->option_descriptors->option[index].value.s, value->s);
                    }

                }

            }

        }

    }


    LOGEXITRC();
    return rc;
}




/*
 *
 */
int DL_SetOption(  task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   u_int32_t      * info )
{
    int rc=EINVAL;

    LOGENTRY();

    if ( context ) {

        switch (context->action) {

        case EVMS_Task_Create:
            rc = SetCreateOption( context, index, value, info );
            break;

        case EVMS_Task_Expand:
        case EVMS_Task_Shrink:
            rc = ENOSYS; // no expand or shrink options

        default:
            rc = EINVAL;
            break;
        }
    }

    LOGEXITRC();

    return rc;
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {

        rc = DLEngFncs->get_object_list( 0,
                                         DATA_TYPE,
                                         NULL,
                                         VALID_INPUT_OBJECT,
                                         &context->acceptable_objects );

    }
    else {
        LOG_ERROR("error, context already has acceptable objects\n");
    }

    LOGEXITRC();
    return rc;
}




/*
 *  Called from get_acceptable_expand_objects
 */
static BOOLEAN prune_unexpandable_objects( ADDRESS   Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           ADDRESS   ObjectHandle,
                                           ADDRESS   Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{
    storage_object_t * obj = (storage_object_t *) Object;

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    if (obj) {

        if ( obj->plugin  != DL_PluginRecord_Ptr ) {

                return FALSE;

        }

    }

    return TRUE;           // tells dlist to prune the object from the list
}

/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_expand_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {

        rc = DLEngFncs->get_object_list( 0,
                                         DATA_TYPE,
                                         NULL,
                                         VALID_INPUT_OBJECT,
                                         &context->acceptable_objects );

        if (rc == DLIST_SUCCESS) {

            PruneList(context->acceptable_objects, prune_unexpandable_objects, NULL);

        }

    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_shrink_objects( task_context_t * context  )
{
    int                      rc = EINVAL;
    void                    *handle;
    Drive_Link_Private_Data *pdata;
    storage_object_t        *child;


    LOGENTRY();

    if ((context) &&
        (context->object) &&
        (context->acceptable_objects) ) {


        pdata = (Drive_Link_Private_Data *) context->object->private_data;

        if (pdata) {

            if (pdata->drive_link_count > 1) {

                rc = 0;

                child = pdata->drive_link[pdata->drive_link_count-1].object;

                rc = InsertObject ( context->acceptable_objects,
                                    sizeof(storage_object_t),
                                    child,
                                    EVMS_OBJECT_TAG,
                                    NULL,
                                    AppendToList,
                                    TRUE,
                                    &handle );
            }

        }

    }

    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
int DL_InitTask( task_context_t * context )
{
    int rc = EINVAL;
    Drive_Link_Private_Data *pdata=NULL;

    LOGENTRY();


    if (context) {

        if (context->object) {
            pdata = (Drive_Link_Private_Data *) context->object->private_data;
        }

        switch (context->action) {

        case EVMS_Task_Create:

            context->min_selected_objects = 1;
            context->max_selected_objects = EVMS_DRIVELINK_MAX_ENTRIES;

            rc = allocate_create_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_create_objects( context );
            }
            break;

        case EVMS_Task_Expand:

            context->min_selected_objects = 1;

            if (pdata) {
                context->max_selected_objects = EVMS_DRIVELINK_MAX_ENTRIES - pdata->drive_link_count;
            }
            else {
                context->max_selected_objects = 1;
            }

            rc = allocate_expand_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_expand_objects( context );
            }
            break;

        case EVMS_Task_Shrink:

            context->min_selected_objects = 1;

            rc = allocate_shrink_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_shrink_objects( context );
                if (rc ==0) {
                    rc = GetListSize(context->acceptable_objects, &context->max_selected_objects);
                }
            }

            break;

        default:
               LOG_ERROR("context->action is unknown or unsupported\n");
               break;
        }
    }


    LOGEXITRC();
    return rc;
}


/*
 * Returns specific information
 */
int DL_GetInfo( storage_object_t  * object, char *name,  extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t    *Info;
    Drive_Link_Private_Data  *pdata = (Drive_Link_Private_Data *) object->private_data;


    LOGENTRY();

    // a measure of protection ...
    if ( (info == NULL) ||
         ((pdata->signature != EVMS_DRIVELINK_SIGNATURE)&&(pdata->signature != MISSING_CHILD_SIGNATURE))) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    if ( object->object_type == EVMS_OBJECT ) {

        Info = DLEngFncs->engine_alloc( sizeof(extended_info_array_t) + ( DL_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

            // the following info isnt available for missing child objects
            if (pdata->signature & MISSING_CHILD_SIGNATURE) {

                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].name, "Name" );
                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].title, "Name" );
                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].desc, "A temporary name given to the missing drivelink object that is used to assist with recovery.");
                Info->info[DL_INFO_NAME_INDEX].type               = EVMS_Type_String;
                Info->info[DL_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].value.s, object->name );
                Info->info[DL_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[DL_INFO_SIZE_INDEX].name, "Size" );
                SET_STRING_FIELD( Info->info[DL_INFO_SIZE_INDEX].title, "Size" );
                SET_STRING_FIELD( Info->info[DL_INFO_SIZE_INDEX].desc, "The size of the object that is missing, i.e. the useable size after subtracting metadata.");
                Info->info[DL_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[DL_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                Info->info[DL_INFO_SIZE_INDEX].value.ui64         = object->size;
                Info->info[DL_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

            }
            else {

                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].name, "Name" );
                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].title, "Name" );
                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].desc, "This is the name given to the storage object. It must be unique on the system.");
                Info->info[DL_INFO_NAME_INDEX].type               = EVMS_Type_String;
                Info->info[DL_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[DL_INFO_NAME_INDEX].value.s, object->name );
                Info->info[DL_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[DL_INFO_SIZE_INDEX].name, "Size" );
                SET_STRING_FIELD( Info->info[DL_INFO_SIZE_INDEX].title, "Size" );
                SET_STRING_FIELD( Info->info[DL_INFO_SIZE_INDEX].desc, "This is the size of the entire drivelink, i.e. the sum of all the drivelink child storage objects.");
                Info->info[DL_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[DL_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                Info->info[DL_INFO_SIZE_INDEX].value.ui64         = object->size;
                Info->info[DL_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[DL_INFO_LINKS_INDEX].name, "Links" );
                SET_STRING_FIELD( Info->info[DL_INFO_LINKS_INDEX].title, "Children" );
                SET_STRING_FIELD( Info->info[DL_INFO_LINKS_INDEX].desc, "This is the number of child storage objects being linked by the drivelink feature.");
                Info->info[DL_INFO_LINKS_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[DL_INFO_LINKS_INDEX].unit               = EVMS_Unit_None;
                Info->info[DL_INFO_LINKS_INDEX].value.ui64         = pdata->drive_link_count;
                Info->info[DL_INFO_LINKS_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_INFO_LINKS_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[DL_INFO_PSN_INDEX].name, "PSN" );
                SET_STRING_FIELD( Info->info[DL_INFO_PSN_INDEX].title, "Parent SN" );
                SET_STRING_FIELD( Info->info[DL_INFO_PSN_INDEX].desc, "The serial number of the parent drivelink object.");
                Info->info[DL_INFO_PSN_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[DL_INFO_PSN_INDEX].unit               = EVMS_Unit_None;
                Info->info[DL_INFO_PSN_INDEX].format             = EVMS_Format_Hex;
                Info->info[DL_INFO_PSN_INDEX].flags              = EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
                Info->info[DL_INFO_PSN_INDEX].value.ui64         = pdata->parent_serial_number;
                Info->info[DL_INFO_PSN_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_INFO_PSN_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

            }


            *info = Info;
            rc = 0;

        }
        else {
            LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *
 */
int DL_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_version_string[64];


    LOGENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    Info = DLEngFncs->engine_alloc( sizeof(extended_info_array_t) + (DL_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
    if (Info) {

       Info->count = DL_PLUGIN_INFO_COUNT;

       sprintf(version_string, "%d.%d.%d",
               MAJOR_VERSION,
               MINOR_VERSION,
               PATCH_LEVEL );

       sprintf(required_version_string, "%d.%d.%d",
               DL_PluginRecord_Ptr->required_api_version.major,
               DL_PluginRecord_Ptr->required_api_version.minor,
               DL_PluginRecord_Ptr->required_api_version.patchlevel );

       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].name, "Short Name" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
       Info->info[DL_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
       Info->info[DL_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].value.s, DL_PluginRecord_Ptr->short_name );
       Info->info[DL_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
       memset( &Info->info[DL_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].name, "Long Name" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].desc, "A long name given to this plugin.");
       Info->info[DL_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
       Info->info[DL_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].value.s, DL_PluginRecord_Ptr->long_name );
       Info->info[DL_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
       memset( &Info->info[DL_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].name, "Type" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
       Info->info[DL_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
       Info->info[DL_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].value.s, "Aggregate Class Feature" );
       Info->info[DL_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
       memset( &Info->info[DL_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].name, "Version" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
       Info->info[DL_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
       Info->info[DL_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
       Info->info[DL_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
       memset( &Info->info[DL_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].name, "Required Version" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].title, "Required Version" );
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].desc, "This is the version of the engine that the plugin requires. It will not run on older versions of the Engine.");
       Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].type               = EVMS_Type_String;
       Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].unit               = EVMS_Unit_None;
       SET_STRING_FIELD( Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].value.s, required_version_string );
       Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].collection_type    = EVMS_Collection_None;
       memset( &Info->info[DL_PLUGIN_INFO_REQVERSION_INDEX].group, 0, sizeof(group_info_t));

       *info = Info;

       rc = 0;
    }


    LOGEXITRC();
    return rc;
}
