/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: internalAPI.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include <fullengine.h>
#include "internalAPI.h"
#include "engine.h"
#include "crc.h"
#include "discover.h"
#include "handlemgr.h"
#include "commit.h"
#include "common.h"
#include "volume.h"
#include "expand.h"
#include "shrink.h"
#include "message.h"


/*-----------*\
| Global data |
\*-----------*/

static name_list_entry_t * name_registry = NULL;


/*-----------------------------*\
| Functions for getting plug-ins |
\*-----------------------------*/

typedef struct plugin_filters_s {
    plugin_type_t         type;
    plugin_search_flags_t flags;
} plugin_filters_t;

/*
 * filter_plugins is used by engine_get_plugin_list to strip out of the list
 * any plug-ins that are not desired.  filter_plugins is called by the PruneList
 * processor.  Parameter points to a plug-in type.  If the plug-in is not of
 * the specified type, then this function returns TRUE to tell PruneList to
 * remove the object from the list.  Otherwise, it returns FALSE to leave the
 * object in the list.
 */
static BOOLEAN filter_plugins(ADDRESS   Object,
                              TAG       ObjectTag,
                              uint      ObjectSize,
                              ADDRESS   ObjectHandle,
                              ADDRESS   Parameters,
                              BOOLEAN * FreeMemory,
                              uint    * Error) {

    plugin_record_t * pPlugRec = (plugin_record_t *) Object;
    plugin_filters_t * filters = (plugin_filters_t *) Parameters;

    BOOLEAN result = FALSE;  /* Assume we want to keep this one. */

    LOG_PROC_ENTRY();

    /*
     * We never want to free the item memory, so set this to FALSE
     * just in case.
     */
    *FreeMemory = FALSE;

    if (filters->type != 0) {
        if (GetPluginType(pPlugRec->id) != filters->type) {
            /* The type doesn't match.  Remove this object from the list. */
            result = TRUE;
        }
    }

    /* Any filter flags set? */
    if (filters->flags != 0) {

        /*
         * If the caller only wants plug-ins that support containers, then
         * filter out this plug-in if it doesn't have a container_functions
         * table.
         */
        if (filters->flags & SUPPORTS_CONTAINERS) {
            if (pPlugRec->container_functions == NULL) {
                result = TRUE;
            }
        }
    }

    *Error = DLIST_SUCCESS;

    LOG_PROC_EXIT_BOOLEAN_INT(result, *Error);
    return result;
}

/*
 * engine_get_plugin_list returns a dlist_t of plugin_record_t structures,
 * optionally filtering on the plug-in type.  If the type is 0, all
 * plugin_record_t structures will be returned.
 */
int engine_get_plugin_list(plugin_type_t         type,
                           plugin_search_flags_t flags,
                           dlist_t             * plugin_list) {
    int rc = 0;

    dlist_t new_plugin_list = CreateList();

    LOG_PROC_ENTRY();

    LOG_DEBUG("Filters:\n");
    LOG_DEBUG("  Plug-in type:  0x%x\n", type);
    LOG_DEBUG("  Flags:         0x%x\n", flags);
    LOG_DEBUG("Destination DLIST:  %p\n", plugin_list);

    if (new_plugin_list != NULL) {
        rc = CopyList(new_plugin_list, PluginList, AppendToList);

        if (rc == 0) {

            /*
             * If the caller specified a plug-in type or flags, use PruneList to
             * remove from the list plug-ins that are not of the specified type.
             */
            if ((type != 0) || (flags != 0)) {

                /* Get a structure for the filters. */
                plugin_filters_t * filters = malloc(sizeof(plugin_filters_t));

                if (filters != NULL) {

                    /*
                     * Fill in the filters and then use PruneList to remove from
                     * the list plug-ins that are not wanted.
                     */
                    filters->type  = type;
                    filters->flags = flags;

                    rc = PruneList(new_plugin_list,
                                   filter_plugins,
                                   filters);

                    free(filters);

                } else {
                    LOG_CRITICAL("Error allocating memory for plug-in filters.\n");
                    rc = ENOMEM;
                }
            }
        }

    } else {
        rc = ENOMEM;
    }

    if (rc == 0) {
        *plugin_list = new_plugin_list;
    } else {
        if (new_plugin_list != NULL) {
            DestroyList(&new_plugin_list, FALSE);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * engine_get_plugin_by_ID will return a pointer to the plugin_record_t that
 * has the specified plugin_ID_t.
 */
int engine_get_plugin_by_ID(plugin_id_t         pluginID,
                            plugin_record_t * * plugin) {
    int rc = 0;
    plugin_record_t * pPlugRec;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Search for plug-in with ID %d (%x).\n", pluginID, pluginID);

    rc = GoToStartOfList(PluginList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(PluginList,
                       sizeof(plugin_record_t),
                       PLUGIN_TAG,
                       NULL,
                       FALSE,
                       (ADDRESS*) &pPlugRec);

        while ((rc == 0) && (pPlugRec->id != pluginID)) {
            rc = GetNextObject(PluginList,
                               sizeof(plugin_record_t),
                               PLUGIN_TAG,
                               (ADDRESS *) &pPlugRec);
        }

        if (rc == 0) {
            *plugin = pPlugRec;

        } else {
            if ((rc == DLIST_EMPTY) ||
                (rc == DLIST_END_OF_LIST)) {
                rc = ENOENT;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * engine_get_plugin_by_name will return a pointer to the plugin_record_t that
 * has the specified short name.
 */
int engine_get_plugin_by_name(char              * plugin_short_name,
                              plugin_record_t * * plugin) {
    int rc = 0;
    plugin_record_t * pPlugRec;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Search for plug-in with short name %s.\n", plugin_short_name);

    rc = GoToStartOfList(PluginList);

    if (rc == DLIST_SUCCESS) {
        rc = GetObject(PluginList,
                       sizeof(plugin_record_t),
                       PLUGIN_TAG,
                       NULL,
                       FALSE,
                       (ADDRESS*) &pPlugRec);

        while ((rc == 0) && (strcmp(pPlugRec->short_name, plugin_short_name) != 0)) {
            rc = GetNextObject(PluginList,
                               sizeof(plugin_record_t),
                               PLUGIN_TAG,
                               (ADDRESS *) &pPlugRec);
        }

        if (rc == 0) {
            *plugin = pPlugRec;

        } else {
            if ((rc == DLIST_EMPTY) ||
                (rc == DLIST_END_OF_LIST)) {
                rc = ENOENT;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*-----------------------------*\
| Functions for getting volumes |
\*-----------------------------*/

/*
 * check_mounted will call is_volume_mounted() for the volume which will update
 * the mount_point string in the logical_volume_t to the current mount point of
 * the volume.
 */
static int check_mounted(ADDRESS object,
                         TAG     object_tag,
                         uint    object_tize,
                         ADDRESS object_tandle,
                         ADDRESS parameters) {

    LOG_PROC_ENTRY();

    /* Safety check */
    if (object_tag == VOLUME_TAG) {
        logical_volume_t * vol = (logical_volume_t *) object;

        is_volume_mounted(vol);
    }

    LOG_PROC_EXIT_INT(DLIST_SUCCESS);
    return DLIST_SUCCESS;
}


/*
 * filter_volumes is used by engine_get_Volume_list to strip out of the list
 * any volumes that are not desired.  filter_volumes is called by the PruneList
 * processor.  Parameter points to a plugin_record_t for an FSIM.  If the volume
 * is not managed by the specified FSIM, then this function returns TRUE to tell
 * PruneList to remove the object from the list.  Otherwise, it returns FALSE
 * to leave the object in the list.
 */
static BOOLEAN filter_volumes(ADDRESS   Object,
                              TAG       ObjectTag,
                              uint      ObjectSize,
                              ADDRESS   ObjectHandle,
                              ADDRESS   Parameters,
                              BOOLEAN * FreeMemory,
                              uint    * Error) {

    logical_volume_t * volume = (logical_volume_t *) Object;
    plugin_record_t * fsim = (plugin_record_t *) Parameters;

    BOOLEAN result = FALSE;  /* Assume we want to keep this one. */

    LOG_PROC_ENTRY();

    /*
     * We never want to free the item memory, so set this to FALSE
     * just in case.
     */
    *FreeMemory = FALSE;

    if (fsim != NULL) {
        if (volume->file_system_manager != fsim) {
            /* The FSIM doesn't match.  Remove this object from the list. */
            result = TRUE;
        }
    }

    *Error = DLIST_SUCCESS;

    LOG_PROC_EXIT_BOOLEAN_INT(result, *Error);
    return result;
}


/*
 * engine_get_volume_list returns a dlist_t of logical_volume_t structures,
 * optionally filtering on the FSIM that manages the volume.  If the FSIM
 * is NULL, all logical_volume_t structures will be returned.
 */
int engine_get_volume_list(plugin_record_t * fsim,
                           dlist_t         * volume_list) {
    int rc = 0;

    dlist_t new_volume_list = CreateList();

    LOG_PROC_ENTRY();

    LOG_DEBUG("Filters:\n");
    LOG_DEBUG("  FSIM:  %s\n", (fsim == NULL) ? "(none)" : fsim->short_name);
    LOG_DEBUG("Destination DLIST:  %p\n", volume_list);

    /* Update the mount points for the volumes. */
    ForEachItem(VolumeList, check_mounted, NULL, TRUE);

    if (new_volume_list != NULL) {
        rc = CopyList(new_volume_list, VolumeList, AppendToList);

        if ((rc == 0) && (fsim != NULL)) {
            /*
             * If the copy went OK and the caller specified a FSIM, use
             * PruneList to remove from the list volumes that are not managed by
             * the FSIM.
             */
            rc = PruneList(new_volume_list,
                           filter_volumes,
                           fsim);
        }

    } else {
        rc = ENOMEM;
    }

    if (rc == 0) {
        *volume_list = new_volume_list;
    } else {
        if (new_volume_list != NULL) {
            DestroyList(&new_volume_list, FALSE);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*-----------------------------*\
| Functions for getting objects |
\*-----------------------------*/

typedef struct object_filters_s {
    object_type_t         object_type;
    data_type_t           data_type;
    plugin_record_t     * plugin;
    object_search_flags_t flags;
} object_filters_t;

/*
 * filter_objects is used by engine_get_object_list to strip out of the list
 * any objects that are not desired.  filter_objects is called by the PruneList
 * processor.  Parameter points to a object_filters_t.  If the object does not
 * match any of the filters specified, then this function returns TRUE to tell
 * PruneList to remove the object from the list.  Otherwise, it returns FALSE
 * to leave the object in the list.
 */
static BOOLEAN filter_objects(ADDRESS   Object,
                              TAG       ObjectTag,
                              uint      ObjectSize,
                              ADDRESS   ObjectHandle,
                              ADDRESS   Parameters,
                              BOOLEAN * FreeMemory,
                              uint    * Error) {

    storage_object_t * object = (storage_object_t *) Object;
    object_filters_t * filters = (object_filters_t *) Parameters;

    BOOLEAN result = FALSE;  /* Assume we want to keep this one. */

    LOG_PROC_ENTRY();

    /*
     * We never want to free the item memory, so set this to FALSE
     * just in case.
     */
    *FreeMemory = FALSE;

    if (filters->object_type != 0) {
        if (!(filters->object_type & object->object_type)) {
            /*
             * The object type doesn't match.
             * Remove this object from the list.
             */
            result = TRUE;
        }
    }

    if (filters->data_type != 0) {
        if (!(filters->data_type & object->data_type)) {
            /*
             * The data type doesn't match.
             * Remove this object from the list.
             */
            result = TRUE;
        }
    }

    if (filters->plugin != NULL) {
        if (filters->plugin != object->plugin) {
            /*
             * The plug-in doesn't match.
             * Remove this object from the list.
             */
            result = TRUE;
        }
    }

    if (filters->flags & TOPMOST) {
        if (!is_top_object(object)) {
            /*
             * The object is not the topmost.
             * Remove this object from the list.
             */
            result = TRUE;
        }
    }

    if (filters->flags & NOT_MUST_BE_TOP) {
        if (object->flags & SOFLAG_MUST_BE_TOP) {
            /*
             * The object insists on being topmost, but the caller doesn't
             * want objects like that.  Remove it from the list.
             */
            result = TRUE;
        }
    }

    if (filters->flags & WRITEABLE) {
        if (object->flags & (SOFLAG_READ_ONLY | SOFLAG_IO_ERROR | SOFLAG_CORRUPT)) {
            /*
             * The object is either read-only, has had an I/O error, or is
             * corrupt.  It is not writeable.  Remove it from the list.
             */
            result = TRUE;
        }
    }

    *Error = DLIST_SUCCESS;

    LOG_PROC_EXIT_BOOLEAN_INT(result, *Error);
    return result;
}


/*
 * engine_get_object_list returns a dlist_t of storage_object_t structures,
 * optionally filtering on the data_type, plug-in, and search flags.
 * feature is NULL, all storage_object_t structures will be returned.
 */
int engine_get_object_list(object_type_t         object_type,
                           data_type_t           data_type,
                           plugin_record_t     * plugin,
                           object_search_flags_t flags,
                           dlist_t             * objects) {
    int rc = 0;

    dlist_t new_object_list = CreateList();

    LOG_PROC_ENTRY();

    LOG_DEBUG("Filters:\n");
    LOG_DEBUG("  Object type:  0x%x\n", object_type);
    LOG_DEBUG("  Data type:    0x%x\n", data_type);
    LOG_DEBUG("  Plug-in:      %s\n", (plugin == NULL) ? "(none)" : plugin->short_name);
    LOG_DEBUG("  Flags:        0x%x\n", flags);
    LOG_DEBUG("Destination DLIST:  %p\n", objects);

    if (new_object_list != NULL) {

        if ((object_type == 0) || (object_type & DISK)) {
            rc = CopyList(new_object_list, DiskList, AppendToList);
        }

        if ((rc == 0) && ((object_type == 0) || (object_type & SEGMENT))) {
            rc = CopyList(new_object_list, SegmentList, AppendToList);
        }

        if ((rc == 0) && ((object_type == 0) || (object_type & REGION))) {
            rc = CopyList(new_object_list, RegionList, AppendToList);
        }

        if ((rc == 0) && ((object_type == 0) || (object_type & EVMS_OBJECT))) {
            rc = CopyList(new_object_list, EVMSObjectList, AppendToList);
        }

        if (rc == 0) {

            /* Get a structure for the filters. */
            object_filters_t * filters = malloc(sizeof(object_filters_t));

            if (filters != NULL) {

                /*
                 * Fill in the filters and then use PruneList to remove from the
                 * list objects that are not wanted.
                 */
                filters->object_type = object_type;
                filters->data_type   = data_type;
                filters->plugin      = plugin;
                filters->flags       = flags;

                rc = PruneList(new_object_list,
                               filter_objects,
                               filters);

                free(filters);

            } else {
                LOG_CRITICAL("Error allocating memory for object filters.\n");
                rc = ENOMEM;
            }
        }

    } else {
        rc = ENOMEM;
    }

    if (rc == 0) {
        *objects = new_object_list;
    } else {
        if (new_object_list != NULL) {
            DestroyList(&new_object_list, FALSE);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*--------------------------------*\
| Functions for getting containers |
\*--------------------------------*/

/*
 * filter_containers is used by engine_get_Container_list to strip out of the
 * list any containers that are not desired.  filter_containers is called by the
 * PruneList processor.  Parameter points to a plugin_record_t for a region
 * manager.  If the object is not managed by the specified region manager, then
 * this function returns TRUE to tell PruneList to remove the object from the
 * list.  Otherwise, it returns FALSE to leave the object in the list.
 */
static BOOLEAN filter_containers(ADDRESS   Object,
                                 TAG       ObjectTag,
                                 uint      ObjectSize,
                                 ADDRESS   ObjectHandle,
                                 ADDRESS   Parameters,
                                 BOOLEAN * FreeMemory,
                                 uint    * Error) {

    storage_container_t * container = (storage_container_t *) Object;
    plugin_record_t * plugin = (plugin_record_t *) Parameters;

    BOOLEAN result = FALSE;  /* Assume we want to keep this one. */

    LOG_PROC_ENTRY();

    /*
     * We never want to free the item memory, so set this to FALSE
     * just in case.
     */
    *FreeMemory = FALSE;

    if (plugin != NULL) {
        if (container->plugin != plugin) {
            /*
             * The plug-in doesn't match.  Remove this container from the list.
             */
            result = TRUE;
        }
    }

    *Error = DLIST_SUCCESS;

    LOG_PROC_EXIT_BOOLEAN_INT(result, *Error);
    return result;
}


/*
 * engine_get_container_list returns a dlist_t of storage_container_t
 * structures, optionally filtering on the plug-in that manages the container.
 * If plugin is NULL, all storage_container_t structures will be returned.
 */
int engine_get_container_list(plugin_record_t * plugin,
                              dlist_t *         container_list) {
    int rc = 0;

    dlist_t new_container_list = CreateList();

    LOG_PROC_ENTRY();

    LOG_DEBUG("Filters:\n");
    LOG_DEBUG("  Plug-in:  %s\n", (plugin == NULL) ? "(none)" : plugin->short_name);
    LOG_DEBUG("Destination DLIST:  %p\n", container_list);

    if (new_container_list != NULL) {
        rc = CopyList(new_container_list, ContainerList, AppendToList);

        if ((rc == 0) && (plugin != NULL)) {
            /*
             * If the copy went OK and the caller specified a plug-in, use
             * PruneList to remove from the list containers that are not managed
             * by the plug-in.
             */
            rc = PruneList(new_container_list,
                           filter_containers,
                           plugin);
        }

    } else {
        rc = ENOMEM;
    }

    if (rc == 0) {
        *container_list = new_container_list;
    } else {
        if (new_container_list != NULL) {
            DestroyList(&new_container_list, FALSE);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*------------------------------------------------*\
| Functions for communicating with the EVMS kernel |
\*------------------------------------------------*/

/*
 * Issue an ioctl to the EVMS kernel block device
 */
static int engine_ioctl_evms_kernel(unsigned long cmd, void * arg) {

    int rc = 0;

    LOG_PROC_ENTRY();

    /* Get the handle to the EVMS kernel block device. */
    if (evms_block_dev_handle == 0) {
        open_evms_block_dev();
    }

    if (evms_block_dev_handle >= 0) {

        /* Send the requested command to the kernel. */
        rc = ioctl(evms_block_dev_handle, cmd, arg);

        /* If the Engine was not opened for write access, close our access to
         * the EVMS kernel block device so that other potential instances of
         * the Engine can open the device.
         */
        if (engine_mode != ENGINE_READWRITE) {
            close_evms_block_dev();
        }

    } else {
        /*
         * If open_evms_block_dev() returns a negative number it is an error
         * code.  Return the error to the caller.
         */
        rc = evms_block_dev_handle;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*-----------------------------------------------------------*\
| Functions for allocating and freeing Engine data structures |
\*-----------------------------------------------------------*/

/*
 * allocate_new_storage_object takes care of the common functions needed to
 * allocate any kind of storage object.  It allocates zero filled memory,
 * creates the dlists, and initializes the flags.
 */
static int allocate_new_storage_object(storage_object_t * * new_object) {
    int rc = 0;

    storage_object_t * object = NULL;

    LOG_PROC_ENTRY();

    object = (storage_object_t *) calloc(1, sizeof(storage_object_t));
    if (object != NULL) {

        object->parent_objects = CreateList();
        if (object->parent_objects != NULL) {

            object->child_objects = CreateList();
            if (object->child_objects != NULL) {

                /*
                 * The default data type is DATA_TYPE.  The plug-in can change
                 * it if necessary.
                 */
                object->data_type = DATA_TYPE;

                /*
                 * If we are not in discovery, this must be a new object so mark
                 * it new and dirty.
                 */
                if (!discover_in_progress) {
                    object->flags |= SOFLAG_NEW | SOFLAG_DIRTY;
                }

            } else {
                DestroyList(&object->parent_objects, FALSE);
                rc = ENOMEM;
            }

        } else {
            rc = ENOMEM;
        }

        if (rc != DLIST_SUCCESS) {
            free(object);
            object = NULL;
        }

    } else {
        rc = ENOMEM;
    }

    *new_object = object;

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * free_old_storage_object takes care of the functions that are common for
 * cleaning up any type of storage object.  It destroys the app-handle, if one
 * was created, destroys the dlists in the object, and frees the object.
 */
static void free_old_storage_object(storage_object_t * object) {

    LOG_PROC_ENTRY();

    if (object->app_handle != 0) {
        destroy_handle(object->app_handle);
    }

    DestroyList(&object->parent_objects, FALSE);
    DestroyList(&object->child_objects, FALSE);

    free(object);

    LOG_PROC_EXIT_VOID();
}


/*
 * Free a storage_object_t for a logical disk.
 */
int engine_free_logical_disk(storage_object_t * disk) {
    int rc = 0;

    LOG_PROC_ENTRY();

    engine_unregister_name(disk->name);

    DeleteObject(DiskList,
                 disk);

    free_old_storage_object(disk);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Allocate a storage_object_t for a logical disk.
 * If a name is specified, register the name.
 */
static int engine_allocate_logical_disk(char * name, storage_object_t * * new_disk) {
    int rc = 0;

    storage_object_t * disk;

    LOG_PROC_ENTRY();

    *new_disk = NULL;

    if (name != NULL) {
        rc = engine_validate_name(name);
    }

    if (rc == 0) {
        rc = allocate_new_storage_object(&disk);

        if (rc == 0) {
            void * trash;

            disk->object_type = DISK;

            rc = InsertObject(DiskList,
                              sizeof(storage_object_t),
                              disk,
                              DISK_TAG,
                              NULL,
                              AppendToList,
                              FALSE,
                              &trash);

            if (rc == DLIST_SUCCESS) {
                if (name != NULL) {
                    rc = engine_register_name(name);

                    if (rc == 0) {
                        strcpy(disk->name, name);
                    } else {
                        DeleteObject(DiskList, disk);
                    }
                }
            }

            if (rc != 0) {
                engine_free_logical_disk(disk);
                disk = NULL;
            }
        }

        *new_disk = disk;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Free a storage_object_t for a disk segment.
 */
int engine_free_segment(storage_object_t * segment) {
    int rc = 0;

    LOG_PROC_ENTRY();

    engine_unregister_name(segment->name);

    DeleteObject(SegmentList,
                 segment);

    free_old_storage_object(segment);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Allocate a storage_object_t for a disk segment.
 * If a name is specified, register the name.
 */
static int engine_allocate_segment(char * name, storage_object_t * * new_segment) {
    int rc = 0;

    storage_object_t * segment;

    LOG_PROC_ENTRY();

    *new_segment = NULL;

    if (name != NULL) {
        rc = engine_validate_name(name);
    }

    if (rc == 0) {
        rc = allocate_new_storage_object(&segment);

        if (rc == 0) {
            void * trash;

            segment->object_type = SEGMENT;

            rc = InsertObject(SegmentList,
                              sizeof(storage_object_t),
                              segment,
                              SEGMENT_TAG,
                              NULL,
                              AppendToList,
                              FALSE,
                              &trash);

            if (rc == DLIST_SUCCESS) {
                if (name != NULL) {
                    rc = engine_register_name(name);

                    if (rc == 0) {
                        strcpy(segment->name, name);
                    } else {
                        DeleteObject(SegmentList, segment);
                    }
                }
            }

            if (rc != 0) {
                engine_free_segment(segment);
                segment = NULL;
            }
        }

        *new_segment = segment;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Free a storage_object_t for a region.
 */
int engine_free_region(storage_object_t * region) {
    int rc = 0;

    LOG_PROC_ENTRY();

    engine_unregister_name(region->name);

    DeleteObject(RegionList,
                 region);

    free_old_storage_object(region);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Allocate a storage_object_t for a region.
 * If a name is specified, register the name.
 */
static int engine_allocate_region(char * name, storage_object_t * * new_region) {
    int rc = 0;

    storage_object_t * region;

    LOG_PROC_ENTRY();

    *new_region = NULL;

    if (name != NULL) {
        rc = engine_validate_name(name);
    }

    if (rc == 0) {
        rc = allocate_new_storage_object(&region);

        if (rc == 0) {
            void * trash;

            region->object_type = REGION;

            rc = InsertObject(RegionList,
                              sizeof(storage_object_t),
                              region,
                              REGION_TAG,
                              NULL,
                              AppendToList,
                              FALSE,
                              &trash);

            if (rc == DLIST_SUCCESS) {
                if (name != NULL) {
                    rc = engine_register_name(name);

                    if (rc == 0) {
                        strcpy(region->name, name);
                    } else {
                        DeleteObject(RegionList, region);
                    }
                }
            }

            if (rc != 0) {
                engine_free_region(region);
                region = NULL;
            }
        }

        *new_region = region;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Free a storage_object_t for an EVMS object.
 */
int engine_free_evms_object(storage_object_t * object) {
    int rc = 0;

    LOG_PROC_ENTRY();

    engine_unregister_name(object->name);

    DeleteObject(EVMSObjectList,
                 object);

    free_old_storage_object(object);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Allocate a storage_object_t for an EVMS object.
 * If a name is specified, register the name.
 */
static int engine_allocate_evms_object(char * name, storage_object_t * * new_object) {
    int rc = 0;

    storage_object_t * object;

    LOG_PROC_ENTRY();

    *new_object = NULL;

    if (name != NULL) {
        rc = engine_validate_name(name);
    }

    if (rc == 0) {
        rc = allocate_new_storage_object(&object);

        if (rc == 0) {
            void * trash;

            object->object_type = EVMS_OBJECT;

            rc = InsertObject(EVMSObjectList,
                              sizeof(storage_object_t),
                              object,
                              EVMS_OBJECT_TAG,
                              NULL,
                              AppendToList,
                              FALSE,
                              &trash);

            if (rc == DLIST_SUCCESS) {
                if (name != NULL) {
                    rc = engine_register_name(name);

                    if (rc == 0) {
                        strcpy(object->name, name);
                    } else {
                        DeleteObject(EVMSObjectList, object);
                    }
                }
            }

            if (rc != 0) {
                engine_free_evms_object(object);
                object = NULL;
            }
        }

        *new_object = object;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Allocate a storage_container structure filled with zeros, create the dlist_t
 * anchors for the segments and regions_exported lists, mark the structure
 * dirty, and add it to ContainerList, the global list of storage containers.
 * If a name is specified, register the name.
 */
static int engine_allocate_container(char * name, storage_container_t * * new_container) {
    int rc = 0;

    storage_container_t * container;

    LOG_PROC_ENTRY();

    *new_container = NULL;

    if (name != NULL) {
        rc = engine_validate_name(name);
    }

    if (rc == 0) {
        container = (storage_container_t *) calloc(1, sizeof(storage_container_t));
        if (container != NULL) {

            container->objects_consumed = CreateList();
            if (container->objects_consumed != NULL) {

                container->objects_produced = CreateList();
                if (container->objects_produced != NULL) {
                    void * trash;

                    /*
                     * If we are not in discovery, this must be a new container
                     * so mark it dirty.
                     */
                    if (!discover_in_progress) {
                        container->flags |= SCFLAG_DIRTY;
                    }

                    rc = InsertObject(ContainerList,
                                      sizeof(storage_container_t),
                                      container,
                                      CONTAINER_TAG,
                                      NULL,
                                      AppendToList,
                                      FALSE,
                                      &trash);

                    if (rc == DLIST_SUCCESS) {
                        if (name != NULL) {
                            rc = engine_register_name(name);

                            if (rc == 0) {
                                strcpy(container->name, name);
                            } else {
                                DeleteObject(ContainerList, container);
                            }
                        }
                    }

                    if (rc != 0) {
                        DestroyList(&container->objects_produced, FALSE);
                    }

                } else {
                    rc = ENOMEM;
                }

                if (rc != 0) {
                    DestroyList(&container->objects_consumed, FALSE);
                }

            } else {
                rc = ENOMEM;
            }

            if (rc != DLIST_SUCCESS) {
                free(container);
                container = NULL;
            }

        } else {
            rc = ENOMEM;
        }

        *new_container = container;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Free a storage_container structure: Destroy the app_handle if we created one,
 * destroy the segments and regions_exported dlist_ts, and remove the structure
 * from ContainerList, the global list of storage containers.
 */
int engine_free_container(storage_container_t * container) {
    int rc = 0;

    LOG_PROC_ENTRY();

    if (container->app_handle != 0) {
        destroy_handle(container->app_handle);
    }

    DestroyList(&container->objects_consumed, FALSE);
    DestroyList(&container->objects_produced, FALSE);

    engine_unregister_name(container->name);

    DeleteObject(ContainerList,
                 container);

    free(container);

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * engine_alloc is a service provided so that plug-ins use the same memory
 * management library that the Engine does.  This is especially needed if
 * the Engine is ever expected to free the memory.
 */
static void * engine_alloc(u_int32_t size) {

    void * pMem = NULL;

    LOG_PROC_ENTRY();

    if ( size > 0 ) {
        pMem = calloc(1, size);
    }

    LOG_PROC_EXIT_PTR(pMem);
    return pMem;
}


/*
 * engine_free is the compliment to engine_alloc.
 */
static void engine_free(void * thing) {

    LOG_PROC_ENTRY();

    LOG_DEBUG("Request to free memory at %08x.\n", thing);

    if (thing) {
        free(thing);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Set the Engine's changes_pending flag.  The Engine will set changes_pending
 * as it sees calls such as create() go by.  However, there are instances, such
 * as with direct-plugin_communication(), that the Engine can't determine that
 * changes are pending.  This function allows plug-in to tell the Engine that
 * changes are pending, i.e., there is stuff to commit to disk.
 */
static void engine_set_changes_pending() {

    LOG_PROC_ENTRY();

    changes_pending = TRUE;

    LOG_PROC_EXIT_VOID();
}


static BOOLEAN engine_commit_in_progress() {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_BOOLEAN(commit_in_progress);
    return(commit_in_progress);
}


static int plugin_write_log_entry(debug_level_t     level,
                                  plugin_record_t * plugin,
                                  char            * fmt,
                                  ...) {
    int rc = 0;
    int len;
    va_list args;

    /*
     * Don't log entry into this function.  It gets in the way of what the
     * plug-in is trying to write.
     */

    if (level <= debug_level) {
        if (log_file > 0) {
            timestamp(log_buf, LOG_BUF_SIZE);

            if (plugin != NULL) {
                strcat(log_buf, plugin->short_name);
            } else {
                strcat(log_buf, "Bad plug-in pointer");
            }

            strcat(log_buf, ": ");
            len = strlen(log_buf);

            va_start(args, fmt);
            len += vsprintf(log_buf + strlen(log_buf), fmt, args);
            va_end(args);

            if (write(log_file, log_buf, len) < 0) {
                rc = errno;
            }

        } else {
            rc = ENOENT;
        }
    }

    /*
     * Don't log exit from this function.  It gets in the way of what the
     * plug-in is trying to write.
     */
    return rc;
}


/*
 * Call the kernel to calculate a checksum.
 */
static int engine_calculate_checksum(unsigned char * buffer,
                                     int             buffer_size,
                                     unsigned int    insum,
                                     unsigned int  * outsum) {
    int rc = 0;

    evms_compute_csum_t csum;

    csum.buffer_address = buffer;
    csum.buffer_size = buffer_size;
    csum.insum = insum;
    csum.outsum = 0;
    csum.status = 0;

    rc = ioctl(evms_block_dev_handle, EVMS_COMPUTE_CSUM, &csum);

    /* If the ioctl worked, return the status code. */
    if (rc == 0) {
        rc = csum.status;

        /* If the call worked, return the checksum generated. */
        if (rc == 0) {
            *outsum = csum.outsum;
        }
    }

    return rc;
}


/*
 * Tell the Engine to put some sectors on its kill list.
 */
static int engine_add_sectors_to_kill_list(storage_object_t * disk, lba_t lba, sector_count_t count) {
    kill_sector_record_t * ksr = calloc(1, sizeof(kill_sector_record_t));
    ADDRESS                handle;
    int                    error = 0;

    LOG_PROC_ENTRY();

    if (ksr != NULL) {
        if (disk != NULL) {
            if (lba <= disk->size) {
                if (lba + count <= disk->size) {
                    if (KillSectorList != NULL) {
                        ksr->logical_disk  = disk;
                        ksr->sector_offset = lba;
                        ksr->sector_count  = count;

                        error = InsertObject(KillSectorList,
                                             sizeof(kill_sector_record_t),
                                             ksr,
                                             KILL_SECTOR_TAG,
                                             NULL,
                                             InsertAtStart,
                                             TRUE,
                                             &handle);

                        if (error == DLIST_SUCCESS) {
                            LOG_DEBUG("Request queued to kill %lld sector%s on disk %s at LBA %lld.\n", ksr->sector_count, (ksr->sector_count == 1) ? "" : "s", ksr->logical_disk->name, ksr->sector_offset);
                        }

                    } else {
                        LOG_CRITICAL("The KillSectorList has not been created.\n");
                        error = ENOMEM;
                    }

                } else {
                    LOG_ERROR("The count of sectors (%d at LBA %lld) goes past the end of the disk (disk size is %lld).\n", ksr->sector_count, ksr->sector_offset, disk->size);
                    error = EINVAL;
                }

            } else {
                LOG_ERROR("The starting LBA of the kill sectors (%lld) is past the end of the disk (disk size is %lld).\n", ksr->sector_offset, disk->size);
                error = EINVAL;
            }

        } else {
            LOG_ERROR("The pointer for the disk is NULL.\n");
            error = EINVAL;
        }

    } else {
        LOG_CRITICAL("Error allocating memory for a kill sector record.\n");
        error = ENOMEM;
    }

    LOG_PROC_EXIT_INT(error);
    return error;
}


/*
 * Tell the Engine that a volume needs to be rediscovered.
 */
int engine_rediscover_volume(logical_volume_t * volume, BOOLEAN sync_fs) {
    int rc = 0;
    ADDRESS handle;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Request to do a rediscover on volume %s.\n", volume->name);

    if (!(volume->flags & VOLFLAG_NEW)) {

        /*
         * We can only allow the volume to appear once on the
         * SoftVolumeDeleteList.  The kernel can only delete the volume once.
         * If the volume were to appear more than once on the list, we would ask
         * the kernel to delete it more than once.  All deletes after the first
         * one would fail, since the volume would be gone, and our commit logic
         * would fail because the delete failed.
         * To ensure there is only one copy of the volume on the
         * SoftVolumeDeleteList, issue a DeleteObject to remove it if it's on
         * the list already.  Ignore the return code from DeleteObject since it
         * may fail if the volume is not on the list.
         */
        DeleteObject(SoftVolumeDeleteList,
                     volume);

        rc = InsertObject(SoftVolumeDeleteList,
                          sizeof(logical_volume_t),
                          volume,
                          VOLUME_TAG,
                          NULL,
                          AppendToList,
                          FALSE,
                          &handle);

        if (rc == 0) {
            if (sync_fs) {
                volume->flags |= VOLFLAG_SYNC_FS;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Validate that an object name is not too long and that there is no other
 * object of the same type with the same name.
 */
int engine_validate_name(char * name) {

    int rc = 0;
    name_list_entry_t * pNameEntry = name_registry;

    LOG_PROC_ENTRY();

    if (name != NULL) {

        LOG_DEBUG("Name to validate is %s.\n", name);

        if (strlen(name) <= EVMS_NAME_SIZE) {

            while ((pNameEntry != NULL) && (strcmp(pNameEntry->name, name) != 0)) {
                pNameEntry = pNameEntry->next;
            }

            if (pNameEntry != NULL) {
                /*
                 * We found a matching name.
                 */
                LOG_DEBUG("Name %s is already in the registry.\n", name);
                rc = EEXIST;
            }

        } else {
            /* The name is too long. */
            LOG_DEBUG("The name is too long.  It must be %d or fewer characters.\n", EVMS_NAME_SIZE);
            rc = EOVERFLOW;
        }

    } else {
        LOG_ERROR("Pointer to name is NULL.\n");
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Add a name to the name registry.
 */
int engine_register_name(char * name) {

    int rc = 0;
    name_list_entry_t * * pNameEntry = &name_registry;
    name_list_entry_t * new_entry;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Name to register is %s.\n", name);

    rc = engine_validate_name(name);

    if (rc == 0) {
        /*
         * Create a new name registry entry and add it to the appropriate list.
         */

        new_entry = (name_list_entry_t *) malloc(sizeof(name_list_entry_t));

        if (new_entry != NULL) {
            new_entry->name = strdup(name);

            if (new_entry->name != NULL) {
                new_entry->next = *pNameEntry;
                *pNameEntry = new_entry;

            } else {
                /* Could not get memory for the name. */
                LOG_CRITICAL("Error getting memory for the name in the new name registry entry.\n");
                free(new_entry);
                rc = ENOMEM;
            }

        } else {
            LOG_CRITICAL("Error getting memory for the new name registry entry.\n");
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Remove a name from the name registry.
 */
int engine_unregister_name(char * name) {
    int rc = 0;

    name_list_entry_t * * pNameEntry = &name_registry;

    LOG_PROC_ENTRY();

    if (name != NULL) {

        LOG_DEBUG("Name to unregister is %s.\n", name);

        while ((*pNameEntry != NULL) && (strcmp((*pNameEntry)->name, name) != 0)) {
            pNameEntry = &((*pNameEntry)->next);
        }

        if (*pNameEntry != NULL) {
            /*
             * We found the name.  Remove it from the list.
             */
            name_list_entry_t * entry = *pNameEntry;

            *pNameEntry = (*pNameEntry)->next;

            free(entry->name);
            free(entry);

        } else {
            /* We did not find the name. */
            LOG_DEBUG("Name %s is not in the registry.\n", name);
            rc = ENOENT;
        }

    } else {
        LOG_ERROR("Pointer to name is NULL.\n");
        rc = EINVAL;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Free up all the entries in the name registry.
 */
void clear_name_registry(void) {

    LOG_PROC_ENTRY();

    while (name_registry != NULL) {
        name_list_entry_t * entry = name_registry;

        LOG_DEBUG("Free name registry entry for \"%s\".\n", entry->name);

        name_registry = name_registry->next;

        free(entry->name);
        free(entry);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 * Check if there are any restrictions that would prevent this object from
 * being renamed.
 */
int engine_can_rename(storage_object_t * obj) {

    int rc;
    uint count = 0;
    BOOLEAN is_top_object = TRUE;
    BOOLEAN has_associative_feature = FALSE;

    LOG_PROC_ENTRY();

    /*
     * If the object is not part of a volume, then it is free to change its
     * name.
     */
    if (obj->volume == NULL) {
        LOG_PROC_EXIT_INT(0);
        return 0;
    }

    /*
     * Check if the object is a top level object, i.e., it has no parent;
     * or its parent is an associative feature that snuk in on top of the
     * object, in which case it is virtually a top level object.
     */
    GetListSize(obj->parent_objects, &count);

    if (count == 0) {
        is_top_object = TRUE;

    } else {
        /*
         * The object is not a top level object.  Check if its parent is an
         * associative feature that snuck in on top of the object, and the
         * associative feature has no parents.
         */
        storage_object_t * parent;
        TAG tag;
        uint size;

        rc = BlindGetObject(obj->parent_objects,
                            &size,
                            &tag,
                            NULL,
                            FALSE,
                            (ADDRESS *) &parent);

        if (rc == DLIST_SUCCESS) {
            GetListSize(parent->parent_objects, &count);
            if ((GetPluginType(parent->plugin->id) == EVMS_ASSOCIATIVE_FEATURE) &&
                (count == 0)) {
                is_top_object = TRUE;
                has_associative_feature = TRUE;

            } else {
                is_top_object = FALSE;
            }

        } else {
            /*
             * We couldn't get the parent object.  Assume the most restrictive
             * case, i.e., it is a top object.
             */
            is_top_object = TRUE;
        }
    }

    if (is_top_object) {

        /*
         * If the object is an EVMS_OBJECT, then it must be part of an
         * EVMS volume and its name is allowed to change since it doesn't
         * affect the volume name.
         */
        if (obj->object_type == EVMS_OBJECT) {
            LOG_PROC_EXIT_INT(0);
            return 0;
        }

        /* If the object has  a feature header on it, that feature header is
         * for making an EVMS volume.  An object that is part of an EVMS
         * volume can always change its name since that doesn't affect the
         * volume name.
         */
        if (obj->feature_header != NULL) {
            LOG_PROC_EXIT_INT(0);
            return 0;

        } else {
            /*
             * The object is the top level object in a compatibility volume.
             * Check to see if the volume is mounted.  If it is, don't allow
             * the object name to be changed since that would change the
             * volume name.  If the volume is not mounted, it's OK to change
             * the object name.
             */
            if (is_volume_mounted(obj->volume)) {
                LOG_PROC_EXIT_INT(EPERM);
                return EPERM;

            } else {

                /*
                 * If this object has an associative feature above it, then
                 * the name cannot be changed.  It is the top object of a
                 * compatibility volume which means the volume name changes
                 * with the object name.  Associative features are tied to
                 * the volume name.  We cannot let the volume name change
                 * and therefore cannot let the object name change.
                 */
                if (has_associative_feature) {
                    LOG_PROC_EXIT_INT(EPERM);
                    return EPERM;

                } else {
                    /*
                     * The object is the top level object of a compatibility
                     * volume that is not mounted.  The name can be changed.
                     * the evms_set_info() code will take care of changing the
                     * volume name if the object name changes.
                     */
                    LOG_PROC_EXIT_INT(0);
                    return 0;
                }
            }
        }

    } else {

        /* The object is not a top level object.  Its name can be changed. */
        LOG_PROC_EXIT_INT(0);
        return 0;
    }

    /*
     * We shouldn't get here.  All of the if branches above should return.
     * But just in case, fail the query.
     */
    LOG_PROC_EXIT_INT(EPERM);
    return EPERM;
}


engine_functions_t engine_functions = {
    get_plugin_list:               engine_get_plugin_list,
    get_plugin_by_ID:              engine_get_plugin_by_ID,
    get_plugin_by_name:            engine_get_plugin_by_name,
    get_volume_list:               engine_get_volume_list,
    get_object_list:               engine_get_object_list,
    get_container_list:            engine_get_container_list,
    ioctl_evms_kernel:             engine_ioctl_evms_kernel,
    allocate_logical_disk:         engine_allocate_logical_disk,
    free_logical_disk:             engine_free_logical_disk,
    allocate_segment:              engine_allocate_segment,
    free_segment:                  engine_free_segment,
    allocate_container:            engine_allocate_container,
    free_container:                engine_free_container,
    allocate_region:               engine_allocate_region,
    free_region:                   engine_free_region,
    allocate_evms_object:          engine_allocate_evms_object,
    free_evms_object:              engine_free_evms_object,
    engine_alloc:                  engine_alloc,
    engine_free:                   engine_free,
    changes_pending:               evms_changes_pending,
    set_changes_pending:           engine_set_changes_pending,
    commit_in_progress:            engine_commit_in_progress,
    write_log_entry:               plugin_write_log_entry,
    calculate_CRC:                 calculate_CRC,
    calculate_checksum:            engine_calculate_checksum,
    add_sectors_to_kill_list:      engine_add_sectors_to_kill_list,
    rediscover_volume:             engine_rediscover_volume,
    validate_name:                 engine_validate_name,
    register_name:                 engine_register_name,
    unregister_name:               engine_unregister_name,
    can_expand_by:                 engine_can_expand_by,
    can_shrink_by:                 engine_can_shrink_by,
    user_message:                  plugin_user_message,
    user_communication:            plugin_user_communication,
    can_rename:                    engine_can_rename,
    is_mounted:                    is_mounted
};

