#!/bin/sh
set -e
# Example upload handler script for em_autobuild
# Ensure you customise this script and make it
# executable for the user running em_autobuild.
# Then pass the full path to this script to em_autobuild
# with the --handler option.

# Exits upon any error so that failures can be traced
# back to the relevant package.

# The skeleton script parses the build log, separates
# the processing of successful builds from unsuccessful builds
# and provides configuration variables for the uploads.

#  Copyright (C) 2008 Neil Williams <codehelp@debian.org>
#
#  This package is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

# In general, two upload methods are needed - dput is the simplest way
# of reliably uploading Debian packages to a repository and rsync is
# probably the simplest way of uploading build logs to a location on
# the server.

# Package uploads need a post_upload script that can put the TDebs into
# a locale repository and the target packages into a target repository.

# The dput.cfg file needs to use the "emdebian-target" and "emdebian-locale"
# labels for the appropriate repositories whether or not www.emdebian.org
# is the destination of the upload. If the relevant labels exist in your
# existing ~/dput.cf file, DPUT_CFG can be left empty.

# To use noninteractively, ensure that the user running em_autobuild
# can use passphrase-less SSH keys. (em_autobuild -l will fail if sudo
# is used).

# The final handler is called once per package by em_autobuild.
# Sample command (as root):
# /# em_autobuild -a arm -p -o -l -u /path/myupload_handler.sh

# Alternatively, run the handler after the complete
# build using find with xargs:
# find /opt/emdebian/buildd/ -maxdepth 4 -a -name '*.log'

# Note that the autobuilder generally creates all files and result
# directories as owned by root:root. The upload handler and the scripts
# that it calls also need to be run as root (or a method arranged
# to run 'sudo chown -R $USER:$USER $WORKDIR/pbuilder/result'.)

LOGFILE="$1"
if [ -z "$LOGFILE" ]; then
	echo "Usage: $0 LOGFILE"
	echo
	echo "Example upload handler script for em_autobuild."
	echo "Ensure you customise this script and make it executable for the "
	echo "user running em_autobuild. Then pass the full path to this script "
	echo "to em_autobuild with the --handler option."
	echo
	echo "Sample command (as root):"
	echo "em_autobuild -a arm -p -o -l -u /path/myupload_handler.sh"
	echo
	exit 0
fi

############# Config #############

# Where to upload the build log (top level directory)
# for rsync: URL="user@server:/path/"
# log for apt then goes into server:/path/a/apt/trunk/
# For Emdebian, the URL should be: 
# username@buildd.emdebian.org:/var/emdebian/buildd/logs/
# Remember: this is run as root, not as the user
# which gives the chance to use different SSH keys.
URL=
# How to upload the log file (including any options)
METHOD=
# e.g. METHOD="rsync -qLzrc" for quiet operation that respects
# existing symlinks.

# Specify an alternative dput config file if needed.
# (will be used by emrecent for package uploads)
DPUT_CFG=

# To use a different uploader for successfully built packages
# modify this config variable - the path to the $CHANGES file 
# and a suitable DPUT_CFG option will be appended, as appropriate.
EMRECENT="emrecent --noninteractive"

# Set to any string value to use quiet mode -
# make sure your configuration works first.
QUIET=

############## End Config ########

# the only argument is the build log path.
if [ -z "$QUIET" ]; then
	echo "Processing $LOGFILE"
fi

# Your emdebian working directory, e.g. /opt/emdebian/
WORKDIR=`perl -e 'use Cwd; use Emdebian::Tools; use Config::Auto; use Debian::DpkgCross; \
&read_config; \
my $w = &get_workdir; \
$w = cwd if (! -d $w); \
print $w;';`

SUCCESS=`grep "Successful build" $LOGFILE`
CHANGES=`grep "Changes file" $LOGFILE | grep -v tdeb`
TCHANGES=`grep "Changes file" $LOGFILE | grep tdeb || true`

# work out where to put the logs
LDIR="${WORKDIR}buildd/"
DIR=`grep "Build directory: " $LOGFILE`
if [ ! -z "$DIR" ]; then
	DIR=`echo $DIR | sed -e 's/Build directory: //'`
	DIR=`echo $DIR | sed -e 's/\/trunk\///'`
	DIR=`dirname $DIR`
else
	# unsuccessful logs might not include "Build directory: "
	DIR=`echo $LOGFILE | sed -e "s:${LDIR}::" | sed -e 's:/[a-z0-9\-]*\.log::'`
fi

if [ ! -z "$CHANGES" ]; then
	CHANGES=`echo $CHANGES | sed -e 's/Changes file://'`
	CHANGES=`echo $CHANGES | sed -e 's/\/trunk\///'`
	CHANGES="${WORKDIR}pbuilder/result/$CHANGES"
	UPLOAD=`echo $CHANGES | sed -e 's/\.changes/.upload/'`
fi
if [ ! -z "$TCHANGES" ]; then
	TCHANGES=`echo $TCHANGES | sed -e 's/Changes file://'`
	TCHANGES=`echo $TCHANGES | sed -e 's/\/trunk\///'`
	TCHANGES=`echo $TCHANGES | sed -e 's/\.\.\///'`
	TCHANGES="$DIR/$TCHANGES"
	TCHANGES="${WORKDIR}pbuilder/result/$TCHANGES"
fi

if [ ! -z "$DPUT_CFG" ]; then
	DPUT_CFG="--dput-cfg $DPUT_CFG"
fi

if [ ! -z "$SUCCESS" -a ! -z "$CHANGES" ]; then
	# run emrecent here and append output to the logfile
	# Append the emrecent output to the logfile so that the report
	# page gets accurate data on failures. Only upload TDebs if the
	# target package upload succeeds.
	if echo "$EMRECENT $DPUT_CFG $CHANGES >> $LOGFILE 2>&1" | /bin/bash; then
		if [ ! -z "$TCHANGES" ]; then
			# no need to log the TDeb upload
			echo $EMRECENT $DPUT_CFG $TCHANGES
			# $EMRECENT $DPUT_CFG $TCHANGES &>/dev/null || true
		fi
	fi
	# (could choose to only append the .upload file)
	if [ -f "$UPLOAD" ]; then
		# comment out the echo and uncomment the replacement
		echo cat $UPLOAD
		# cat $UPLOAD >> $LOGFILE
	fi
else
	if [ -z "$QUIET" ]; then
		echo "Unsuccessful build, only uploading the build logs in $LDIR."
	fi
fi

# Upload $LOGFILE here - actually rsync the entire path
# so that missing directories are created on the server.
# e.g. rsync -vaz /opt/emdebian/buildd/* $URL
# rsync does this efficiently because only the changes are sent.
# (This could be done in one go at the end but this way the webpage
# is updated in real time.)
if [ ! -z "$METHOD" ]; then
	if [ -z "$QUIET" ]; then
		echo "Running $METHOD ${LDIR}* $URL"
	fi
	$METHOD ${LDIR}* $URL
fi
