// ------------------------------------------------------------------------
// interface.cpp: Main user interface widget for ecamegapedal
// Copyright (C) 2001 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <qlayout.h>
#include <qstatusbar.h>
#include <qtimer.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qaccel.h>
#include <qlabel.h>
#include <qfiledialog.h>
#include <qtimer.h>

#include <kvutils/kvu_numtostr.h>
#include <ecasound/eca-session.h>
#include <ecasound/eca-control.h>
#include <ecasound/eca-version.h>

#include "version.h"
#include "qeaudiotrap.h"
#include "qevumeter.h"
#include "qebuttonrow.h"
#include "qechainoperatorinput.h"

#include "interface.h"

QEInterface::QEInterface(const string& input,
			 const string& output,
			 QWidget *parent, 
			 const char *name)
  : QWidget(parent, name)
{
  setCaption("ecamegapedal v" + QString(ecamegapedal_version) + " - the real audio processor!");
  qApp->setMainWidget(this);
  accel_repp = new QAccel(this);
  copinput_repp = 0;
  esession_repp = new ECA_SESSION();
  ectrl_repp = new ECA_CONTROL(esession_repp);
  input_rep = input;
  output_rep = output;
  io_changed_rep = false;
  init_layout();
  init_chainsetup();
}

QEInterface::~QEInterface(void) {
  delete ectrl_repp;
  delete esession_repp;
}

void QEInterface::init_chainsetup(void) {
  /* make sure we're not running */
  if (ectrl_repp->is_connected() == true) stop_processing();

  /* clear out the previous chainsetup */
  if (ectrl_repp->selected_chainsetup() == "ecamegapedal") {
    ectrl_repp->remove_chainsetup();
  }

  /* initialize a new chainsetup */
  ectrl_repp->add_chainsetup("ecamegapedal");
  ectrl_repp->toggle_interactive_mode(true);

#if ECASOUND_LIBRARY_VERSION_CURRENT > 7
  ectrl_repp->set_chainsetup_buffersize(256);
#else
  ectrl_repp->set_buffersize(256);
#endif

  ectrl_repp->set_chainsetup_parameter("-z:nointbuf");
  ectrl_repp->add_chain("ecamegapedal");
  ectrl_repp->add_audio_input(input_rep);
  ectrl_repp->add_audio_output(output_rep);

  io_changed_rep = false;
}

void QEInterface::init_layout(void) {
  QBoxLayout* topLayout = new QVBoxLayout( this );
  QBoxLayout* bottomrow = new QHBoxLayout();

  init_buttons();
  init_io_rows();

  copinput_repp = new QEChainOperatorInput(this, "qechainop"); 
  QObject::connect(copinput_repp, 
		   SIGNAL(parameters_changed()), 
		   this, 
		   SLOT(update_processing_parameters()));
  QObject::connect(copinput_repp, 
		   SIGNAL(operator_changed()), 
		   this, 
		   SLOT(update_processing_operator()));

  copinput_repp->setMaximumHeight(500);
  topLayout->addWidget(copinput_repp);
  topLayout->addLayout(io_rows_repp);
  topLayout->addWidget(buttonrow_repp);

  vumeter_repp = new QEVUMeter(this, 0);
  vumeter_repp->setDelay(500);
  topLayout->addWidget(vumeter_repp);

  topLayout->addLayout(bottomrow,0);
  init_bottomrow(bottomrow);
}


void QEInterface::init_bottomrow(QBoxLayout* bottomrow) {
  init_statusbar();
  bottomrow->addWidget(statusbar_repp, 2, 0);
}

void QEInterface::init_statusbar(void) {
  statusbar_repp = new QStatusBar(this, "qsbar");
  statusbar_repp->message(QString("ecamegapedal v") + QString(ecamegapedal_version) + QString(" ready.")); 

  QTimer *timer = new QTimer( this );
  connect( timer, SIGNAL(timeout()), this, SLOT(update_statusbar()));
  timer->start( 500, false);
}

void QEInterface::update_statusbar(void) { 
  static string last_epstatus;
  string new_epstatus = ectrl_repp->engine_status();
  position_update();

  //  if (new_epstatus == last_epstatus) return;
  //  else 
  last_epstatus = new_epstatus;

  string pos = kvu_numtostr(curpos_rep, 3);
  statusbar_repp->message(QString(" [ ") + 
			  last_epstatus.c_str() + 
			  ", " + 
			  QString(pos.c_str()) + 
			  "sec ] . (C) 2001 Kai Vehmanen");
}

void QEInterface::init_io_rows(void) {
  io_rows_repp = new QGridLayout(0, 2, 4, 2, -1, "io_grid");

  QLabel* tmp = new QLabel("(I)nput: ", this, 0);
  io_rows_repp->addWidget(tmp, 0, 0);

  inputrow_name_repp = new QLineEdit(this, "inputrow_name");
  inputrow_name_repp->setText(input_rep.c_str());
  io_rows_repp->addWidget(inputrow_name_repp, 0, 1);

//    inputrow_format_repp = new QLineEdit(this, "inputrow_format");
//    io_rows_repp->addWidget(inputrow_format_repp, 0, 2);

  QPushButton* inputrow_browse = new QPushButton("(B)rowse", this, 0);
  io_rows_repp->addWidget(inputrow_browse, 0, 3);

  io_rows_repp->addWidget(new QLabel("(O)utput: ", this, 0), 1, 0);

  outputrow_name_repp = new QLineEdit(this, "outputrow_name");
  outputrow_name_repp->setText(output_rep.c_str());
  io_rows_repp->addWidget(outputrow_name_repp, 1, 1);

//    outputrow_format_repp = new QLineEdit(this, "outputrow_format");
//    io_rows_repp->addWidget(outputrow_format_repp, 1, 2);

  QPushButton* outputrow_browse = new QPushButton("Bro(w)se", this, 0);
  io_rows_repp->addWidget(outputrow_browse, 1, 3);

  io_rows_repp->setMargin(5);

  QObject::connect(inputrow_browse, SIGNAL(clicked()), this, SLOT(button_input_browse()));
  QObject::connect(inputrow_name_repp, SIGNAL(textChanged(const QString&)), 
		   this, SLOT(set_input_name(const QString&)));
					      
  QObject::connect(outputrow_browse, SIGNAL(clicked()), this, SLOT(button_output_browse()));
  QObject::connect(outputrow_name_repp, SIGNAL(textChanged(const QString&)), 
		   this, SLOT(set_output_name(const QString&)));

  accel_repp->connectItem(accel_repp->insertItem(CTRL+Key_B), this, SLOT(button_input_browse()));
  accel_repp->connectItem(accel_repp->insertItem(CTRL+Key_W), this, SLOT(button_output_browse()));

  accel_repp->connectItem(accel_repp->insertItem(CTRL+Key_I), inputrow_name_repp, SLOT(setFocus()));
  accel_repp->connectItem(accel_repp->insertItem(CTRL+Key_O), outputrow_name_repp, SLOT(setFocus()));
}

void QEInterface::init_buttons(void) {
  buttonrow_repp = new QEButtonRow(this, "buttonrow");
  buttonrow_repp->add_button(new QPushButton("Be(g)in",buttonrow_repp),
			CTRL+Key_G,
			this, SLOT(button_rewind_begin()));

  buttonrow_repp->add_button(new QPushButton("Rew (<)",buttonrow_repp), 
			Key_Less,
			this, SLOT(button_rewind()));

  buttonrow_repp->add_button(new QPushButton("S(t)art",buttonrow_repp), 
			CTRL+Key_T,
			this, SLOT(start_processing()));

  buttonrow_repp->add_button(new QPushButton("(S)top",buttonrow_repp), 
			CTRL+Key_S,
			this, SLOT(stop_processing()));

  buttonrow_repp->add_button(new QPushButton("Fw (>)",buttonrow_repp), 
			SHIFT+Key_Greater,
			this, SLOT(button_forward()));

  buttonrow_repp->add_button(new QPushButton("(Q)uit",buttonrow_repp), 
			CTRL+Key_Q,
			qApp, SLOT(closeAllWindows()));


}

void QEInterface::start_processing(void) {  
  bool full_stop = false;

  if (io_changed_rep == true ||
      ectrl_repp->is_connected() != true) full_stop = true;

  if (ectrl_repp->is_running() == true) ectrl_repp->stop_on_condition();    
  if (full_stop == true) {
    if (ectrl_repp->is_connected() == true) ectrl_repp->disconnect_chainsetup();
    init_chainsetup();
  }

  ectrl_repp->select_chainsetup("ecamegapedal");
  ectrl_repp->select_chain("ecamegapedal");
  ectrl_repp->clear_chains();

  QEAudioTrap* atrap = new QEAudioTrap(this, 0, vumeter_repp);
//    QObject::connect(atrap, 
//  		   SIGNAL(dataUpdated(const SAMPLE_BUFFER*)), 
//  		   vumeter_repp, 
//  		   SLOT(dataUpdated(const SAMPLE_BUFFER*)));

  if (copinput_repp != 0) {
    copinput_repp->update_results();
    OPERATOR* c = dynamic_cast<OPERATOR*>(copinput_repp->result());
    if (c != 0) {
      c = c->clone();
      ectrl_repp->add_chain_operator(dynamic_cast<CHAIN_OPERATOR*>(c));
    }
    /* FIXME: should we manually set the parameters here? */
  }
  ectrl_repp->add_chain_operator(dynamic_cast<CHAIN_OPERATOR*>(atrap));

  if (full_stop == true) {
    ectrl_repp->connect_chainsetup();
  }

  if (ectrl_repp->is_connected() == true) ectrl_repp->start();
  else {
    QMessageBox* mbox = new QMessageBox(this, "mbox");
    mbox->information(this, "ecamegapedal", "Unable to start processing!",0);
    io_changed_rep = true;
  }
}

void QEInterface::update_processing_operator(void) {  
  if (ectrl_repp->is_running() == true) {
    start_processing();
  }
}

void QEInterface::update_processing_parameters(void) {  
  if (ectrl_repp->is_running() == true) {
    if (copinput_repp != 0) {
      /* get the chainop controller by the GUI controls */
      copinput_repp->update_results();
      CHAIN_OPERATOR* newcop = copinput_repp->result();
      if (newcop != 0) {
	/* select the current setup */
	ectrl_repp->select_chainsetup("ecamegapedal");
	ectrl_repp->select_chain("ecamegapedal");

	/* get the active chainop from the engine */
	ectrl_repp->select_chain_operator(1);
	const CHAIN_OPERATOR* cop = 0;
	cop = ectrl_repp->get_chain_operator();
	if (cop != 0) {
//  	  cerr << "Found a chain operator." << endl;
	  for(int n = 0; n < cop->number_of_params(); n++) {
//  	    cerr << "Param " << n << ": ";
	    ectrl_repp->select_chain_operator_parameter(n + 1);
//  	    cerr << " old:" <<
//  	      ectrl_repp->get_chain_operator_parameter();
//  	    cerr << ", new:" << newcop->get_parameter(n + 1) << endl;
	    ectrl_repp->set_chain_operator_parameter(newcop->get_parameter(n + 1));
	  }
	}
      }
    }
  }
}

void QEInterface::stop_processing(void) {  
  if (ectrl_repp->is_running() == true) ectrl_repp->stop();
  if (ectrl_repp->is_connected() == true) ectrl_repp->disconnect_chainsetup();
}

void QEInterface::close_session(void) {  }

void QEInterface::button_input_browse(void) {  
  QFileDialog* fdialog = new QFileDialog(0, 0, true);
  fdialog->setMode(QFileDialog::ExistingFile);    
  fdialog->setCaption("Selecting an input file...");
  fdialog->setDir(current_dir_rep.c_str());
  fdialog->exec();
  QString fname = fdialog->selectedFile();
  if (!fname.isEmpty()) {
    current_dir_rep = string(fdialog->dirPath().latin1());
    inputrow_name_repp->setText(fname);
  }
}

void QEInterface::button_output_browse(void) {  
  QFileDialog* fdialog = new QFileDialog(0, 0, true);
  fdialog->setMode(QFileDialog::AnyFile);
  fdialog->setCaption("Selecting an output file...");
  fdialog->setDir(current_dir_rep.c_str());
  fdialog->exec();
  QString fname = fdialog->selectedFile();
  if (!fname.isEmpty()) {
    current_dir_rep = string(fdialog->dirPath().latin1());
    outputrow_name_repp->setText(fname);
  }
}

void QEInterface::button_rewind_begin(void) { ectrl_repp->command("setpos 0"); }
void QEInterface::button_rewind(void) { ectrl_repp->command("rw 5"); }
void QEInterface::button_forward(void) { ectrl_repp->command("fw 5"); }

void QEInterface::set_input_name(const QString& q) {
  string s (q.latin1());
  if (s != input_rep) {
    input_rep = s;
    io_changed_rep = true;
  }
}

void QEInterface::set_output_name(const QString& q) {
  string s (q.latin1());
  if (s != output_rep) {
    output_rep = s;
    io_changed_rep = true;
  }
}

void QEInterface::position_update(void) {
  curpos_rep = ectrl_repp->position_in_seconds_exact();
}

void QEInterface::not_implemented(void) {
  QMessageBox* mbox = new QMessageBox(this, "mbox");
  mbox->information(this, "ecamegapedal", "This feature is not implemented...",0);
}
