/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#include <wx/filename.h>

#include "Common.hh"
#include "AddressBookFrame.hh"
#include "ContactDlg.hh"
#include "ImportContactFrame.hh"
#include "ImportContactChoice.hh"
#include "PropertiesFrame.hh"
#include "AddCrlDlg.hh"

#include "../AddressBook.hh"

//#include "pics/resources.h" //icons 
#ifndef __WXMSW__ // en attendant
#include "pics/add_contact.xpm"
#include "pics/search_contact.xpm"
#include "pics/remove_contact.xpm"
#include "pics/exit.xpm"
#include "pics/import.xpm"
#include "pics/config.xpm"
#endif


using namespace Cryptonit;


BEGIN_EVENT_TABLE(AddressBookFrame, wxFrame)
    EVT_TOOL (AB_CLOSE_BTN_ID,AddressBookFrame::exit)
    EVT_CLOSE(AddressBookFrame::exit2)
    EVT_TOOL (AB_ADD_CONTACT_BTN_ID, AddressBookFrame::addContact)
    EVT_TOOL (AB_REMOVE_CONTACT_BTN_ID, AddressBookFrame::removeContact)
    EVT_TOOL (AB_SETTINGS_BTN_ID, AddressBookFrame::configFields)
    EVT_LIST_ITEM_ACTIVATED (AB_SELECT_CONTACT_ID, AddressBookFrame::editContact)
    EVT_LIST_COL_CLICK (AB_SELECT_CONTACT_ID, AddressBookFrame::onColumnClick)
END_EVENT_TABLE()


/* Struct and function used by wxListCtrl::SortItems */
struct compareContactsDatas {
    User *user;
    wxString *column;
    bool ascending;
};

int wxCALLBACK compareContacts(long item1, long item2, long sortData)
{
    struct compareContactsDatas* datas;
    datas = (compareContactsDatas*)sortData;

    std::string col = wx2std(*((wxString*)datas->column));
    std::string* dn1 = (std::string*)item1;
    std::string* dn2 = (std::string*)item2;

    // Do we have to sort in ascending order
    if( datas->ascending ) {
	return strcasecmp( datas->user->addressBook->getContactInfo(*dn1, col).c_str(),
			   datas->user->addressBook->getContactInfo(*dn2, col).c_str() );
    } else { // or descending order
	return( -1 * strcasecmp( datas->user->addressBook->getContactInfo(*dn1, col).c_str(),
				 datas->user->addressBook->getContactInfo(*dn2, col).c_str() ));
    }
}



AddressBookFrame::AddressBookFrame( wxWindow* parent, 
				    wxWindowID id, 
				    const wxString& title, 
				    User* u,
				    std::vector<KeyStore *> *kstore,
				    const wxPoint& pos, 
				    const wxSize& size, 
				    long style, 
				    const wxString& name)
    :wxFrame(parent, id, title,  pos,wxSize(510,300) ,style, name)
{
    p = (MainWindow *) parent;
    user = u;
    keyStore = kstore;
    contactList = NULL;

    initFieldList();  
    createToolBar();
    CreateStatusBar();
      
    wxBoxSizer *mainBox = new wxBoxSizer(wxVERTICAL);
      
    wxPanel *panel1 = new wxPanel( this );
      
    createLocalAddressBookPage( panel1 );

    mainBox->Add( panel1 );

#ifdef __WXMSW__
    SetIcon(wxICON(aaa));
#endif

    Centre();
    Show();
}
  


/* Initalize the contact fields to be displayed in the contactList.
 * contactFieldList[0] is the text to be displayed in the contactList header
 * contactFieldList[1] is the real information field in the Contact class.
 */
void AddressBookFrame::initFieldList()
{
    contactFieldList[0].Clear();
    contactFieldList[1].Clear();
    
    std::vector<std::string> fields = user->getInfos("AddressBookFields");

    // Use the user's fields
    if( fields.size() > 0 ) {
	std::vector<std::string>::iterator it;
	for( it = fields.begin(); it != fields.end(); it++ ) {

	    /* Search information's details from the ContactInfo::AttributeList */
	    unsigned int i = 0;
	    while( ContactInfo::AttributeList[i][0] != NULL && strcmp(ContactInfo::AttributeList[i][0], it->c_str()) != 0 )
		i++;
		
	    if( ContactInfo::AttributeList[i][0] != NULL ) {
					contactFieldList[0].Add( std2wx(ContactInfo::AttributeList[i][1]) );
					contactFieldList[1].Add( std2wx(ContactInfo::AttributeList[i][0]) );
	    }
	}
	
    } // Otherwise add some defaults fields
    else {
	contactFieldList[0].Add(_T("CN"));
	contactFieldList[0].Add(_T("Mail"));
	contactFieldList[0].Add(_T("SN"));

	contactFieldList[1].Add(_T("cn"));
	contactFieldList[1].Add(_T("mail"));
	contactFieldList[1].Add(_T("sn"));
	
	/* And add it to the Preference file */
	std::vector<std::string> fields;
	fields.push_back("cn");
	fields.push_back("mail");
	fields.push_back("sn");
	user->setInfos( "AddressBookFields", fields );
    }

    /* This array is used for sorting in an ascending/descending way
     * each columns.
     */
	int count = contactFieldList[0].GetCount();
    contactFieldListAscending = new bool[count];
	for(int i = 0; i < count; i++) {
		contactFieldListAscending[i] = false;
	}
}


AddressBookFrame::~AddressBookFrame()
{
    if( contactList != NULL ) {
	for( int i=0 ; i < contactList->GetItemCount() ; i++ ){
	    std::string *data = (std::string*) contactList->GetItemData( i );
	    if( data ){
		delete data;
	    }
	}
    }

    contactFieldList[0].Clear();
    contactFieldList[1].Clear();
}



void AddressBookFrame::createLocalAddressBookPage( wxPanel *panel )
{
    contactList = new wxListCtrl( panel, AB_SELECT_CONTACT_ID, wxDefaultPosition, wxDefaultSize, 
				  wxLC_REPORT | wxLC_HRULES | wxLC_SORT_ASCENDING );
    
    for( unsigned int i = 0; i < contactFieldList[0].GetCount(); i++ ) {
	contactList->InsertColumn( i, contactFieldList[0][i], wxLIST_FORMAT_LEFT, 195 );
    }

    refreshContactList();

    wxBoxSizer *sizerPanel = new wxBoxSizer( wxVERTICAL );
    sizerPanel->Add( contactList, 1, wxEXPAND);
    panel->SetSizer( sizerPanel );
}


void AddressBookFrame::refreshLocalAddressBookPage()
{
    initFieldList();
    contactList->ClearAll();

    for( unsigned int i = 0; i < contactFieldList[0].GetCount(); i++ ) {
	contactList->InsertColumn( i, contactFieldList[0][i], wxLIST_FORMAT_LEFT, 195 );
    }
    
    refreshContactList();
}


void AddressBookFrame::createToolBar()
{
    wxBitmap toolBarBitmaps[6];

    toolBarBitmaps[0] = wxBITMAP(add_contact);
    toolBarBitmaps[1] = wxBITMAP(remove_contact);
    toolBarBitmaps[2] = wxBITMAP(search_contact);
    toolBarBitmaps[3] = wxBITMAP(import);
    toolBarBitmaps[4] = wxBITMAP(exit);
    toolBarBitmaps[5] = wxBITMAP(config);

#ifdef __WXMSW__
    addressBookToolBar = CreateToolBar(wxTB_FLAT |wxTB_TEXT |  wxTB_HORIZONTAL | wxNO_BORDER, 500);
    addressBookToolBar->SetToolBitmapSize(wxSize(32, 32));
#else
    addressBookToolBar = new wxToolBar( this, AB_TOOLBAR_ID, wxDefaultPosition, wxDefaultSize, 
					wxTB_FLAT |wxTB_TEXT |  wxTB_HORIZONTAL | wxNO_BORDER);
#endif

    addressBookToolBar->AddTool( AB_ADD_CONTACT_BTN_ID, _("Add"), toolBarBitmaps[0], _("Add a new contact"));
    addressBookToolBar->AddTool( AB_REMOVE_CONTACT_BTN_ID, _("Remove"), toolBarBitmaps[1], _("Remove selected contact"));
    //  addressBookToolBar->AddTool( AB_SEARCH_CONTACT_BTN_ID, _("Search"), toolBarBitmaps[2],_("Search a contact"));
    addressBookToolBar->AddTool( AB_SETTINGS_BTN_ID, _("Settings"), toolBarBitmaps[5],_("Configure the displayed fields"));
    addressBookToolBar->AddSeparator();
    addressBookToolBar->AddTool( AB_CLOSE_BTN_ID, _("Close"), toolBarBitmaps[4], _("Close the address book"));

    addressBookToolBar->EnableTool( AB_SEARCH_CONTACT_BTN_ID, false );

#ifndef __WXMSW__
    SetToolBar(addressBookToolBar);
#endif
    addressBookToolBar->Realize();
 
    addressBookToolBar->SetRows(1);
}

void AddressBookFrame::createStatusBar()
{
    addressBookStatusBar = new wxStatusBar( this, -1 );
    CreateStatusBar(1);
}


void AddressBookFrame::refreshContactList()
{
    contactList->DeleteAllItems();

    if( user->addressBook != NULL ) {
	std::vector<std::string> *names = user->addressBook->getAllContactName();
	std::vector<std::string>::iterator itr;
	long index;
	
	// Hide temporarly the list to speed up inserting
	contactList->Hide();
	
	for( itr = names->begin(); itr != names->end(); itr++ ) {
	    wxListItem item;
	    item.SetId(0);
	    item.SetColumn(0);
	    item.SetMask(wxLIST_MASK_DATA | wxLIST_MASK_TEXT);

	    // Store the contact DN into the item 'data' attribute
	    std::string *data = new std::string( *itr );
	    item.SetData( (void*)data );

	    // Add the first field
	    item.SetText( std2wx(user->addressBook->getContactInfo(*itr, wx2std(contactFieldList[1][0]))) );

	    // Get the index where the item was inserted since the original ID
	    // is not kept under Windows
	    index = contactList->InsertItem( item );

	    // Fill columns with all other fields
	    for( size_t j = 1; j < contactFieldList[1].GetCount(); j++ ) {
		contactList->SetItem( index, j, 
													std2wx(user->addressBook->getContactInfo(*itr, wx2std(contactFieldList[1][j]))) );
	    }
			  
	}

	contactList->Show();

	// Fill the status bar with the number of contacts
	if( contactList->GetItemCount() > 0 ) {
	    wxString buffer;
	    buffer << contactList->GetItemCount() << _(" entries");
	    SetStatusText( buffer );
	}
	if( contactList->GetItemCount() == 1 )
	    SetStatusText( _("One entry.") );
	
	if( contactList->GetItemCount() == 0 )
	    SetStatusText( _("No entry.") );

	// Sort contacts using the first field
	struct compareContactsDatas data;
	data.user = user;
	data.column = new wxString(contactFieldList[1][0]);
	data.ascending = contactFieldListAscending[0];

	contactList->SortItems( compareContacts, (long)&data );

	contactFieldListAscending[0] = true;
	delete data.column;
	    
	names->clear();
	delete names;
    }
}

void AddressBookFrame::exit(wxCommandEvent& WXUNUSED(event))
{
    p->enableAddressBookBtn( true );
    Destroy();
}

void AddressBookFrame::exit2(wxCloseEvent& WXUNUSED(event))
{
		wxCommandEvent e;
		this->exit(e);
}

void AddressBookFrame::addContact(wxCommandEvent& WXUNUSED(event))
{
    ImportContactChoice icc(this, -1, _("Import a new contact"));
  
    if(icc.showModal() == wxID_OK) {

	if(icc.getTypeOfImport() == FILE_CONTACT_IMPORT ) {
	    // Import from a file
	    wxString wildcard;
	    wildcard << _("All supported formats") << _T("|*.cer;*.der;*.pem")
							 << _T("|") << _("Certificate") << _T(" (*.cer)|*.cer")
							 << _T("|") << _("DER encoded certificate") << _T(" (*.der)|*.der")
							 << _T("|") << _("PEM encoded certificate") << _T(" (*.pem)|*.pem");
	    
	    wxFileDialog fileDlg( this, _("Choose a certificate to import"),
														wxGetCwd(), _T(""), wildcard,
				  wxOPEN | wxCHANGE_DIR);

	    if( fileDlg.ShowModal() == wxID_OK ) {
		wxFileName certFilename(fileDlg.GetFilename());
		certFilename = wxMyPrepend(certFilename,fileDlg.GetDirectory());
		
		ContactDlg cd(this, -1, certFilename.GetFullPath(), user ,_("Add a contact"));
		
		if( cd.showModal(wxSize(450,400)) == wxID_OK){
		    int ret;
		    wxString errorMsg = wxEmptyString;
		    ret = cd.addContact(user, certFilename.GetFullPath());

		    switch( ret ) {
		    case -1: errorMsg << wxString::Format(_("Cannot read contact's certificate %s, aborting"),certFilename.GetFullPath().c_str());
			break;
		    case -2: errorMsg = _("An error occured while writing the contact's certificate on disk.");
			break;
		    case -3: errorMsg = _("An error occured while adding contact: ");
			break;
		    }

		    if( errorMsg != wxEmptyString ) {
			wxMessageBox( errorMsg, _("Error"), wxICON_ERROR | wxOK, this );
		    }

		    // ask the user if he wants to download the new contact crl.
		    Certificate newCert;
		    if( newCert.load( wx2std(certFilename.GetFullPath()).c_str() ) == SUCCESS ){
			std::string uri = newCert.extractCrlUri();
			if( uri != "" && Crl::outDated(uri , user , &newCert)){
			    int answer = wxMessageBox( _("Do you want to download a CRL for this certificate? "), _("Question") , wxICON_QUESTION | wxYES_NO , this );
			    if( answer == wxYES ){
							wxString newCrlFile = AddCrlDlg::fetchCrl( std2wx(uri) );
				Crl newCrl;
				if( newCrl.load( wx2std(newCrlFile).c_str()) == SUCCESS ){
				    AddCrlDlg::saveCrl( std2wx(uri) , &newCrl , newCert , user ); 
				    
				} else {
				    wxMessageBox( _("The downloaded CRL is invalid! "), _("Error") , wxICON_ERROR | wxOK , this );
				}
			    }
			} else if(uri ==""){
			    wxMessageBox( _("This contact certificate have no CRL.\nPlease add a CRL in the Certificate Viewer 'CRL' tab."),
					  _("Info") , wxICON_INFORMATION | wxOK , this );
			}
						
		    }
		    refreshContactList();
		}
	    }

	} else {
	    // Import from LDAP
	    ImportContactFrame *ic = new  ImportContactFrame(this, -1 ,_("Import a Contact"), user, keyStore);
	    ic->Show();
	}
    }
}




void AddressBookFrame::editContact(wxListEvent& event)
{
    wxListItem info;
    info.m_itemId = event.m_itemIndex;
    info.m_col = 3;
    info.m_mask = wxLIST_MASK_TEXT;
    
    wxListItem name;
    name.SetId(event.m_itemIndex);
    name.SetColumn(0);
    name.SetMask( wxLIST_MASK_TEXT | wxLIST_MASK_DATA );
    
    if( contactList->GetItem(info) && contactList->GetItem(name) ) {
	wxString title = _("Edit contact: ") + info.m_text;
	
	std::string *dn = (std::string*)name.GetData();

	if(dn){
	    ContactDlg cd(this, -1, user->addressBook->getContactDSE(*dn), user ,false, title );
	    
	    if(cd.showModal(wxSize(450,400)) == wxID_OK){
		cd.saveContact( user->addressBook->getContactDSE(*dn ));
		refreshContactList();
	    }
	}
    }
    else {
#ifdef DEBUG	    
	std::cout << "GetItem failed" << std::endl;
#endif
    }
}


  
void AddressBookFrame::removeContact(wxCommandEvent& event)
{
    if( contactList->GetSelectedItemCount() == 0 ) {
	wxMessageDialog errorMsg(this, _("There is no selected contact"), _("Error"), wxOK | wxICON_ERROR);
	errorMsg.ShowModal();
	return;
    }

    long item = -1;
    while( true ) {

	/* Iterate through all selected items */
        item = contactList->GetNextItem(item,
					wxLIST_NEXT_ALL,
					wxLIST_STATE_SELECTED);
        if ( item == -1 )
            break;


	wxListItem currentSelectedItem;
	currentSelectedItem.SetId(item);
	currentSelectedItem.SetData(wxLIST_MASK_DATA);

	if( contactList->GetItem(currentSelectedItem) ) {
	  std::string *dn = (std::string*)currentSelectedItem.GetData();
	  if(dn) {
	    /* Delete associated certificate(s) file(s) */
	    wxFileName fileToDelete = std2wx(user->addressBook->getContactInfo(*dn, "userCertificate"));
	    fileToDelete=wxMyPrepend(fileToDelete,std2wx(user->getCertificatesDir()));
	    //wxLogMessage("file to Delete : %s", (fileToDelete.GetFullPath()).c_str());

	    wxRemoveFile(fileToDelete.GetFullPath() );
	    
	    /* Remove the contact from the address book */
	    user->addressBook->removeContact( *dn );
	  } else {
	    wxLogMessage(_("An error occurred!"));
	  }
	}
    }
    refreshContactList();
}


void AddressBookFrame::onColumnClick(wxListEvent& event)
{
    struct compareContactsDatas datas;


    /* Give the adresse of the buffer containing 
     * the title of the clicked column.
     */
    wxString *col = new wxString( contactFieldList[1][event.GetColumn()] );

    datas.column = col;
    datas.user = user;
    datas.ascending = contactFieldListAscending[event.GetColumn()];

    contactList->SortItems( compareContacts, (long)&datas );

    contactFieldListAscending[event.GetColumn()] = contactFieldListAscending[event.GetColumn()] ? false : true;

    delete col;
}


void AddressBookFrame::configFields(wxCommandEvent& WXUNUSED(event))
{
    PropertiesFrame *pf = new  PropertiesFrame(this, -1 ,_("Properties"), user, keyStore);
    pf->displayPage( 4 );
    pf->ShowModal();
    refreshLocalAddressBookPage();
}
