/* cdrom.c
Copyright (c) 2000-2002 Craig Condit, Stefan Hlswitt.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met: 

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer. 
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution. 

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.
*/

#define _LARGEFILE64_SOURCE

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <linux/cdrom.h>
#include <sys/ioctl.h>

#include "cdbackup.h"
#include "misc.h"

/* size of leadin/out depending of how many tracks are on cd */
#define MIN_FREE_1	(11400*CD_FRAMESIZE) /* 1th track */
#define MIN_FREE_2	(6900*CD_FRAMESIZE)  /* 2nd and more tracks */

struct toc_entry *toc=0;
long long cd_used, cd_avail;

extern long cd_len;
extern int verbose;
extern char *prg_name;

/****************************************************************************/

int open_cdr(char *device) 
{
  return open(device, O_RDONLY | O_NONBLOCK);
}

/****************************************************************************/

void close_cdr(int cd_fd)
{
  close(cd_fd);
}

/****************************************************************************/

int full_read(int fd, void *buf, int count)
{
  int total=0;

  while(count>0) {
    int bytes;
    bytes=read(fd,buf,count); if(bytes<0) return bytes;
    if(bytes==0) break;
    count-=bytes; total+=bytes; buf+=bytes;
    }
  return total;
}

/****************************************************************************/

void free_toc()
{
  if(toc) { free(toc); toc=0; }
}

/****************************************************************************/

int read_toc(int cd_fd, int trackinfos)
{
  int i;
  struct cdrom_tochdr cd_header;
  struct cdrom_tocentry cd_entry;
  int tracks,start_track;
 
  /* read table of contents header */
  if(ioctl(cd_fd,CDROMREADTOCHDR,&cd_header)) {
    if(verbose) fprintf(stderr,"%s: Unable to read CD header, assuming empty CD-R\n", prg_name);
    cd_used=0; cd_avail=(long long)cd_len*CD_FRAMESIZE; return 0;
    }

  /* get start and end tracks */
  start_track = cd_header.cdth_trk0;
  tracks = cd_header.cdth_trk1-start_track+1;

  free_toc();
  if(!(toc=calloc(tracks+1,sizeof(struct toc_entry)))) serror("No memory for TOC");

  /* set some parameters */
  cd_entry.cdte_format=CDROM_LBA;

  /* read lead-out */
  cd_entry.cdte_track=CDROM_LEADOUT;
  if(ioctl(cd_fd,CDROMREADTOCENTRY,&cd_entry)) error("Error reading lead-out");

  toc[0].track_no=CDROM_LEADOUT;  /* not a real track */
  toc[0].sec_start=cd_entry.cdte_addr.lba;

  cd_used=(long long)toc[0].sec_start*CD_FRAMESIZE; cd_avail=(long long)cd_len*CD_FRAMESIZE-cd_used;
  cd_avail-=(tracks>1?MIN_FREE_2:MIN_FREE_1);
  if(cd_avail<0) cd_avail=0; /* can be <0 due to assumed additional lead-in/out */

  /* read rest of tracks */
  for(i=1; i<=tracks; i++) {
    cd_entry.cdte_track=start_track+i-1;
    if(ioctl(cd_fd,CDROMREADTOCENTRY,&cd_entry)) error("Error reading TOC");  

    toc[i].track_no=start_track+i-1;
    toc[i].sec_start=cd_entry.cdte_addr.lba;
    if(cd_entry.cdte_ctrl&CDROM_DATA_TRACK) toc[i].is_data=1;
    }    

 if(trackinfos)
  /* now loop through tracks and read header info */
  for(i=1; i<=tracks; i++) {
    char inbuffer[CD_FRAMESIZE];
    struct header_block *track_header=(struct header_block *)&inbuffer[0];

    if (toc[i].is_data == 1) {
      if(lseek64(cd_fd,(long long)toc[i].sec_start*CD_FRAMESIZE,SEEK_SET)<0) error("Seek failed");

      if(full_read(cd_fd,inbuffer,CD_FRAMESIZE)==CD_FRAMESIZE) {
        if(!strncmp(SHORT_HDR,track_header->id_str,strlen(SHORT_HDR))) {
          toc[i].is_cdbackup=1;
          strncpy(toc[i].id_str,track_header->id_str,32); toc[i].id_str[32]=0;
          strncpy(toc[i].vol_id, track_header->vol_id,32); toc[i].vol_id[32]=0;
          strncpy(toc[i].t_stamp, track_header->t_stamp,12); toc[i].t_stamp[12]=0;
          toc[i].disk_set = track_header->disk_set;
          }
        }      
      }
    }
  
  return tracks;
}

/****************************************************************************/

void print_space() 
{
  if(verbose)
    fprintf(stderr,
      "Disk size:  %7lld kB (%7ld blocks)\n"
      "Space used: %7lld kB (%7lld blocks)\n"
      "Space avail:%7lld kB (%7lld blocks)\n",
      (long long)cd_len*CD_FRAMESIZE/1024,cd_len,
      cd_used/1024, cd_used/CD_FRAMESIZE,
      cd_avail/1024, cd_avail/CD_FRAMESIZE);
}
