/* Dear Emacs, please note this is a -*-pike-*- file. Thank you. */

/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2001 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: calta.pike,v 1.6.4.1 2001/08/20 18:17:04 kiwi Exp $
 *
 * CALTA - CAmas Language Translator Aid
 *
 * A utility to aid in translating/maintaining the CAMAS language
 * files.
 *
 * v0.1
 *
 */

// Force Pike 7.0 compatibility
#pike 7.0 

#define LINE_COMMENT        0x00
#define LINE_MSG_COMMENT    0x01
#define LINE_EMPTY          0x02
#define LINE_MSG_DEF        0x03
#define LINE_DIRECTIVE      0x04

#define MSG_DEFINE "#define"

#define FIRST_SPECIAL       11111

#define	TS_MAINTAINER	    11111
#define	TS_VERSION	        11112
#define	TS_NOMSGFOUND	    11113
#define	TS_LANGNAME	        11114
#define	TS_CHARSET	        11115
#define	TS_ENCCHAR	        11116
#define	TS_CVSVER	        11117
#define	TS_COPYRIGHT	    11118

private static multiset whitespace = (<' ', '\t'>);

private static string MyLocation;
private static string ModuleLocation;

void debug(string fmt)
{
    write(sprintf("Debug: %s\n", fmt));
}

void trace(string fmt)
{
    write(fmt);
}

void wrout(string fmt)
{
    trace(fmt);
}

void warning(string fmt)
{
    write(sprintf("Warning: %s\n", fmt));
}

class LanguageFile 
{
    inherit Stdio.File;

    string                                   language;
    string                                   file;
    array(string)                            lines;
    mapping(int:mapping)                     ids;
    mapping(string:mapping)                  names;
    mapping(string:mapping)                  merged;
    array(string)                            msg_comment;
    mapping(string:string)                   msg_directive;

    static int                               lineno;
    static int                               msg_count;
    static int                               spec_count;
    static int                               isempty;
    static int                               combined, saved;
    
    /*
     * PRIVATE FUNCTIONS
     */
    static int qualify_line(string line)
    {
        if (!line || line == "")
            return LINE_EMPTY;

        if (sizeof(line) >= 3 && line[0..2] == "//!")
            return LINE_MSG_COMMENT;

        if (sizeof(line) >= 3 && line[0..2] == "//#")
            return LINE_DIRECTIVE;
        
        if (sizeof(line) >= 2 && line[0..1] == "//")
            return LINE_COMMENT;

        if (sizeof(line) >= sizeof(MSG_DEFINE) && line[0..(sizeof(MSG_DEFINE) - 1)] == MSG_DEFINE)
            return LINE_MSG_DEF;

        return -1;
    }

    static void handle_msg_comment(string line)
    {
        if (line == "")
            return;

        msg_comment += ({line});
    }

    static string format_msg(string line)
    {
        if (line == "")
            return "**UNSET MESSAGE**";

        if (sizeof(line) >= 2 && line[0..1] == "\\#")
            return replace(line[2..], ({"\\s","\\t"}), ({" ","\t"}));
        else
            return "\"" + replace(line, ({"\\s","\\t"}), ({" ","\t"})) + "\"";
    }

    static void handle_msg_directive(string line)
    {
        if (line == "")
            return;

        int i = 0;
        while(i <= sizeof(line))
            if (!whitespace[line[i]])
                i++;
            else
                break;

        switch(upper_case(line[0..(i-1)])) {
	  
            case "ORIG":
                msg_directive->orig = String.trim_whites(line[i..]);
                break;

            case "FLAG":
                /* Ignored */
                break;
        }
    }
    
    static void handle_msg_define(string line)
    {
        array(string)   msgdef = line / "\t";
        int             withid = 0, msgidx = 3;
	
        if (sizeof(msgdef) < 3 || sizeof(msgdef) > 4)
            throw(({sprintf("Incorrect message definition format at %s(%d)\n"
                            "Expected 3 or 4, got %d elements.\n",
                            file, lineno, sizeof(msgdef)), backtrace()}));

        /* First store it by ID */
	if (sizeof(msgdef) == 4) {
	    withid = (int)String.trim_whites(msgdef[2]);
    	    if (ids[withid])
        	throw(({sprintf("Duplicate ID %d at %s(%d)\nPrevious definition on line %d\n",
                                withid, file, lineno, ids[withid]->lineno), backtrace()}));
	} else
	    msgidx = 2;

        string name = String.trim_whites(msgdef[1]);
        if (names[name])
            throw(({sprintf("Duplicate NAME '%s' at %s(%d)\nPrevious definition on line %d\n",
                            name, file, lineno, names[name]->lineno), backtrace()}));
			    
	if (withid) {
    	    ids[withid] = ([]);
    	    ids[withid]->id = withid;
    	    ids[withid]->lineno = lineno;
    	    ids[withid]->name = String.trim_whites(msgdef[1]);
    	    ids[withid]->msg = String.trim_whites(msgdef[msgidx]);
    	    ids[withid]->comment = "";
    	    ids[withid]->directives = ([]);
	}
        
        /* Now store it by name */
        names[name] = ([]);
        names[name]->name = name;
        names[name]->lineno = lineno;
	if (withid)
    	    names[name]->id = withid;
        names[name]->msg = String.trim_whites(msgdef[msgidx]);
        names[name]->comment = "";
        names[name]->directives = ([]);

        if (withid >= FIRST_SPECIAL) {
            ids[withid]->special = 1;
            names[name]->special = 1;
            spec_count++;
        } else {
            msg_count++;
        }

        if (msg_comment && msg_comment != ({})) {
    	    names[name]->comment = ({});
            foreach(msg_comment, string l)
                names[name]->comment += ({l});

	    if (withid)
        	ids[withid]->comment = names[name]->comment;
            msg_comment = ({});
        }

        if (msg_directive && msg_directive != ([])) {
	    if (withid)
        	ids[withid]->directives = msg_directive;
            names[name]->directives = msg_directive;
            msg_directive = ([]);
        }
    }
    
    static void read_and_parse(int|void re)
    {
        if (isempty) {
            trace(sprintf("Creating file '%s'... \n", basename(file)));
            return;
        }

        if (re) {
            foreach(indices(names), string name)
                m_delete(names, name);
            foreach(indices(ids), int id)
                m_delete(ids, id);
            names = ([]);
            ids = ([]);
            if (!open(file, "rc"))
                throw(({sprintf("Cannot open the input file '%s' for reparse\n", file), backtrace()}));
            lineno = 1;
            msg_count = spec_count = 0;
        }
        
        trace(sprintf("%sarsing file '%s'... ", re ? "Rep" : "P", basename(file)));
        
        lines = read() / "\n";

        close();
        
        /*
         * Walk the array and convert all relevant data into
         * structures which will be used in the further processing.
         */
        lineno = 1;
        foreach(lines, string line) {
            switch(qualify_line(String.trim_whites(line))) {
                case LINE_COMMENT: /* fall through */
                case LINE_EMPTY:
                    break;                    

                case LINE_MSG_COMMENT:
                    handle_msg_comment(String.trim_whites(line)[3..]);
                    break;

                case LINE_DIRECTIVE:
                    handle_msg_directive(String.trim_whites(line)[3..]);
                    break;
                    
                case LINE_MSG_DEF:
                    handle_msg_define(String.trim_whites(line));
                    break;

                default:
                    throw(({sprintf("Unknown line format at %s(%d)\n",
                                    file, lineno), backtrace()}));
            }
            lineno++;
        }
        trace(sprintf("done (%d lines, %d msgs, %d special)\n", sizeof(lines),
                      msg_count, spec_count));
    }

    /*
     * Retrieve a message mapping either by ID or by NAME
     */
    mapping get(string|int msg) 
    {
        if (!msg)
            return 0;
        if (stringp(msg))
            return names[msg];
        else if (intp(msg))
            return ids[msg];

        return 0;
    }
    
    /*
     * public FUNCTIONS
     */

    /*
     * Combine the given file with this object. Combining consists in
     * walking the collection of symbol names present in the 'from'
     * file and modifying accordingly this file.
     * If a message symbol is found in both files, then the routine
     * tests wheter the target file contains an up-to-date version
     * of the message (using the //#ORIG directive). If not, the
     * message is flagged <changed> and the //#ORIG directive updated
     * accordingly.
     * If the target file doesn't contain a message, it is added and
     * marked as new.
     * If the target file contains a message that wasn't found in the
     * 'from' file, the message is deleted from the former.
     *
     * Note that after return from this function, the ids mapping is
     * completely outdated.
     *
     * Returns an array with statistics:
     *
     *   [0] - # of removed entries
     *   [1] - # of modified entries
     *   [2] - # of new entries
     *   [3] - total # of entries
     */
    array(int) combine(object(LanguageFile) from)
    {
        trace(sprintf("Combining file '%s' with '%s'... ",
                      basename(file), basename(from->file)));
        
        foreach(indices(from->names), string name) {
            mapping entry = from->names[name];
            
            if (names[entry->name]) {
                mapping elocal = names[entry->name];
                int     modified = 0;
                
                if (!elocal->special && (elocal->directives == ([]) || !elocal->directives->orig)) {
                    /*
                     * No original text, we will modify this entry
                     * unconditionally
                     */
                    elocal = entry;
                    elocal->directives = ([]);
                    elocal->directives->orig = entry->msg;
                    elocal->directives->flags = "<changed>";
                    modified = 1;
                } else if (!elocal->special && elocal->directives->orig != entry->msg) {
                    /*
                     * The entries differ. Mark the message as
                     * changed.
                     */
                    elocal->directives->orig = entry->msg;
                    elocal->directives->flags = "<changed>";
                    modified = 1;
                }

                /*
                 * If IDs differ, update.
                 */
                if (elocal->id && (elocal->id != entry->id)) {
                    elocal->id = entry->id;
                    elocal->directives->flags = "<changed>";
                    modified = 1;
                }

                if (!modified) {
                    elocal->touched = 1; /* 1 == found */
                } else {
                    elocal->touched = 2; /* 2 == modified */
                }
            } else {
                mapping    elocal;
                
                /* We've got a new entry */
                names += ([]);
                elocal = names[entry->name] = entry;
                elocal->directives = ([]);
                elocal->directives->orig = entry->msg;
                elocal->directives->flags = "<new>";
                elocal->touched = 3; /* 3 == new */
            }
        }

        /*
         * Now walk the list of the local names and remove those
         * which weren't found in the 'from' file
         */
        int  removed = 0, modified = 0, news = 0, total = 0;
        
        foreach(indices(names), string name) {
            if (!names[name]->touched) {
                removed++;
                m_delete(names, name);
            } else switch(names[name]->touched) {
                case 1:
                    total++;
                    break;

                case 2:
                    modified++;
                    break;

                case 3:
                    news++;
                    break;
            }   
        }

        trace(sprintf("done\n\tEntries: %d removed, %d modified, %d new, %d unchanged\n",
                      removed, modified, news, total));

        combined = 1;
	isempty = 0;
	    
        return ({removed, modified, news, total + modified});
    }

    /*
     * Save the file to disk
     */
    static private void
    write_entry(mapping entry)
    {
        /* First the comments */
        if (entry->comment && entry->comment != "")
            foreach(entry->comment, string line)
                write(sprintf("//!%s\n", line));

        /* Now the directives */
        if (entry->directives && entry->directives != ([])) {
            if (entry->directives->flags)
                write(sprintf("//#FLAG %s\n", entry->directives->flags));
            if (entry->directives->orig)
                write(sprintf("//#ORIG %s\n", entry->directives->orig));
        }

        /*
         * And the entry itself
         */
        write(sprintf("#define\t%s%s\t%s\n\n", 
		      entry->name, 
		      entry->id ? sprintf("\t%d", entry->id) : "",
                      entry->msg));
    }

    static private int id_compare(mapping first, mapping second)
    {
	if (!first->id || !second->id)
	    return 0;
	    
        if (first->id > second->id)
            return 1;
        return 0;
    }
    
    void save()
    {
        array(int) fst = file_stat(file, 1);

        trace(sprintf("Saving the '%s' file... ", basename(file)));
        
        if (!fst)
            throw(({sprintf("Something wicked happened while saving the '%s' file\n", file), backtrace()}));
        
        if (fst[1] < 0)
            throw(({sprintf("Something even more wicked happened while"
                            "saving the '%s' file!\n", file), backtrace()}));

        if (fst[1] > 0) {
            /*
             * let's make a backup.
             */
            rm (file + ".orig");
            hardlink(file, file + ".orig");
            
            fst = file_stat(file + ".orig");
            if (!fst) {
                /* Oops, cannot hardlink? */
                if (!mv(file, file + ".orig"))
                    throw(({sprintf("Cannot create a backup copy of '%s'\n", file),
                            backtrace()})); /* Better be safe than sorry... */
            } else {
                rm(file);
            }
        }
        
        if (!open(file, "wct"))
            throw(({sprintf("Cannot open '%s' for writing\n", file), backtrace()}));

        /*
         * First the intro
         */
        object(Stdio.File) introfile = Stdio.File(MyLocation + "/intro.txt", "r");
        
        if (introfile) {
            write(introfile->read() + "\n\n");
            introfile->close();
        }
        
        /*
         * Now walk the list of names and save them.
         * We will be nice and sort the stuff (although it isn't
         * really necessary...). We do it by creating two arrays out
         * of the names mapping - first with the specials and second
         * with the other entries. Then we sort those arrays.
         * First we save all the specials. Waste of time? Yep, but
         * nice :-)
         */
        array(mapping) specials = ({}), other = ({});
        
        foreach(indices(names), string name) {
            mapping entry = names[name];
            
            if (entry->special)
                specials += ({entry});
        }
        specials = Array.sort_array(specials, id_compare);
        
        foreach(indices(names), string name) {
            mapping entry = names[name];
            
            if (!entry->special)
                other += ({entry});
        }
        other = Array.sort_array(other, id_compare);

        /* Now the output itself */
        write("// For the translation system internal usage. Not used by CAMAS.\n\n");
        foreach(specials, mapping entry)
            write_entry(entry);

        write("\n// CAMAS Messages\n\n");
        foreach(other, mapping entry)
            write_entry(entry);
        
        close();
        saved = 1;
        
        trace("done\n");

        /*
         * Now let's reparse the file, so that names and ids reflect
         * the actual content of the file and the syntax checks are
         * done, just in case...
         */
        read_and_parse(1);
    }

    /*
     * Compile the language definition into a CAMAS language module
     */
    void compile() 
    {
        if (combined && !saved)
            save();
	
        if (!names || names == ([]))
            throw(({"Cannot compile an empty file.\n", backtrace()}));

        string modname = sprintf("%s/camas_%s.pike", ModuleLocation, language);
        
        trace(sprintf("Compiling '%s' -> '%s'... ",
                      basename(file), basename(modname)));
        
        object(Stdio.File) modfile = Stdio.File(modname, "wct");
        object(Stdio.File) gplfile = Stdio.File(MyLocation + "/gpl.txt", "r");
        
        if (!modfile)
            throw(({sprintf("Error opening the '%s' module file for writing.\n", modname),
                    backtrace()}));

        /* First some intro stuff */
        modfile->write(sprintf("/* WARNING! THIS IS A GENERATED FILE. DO NOT EDIT BY HAND! */\n\n"
                               "/*\n"
                               " * Copyright  %s\n"
                               " * Maintainer: %s\n *\n"
                               "%s\n *\n" /* GPL blurb here */
                               " * Generated from template: %s v%s\n *\n"
                               " * $Id: calta.pike,v 1.6.4.1 2001/08/20 18:17:04 kiwi Exp $\n"
                               " */",
                               ids[TS_COPYRIGHT]->msg, ids[TS_MAINTAINER]->msg,
                               gplfile ? gplfile->read() : "This module is released under the GPL license",
                               basename(file), ids[TS_VERSION]->msg));
        if (gplfile)
            gplfile->close();

        /*
         * Now the common functions
         */
        modfile->write("\n#include <camas/language.h>\n\n");

        /* The _lang function */
        modfile->write("string imho_lang() {\n"
                       "\treturn \"%s\";\n}\n\n", ids[TS_LANGNAME]->msg);

        /* The _lang_short function */
        modfile->write("string imho_lang_short() {\n"
                       "\treturn \"%s\";\n}\n\n", language);

        /* The _lang_charset function */
        modfile->write("string imho_lang_charset() {\n"
                       "\treturn \"%s\";\n}\n\n", ids[TS_CHARSET]->msg);

        /* The message translation function itself */
        modfile->write("string msg(object sessobj, int m, array arg) {\n"
                       "\tswitch(m) {\n");

        foreach(indices(names), string name) {
            mapping entry = names[name];

            if (entry->special)
                continue;
            
            modfile->write(sprintf("\t\tcase %s:\n"
                                   "\t\t\treturn %s;\n",
                                   entry->name, format_msg(entry->msg)));
        }

        modfile->write(sprintf("\t\tdefault:\n"
                               "\t\t\treturn %s;\n", format_msg(ids[TS_NOMSGFOUND]->msg)));
        modfile->write("\t}\n}\n");
        modfile->close();

        trace("done\n");
    }
    
    /*
     * Creates new LanguageFile object.
     * Looks for the language files in the given directory. The
     * language file names are in the following format:
     *
     *    language.LANG
     *
     * Where LANG is the requested language code in the following
     * format:
     *
     *    countrycode_VARIANT
     *
     * for example en_GB
     *
     * Parameters:
     *   FPATH  - absolute path to the directory where the language
     *            files are kept
     *   LANG - a language code in the above format. The only exception
     *          is when the default language file is requested. In
     *          that case this parameter should be omitted.
     */
    void create(string lpath, void|string lang)
    {
        if (!lang)
            language = "def";
        else
            language = lang;

        if (lpath[-1] != '/')
            lpath += "/";

        string cwd = getcwd();
        if (!cd(lpath))
            throw(({sprintf("Cannot change to the language directory '%s'\n", lpath),
                    backtrace()}));
        cd(cwd);
        
        file = lpath + sprintf("language.%s", language);

        array(int) fst = file_stat(file, 1);
        
        if (fst && fst[1] < 0)
            throw(({sprintf("'%s' is not a regular file\n",file), backtrace()}));

        if (fst)
            isempty = fst[1] == 0;
        else
            isempty = 1;
        
        if (!open(file, "rc"))
            throw(({sprintf("Cannot open the input file '%s'\n", file), backtrace()}));
        
        msg_comment = ({});
        msg_directive = ([]);
        ids = ([]);
        names = ([]);
        read_and_parse();
        msg_count = spec_count = combined = saved = 0;
    }
}

void usage()
{
    write ("Usage\n");
    exit(1);
}

int main(int argc, array(string) argv)
{
    string   path;
    
    if (argc < 2)
        usage();

    MyLocation = dirname(argv[0]);
    ModuleLocation = combine_path(MyLocation, "../../server/modules/camas/languages/");
    
    if (argv[1][0] != '/')
        path = combine_path(getcwd(), argv[1]);
    else
        path = argv[1];

    object(LanguageFile)   def = LanguageFile(path);
    object(LanguageFile)   target;

    if (argc > 2)
        target = LanguageFile(path, argv[2]);

    if (target) {
        target->combine(def);
        target->compile();
    }
    
    return 0;
}
