/*
 * log.h - logging function library
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#ifndef LOG_H
#define LOG_H

#include <config.h>


#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif /* HAVE_SYSLOG_H */

typedef enum log_level_values log_level;
enum log_level_values
{
	LOG_DBG = 1, LOG_RAW = 2, LOG_ERROR = 3, LOG_WARN = 4, LOG_MSG = 5, LOG_VERBOSE = 6, LOG_FATAL
};

#define STATUS_OK		0
#define STATUS_NULL		1
#define STATUS_USAGE		2
#define STATUS_CHANGED		3
#define STATUS_ERR_MALLOC	4
#define STATUS_ERR_GETOPT	5
#define STATUS_ERR_CHDIR	6
#define STATUS_ERR_CONFIG	7
#define STATUS_ERR_PARSE_ERROR	8
#define STATUS_ERR_BUFFERSIZE	9
#define STATUS_ERR_CFG_OPTION	10
#define STATUS_ERR_AUTO_UPDATE	11
#define STATUS_ERR_QUERY	12
#define STATUS_ERR_INCLUDE	13
#define STATUS_UNKNOWN		99

/*
 * Type of function to use for the actual logging.
 */
typedef void log_function_type(int level, char *message);

/* Open the logfile.
 * @param filename: filename of the logfile.
 * @param verbosity: verbosity level.
 */
void open_logfile(const char* filename, uint8_t verbosity);

/* Close the logfile.
 */
void close_logfile(void);

/* Set the log function.
 * @param log_function: new log function.
 */
void set_logfunction(log_function_type* log_function);

log_function_type log_file;
log_function_type log_syslog;


/* Function to log informational messages.
 * @param s: the log message.
 */
void notice(const char* s, ...);

/* Function to log warnings.
 * @param s: the log message.
 */
void warning(const char* s, ...);

/* Function to log error messages.
 * @param s: the log message.
 */
void error(const char* s, ...);

/* Function to log fatal messages.
 * @param s: the log message.
 */
void fatal(const char* s, ...);

/* Put the string s into the logs without modifications.
 * @param s: the log message.
 */
void raw(const char* s, ...);

/* Function to log more verbose messages.
 * @param s: the level of verbosity.
 * @param s: the log message.
 */
void verbos(uint8_t level, const char* s, ...);

/* Function to log debug messages.
 * @param s: the log message.
 */
void print_debug(const char* s, ...);

/* Function to log debug messages when debugging is disabled.
 * @param s: the log message.
 */
void print_verbose(const char* s, ...);

/* Flush the logs. */
void flush(void);

#ifdef TDEBUG
	#define debug(x) print_debug x
#else /* !TDEBUG */
	#define debug(x)
#endif /* TDEBUG */

/* Function to return message by status id.
 * @param status: status id.
 * @return: the string matching the status id.
 */
const char* status_by_id(int status);

/* Define a function to verify assertions and print failed assertions.
 * @param x: the assertion to be verified.
 */
#ifdef TDEBUG
	#define log_assert(x) \
	do { if (!(x)) { \
		fatal("assertion failed (%s)", #x); \
		exit(1); \
		} \
	} while(0)
#else /* !TDEBUG */
	#define log_assert(x)
#endif /* TDEBUG */

#endif /* LOG_H */
