/* PCNonBipartiteMatchingNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

import java.util.LinkedList;
import java.util.Queue;


/**
 * This class computes the matching number(maximum size of matching) of a
 * non-biparite graph based on the algorithm of Pape and Conradt.
 */
public class PCNonBipartiteMatchingNumber extends GeneralMatchingNumber {
    
    // PRIVATE VARIABLES
    /** A value false means the corresponding vertex is the root of the
     * alternating tree or it belongs to an odd level in the tree. */
    private boolean[] level;
    /** The alternating tree (only store grandparents of vertices at an
     even level in the alternating tree). */
    private int[] grandparent;
    /** Queue needed to construct alternating tree. */
    private Queue<Integer> q;
    
    /**
     * Creates a new instance of NonBipartiteMatchingNumber
     */
    public PCNonBipartiteMatchingNumber() {
        super();
        level = null;
        grandparent = null;
        q = new LinkedList<Integer>();
    }
    
    /**
     * Computes the invariant value(the maximum matching number)
     * for a general graph.
     */
    public int compute(int[][] adjlist) {
        super.adjlist = adjlist;
        initializeMatching(SIMPLE_GREEDY_HEUR);
        level = new boolean[adjlist.length];
        grandparent = new int[adjlist.length];
        int unmatched = adjlist.length - (2*matchingSize);
        for(int i = 0; i < adjlist.length && unmatched >= 2; i++) {
            if(matching[i] == UNSATURATED) {
                q.clear();
                q.add(i);
                for(int j = 0; j < level.length; j++) {
                    level[j] = true;
                }
                level[i] = false;
                initializeTree();
                if(findAugmentingPath(i)) {
                    unmatched = unmatched - 2;
                }
            }
        }
        
        return matchingSize;
    }

    /**
     * This methods searches for an augmenting path starting from
     * the given root (for the corresponding alternating tree)
     * in the graph w.r.t. the current matching.
     * @param root The root of the alternating tree to be build.
     * @return true if an augmenting path was found
     */
    private boolean findAugmentingPath(int root) {
        while(!q.isEmpty()) {
            int x = q.poll().intValue();
            searchInNeighbours:
            for(int j = 0; j < adjlist[x].length; j++) {
                int y = adjlist[x][j];
                if(level[y]) {
                    if(matching[y] == UNSATURATED) {
                        updateMatching(x,y);
                        return true;
                    }
                    else {
                        if(x != root) {
                            int u = grandparent[x];
                            while(u != root) {
                                if(u == y) {
                                    continue searchInNeighbours;
                                }
                                else {
                                    u = grandparent[u];
                                }
                            }
                        }
                        level[y] = false;
                        q.offer(matching[y]);
                        grandparent[matching[y]] = x;                        
                    }
                }
            }
        }
        return false;
    }

    /**
     * This method updates the current matching along the augmenting
     * path found by findAugmentingPath starting for the leave y with
     * parent x in the alternating tree.
     * @param x Parent of y in the alternating tree.
     * @param y The last vertex on the augmenting path.
     */
    private void updateMatching(int x, int y) {
        matching[y] = x;
        int next = matching[x];
        matching[x] = y;
        while(next != -1) {
            x = grandparent[x];
            matching[next] = x;
            y = next;
            next = matching[x];
            matching[x] = y;
        }
        matchingSize++;
    }

    /**
     * Initialize array grandparent.
     */
    private void initializeTree() {
        for(int i = 0; i < grandparent.length; i++) {
            grandparent[i] = -1;
        }
    }
 
}
