/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  April 1996                              */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*         Readline interface to siod with history and completion        */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include <stdlib.h>
#include "EST_unix.h"
#include <string.h>
#include "EST_cutils.h"
#include "siodline.h"

FILE *stddebug = NULL;
int readline_histsize = 256;
extern int rl_pos;
extern char *repl_prompt;

#ifndef SUPPORT_READLINE

/* If for some reason you don't have/can't get readline the following */
/* functions are provided.  They are functional but minimal.  They could */
/* be improved even without readline but I'd rather not spend time */
/* on them.  Possibly when Festival is embedded in commercial (binary  */
/* only versions) you'll need to use these, but in that case you'll probably */
/* not be offering the command interpreter. ... */

int siod_rl_getc(FILE *f)
{
    int c;

    if (rl_pos == -1)
    {
	fprintf(stdout,"%s",repl_prompt);
	fflush(stdout);
	rl_pos = 0;
    }

    c = getc(f);

    if (c == '\n')
	rl_pos = -1;
	
    return c;
}

void siod_rl_ungetc(int c, FILE *f)
{
    ungetc(c,f);
}

void siod_rl_init(void)
{
    return;
}
#else
#include <readline/readline.h>
#include <readline/history.h>

static int possible_commandp(char *text, int start, int end);
static int possible_variablep(char *text, int start, int end);
static char **command_completion (char *text,int start,int end);

static char *rl_line = NULL;

static char *history_file = ".festival_history";

static char *current_sym()
{
    /* Get current symbol at point */
    char *symbol;
    int i,j;

    if (rl_end == 0)
	return NULL;
    if (rl_point == rl_end)
	i=rl_point-1;
    else
	i=rl_point;
	
    for ( ;
	 ((i >= 0) &&
	  (strchr("()' \t\n\r",rl_line_buffer[i]) != NULL));
	 i--);
    /* i will be on final or before final character */
    if (i < 0)
	return NULL;
    /* But if its not at the end of the current symbol move it there */
    for (; i < rl_end; i++)
	if (strchr("()' \t\n\r\"",rl_line_buffer[i]) != NULL)
	    break;
    for (j=i-1; j >=0; j--)
	if (strchr("()' \t\n\r\"",rl_line_buffer[j]) != NULL)
	    break;

    symbol = walloc(char,i-j);
    strncpy(symbol,&rl_line_buffer[j+1],i-(j+1));
    symbol[i-(j+1)] = '\0';

    return symbol;
}

static void siod_display_doc (int ignore)
{
    /* Find the current symbol and check for a documentation string */
    char *symbol, *docstring;
    int i;

    symbol = current_sym();
    crlf();
    docstring = siod_docstring(symbol);
    for (i=0; docstring[i] != '\0'; i++)
	putc(docstring[i],rl_outstream);
    putc('\n',rl_outstream);
    fflush(rl_outstream);
    rl_on_new_line();
    rl_redisplay();
    wfree(symbol);
}

static void siod_say_doc (int ignore)
{
    /* Find the current symbol and check for a documentation string */
    /* Now this is what you call wasting your time.  Here we get the */
    /* synthesizer to say the documentation string                   */
    char *symbol;

    symbol = current_sym();
    crlf();
    fprintf(rl_outstream,"synthesizing doc string ...");
    fflush(rl_outstream);
    siod_saydocstring(symbol);
    putc('\n',rl_outstream);
    fflush(rl_outstream);
    rl_on_new_line();
    rl_redisplay();
    wfree(symbol);
}

static void siod_manual(int ignore)
{
    /* Find the current symbol and check for a documentation string    */
    /* Look for a "see " reference in its documentation string, if so  */
    /* access that section of the manual by sending a call to netscape */
    char *symbol;
    char *infostring;

    symbol = current_sym();
    crlf();
    infostring = siod_manual_sym(symbol);
    fprintf(rl_outstream,infostring);
    fflush(rl_outstream);
    putc('\n',rl_outstream);
    fflush(rl_outstream);
    rl_on_new_line();
    rl_redisplay();
    wfree(symbol);
}

void siod_rl_init(void)
{
    /* Various initialization completion, history etc */
    char *home;

    rl_readline_name = "festival";
    rl_outstream = stderr;

    home = getenv("HOME");
    history_file = walloc(char,strlen(home)+20);
    sprintf(history_file,"%s/.festival_history",home);
    read_history(history_file);
    stifle_history(readline_histsize);

    rl_add_defun("siod-display-doc",(Function *)siod_display_doc, -1);
    rl_bind_key_in_map('h',(Function *)siod_display_doc, emacs_meta_keymap);
    rl_add_defun("siod-say-doc",(Function *)siod_say_doc, -1);
    rl_bind_key_in_map('s',(Function *)siod_say_doc, emacs_meta_keymap);
    rl_add_defun("siod-manual-doc",(Function *)siod_manual, -1);
    rl_bind_key_in_map('m',(Function *)siod_manual, emacs_meta_keymap);
    rl_attempted_completion_function = (CPPFunction *)command_completion;
}

int siod_rl_getc(FILE *f)
{
    int c;

    if (rl_pos == -1)
    {
	rl_line=readline(repl_prompt);
	if (rl_line != NULL)
	{
	    add_history (rl_line);
	    write_history(history_file);
	}
	rl_pos = 0;
    }
    if ((rl_line==NULL) ||
	(strlen(rl_line) <= rl_pos))
	rl_pos = -1;
    if (rl_line==NULL)
	c = EOF;
    else if (rl_pos == -1)
	c = '\n';   /* whitespace representing end of line */
    else 
    {
	c = rl_line[rl_pos];
	rl_pos++;
    }

    return c;
}

void siod_rl_ungetc(int c, FILE *f)
{
    if (rl_pos > 0)
	rl_pos--;
    else
    {
	fprintf(stderr,"fix ungetc when nothing is there");
    }
}
	
static char **command_completion (char *text,int start,int end)
{
    char **matches = NULL;

    /* If preceeding non-alpanum character is a left paren, */
    /* look for a command else look for any variable */
    if (possible_commandp(text,start,end))
	matches = completion_matches(text, siod_command_generator);
    else if (possible_variablep(text,start,end))
	matches = completion_matches(text, siod_variable_generator);

    return matches;
}

static int possible_commandp(char *text, int start, int end)
{
    /* If non-white space previous to this is a left paren */
    /* signal we are looking for a function name           */
    int t;

    for (t=start-1; t >= 0; t--)
	if (strchr(" \t\n\r",rl_line_buffer[t]) != NULL)
	    continue;
	else if (rl_line_buffer[t] == '(')
	    return TRUE;
	else
	    return FALSE;

    return FALSE;
}

static int possible_variablep(char *text, int start, int end)
{
    /* Almost negative of above but if previous symbol is a quote */
    /* let the file completion stuff do it                        */
    int t;

    for (t=start-1; t >= 0; t--)
	if (strchr(" \t\n",rl_line_buffer[t]) != NULL)
	    continue;
	else if (rl_line_buffer[t] == '(')
	    return FALSE;
	else if ((rl_line_buffer[t] == '"') &&
		 (t == start-1))
	    return FALSE;
	else
	    return TRUE;

    return TRUE;
}

#ifdef solaris_2_4_compat
/* If you include libucb.a to get index and rindex readline fails */
/* so include versions here, just map them to ansi versions       */
char *rindex(char *s,int c)
{
    return strrchr(s,c);
}

char *index(char *s,int c)
{
    return strchr(s,c);
}

#endif

#ifdef READLINE_sigmask_fix
int sigmask(int sig)
{
   return (1L << ((sig)-1));
}
#endif /* READLINE_sigmask_fix */

#endif /* SUPPORT_READLINE */
