/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  April 1995                             */
/*-----------------------------------------------------------------------*/
/*                           Filter signals                              */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <iostream.h>
#include <math.h>
#include "EST_Wave.h"
#include "EST_cmd_line.h"
#include "EST_cmd_line_options.h"
#include "EST_sigpr.h"
#include "EST_wave_aux.h"

void inv_filter_ola(EST_Wave &sig, EST_Track &lpc, EST_Wave &res);
void frame_filter(EST_Wave &sig, EST_Track &lpc, EST_Wave &res);
void inv_lpc_filter_ola(EST_Wave &in_sig, EST_Track &lpc, EST_Wave &out_sig);

int main (int argc, char *argv[])
{
    EST_Wave sig, fsig;
    EST_String in_file("-"), out_file("-"), op_file(""), test;
    EST_Option al;
    EST_TList<EST_String> files;
    EST_Track filter;

    parse_command_line(argc, argv, 
       EST_String("Usage:   ")+
       "sigfilter <input file0> -o <output file> <options>\n"+
       "Summary: change/copy/combine waveform files\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_wave_input()+
       "\n"+
       options_wave_output()+
       "\n"+
       "-s <float>       Scaling factor (1.0 is no change)\n"+
       "-scaleN <float>  Scaling factor with normalization\n"+
       "                 waveform is maximized before scaling\n"+
       "-lpfilter <int>  Low pass filter cutoff frequency\n"+
       "-hpfilter <int>  High pass filter cutoff frequency\n"+
       "-forder <int>    Order of FIR filter used for lpfilter and\n"+
       "                 hpfilter (must be ODD; default is 199 to give\n"+
       "                 steep rolloff)\n"+
       "-inv_filter      use filter coefficients for inverse filtering\n"+
       "-lpcfilter <ifile>  Track file containing lpc filter coefficients\n",
			files, al);

    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    if (read_wave(sig, files.first(), al) != format_ok)
	exit(-1);

    if (al.present("-s"))	// rescale
    {
	float scale = al.fval("-s", 0);
	sig.rescale(scale);
    }
    else if (al.present("-scaleN"))	// rescale
    {
	float scale = al.fval("-scaleN", 0);
	if ((scale < 0) || (scale > 1.0))
	{
	    cerr << "ch_wave: -scaleN must be in range 0 to 1" << endl;
	    exit(-1);
	}
	sig.rescale(scale,1);
    }

    // default is to filter before any resampling etc.
    // (this may cause problems for multiplexed data !)

    if (al.present("-lpfilter"))
    {
	FIRlowpass_filter(sig, al.ival("-lpfilter"),al.ival("-forder"));
	fsig = sig;
    }
    if (al.present("-hpfilter"))
    {
	FIRhighpass_filter(sig,al.ival("-hpfilter"),al.ival("-forder"));
	fsig = sig;
    }

    if (al.present("-lpcfilter"))
    {
	filter.load(al.val("-lpcfilter"));
	if (al.present("-inv_filter"))
	    inv_lpc_filter_ola(sig, filter, fsig);
	else
//	    frame_filter(sig, filter, fsig);
	    cout << "not done yet\n";
    }

    if (write_wave(fsig, out_file, al) != write_ok)
    {
	cerr << "sigfilter: failed to write output to \"" << out_file 
	    << "\"" << endl;
	exit(-1);
    }
    return 0;
}
