/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  June 1995                              */
/*-----------------------------------------------------------------------*/
/*                      Stream class file i/o                            */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include "EST_unix.h"
#include "EST_types.h"
#include "ling_class/EST_Stream.h"
#include "EST_string_aux.h"
#include "EST_cutils.h"
#include "EST_TList.h"
#include "EST_Option.h"
#include "stream_io.h"

#define DEF_SAMPLE_RATE 16000
#define HTK_UNITS_PER_SECOND 10000000

EST_read_status read_label_portion(EST_TokenStream &ts, EST_Stream &s, int sample);

EST_read_status load_ogi_label(const EST_String &filename, EST_Stream &s)
{
// This function reads OGI style label files. The start, end
// time and names of the labels are mandatory. 
    EST_TokenStream ts;
    EST_String key, val;
    float sr;
    int isr;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "Error: can't open label input file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");

    // Skip over header

    while(!ts.eof())
      {
	if ((ts.peek().col() == 0) && (ts.peek() == "END"))
	  {
	    if (ts.peek() == "END")
	      { // read rest of header
		ts.get();
		ts.get();
		ts.get();
	      }
	    break;
	  }
	key = ts.get().string();
	val = ts.get().string();
      }

    sr = 1000.0 / atof(val);
    isr = (int)sr;
    
    if (ts.eof())
    {
	cerr << "Error: couldn't find header in label file " 
	     << filename << endl;
	return wrong_format;
    }

    if (read_label_portion(ts, s, isr) == misc_read_error)
    {
	cerr << "error: in label file " << filename << " at line " <<
	    ts.linenum() << endl;
	return misc_read_error;
    }
    s.set_name(filename);
    ts.close();
    return format_ok;
}

EST_read_status load_esps_label(const EST_String &filename, EST_Stream &s)
{
// This function reads xlabel (esps label) style label files. The end
// time and names of the labels are mandatory. Everything after the
// name is put in a list, where each item in the list corresponds to a
// chunk of text between two separators (by default a ";").
    EST_TokenStream ts;
    EST_String t;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
        cerr << "can't open label input file " << filename << endl;
        return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");

    // Skip over header
    for(; !ts.eof(); ts.get())
        if ((ts.peek().col() == 0) && (ts.peek() == "#"))
            break;
    
    if (ts.eof())
    {
        cerr << "couldn't find header in label file " << filename << endl;
        return wrong_format;
    }
    t = ts.get();

    if (!ts.eoln()) // make anything after # the stream name
	s.set_stream_name(ts.get().string());

    read_esps_portion(ts, s);
    s.set_name(filename);
    ts.close();

    return format_ok;
}


EST_read_status load_words_label(const EST_String &filename, EST_Stream &s)
{
// This function reads label files in the form of simple word strings 
// with no timing information.

    EST_StrList words;
    EST_TBI *p;
    EST_Stream_Item item;

    if (load_StrList(filename, words) != format_ok)
	return misc_read_error;

    item.set_end(0.0);

    for (p = words.head(); p; p = next(p))
    {
	item.set_name(words(p));
	s.append(item);
    }
    s.set_name(filename);

    return format_ok;
}

EST_read_status read_esps_portion(EST_TokenStream &ts, EST_Stream &s)
{
    EST_String name, key, val;

    ts.set_SingleCharSymbols(";");
    ts.set_quotes('"','\\');
    
    while(!ts.eof() && (ts.peek() != "#"))
    {
	EST_Stream_Item *item = s.append();

	item->set_end(atof(ts.get().string())); // time
	ts.get();		// ignore that all important color
	
	for (name = ""; (!ts.eoln()) && (ts.peek() != ";"); )
	{
	    EST_Token &t = ts.get();
	    if (name.length() > 0)	// preserve internal whitespace
		name += t.whitespace();  
	    name += t.string();
	}
	
	item->set_name(name);
	
	if (ts.peek().string() == ";") // absorb separator
	{
	    ts.get();
	    item->f.load(ts);
	}
    }
    return format_ok;
}    

EST_read_status read_label_portion(EST_TokenStream &ts, EST_Stream &s, int sample)
{
    EST_Stream_Item item;
    long_long hstart, hend;
    EST_String str;
    
    while(!ts.eof())
    {
	str = ts.get().string();
	if (str == ".")
	    return format_ok;
	
	item.init(s.stream_name());
	item.set_addr(0);
	
	hstart = watoll(str);
	str = ts.get().string();
	hend =  watoll(str);
	
	item.set_end((float)hend/(float)sample); // time
	item.set_name(ts.get().string());	 // name
	
	if (!ts.eoln())
	    item.f.set("lab_fields",ts.get_upto_eoln().string());
	
	s.append(item);
    }
    
    //    return ts.eof() ? format_ok : misc_read_error;
    return format_ok;
}    

EST_read_status load_sample_label(const EST_String &filename, 
				  EST_Stream &s, int sample)
{
    EST_TokenStream ts;
    
    if (sample == 0)	// maybe this should be an error
	sample = DEF_SAMPLE_RATE;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "can't open label input file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    
    s.clear();
    if (read_label_portion(ts, s, 10000000) == misc_read_error)
    {
	cerr << "error: in label file " << filename << " at line " <<
	    ts.linenum() << endl;
	return misc_read_error;
    }
    s.set_name(filename);
    return format_ok;
    
}

EST_write_status save_label_esps(const EST_String &filename, 
				 const EST_Stream &a, 
				 const EST_String &features)
{
    EST_Stream_Item *ptr;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))  
    {
	cerr << "save_label_epsp: can't open label output file \"" << 
	    filename << "\"" << endl;
	return write_fail;
    }
    
    *outf << "separator ;\n";
    *outf << "nfields 1\n";
    *outf << "# " << a.stream_name() << endl;
    
    for (ptr = a.head(); ptr != 0; ptr = next(ptr))
    {
	*outf << "\t";
	outf->precision(5);
	outf->setf(ios::fixed, ios::floatfield);
	outf->width(8);
	//	outf->fill('0');
	*outf << ptr->end();
	
	*outf << " 26 \t" << ptr->name() << " ; ";
	
	if (features == "tilt") 
	{
	    EST_String tiltfeats = ptr->f("tilt");
	    if (tiltfeats == "0")
		cerr << "stream_io.cc:save_label_esps no tilt feature" << endl;
	    else
		*outf << "tilt: " << tiltfeats;
	}
	else if (features == "Features")
	    ptr->f.save(*outf);
	
	else if (features != "None")
	{
	    cerr << "stream_io.cc:save_label_esps unknown feature type to be saved" << endl;
	}
	
	*outf << endl;
    }
    
    if (outf != &cout)
	delete outf;
    return write_ok;
}

EST_write_status save_label_htk(const EST_String &filename, 
				const EST_Stream &a)
{
    EST_Stream_Item *ptr;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))
    {
	cerr << "save_label_htk: can't open label output file \"" << 
	    filename << "\"" << endl;
	return write_fail;
    }
    
    outf->precision(6);
    
    for (ptr = a.head(); ptr != 0; ptr = next(ptr))
    {
	outf->width(15);
	cout.setf(ios::left,ios::adjustfield);
	*outf << (int)(ptr->start() * HTK_UNITS_PER_SECOND);
	outf->width(15);
	*outf << (int)(ptr->end() * HTK_UNITS_PER_SECOND);
	*outf << " " << ptr->name() << endl;
    }
    
    if (outf != &cout)
	delete outf;
    return write_ok;
}

EST_write_status save_label_spn(const EST_String &filename, 
				const EST_Stream &a)
{
    EST_Stream_Item *ptr;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))
    {
	cerr << "save_label_spn: can't open label output file \"" 
	    << filename << "\"" << endl;
	return write_fail;
    }
    
    ptr = a.head();
    outf->precision(3);
    outf->setf(ios::left, ios::adjustfield);
    outf->width(8);
    *outf << ptr->name();
    outf->setf(ios::fixed, ios::floatfield);
    outf->width(8);
    *outf << (ptr->dur() * 1000.0) << "\t (0,140)" << endl;
    
    for (; next(ptr) != 0; ptr = next(ptr))
    {
	outf->precision(3);
	outf->setf(ios::left, ios::adjustfield);
	outf->width(8);
	*outf << ptr->name();
	outf->setf(ios::fixed, ios::floatfield);
	outf->width(8);
	*outf << (ptr->dur() * 1000.0) << endl;
    }
    //    outf->precision(3);
    //    outf->setf(ios::left, ios::adjustfield);
    outf->width(8);
    *outf << ptr->name();
    outf->setf(ios::fixed, ios::floatfield);
    outf->width(8);
    *outf << (ptr->dur() * 1000.0) << "\t (99,80)" << endl;
    
    if (outf != &cout)
	delete outf;
    
    return write_ok;
}

EST_write_status save_label_names(const EST_String &filename, 
				  const EST_Stream &a, 
				  const EST_String &features)
{
    EST_Stream_Item *ptr;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))  
    {
	cerr << "save_label_name: can't open label output file \"" 
	    << filename << "\"" << endl;
	return misc_write_error;
    }
    
    for (ptr = a.head(); next(ptr) != 0; ptr = next(ptr))
    {
	*outf << ptr->name();
	if ((features != "") && (features != "OneLine"))
	    *outf << endl;
	else
	    *outf << " ";
    }
    
    *outf << ptr->name() << endl;
    
    if (outf != &cout)
	delete outf;
    return write_ok;
}

EST_write_status save_StreamList(const EST_String &filename, 
				 const EST_StreamList &plist, 
				 int time, int path)
{
    EST_TBI *p;
    EST_Stream_Item *ptr;
    EST_String outname;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))
    {
	cerr << "save_StreamList: can't open MLF output file \"" 
	    << filename << "\"\n";
	return write_fail;
    }
    
    *outf << "#!MLF!#\n";	// MLF header/identifier
    outf->precision(6);
    
    for (p = plist.head(); p != 0; p = next(p))
    {
	outname = path ? plist(p).name() : basename(plist(p).name());
	*outf << "\"*/" << outname<<"\"\n";
	for (ptr = plist(p).head(); ptr != 0; ptr = next(ptr))
	{
	    if (time)
	    {
		outf->width(15);
		cout.setf(ios::left,ios::adjustfield);
		*outf << (int)(ptr->start() * HTK_UNITS_PER_SECOND);
		outf->width(15);
		*outf << (int)(ptr->end() * HTK_UNITS_PER_SECOND) << " ";
	    }
	    *outf << ptr->name() << endl;
	}
	*outf << ".\n";
    }
    
    if (outf != &cout)
	delete outf;
    return write_ok;
}    

EST_write_status save_WordList(const EST_String &filename, 
			       const EST_StreamList &plist, 
			       int style)
{
    EST_TBI *p;
    EST_Stream_Item *ptr;
    
    ostream *outf;
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))
    {
	cerr << "save:WordList: can't open WordList output file \"" 
	    << filename << "\"\n";
	return write_fail;
    }
    
    for (p = plist.head(); p != 0; p = next(p))
    {
	for (ptr = plist(p).head(); next(ptr) != 0; ptr = next(ptr))
	{
	    *outf << ptr->name();
	    if (style == 0)
		*outf << endl;
	    else
		*outf << " ";
	}
	if (ptr != 0)
	    *outf << ptr->name() << endl;
    }
    
    if (outf != &cout)
	delete outf;
    return write_ok;
}    

EST_write_status save_ind_StreamList(const EST_String &filename, 
				     const EST_StreamList &plist, 
				     const EST_String &features, 
				     int path)
{
    EST_TBI *p;
    EST_String outname;
    (void) filename;
    
    for (p = plist.head(); p != 0; p = next(p))
    {
	outname = path ? plist(p).name() : basename(plist(p).name());
	if (plist(p).save(outname, "esps", features) != write_ok)
	    return misc_write_error;
    }
    
    return write_ok;
}    

EST_read_status load_StreamList(const EST_String &filename, 
				EST_StreamList &plist)
{
    EST_TokenStream ts;
    EST_String fns, name;
    EST_Stream s;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "Can't open label input file " << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    
    // Skip over header
    if (ts.get().string() != "#!MLF!#")
    {
	cerr << "Not MLF file\n";
	return wrong_format;
    }
    
    while(!ts.eof())
    {
	s.clear();
	// put filename in as stream name. The filename is usually surrounded
	// by quotes, so remove these.
	fns = ts.get().string();
	strip_quotes(fns);
	s.set_name(fns);
	
    	if (read_label_portion(ts, s, 10000000) == misc_read_error)
	{
	    cerr << "error: in reading MLF file\n";
	    cerr << "section for file " << s.stream_name() << 
		" at line " << ts.linenum() << " is badly formatted\n";
	    
	    return misc_read_error;
	}
	plist.append(s);
    }
    
    return format_ok;
}

// add evenly spaced dummny end values to Stream
static void pad_ends(EST_Stream &s, float length)
{
    EST_Stream_Item *p;
    int i;
    
    for (i = 0, p = s.head(); p; p = next(p), ++i)
	p->set_end(length * float(i)/float(s.length()));
}

EST_read_status read_StreamList(EST_StreamList &plist, EST_StrList &files, EST_Option &al)
{
    EST_Stream s;
    EST_TBI *p, *plp;
    
    if (al.val("-itype", 0) == "mlf")
    {
	if (load_StreamList(files.first(), plist) != format_ok)
	    exit (-1);
    }
    else
	for (p = files.head(); p; p = next(p))
	{
	    s.clear();
	    plist.append(s);
	    plp = plist.tail();
	    if (al.present("-itype"))
	    {
		if (plist(plp).load_type(files(p), al.val("-itype"), 
					 al.ival("-lf", 0)) != format_ok)
		    exit (-1);
	    }
	    else if (plist(plp).load(files(p)) != format_ok)
		exit (-1);
	    plist(plp).set_stream_name(files(p));
	    if ((al.val("-itype", 0) == "words") && (al.present("-length")))
		pad_ends(s, al.fval("-length"));
	    
	}
    
    return format_ok;
}
