/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  February 1996                            */
/*-----------------------------------------------------------------------*/
/*                    Event RFC Utilities                                */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <math.h>
#include "tilt.h"
#include "EST_error.h"

void validate_rfc_stream(EST_Relation &ev);

EST_Val simple_start_time(EST_Item *s)
{
    if ((s == 0) || (prev(s) == 0))
	return 0.0;
    else
	return prev(s)->f("end");
}

float rfc_to_tilt_amp(EST_Item *e)
{
    return fabs(e->fF("rfc:rise_amp")) + fabs(e->fF("rfc:fall_amp"));
}

float rfc_to_tilt_dur(EST_Item *e)
{
    return e->fF("rfc:rise_dur") + e->fF("rfc:fall_dur");
}

float rfc_to_a_tilt(EST_Item *e)
{
    return (float)(fabs(e->fF("rfc:rise_amp")) - fabs(e->fF("rfc:fall_amp"))) / 
	(float)(fabs(e->fF("rfc:rise_amp")) + fabs(e->fF("rfc:fall_amp")));
}

float rfc_to_d_tilt(EST_Item *e)
{
    return (float)(fabs(e->fF("rfc:rise_dur")) - fabs(e->fF("rfc:fall_dur"))) / 
	(float)(e->fF("rfc:rise_dur") + e->fF("rfc:fall_dur"));
}

float rfc_to_t_tilt(EST_Item *e)
{
    float t_tilt;
    t_tilt = (rfc_to_a_tilt(e) + rfc_to_d_tilt(e)) / 2;
    if (isnan(t_tilt))
	t_tilt = 0.0;
    return t_tilt;
}

float tilt_to_rise_amp(EST_Item *e)
{
    return e->fF("tilt:amp") * (1 + e->fF("tilt:tilt")) / 2.0;
}

float tilt_to_rise_dur(EST_Item *e)
{
    return e->fF("tilt:dur") * (1 + e->fF("tilt:tilt")) / 2.0;
}

float tilt_to_fall_amp(EST_Item *e)
{
    return -e->fF("tilt:amp") * (1 - e->fF("tilt:tilt")) / 2.0;
}

float tilt_to_fall_dur(EST_Item *e)
{
    return e->fF("tilt:dur") * (1 - e->fF("tilt:tilt")) / 2.0;
}

float tilt_to_peak_f0(EST_Item *e)
{
    return  e->fF("ev:start_f0") + tilt_to_rise_amp(e);
}

float tilt_to_peak_pos(EST_Item *e)
{
    return  e->fF("start") + tilt_to_rise_dur(e);
}

void rfc_to_tilt(EST_Relation &ev)
{
    EST_Item *e;

    if (ev.f("intonation_style") != "rfc")
	EST_error("Can't create Tilt parameters from intonation style: %s\n", 
		  (const char *)ev.f.S("intonation_style"));

    cout << "converting to tilt\n";
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout << "converting " << *e << " to tilt\n";
	if (event_item(*e))
	{
	    cout << "converting to tilt event\n";
	    cout << "a tilt" << rfc_to_a_tilt(e) << endl;
	    cout << "d tilt" << rfc_to_d_tilt(e) << endl;
	    cout << "t tilt" << rfc_to_t_tilt(e) << endl;
	    e->fset("tilt:amp", rfc_to_tilt_amp(e));
	    e->fset("tilt:dur", rfc_to_tilt_dur(e));
	    e->fset("tilt:tilt", rfc_to_t_tilt(e));
	}
    }
    ev.f.set("intonation_style", "tilt");
}

void tilt_to_rfc(EST_Relation &ev)
{
    EST_Item *e;
    bool rise_fall;

//    if (ev.f("intonation_style") != "tilt")
//	EST_error("Can't create RFC parameters stream type: %s\n", 
//		  (const char *)ev.f.S("intonation_style"));
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	if (event_item(*e))
	{
	    rise_fall = ((e->fF("tilt:tilt") < 0.9) 
			 && (e->fF("tilt:tilt") > -0.9)) ? true : false;

	    e->fset("rfc:peak_f0", (rise_fall) ? tilt_to_peak_f0(e) : 0.0);
	    e->fset("rfc:peak_pos",(rise_fall) ? tilt_to_peak_pos(e): 0.0);
	    e->fset("rfc:rise_amp", tilt_to_rise_amp(e));
	    e->fset("rfc:rise_dur", tilt_to_rise_dur(e));
	    e->fset("rfc:fall_amp", tilt_to_fall_amp(e));
	    e->fset("rfc:fall_dur", tilt_to_fall_dur(e));

	    if (rise_fall)
		e->fset("rfc:type", "RISEFALL");
	    else if (e->fF("tilt:tilt") > 0.0)
		e->fset("rfc:type", "RISE");
	    else 
		e->fset("rfc:type", "FALL");
	}
	else
	    e->fset("rfc:type", "SIL");
    }

    ev.f.set("intonation_style", "rfc"); // say that this now contains rfc
    
}

void fill_rfc_types(EST_Relation &ev)
{
    EST_Item *e;
    
    for (e = ev.head(); e; e = next(e))
    {
	if (event_item(*e))
	{
	    if ((e->fF("rfc:rise_amp", 1) > 0.0) && (e->fF("rfc:fall_amp", 1) < 0.0))
		e->fset("rfc:type", "RISEFALL");
	    else if (e->fF("rfc:rise_amp", 1) > 0.0)
		e->fset("rfc:type", "RISE");
	    else 
		e->fset("rfc:type", "FALL");
	}
	else
	    e->fset("rfc:type", "SIL");
    }
}

void int_segment_to_unit(EST_Relation &int_lab, EST_Relation &ev_lab)
{
    EST_Item *e, *inext;
    (void)ev_lab;

    if (int_lab.f("timing_style") != "segment")
	EST_error("Undefined timing style:%s in relation\n", 
		  (const char *)int_lab.f.S("timing_style"));

    for (e = int_lab.head(); e != 0; e = next(e))
	e->fset("start", e->fF("start"));

    for (e = int_lab.head(); e != 0; e = inext)
    {
	inext = next(e);
	if (event_item(*e) || sil_item(*e))
	    continue;
	int_lab.remove_item(e);
    }

/*    for (e = int_lab.head(); e != 0; e = inext)
    {
	if (event_item(*e))
	    ev_lab.append(e);
    }
*/

    int_lab.f.set("timing_style", "unit");

}

void fn_start_to_real_start(EST_Relation &ev)
{
    for (EST_Item *e = ev.head(); e; e = next(e))
	e->fset("start", e->fF("start"));
}

void set_fn_start(EST_Relation &ev)
{
    for (EST_Item *e = ev.head(); e; e = next(e))
	e->fset("start", simple_start_time);

}

// this should move to EST_Relation_aux.cc after REORG. This
// adds a dummy stream_item with no name.
void event_to_segment(EST_Relation &ev, float min_length)
{
    EST_Item *e, *n;
    EST_Item *dummy;

    // REORG - replace by stream feature
    if (ev.f.S("timing_style") != "event")
	return;

    for (e = ev.head(); next(e) != 0; e = next(e))
    {
	n = next(e);
	if ((n->fF("start") - e->fF("end"))  > min_length)
	{
	    dummy = e->insert_after();
//	    dummy->fset("end", 19.0);
	    dummy->fset("end", n->fF("start"));
//	    cout << n->fF("start") << endl;
//	    sleep(2);
//	    dummy->fset("start", e->fF("end"));
//	    cout << *n << endl;
	    cout << *dummy << endl;
//	    cout << dummy << endl;
//	    sleep(1);
	}
/*	else // make sure starts and ends are now properly contiguous
	{
	    float pos = (n->fF("start") - e->fF("end")) / 2.0;
	    cout << "balanced pos = " << pos << endl;
	    e->fset("end", pos);
	    n->fset("start", pos);
	}
*/
    }
    cout << "Event to segment\n";
    cout << ev;
//  sleep(2);
    set_fn_start(ev);

    // REORG - replace by stream feature
    ev.f.set("timing_style", "segment");
    cout << "Event to segment\n";
    cout << ev;
//    sleep(2);

}

void remove_rfc_features(EST_Relation &ev)
{
    for (EST_Item *e = ev.head(); e != 0; e = next(e))
    {
	e->f_remove("rfc:rise_amp");
	e->f_remove("rfc:rise_dur");
	e->f_remove("rfc:fall_amp");
	e->f_remove("rfc:fall_dur");
	e->f_remove("rfc:type");
    }
}

void remove_tilt_features(EST_Relation &ev)
{
    for (EST_Item *e = ev.head(); e != 0; e = next(e))
    {
	e->f_remove("tilt:amp");
	e->f_remove("tilt:dur");
	e->f_remove("tilt:tilt");
    }
}

float unit_curve(EST_String type, float amp, float dur, float t)
{
    float val;
    float x;
    float gradient;
    
    if (type == "RISE")
    {
	x = (t / (dur)) * 2.0;
	if (x < 1.0)
	    val = pow(x, 2.0);
	else
	    val = 2 - pow((2 - x), 2.0);
	
	val = (val / 2.0);
	
	val *= amp;
	val += 0;		// vert dist.
    }
    else if (type == "FALL")
    {
	x = (t / (dur)) * 2.0;
	if (x < 1.0)
	    val = pow(x, 2.0);
	else
	    val = 2 - pow((2 - x), 2.0);
	
	val = (val / 2.0);
	
	val *= amp;
	val += 0;		// vert dist.
    }
    else
    {
	gradient = amp / dur;
	val = (gradient * t); // + el1->s_freq; vert dist.
    }
    
    return (val);
}

/* find index of contour given position in ms. Return -1 if */
float fncurve(float length, float t, float curve)
{
    float val;
    float x;
    
    x = (t / length) * 2.0;
    
    if (x < 1.0)
	val = pow(x, curve);
    else
	val = 2 - pow((2 - x), curve);
    
    val = val / 2.0;
    
    return (val);
}

int event_item(EST_Item &e)
{	
    return ((int)e.f("int_event"));
}
int sil_item(EST_Item &e)
{
    return ((e.name() == "sil") || (e.name() == "SIL"));
}
int connection_item(EST_Item &e)
{
    return ((e.name() == "c") || (e.name() == "C"));
}
