/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author :  Alan W Black                          */
/*                       Date   :  February 1998                         */
/* --------------------------------------------------------------------- */
/*                        another architecture                           */
/*                                                                       */
/*************************************************************************/
#ifndef __EST_RELATION_H__
#define __EST_RELATION_H__

#include "EST_String.h"
#include "EST_TList.h"
#include "EST_KV.h"
#include "EST_Val.h"
#include "EST_types.h"
#include "EST_Token.h"
#include "EST_Features.h"
#include "ling_class/EST_Item.h"

class EST_Utterance;

class EST_Relation_Itterator;

/** relations are between \Ref{EST_Item}s in an \Ref{EST_Utterance}.
These defined trees or lists over related items, representing things like
syntax tree, lists of words, syllables etc.

*/

class EST_Relation
{
    EST_String p_name;
    EST_Utterance *p_utt;
    EST_Item *p_head;   
    EST_Item *p_tail;   // less meaningful in a tree

    EST_Item *get_item_from_name(KVL<int,EST_Item *> &inames,int name);
    EST_write_status save_items(EST_Item *item, 
				ostream &outf,
				KVL<void *,int> &contentnames,
				KVL<void *,int> &itemnames,
				int &node_count) const;
    EST_read_status load_items(EST_TokenStream &ts,
			       KVL<int,EST_Item_Content *> contents);
    void copy(const EST_Relation &r);
  public:
    EST_Relation();
    EST_Relation(const EST_String &name);
    EST_Relation(const EST_Relation &r) { copy(r); }
    ~EST_Relation();

    EST_Features f;

    void clear();

    void set_utt(EST_Utterance *u) { p_utt = u; }
    EST_Utterance *utt(void) { return p_utt; }
    
    const EST_String &name() const { return p_name; }
    EST_Item *head() const {return (this == 0) ? 0 : p_head;}
    EST_Item *root() const {return head();}
    EST_Item *tail() const {return (this == 0) ? 0 : p_tail;}

    EST_Item *first() const { return head(); }
    EST_Item *first_leaf() const;
    EST_Item *last() const { return tail(); }
    EST_Item *last_leaf() const;

    typedef EST_Relation_Itterator Itterator;
    
    int length() const;
    int num_nodes() const;
    int num_leafs() const;
    int empty() const { return p_head == 0; }

    EST_Item *append(EST_Item *si);
    EST_Item *append(); 
    EST_Item *prepend(EST_Item *si);
    EST_Item *prepend(); 
    void remove_item(EST_Item *item);

    // remove all occurances of feature name from relation's items
    void remove_item_feature(const EST_String &name);

    EST_read_status load(const EST_String &filename,
			 const EST_String &type="esps");
    EST_read_status load(EST_TokenStream &ts,
			 KVL<int,EST_Item_Content *> contents);
    EST_write_status save(const EST_String &filename);
    EST_write_status save(ostream &outf,KVL<void *,int> contents) const;

    EST_Relation &operator=(const EST_Relation &s);
    friend ostream& operator << (ostream &s, const EST_Relation &u);

    friend EST_Item;
};

inline bool operator==(const EST_Relation &a, const EST_Relation &b)
   { return (&a == &b); }

void copy_relation(const EST_Relation &from, EST_Relation &to);

EST_Utterance *get_utt(EST_Item *s);

class EST_Relation_Itterator
{
private:
  const EST_Relation &rel;
  EST_Item *next;

public:
  EST_Relation_Itterator(const EST_Relation &r)
    : rel(r), next(NULL) { reset();};

  void reset() 
    { next=rel.head(); }
  bool has_more_elements()
    { return next != NULL; }

  EST_Item *next_element();
};


#endif
